/*
 * rtmsg.c - route message parser
 * Copyright (C) 2014 Tetsumune KISO <t2mune@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "nield.h"
#include "rtnetlink.h"

/*
 * parse route message
 */
int parse_rtmsg(struct nlmsghdr *nlh)
{
    struct rtmsg *rtm;
    int rtm_len;
    struct rtattr *rta[__RTA_MAX];
    char ipv[MAX_STR_SIZE] = "";
    char msg[MAX_MSG_SIZE] = "";
    char *mp = msg;
    int log_opts = get_log_opts();
    int res;

    /* debug nlmsghdr */
    if(log_opts & L_DEBUG)
        debug_nlmsg(0, nlh);

    /* get rtmsg */
    rtm_len = NLMSG_PAYLOAD(nlh, 0);
    if(rtm_len < sizeof(*rtm)) {
        rec_log("error: %s: rtmsg: length too short", __func__);
        return(1);
    }
    rtm = (struct rtmsg *)NLMSG_DATA(nlh);

    /* parse route attributes */
    parse_rtattr(rta, RTA_MAX, RTM_RTA(rtm), RTM_PAYLOAD(nlh));

    /* debug rtmsg */
    if(log_opts & L_DEBUG)
        debug_rtmsg(0, rtm, rta, rtm_len);

    /* check address family */
    char dst[INET6_ADDRSTRLEN] = "";
    if(rtm->rtm_family == AF_INET) {
        strcpy(ipv, "ipv4");
        strcpy(dst, "0.0.0.0");
    } else if(rtm->rtm_family == AF_INET6) {
        strcpy(ipv, "ipv6");
        strcpy(dst, "::");
    } else {
        rec_log("error: %s: unknown address family: %d",
                __func__, rtm->rtm_family);
        return(1);
    }

    /* convert from table id to table name */
    char table[MAX_STR_SIZE] = "";
    snprintf(table, sizeof(table), "%s", convert_rt_table(rtm->rtm_table, 0));
    if(!strncmp(table, "unknown", sizeof(table)))
        snprintf(table, sizeof(table), "%d", rtm->rtm_table);

    /* check route table id(other than RT_TABLE_LOCAL) */
    if(rtm->rtm_table == RT_TABLE_LOCAL)
        return(1);

    /* check route protocol(other than RTPROT_UNSPEC) */
    if(rtm->rtm_protocol == RTPROT_UNSPEC)
        return(1);
    
    /* check route flags(other then RTM_F_CLONED) */
    if(rtm->rtm_flags & RTM_F_CLONED)
        return(1);

    /* get destination prefix */
    if(rta[RTA_DST]) {
        res = inet_ntop_ifa(rtm->rtm_family, rta[RTA_DST], dst, sizeof(dst));
        if(res) {
            rec_log("error: %s: RTA_DST: %s", __func__,
                (res == 1) ? strerror(errno) : "payload too short");
            return(1);
        }
    }
    /* no RTA_DST attribute if destination is a default gateway */
    mp = add_log(msg, mp, "destination=%s/%d ", dst, rtm->rtm_dst_len);

    /* get source prefix */
    if(rta[RTA_SRC]) {
        char src[INET6_ADDRSTRLEN] = "";

        res = inet_ntop_ifa(rtm->rtm_family, rta[RTA_SRC], src, sizeof(src));
        if(res == 1) {
            rec_log("error: %s: RTA_SRC: %s", __func__,
                (res == 1) ? strerror(errno) : "payload too short");
            return(1);
        }
        mp = add_log(msg, mp, "source=%s/%d ", src, rtm->rtm_src_len);
    }

    /* get preferred source address */
    if(rta[RTA_PREFSRC]) {
        char prefsrc[INET6_ADDRSTRLEN] = "";

        res = inet_ntop_ifa(rtm->rtm_family, rta[RTA_PREFSRC], prefsrc, sizeof(prefsrc));
        if(res) {
            rec_log("error: %s: RTA_PREFSRC: %s", __func__,
                (res == 1) ? strerror(errno) : "payload too short");
            return(1);
        }
        mp = add_log(msg, mp, "preferred-source=%s ", prefsrc);
    }

    /* get tos */
    if(rtm->rtm_tos)
        mp = add_log(msg, mp, "tos=0x%.2x ", rtm->rtm_tos);

    /* get ingress interface */
    if(rta[RTA_IIF]) {
        unsigned iifindex;
        char iifname[IFNAMSIZ] = "";

        if(RTA_PAYLOAD(rta[RTA_IIF]) < sizeof(iifindex)) {
            rec_log("error: %s: RTA_IIF: payload too short", __func__);
            return(1);
        }
        iifindex = *((unsigned *)RTA_DATA(rta[RTA_IIF]));
        if_indextoname_from_lists(iifindex, iifname);

        mp = add_log(msg, mp, "in=%s ", iifname);
    }

    /* get gateway address */
    if(rta[RTA_GATEWAY]) {
        char nexthop[INET6_ADDRSTRLEN] = "";

        res = inet_ntop_ifa(rtm->rtm_family, rta[RTA_GATEWAY], nexthop, sizeof(nexthop));
        if(res) {
            rec_log("error: %s: RTA_GATEWAY: %s", __func__,
                (res == 1) ? strerror(errno) : "payload too short");
            return(1);
        }
        mp = add_log(msg, mp, "nexthop=%s ", nexthop);
    }

    /* get egress interface */
    if(rta[RTA_OIF]) {
        unsigned oifindex;
        char oifname[IFNAMSIZ] = "";

        if(RTA_PAYLOAD(rta[RTA_OIF]) < sizeof(oifindex)) {
            rec_log("error: %s: RTA_OIF: payload too short", __func__);
            return(1);
        }
        oifindex = *((unsigned *)RTA_DATA(rta[RTA_OIF]));
        if_indextoname_from_lists(oifindex, oifname);

        mp = add_log(msg, mp, "interface=%s ", oifname);
    }

    /* get priority(but metric) */
    char metric[MAX_STR_SIZE] = "";
    if(rta[RTA_PRIORITY]) {
        if(RTA_PAYLOAD(rta[RTA_PRIORITY]) < sizeof(int)) {
            rec_log("error: %s: RTA_PRIORITY: payload too short", __func__);
            return(1);
        }
        snprintf(metric, sizeof(metric), "metric=%d ", *((int *)RTA_DATA(rta[RTA_PRIORITY])));
    }

    /* convert route message type */
    char type[MAX_STR_SIZE] = "";
    snprintf(type, sizeof(type), "%s", convert_rtn_type(rtm->rtm_type, 0));

    /* convert route message protocol */
    char proto[MAX_STR_SIZE] = "";
    snprintf(proto, sizeof(proto), "%s", convert_rtprot(rtm->rtm_protocol, 0));

    /* get table id & name */
    if(rta[RTA_TABLE]) {
        int table_id = *(int *)RTA_DATA(rta[RTA_TABLE]);

        if(RTA_PAYLOAD(rta[RTA_TABLE]) < sizeof(int)) {
            rec_log("error: %s: RTA_TABLE: payload too short", __func__);
            return(1);
        }
        snprintf(table, sizeof(table), "%s", convert_rt_table(table_id, 0));
        if(!strncmp(table, "unknown", sizeof(table)))
            snprintf(table, sizeof(table), "%d", table_id);
    }

    /* get multipath */
    if(rta[RTA_MULTIPATH]) {
        struct rtnexthop *rtnh;
        int rtnh_len = RTA_PAYLOAD(rta[RTA_MULTIPATH]);
        struct rtattr *rtna[__RTA_MAX];
        char rtnh_ifname[IFNAMSIZ] = "";
        char rtnh_nexthop[INET6_ADDRSTRLEN] = "";

        if(RTA_PAYLOAD(rta[RTA_MULTIPATH]) < sizeof(*rtnh)) {
            rec_log("error: %s: RTA_MULTIPATH: payload too short", __func__);
            return(1);
        }
        rtnh = RTA_DATA(rta[RTA_MULTIPATH]);

        for(; RTNH_OK(rtnh, rtnh_len);
            rtnh = RTNH_NEXT(rtnh), rtnh_len -= RTNH_ALIGN(rtnh->rtnh_len)) {
            parse_rtattr(rtna, RTA_MAX, RTNH_DATA(rtnh), rtnh->rtnh_len - sizeof(*rtnh));

            if(rtna[RTA_GATEWAY]) {
                res = inet_ntop_ifa(rtm->rtm_family, rtna[RTA_GATEWAY],
                    rtnh_nexthop, sizeof(rtnh_nexthop));
                if(res) {
                    rec_log("error: %s: RTA_GATEWAY: %s", __func__,
                        (res == 1) ? strerror(errno) : "payload too short");
                    return(1);
                }
            }

            /* get interface name & logging routing table message */
            if_indextoname_from_lists(rtnh->rtnh_ifindex, rtnh_ifname);
            if(nlh->nlmsg_type == RTM_NEWROUTE)
                rec_log("%s route added: %snexthop=%s interface=%s "
                    "%sweight=%d type=%s protocol=%s table=%s",
                    ipv, msg, rtnh_nexthop, rtnh_ifname,
                    metric, rtnh->rtnh_hops+1, type, proto, table);
            else if(nlh->nlmsg_type == RTM_DELROUTE)
                rec_log("%s route deleted: %snexthop=%s interface=%s " 
                    "%sweight=%d type=%s protocol=%s table=%s",
                    ipv, msg, rtnh_nexthop, rtnh_ifname,
                    metric, rtnh->rtnh_hops+1, type, proto, table);
        }

        return(0);
    }

    /* logging routing message */
    if(nlh->nlmsg_type == RTM_NEWROUTE)
        rec_log("%s route added: %s%stype=%s protocol=%s table=%s",
            ipv, msg, metric, type, proto, table);
    else if(nlh->nlmsg_type == RTM_DELROUTE)
        rec_log("%s route deleted: %s%stype=%s proto=%s table=%s",
            ipv, msg, metric, type, proto, table);

    return(0);
}

/*
 * debug route message
 */
void debug_rtmsg(int lev, struct rtmsg *rtm, struct rtattr *rta[], int rtm_len)
{
    /* debug rtmsg */
    char flags_list[MAX_STR_SIZE] = "";

    convert_rtm_flags(rtm->rtm_flags, flags_list, sizeof(flags_list));

    rec_dbg(lev, "*********************************************************************");
    rec_dbg(lev, "[ rtmsg(%d) ]",
        NLMSG_ALIGN(sizeof(struct rtmsg)));
    rec_dbg(lev, "    rtm_family(%d): %d(%s)",
        sizeof(rtm->rtm_family), rtm->rtm_family,
        convert_af_type(rtm->rtm_family));
    rec_dbg(lev, "    rtm_dst_len(%d): %d",
        sizeof(rtm->rtm_dst_len), rtm->rtm_dst_len);
    rec_dbg(lev, "    rtm_src_len(%d): %d",
        sizeof(rtm->rtm_src_len), rtm->rtm_src_len);
    rec_dbg(lev, "    rtm_tos(%d): %d",
        sizeof(rtm->rtm_tos), rtm->rtm_tos);
    rec_dbg(lev, "    rtm_table(%d): %d(%s)",
        sizeof(rtm->rtm_table), rtm->rtm_table,
        convert_rt_table(rtm->rtm_table, 1));
    rec_dbg(lev, "    rtm_protocol(%d): %d(%s)",
        sizeof(rtm->rtm_protocol), rtm->rtm_protocol,
        convert_rtprot(rtm->rtm_protocol, 1));
    rec_dbg(lev, "    rtm_scope(%d): %d(%s)",
        sizeof(rtm->rtm_scope), rtm->rtm_scope,
        convert_rt_scope(rtm->rtm_scope));
    rec_dbg(lev, "    rtm_type(%d): %d(%s)",
        sizeof(rtm->rtm_type), rtm->rtm_type,
        convert_rtn_type(rtm->rtm_type, 1));
    rec_dbg(lev, "    rtm_flags(%d): %d(%s)",
        sizeof(rtm->rtm_flags), rtm->rtm_flags,
        flags_list);

    /* debug route attributes */
    rec_dbg(lev, "*********************************************************************");
    rec_dbg(lev, "[ rtmsg attributes(%d) ]",
        NLMSG_ALIGN(rtm_len - NLMSG_ALIGN(sizeof(struct rtmsg))));

    if(rta[RTA_DST])
        debug_rta_dst(lev+1, rtm, rta[RTA_DST]);

    if(rta[RTA_SRC])
        debug_rta_src(lev+1, rtm, rta[RTA_SRC]);

    if(rta[RTA_IIF])
        debug_rta_iif(lev+1, rta[RTA_IIF]);

    if(rta[RTA_OIF])
        debug_rta_oif(lev+1, rta[RTA_OIF]);

    if(rta[RTA_GATEWAY])
        debug_rta_gateway(lev+1, rtm, rta[RTA_GATEWAY]);

    if(rta[RTA_PRIORITY])
        debug_rta_priority(lev+1, rta[RTA_PRIORITY]);

    if(rta[RTA_PREFSRC])
        debug_rta_prefsrc(lev+1, rtm, rta[RTA_PREFSRC]);

    if(rta[RTA_METRICS])
        debug_rta_metrics(lev+1, rta[RTA_METRICS]);

    if(rta[RTA_MULTIPATH])
        debug_rta_multipath(lev+1, rtm, rta[RTA_MULTIPATH]);

    if(rta[RTA_FLOW])
        debug_rta_flow(lev+1, rta[RTA_FLOW]);

    if(rta[RTA_CACHEINFO])
        debug_rta_cacheinfo(lev+1, rta[RTA_CACHEINFO]);

    if(rta[RTA_TABLE])
        debug_rta_table(lev+1, rta[RTA_TABLE]);

#if HAVE_DECL_RTA_MARK
    if(rta[RTA_MARK])
        debug_rta_mark(lev+1, rta[RTA_TABLE]);
#endif

    rec_dbg(lev, "");

    return;
}

/*
 * debug attribute RTA_DST
 */
void debug_rta_dst(int lev, struct rtmsg *rtm, struct rtattr *rta)
{
    char dst[INET6_ADDRSTRLEN] = "";
    int res;

    res = inet_ntop_ifa(rtm->rtm_family, rta, dst, sizeof(dst));
    if(res) {
        rec_dbg(lev, "RTA_DST(%hu): -- %s --",
            RTA_ALIGN(rta->rta_len),
            (res == 1) ? strerror(errno) : "payload too short");
        return;
    }

    rec_dbg(lev, "RTA_DST(%hu): %s", RTA_ALIGN(rta->rta_len), dst);
}

/*
 * debug attribute RTA_SRC
 */
void debug_rta_src(int lev, struct rtmsg *rtm, struct rtattr *rta)
{
    char src[INET6_ADDRSTRLEN] = "";
    int res;

    res = inet_ntop_ifa(rtm->rtm_family, rta, src, sizeof(src));
    if(res) {
        rec_dbg(lev, "RTA_SRC(%hu): -- %s --",
            RTA_ALIGN(rta->rta_len),
            (res == 1) ? strerror(errno) : "payload too short");
        return;
    }

    rec_dbg(lev, "RTA_SRC(%hu): %s", RTA_ALIGN(rta->rta_len), src);
}

/*
 * debug attribute RTA_IIF
 */
void debug_rta_iif(int lev, struct rtattr *rta)
{
    char ifname[IFNAMSIZ] = "";
    int ifindex;

    if(RTA_PAYLOAD(rta) < sizeof(ifindex)) {
        rec_dbg(lev, "RTA_IIF(%hu): -- payload too short --",
            RTA_ALIGN(rta->rta_len));
        return;
    }
    ifindex = *((int *)RTA_DATA(rta));
    if_indextoname_from_lists(ifindex, ifname);

    rec_dbg(lev, "RTA_IIF(%hu): %d(%s)",
        RTA_ALIGN(rta->rta_len), ifindex, ifname);
}

/*
 * debug attribute RTA_OIF
 */
void debug_rta_oif(int lev, struct rtattr *rta)
{
    char ifname[IFNAMSIZ] = "";
    int ifindex;

    if(RTA_PAYLOAD(rta) < sizeof(ifindex)) {
        rec_dbg(lev, "RTA_OIF(%hu): -- payload too short --",
            RTA_ALIGN(rta->rta_len));
        return;
    }
    ifindex = *((int *)RTA_DATA(rta));
    if_indextoname_from_lists(ifindex, ifname);

    rec_dbg(lev, "RTA_OIF(%hu): %d(%s)",
        RTA_ALIGN(rta->rta_len), ifindex, ifname);
}

/*
 * debug attribute RTA_GATEWAY
 */
void debug_rta_gateway(int lev, struct rtmsg *rtm, struct rtattr *rta)
{
    char nexthop[INET6_ADDRSTRLEN] = "";
    int res;

    res = inet_ntop_ifa(rtm->rtm_family, rta, nexthop, sizeof(nexthop));
    if(res) {
        rec_dbg(lev, "RTA_GATEWAY(%hu): -- %s --",
            RTA_ALIGN(rta->rta_len),
            (res == 1) ? strerror(errno) : "payload too short");
        return;
    }

    rec_dbg(lev, "RTA_GATEWAY(%hu): %s", RTA_ALIGN(rta->rta_len), nexthop);
}

/*
 * debug attribute RTA_PRIORITY
 */
void debug_rta_priority(int lev, struct rtattr *rta)
{
    int priority;

    if(RTA_PAYLOAD(rta) < sizeof(priority)) {
        rec_dbg(lev, "RTA_PRIORITY(%hu): -- payload too short --",
            RTA_ALIGN(rta->rta_len));
        return;
    }
    priority = *((int *)RTA_DATA(rta));

    rec_dbg(lev, "RTA_PRIORITY(%hu): %d", RTA_ALIGN(rta->rta_len), priority);
}

/*
 * debug attribute RTA_PREFSRC
 */
void debug_rta_prefsrc(int lev, struct rtmsg *rtm, struct rtattr *rta)
{
    char prefsrc[INET6_ADDRSTRLEN] = "";
    int res;

    res = inet_ntop_ifa(rtm->rtm_family, rta, prefsrc, sizeof(prefsrc));
    if(res) {
        rec_dbg(lev, "RTA_PREFSRC(%hu): -- %s --",
            RTA_ALIGN(rta->rta_len),
            (res == 1) ? strerror(errno) : "payload too short");
        return;
    }

    rec_dbg(lev, "RTA_PREFSRC(%hu): %s", RTA_ALIGN(rta->rta_len), prefsrc);
}

/*
 * debug attribute RTA_METRICS
 */
void debug_rta_metrics(int lev, struct rtattr *rta)
{
    struct rtattr *rtax[RTAX_MAX+1];

    rec_dbg(lev, "RTA_METRICS(%hu):", RTA_ALIGN(rta->rta_len));

    parse_rtattr(rtax, RTAX_MAX, RTA_DATA(rta), RTA_PAYLOAD(rta));

    if(rtax[RTAX_LOCK])
        debug_rtax_lock(lev+1, rtax[RTAX_LOCK]);

    if(rtax[RTAX_MTU])
        debug_rtax_mtu(lev+1, rtax[RTAX_MTU]);

    if(rtax[RTAX_ADVMSS])
        debug_rtax_advmss(lev+1, rtax[RTAX_ADVMSS]);

    if(rtax[RTAX_HOPLIMIT])
        debug_rtax_hoplimit(lev+1, rtax[RTAX_HOPLIMIT]);

    if(rtax[RTAX_WINDOW])
        debug_rtax_window(lev+1, rtax[RTAX_WINDOW]);
}

/*
 * debug attribute RTAX_LOCK
 */
void debug_rtax_lock(int lev, struct rtattr *rtax)
{
    if(RTA_PAYLOAD(rtax) < sizeof(unsigned)) {
        rec_dbg(lev, "RTAX_LOCK(%hu): -- payload too short --",
            RTA_ALIGN(rtax->rta_len));
        return;
    }
    rec_dbg(lev, "RTAX_LOCK(%hu): %u",
        RTA_ALIGN(rtax->rta_len), *(unsigned *)RTA_DATA(rtax));
}

/*
 * debug attribute RTAX_MTU
 */
void debug_rtax_mtu(int lev, struct rtattr *rtax)
{
    if(RTA_PAYLOAD(rtax) < sizeof(unsigned)) {
        rec_dbg(lev, "RTAX_MTU(%hu): -- payload too short --",
            RTA_ALIGN(rtax->rta_len));
        return;
    }
    rec_dbg(lev, "RTAX_MTU(%hu): %u",
        RTA_ALIGN(rtax->rta_len), *(unsigned *)RTA_DATA(rtax));
}

/*
 * debug attribute RTAX_ADVMSS
 */
void debug_rtax_advmss(int lev, struct rtattr *rtax)
{
    if(RTA_PAYLOAD(rtax) < sizeof(unsigned)) {
        rec_dbg(lev, "RTAX_ADVMSS(%hu): -- payload too short --",
            RTA_ALIGN(rtax->rta_len));
        return;
    }
    rec_dbg(lev, "RTAX_ADVMSS(%hu): %u",
        RTA_ALIGN(rtax->rta_len), *(unsigned *)RTA_DATA(rtax));
}

/*
 * debug attribute RTAX_HOPLIMIT
 */
void debug_rtax_hoplimit(int lev, struct rtattr *rtax)
{
    if(RTA_PAYLOAD(rtax) < sizeof(int)) {
        rec_dbg(lev, "RTAX_HOPLIMIT(%hu): -- payload too short --",
            RTA_ALIGN(rtax->rta_len));
        return;
    }
    rec_dbg(lev, "RTAX_HOPLIMIT(%hu): %d",
        RTA_ALIGN(rtax->rta_len), *(int *)RTA_DATA(rtax));
}

/*
 * debug attribute RTAX_WINDOW
 */
void debug_rtax_window(int lev, struct rtattr *rtax)
{
    if(RTA_PAYLOAD(rtax) < sizeof(unsigned)) {
        rec_dbg(lev, "RTAX_WINDOW(%hu): -- payload too short --",
            RTA_ALIGN(rtax->rta_len));
        return;
    }
    rec_dbg(lev, "RTAX_WINDOW(%hu): %u",
        RTA_ALIGN(rtax->rta_len), *(unsigned *)RTA_DATA(rtax));
}

/*
 * debug attribute RTA_MULTIPATH
 */
void debug_rta_multipath(int lev, struct rtmsg *rtm, struct rtattr *rta)
{
    struct rtnexthop *rtnh;
    int rtnh_len = RTA_PAYLOAD(rta);
    struct rtattr *rtnha[__RTA_MAX];
    char ifname[IFNAMSIZ] = "";
    char flags_list[MAX_STR_SIZE] = "";

    if(RTA_PAYLOAD(rta) < sizeof(*rtnh)) {
        rec_dbg(lev, "RTA_MULTIPATH(%hu): -- payload too short --",
            RTA_ALIGN(rta->rta_len));
        return;
    }
    rtnh = RTA_DATA(rta);

    rec_dbg(lev, "RTA_MULTIPATH(%hu):", RTA_ALIGN(rta->rta_len));

    for(; RTNH_OK(rtnh, rtnh_len);
        rtnh = RTNH_NEXT(rtnh), rtnh_len -= RTNH_ALIGN(rtnh->rtnh_len)) {
        convert_rtnh_flags(rtnh->rtnh_flags, flags_list, sizeof(flags_list));
        if_indextoname_from_lists(rtnh->rtnh_ifindex, ifname);

        rec_dbg(lev, "    [ rtnexthop(%d) ]", sizeof(*rtnh));
        rec_dbg(lev, "        rtnh_len(%d): %hu",
            sizeof(rtnh->rtnh_len), rtnh->rtnh_len);
        rec_dbg(lev, "        rtnh_flags(%d): %d(%s)",
            sizeof(rtnh->rtnh_flags), rtnh->rtnh_flags, flags_list);
        rec_dbg(lev, "        rtnh_hops(%d): %d",
            sizeof(rtnh->rtnh_hops), rtnh->rtnh_hops);
        rec_dbg(lev, "        rtnh_ifindex(%d): %d(%s)",
            sizeof(rtnh->rtnh_ifindex), rtnh->rtnh_ifindex, ifname);

        parse_rtattr(rtnha, RTA_MAX, RTNH_DATA(rtnh), rtnh->rtnh_len - sizeof(*rtnh));

        if(rtnha[RTA_GATEWAY])
            debug_rta_gateway(lev+3, rtm, rtnha[RTA_GATEWAY]);
    }
}

/*
 * debug attribute RTA_FLOW
 */
void debug_rta_flow(int lev, struct rtattr *rta)
{
    if(RTA_PAYLOAD(rta) < sizeof(unsigned)) {
        rec_dbg(lev, "RTA_FLOW(%hu): -- payload too short --",
            RTA_ALIGN(rta->rta_len));
        return;
    }
    rec_dbg(lev, "RTA_FLOW(%hu): %u",
        RTA_ALIGN(rta->rta_len), *((unsigned *)RTA_DATA(rta)));
}

/*
 * debug attribute RTA_CACHEINFO
 */
void debug_rta_cacheinfo(int lev, struct rtattr *rta)
{
    struct rta_cacheinfo *rtac;

    if(RTA_PAYLOAD(rta) < sizeof(*rtac)) {
        rec_dbg(lev, "RTA_CACHEINFO(%hu): -- payload too short --",
            RTA_ALIGN(rta->rta_len));
        return;
    }
    rtac = (struct rta_cacheinfo *)RTA_DATA(rta);

    rec_dbg(lev, "RTA_CACHEINFO(%hu):", RTA_ALIGN(rta->rta_len));
    rec_dbg(lev, "    [ rta_cacheinfo(%d) ]", sizeof(*rtac));
    rec_dbg(lev, "        rta_clntref(%d): %u",
        sizeof(rtac->rta_clntref), rtac->rta_clntref);
    rec_dbg(lev, "        rta_lastuse(%d): %u",
        sizeof(rtac->rta_lastuse), rtac->rta_lastuse);
    rec_dbg(lev, "        rta_expires(%d): %d",
        sizeof(rtac->rta_expires), rtac->rta_expires);
    rec_dbg(lev, "        rta_error(%d): %u",
        sizeof(rtac->rta_error), rtac->rta_error);
    rec_dbg(lev, "        rta_used(%d): %u",
        sizeof(rtac->rta_used), rtac->rta_used);
    rec_dbg(lev, "        rta_id(%d): %u",
        sizeof(rtac->rta_id), rtac->rta_id);
    rec_dbg(lev, "        rta_ts(%d): %u",
        sizeof(rtac->rta_ts), rtac->rta_ts);
    rec_dbg(lev, "        rta_tsage(%d): %u",
        sizeof(rtac->rta_tsage), rtac->rta_tsage);
}

/*
 * debug attribute RTA_TABLE
 */
void debug_rta_table(int lev, struct rtattr *rta)
{
    if(RTA_PAYLOAD(rta) < sizeof(int)) {
        rec_dbg(lev, "RTA_TABLE(%hu): -- payload too short --",
            RTA_ALIGN(rta->rta_len));
        return;
    }
    rec_dbg(lev, "RTA_TABLE(%hu): %d(%s)",
        RTA_ALIGN(rta->rta_len), *(int *)RTA_DATA(rta),
        convert_rt_table(*(int *)RTA_DATA(rta), 1));
}

#if HAVE_DECL_RTA_MARK
/*
 * debug attribute RTA_MARK
 */
void debug_rta_mark(int lev, struct rtattr *rta)
{
    if(RTA_PAYLOAD(rta) < sizeof(unsigned)) {
        rec_dbg(lev, "RTA_MARK(%hu): -- payload too short --",
            RTA_ALIGN(rta->rta_len));
        return;
    }
    rec_dbg(lev, "RTA_MARK(%hu): %u",
        RTA_ALIGN(rta->rta_len), *(unsigned *)RTA_DATA(rta));
}
#endif

/*
 * convert rtm_table from number to string
 */
const char *convert_rt_table(int table, int debug)
{
#define _RT_TABLE(s1, s2) \
    if(table == RT_TABLE_##s1) \
        return(debug ? #s1 : #s2);
    _RT_TABLE(UNSPEC, none);
#if HAVE_DECL_RT_TABLE_COMPAT
    _RT_TABLE(COMPAT, compat);
#endif
    _RT_TABLE(DEFAULT, default);
    _RT_TABLE(MAIN, main);
    _RT_TABLE(LOCAL, local);
#undef _RT_TABLE
    return(debug ? "UNKNOWN" : "unknown");
}

/*
 * convert rtm_protocol from number to string
 */
const char *convert_rtprot(int protocol, int debug)
{
#define _RTM_PROTOCOL(s1, s2) \
    if(protocol == RTPROT_##s1) \
        return(debug ? #s1: #s2);
    _RTM_PROTOCOL(UNSPEC, none);
    _RTM_PROTOCOL(REDIRECT, redirect);
    _RTM_PROTOCOL(KERNEL, kernel);
    _RTM_PROTOCOL(BOOT, boot);
    _RTM_PROTOCOL(STATIC, static);
    _RTM_PROTOCOL(GATED, gated);
    _RTM_PROTOCOL(RA, ra);
    _RTM_PROTOCOL(MRT, ra);
    _RTM_PROTOCOL(ZEBRA, zebra);
    _RTM_PROTOCOL(BIRD, bird);
    _RTM_PROTOCOL(DNROUTED, dnrouted);
    _RTM_PROTOCOL(XORP, xorp);
    _RTM_PROTOCOL(NTK, ntk);
#if HAVE_DECL_RTPROT_DHCP
    _RTM_PROTOCOL(DHCP, dhcp);
#endif
#undef _RTM_PROTOCOL
    return(debug ? "UNKNOWN" : "unknown");
}

/*
 * convert rtm_scope from number to string
 */
const char *convert_rt_scope(int scope)
{
#define _RT_SCOPE(s) \
    if(scope == RT_SCOPE_##s) \
        return(#s);
    _RT_SCOPE(UNIVERSE);
    _RT_SCOPE(SITE);
    _RT_SCOPE(LINK);
    _RT_SCOPE(HOST);
    _RT_SCOPE(NOWHERE);
#undef _RT_SCOPE
    return("UNKNOWN");
}

/*
 * convert rtm_type from number to string
 */
const char *convert_rtn_type(int type, int debug)
{
#define _RTN_TYPE(s1, s2) \
    if(type == RTN_##s1) \
        return(debug ? #s1 : #s2);
    _RTN_TYPE(UNSPEC, none);
    _RTN_TYPE(UNICAST, unicast);
    _RTN_TYPE(LOCAL, local);
    _RTN_TYPE(BROADCAST, broadcast);
    _RTN_TYPE(ANYCAST, anycast);
    _RTN_TYPE(MULTICAST, multicast);
    _RTN_TYPE(BLACKHOLE, blackhole);
    _RTN_TYPE(UNREACHABLE, unreachable);
    _RTN_TYPE(PROHIBIT, prohibit);
    _RTN_TYPE(THROW, throw);
    _RTN_TYPE(NAT, nat);
    _RTN_TYPE(XRESOLVE, external);
#undef _RTN_TYPE
    return(debug ? "UNKNOWN" : "unknown");
}

/*
 * convert rtm_flags from number to string
 */
void convert_rtm_flags(int flags, char *flags_list, int len)
{
    if(!flags) {
        strncpy(flags_list, "NONE", len);
        return;
    }
#define _RTM_FLAGS(s) \
    if((flags & RTM_F_##s) && (len - strlen(flags_list) - 1 > 0)) \
        (flags &= ~RTM_F_##s) ? \
            strncat(flags_list, #s ", ", len - strlen(flags_list) - 1) : \
            strncat(flags_list, #s, len - strlen(flags_list) - 1);
    _RTM_FLAGS(NOTIFY);
    _RTM_FLAGS(CLONED);
    _RTM_FLAGS(EQUALIZE);
    _RTM_FLAGS(PREFIX);
#undef _RTM_FLAGS
    if(!strlen(flags_list))
        strncpy(flags_list, "UNKNOWN", len);
}

/*
 * convert rtnh_flags from number to string
 */
void convert_rtnh_flags(int flags, char *flags_list, int len)
{
    if(!flags) {
        strncpy(flags_list, "NONE", len);
        return;
    }
#define _RTNH_FLAGS(s) \
    if((flags & RTNH_F_##s) && (len - strlen(flags_list) - 1 > 0)) \
        (flags &= ~RTNH_F_##s) ? \
            strncat(flags_list, #s ", ", len - strlen(flags_list) - 1) : \
            strncat(flags_list, #s, len - strlen(flags_list) - 1);
    _RTNH_FLAGS(DEAD);
    _RTNH_FLAGS(PERVASIVE);
    _RTNH_FLAGS(ONLINK);
#undef _RTNH_FLAGS
    if(!strlen(flags_list))
        strncpy(flags_list, "UNKNOWN", len);
}
