# -*- coding: utf-8 -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2004-2009 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003-2005 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import mimetools
import os

import gtk
import gobject

import ninix.home


range_scale = [(' 100%',  100),
               ('  90%',   90),
               ('  80%',   80),
               ('  70%',   70),
               ('  60%',   60),
               ('  50%',   50),
               ('  40%',   40),
               (' 200%',  200),
               (' 300%',  300),
               ('1000%', 1000)]

range_script_speed = [(_('None'), -1), # ordinal (-1: no wait)
                      (''.join(('1 (', _('Fast'), ')')), 0),
                      ('2', 1),
                      ('3', 2),
                      ('4', 3),
                      ('5', 4),
                      ('6', 6),
                      (''.join(('7 (', _('Slow'), ')')), 8)]

# default settings
DEFAULT_BALLOON_FONTS = 'Sans'

def get_default_surface_scale():
    return range_scale[0][1]

def get_default_script_speed():
    i = len(range_script_speed) / 2
    return range_script_speed[i][1]


class Preferences(dict):

    def __init__(self, filename):
        dict.__init__(self)
        self.filename = filename

    def load(self):
        try:
            f = open(self.filename)
        except IOError:
            return
        prefs = mimetools.Message(f)
        for key, value in prefs.items():
            self[key] = value
        f.close()

    def save(self):
        try:
            os.makedirs(os.path.dirname(self.filename))
        except OSError:
            pass
        f = open(self.filename, 'w')
        keys = self.keys()
        keys.sort()
        for key in keys:
            f.write('%s: %s\n' % (key, self[key]))
        f.close()

    def _get(self, name, default, conv):
        value = self.get(name)
        if value:
            try:
                return conv(value)
            except ValueError:
                pass
        return default

    def getint(self, name, default=None):
        return self._get(name, default, int)

    def getfloat(self, name, default=None):
        return self._get(name, default, float)


def new_prefs():
    filename = ninix.home.get_preferences()
    return Preferences(filename)

def load_prefs():
    prefs = new_prefs()
    prefs.load()
    return prefs


class PreferenceDialog:

    PREFS_TYPE = {'sakura_name': None,
                  'sakura_surface': int,
                  'default_balloon': None,
                  'ignore_default': int,
                  'script_speed': int,
                  'surface_scale': int,
                  'balloon_scalling': int,
                  'balloon_fonts': None,
                  'allowembryo': int,
                  'check_collision': int,
                  'use_pna': int,
                  'sink_after_talk': int,
                  'raise_before_talk': int,
                  'surface_alpha': float,
                  'balloon_alpha': float,
                  'animation_quality': float,
                  'seriko_inactive': int,
                  }

    def __init__(self, app):
        self.app = app
        self.window = gtk.Dialog()
        self.window.set_title('Preferences')
        self.window.connect('delete_event', self.cancel)
        self.notebook = gtk.Notebook()
        self.notebook.set_tab_pos(gtk.POS_TOP)
        self.window.vbox.pack_start(self.notebook)
        self.notebook.show()
        for name, constructor in [
            (_('Font'),            self.make_page_fonts),
            (_('Surface&Balloon'), self.make_page_surface_n_balloon),
            (_('Misc'),            self.make_page_misc),
            (_('Debug'),           self.make_page_debug),
            ]:
            self.notebook.append_page(constructor(),
                                      gtk.Label(unicode(name, 'utf-8')))
        box = gtk.HButtonBox()
        box.set_layout(gtk.BUTTONBOX_END)
        self.window.action_area.pack_start(box)
        box.show()
        button = gtk.Button('OK')
        button.connect('clicked', self.ok)
        box.add(button)
        button.show()
        button = gtk.Button('Apply')
        button.connect('clicked', self.apply)
        box.add(button)
        button.show()
        button = gtk.Button('Cancel')
        button.connect('clicked', self.cancel)
        box.add(button)
        button.show()

    def load(self):
        self.__prefs = load_prefs()
        self.reset()
        self.__saved_prefs = self.__prefs.copy()
        self.notify() ## FIXME
        
    def reset(self): ### FIXME ###
        self.set_balloon_fonts(self.get('balloon_fonts', DEFAULT_BALLOON_FONTS))
        name = self.get('default_balloon')
        if self.app.find_balloon_by_name(name) is not None:
            self.set_default_balloon(name)
        self.set_ignore_default(self.get('ignore_default', 0))
        self.set_surface_scale(self.get('surface_scale',
                                        get_default_surface_scale()))
        self.set_script_speed(self.get('script_speed',
                                       get_default_script_speed()))
        self.set_balloon_scalling(self.get('balloon_scalling'))
        self.set_allowembryo(self.get('allowembryo'))
        self.set_check_collision(self.get('check_collision', 0))
        self.set_use_pna(self.get('use_pna'))
        self.set_sink_after_talk(self.get('sink_after_talk'))
        self.set_surface_alpha(self.get('surface_alpha'))
        self.set_balloon_alpha(self.get('balloon_alpha'))
        self.set_raise_before_talk(self.get('raise_before_talk'))
        self.set_animation_quality(self.get('animation_quality'))
        self.set_seriko_inactive(self.get('seriko_inactive'))

    def get(self, name, default=None):
        assert name in self.PREFS_TYPE
        if self.PREFS_TYPE[name] == int:
            return self.__prefs.getint(name, default)
        elif self.PREFS_TYPE[name] == float:
            return self.__prefs.getfloat(name, default)
        else:
            return self.__prefs.get(name, default)

    def save(self): ## FIXME: check key is valid
        name, surface = self.app.get_current_sakura_name() ## FIXME
        self.__prefs['sakura_name'] = self.__saved_prefs['sakura_name'] = name
        self.__prefs['sakura_surface'] = self.__saved_prefs['sakura_surface'] = surface
        self.__prefs.save()

    def edit_preferences(self):
        self.show()

    def update(self): ## FIXME
        self.__prefs['allowembryo'] = str(1 if self.allowembryo_button.get_active() else 0)
        self.__prefs['balloon_fonts'] = self.fontsel.get_font_name()
        name = None
        selected = self.balloon_treeview.get_selection().get_selected()
        if selected:
            model, listiter = selected
            name = model.get_value(listiter, 0)
        if self.app.find_balloon_by_name(name) is not None:
            self.__prefs['default_balloon'] = name
        self.__prefs['ignore_default'] = 1 if self.ignore_button.get_active() else 0
        label, value = range_scale[self.surface_scale_combo.get_active()]
        self.__prefs['surface_scale'] = str(int(value))
        label, value = range_script_speed[self.script_speed_combo.get_active()]
        self.__prefs['script_speed'] = str(int(value))
        self.__prefs['balloon_scalling'] = str(1 if self.balloon_scalling_button.get_active() else 0)
        self.__prefs['check_collision'] = str(1 if self.check_collision_button.get_active() else 0)
        self.__prefs['use_pna'] = str(1 if self.use_pna_button.get_active() else 0)
        self.__prefs['sink_after_talk'] = str(1 if self.sink_after_talk_button.get_active() else 0)
        self.__prefs['raise_before_talk'] = str(1 if self.raise_before_talk_button.get_active() else 0)
        self.__prefs['surface_alpha'] = str(float(self.surface_alpha_adjustment.get_value()))
        self.__prefs['balloon_alpha'] = str(float(self.balloon_alpha_adjustment.get_value()))
        self.__prefs['animation_quality'] = str(float(self.animation_quality_adjustment.get_value()))
        self.__prefs['seriko_inactive'] = str(1 if self.seriko_inactive_button.get_active() else 0)

    def ok(self, widget):
        self.hide()
        self.update()
        self.__saved_prefs.update(self.__prefs)
        self.notify() ## FIXME

    def apply(self, widget):
        self.update()
        self.notify() ## FIXME

    def cancel(self, widget, event=None):
        self.hide()
        self.__prefs.update(self.__saved_prefs)
        self.reset()
        self.notify() ## FIXME
        return True

    def show(self):
        self.window.show()

    def hide(self):
        self.window.hide()

    def make_page_fonts(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        # font
        frame = gtk.Frame(unicode(_('Font(s) for balloons'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame)
        frame.show()
        self.fontsel = gtk.FontSelection()
        self.fontsel.show()
        frame.add(self.fontsel)
        page.show()
        return page

    def make_page_surface_n_balloon(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        page.show()
        frame = gtk.Frame(unicode(_('Surface Scaling'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        hbox = gtk.HBox(spacing=5)
        box.pack_start(hbox, False)
        hbox.show()
        label = gtk.Label(unicode(_('Default Setting'), 'utf-8'))
        hbox.pack_start(label, False)
        label.show()
        self.surface_scale_combo = gtk.combo_box_new_text()
        for label, value in range_scale:
            self.surface_scale_combo.append_text(label)
        hbox.pack_start(self.surface_scale_combo, False)
        self.surface_scale_combo.show()
        button = gtk.CheckButton(unicode(_('Scale Balloon'), 'utf-8'))
        self.balloon_scalling_button = button
        box.pack_start(button, False)
        button.show()
        frame = gtk.Frame(unicode(_('Default Balloon'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        scrolled = gtk.ScrolledWindow()
        scrolled.set_policy(gtk.POLICY_NEVER, gtk.POLICY_ALWAYS)
        scrolled.set_shadow_type(gtk.SHADOW_ETCHED_IN)
        box.pack_start(scrolled, True)
        scrolled.show()
        model = gtk.ListStore(gobject.TYPE_STRING)
        for desc, balloon in self.app.balloons:
            name = desc.get('name', '')
            listiter = model.append()
            model.set_value(listiter, 0, name)
        treeview = gtk.TreeView(model)
        column = gtk.TreeViewColumn(
            _('Balloon Name'), gtk.CellRendererText(), text=0)
        treeview.append_column(column)
        treeview.get_selection().set_mode(gtk.SELECTION_SINGLE)
        self.balloon_treeview = treeview
        scrolled.add(treeview)
        treeview.show()
        button = gtk.CheckButton(unicode(_('Always Use This Balloon'), 'utf-8'))
        self.ignore_button = button
        box.pack_start(button, False)
        button.show()
        frame = gtk.Frame(unicode(_('Translucency'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        button = gtk.CheckButton(unicode(_('Use PNA file'), 'utf-8'))
        self.use_pna_button = button
        box.pack_start(button, False)
        button.show()
        hbox = gtk.HBox(spacing=5)
        box.add(hbox)
        hbox.show()
        label = gtk.Label(unicode(_("Surfaces' alpha channel"), 'utf-8'))
        hbox.pack_start(label, False)
        label.show()
        self.surface_alpha_adjustment = gtk.Adjustment(1.0, 0.1, 1.0, 0.1, 0.5)
        button = gtk.SpinButton(self.surface_alpha_adjustment, 0.2, 1)
        hbox.pack_start(button, False)
        button.show()
        hbox = gtk.HBox(spacing=5)
        box.add(hbox)
        hbox.show()
        label = gtk.Label(unicode(_("Balloons' alpha channel"), 'utf-8'))
        hbox.pack_start(label, False)
        label.show()
        self.balloon_alpha_adjustment = gtk.Adjustment(1.0, 0.1, 1.0, 0.1, 0.5)
        button = gtk.SpinButton(self.balloon_alpha_adjustment, 0.2, 1)
        hbox.pack_start(button, False)
        button.show()
        frame = gtk.Frame(unicode(_('Animation'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        hbox = gtk.HBox(spacing=5)
        box.add(hbox)
        hbox.show()
        label = gtk.Label(unicode(_("Quality"), 'utf-8'))
        hbox.pack_start(label, False)
        label.show()        
        self.animation_quality_adjustment = gtk.Adjustment(1.0, 0.1, 1.0, 0.1, 0.1)
        button = gtk.SpinButton(self.animation_quality_adjustment, 0.2, 1)
        hbox.pack_start(button, False)
        button.show()
        hbox.show()
        button = gtk.CheckButton(unicode(_('SERIKO INACTIVE'), 'utf-8'))
        self.seriko_inactive_button = button
        box.pack_start(button, False)
        button.show()
        return page        

    def make_page_misc(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        page.show()
        frame = gtk.Frame(unicode(_('SSTP Setting'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        button = gtk.CheckButton(unicode(_('Allowembryo'), 'utf-8'))
        self.allowembryo_button = button
        frame.add(button)
        button.show()
        frame = gtk.Frame(unicode(_('Script Wait'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        hbox = gtk.HBox(spacing=5)
        frame.add(hbox)
        hbox.show()
        label = gtk.Label(unicode(_('Default Setting'), 'utf-8'))
        hbox.pack_start(label, False)
        label.show()
        self.script_speed_combo = gtk.combo_box_new_text()
        for label, value in range_script_speed:
            self.script_speed_combo.append_text(label)
        hbox.pack_start(self.script_speed_combo, False)
        self.script_speed_combo.show()
        frame = gtk.Frame(unicode(_('Raise & Lower'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        button = gtk.CheckButton(unicode(_('Sink after Talk'), 'utf-8'))
        self.sink_after_talk_button = button
        box.pack_start(button, False)
        button.show()
        button = gtk.CheckButton(unicode(_('Raise before Talk'), 'utf-8'))
        self.raise_before_talk_button = button
        box.pack_start(button, False)
        button.show()
        return page

    def make_page_debug(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        page.show()
        frame = gtk.Frame(unicode(_('Surface Debugging'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        button = gtk.CheckButton(unicode(_('Display Collision Area'), 'utf-8'))
        self.check_collision_button = button
        frame.add(button)
        button.show()
        return page

    def notify(self): ## FIXME
        for sakura in self.app.get_working_ghost():
            sakura.reset_balloon_fonts() ## FIXME
            flag = self.get('balloon_scalling')
            sakura.set_balloon_scalling(flag) ## FIXME
            scale = self.get('surface_scale')
            sakura.set_surface_scale(scale) ## FIXME
            flag = self.get('use_pna')
            sakura.set_use_pna(flag) ## FIXME
            alpha = self.get('surface_alpha')
            sakura.set_surface_alpha(alpha) ## FIXME
            alpha = self.get('balloon_alpha')
            sakura.set_balloon_alpha(alpha) ## FIXME
            quality = self.get('animation_quality')
            sakura.set_animation_quality(quality) ## FIXME
            flag = self.get('seriko_inactive')
            sakura.set_seriko_inactive(flag) ## FIXME

    def set_balloon_fonts(self, name):
        self.fontsel.set_font_name(name)

    def set_balloon_scalling(self, flag):
        self.balloon_scalling_button.set_active(True if flag else False)

    def set_script_speed(self, speed):
        index = 0
        for label, value in range_script_speed:
            if speed == value:
                self.script_speed_combo.set_active(index)
                break
            index += 1

    def set_surface_scale(self, scale):
        index = 0
        for label, value in range_scale:
            if scale == value:
                self.surface_scale_combo.set_active(index)
                break
            index += 1

    def set_default_balloon(self, name):
        model = self.balloon_treeview.get_model()
        listiter = model.get_iter_first()
        while listiter is not None:
            value = model.get_value(listiter, 0)
            if value == name or name is None:
                self.balloon_treeview.get_selection().select_iter(listiter)
                break
            listiter = model.iter_next(listiter) 
        else:
            listiter = model.get_iter_first()
            assert listiter is not None
            self.balloon_treeview.get_selection().select_iter(listiter)

    def set_ignore_default(self, flag):
        self.ignore_button.set_active(True if flag else False)

    def set_check_collision(self, flag):
        self.check_collision_button.set_active(True if flag else False)

    def set_allowembryo(self, flag):
        self.allowembryo_button.set_active(True if flag else False)

    def set_use_pna(self, flag):
        self.use_pna_button.set_active(True if flag else False)

    def set_sink_after_talk(self, flag):
        self.sink_after_talk_button.set_active(True if flag else False)

    def set_raise_before_talk(self, flag):
        self.raise_before_talk_button.set_active(True if flag else False)

    def set_surface_alpha(self, alpha):
        self.surface_alpha_adjustment.set_value(1.0 if alpha is None else alpha)

    def set_balloon_alpha(self, alpha):
        self.balloon_alpha_adjustment.set_value(1.0 if alpha is None else alpha)

    def set_animation_quality(self, quality):
        self.animation_quality_adjustment.set_value(1.0 if quality is None else quality)

    def set_seriko_inactive(self, flag):
        self.seriko_inactive_button.set_active(True if flag else False)
