# -*- coding: utf-8 -*-
#
#  hanayu.py - a "花柚" compatible Saori module for ninix
#  Copyright (C) 2002-2011 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

# TODO:
# - usetime, line, radar 以外の形式のグラフへの対応.


import sys
import os
import math
import time

if 'DISPLAY' in os.environ:
    if 'gtk' not in sys.modules:
        try:
            import pygtk
            pygtk.require('2.0')
        except ImportError:
            pass
    import gtk
    import pango
    import cairo
    import ninix.pix
else:
    gtk = None

from ninix.dll import SAORI


class Saori(SAORI):

    __DBNAME = 'HANAYU.db'

    def __init__(self):
        SAORI.__init__(self)
        self.graphs = {}
        self.data = {}

    def check_import(self):
        return 0 if not gtk else 1

    def setup(self):
        self.dbpath = os.path.join(self.dir, self.__DBNAME)
        self.graphs = {}
        self.data = self.read_hanayu_txt(self.dir)
        if self.data:
            self.read_db()
            return 1
        else:
            return 0

    def read_hanayu_txt(self, dir):
        graphs = {}
        try:
            f = open(os.path.join(dir, 'hanayu.txt'), 'r')
            data = {}
            name = ''
            tmp_name = ''
            for line in f:
                line = line.strip()
                if not line:
                    continue
                if line.startswith('//'):
                    continue
                if line.startswith('['): # bln.txt like format
                    graphs[name] = data
                    data = {}
                    end = line.find(']')
                    if end < 0:
                        end = len(line)
                    name = line[1:end]
                elif line == '{': # surfaces.txt like format
                    graphs[name] = data
                    data = {}
                    name = tmp_name ## FIXME
                elif line == '}': # surfaces.txt like format
                    graphs[name] = data
                    data = {}
                    name = ''
                elif ',' in line:
                    key, value = [x.strip() for x in line.split(',', 1)]
                    data[key] = value
                elif not name:
                    tmp_name = line
            if data:
                graphs[name] = data
            return graphs
        except:
            return None

    def finalize(self):
        for name in self.graphs.keys():
            self.graphs[name].destroy()
        self.graphs = {}
        self.data = {}
        self.write_db()
        return 1

    def time_to_key(self, secs, offset):
        year = int(time.strftime('%Y', time.localtime(secs)))
        month = int(time.strftime('%m', time.localtime(secs)))
        day = int(time.strftime('%d', time.localtime(secs)))
        target_time = time.mktime(
            (year, month, day + offset, 0, 0, 0, -1, -1, -1))
        year = int(time.strftime('%Y', time.localtime(target_time)))
        month = int(time.strftime('%m', time.localtime(target_time)))
        day = int(time.strftime('%d', time.localtime(target_time)))
        key = str(year * 10000 + month * 100 + day)
        return key, year, month, day

    def read_db(self):
        self.seven_days = []
        current_time = self.last_update = time.time()
        for index in range(-6, 1):
            key, year, month, day = self.time_to_key(current_time, index)
            self.seven_days.append([key, year, month, day, 0.0])
        try:
            f = open(self.dbpath)
        except:
            return
        else:
            ver = None
            for line in f:
                line = line.strip()
                if not line:
                    continue
                if ver is None:
                    if line == '# Format: v1.0':
                        ver = 1
                    continue
                if ',' not in line:
                    continue
                key, value = line.split(',', 1)
                for index in range(7):
                    if self.seven_days[index][0] == key:
                        self.seven_days[index][4] = float(value)
            f.close()

    def update_db(self):
        current_time = time.time()
        old_seven_days = []
        old_seven_days.extend(self.seven_days)
        self.seven_days = []
        for index in range(-6, 1):
            key, year, month, day = self.time_to_key(current_time, index)
            self.seven_days.append([key, year, month, day, 0.0])
        for i in range(7):
            key = old_seven_days[i][0]
            for j in range(7):
                if self.seven_days[j][0] == key:
                    self.seven_days[j][4] = old_seven_days[i][4]
                    if j == 6:
                        self.seven_days[j][4] = self.seven_days[j][4] + \
                                                (current_time - \
                                                 self.last_update) / \
                                                 (60.0 * 60.0)
        self.last_update = current_time

    def write_db(self):
        self.update_db()
        try:
            f = open(self.dbpath, 'w')
        except IOError:
            print 'HANAYU: cannot write database (ignored)'
        else:
            f.write('# Format: v1.0\n')
            for index in range(7):
                f.write('%s, %s\n' % \
                        (self.seven_days[index][0],
                         self.seven_days[index][4]))
            f.close()

    def execute(self, argument):
        if not argument:
            return self.RESPONSE[400]
        command = argument[0]
        if command == 'show':
            if len(argument) >= 2:
                name = argument[1]
                if name in self.graphs:
                    self.graphs[name].destroy()
            else:
                name = ''
            if name not in self.data:
                return self.RESPONSE[400]
            if 'graph' in self.data[name] and \
                   self.data[name]['graph'] in ['line', 'bar',
                                                'radar', 'radar2']:
                graph_type = self.data[name]['graph']
            else:
                graph_type = 'usetime'
            if graph_type == 'usetime':
                self.update_db()
                new_args = []
                for index in range(7):
                    date = ''.join((str(self.seven_days[index][2]),
                                    '/',
                                    str(self.seven_days[index][3])))
                    new_args.append(date)
                    hours = self.seven_days[index][4]
                    new_args.append(hours)
                self.graphs[name] = Line(
                    self.dir, self.data[name], new_args, 0, 24)
            elif graph_type == 'line':
                self.graphs[name] = Line(
                    self.dir, self.data[name], argument[2:])
            elif graph_type == 'bar':
                self.graphs[name] = Bar(
                    self.dir, self.data[name], argument[2:])
            elif graph_type == 'radar':
                self.graphs[name] = Radar(
                    self.dir, self.data[name], argument[2:])
            elif graph_type == 'radar2':
                self.graphs[name] = Radar2(
                    self.dir, self.data[name], argument[2:])
        elif command == 'hide':
            if len(argument) >= 2:
                name = argument[1]
            else:
                name = ''
            if name in self.graphs:
                self.graphs[name].destroy()
            else:
                return self.RESPONSE[400]
        else:
            return self.RESPONSE[400]
        return self.RESPONSE[204]


class Graph:

    width = 450
    height = 340

    def __init__(self, dir, data, args=[], limit_min=None, limit_max=None):
        self.dir = dir
        self.data = data
        self.args = args
        self.min = limit_min
        self.max = limit_max
        self.create_window()
        self.window.show()

    def create_window(self):
        self.window = gtk.Window()
        self.window.set_title('花柚') # UTF-8
        self.window.set_decorated(False)
        self.window.set_resizable(False)
        self.window.connect('delete_event', self.delete)
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        self.x = left + scrn_w // 2
        self.y = top + scrn_h // 4
        self.window.move(self.x, self.y)
        self.darea = gtk.DrawingArea()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK|
                              gtk.gdk.BUTTON_PRESS_MASK)
        self.darea.connect('expose_event', self.redraw)
        self.darea.connect('button_press_event', self.button_press)
        self.darea.set_size_request(self.width, self.height)
        self.darea.show()
        self.window.add(self.darea)
        self.darea.realize()
        self.layout = pango.Layout(self.darea.get_pango_context())
        pixbuf = None
        if 'background.filename' in self.data:
            path = os.path.join(
                self.dir, self.data['background.filename'].replace('\\', '/'))
            
            try:
                pixbuf = ninix.pix.create_pixbuf_from_file(path, is_pnr=0)
            except:
                pixbuf = None
        self.surface_pixbuf = pixbuf

    def get_color(self, target):
        assert target in ['font', 'line', 'frame', 'bar', 'background']
        if target == 'background':
            r = g = b = 255 # white
        else:
            r = g = b = 0 # black
        name = ''.join((target, '.color'))
        if name in self.data:
            r = int(self.data[name][:2], 16)
            g = int(self.data[name][2:4], 16)
            b = int(self.data[name][4:6], 16)
        else:
            name_r = ''.join((name, '.r'))
            if name_r in self.data:
                r = int(self.data[name_r])
            name_g = ''.join((name, '.g'))
            if name_g in self.data:
                g = int(self.data[name_g])
            name_b = ''.join((name, '.b'))
            if name_b in self.data:
                b = int(self.data[name_b])
        return (r / 255., g / 255., b / 255.)

    def draw_title(self):
        if 'title' in self.data:
            self.title = unicode(self.data['title'], 'Shift_JIS', 'ignore')
        font_size = 12 # pixel
        self.font_desc = pango.FontDescription()
        self.font_desc.set_family('Sans') # FIXME
        self.font_desc.set_size(font_size * pango.SCALE)
        cr = self.darea.window.cairo_create()
        cr.set_source_rgb(*self.get_color('font'))
        w, h = self.darea.window.get_size()
        cr.translate(w, 0)
        cr.rotate(math.pi / 2.0)
        layout = cr.create_layout()
        layout.set_font_description(self.font_desc)
        context = layout.get_context()
        default_gravity = context.get_gravity() # XXX
        context.set_base_gravity(pango.GRAVITY_EAST) # Vertical Text
        layout.set_text(self.title)
        layout.set_wrap(pango.WRAP_WORD)
        tw, th = layout.get_pixel_size()
        cr.move_to(58, w - 20 - th)
        cr.show_layout(layout)
        context.set_base_gravity(default_gravity) # XXX
        del cr

    def draw_frame(self):
        pass

    def draw_graph(self):
        pass

    def redraw(self, darea, event):
        cr = darea.window.cairo_create()
        cr.set_source_rgb(*self.get_color('background'))
        cr.paint()
        if self.surface_pixbuf:
            width = self.surface_pixbuf.get_width()
            height = self.surface_pixbuf.get_height()
            xoffset = (self.width - width) // 2
            yoffset = (self.height - height) // 2
            cr.set_source_pixbuf(self.surface_pixbuf, xoffset, yoffset)
            cr.paint()
        del cr
        self.draw_title()
        self.draw_frame()
        self.draw_graph()

    def button_press(self, window, event):
        if event.type == gtk.gdk.BUTTON_PRESS:
            self.window.begin_move_drag(
                event.button, int(event.x_root), int(event.y_root),
                event.time)
        elif event.type == gtk.gdk._2BUTTON_PRESS: # double click
            self.destroy()
        return True

    def delete(self, window, event):
        return True

    def destroy(self):
        if self.window:
            self.window.destroy()
            self.window = None
            self.timeout_id = None


class Line(Graph):

    def draw_frame(self):
        frame_width = 2
        if 'frame.width' in self.data:
            frame_width = int(self.data['frame.width'])
        cr = self.darea.window.cairo_create()
        cr.set_line_width(frame_width)
        cr.set_source_rgb(*self.get_color('frame'))
        cr.move_to(60, 48)
        cr.line_to(60, 260)
        cr.line_to(420, 260)
        cr.stroke()
        del cr

    def draw_graph(self):
        cr = self.darea.window.cairo_create()
        cr.set_source_rgb(*self.get_color('font'))
        num = len(self.args) // 2
        step = 368 // num
        for index in range(num):
            self.layout.set_text(self.args[index * 2])
            w, h = self.layout.get_pixel_size()
            pos_x = 60 + index * step + step // 2 - w // 2
            pos_y = 268
            cr.move_to(pos_x, pos_y)
            cr.show_layout(self.layout)
        if self.min is not None:
            limit_min = self.min
        else:
            limit_min = self.args[1]
            for index in range(2, num):
                if self.args[index * 2] < limit_min:
                    limit_min = self.args[index * 2]
        if self.max is not None:
            limit_max = self.max
        else:
            limit_max = self.args[1]
            for index in range(2, num):
                if self.args[index * 2] > limit_max:
                    limit_max = self.args[index * 2]
        line_width = 2
        if 'line.width' in self.data:
            line_width = int(self.data['line.width'])
        cr.set_line_width(line_width)
        cr.set_source_rgb(*self.get_color('line'))
        for index in range(1, num):
            src_x = 60 + (index - 1) * step + step // 2
            src_y = 220 - int(
                168 * self.args[(index - 1) * 2 + 1] / (limit_max - limit_min))
            dst_x = 60 + index * step + step // 2
            dst_y = 220 - int(168 * self.args[index * 2 + 1] / (limit_max - limit_min))
            cr.move_to(src_x, src_y)
            cr.line_to(dst_x, dst_y)
            cr.stroke()
        for index in range(num):
            pixbuf = None
            if self.args[index * 2 + 1] == limit_min and \
               'mark0.filename' in self.data:
                path = os.path.join(
                    self.dir, self.data['mark0.filename'].replace('\\', '/'))
                if os.path.exists(path):
                    try:
                        pixbuf = ninix.pix.create_pixbuf_from_file(path)
                    except:
                        pixbuf = None
            elif self.args[index * 2 + 1] == limit_max and \
                 'mark2.filename' in self.data:
                path = os.path.join(
                    self.dir, self.data['mark2.filename'].replace('\\', '/'))
                if os.path.exists(path):
                    try:
                        pixbuf = ninix.pix.create_pixbuf_from_file(path)
                    except:
                        pixbuf = None
            elif 'mark1.filename' in self.data:
                path = os.path.join(
                    self.dir, self.data['mark1.filename'].replace('\\', '/'))
                if os.path.exists(path):
                    try:
                        pixbuf = ninix.pix.create_pixbuf_from_file(path)
                    except:
                        pixbuf = None
            if pixbuf:
                w = pixbuf.get_width()
                h = pixbuf.get_height()
                x = 60 + index * step + step // 2 - w // 2
                y = 220 - int(
                    168 * self.args[index * 2 + 1] / (limit_max - limit_min)) - h / 2
                cr.set_source_pixbuf(pixbuf, x, y)
                cr.paint()
        del cr


class Bar(Graph):

    def draw_frame(self):
        frame_width = 2
        if 'frame.width' in self.data:
            frame_width = int(self.data['frame.width'])
        cr = self.darea.window.cairo_create()
        cr.set_line_width(frame_width)
        cr.set_source_rgb(*self.get_color('frame'))
        cr.move_to(60, 48)
        cr.line_to(60, 260)
        cr.line_to(420, 260)
        cr.stroke()
        del cr

    def draw_graph(self): ## FIXME
        cr = self.darea.window.cairo_create()
        cr.set_source_rgb(*self.get_color('bar'))
        bar_with = 20 ## FIXME
        if 'bar.width' in self.data:
            bar_width = int(self.data['bar.width'])
        ### NOT YET ###


class Radar(Graph):

    width = 288
    height = 288

    def __init__(self, dir, data, args=[]):
        Graph.__init__(self, dir, data, args)

    def draw_frame(self):
        frame_width = 2
        if 'frame.width' in self.data:
            frame_width = int(self.data['frame.width'])
        cr = self.darea.window.cairo_create()
        cr.set_line_width(frame_width)
        cr.set_source_rgb(*self.get_color('frame'))
        num = len(self.args) // 2
        for index in range(num):
            x = 146 + int(math.cos(math.pi * (0.5 - 2.0 * index / num)) * 114)
            y = 146 - int(math.sin(math.pi * (0.5 - 2.0 * index / num)) * 114)
            cr.move_to(146, 146,)
            cr.line_to(x, y)
            cr.stroke()
        del cr

    def draw_graph(self):
        num = len(self.args) // 2
        for index in range(num):
            try:
                value = self.args[index * 2 + 1]
                self.args[index * 2 + 1] = float(value)
            except:
                self.args[index * 2 + 1] = 0.0
            if self.args[index * 2 + 1] < 0:
                self.args[index * 2 + 1] = 0.0
        limit_min = self.args[1]
        for index in range(num):
            if self.args[index * 2 + 1] < limit_min:
                limit_min = self.args[index * 2 + 1]
        limit_max = self.args[1]
        for index in range(num):
            if self.args[index * 2 + 1] > limit_max:
                limit_max = self.args[index * 2 + 1]
        line_width = 2
        if 'line.width' in self.data:
            line_width = int(self.data['line.width'])
        cr = self.darea.window.cairo_create()
        cr.set_line_width(line_width)
        cr.set_source_rgb(*self.get_color('line'))
        if limit_max > 0:
            value = self.args[(num - 1) * 2 + 1] / limit_max
        else:
            value = 1.0
        src_x = 146 + int(math.cos(
            math.pi * (0.5 - 2.0 * (num - 1) / num)) * value * 100)
        src_y = 146 - int(math.sin(
            math.pi * (0.5 - 2.0 * (num - 1) / num)) * value * 100)
        cr.move_to(src_x, src_y)
        for index in range(num):
            if limit_max > 0:
                value = self.args[index * 2 + 1] / limit_max
            else:
                value = 1.0
            dst_x = 146 + int(
                math.cos(math.pi * (0.5 - 2.0 * index / num)) * value * 100)
            dst_y = 146 - int(
                math.sin(math.pi * (0.5 - 2.0 * index / num)) * value * 100)
            cr.line_to(dst_x, dst_y)
        cr.stroke()
        font_size = 9 # pixel
        self.font_desc.set_size(font_size * pango.SCALE)
        self.layout.set_font_description(self.font_desc)
        cr.set_source_rgb(*self.get_color('font'))
        for index in range(num):
            ##if limit_max > 0:
            ##    value = self.args[index * 2 + 1] / limit_max
            ##else:
            ##    value = 1.0
            value = 1.2 # XXX
            x = 146 + int(math.cos(
                math.pi * (0.5 - 2.0 * index / num)) * value * 100)
            y = 146 - int(math.sin(
                math.pi * (0.5 - 2.0 * index / num)) * value * 100)
            self.layout.set_text(self.args[index * 2])
            w, h = self.layout.get_pixel_size()
            x -= w // 2
            y -= h // 2
            cr.move_to(x, y)
            cr.show_layout(self.layout)
        del cr

class Radar2(Graph):

    width = 288
    height = 288

    def __init__(self, dir, data, args=[]):
        Graph.__init__(self, dir, data, args)


if __name__ == '__main__':
    USAGE= 'Usage: hanayu.py <dir>'
    if len(sys.argv) == 2:
        saori = Saori()
        data = saori.read_hanayu_txt(sys.argv[1])
        print data
    else:
        print USAGE
