/* Asynchronous repaint stuff for imagedisplay
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

/* Clear all bits we will paint, not just out-of-image ... makes it clear
 * which bits we are painting if there are pauses.
#define DEBUG_PAINT
 */

#include "ip.h"

/* List of Imagedisplay for which there are background repaints pending.
 */
static GSList *imagedisplay_update_list = NULL;

/* Our idle process for image repaints.
 */
static gint imagedisplay_idle_running = 0;

/* The timer we use to check for too long spent repainting.
 */
static GTimer *imagedisplay_paint_timer = NULL;

/* The last value we saw from the elapsed time.
 */
static gdouble imagedisplay_elapsed_time = 0.0;

/* The maximum amount of time we spend repainting.
 */
static const gdouble imagedisplay_max_time = 0.2;

/* The sheight adjust threshold ... if we take longer than this on a single
 * repaint call, make repaint strips smaller, else make them larger.
 */
static const gdouble imagedisplay_adjust_thresh = 0.05;

/* Range of values strip height moves between.
 */
static const int imagedisplay_max_strip_height = 20;
static const int imagedisplay_min_strip_height = 1;

/* Fewer pixels than this in an expose, don't bother decomposing.
 */
static const int imagedisplay_max_nodecompose_pixels = 1000;

/* How narrow does something have to be before being called a narrow strip?
 */
static const int imagedisplay_max_nostrip_width = 20;

/* Paint an area with the background pattern.
 */
static void
imagedisplay_paint_background( Imagedisplay *id, GdkGC *gc, Rect *area )
{
	if( !GTK_WIDGET( id )->window )
		return;

	if( im_rect_isempty( area ) )
		return;

	gdk_gc_set_ts_origin( gc, 
		-id->visible.left, -id->visible.top );
	gdk_draw_rectangle( GTK_WIDGET( id )->window, 
		gc, TRUE,
		area->left - id->visible.left + id->screen.left, 
		area->top - id->visible.top + id->screen.top,
		area->width, area->height );
}

/* Repaint an area of the image.
 */
static void
imagedisplay_repaint_area( Imagedisplay *id, Rect *expose )
{
	Conversion *conv = id->conv;

	guchar *buf;
	int lsk;

	if( !GTK_WIDGET( id )->window )
		return;
	if( !GTK_WIDGET_VISIBLE( id ) )
		return;
	if( conv->err ) {
		imagedisplay_paint_background( id, id->err_gc, expose );
		return;
	}

	/* Generate pixels we need.
	 */
#ifdef DEBUG
	printf( "imagedisplay_repaint_area: requesting %dx%d pixels from "
		"VIPS\n", expose->width, expose->height );
#endif /*DEBUG*/
	if( im_prepare_thread( conv->tg, conv->ireg, expose ) ) {
		verrors( "unable to paint display" );
		conversion_error_set( conv );
		imagedisplay_paint_background( id, id->err_gc, expose );
		return;
	}
#ifdef DEBUG
	printf( "imagedisplay_repaint_area: back from VIPS\n" );
#endif /*DEBUG*/

        /* Find a pointer to the start of the data, plus a line skip value.
         */
        buf = (guchar *) IM_REGION_ADDR( conv->ireg, 
		expose->left, expose->top );
        lsk = IM_REGION_LSKIP( conv->ireg );

	/* Paint into window.
 	 */
	if( conv->ireg->im->Bands == 3 )
		gdk_draw_rgb_image( GTK_WIDGET( id )->window,
			GTK_WIDGET( id )->style->white_gc,
			expose->left - id->visible.left + id->screen.left, 
			expose->top - id->visible.top + id->screen.top,
			expose->width, expose->height,
			GDK_RGB_DITHER_MAX,
			buf, lsk );
	else if( conv->ireg->im->Bands == 1 )
		gdk_draw_gray_image( GTK_WIDGET( id )->window,
			GTK_WIDGET( id )->style->white_gc,
			expose->left - id->visible.left + id->screen.left, 
			expose->top - id->visible.top + id->screen.top,
			expose->width, expose->height,
			GDK_RGB_DITHER_MAX,
			buf, lsk );

	/* Tell our clients we have just repainted a section.
	 */
	imagedisplay_emit_repaint( id, expose );
}

/* Paint areas outside the image.
 */
static void
imagedisplay_paint_background_clipped( Imagedisplay *id, Rect *expose )
{
#ifdef DEBUG_PAINT
	imagedisplay_paint_background( id, id->back_gc, expose );
#else
	Conversion *conv = id->conv;
	Rect area, clipped;

	/* Any stuff to the right of the canvas?
	 */
	area.left = conv->canvas.width;
	area.top = id->visible.top;
	area.width = IM_MAX( 0, 
		id->visible.left + id->screen.width - conv->canvas.width );
	area.height = id->screen.height;
	im_rect_intersectrect( expose, &area, &clipped );
	imagedisplay_paint_background( id, id->back_gc, &clipped );

	/* Any stuff below the canvas?
	 */
	area.left = id->visible.left;
	area.top = conv->canvas.height;
	area.width = IM_MAX( 0, conv->canvas.width - id->visible.left );
	area.height = IM_MAX( 0, 
		id->visible.top + id->screen.height - conv->canvas.height );
	im_rect_intersectrect( expose, &area, &clipped );
	imagedisplay_paint_background( id, id->back_gc, &clipped );
#endif /*DEBUG_PAINT*/
}

/* Repaint a rect, clipping against visible and image. 
 */
static void 
imagedisplay_repaint_clip( Imagedisplay *id, Rect *expose )
{
	Conversion *conv = id->conv;

	Rect clip;
	Rect clip2;

	/* Clip as much as we can.
	 */
	im_rect_intersectrect( &id->visible, expose, &clip );
	im_rect_intersectrect( &conv->canvas, &clip, &clip2 );

	if( im_rect_isempty( &clip2 ) ) 
		/* Clipped to nothing! D'oh.
		 */
		return;

	/* Do a paint.
	 */
	imagedisplay_repaint_area( id, &clip2 );

	/* We've been busy ... refresh the timer.
	 */
	imagedisplay_elapsed_time = 
		g_timer_elapsed( imagedisplay_paint_timer, NULL );
}

/* Repaint expose in strips. Remove pixels from expose as we paint them.
 */
static void
imagedisplay_repaint_strips( Imagedisplay *id, Rect *expose )
{
	Conversion *conv = id->conv;

	/* No image installed? Do nothing.
	 */
	if( !conv->ii ) {
		expose->height = 0;
		return;
	}

	/* Is it a small or narrow rect? Don't bother decomposing.
	 */
	if( expose->width * expose->height < 
		imagedisplay_max_nodecompose_pixels ||
		expose->width < imagedisplay_max_nostrip_width ) {
		imagedisplay_repaint_clip( id, expose );
		expose->width = expose->height = 0;
		return;
	}

	/* It's a big rect ... we repaint in strips of sheight rows.
	 */
	while( !im_rect_isempty( expose ) && 
		imagedisplay_elapsed_time < imagedisplay_max_time ) {
		Rect sub;
		gdouble start = imagedisplay_elapsed_time;

		/* Chop a bit off the top.
		 */
		sub = *expose;
		sub.height = IM_MIN( id->sheight, sub.height );
		expose->top += id->sheight;
		expose->height = IM_MAX( 0, expose->height - id->sheight );

		/* And repaint that.
		 */
		imagedisplay_repaint_clip( id, &sub );

		/* Adjust strip height if we took too long.
		 */
		if( imagedisplay_elapsed_time - start > 
			imagedisplay_adjust_thresh ) 
			id->sheight = IM_MAX( imagedisplay_min_strip_height, 
				id->sheight - 1 );
		else 
			id->sheight = IM_MIN( imagedisplay_max_strip_height, 
				id->sheight + 1 );
	}
}

/* Clear the pending repaints on an image.
 */
static void
imagedisplay_clear_pending( Imagedisplay *id )
{
#ifdef DEBUG
	printf( "flushing repaint buffer\n" );
#endif /*DEBUG*/
	slist_map( id->pending,
		(SListMapFn) rect_free, NULL );
	g_slist_free( id->pending );
	id->pending = NULL;
}

/* Make sure a display is not on the update list.
 */
void
imagedisplay_repaint_remove( Imagedisplay *id )
{
	if( id->update ) {
#ifdef DEBUG
		printf( "imagedisplay_repaint_remove\n" );
#endif /*DEBUG*/

		imagedisplay_update_list = 
			g_slist_remove( imagedisplay_update_list, id );
		id->update = FALSE;

		imagedisplay_clear_pending( id );
	}
}

/* Make sure a display is on the update list.
 */
static void
imagedisplay_repaint_add( Imagedisplay *id )
{
	if( !id->update ) {
#ifdef DEBUG
		printf( "imagedisplay_repaint_add\n" );
#endif /*DEBUG*/

		imagedisplay_update_list = 
			g_slist_prepend( imagedisplay_update_list, id );
		id->update = TRUE;
	}
}

/* Our idle process.
 */
static gint
imagedisplay_idle_repaint( gpointer user_data )
{
	Imagedisplay *id;
	Rect *expose;

	/* Make sure we have a timer, and reset it.
	 */
	if( !imagedisplay_paint_timer )
		imagedisplay_paint_timer = g_timer_new();
	g_timer_reset( imagedisplay_paint_timer );
	imagedisplay_elapsed_time = 0.0;

	/* Loop, looking for repaint candidates.
	 */
	while( imagedisplay_update_list && 
		imagedisplay_elapsed_time < imagedisplay_max_time ) {
		/* First display candidate.
 		 */
		id = IMAGEDISPLAY( imagedisplay_update_list->data );

		/* Loop for this imagedisplay.
 		 */
		while( id->pending && 
			imagedisplay_elapsed_time < imagedisplay_max_time ) {
			/* First expose candidate.
			 */
			expose = (Rect *) id->pending->data;

			/* Paint it in strips. 
			 */
			imagedisplay_repaint_strips( id, expose );

			/* If finished this expose, remove from list.
			 */
			if( im_rect_isempty( expose ) ) {
#ifdef DEBUG
				printf( "finished expose block\n" );
#endif /*DEBUG*/
				id->pending = g_slist_remove( 
					id->pending, (gpointer) expose );
				rect_free( expose );
			}
		}

		/* Anything left?
		 */
		if( !id->pending ) 
			imagedisplay_repaint_remove( id );
	}

#ifdef DEBUG
	printf( "repaint idle done after %g secs\n", 
		imagedisplay_elapsed_time );
#endif /*DEBUG*/

	/* Any repaints left?
	 */
	if( !imagedisplay_update_list ) {
#ifdef DEBUG
		printf( "stopping background repaint process\n" );
#endif /*DEBUG*/
		imagedisplay_idle_running = 0;
		return( FALSE );
	}

	return( TRUE );
}

/* old is an existing pending repaint, new is a possible new repaint. If new
 * fits inside old, don't bother with new. If old fits inside new, replace
 * old with new. 
 */
static gpointer
imagedisplay_test_existing( gpointer old, gpointer new )
{
	Rect *oldr = (Rect *) old;
	Rect *newr = (Rect *) new;

	if( im_rect_includesrect( oldr, newr ) ) {
#ifdef DEBUG
		printf( "expose area already pending, junking\n" );
#endif /*DEBUG*/
		/* new is inside old. Do nothing.
		 */
		return( old );
	}
	else if( im_rect_includesrect( newr, oldr ) ) {
#ifdef DEBUG
		printf( "expose area extends existing expose\n" );
#endif /*DEBUG*/
		/* old is inside new. Update old.
		 */
		*oldr = *newr;

		return( old );
	}

	return( NULL );
}

/* A new expose has come in. Add to pending list for this imagedisplay.
 */
void
imagedisplay_repaint( Imagedisplay *id, Rect *expose )
{
	Rect screen, clip;

	/* Iconised, or hidden?
	 */
	if( !GTK_WIDGET_DRAWABLE( GTK_WIDGET( id ) ) ) 
		return;

	/* Clip against screen.
	 */
	screen = id->screen;
	screen.left = id->visible.left;
	screen.top = id->visible.top;
	im_rect_intersectrect( &screen, expose, &clip );
	if( im_rect_isempty( &clip ) )
		return;

	/* Clear to background. Always do this, to make sure we paint outside
	 * the image area.
 	 */
	imagedisplay_paint_background_clipped( id, &clip );

	/* Is it a rect we already have, or can patch? 
	 */
	if( slist_map( id->pending, imagedisplay_test_existing, &clip ) )
		return;

	/* Tell our clients we are about to repaint.
	 */
	imagedisplay_emit_expose_start( id, &clip );

	/* Add as new pending.
 	 */
#ifdef DEBUG
	printf( "adding new pending expose: at %dx%d size %dx%d\n",
		clip.left, clip.top, clip.width, clip.height );
#endif /*DEBUG*/
	id->pending = g_slist_append( id->pending, rect_dup( &clip ) );

	/* Make sure this display is on the list of displays with pending
	 * repaints.
	 */
	imagedisplay_repaint_add( id );

	/* Make sure the background paint process is running.
 	 */
	if( !imagedisplay_idle_running ) {
#ifdef DEBUG
		printf( "starting background repaint process\n" );
#endif /*DEBUG*/
		imagedisplay_idle_running = 
			gtk_idle_add( imagedisplay_idle_repaint, NULL );
	}
}

/* Do a complete repaint, junking any pending paints.
 */
void
imagedisplay_repaint_all( Imagedisplay *id )
{
	Rect all;

	imagedisplay_clear_pending( id );
	all = id->visible;
	all.width = id->screen.width;
	all.height = id->screen.height;
	imagedisplay_repaint( id, &all );
}
