/* like a heapmodel, but we represent a class in the heap
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#include "ip.h"

/*
#define DEBUG
 */

static HeapmodelClass *parent_class = NULL;

void *
classmodel_get_instance( Classmodel *classmodel )
{
	ClassmodelClass *klass = 
		CLASSMODEL_CLASS( GTK_OBJECT( classmodel )->klass );

	if( klass->get_instance )
		return( klass->get_instance( classmodel ) );

	return( NULL );
}

/* Save graphic dialog.
 */
void
classmodel_graphic_save( GtkWidget *parent, Classmodel *classmodel )
{
	ClassmodelClass *klass = 
		CLASSMODEL_CLASS( GTK_OBJECT( classmodel )->klass );

	if( !klass->graphic_save ) {
		ierrors( "graphic_save not implemented for class %s", 
			OBJECT_CLASS_NAME( classmodel ) );
		box_alert( parent );
	}

	klass->graphic_save( parent, classmodel );
}

/* Replace graphic dialog.
 */
void
classmodel_graphic_replace( GtkWidget *parent, Classmodel *classmodel )
{
	ClassmodelClass *klass = 
		CLASSMODEL_CLASS( GTK_OBJECT( classmodel )->klass );

	if( !klass->graphic_replace ) {
		ierrors( "graphic_replace not implemented for class %s", 
			OBJECT_CLASS_NAME( classmodel ) );
		box_alert( parent );
	}

	klass->graphic_replace( parent, classmodel );
}

/* Make and break links between classmodels and the iimages displaying them.
 */
static void 
classmodel_iimage_link( Classmodel *classmodel, Iimage *iimage )
{
	if( !g_slist_find( classmodel->iimages, iimage ) ) {
#ifdef DEBUG
		printf( "classmodel_iimage_link: linking " );
		row_name_print( HEAPMODEL( classmodel )->row );
		printf( " to " );
		row_name_print( HEAPMODEL( iimage )->row );
		printf( "\n" );
#endif /*DEBUG*/

		iimage->classmodels = 
			g_slist_prepend( iimage->classmodels, classmodel );
		classmodel->iimages = 
			g_slist_prepend( classmodel->iimages, iimage );
	}
}

void *
classmodel_iimage_unlink( Classmodel *classmodel, Iimage *iimage )
{
	if( g_slist_find( classmodel->iimages, iimage ) ) {
#ifdef DEBUG
		printf( "classmodel_iimage_unlink: unlinking " );
		row_name_print( HEAPMODEL( classmodel )->row );
		printf( " from " );
		row_name_print( HEAPMODEL( iimage )->row );
		printf( "\n" );
#endif /*DEBUG*/

		iimage->classmodels = 
			g_slist_remove( iimage->classmodels, classmodel );
		classmodel->iimages = 
			g_slist_remove( classmodel->iimages, iimage );
	}

	return( NULL );
}

static void *
classmodel_iimage_unlink_rev( Iimage *iimage, Classmodel *classmodel )
{
	return( classmodel_iimage_unlink( classmodel, iimage ) );
}

typedef struct {
	Classmodel *classmodel;
	Imageinfo *ii;
} ClassmodelSearch;

static void *
classmodel_iimage_expr_model( Model *model, ClassmodelSearch *parms )
{
	/* Look for iimages which aren't super ... ie. if this is a class
	 * derived from Image, display on the derived class, not on the
	 * superclass.
	 */
	if( IS_IIMAGE( model ) && !is_super( HEAPMODEL( model )->row->sym ) &&
		!is_this( HEAPMODEL( model )->row->sym ) ) {
		Iimage *iimage = IIMAGE( model );

		if( iimage->instance.image == parms->ii ) 
			classmodel_iimage_link( parms->classmodel, iimage );
	}

	return( NULL );
}

/* This classmodel is defined on an Imageinfo recorded as having been the value
 * of expr ... find an associated Iimage, and link to that. 
 */
static void *
classmodel_iimage_expr( Expr *expr, ClassmodelSearch *parms )
{
	if( expr->row ) {
#ifdef DEBUG
		printf( "classmodel_iimage_expr: starting for " );
		row_name_print( expr->row );
		printf( "\n" );
#endif /*DEBUG*/

		/* Search this part of the tally for an Iimage with ii as its
		 * derived value, and link to us. 
		 */
		(void) model_map_all( MODEL( expr->row->top_row ), 
			(model_map_fn) classmodel_iimage_expr_model, parms );
	}

	return( NULL );
}

/* classmodel is defined on ii ... update all the classmodel->iimage links.
 */
void
classmodel_iimage_update( Classmodel *classmodel, Imageinfo *ii )
{
	ClassmodelSearch parms;

	parms.classmodel = classmodel;
	parms.ii = ii;
	slist_map( classmodel->iimages, 
		(SListMapFn) classmodel_iimage_unlink_rev, classmodel );

	/* Don't make links for supers/this.
	 */
	if( HEAPMODEL( classmodel )->row->sym &&
		!is_super( HEAPMODEL( classmodel )->row->sym ) &&
		!is_this( HEAPMODEL( classmodel )->row->sym ) ) {
#ifdef DEBUG
		printf( "classmodel_iimage_update: " );
		row_name_print( HEAPMODEL( classmodel )->row );
		printf( " is defined on ii \"%s\" ... searching for client "
			"displays\n", ii->im->filename );
#endif /*DEBUG*/
		slist_map( imageinfo_expr_which( ii ), 
			(SListMapFn) classmodel_iimage_expr, &parms );
	}
}

/* Trigger the class_new method for a classmodel ... look for a constructor:
 * try CLASS_edit, then if that's not defined, try CLASS. Eg.  
 *	"A1.Slider_edit" from to value
 * if Slider_edit is not defined, try
 *	"A1.Slider" from to value
 */
static gboolean
classmodel_class_instance_new( Classmodel *classmodel )
{
	ClassmodelClass *klass = CLASSMODEL_CLASS( 
		GTK_OBJECT( classmodel )->klass );
	Row *row = HEAPMODEL( classmodel )->row;
	PElement *root = &row->expr->root;
	const char *cname = MODEL( classmodel )->name;
	Reduce *rc = reduce_context;

	char cname_new[256];
	PElement fn;

	/* Find and build.
	 */
	im_snprintf( cname_new, 256, "%s_edit", cname );
	if( !class_get_member( root, cname_new, &fn ) ) {
		if( !class_get_member( root, cname, &fn ) ) 
			return( FALSE );
	}
	if( !klass->class_new( classmodel, &fn, root ) )
		return( FALSE );

	/* Reduce to base type.
	 */
	if( !reduce_pelement( rc, reduce_spine, root ) ) 
		return( FALSE );

	/* We have a new heap struct ... tell everyone to get new pointers.
	 */
	if( heapmodel_new_heap( HEAPMODEL( row ), root ) )
		return( FALSE );

	return( TRUE );
}

static void
classmodel_destroy( GtkObject *object )
{
	Classmodel *classmodel;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_CLASSMODEL( object ) );

	classmodel = CLASSMODEL( object );

	/* My instance destroy stuff.
	 */
	slist_map( classmodel->iimages, 
		(SListMapFn) classmodel_iimage_unlink_rev, classmodel );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

static void
classmodel_parent_add( Model *child, Model *parent )
{
	assert( IS_CLASSMODEL( child ) );

	MODEL_CLASS( parent_class )->parent_add( child, parent );
}

static void *
classmodel_update_model( Heapmodel *heapmodel )
{
	ClassmodelClass *klass = CLASSMODEL_CLASS( 
		GTK_OBJECT( heapmodel )->klass );

#ifdef DEBUG
	printf( "classmodel_update_model: " );
	row_name_print( heapmodel->row );
	printf( "\n" );
#endif /*DEBUG*/

	if( heapmodel->row && heapmodel->row->expr ) {
		Classmodel *classmodel = CLASSMODEL( heapmodel );
		Expr *expr = heapmodel->row->expr;

		if( !heapmodel->modified ) {
			if( !klass->class_get( classmodel, &expr->root ) )
				return( heapmodel );
		}
	}

	return( HEAPMODEL_CLASS( parent_class )->update_model( heapmodel ) );
}

static void *
classmodel_update_heap( Heapmodel *heapmodel )
{
	Classmodel *classmodel = CLASSMODEL( heapmodel );

	/* Nasty: classmodel_class_instance_new() can (indirectly) destroy us.
	 * Wrap a _ref()/_unref() pair around it to make sure we stay alive.
	 */
	gtk_object_ref( GTK_OBJECT( heapmodel ) );

	/* Build a new instance from the model.
	 */
	if( !classmodel_class_instance_new( classmodel ) ) {
		gtk_object_unref( GTK_OBJECT( heapmodel ) );
		return( heapmodel );
	}

	if( HEAPMODEL_CLASS( parent_class )->update_heap( heapmodel ) ) {
		gtk_object_unref( GTK_OBJECT( heapmodel ) );
		return( heapmodel );
	}

	gtk_object_unref( GTK_OBJECT( heapmodel ) );

	return( NULL );
}

static void *
classmodel_clear_edited( Heapmodel *heapmodel )
{
	Classmodel *classmodel = CLASSMODEL( heapmodel );

	classmodel_set_edited( classmodel, FALSE );

	return( HEAPMODEL_CLASS( parent_class )->clear_edited( heapmodel ) );
}

static gboolean
classmodel_real_class_get( Classmodel *classmodel, PElement *root )
{
	return( TRUE );
}

static void
classmodel_class_init( ClassmodelClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	ModelClass *model_class = (ModelClass *) klass;
	HeapmodelClass *heapmodel_class = (HeapmodelClass *) klass;
	ClassmodelClass *classmodel_class = (ClassmodelClass *) klass;

	parent_class = gtk_type_class( TYPE_HEAPMODEL );

	/* Init methods.
	 */
	object_class->destroy = classmodel_destroy;

	model_class->parent_add = classmodel_parent_add;

	heapmodel_class->update_model = classmodel_update_model;
	heapmodel_class->update_heap = classmodel_update_heap;
	heapmodel_class->clear_edited = classmodel_clear_edited;

	classmodel_class->get_instance = NULL;

	classmodel_class->class_get = classmodel_real_class_get;
	classmodel_class->class_new = NULL;

	classmodel_class->graphic_save = NULL;
	classmodel_class->graphic_replace = NULL;
}

static void
classmodel_init( Classmodel *classmodel )
{
	Model *model = MODEL( classmodel );

	model->display = FALSE;

        classmodel->edited = FALSE;

        classmodel->iimages = NULL;
        classmodel->views = NULL;
}

GtkType
classmodel_get_type( void )
{
	static GtkType classmodel_type = 0;

	if( !classmodel_type ) {
		static const GtkTypeInfo info = {
			"Classmodel",
			sizeof( Classmodel ),
			sizeof( ClassmodelClass ),
			(GtkClassInitFunc) classmodel_class_init,
			(GtkObjectInitFunc) classmodel_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		classmodel_type = gtk_type_unique( TYPE_HEAPMODEL, &info );
	}

	return( classmodel_type );
}

void
classmodel_set_edited( Classmodel *classmodel, gboolean edited )
{
	if( classmodel->edited != edited ) {
		Row *row = HEAPMODEL( classmodel )->row;

#ifdef DEBUG
		printf( "classmodel_set_edited: " );
		row_name_print( row );
		printf( " %s\n", bool_to_char( edited ) );
#endif /*DEBUG*/

		classmodel->edited = edited;
		model_changed( MODEL( classmodel ) );

		if( row && row->expr )
			expr_dirty( row->expr, link_serial_new() );
	}

	/* Mark eds for application.
	 */
	if( edited )
		heapmodel_set_modified( HEAPMODEL( classmodel ), TRUE );
}

/* The model has changed: mark for recomp.
 */
void
classmodel_update( Classmodel *classmodel )
{
	Row *row = HEAPMODEL( classmodel )->row;

	/* Eg. for no symol on load.
	 */
	if( !row->expr )
		return;

#ifdef DEBUG
	printf( "classmodel_update: " );
	row_name_print( HEAPMODEL( classmodel )->row );
	printf( "\n" );
#endif /*DEBUG*/

	/* classmodel_update_heap() will rebuild us on recomp.
	 */
	classmodel_set_edited( classmodel, TRUE );
	expr_dirty( row->expr, link_serial_new() );
	workspace_set_modified( row->sym );
}
