/* a view of a coloumn
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/* 
#define DEBUG
 */

/* trace pointer grab/ungrab
#define DEBUG_GRAB
 */

#include "ip.h"

#include "BITMAPS/delete.xpm"

/* Define to do true pointer grabs on column move.
 */
#define GRAB

static ViewClass *parent_class = NULL;

/* The columnview popup menu.
 */
static GtkWidget *columnview_menu = NULL;

static void
columnview_destroy( GtkObject *object )
{
	Columnview *cview;

#ifdef DEBUG
	printf( "columnview_destroy\n" );
#endif /*DEBUG*/

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_COLUMNVIEW( object ) );

	cview = COLUMNVIEW( object );

	if( cview->wview && cview->wview->front == cview )
		cview->wview->front = NULL;

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

/* Edit caption ... right button menu on title bar.
 */
static void
columnview_caption_cb( GtkWidget *wid, GtkWidget *host, Columnview *cview )
{
        /* Edit caption!
         */
        if( cview->state == COL_EDIT )
                return;

        cview->state = COL_EDIT;
        view_refresh_queue( VIEW( cview ) );
}

/* Select all objects in menu's column.
 */
/*ARGSUSED*/
static void
columnview_select_cb( GtkWidget *wid, GtkWidget *host, Columnview *cview )
{
	Column *col = COLUMN( VIEW( cview )->model );
	Workspace *ws = col->ws;

	workspace_deselect_all( ws );
        column_select_symbols( col );
}

/* Clone a column.
 */
static void
columnview_clone_cb( GtkWidget *wid, GtkWidget *host, Columnview *cview )
{
	Column *col = COLUMN( VIEW( cview )->model );
	Workspace *ws = col->ws;
	const char *newname = workspace_column_name_new( ws, NULL );
        Column *newcol = workspace_column_get( ws, newname );

	FREE( newname );
        model_set( MODEL( newcol ), NULL, MODEL( col )->caption );
        newcol->x = col->x + 30;
        newcol->y = col->y + 30;

	workspace_deselect_all( ws );
        column_select_symbols( col );
	workspace_column_select( ws, newcol );
        if( !workspace_clone_selected( ws ) )
                box_alert( NULL );
	workspace_deselect_all( ws );

        symbol_recalculate_all();
}

/* Callback from save browser.
 */
static void
columnview_save_as_sub( iWindow *iwnd, 
	void *client, iWindowNotifyFn nfn, void *sys )
{
	Filesel *filesel = FILESEL( iwnd );
	const char *filename = gtk_file_selection2_get_filename(
		GTK_FILE_SELECTION2( filesel ) );
	Column *col = COLUMN( client );
	Workspace *ws = col->ws;

	workspace_deselect_all( ws );
        column_select_symbols( col );

	if( !workspace_selected_save( ws, filename ) ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	workspace_deselect_all( ws );

	nfn( sys, IWINDOW_TRUE );
}

/* Save a column ... can't just do view_save_as_cb(), since we need to save
 * the enclosing workspace too. Hence we have to save_selected on the ws, but
 * only after we have the filename.
 */
static void
columnview_save_as_cb( GtkWidget *wid, GtkWidget *host, Columnview *cview )
{
	Column *col = COLUMN( VIEW( cview )->model );
	iWindow *iwnd = IWINDOW( view_get_toplevel( VIEW( cview ) ) );
	GtkWidget *filesel = filesel_new();

	iwindow_set_title( IWINDOW( filesel ), 
		"Save Column %s", MODEL( col )->name );
	filesel_set_flags( FILESEL( filesel ), FALSE, TRUE );
	filesel_set_type( FILESEL( filesel ), filesel_type_workspace, 0 ); 
	idialog_set_parent( IDIALOG( filesel ), GTK_WIDGET( iwnd ) );
	filesel_set_done( FILESEL( filesel ), columnview_save_as_sub, col );
	iwindow_build( IWINDOW( filesel ) );

	gtk_widget_show( GTK_WIDGET( filesel ) );
}

/* Make a name for a column menu file, based on what we're going to call the
 * menu item.
 */
static void
columnview_filename( char *file, const char *caption )
{
	int i;
	char name[PATHLEN];

	im_strncpy( name, caption, 10 );
	for( i = 0; i < (int) strlen( name ); i++ )
		if( name[i] == ' ' )
			name[i] = '_';

	for( i = 0; ; i++ ) {
		im_snprintf( file, PATHLEN, 
			"$HOME/.$PACKAGE-$VERSION/data/%s-%d.ws", name, i );
		if( !existsf( "%s", file ) )
			break;
	}
}

/* Remember the name of the last toolkit the user asked to add to.
 */
static char *columnview_to_menu_last_toolkit = NULL;

/* Done button hit.
 */
static void
columnview_to_menu_done_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	Column *col = COLUMN( client );
	Workspace *ws = col->ws;
	Stringset *ss = STRINGSET( iwnd );
	StringsetChild *name = stringset_child_get( ss, "Name" );
	StringsetChild *toolkit = stringset_child_get( ss, "Toolkit" );
	StringsetChild *file = stringset_child_get( ss, "Filename" );

	char name_text[1024];
	char toolkit_text[1024];
	char file_text[1024];

	if( !get_geditable_string( name->entry, name_text, 1024 ) ||
		!get_geditable_name( toolkit->entry, toolkit_text, 1024 ) ||
		!get_geditable_filename( file->entry, file_text, 1024 ) ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	/* Save column to file.
	 */
	workspace_deselect_all( ws );
        column_select_symbols( col );

	if( !workspace_selected_save( ws, file_text ) ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	workspace_deselect_all( ws );

	if( !tool_new_dia( toolkit_by_name( toolkit_text ), 
			-1, name_text, file_text ) ) {
		unlink( file_text );
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	SETSTR( columnview_to_menu_last_toolkit, toolkit_text );

	nfn( sys, IWINDOW_TRUE );
}

/* Help button hit.
 */
static void
columnview_to_menu_help_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	box_help( GTK_WIDGET( iwnd ), "sec:diaref" );

	nfn( sys, IWINDOW_TRUE );
}

/* Make a column into a menu item. 
 */
static void
columnview_to_menu_cb( GtkWidget *wid, GtkWidget *host, Columnview *cview )
{
	Column *col = COLUMN( VIEW( cview )->model );
	iWindow *iwnd = IWINDOW( view_get_toplevel( VIEW( cview ) ) );
	GtkWidget *ss = stringset_new();
	char *name;
	char *kit_name;
	char filename[PATHLEN];

	if( !(name = MODEL( col )->caption) )
		name = "untitled";
	columnview_filename( filename, name );

	if( columnview_to_menu_last_toolkit )
		kit_name = columnview_to_menu_last_toolkit;
	else
		kit_name = "untitled";

	stringset_child_new( STRINGSET( ss ), 
		"Name", name, "Set menu item text here" );
	stringset_child_new( STRINGSET( ss ), 
		"Toolkit", kit_name, "Add to this toolkit" );
	stringset_child_new( STRINGSET( ss ), 
		"Filename", filename, "Store column in this file" );

	iwindow_set_title( IWINDOW( ss ), 
		"New menu item from column \"%s\"", MODEL( col )->name );
	idialog_set_callbacks( IDIALOG( ss ), 
		iwindow_true_cb, columnview_to_menu_help_cb, NULL, NULL, col );
	idialog_add_ok( IDIALOG( ss ), columnview_to_menu_done_cb, "Menuize" );
	idialog_set_parent( IDIALOG( ss ), GTK_WIDGET( iwnd ) );
	iwindow_build( IWINDOW( ss ) );

	gtk_widget_show( ss );
}

/* Find the position and size of a columnview.
 */
void
columnview_get_position( Columnview *cview, int *x, int *y, int *w, int *h )
{
	Column *col = COLUMN( VIEW( cview )->model );
	Workspaceview *wview = cview->wview;

        if( (wview->vp.width == 0 && wview->vp.height == 0) ||
		(GTK_WIDGET( cview )->allocation.x == -1 && 
		GTK_WIDGET( cview )->allocation.y == -1) ) {
                /* Nothing there yet, guess.
                 */
		*x = col->x; 
		*y = col->y;
                *w = 200;
                *h = 50;
        }
        else {
                *x = GTK_WIDGET( cview )->allocation.x;
                *y = GTK_WIDGET( cview )->allocation.y;
                *w = GTK_WIDGET( cview )->allocation.width;
                *h = GTK_WIDGET( cview )->allocation.height;

#ifdef DEBUG
		printf( "columnview_get_position: %s, "
			"x = %d, y = %d, w = %d, h = %d\n",
			MODEL( col )->name, *x, *y, *w, *h );
#endif /*DEBUG*/
        }
}

/* Pop columnview to front.
 */
static void
columnview_front( Columnview *cview )
{
	Workspaceview *wview = cview->wview;

	if( wview->front != cview ) {
		gtk_widget_ref( GTK_WIDGET( cview ) );
		gtk_container_remove( GTK_CONTAINER( wview->fixed ),
			GTK_WIDGET( cview ) );
		gtk_fixed_put( GTK_FIXED( wview->fixed ),
			GTK_WIDGET( cview ), cview->lx, cview->ly );
		gtk_widget_unref( GTK_WIDGET( cview ) );

		wview->front = cview;
	}
}

/* Transition functions for mouse stuff on columnviews.
 */
static void
columnview_left_press( Columnview *cview, GdkEvent *ev )
{
	Workspaceview *wview = cview->wview;

        int ix, iy;
        int jx, jy;
        int kx, ky;
        int wx, wy, ww, wh;

        /* Find pos of columnview.
         */
        columnview_get_position( cview, &wx, &wy, &ww, &wh );

        /* Position in virtual tally window.
         */
        ix = ev->button.x + wx;
        iy = ev->button.y + wy;

        /* Position in tally viewport.
         */
        jx = ix - wview->vp.left;
        jy = iy - wview->vp.top;

        /* So ... position of top LH corner of tally viewport in root window.
         */
        kx = ev->button.x_root - jx;
        ky = ev->button.y_root - jy;

        switch( cview->state ) {
        case COL_WAIT:
                cview->state = COL_SELECT;

                /* Record offset of mouse in columnview title bar.
                 */
                cview->rx = ev->button.x;
                cview->ry = ev->button.y;

                /* Position of tally window in root window.
                 */
                cview->tx = kx;
                cview->ty = ky;

                /* Start position of mouse in virtual tally window.
                 */
                cview->sx = ix;
                cview->sy = iy;

                break;

        case COL_SELECT:
        case COL_DRAG:
        case COL_EDIT:
                break;

        default:
		assert( FALSE );
        }
}

static void
columnview_left_motion( Columnview *cview, GdkEvent *ev )
{
	Column *col = COLUMN( VIEW( cview )->model );
	Workspaceview *wview = cview->wview;

	int u, v;

        /* Posn of pointer in tally viewport.
         */
        int ix = ev->motion.x_root - cview->tx;
        int iy = ev->motion.y_root - cview->ty;

        /* Posn in virtual tally cods.
         */
        int jx = ix + wview->vp.left;
        int jy = iy + wview->vp.top;

        /* Amount of drag since we started.
         */
        int xoff = jx - cview->sx;
        int yoff = jy - cview->sy;

        /* New columnview position.
         */
        int xnew = IM_MAX( 0, jx - cview->rx );
        int ynew = IM_MAX( 0, jy - cview->ry );

        switch( cview->state ) {
        case COL_EDIT:
        case COL_WAIT:
                break;

        case COL_SELECT:
                /* How much drag?
                 */
                if( abs( xoff ) > 5 || abs( yoff ) > 5 ) {
                        cview->state = COL_DRAG;
			workspaceview_set_cursor( wview, IWINDOW_SHAPE_MOVE );

#ifdef GRAB
                        /* And grab the mouse.
                         */
#ifdef DEBUG_GRAB
			printf( "columnview_left_motion: grabbing pointer\n" );
#endif /*DEBUG_GRAB*/

                        gdk_pointer_grab( GTK_WIDGET( cview->title )->window,
                                FALSE,
                                GDK_BUTTON1_MOTION_MASK |
                                GDK_POINTER_MOTION_MASK |
                                GDK_BUTTON_RELEASE_MASK,
                                NULL, NULL, ev->button.time );
                        gtk_grab_add( GTK_WIDGET( cview->title ) );
#endif /*GRAB*/
                }

                break;

        case COL_DRAG:
		col->x = xnew;
		col->y = ynew;
		model_changed( MODEL( col ) );

#ifdef DEBUG
		g_message( "drag columnview: x=%d, y=%d", col->x, col->y );
#endif /*DEBUG*/

                /* Set vars for bg scroll.
                 */
		u = 0;
                if( ix > wview->vp.width )
                        u = 10;
                else if( ix < 0 )
                        u = -10;

		v = 0;
                if( iy > wview->vp.height )
                        v = 10;
                else if( iy < 0 )
                        v = -10;

		workspaceview_scroll_background( wview, u, v );

                break;

        default:
                assert( FALSE );
        }
}

static void
columnview_left_release( Columnview *cview, GdkEvent *ev )
{
	Column *col = COLUMN( VIEW( cview )->model );
	Workspace *ws = col->ws;
	Workspaceview *wview = cview->wview;

        /* Back to wait.
         */
        switch( cview->state ) {
        case COL_SELECT:
                cview->state = COL_WAIT;
                workspace_column_select( ws, col );
                columnview_front( cview );

                break;

        case COL_DRAG:
                cview->state = COL_WAIT;
		workspaceview_scroll_background( wview, 0, 0 );
		filemodel_set_modified( FILEMODEL( ws ), TRUE );
		workspaceview_set_cursor( wview, IWINDOW_SHAPE_NONE );

#ifdef GRAB
                /* And release the mouse.
                 */
#ifdef DEBUG_GRAB
		printf( "columnview_left_release: ungrabbing pointer\n" );
#endif /*DEBUG_GRAB*/

                gtk_grab_remove( GTK_WIDGET( cview->title ) );
                gdk_pointer_ungrab( ev->button.time );
#endif /*GRAB*/

                break;

        case COL_EDIT:
        case COL_WAIT:
                break;

        default:
                assert( FALSE );
        }
}

/* Event in columnview title bar.
 */
static gboolean
columnview_event_cb( GtkWidget *widget, GdkEvent *ev, Columnview *cview )
{
	gboolean used = FALSE;

        switch( ev->type ) {
        case GDK_BUTTON_PRESS:
                if( ev->button.button == 1 ) {
                        columnview_left_press( cview, ev );
                        used = TRUE;
                }

                break;

        case GDK_MOTION_NOTIFY:
                if( ev->motion.state & GDK_BUTTON1_MASK ) {
                        columnview_left_motion( cview, ev );
                        used = TRUE;
                }

                break;

        case GDK_BUTTON_RELEASE:
                if( ev->button.button == 1 ) {
                        columnview_left_release( cview, ev );
                        used = TRUE;
                }

                break;

        default:
                break;
        }

	if( used ) 
		gtk_signal_emit_stop_by_name( GTK_OBJECT( widget ), "event" );

        return( used );
}

/* Double click on titlebar callback. Edit the caption.
 */
static void
columnview_double_cb( GtkWidget *wid, Columnview *cview )
{
        /* Edit caption!
         */
        if( cview->state == COL_EDIT )
                return;

        cview->state = COL_EDIT;
        view_refresh_queue( VIEW( cview ) );
}

/* Arrow button on title bar.
 */
static void
columnview_updown_cb( GtkWidget *wid, Columnview *cview )
{
	Column *col = COLUMN( VIEW( cview )->model );
	Workspace *ws = col->ws;

        col->open = !col->open;
	filemodel_set_modified( FILEMODEL( ws ), TRUE );
        model_changed( MODEL( col ) );
}

/* Delete this column from the popup menu.
 */
static void
columnview_destroy_cb( GtkWidget *wid, GtkWidget *host, Columnview *cview )
{
	Column *col = COLUMN( VIEW( cview )->model );

	model_check_destroy( view_get_toplevel( VIEW( cview ) ), MODEL( col ) );
}

/* Delete this column with a click on the 'x' button.
 */
static void
columnview_destroy2_cb( GtkWidget *wid, Columnview *cview )
{
	Column *col = COLUMN( VIEW( cview )->model );

	model_check_destroy( view_get_toplevel( VIEW( cview ) ), MODEL( col ) );
}

/* Callback for enter in caption edit box.
 */
static void
columnview_caption_enter_cb( GtkWidget *wid, Columnview *cview )
{
        gchar *text = gtk_entry_get_text( GTK_ENTRY( cview->capedit ) );
	Column *col = COLUMN( VIEW( cview )->model );
	Workspace *ws = col->ws;

        cview->state = COL_WAIT;
	model_set( MODEL( col ), NULL, text );
	filemodel_set_modified( FILEMODEL( ws ), TRUE );
}

/* Detect cancel in a caption field.
 */
static gboolean
columnview_caption_cancel_cb( GtkWidget *widget, 
	GdkEvent *ev, Columnview *cview )
{
        if( ev->type != GDK_KEY_PRESS || ev->key.keyval != GDK_Escape )
                return( FALSE );

        /* Turn off edit.
         */
        cview->state = COL_WAIT;
        view_refresh_queue( VIEW( cview ) );

        return( TRUE );
}

/* Add a caption entry to a columnview if not there.
 */
static void
columnview_add_caption( Columnview *cview )
{
        if( cview->capedit )
                return;

        cview->capedit = gtk_entry_new();
        gtk_box_pack_start( GTK_BOX( cview->titlehb ),
                cview->capedit, FALSE, FALSE, 0 );
        set_tooltip( cview->capedit, "Edit caption, press enter to "
                "accept changes, press escape to cancel" );

        gtk_signal_connect( GTK_OBJECT( cview->capedit ), "activate",
                GTK_SIGNAL_FUNC( columnview_caption_enter_cb ), cview );
        gtk_signal_connect( GTK_OBJECT( cview->capedit ), "event",
                GTK_SIGNAL_FUNC( columnview_caption_cancel_cb ), cview );
}

/* Callback for enter in new def widget.
 */
static void
columnview_text_enter_cb( GtkWidget *wid, Columnview *cview )
{
        gchar *text = gtk_entry_get_text( GTK_ENTRY( cview->text ) );
	Column *col = COLUMN( VIEW( cview )->model );
	Workspace *ws = col->ws;
        Symbol *sym;

        if( !text || strspn( text, " \t\b\n" ) == strlen( text ) ||
		!(sym = workspace_add_def( ws, text )) ) 
		return;

	if( sym->expr->err ) {
		expr_error_get( sym->expr );
		box_alert( NULL );
		gtk_object_destroy( GTK_OBJECT( sym ) );
		symbol_recalculate_all();
		return;
	}

	set_gentry( cview->text, NULL );
	model_scrollto( MODEL( col ) );
}

/* Add bottom entry widget.
 */
static void
columnview_add_text( Columnview *cview )
{
        GtkWidget *inv;

        if( cview->textfr )
                return;

        cview->textfr = gtk_hbox_new( FALSE, 0 );
        gtk_box_pack_end( GTK_BOX( cview->vbox ), 
		cview->textfr, FALSE, FALSE, 0 );
        inv = gtk_label_new( "" );
        gtk_box_pack_start( GTK_BOX( cview->textfr ), inv, FALSE, FALSE, 25 );
        gtk_widget_show( inv );
        cview->text = gtk_entry_new();
        gtk_box_pack_start( GTK_BOX( cview->textfr ), 
		cview->text, TRUE, TRUE, 0 );
        gtk_signal_connect( GTK_OBJECT( cview->text ), "activate",
                GTK_SIGNAL_FUNC( columnview_text_enter_cb ), cview );
        gtk_widget_show( cview->text );
        set_tooltip( cview->text, "Enter expressions here" );
}

static void 
columnview_refresh( View *view )
{
	Columnview *cview = COLUMNVIEW( view );
	Column *col = COLUMN( VIEW( cview )->model );
	Workspaceview *wview = cview->wview;

	/* Update position.
	 */
	if( col->x != cview->lx || col->y != cview->ly ) {
#ifdef DEBUG
		printf( "columnview_refresh: move column %s to %d x %d\n",
			MODEL( col )->name, col->x, col->y );
#endif /*DEBUG*/

		cview->lx = col->x;
		cview->ly = col->y;
		gtk_fixed_move( GTK_FIXED( wview->fixed ),
			GTK_WIDGET( cview ), cview->lx, cview->ly );

		/* Update the save offset hints too.
		 */
		filemodel_set_offset( FILEMODEL( col ), cview->lx, cview->ly );
	}

	/* Update names.
	 */
        set_glabel( cview->lab, "%s - ", MODEL( col )->name );
        set_glabel( cview->head, "%s", MODEL( col )->caption );

	/* Set open/closed.
	 */
	if( col->open ) {
                if( !GTK_WIDGET_VISIBLE( cview->frame ) ) {
                        gtk_widget_show( cview->frame );

                        /* Pop to front ... if we have opened it, we
                         * probably want to look at it.
                         */
                        columnview_front( cview );
                }

                gtk_arrow_set( GTK_ARROW( cview->updown ),
                        GTK_ARROW_DOWN, GTK_SHADOW_OUT );
                set_tooltip( cview->updownb, "Close the columnview" );
	}
	else {
                gtk_widget_hide( cview->frame );
                gtk_arrow_set( GTK_ARROW( cview->updown ),
                        GTK_ARROW_UP, GTK_SHADOW_OUT );
                set_tooltip( cview->updownb, "Open the columnview" );
		view_resize( VIEW( cview ) );
	}

        /* Set caption edit.
         */
        if( cview->state == COL_EDIT ) {
                columnview_add_caption( cview );

                gtk_widget_show( cview->capedit );
                gtk_widget_hide( cview->headfr );

                set_gentry( cview->capedit, "%s", MODEL( col )->caption );
                gtk_editable_select_region( GTK_EDITABLE( cview->capedit ),
                        0, strlen( MODEL( col )->caption ) );
                gtk_widget_grab_focus( cview->capedit );
        }
        else {
                gtk_widget_show( cview->headfr );
                FREEW( cview->capedit );
        }

        /* Set bottom entry.
         */
        if( col->selected && col->open ) {
                columnview_add_text( cview );
                gtk_widget_show( cview->textfr );
        }
        else
                FREEW( cview->textfr );

	/* Set select state.
	 */
        if( col->selected && !cview->selected ) {
		int x, y, w, h;
		Rect column, overlap;

                set_name( cview->title, "selected_widget" );
		cview->selected = TRUE;

		/* If the column is completely off the screen, scroll to make
		 * the titlebar visible. Not the same as model_scrollto().
		 */
		columnview_get_position( cview, &x, &y, &w, &h );
		column.left = x;
		column.top = y;
		column.width = w;
		column.height = h;
		im_rect_intersectrect( &column, &wview->vp, &overlap );
		if( im_rect_isempty( &overlap ) )
			workspaceview_scroll( wview, x, y, w, 25 );

		/* Grab focus the 1st time we're selected.
		 */
		if( !cview->stale && cview->text ) {
			gtk_widget_grab_focus( cview->text );
			cview->stale = TRUE;
		}
	}
        else if( !col->selected && cview->selected ) {
                set_name( cview->title, "default_widget" );
		cview->selected = FALSE;
	}

	VIEW_CLASS( parent_class )->refresh( view );
}

/* Scroll to keep the text entry at the bottom of the columnview on screen.
 * We can't use the position/size of the text widget for positioning, since it
 * may not be properly realized yet ... make the bottom of the column visible
 * instead.
 */
static void
columnview_scrollto( View *view )
{
	Columnview *cview = COLUMNVIEW( view );
	Workspaceview *wview = cview->wview;
	int x, y, w, h;

	columnview_get_position( cview, &x, &y, &w, &h );
	workspaceview_scroll( wview, x, y + h, w, 25 );
}

static void
columnview_link( View *view, Model *model, View *parent )
{
	Columnview *cview = COLUMNVIEW( view );
	Column *col = COLUMN( model );
	Workspaceview *wview = WORKSPACEVIEW( parent );

	VIEW_CLASS( parent_class )->link( view, model, parent );

	cview->wview = wview;

	/* Pick start xy pos. 
	 */
        workspaceview_pick_xy( wview, &col->x, &col->y );
	gtk_fixed_put( GTK_FIXED( wview->fixed ),
		GTK_WIDGET( cview ), cview->lx, cview->ly );
	cview->lx = -1;
	cview->ly = -1;
	wview->front = cview;

	/* Refresh to make sure ws->area is updated.
	 */
        view_refresh_queue( VIEW( cview ) );
}

static void
columnview_class_init( ColumnviewClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	ViewClass *view_class = (ViewClass *) klass;

	GtkWidget *pane;

	parent_class = gtk_type_class( TYPE_VIEW );

	object_class->destroy = columnview_destroy;

	/* Create signals.
	 */

	/* Init methods.
	 */
	view_class->refresh = columnview_refresh;
	view_class->scrollto = columnview_scrollto;
	view_class->link = columnview_link;

	pane = columnview_menu = popup_build( "Column menu" );
	popup_add_but( pane, "Edit caption ...", columnview_caption_cb );
	popup_add_but( pane, "Select all items in column", 
		columnview_select_cb );
	popup_add_but( pane, "Clone column", columnview_clone_cb );
	popup_add_but( pane, "Save column ...", columnview_save_as_cb );
	menu_add_sep( pane );
	popup_add_but( pane, "Make column into menu item...",
		columnview_to_menu_cb );
	menu_add_sep( pane );
	popup_add_but( pane, "Delete column ...", columnview_destroy_cb );
}

static void
columnview_init( Columnview *cview )
{
	/* Delete columnview graphic.
	 */
        static GdkPixmap *delete_px = NULL;
        static GdkBitmap *delete_mask = NULL;

        GtkWidget *sb;
        GtkWidget *sep;
        GtkWidget *frame;
        GtkWidget *px;
        GtkWidget *but;

	/* Make sure the delete columnview graphic is built.
	 */
        if( !delete_px ) {
                delete_px = gdk_pixmap_create_from_xpm_d( 
			main_window_gdk,
                        &delete_mask, NULL, column_delete_xpm );
        }

        /* No position yet.
         */
        cview->lx = -1;
        cview->ly = -1;

	cview->state = COL_WAIT;

	cview->selected = FALSE;
	cview->stale = FALSE;

        /* Make outer vb.
         */
        cview->main = gtk_event_box_new();
        cview->vbox = gtk_vbox_new( FALSE, 0 );
        gtk_container_add( GTK_CONTAINER( cview->main ), cview->vbox );

        /* Frame for whole title bar. Need an event_box to catch clicks.
         */
        cview->title = gtk_event_box_new();
        gtk_box_pack_start( GTK_BOX( cview->vbox ), 
		cview->title, FALSE, FALSE, 0 );
        set_tooltip( cview->title, "Left-drag to move, left-double-click to "
                "set title, right-click for menu" );
        frame = gtk_frame_new( NULL );
        gtk_frame_set_shadow_type( GTK_FRAME( frame ), GTK_SHADOW_OUT );
        gtk_container_add( GTK_CONTAINER( cview->title ), frame );
        doubleclick_add( cview->title, 
		TRUE, NULL, NULL, columnview_double_cb, cview );
        popup_attach( cview->title, columnview_menu, cview );
        gtk_signal_connect( GTK_OBJECT( cview->title ), "event",
                GTK_SIGNAL_FUNC( columnview_event_cb ), cview );

        /* Layout contents of title bar.
         */
        cview->titlehb = gtk_hbox_new( FALSE, 0 );
        gtk_container_add( GTK_CONTAINER( frame ), cview->titlehb );

        /* Up/down button.
         */
        cview->updownb = gtk_button_new();
        gtk_button_set_relief( GTK_BUTTON( cview->updownb ), GTK_RELIEF_NONE );
        gtk_container_set_border_width( GTK_CONTAINER( cview->updownb ), 0 );
        gtk_box_pack_end( GTK_BOX( cview->titlehb ),
                cview->updownb, FALSE, FALSE, 0 );
        cview->updown = gtk_arrow_new( GTK_ARROW_UP, GTK_SHADOW_OUT );
        gtk_container_add( GTK_CONTAINER( cview->updownb ), cview->updown );
        gtk_signal_connect( GTK_OBJECT( cview->updownb ), "clicked",
                GTK_SIGNAL_FUNC( columnview_updown_cb ), cview );

        /* Up/down - title separator.
         */
        sep = gtk_vseparator_new();
        gtk_box_pack_end( GTK_BOX( cview->titlehb ), sep, FALSE, FALSE, 0 );

        /* Remove columnview button.
         */
        sb = gtk_vbox_new( FALSE, 0 );
        gtk_box_pack_start( GTK_BOX( cview->titlehb ), sb, FALSE, FALSE, 4 );
        but = gtk_button_new();
        gtk_box_pack_start( GTK_BOX( sb ), but, TRUE, FALSE, 0 );
        set_tooltip( but, "Delete the column" );
        px = gtk_pixmap_new( delete_px, delete_mask );
        gtk_container_add( GTK_CONTAINER( but ), px );
        gtk_signal_connect( GTK_OBJECT( but ), "clicked",
                GTK_SIGNAL_FUNC( columnview_destroy2_cb ), cview );

        /* Columnview name.
         */
        cview->lab = gtk_label_new( "" );
        gtk_box_pack_start( GTK_BOX( cview->titlehb ), 
		cview->lab, FALSE, FALSE, 0 );

        /* Comment. Wrap a frame around it, to make it the same size as
         * an entry widget.
         */
        cview->headfr = gtk_frame_new( NULL );
        gtk_frame_set_shadow_type( GTK_FRAME( cview->headfr ), 
		GTK_SHADOW_NONE );
        gtk_container_set_border_width( GTK_CONTAINER( cview->headfr ), 1 );
        gtk_box_pack_start( GTK_BOX( cview->titlehb ),
                cview->headfr, FALSE, FALSE, 0 );
        cview->head = gtk_label_new( "" );
        gtk_container_add( GTK_CONTAINER( cview->headfr ), cview->head );

        /* Make central table for tally roll.
         */
        cview->frame = gtk_frame_new( NULL );
        gtk_box_pack_start( GTK_BOX( cview->vbox ), 
		cview->frame, TRUE, TRUE, 0 );

        gtk_box_pack_start( GTK_BOX( cview ), cview->main, FALSE, FALSE, 0 );

        gtk_widget_show_all( GTK_WIDGET( cview ) );
}

GtkType
columnview_get_type( void )
{
	static GtkType columnview_type = 0;

	if (!columnview_type) {
		static const GtkTypeInfo cview_info = {
			"Columnview",
			sizeof( Columnview ),
			sizeof( ColumnviewClass ),
			(GtkClassInitFunc) columnview_class_init,
			(GtkObjectInitFunc) columnview_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		columnview_type = gtk_type_unique( TYPE_VIEW, &cview_info );
	}

	return( columnview_type );
}

View *
columnview_new( void )
{
	Columnview *cview = gtk_type_new( TYPE_COLUMNVIEW );

	return( VIEW( cview ) );
}

