/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Modified by the GTK+ Team and others 1997-1999.  See the AUTHORS
 * file for a list of people on the GTK+ Team.  See the ChangeLog
 * files for a list of changes.  These files are distributed with
 * GTK+ at ftp://ftp.gtk.org/pub/gtk/. 
 */

/* 
 * Hacked to derive from iDialog, stuff added.
 */

#include "config.h"

#ifdef HAVE_WINDOWS_H
#include <io.h>
#define S_ISDIR(m) (!!(m & _S_IFDIR))
#include <direct.h>
#endif

#include "ip.h"

/*
#define DEBUG
 */

#define DIR_LIST_WIDTH   (180)
#define DIR_LIST_HEIGHT  (180)
#define FILE_LIST_WIDTH  (180)
#define FILE_LIST_HEIGHT (180)

/* Our signals. A new file has been selected, a new directory has been
 * entered, we are asked to refresh.
 */
enum {
  FILE_SELECT,
  DIR_ENTER,
  REFRESH,
  LAST_SIGNAL
};

typedef struct _CompletionState    CompletionState;
typedef struct _CompletionDir      CompletionDir;
typedef struct _CompletionDirSent  CompletionDirSent;
typedef struct _CompletionDirEntry CompletionDirEntry;
typedef struct _CompletionUserDir  CompletionUserDir;
typedef struct _PossibleCompletion PossibleCompletion;

/* Non-external file completion decls and structures */

/* A contant telling PRCS how many directories to cache.  Its actually
 * kept in a list, so the geometry isn't important. 
 */
#define CMPL_DIRECTORY_CACHE_SIZE 10

/* A constant used to determine whether a substring was an exact
 * match by first_diff_index()
 */
#define PATTERN_MATCH -1

/* Use fnmatch() if available, otherwise bodge with wild_*() code.
 */
#ifdef HAVE_FNMATCH
#define FNMATCH( PATT, STR ) \
	(fnmatch( (PATT), (STR), FNM_PATHNAME | FNM_PERIOD) != FNM_NOMATCH)
#else /*HAVE_FNMATCH*/
#define FNMATCH( PATT, STR ) \
	(wild_match_patt( (PATT), (STR) ))
#endif /*HAVE_FNMATCH*/

#define CMPL_ERRNO_TOO_LONG ((1<<16)-1)

/* This structure contains all the useful information about a directory
 * for the purposes of filename completion.  These structures are cached
 * in the CompletionState struct.  CompletionDir's are reference counted.
 */
struct _CompletionDirSent
{
  ino_t inode;
  time_t mtime;
  dev_t device;

  gint entry_count;
  gchar *name_buffer; /* memory segment containing names of all entries */

  struct _CompletionDirEntry *entries;
};

struct _CompletionDir
{
  CompletionDirSent *sent;

  gchar *fullname;
  gint fullname_len;

  struct _CompletionDir *cmpl_parent;
  gint cmpl_index;
  const char *cmpl_text;
};

/* This structure contains pairs of directory entry names with a flag saying
 * whether or not they are a valid directory.  NOTE: This information is used
 * to provide the caller with information about whether to update its completions
 * or try to open a file.  Since directories are cached by the directory mtime,
 * a symlink which points to an invalid file (which will not be a directory),
 * will not be reevaluated if that file is created, unless the containing
 * directory is touched.  I consider this case to be worth ignoring (josh).
 */
struct _CompletionDirEntry
{
  gint is_dir;
  gchar *entry_name;
};

struct _CompletionUserDir
{
  gchar *login;
  gchar *homedir;
};

struct _PossibleCompletion
{
  /* accessible fields, all are accessed externally by functions
   * declared above
   */
  gchar *text;
  gint is_a_completion;
  gint is_directory;

  /* Private fields
   */
  gint text_alloc;
};

struct _CompletionState
{
  gint last_valid_char;
  gchar *updated_text;
  gint updated_text_len;
  gint updated_text_alloc;
  gint re_complete;

  gchar *user_dir_name_buffer;
  gint user_directories_len;

  const char *last_completion_text;

  gint user_completion_index; /* if >= 0, currently completing ~user */

  gint environment_completion_index; /* if >= 0, currently completing $xx */

  struct _CompletionDir *completion_dir; /* directory completing from */
  struct _CompletionDir *active_completion_dir;

  struct _PossibleCompletion the_completion;

  struct _CompletionDir *reference_dir; /* initial directory */

  GList* directory_storage;
  GList* directory_sent_storage;

  struct _CompletionUserDir *user_directories;

  GtkFileSelection2FilenamePredicate predicate;
  gpointer predicate_data;
};

/* File completion functions which would be external, were they used
 * outside of this file.
 */

static CompletionState*    cmpl_init_state        (void);
static void                cmpl_free_state        (CompletionState *cmpl_state);
static gint                cmpl_state_okay        (CompletionState* cmpl_state);
static const char*         cmpl_strerror          (gint);

static PossibleCompletion* cmpl_completion_matches(const char *text_to_complete,
						   const char **remaining_text,
						   CompletionState *cmpl_state);

/* Returns a name for consideration, possibly a completion, this name
 * will be invalid after the next call to cmpl_next_completion.
 */
static char*               cmpl_this_completion   (PossibleCompletion*);

/* True if this completion matches the given text.  Otherwise, this
 * output can be used to have a list of non-completions.
 */
static gint                cmpl_is_a_completion   (PossibleCompletion*);

/* True if the completion is a directory
 */
static gint                cmpl_is_directory      (PossibleCompletion*);

/* Obtains the next completion, or NULL
 */
static PossibleCompletion* cmpl_next_completion   (CompletionState*);

/* Updating completions: the return value of cmpl_updated_text() will
 * be text_to_complete completed as much as possible after the most
 * recent call to cmpl_completion_matches.  For the present
 * application, this is the suggested replacement for the user's input
 * string.  You must CALL THIS AFTER ALL cmpl_text_completions have
 * been received.
 */
static gchar*              cmpl_updated_text       (CompletionState* cmpl_state);

/* After updating, to see if the completion was a directory, call
 * this.  If it was, you should consider re-calling completion_matches.
 */
static gint                cmpl_updated_dir        (CompletionState* cmpl_state);

/* Current location: if using file completion, return the current
 * directory, from which file completion begins.  More specifically,
 * the cwd concatenated with all exact completions up to the last
 * directory delimiter (IM_DIR_SEP).
 */
static gchar*              cmpl_reference_position (CompletionState* cmpl_state);

/* backing up: if cmpl_completion_matches returns NULL, you may query
 * the index of the last completable character into cmpl_updated_text.
 */
static gint                cmpl_last_valid_char    (CompletionState* cmpl_state);

/* When the user selects a non-directory, call cmpl_completion_fullname
 * to get the full name of the selected file.
 */
static gchar*              cmpl_completion_fullname (gchar*, CompletionState* cmpl_state);


/* Directory operations. */
static CompletionDir* open_ref_dir         (const char* text_to_complete,
					    const char** remaining_text,
					    CompletionState* cmpl_state);
static gboolean       check_dir            (const char *dir_name, 
					    struct stat *result, 
					    gboolean *stat_subdirs);
static CompletionDir* open_dir             (const char* dir_name,
					    CompletionState* cmpl_state);
static CompletionDir* open_environment_dir (const char* text_to_complete,
					    CompletionState *cmpl_state);
static CompletionDir* open_user_dir        (const char* text_to_complete,
					    CompletionState *cmpl_state);
static CompletionDir* open_relative_dir    (const char* dir_name, 
					    CompletionDir* dir,
					    CompletionState *cmpl_state);
static CompletionDirSent* open_new_dir     (const char* dir_name, 
					    struct stat* sbuf,
					    gboolean stat_subdirs);
static gint           correct_dir_fullname (CompletionDir* cmpl_dir);
static gint           correct_parent       (CompletionDir* cmpl_dir);
static gchar*         find_parent_dir_fullname    (gchar* dirname);
static CompletionDir* attach_dir           (CompletionDirSent* sent,
					    const char* dir_name,
					    CompletionState *cmpl_state);
static void           free_dir_sent (CompletionDirSent* sent);
static void           free_dir      (CompletionDir  *dir);
static void           prune_memory_usage(CompletionState *cmpl_state);

/* Completion operations */
static PossibleCompletion* 
	attempt_environment_completion(const char* text_to_complete,
				       CompletionState *cmpl_state);
static PossibleCompletion* attempt_homedir_completion(const char* text_to_complete,
						      CompletionState *cmpl_state);
static PossibleCompletion* attempt_file_completion(CompletionState *cmpl_state);
static CompletionDir* find_completion_dir(const char* text_to_complete,
					  const char** remaining_text,
					  CompletionState* cmpl_state);
static PossibleCompletion* append_completion_text(const char* text,
						  CompletionState* cmpl_state);
static gint get_pwdb(CompletionState* cmpl_state);
static gint first_diff_index(const char* pat, const char* text);
static gint compare_user_dir(const void* a, const void* b);
static gint compare_cmpl_dir(const void* a, const void* b);
static void update_cmpl(PossibleCompletion* poss,
			CompletionState* cmpl_state);

static void gtk_file_selection2_class_init    (GtkFileSelection2Class *klass);
static void gtk_file_selection2_init          (GtkFileSelection2      *fs);
static void gtk_file_selection2_destroy       (GtkObject             *object);
static void gtk_file_selection2_build         (GtkWidget             *widget);
static gint gtk_file_selection2_key_press     (GtkWidget             *widget,
					      GdkEventKey           *event,
					      gpointer               user_data);
static gint gtk_file_selection2_dir_key_press (GtkWidget             *widget,
					      GdkEventKey           *event,
					      gpointer               user_data);
static void gtk_file_selection2_dir_activate  (GtkWidget             *widget,
					      gpointer               user_data);

static void gtk_file_selection2_file_button (GtkWidget *widget,
					    gint row, 
					    gint column, 
					    GdkEventButton *bevent,
					    gpointer user_data);

static void gtk_file_selection2_select_row (GtkWidget *widget,
					    gint row, 
					    gint column, 
					    GdkEventButton *bevent,
					    gpointer user_data);
static void gtk_file_selection2_unselect_row (GtkWidget *widget,
					    gint row, 
					    gint column, 
					    GdkEventButton *bevent,
					    gpointer user_data);

static void gtk_file_selection2_dir_button (GtkWidget *widget,
					   gint row, 
					   gint column, 
					   GdkEventButton *bevent,
					   gpointer data);

static void gtk_file_selection2_abort        (GtkFileSelection2      *fs);
static void gtk_real_refresh   		     (GtkFileSelection2      *fs);
static void gtk_real_dir_enter 		     (GtkFileSelection2      *fs,
					      const gchar 	     *txt);
static void gtk_real_file_select	     (GtkFileSelection2      *fs,
					      const gchar 	     *txt);

static void gtk_file_selection2_create_dir (GtkWidget *widget, gpointer data);
static void gtk_file_selection2_delete_file (GtkWidget *widget, gpointer data);
static void gtk_file_selection2_rename_file (GtkWidget *widget, gpointer data);

static void gtk_file_selection2_dir_enter (GtkFileSelection2 *fs, 
					 gchar *rel_path,
					 gint try_complete);
static void gtk_file_selection2_new_file (GtkFileSelection2 *fs, 
					 gchar *rel_path,
				         gint try_complete);

static iDialogClass *parent_class = NULL;
static guint filesel_signals[LAST_SIGNAL] = { 0 };

/* Saves errno when something cmpl does fails. 
 */
static gint cmpl_errno;

/* The Quark we use to hide suffix numbers in the file type menu.
 */
static GQuark quark_suffix = 0;

GtkType
gtk_file_selection2_get_type (void)
{
  static GtkType file_selection_type = 0;

  if (!file_selection_type)
    {
      static const GtkTypeInfo fs_info =
      {
	"GtkFileSelection2",
	sizeof (GtkFileSelection2),
	sizeof (GtkFileSelection2Class),
	(GtkClassInitFunc) gtk_file_selection2_class_init,
	(GtkObjectInitFunc) gtk_file_selection2_init,
	/* reserved_1 */ NULL,
	/* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      file_selection_type = gtk_type_unique (TYPE_IDIALOG, &fs_info);
    }

  return file_selection_type;
}

static void
gtk_file_selection2_class_init (GtkFileSelection2Class *klass)
{
  GtkObjectClass *object_class;
  iWindowClass *iwindow_class;

  object_class = (GtkObjectClass*) klass;
  iwindow_class = (iWindowClass*) klass;
  parent_class = gtk_type_class (TYPE_IDIALOG);

  object_class->destroy = gtk_file_selection2_destroy;
  iwindow_class->build = gtk_file_selection2_build;

  filesel_signals[FILE_SELECT] =
    gtk_signal_new ("file_select",
                    GTK_RUN_FIRST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GtkFileSelection2Class, file_select),
                    gtk_marshal_NONE__STRING,
                    GTK_TYPE_NONE, 1,
		    GTK_TYPE_STRING);
  filesel_signals[DIR_ENTER] =
    gtk_signal_new ("dir_enter",
                    GTK_RUN_FIRST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GtkFileSelection2Class, dir_enter),
                    gtk_marshal_NONE__STRING,
                    GTK_TYPE_NONE, 1,
		    GTK_TYPE_STRING);
  filesel_signals[REFRESH] =
    gtk_signal_new ("refresh",
                    GTK_RUN_FIRST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GtkFileSelection2Class, refresh),
                    gtk_marshal_NONE__NONE,
                    GTK_TYPE_NONE, 1,
		    GTK_TYPE_NONE);

  gtk_object_class_add_signals (object_class, filesel_signals, LAST_SIGNAL);

  klass->file_select = gtk_real_file_select;
  klass->dir_enter = gtk_real_dir_enter;
  klass->refresh = gtk_real_refresh;
}

/* Is suff a postfix of name, strcmp()-type return: 0 for match.
 */
static gboolean
is_postfix( gconstpointer suff, gconstpointer name )
{
  char *a = (char *) suff;
  char *b = (char *) name;
  int n = strlen (a);
  int m = strlen (b);

  if( m < n )
    {
      return -1;
    }

  return (strcasecmp (a, b + m - n));
}

/* Check a prospective match against the list of current suffixes.
 */
static gboolean
check_match (const gchar *name, gpointer data)
{
  GtkFileSelection2 *fs = data;

  /* Suffix stuff turned off? Do nowt.
   */
  if (!fs->suffixes || !GTK_WIDGET_VISIBLE (fs->type_option_menu))
    return TRUE;

  /* Needs to have one of our postfixes.
   */
  return (g_list_find_custom (fs->suffixes, (void *) name, 
  		(GCompareFunc) is_postfix) != NULL);
}

static void
gtk_file_selection2_init (GtkFileSelection2 *fs)
{
  CompletionState *cmpl_state;

  cmpl_state = cmpl_init_state ();
  cmpl_state->predicate = check_match;
  cmpl_state->predicate_data = fs;
  fs->cmpl_state = cmpl_state;

  /* Init other info fields.
   */
  fs->file_types = NULL;
  fs->suffixes = NULL;
  fs->current_file = NULL;
  fs->current_dir = NULL;
  fs->multi = FALSE;
  fs->selected_names = NULL;
}

static void
gtk_file_selection2_build (GtkWidget *widget)
{
  GtkFileSelection2 *fs = GTK_FILE_SELECTION2 (widget);
  iDialog *idlg = IDIALOG (widget);

  GtkWidget *entry_vbox;
  GtkWidget *label;
  GtkWidget *list_hbox;
  GtkWidget *dir_vbox;
  GtkWidget *pulldown_hbox;
  GtkWidget *scrolled_win;
  GtkWidget *popwin;
  GtkWidget *work;

  char *dir_title [2];
  char *file_title [2];

  /* Call all builds in superclasses.
   */
  if( IWINDOW_CLASS( parent_class )->build )
    (*IWINDOW_CLASS( parent_class )->build)( widget );

  /* The dialog-sized vertical box  
   */ 
  work = idlg->work;

  /* The horizontal box containing create, rename etc. buttons 
   */
  fs->button_area = gtk_hbutton_box_new ();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(fs->button_area), GTK_BUTTONBOX_START);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(fs->button_area), 0);
  gtk_box_pack_start (GTK_BOX (work), fs->button_area, 
		      FALSE, FALSE, 0);
  gtk_widget_show (fs->button_area);

  gtk_file_selection2_show_fileop_buttons(fs);

  /* The label above the combo showing the current directory and error messages.
   */
  fs->path_text = label = gtk_label_new (_("Current:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (work), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  /* hbox for pulldown menu 
   */
  pulldown_hbox = gtk_hbox_new (TRUE, 5);
  gtk_box_pack_start (GTK_BOX (work), pulldown_hbox, FALSE, FALSE, 0);
  gtk_widget_show (pulldown_hbox);

  /* Pulldown menu */
  fs->history_pulldown = gtk_combo_new ();

  /* We change Tab to be name-complete, so user has to have arrows
   * to shift the focus about.
   */
  gtk_combo_set_use_arrows (GTK_COMBO (fs->history_pulldown), FALSE);

  /* Catch this signal to spot end of select in combo popdown. Not
   * really supposed to look at popwin ... :(
   */
  popwin = GTK_COMBO (fs->history_pulldown)->popwin;
  gtk_signal_connect (GTK_OBJECT (popwin), "hide",
		      GTK_SIGNAL_FUNC (gtk_file_selection2_dir_activate), 
		      (gpointer) fs);

  /* Use this signal to do our own interpretation of Tab and Return
   * for the combo.
   */
  fs->path_entry = GTK_COMBO (fs->history_pulldown)->entry;
  gtk_signal_connect (GTK_OBJECT (fs->path_entry), "key_press_event",
		      GTK_SIGNAL_FUNC (gtk_file_selection2_dir_key_press), 
		      (gpointer) fs);

  gtk_widget_show (fs->history_pulldown);
  gtk_box_pack_start (GTK_BOX (pulldown_hbox), fs->history_pulldown, 
		      TRUE, TRUE, 5);

  /*  The horizontal box containing the directory and file listboxes  
   */
  list_hbox = gtk_hbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX (work), list_hbox, TRUE, TRUE, 0);
  gtk_widget_show (list_hbox);

  /* vbox for file type and dir list.
   */
  dir_vbox = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (list_hbox), dir_vbox, TRUE, TRUE, 0);
  gtk_widget_show (dir_vbox);

  /* File type option menu.
   */
  fs->type_menu_pane = gtk_menu_new();
  fs->type_option_menu = gtk_option_menu_new();
  gtk_option_menu_set_menu (GTK_OPTION_MENU (fs->type_option_menu), 
	    fs->type_menu_pane);
  gtk_box_pack_start( GTK_BOX( dir_vbox ), fs->type_option_menu, FALSE, FALSE, 5 );

  /* The directories clist 
   */
  dir_title[0] = _("Directories");
  dir_title[1] = NULL;
  fs->dir_list = gtk_clist_new_with_titles (1, (gchar**) dir_title);
  gtk_widget_set_usize (fs->dir_list, DIR_LIST_WIDTH, DIR_LIST_HEIGHT);
  gtk_signal_connect (GTK_OBJECT (fs->dir_list), "select_row",
		      (GtkSignalFunc) gtk_file_selection2_dir_button, 
		      (gpointer) fs);
  gtk_clist_column_titles_passive (GTK_CLIST (fs->dir_list));

  scrolled_win = gtk_scrolled_window_new (NULL, NULL);
  gtk_container_add (GTK_CONTAINER (scrolled_win), fs->dir_list);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_win),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_container_set_border_width (GTK_CONTAINER (scrolled_win), 5);
  gtk_box_pack_start (GTK_BOX (dir_vbox), scrolled_win, TRUE, TRUE, 0);
  gtk_widget_show (fs->dir_list);
  gtk_widget_show (scrolled_win);

  /* The files clist */
  file_title[0] = _("Files");
  file_title[1] = NULL;
  fs->file_list = gtk_clist_new_with_titles (1, (gchar**) file_title);
  gtk_widget_set_usize (fs->file_list, FILE_LIST_WIDTH, FILE_LIST_HEIGHT);
  gtk_signal_connect (GTK_OBJECT (fs->file_list), "select_row",
		      (GtkSignalFunc) gtk_file_selection2_file_button, 
		      (gpointer) fs);
  gtk_clist_column_titles_passive (GTK_CLIST (fs->file_list));

  scrolled_win = gtk_scrolled_window_new (NULL, NULL);
  gtk_container_add (GTK_CONTAINER (scrolled_win), fs->file_list);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_win),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_container_set_border_width (GTK_CONTAINER (scrolled_win), 5);
  gtk_box_pack_start (GTK_BOX (list_hbox), scrolled_win, TRUE, TRUE, 0);
  gtk_widget_show (fs->file_list);
  gtk_widget_show (scrolled_win);

  if (fs->multi) 
    {
      gtk_clist_set_selection_mode (GTK_CLIST (fs->file_list), 
   			           GTK_SELECTION_EXTENDED);
      gtk_signal_connect (GTK_OBJECT (fs->file_list), "select_row",
		      (GtkSignalFunc) gtk_file_selection2_select_row, 
		      (gpointer) fs);
      gtk_signal_connect (GTK_OBJECT (fs->file_list), "unselect_row",
		      (GtkSignalFunc) gtk_file_selection2_unselect_row, 
		      (gpointer) fs);
    }

  /* action area for packing any child stuff into. */
  fs->work = gtk_vbox_new (FALSE, 3);
  gtk_box_pack_start (GTK_BOX (work), fs->work, FALSE, FALSE, 0);
  gtk_widget_show (fs->work);
  
  /*  The selection entry widget  */
  entry_vbox = gtk_vbox_new (FALSE, 2);
  gtk_box_pack_end (GTK_BOX (work), entry_vbox, FALSE, FALSE, 0);

  if (!fs->multi) 
    {
      gtk_widget_show (entry_vbox);
    }

  fs->selection_text = label = gtk_label_new (_("Selection:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (entry_vbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  fs->selection_entry = gtk_entry_new ();
  gtk_signal_connect (GTK_OBJECT (fs->selection_entry), "key_press_event",
		      (GtkSignalFunc) gtk_file_selection2_key_press, fs);
  gtk_box_pack_start (GTK_BOX (entry_vbox), fs->selection_entry, TRUE, TRUE, 0);
  gtk_widget_show (fs->selection_entry);

  if (!fs->multi) 
    {
      idialog_set_default_entry (idlg, GTK_ENTRY (fs->selection_entry));
    }
  else
    {
      gtk_widget_grab_focus (GTK_WIDGET (fs->file_list));
      idialog_set_ok_button_state (IDIALOG (fs), fs->selected_names != NULL);
    }

  if (!cmpl_state_okay (fs->cmpl_state))
    {
      gtk_file_selection2_abort (fs);
    }
  else
    {
      gtk_file_selection2_dir_enter (fs, "", FALSE);
    }
}

GtkWidget*
gtk_file_selection2_new (const gchar *title)
{
  GtkFileSelection2 *fs;

  fs = gtk_type_new (GTK_TYPE_FILE_SELECTION2);
  gtk_window_set_title (GTK_WINDOW (fs), title);

  return GTK_WIDGET (fs);
}

void
gtk_file_selection2_show_fileop_buttons (GtkFileSelection2 *fs)
{
  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));
    
  /* delete, create directory, and rename */
  if (!fs->fileop_c_dir) 
    {
      fs->fileop_c_dir = gtk_button_new_with_label (_("Create Dir"));
      gtk_signal_connect (GTK_OBJECT (fs->fileop_c_dir), "clicked",
			  (GtkSignalFunc) gtk_file_selection2_create_dir, 
			  (gpointer) fs);
      gtk_box_pack_start (GTK_BOX (fs->button_area), 
			  fs->fileop_c_dir, TRUE, TRUE, 0);
      gtk_widget_show (fs->fileop_c_dir);
    }
	
  if (!fs->fileop_del_file) 
    {
      fs->fileop_del_file = gtk_button_new_with_label (_("Delete File"));
      gtk_signal_connect (GTK_OBJECT (fs->fileop_del_file), "clicked",
			  (GtkSignalFunc) gtk_file_selection2_delete_file, 
			  (gpointer) fs);
      gtk_box_pack_start (GTK_BOX (fs->button_area), 
			  fs->fileop_del_file, TRUE, TRUE, 0);
      gtk_widget_show (fs->fileop_del_file);
    }

  if (!fs->fileop_ren_file)
    {
      fs->fileop_ren_file = gtk_button_new_with_label (_("Rename File"));
      gtk_signal_connect (GTK_OBJECT (fs->fileop_ren_file), "clicked",
			  (GtkSignalFunc) gtk_file_selection2_rename_file, 
			  (gpointer) fs);
      gtk_box_pack_start (GTK_BOX (fs->button_area), 
			  fs->fileop_ren_file, TRUE, TRUE, 0);
      gtk_widget_show (fs->fileop_ren_file);
    }

  gtk_widget_queue_resize(GTK_WIDGET(fs));
}

static void
set_new_file (GtkFileSelection2 *fs, const gchar *file)
{
  if (fs->selection_entry)
    {
      gtk_entry_set_text (GTK_ENTRY (fs->selection_entry), file);
    }

  if (!fs->current_file || strcmp (fs->current_file, file) != 0)
    {
      if (fs->current_file)
        {
	  g_free (fs->current_file);
	}
      fs->current_file = g_strdup (file);

      gtk_signal_emit (GTK_OBJECT (fs), filesel_signals[FILE_SELECT], 
      		       fs->current_file);
    }
}

gboolean
path_is_root( const char *path )
{
  return (strcmp (g_path_skip_root ((char *) path), "") == 0);
}

static void
set_dir_enter (GtkFileSelection2 *fs, const gchar *new)
{
  gchar *path;

  if (new[0] == '~' || new[0] == '$' || g_path_is_absolute (new))
    {
      /* We're trying to expand a tilda or an env var, don't add the 
       * base dir. If new is an absolute path, don't add the base dir.
       */
      path = (gchar*) new;
    }
  else if (path_is_root (cmpl_reference_position (fs->cmpl_state)))
    {
      /* If we're in root, make sure we don't add more.
       */
      path = g_strconcat (cmpl_reference_position (fs->cmpl_state), new, NULL);
    }
  else 
    {
      path = g_strconcat (cmpl_reference_position (fs->cmpl_state),
		IM_DIR_SEP_STR, new, NULL);
    }

  if (fs->path_entry)
    {
      gtk_entry_set_text (GTK_ENTRY (fs->path_entry), path);
    }

  if (!fs->current_dir || strcmp (fs->current_dir, path) != 0)
    {
      if (fs->current_dir)
        {
	  g_free (fs->current_dir);
	}
      fs->current_dir = g_strdup (path);
      gtk_signal_emit (GTK_OBJECT (fs), filesel_signals[DIR_ENTER], 
      		       fs->current_dir);
    }

  if (path != new)
    {
      g_free (path);
    }
}

/* Set a new suffix set. Mangle the entered filename as required.
 */
static void
set_new_suffix (GtkFileSelection2 *fs, GList *new_suffixes)
{
  /* Don't mangle if there's no filename entered.
   */
  if (fs->selection_entry && strlen (gtk_entry_get_text
	  (GTK_ENTRY (fs->selection_entry))) > 0)
    {
      gchar *filename = g_strdup (gtk_entry_get_text 
		(GTK_ENTRY (fs->selection_entry)));
      GList *q;

      /* If we're switching to 'all', don't chop the existing suffix.
       */
      if (new_suffixes && new_suffixes->data && 
		strlen (new_suffixes->data) > 0)
	{
          /* Chop off the current suffix, if any. Look for any suffix in the
           * file_types we know about.
           */
          for (q = fs->file_types; q; q = q->next)
	    {
	      GList *p = q->data;

              if (p && p->data && strlen (p->data) > 0 &&
	      		(p = g_list_find_custom (p, (void *) filename, 
  		    		(GCompareFunc) is_postfix)))
                {
	          filename[strlen (filename) - strlen (p->data)] = '\0';
	          break;
	        }
	    }
	}

      /* If it does not already have one of the new suffixes, append the 1st
       * one.
       */
      if (new_suffixes && !g_list_find_custom 
		(new_suffixes, (void *) filename, (GCompareFunc) is_postfix))
        {
          gchar *filename2 = g_strconcat (filename, new_suffixes->data, NULL);
	  
	  g_free (filename);
	  filename = filename2;
	}

      /* Write back to entry.
       */
      set_new_file (fs, filename);

      g_free (filename);
    }

  fs->suffixes = new_suffixes;
}

/* Activate callback in file type option.
 */
static void
set_type_cb (GtkWidget *wid, gpointer data)
{
  GtkFileSelection2 *fs = data;
  int n = (int) gtk_object_get_data_by_id (GTK_OBJECT (wid), 
  						   quark_suffix);

  gtk_file_selection2_set_file_type (fs, n);
}

/* Update the file type option menu with a new set of file types.
 */
void
gtk_file_selection2_set_file_types (GtkFileSelection2 *fs, 
				    GtkFileSelection2FileType *types[])
{
  GList *p, *q;
  int i, j;

  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));

  /* Junk the old menu and file type suffix lists.
   */
  if (fs->type_menu_pane)
    {
      gtk_widget_destroy (fs->type_menu_pane);
      fs->type_menu_pane = NULL;
    }
  for (p = fs->file_types; p; p = p->next)
    {
      for (q = p->data; q; q = q->next)
	{
          g_free (q->data);
	}
      g_list_free (p->data);
    }
  g_list_free (fs->file_types);
  fs->file_types = NULL;
  fs->suffixes = NULL;

  /* Build the new one!
   */
  fs->type_menu_pane = gtk_menu_new();

  if (types)
    {
      for (i = 0; types[i]; i++ )
        {
          GtkWidget *item = gtk_menu_item_new_with_label (types[i]->name);
          GList *suffixes;
    
          /* Take a copy of the suffix list.
           */
          for (suffixes = NULL, j = 0; types[i]->suffixes[j]; j++)
            {
              suffixes = g_list_append (suffixes, 
			g_strdup (types[i]->suffixes[j]));
	    }
    
          /* Save the pattern index inside the item.
           */
          if (!quark_suffix)
	    quark_suffix = g_quark_from_static_string ("GtkFileSelection2_suf");
          gtk_object_set_data_by_id (GTK_OBJECT (item), quark_suffix, 
	  	(void *) i);
          fs->file_types = g_list_append (fs->file_types, suffixes);
          gtk_signal_connect (GTK_OBJECT (item), "activate",
	  		    GTK_SIGNAL_FUNC (set_type_cb), fs);
          gtk_menu_append (GTK_MENU (fs->type_menu_pane), item);
          gtk_widget_show (item);
        }
    }
    
  gtk_option_menu_set_menu (GTK_OPTION_MENU (fs->type_option_menu), 
	    fs->type_menu_pane);
  gtk_option_menu_set_history (GTK_OPTION_MENU (fs->type_option_menu), 0);
  if (fs->file_types)
    {
      fs->suffixes = fs->file_types->data;
    }

  gtk_file_selection2_dir_enter (fs, "", FALSE);
}

void
gtk_file_selection2_set_file_type (GtkFileSelection2 *fs, int n)
{
  GList *suffixes;

  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));
  g_return_if_fail (fs->file_types);
  g_return_if_fail (n >= 0 && n < (int) g_list_length (fs->file_types));

  suffixes = g_list_nth_data (fs->file_types, n);
  if (fs->suffixes == suffixes)
    {
      return;
    }

  if (fs->type_option_menu)
    {
      gtk_option_menu_set_history (GTK_OPTION_MENU (fs->type_option_menu), n);
    }

  set_new_suffix (fs, suffixes);
  gtk_file_selection2_refresh (fs);
}

gchar*
gtk_file_selection2_get_file_type (GtkFileSelection2 *fs)
{
  g_return_val_if_fail (fs != NULL, NULL);
  g_return_val_if_fail (GTK_IS_FILE_SELECTION2 (fs), NULL);

  if (!fs->suffixes)
    {
      return ("");
    }

  return (fs->suffixes->data);
}

/* Search for the file type that matches this filename.
 */
static int
find_file_type (GtkFileSelection2 *fs, const char *filename)
{
  GList *q;
  int i;

  for (i = 0, q = fs->file_types; q; q = q->next, i++)
    {
      GList *p = q->data;

      if (p && p->data && strlen (p->data) > 0 &&
		(p = g_list_find_custom (p, (void *) filename, 
			(GCompareFunc) is_postfix)))
	{
	  return i;
	}
    }

  return -1;
}

/* If the text in the filebox does not have any recognised suffix, and if
 * we're not set to "all", append the 1st of the current set.
 */
static void
set_suffix (GtkFileSelection2 *fs)
{
  char *filename;
  int i;

  /* Only if we're not displaying "all".
   */
  if (fs->selection_entry && 
  	fs->suffixes &&
	fs->suffixes->data &&
	strlen (fs->suffixes->data) > 0 &&
	(filename = gtk_entry_get_text (GTK_ENTRY (fs->selection_entry))) &&
  	strlen (filename) > 0 &&
	(i = find_file_type (fs, filename)) < 0)
     {
	char *filename2 = g_strconcat (filename, fs->suffixes->data, NULL);

	set_new_file (fs, filename2);

	g_free (filename2);
      }
}

static void
gtk_file_selection2_set_file_type_from_filename (GtkFileSelection2 *fs, 
		      				const char *name)
{
  int i;
  char *p;

  /* If we're showing "all", any filename is OK, so don't change the file
   * type.
   */
  if (g_list_index (fs->file_types, fs->suffixes) ==
      (int) g_list_length (fs->file_types) - 1)
    {
      return;
    }

  /* If we've not got a sensible filename, don't bother.
   */
  if ((p = strrchr (name, IM_DIR_SEP)) && 
      strspn (p + 1, " \n\t") == strlen (p + 1)) 
    {
      return;
    }

  if ((i = find_file_type (fs, name)) >= 0)
    {
      gtk_file_selection2_set_file_type (fs, i);
      return;
    }

  /* No match, or no suffix. Set the last type (should be "All").
   */
  i = g_list_length (fs->file_types);
  if (i > 0)
    {
      gtk_file_selection2_set_file_type (fs, i - 1);
    }
}

void
gtk_file_selection2_set_filter (GtkFileSelection2 *fs,
			        GtkFileSelection2FilenamePredicate predicate,
			        gpointer predicate_data)
{
  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));

  if (fs->cmpl_state)
    {
      CompletionState *cmpl_state = fs->cmpl_state;

      cmpl_state->predicate = predicate;
      cmpl_state->predicate_data = predicate_data;
      gtk_file_selection2_dir_enter (fs, "", FALSE);
    }
}

void       
gtk_file_selection2_hide_file_types (GtkFileSelection2 *fs)
{
  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));
    
  if (fs->type_option_menu) 
    {
      gtk_widget_hide (fs->type_option_menu);
      gtk_file_selection2_dir_enter (fs, "", FALSE);
    }
}

void       
gtk_file_selection2_show_file_types (GtkFileSelection2 *fs)
{
  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));
    
  if (fs->type_option_menu) 
    {
      gtk_widget_show (fs->type_option_menu);
      gtk_file_selection2_dir_enter (fs, "", FALSE);
    }
}

void       
gtk_file_selection2_hide_fileop_buttons (GtkFileSelection2 *fs)
{
  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));
    
  if (fs->fileop_ren_file) 
    {
      gtk_widget_destroy (fs->fileop_ren_file);
      fs->fileop_ren_file = NULL;
    }

  if (fs->fileop_del_file)
    {
      gtk_widget_destroy (fs->fileop_del_file);
      fs->fileop_del_file = NULL;
    }

  if (fs->fileop_c_dir)
    {
      gtk_widget_destroy (fs->fileop_c_dir);
      fs->fileop_c_dir = NULL;
    }
}

void
gtk_file_selection2_set_filename (GtkFileSelection2 *fs,
				 const gchar      *filename)
{
  char  buf[MAXPATHLEN];
  const char *name, *last_slash;
  char *text;
  int n;

  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));
  g_return_if_fail (filename != NULL);

  last_slash = strrchr (filename, IM_DIR_SEP);

  if (!last_slash)
    {
      buf[0] = 0;
      name = filename;
    }
  else
    {
      gint len = IM_MIN (MAXPATHLEN - 1, last_slash - filename + 1);

      strncpy (buf, filename, len);
      buf[len] = 0;

      name = last_slash + 1;
    }

  gtk_file_selection2_dir_enter (fs, buf, FALSE);

  if (fs->selection_entry)
    {
      set_new_file (fs, name);
      set_new_suffix (fs, fs->suffixes);
    }

  /* Try to select the clist row for this filename.
   */
  for (n = 0; n < GTK_CLIST (fs->file_list)->rows; n++)
    {
      if (gtk_clist_get_text (GTK_CLIST (fs->file_list), n, 0, &text) &&
          strcmp (text, name) == 0 )
	{
	  gtk_clist_select_row (GTK_CLIST (fs->file_list), n, 0);
	  gtk_clist_moveto (GTK_CLIST (fs->file_list), n, 0, 0.5, 0.5);
	}
    }
}

void
gtk_file_selection2_set_multi (GtkFileSelection2 *fs, 
			      gboolean            multi)
{
  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));

  fs->multi = multi;
}

gchar*
gtk_file_selection2_get_filename (GtkFileSelection2 *fs)
{
  static char nothing[2] = "";
  char *text;

  g_return_val_if_fail (fs != NULL, nothing);
  g_return_val_if_fail (GTK_IS_FILE_SELECTION2 (fs), nothing);

  /* Add a suffix, if there isn't one and there should be.
   */
  set_suffix (fs);

  text = gtk_entry_get_text (GTK_ENTRY (fs->selection_entry));
  if (text)
    {
      text = cmpl_completion_fullname (text, fs->cmpl_state);

      /* Only mess with filetypes if we're not showing "all".
       */
      if (fs->suffixes)
        {
          gtk_file_selection2_set_file_type_from_filename (fs, text);
	}

      return text;
    }

  return nothing;
}

static void
gtk_file_selection2_unselect (GtkFileSelection2 *fs)
{
  if (fs->selected_names)
    {
      g_slist_free (fs->selected_names);
      fs->selected_names = NULL;     
      idialog_set_ok_button_state (IDIALOG (fs), FALSE);
    }
}

GSList*     
gtk_file_selection2_get_filename_multi (GtkFileSelection2 *fs)
{
  GSList *names, *p;

  g_return_val_if_fail (fs != NULL, NULL);
  g_return_val_if_fail (GTK_IS_FILE_SELECTION2 (fs), NULL);
  g_return_val_if_fail (fs->multi, NULL);

  for (names = NULL, p = fs->selected_names; p; p = p->next)
    {
    	char *text = (char *) p->data;
	char *full_name = cmpl_completion_fullname (text, fs->cmpl_state);

	names = g_slist_prepend (names, g_strdup (full_name));
    }

#ifdef DEBUG
  {
	printf( "gtk_file_selection2_get_filename_multi: returning:\n" );
	for( p = names; p; p = p->next )
		printf( "- \"%s\"\n", (char *) p->data );
  }
#endif /*DEBUG*/

  return (names);
}

void
gtk_file_selection2_complete (GtkFileSelection2 *fs,
			     const gchar      *pattern)
{
  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));
  g_return_if_fail (pattern != NULL);

  if (fs->selection_entry)
    {
      set_new_file (fs, pattern);
    }
  gtk_file_selection2_dir_enter (fs, (gchar*) pattern, TRUE);
}

void
gtk_file_selection2_refresh (GtkFileSelection2 *fs)
{
  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));

  gtk_file_selection2_dir_enter (fs, "", FALSE);
  gtk_signal_emit (GTK_OBJECT (fs), filesel_signals[REFRESH]);
}

static void
gtk_file_selection2_destroy (GtkObject *object)
{
  GtkFileSelection2 *fs;
  GList *list, *list2;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (object));

  fs = GTK_FILE_SELECTION2 (object);
  
  if (fs->fileop_dialog)
    gtk_widget_destroy (fs->fileop_dialog);
  
  cmpl_free_state (fs->cmpl_state);
  fs->cmpl_state = NULL;

  if (fs->history_list)
    {
      for (list = fs->history_list; list; list = list->next)
	{
	  g_free (list->data);
	}
      g_list_free (fs->history_list);
      fs->history_list = NULL;
    }
  
  for (list = fs->file_types; list; list = list->next)
    {
      for (list2 = list->data; list2; list2 = list2->next)
	{
          g_free (list2->data);
	}
      g_list_free (list->data);
    }
  g_list_free (fs->file_types);
  fs->file_types = NULL;
  fs->suffixes = NULL;

  if (fs->current_file)
    {
      g_free (fs->current_file);
    }
  if (fs->current_dir)
    {
      g_free (fs->current_dir);
    }

  gtk_file_selection2_unselect (fs);

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

/* Begin file operations callbacks */

static void
gtk_file_selection2_fileop_error (gchar *error_message)
{
  GtkWidget *label;
  GtkWidget *vbox;
  GtkWidget *button;
  GtkWidget *dialog;
  
  g_return_if_fail (error_message != NULL);
  
  /* main dialog */
  dialog = gtk_dialog_new ();
  /*
  gtk_signal_connect (GTK_OBJECT (dialog), "destroy",
		      (GtkSignalFunc) gtk_file_selection2_fileop_destroy, 
		      (gpointer) fs);
  */
  gtk_window_set_title (GTK_WINDOW (dialog), _("Error"));
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
  
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), 8);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), vbox,
		     FALSE, FALSE, 0);
  gtk_widget_show(vbox);

  label = gtk_label_new(error_message);
  gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 5);
  gtk_widget_show(label);

  /* yes, we free it */
  g_free (error_message);
  
  /* close button */
  button = gtk_button_new_with_label (_("Close"));
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     (GtkSignalFunc) gtk_widget_destroy, 
			     (gpointer) dialog);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),
		     button, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default(button);
  gtk_widget_show (button);

  gtk_widget_show (dialog);
}

static void
gtk_file_selection2_fileop_destroy (GtkWidget *widget, gpointer data)
{
  GtkFileSelection2 *fs = data;

  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));
  
  fs->fileop_dialog = NULL;
}


static void
gtk_file_selection2_create_dir_confirmed (GtkWidget *widget, gpointer data)
{
  GtkFileSelection2 *fs = data;
  gchar *dirname;
  gchar *path;
  gchar *full_path;
  gchar *buf;
  CompletionState *cmpl_state;
  
  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));

  dirname = gtk_entry_get_text (GTK_ENTRY (fs->fileop_entry));
  cmpl_state = (CompletionState*) fs->cmpl_state;
  path = cmpl_reference_position (cmpl_state);
  
  full_path = g_strconcat (path, IM_DIR_SEP_STR, dirname, NULL);
  if ( (mkdir (full_path, 0755) < 0) ) 
    {
      buf = g_strconcat ("Error creating directory \"", dirname, "\":  ", 
			 g_strerror(errno), NULL);
      gtk_file_selection2_fileop_error (buf);
    }
  g_free (full_path);
  
  gtk_widget_destroy (fs->fileop_dialog);
  gtk_file_selection2_dir_enter (fs, "", FALSE);
}
  
static void
gtk_file_selection2_create_dir (GtkWidget *widget, gpointer data)
{
  GtkFileSelection2 *fs = data;
  GtkWidget *label;
  GtkWidget *dialog;
  GtkWidget *vbox;
  GtkWidget *button;

  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));

  if (fs->fileop_dialog)
	  return;
  
  /* main dialog */
  fs->fileop_dialog = dialog = gtk_dialog_new ();
  gtk_signal_connect (GTK_OBJECT (dialog), "destroy",
		      (GtkSignalFunc) gtk_file_selection2_fileop_destroy, 
		      (gpointer) fs);
  gtk_window_set_title (GTK_WINDOW (dialog), _("Create Directory"));
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);

  /* If file dialog is grabbed, grab option dialog */
  /* When option dialog is closed, file dialog will be grabbed again */
  if (GTK_WINDOW(fs)->modal)
      gtk_window_set_modal (GTK_WINDOW(dialog), TRUE);

  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), 8);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), vbox,
		     FALSE, FALSE, 0);
  gtk_widget_show(vbox);
  
  label = gtk_label_new(_("Directory name:"));
  gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 5);
  gtk_widget_show(label);

  /*  The directory entry widget  */
  fs->fileop_entry = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (vbox), fs->fileop_entry, 
		      TRUE, TRUE, 5);
  GTK_WIDGET_SET_FLAGS(fs->fileop_entry, GTK_CAN_DEFAULT);
  gtk_widget_show (fs->fileop_entry);
  
  /* buttons */
  button = gtk_button_new_with_label (_("Create"));
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      (GtkSignalFunc) gtk_file_selection2_create_dir_confirmed, 
		      (gpointer) fs);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),
		     button, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
  gtk_widget_show(button);
  
  button = gtk_button_new_with_label (_("Cancel"));
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     (GtkSignalFunc) gtk_widget_destroy, 
			     (gpointer) dialog);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),
		     button, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default(button);
  gtk_widget_show (button);

  gtk_widget_show (dialog);
}

static void
gtk_file_selection2_delete_file_confirmed (GtkWidget *widget, gpointer data)
{
  GtkFileSelection2 *fs = data;
  CompletionState *cmpl_state;
  gchar *path;
  gchar *full_path;
  gchar *buf;
  
  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));

  cmpl_state = (CompletionState*) fs->cmpl_state;
  path = cmpl_reference_position (cmpl_state);
  
  full_path = g_strconcat (path, IM_DIR_SEP_STR, fs->fileop_file, NULL);
  if ( (unlink (full_path) < 0) ) 
    {
      buf = g_strconcat ("Error deleting file \"", fs->fileop_file, "\":  ", 
			 g_strerror(errno), NULL);
      gtk_file_selection2_fileop_error (buf);
    }
  g_free (full_path);
  
  gtk_widget_destroy (fs->fileop_dialog);
  gtk_file_selection2_dir_enter (fs, "", FALSE);
}

static void
gtk_file_selection2_delete_file (GtkWidget *widget, gpointer data)
{
  GtkFileSelection2 *fs = data;
  GtkWidget *label;
  GtkWidget *vbox;
  GtkWidget *button;
  GtkWidget *dialog;
  gchar *filename;
  gchar *buf;
  
  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));

  if (fs->fileop_dialog)
	  return;

  filename = gtk_entry_get_text (GTK_ENTRY (fs->selection_entry));
  if (strlen(filename) < 1)
	  return;

  fs->fileop_file = filename;
  
  /* main dialog */
  fs->fileop_dialog = dialog = gtk_dialog_new ();
  gtk_signal_connect (GTK_OBJECT (dialog), "destroy",
		      (GtkSignalFunc) gtk_file_selection2_fileop_destroy, 
		      (gpointer) fs);
  gtk_window_set_title (GTK_WINDOW (dialog), _("Delete File"));
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);

  /* If file dialog is grabbed, grab option dialog */
  /* When option dialog is closed, file dialog will be grabbed again */
  if (GTK_WINDOW(fs)->modal)
      gtk_window_set_modal (GTK_WINDOW(dialog), TRUE);
  
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), 8);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), vbox,
		     FALSE, FALSE, 0);
  gtk_widget_show(vbox);

  buf = g_strconcat ("Really delete file \"", filename, "\" ?", NULL);
  label = gtk_label_new(buf);
  gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 5);
  gtk_widget_show(label);
  g_free(buf);
  
  /* buttons */
  button = gtk_button_new_with_label (_("Delete"));
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      (GtkSignalFunc) gtk_file_selection2_delete_file_confirmed, 
		      (gpointer) fs);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),
		     button, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
  gtk_widget_show(button);
  
  button = gtk_button_new_with_label (_("Cancel"));
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     (GtkSignalFunc) gtk_widget_destroy, 
			     (gpointer) dialog);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),
		     button, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default(button);
  gtk_widget_show (button);

  gtk_widget_show (dialog);

}

static void
gtk_file_selection2_rename_file_confirmed (GtkWidget *widget, gpointer data)
{
  GtkFileSelection2 *fs = data;
  gchar *buf;
  gchar *file;
  gchar *path;
  gchar *new_filename;
  gchar *old_filename;
  CompletionState *cmpl_state;
  
  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));

  file = gtk_entry_get_text (GTK_ENTRY (fs->fileop_entry));
  cmpl_state = (CompletionState*) fs->cmpl_state;
  path = cmpl_reference_position (cmpl_state);
  
  new_filename = g_strconcat (path, IM_DIR_SEP_STR, file, NULL);
  old_filename = g_strconcat (path, IM_DIR_SEP_STR, fs->fileop_file, NULL);

  if ( (rename (old_filename, new_filename)) < 0) 
    {
      buf = g_strconcat ("Error renaming file \"", file, "\":  ", 
			 g_strerror(errno), NULL);
      gtk_file_selection2_fileop_error (buf);
    }
  g_free (new_filename);
  g_free (old_filename);
  
  gtk_widget_destroy (fs->fileop_dialog);
  gtk_file_selection2_dir_enter (fs, "", FALSE);
}
  
static void
gtk_file_selection2_rename_file (GtkWidget *widget, gpointer data)
{
  GtkFileSelection2 *fs = data;
  GtkWidget *label;
  GtkWidget *dialog;
  GtkWidget *vbox;
  GtkWidget *button;
  gchar *buf;
  
  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));

  if (fs->fileop_dialog)
	  return;

  fs->fileop_file = gtk_entry_get_text (GTK_ENTRY (fs->selection_entry));
  if (strlen(fs->fileop_file) < 1)
	  return;
  
  /* main dialog */
  fs->fileop_dialog = dialog = gtk_dialog_new ();
  gtk_signal_connect (GTK_OBJECT (dialog), "destroy",
		      (GtkSignalFunc) gtk_file_selection2_fileop_destroy, 
		      (gpointer) fs);
  gtk_window_set_title (GTK_WINDOW (dialog), _("Rename File"));
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);

  /* If file dialog is grabbed, grab option dialog */
  /* When option dialog  closed, file dialog will be grabbed again */
  if (GTK_WINDOW(fs)->modal)
    gtk_window_set_modal (GTK_WINDOW(dialog), TRUE);
  
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_set_border_width (GTK_CONTAINER(vbox), 8);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), vbox,
		     FALSE, FALSE, 0);
  gtk_widget_show(vbox);
  
  buf = g_strconcat ("Rename file \"", fs->fileop_file, "\" to:", NULL);
  label = gtk_label_new(buf);
  gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 5);
  gtk_widget_show(label);
  g_free(buf);

  /* New filename entry */
  fs->fileop_entry = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (vbox), fs->fileop_entry, 
		      TRUE, TRUE, 5);
  GTK_WIDGET_SET_FLAGS(fs->fileop_entry, GTK_CAN_DEFAULT);
  gtk_widget_show (fs->fileop_entry);
  
  gtk_entry_set_text (GTK_ENTRY (fs->fileop_entry), fs->fileop_file);
  gtk_editable_select_region (GTK_EDITABLE (fs->fileop_entry),
			      0, strlen (fs->fileop_file));

  /* buttons */
  button = gtk_button_new_with_label (_("Rename"));
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      (GtkSignalFunc) gtk_file_selection2_rename_file_confirmed, 
		      (gpointer) fs);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),
		     button, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
  gtk_widget_show(button);
  
  button = gtk_button_new_with_label (_("Cancel"));
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     (GtkSignalFunc) gtk_widget_destroy, 
			     (gpointer) dialog);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),
		     button, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default(button);
  gtk_widget_show (button);

  gtk_widget_show (dialog);
}

/* Make sure we have a trailing IM_DIR_SEP character to avoid confusing 
 * filecomplete.  Caller to free result.
 */
static gchar *
ensure_trailing_slash (const gchar *dir)
{
  int len = strlen (dir);

  if (len == 0 || dir[len - 1] != IM_DIR_SEP)
    return (g_strconcat (dir, IM_DIR_SEP_STR, NULL));
  else
    return (g_strdup (dir));
}

/* Read the path from the combo, and jump to it.
 */
static void
goto_dir (GtkFileSelection2 *fs, gint complete)
{
  char *dir = gtk_entry_get_text (GTK_ENTRY (fs->path_entry));
  gchar *txt;

  if (!complete)
    {
      txt = ensure_trailing_slash (dir);
    }
  else
    {
      txt = g_strdup (dir);
    }
  gtk_file_selection2_dir_enter (fs, txt, complete);
  g_free (txt);
}

/* key-press-event in path entry.
 */
static gint
gtk_file_selection2_dir_key_press (GtkWidget   *widget,
			      GdkEventKey *event,
			      gpointer     user_data)
{
  GtkFileSelection2 *fs;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  if (event->keyval == GDK_Tab || event->keyval == GDK_Return)
    {
      fs = GTK_FILE_SELECTION2 (user_data);

      goto_dir (fs, event->keyval == GDK_Tab);

      gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "key_press_event");

      return TRUE;
    }

  return FALSE;
}

/* Something selected in the combo popdown list.
 */
static void 
gtk_file_selection2_dir_activate  (GtkWidget             *widget,
			      gpointer               user_data)
{
  GtkFileSelection2 *fs = user_data;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));

  goto_dir (fs, FALSE);
}

static gint
gtk_file_selection2_key_press (GtkWidget   *widget,
			      GdkEventKey *event,
			      gpointer     user_data)
{
  GtkFileSelection2 *fs = user_data;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (event != NULL, FALSE);
  g_return_val_if_fail (fs != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_FILE_SELECTION2 (fs), FALSE);

  if (event->keyval == GDK_Tab)
    {
      char *text = gtk_entry_get_text (GTK_ENTRY (fs->selection_entry));

      text = g_strdup (text);

      gtk_file_selection2_new_file (fs, text, TRUE);

      g_free (text);

      gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "key_press_event");

      return TRUE;
    }

  return FALSE;
}

/* We've entered a new directory. If it's not been seen before, add it 
 * to the history list. 
 */
void 
gtk_file_selection2_update_history_menu (GtkFileSelection2 *fs,
					const gchar *dir)
{
  GtkCombo *combo;
  gchar *txt;

  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));
  g_return_if_fail (dir != NULL);

  if (!fs->history_pulldown)
    return;

  txt = ensure_trailing_slash (dir);
  if( g_list_find_custom (fs->history_list, txt, (GCompareFunc) strcmp) ) 
    {
      g_free (txt);
      return;
    }
  fs->history_list = g_list_append (fs->history_list, txt);

  /* Woah, gross, we have to stop gtk_combo updating the entry when we
   * add the new popdown string.
   */
  combo = GTK_COMBO (fs->history_pulldown);
  gtk_signal_handler_block (GTK_OBJECT (combo->list), combo->list_change_id);
  gtk_combo_set_popdown_strings (combo, fs->history_list);
  gtk_signal_handler_unblock (GTK_OBJECT (combo->list), combo->list_change_id);
}

static void
gtk_file_selection2_file_button (GtkWidget *widget,
			       gint row, 
			       gint column, 
			       GdkEventButton *bevent,
			       gpointer user_data)
{
  GtkFileSelection2 *fs = NULL;
  gchar *filename, *temp = NULL;
  
  g_return_if_fail (GTK_IS_CLIST (widget));

  fs = user_data;
  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));
  
  gtk_clist_get_text (GTK_CLIST (fs->file_list), row, 0, &temp);
  filename = g_strdup (temp);

  if (filename)
    {
      if (bevent)
	switch (bevent->type)
	  {
	  case GDK_2BUTTON_PRESS:
	    idialog_done_trigger( IDIALOG( fs ), 0 );
	    break;

	  default:
	    set_new_file (fs, filename);
	    break;
	  }
      else
	set_new_file (fs, filename);

      g_free (filename);
    }
}

static void
gtk_file_selection2_select_row (GtkWidget *widget,
			        gint row, 
			        gint column, 
			        GdkEventButton *bevent,
			        gpointer user_data)
{
  GtkFileSelection2 *fs = NULL;
  gchar *filename;
  
  g_return_if_fail (GTK_IS_CLIST (widget));

  fs = user_data;
  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));
  
  gtk_clist_get_text (GTK_CLIST (fs->file_list), row, 0, &filename);

  if (!g_slist_find (fs->selected_names, filename))
	  fs->selected_names = g_slist_prepend( fs->selected_names, filename );

  idialog_set_ok_button_state (IDIALOG (fs), TRUE);

#ifdef DEBUG
  {
  	GSList *p;

	printf( "gtk_file_selection2_select_row: selected rows now:\n" );
	for( p = fs->selected_names; p; p = p->next )
		printf( "- \"%s\"\n", (char *) p->data );
  }
#endif /*DEBUG*/
}

static void
gtk_file_selection2_unselect_row (GtkWidget *widget,
			          gint row, 
			          gint column, 
			          GdkEventButton *bevent,
			          gpointer user_data)
{
  GtkFileSelection2 *fs = NULL;
  gchar *filename;
  
  g_return_if_fail (GTK_IS_CLIST (widget));

  fs = user_data;
  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));
  
  gtk_clist_get_text (GTK_CLIST (fs->file_list), row, 0, &filename);

  fs->selected_names = g_slist_remove( fs->selected_names, filename );

  if (!fs->selected_names)
    {
      idialog_set_ok_button_state (IDIALOG (fs), FALSE);
    }

#ifdef DEBUG
  {
  	GSList *p;

	printf( "gtk_file_selection2_unselect_row: selected rows now:\n" );
	for( p = fs->selected_names; p; p = p->next )
		printf( "- \"%s\"\n", (char *) p->data );
  }
#endif /*DEBUG*/
}

static void
gtk_file_selection2_dir_button (GtkWidget *widget,
			       gint row, 
			       gint column, 
			       GdkEventButton *bevent,
			       gpointer user_data)
{
  GtkFileSelection2 *fs = NULL;
  gchar *dirname, *temp = NULL;

  g_return_if_fail (GTK_IS_CLIST (widget));

  fs = GTK_FILE_SELECTION2 (user_data);
  g_return_if_fail (fs != NULL);
  g_return_if_fail (GTK_IS_FILE_SELECTION2 (fs));

  gtk_clist_get_text (GTK_CLIST (fs->dir_list), row, 0, &temp);
  dirname = g_strdup (temp);

  if (dirname)
    {
      if (bevent)
        switch (bevent->type)
	  {
	  case GDK_2BUTTON_PRESS:
	    gtk_file_selection2_dir_enter (fs, dirname, FALSE);
	    break;

	  default:
	    break;
          }
      else
	gtk_file_selection2_dir_enter (fs, dirname, FALSE);

      g_free (dirname);
    }
}

#ifdef HAVE_WINDOWS_H
static void
win32_gtk_add_drives_to_dir_list(GtkWidget *the_dir_list)
{
  gchar *text[2], *textPtr;
  gchar buffer[128];
  char volumeNameBuf[128];
  char formatBuffer[128];
  gint row;

  text[1] = NULL;

  /* Get the Drives string */
  GetLogicalDriveStrings(sizeof(buffer), buffer);

  /* Add the drives as necessary */
  textPtr = buffer;
  while (*textPtr != '\0') {
    /* Get the volume information for this drive */
    if ((tolower(textPtr[0]) != 'a') && (tolower(textPtr[0]) != 'b'))
      {
       /* Ignore floppies (?) */
       DWORD maxComponentLength, flags;

       GetVolumeInformation(textPtr,
                            volumeNameBuf, sizeof(volumeNameBuf),
                            NULL, &maxComponentLength,
                            &flags, NULL, 0);
       /* Build the actual displayable string */

       sprintf(formatBuffer, "%c:\\", toupper(textPtr[0]));
       /* Add to the list */
       text[0] = formatBuffer;
       row = gtk_clist_append (GTK_CLIST (the_dir_list), text);
      }
    textPtr += (strlen(textPtr) + 1);
  }
}
#endif

/* Populate the dir and file lists for a path. Return a pointer to the start 
 * of the unused section in the path, or NULL for error. 
 */
static const char *
refresh_lists (GtkFileSelection2 *fs,
		const char *path)
{
  CompletionState *cmpl_state;
  PossibleCompletion* poss;
  const char* rem_path = path;
  gchar* filename;
  gint row;
  gchar* text[2];
  gint possible_count = 0;
  gint file_list_width;
  gint dir_list_width;
  
  g_return_val_if_fail (fs != NULL, NULL);
  g_return_val_if_fail (GTK_IS_FILE_SELECTION2 (fs), NULL);

  cmpl_state = (CompletionState*) fs->cmpl_state;
  poss = cmpl_completion_matches (path, &rem_path, cmpl_state);

  if (!cmpl_state_okay (cmpl_state))
    {
      return NULL;
    }
  g_assert (cmpl_state->reference_dir);

  gtk_clist_freeze (GTK_CLIST (fs->dir_list));
  gtk_clist_clear (GTK_CLIST (fs->dir_list));
  gtk_clist_freeze (GTK_CLIST (fs->file_list));
  gtk_clist_clear (GTK_CLIST (fs->file_list));
  gtk_file_selection2_unselect (fs);

  /* Set the dir_list to include ./ and ../, provided we're not showing
   * "~xx" or "$xx".
   */
  text[1] = NULL;
  if (cmpl_updated_text (fs->cmpl_state)[0] != '~' &&
      cmpl_updated_text (fs->cmpl_state)[0] != '$')
    {
      text[0] = "." IM_DIR_SEP_STR;
      row = gtk_clist_append (GTK_CLIST (fs->dir_list), text);
      text[0] = ".." IM_DIR_SEP_STR;
      row = gtk_clist_append (GTK_CLIST (fs->dir_list), text);
    }

  /* Reset the max widths of the lists
   */
  dir_list_width = gdk_string_width(fs->dir_list->style->font,".." IM_DIR_SEP_STR);
  gtk_clist_set_column_width(GTK_CLIST(fs->dir_list),0,dir_list_width);
  file_list_width = 1;
  gtk_clist_set_column_width(GTK_CLIST(fs->file_list),0,file_list_width);

  while (poss)
    {
      if (cmpl_is_a_completion (poss))
        {
          possible_count += 1;

          filename = cmpl_this_completion (poss);

	  text[0] = filename;
	  
          if (cmpl_is_directory (poss))
            {
              if (strcmp (filename, "." IM_DIR_SEP_STR) != 0 &&
                  strcmp (filename, ".." IM_DIR_SEP_STR) != 0)
		{
		  int width = gdk_string_width(fs->dir_list->style->font,
					       filename);
		  row = gtk_clist_append (GTK_CLIST (fs->dir_list), text);
		  if(width > dir_list_width)
		    {
		      dir_list_width = width;
		      gtk_clist_set_column_width(GTK_CLIST(fs->dir_list),0,
						 width);
		    }
 		}
	    }
          else
	    {
	      int width = gdk_string_width(fs->file_list->style->font,
				           filename);
	      row = gtk_clist_append (GTK_CLIST (fs->file_list), text);
	      if(width > file_list_width)
	        {
	          file_list_width = width;
	          gtk_clist_set_column_width(GTK_CLIST(fs->file_list),0,
					     width);
	        }
            }
	}

      poss = cmpl_next_completion (cmpl_state);
    }

#ifdef HAVE_WINDOWS_H
  /* For Windows, add drives as potential selections */
  win32_gtk_add_drives_to_dir_list (fs->dir_list);
#endif

  gtk_clist_thaw (GTK_CLIST (fs->dir_list));
  gtk_clist_thaw (GTK_CLIST (fs->file_list));

  g_assert (cmpl_state->reference_dir);

  return rem_path;
}

/* Set file and dir lists for a path, completing if requested.
 * Return 0 for success, non-z for completion failed. 
 */
static gint
path_complete (GtkFileSelection2 *fs,
	       const char        *rel_path,
	       gint               try_complete)
{
  CompletionState *cmpl_state;
  const char *rem_path;

  g_return_val_if_fail (fs != NULL, -1);
  g_return_val_if_fail (GTK_IS_FILE_SELECTION2 (fs), -1);

  cmpl_state = (CompletionState*) fs->cmpl_state;

  if (!(rem_path = refresh_lists (fs, rel_path)))
    {
      return -1;
    }

  if (try_complete)
    {
      if (!cmpl_updated_text (cmpl_state)[0])
        {
	  /* No completions: nothing matches, or match is complete. 
	   */
	  if (rem_path[0])
	    {
	      /* Stuff left: no matches.
	       */
	      return -1;
	    }
	  else
	    {
	      /* Perfect match.
	       */
	      return 0;
	    }
        }

      /* Have we completed to a directory? Recurse into it.
       */
      if (cmpl_updated_dir (cmpl_state))
        {
          gchar *dir2 = g_strdup (cmpl_updated_text (cmpl_state));
	  gint res = path_complete (fs, dir2, FALSE);

	  g_free (dir2);
	  return res;
        }
    }
  else
    {
      if (rem_path[0])
        {
          /* There's stuff left, and we're not completing. Must be a file-
	   * not-found/permissions problem.
           */
          return -1;
        }
    }

  return 0;
}

/* Jump to a new path, optionally completing. Give completion feedback in 
 * the path_entry widget. Only complete directories.
 */
static void
gtk_file_selection2_dir_enter (GtkFileSelection2 *fs,
			     gchar            *rel_path,
			     gint              try_complete)
{
#ifdef DEBUG
  printf( "gtk_file_selection2_dir_enter: \"%s\" %d\n", 
  	rel_path, try_complete );
#endif /*DEBUG*/

  if (path_complete (fs, rel_path, try_complete))
    {
#ifdef DEBUG
      printf( "gtk_file_selection2_dir_enter: path_complete failed!\n" );
#endif /*DEBUG*/

      gtk_file_selection2_abort (fs);

      /* Highlight the error stuff.
       */
      if (fs->path_entry)
	{
          gtk_entry_set_position (GTK_ENTRY (fs->path_entry), 
			cmpl_last_valid_char (fs->cmpl_state));
          gtk_editable_select_region (GTK_EDITABLE (fs->path_entry), 
			cmpl_last_valid_char (fs->cmpl_state), -1);
	}

      return;
    }

  if (try_complete)
    {
      set_dir_enter (fs, cmpl_updated_text (fs->cmpl_state));
    }
  else
    {
      set_dir_enter (fs, cmpl_reference_position (fs->cmpl_state));
    }
}

/* As above, but feedback to the file entry bit.
 */
static void
gtk_file_selection2_new_file (GtkFileSelection2 *fs,
			      gchar            *rel_path,
			      gint              try_complete)
{
  if (path_complete (fs, rel_path, try_complete))
    {
      gtk_file_selection2_abort (fs);

      /* Highlight the error stuff.
       */
      if (fs->selection_entry)
	{
          gtk_entry_set_position (GTK_ENTRY (fs->selection_entry), 
			cmpl_last_valid_char (fs->cmpl_state));
          gtk_editable_select_region (GTK_EDITABLE (fs->selection_entry), 
			cmpl_last_valid_char (fs->cmpl_state), -1);
	}

      return;
    }

  set_dir_enter (fs, cmpl_reference_position (fs->cmpl_state));
  set_new_file (fs, cmpl_updated_text (fs->cmpl_state));
}

static void
gtk_file_selection2_abort (GtkFileSelection2 *fs)
{
  /*  BEEP gdk_beep();  */

  if (fs->selection_entry)
    {
      gchar *txt = g_strconcat (_("Directory unreadable: "), 
		cmpl_strerror (cmpl_errno), NULL);

      gtk_label_set_text (GTK_LABEL (fs->path_text), txt);
      g_free (txt);
    }
}

/* Default signal handler for new dir ... update the history menu and the
 * current directory label.
 */
static void
gtk_real_dir_enter (GtkFileSelection2 *fs, const gchar *dir)
{
#ifdef DEBUG
  printf( "gtk_real_dir_enter: \"%s\"\n", dir );
#endif /*DEBUG*/

  if (fs->selection_entry)
    {
      gchar *txt = g_strconcat (_("Current: "), dir, NULL);

      gtk_label_set_text (GTK_LABEL (fs->path_text), txt); 
      g_free (txt);
    }

  gtk_file_selection2_update_history_menu (fs, dir);
}

/* Default signal handler for new file ... nothing.
 */
static void
gtk_real_file_select (GtkFileSelection2 *fs, const gchar *filename)
{
#ifdef DEBUG
	printf( "gtk_real_new_file: \"%s\"\n", filename );
#endif /*DEBUG*/
}

/* Default signal handler for refresh ... do nothing.
 */
static void
gtk_real_refresh (GtkFileSelection2 *fs)
{
#ifdef DEBUG
	printf( "gtk_real_refresh\n" );
#endif /*DEBUG*/
}

/**********************************************************************/
/*			  External Interface                          */
/**********************************************************************/

/* The four completion state selectors
 */
static gchar*
cmpl_updated_text (CompletionState* cmpl_state)
{
  return cmpl_state->updated_text;
}

static gint
cmpl_updated_dir (CompletionState* cmpl_state)
{
  return cmpl_state->re_complete;
}

static gchar*
cmpl_reference_position (CompletionState* cmpl_state)
{
  return cmpl_state->reference_dir->fullname;
}

static gint
cmpl_last_valid_char (CompletionState* cmpl_state)
{
  return cmpl_state->last_valid_char;
}

static gchar*
cmpl_completion_fullname (gchar* text, CompletionState* cmpl_state)
{
  static char nothing[2] = "";

  if (!cmpl_state_okay (cmpl_state))
    {
      return nothing;
    }
  else if (g_path_is_absolute (text))
    {
      strcpy (cmpl_state->updated_text, text);
    }
  else if (text[0] == '~')
    {
      CompletionDir* dir;
      char* slash;

      dir = open_user_dir (text, cmpl_state);

      if (!dir)
	{
	  /* spencer says just return ~something, so
	   * for now just do it. */
	  strcpy (cmpl_state->updated_text, text);
	}
      else
	{
	  strcpy (cmpl_state->updated_text, dir->fullname);

	  slash = strchr (text, IM_DIR_SEP);

	  if (slash)
	    strcat (cmpl_state->updated_text, slash);
	}
    }
  else if (text[0] == '$')
    {
      CompletionDir* dir;
      char* slash;

      dir = open_environment_dir (text, cmpl_state);

      if (!dir)
	{
	  strcpy (cmpl_state->updated_text, text);
	}
      else
	{
	  strcpy (cmpl_state->updated_text, dir->fullname);

	  slash = strchr (text, IM_DIR_SEP);

	  if (slash)
	    strcat (cmpl_state->updated_text, slash);
	}
    }
  else
    {
      strcpy (cmpl_state->updated_text, cmpl_state->reference_dir->fullname);
      if (cmpl_state->updated_text[strlen (cmpl_state->updated_text) - 1] != IM_DIR_SEP)
        strcat (cmpl_state->updated_text, IM_DIR_SEP_STR);

      strcat (cmpl_state->updated_text, text);
    }

  return cmpl_state->updated_text;
}

/* The three completion selectors
 */
static gchar*
cmpl_this_completion (PossibleCompletion* pc)
{
  return pc->text;
}

static gint
cmpl_is_directory (PossibleCompletion* pc)
{
  return pc->is_directory;
}

static gint
cmpl_is_a_completion (PossibleCompletion* pc)
{
  return pc->is_a_completion;
}

/**********************************************************************/
/*	                 Construction, deletion                       */
/**********************************************************************/

static CompletionState*
cmpl_init_state (void)
{
  gchar getcwd_buf[2*MAXPATHLEN];
  CompletionState *new_state;

  new_state = g_new (CompletionState, 1);

  /* We don't use getcwd() on SUNOS, because, it does a popen("pwd")
   * and, if that wasn't bad enough, hangs in doing so.
   */
#if defined(sun) && !defined(__SVR4)
  if (!getwd (getcwd_buf))
#else    
  if (!getcwd (getcwd_buf, MAXPATHLEN))
#endif    
    {
      /* Oh joy, we can't get the current directory. Um..., we should have
       * a root directory, right? Right? (Probably not portable to non-Unix)
       */
      strcpy (getcwd_buf, IM_DIR_SEP_STR);
    }

tryagain:

  new_state->reference_dir = NULL;
  new_state->completion_dir = NULL;
  new_state->active_completion_dir = NULL;
  new_state->directory_storage = NULL;
  new_state->directory_sent_storage = NULL;
  new_state->last_valid_char = 0;
  new_state->updated_text = g_new (gchar, MAXPATHLEN);
  new_state->updated_text_alloc = MAXPATHLEN;
  new_state->the_completion.text = g_new (gchar, MAXPATHLEN);
  new_state->the_completion.text_alloc = MAXPATHLEN;
  new_state->user_dir_name_buffer = NULL;
  new_state->user_directories = NULL;
  new_state->predicate = NULL;
  new_state->predicate_data = NULL;

  new_state->reference_dir =  open_dir (getcwd_buf, new_state);

  if (!new_state->reference_dir)
    {
      /* Directories changing from underneath us, grumble */
      strcpy (getcwd_buf, IM_DIR_SEP_STR);
      goto tryagain;
    }

  return new_state;
}

static void
cmpl_free_dir_list(GList* dp0)
{
  GList *dp = dp0;

  while (dp) {
    free_dir (dp->data);
    dp = dp->next;
  }

  g_list_free(dp0);
}

static void
cmpl_free_dir_sent_list(GList* dp0)
{
  GList *dp = dp0;

  while (dp) {
    free_dir_sent (dp->data);
    dp = dp->next;
  }

  g_list_free(dp0);
}

static void
cmpl_free_state (CompletionState* cmpl_state)
{
  cmpl_free_dir_list (cmpl_state->directory_storage);
  cmpl_free_dir_sent_list (cmpl_state->directory_sent_storage);

  if (cmpl_state->user_dir_name_buffer)
    g_free (cmpl_state->user_dir_name_buffer);
  if (cmpl_state->user_directories)
    g_free (cmpl_state->user_directories);
  if (cmpl_state->the_completion.text)
    g_free (cmpl_state->the_completion.text);
  if (cmpl_state->updated_text)
    g_free (cmpl_state->updated_text);

  g_free (cmpl_state);
}

static void
free_dir(CompletionDir* dir)
{
  g_free(dir->fullname);
  g_free(dir);
}

static void
free_dir_sent(CompletionDirSent* sent)
{
  g_free(sent->name_buffer);
  g_free(sent->entries);
  g_free(sent);
}

static void
prune_memory_usage(CompletionState *cmpl_state)
{
  GList* cdsl = cmpl_state->directory_sent_storage;
  GList* cdl = cmpl_state->directory_storage;
  GList* cdl0 = cdl;
  gint len = 0;

  for(; cdsl && len < CMPL_DIRECTORY_CACHE_SIZE; len += 1)
    cdsl = cdsl->next;

  if (cdsl) {
    cmpl_free_dir_sent_list(cdsl->next);
    cdsl->next = NULL;
  }

  cmpl_state->directory_storage = NULL;
  while (cdl) {
    if (cdl->data == cmpl_state->reference_dir)
      cmpl_state->directory_storage = g_list_prepend(NULL, cdl->data);
    else
      free_dir (cdl->data);
    cdl = cdl->next;
  }

  g_list_free(cdl0);
}

/**********************************************************************/
/*                        The main entrances.                         */
/**********************************************************************/

static PossibleCompletion*
cmpl_completion_matches (const char* text_to_complete,
			 const char** remaining_text,
			 CompletionState* cmpl_state)
{
  gchar* first_slash;
  PossibleCompletion *poss;

  prune_memory_usage(cmpl_state);

  g_assert (text_to_complete != NULL);

  cmpl_state->user_completion_index = -1;
  cmpl_state->environment_completion_index = -1;
  cmpl_state->last_completion_text = text_to_complete;
  cmpl_state->the_completion.text[0] = 0;
  cmpl_state->last_valid_char = 0;
  cmpl_state->updated_text_len = -1;
  cmpl_state->updated_text[0] = 0;
  cmpl_state->re_complete = FALSE;

  first_slash = strchr (text_to_complete, IM_DIR_SEP);

  if (text_to_complete[0] == '~' && !first_slash)
    {
      /* Text starts with ~ and there is no slash, show all the
       * home directory completions.
       */
      poss = attempt_homedir_completion (text_to_complete, cmpl_state);

      update_cmpl(poss, cmpl_state);

      return poss;
    }

  if (text_to_complete[0] == '$' && !first_slash)
    {
      /* Text starts with $ and there is no slash, show all the
       * environment completions.
       */
      poss = attempt_environment_completion (text_to_complete, cmpl_state);

      update_cmpl(poss, cmpl_state);

      return poss;
    }

  cmpl_state->reference_dir =
    open_ref_dir (text_to_complete, remaining_text, cmpl_state);

  if(!cmpl_state->reference_dir)
    return NULL;

  cmpl_state->completion_dir =
    find_completion_dir (*remaining_text, remaining_text, cmpl_state);

  cmpl_state->last_valid_char = *remaining_text - text_to_complete;

  if(!cmpl_state->completion_dir)
    return NULL;

  cmpl_state->completion_dir->cmpl_index = -1;
  cmpl_state->completion_dir->cmpl_parent = NULL;
  cmpl_state->completion_dir->cmpl_text = *remaining_text;

  cmpl_state->active_completion_dir = cmpl_state->completion_dir;

  cmpl_state->reference_dir = cmpl_state->completion_dir;

  poss = attempt_file_completion(cmpl_state);

  update_cmpl(poss, cmpl_state);

  return poss;
}

static PossibleCompletion*
cmpl_next_completion (CompletionState* cmpl_state)
{
  PossibleCompletion* poss = NULL;

  cmpl_state->the_completion.text[0] = '\0';

  if(cmpl_state->user_completion_index >= 0)
    poss = attempt_homedir_completion(cmpl_state->last_completion_text, cmpl_state);
  else if(cmpl_state->environment_completion_index >= 0)
    poss = attempt_environment_completion(cmpl_state->last_completion_text, cmpl_state);
  else
    poss = attempt_file_completion(cmpl_state);

  update_cmpl(poss, cmpl_state);

  return poss;
}

/**********************************************************************/
/*			 Directory Operations                         */
/**********************************************************************/

static char *
path_get_root( const char *path )
{
        static char buf[MAXPATHLEN];
	char *p;

        im_strncpy( buf, path, MAXPATHLEN );
        if( (p = (char *) g_path_skip_root( buf )) )
		*p = '\0';

        return( buf );
}

/* Open the directory where completion will begin from, if possible. */
static CompletionDir*
open_ref_dir(const char* text_to_complete,
	     const char** remaining_text,
	     CompletionState* cmpl_state)
{
  gchar* first_slash;
  CompletionDir *new_dir;

  first_slash = strchr(text_to_complete, IM_DIR_SEP);

  if (g_path_is_absolute (text_to_complete) || !cmpl_state->reference_dir)
    {
      new_dir = open_dir (path_get_root (text_to_complete), cmpl_state);

      if(new_dir)
	*remaining_text = g_path_skip_root ((char *) text_to_complete);
    }
  else if (text_to_complete[0] == '~')
    {
      new_dir = open_user_dir(text_to_complete, cmpl_state);

      if(new_dir)
	{
	  if(first_slash)
	    *remaining_text = first_slash + 1;
	  else
	    *remaining_text = text_to_complete + strlen (text_to_complete);
	}
      else
	{
	  return NULL;
	}
    }
  else if (text_to_complete[0] == '$')
    {
      new_dir = open_environment_dir(text_to_complete, cmpl_state);

      if(new_dir)
	{
	  if(first_slash)
	    *remaining_text = first_slash + 1;
	  else
	    *remaining_text = text_to_complete + strlen(text_to_complete);
	}
      else
	{
	  return NULL;
	}
    }
  else
    {
      *remaining_text = text_to_complete;

      new_dir = open_dir(cmpl_state->reference_dir->fullname, cmpl_state);
    }

  if(new_dir)
    {
      new_dir->cmpl_index = -1;
      new_dir->cmpl_parent = NULL;
    }

  return new_dir;
}

/* open a directory by user name */
static CompletionDir*
open_user_dir(const char* text_to_complete,
	      CompletionState *cmpl_state)
{
  gchar *first_slash;
  gint cmp_len;

  g_assert(text_to_complete && text_to_complete[0] == '~');

  first_slash = strchr(text_to_complete, IM_DIR_SEP);

  if (first_slash)
    cmp_len = first_slash - text_to_complete - 1;
  else
    cmp_len = strlen(text_to_complete + 1);

  if(!cmp_len)
    {
      /* ~/ */
      const char *homedir = g_get_home_dir ();

      if (homedir)
	return open_dir(homedir, cmpl_state);
      else
	return NULL;
    }
  else
    {
#ifdef HAVE_GETPWNAM
      /* ~user/ */
      char* copy = g_new(char, cmp_len + 1);
      struct passwd *pwd;
      strncpy(copy, text_to_complete + 1, cmp_len);
      copy[cmp_len] = 0;
      pwd = getpwnam(copy);
      g_free(copy);
      if (!pwd)
	{
	  cmpl_errno = errno;
	  return NULL;
	}

      return open_dir(pwd->pw_dir, cmpl_state);
#endif /*HAVE_GETPWNAM*/
      return NULL;
    }
}

/* open a directory by environment variable */
static CompletionDir*
open_environment_dir(const char* text_to_complete,
	      CompletionState *cmpl_state)
{
  gchar *first_slash;
  gint cmp_len;
  char* copy;
  char* val;

  g_assert(text_to_complete && text_to_complete[0] == '$');

  first_slash = strchr(text_to_complete, IM_DIR_SEP);

  if (first_slash)
    cmp_len = first_slash - text_to_complete - 1;
  else
    cmp_len = strlen(text_to_complete + 1);

  copy = g_new(char, cmp_len + 1);
  strncpy(copy, text_to_complete + 1, cmp_len);
  copy[cmp_len] = 0;

  val = getenv (copy);
  g_free(copy);

  if (!val)
    {
      cmpl_errno = ENOENT;
      return NULL;
    }

  return open_dir(val, cmpl_state);
}

/* open a directory relative the the current relative directory */
static CompletionDir*
open_relative_dir(const char* dir_name,
		  CompletionDir* dir,
		  CompletionState *cmpl_state)
{
  gchar path_buf[2*MAXPATHLEN];

  if(dir->fullname_len + strlen(dir_name) + 2 >= MAXPATHLEN)
    {
      cmpl_errno = CMPL_ERRNO_TOO_LONG;
      return NULL;
    }

  strcpy(path_buf, dir->fullname);

  if(dir->fullname_len > 1 && 
    path_buf[dir->fullname_len - 1] != IM_DIR_SEP)
    {
      path_buf[dir->fullname_len] = IM_DIR_SEP;
      strcpy(path_buf + dir->fullname_len + 1, dir_name);
    }
  else
    {
      strcpy(path_buf + dir->fullname_len, dir_name);
    }

  return open_dir(path_buf, cmpl_state);
}

/* after the cache lookup fails, really open a new directory */
static CompletionDirSent*
open_new_dir(const char* dir_name, struct stat* sbuf, gboolean stat_subdirs)
{
  CompletionDirSent* sent;
  DIR* directory;
  gchar *buffer_ptr;
  struct dirent *dirent_ptr;
  gint buffer_size = 0;
  gint entry_count = 0;
  gint i;
  struct stat ent_sbuf;
  char path_buf[MAXPATHLEN*2];
  gint path_buf_len;

  sent = g_new(CompletionDirSent, 1);
  sent->mtime = sbuf->st_mtime;
  sent->inode = sbuf->st_ino;
  sent->device = sbuf->st_dev;

  path_buf_len = strlen(dir_name);

  if (path_buf_len > MAXPATHLEN)
    {
      cmpl_errno = CMPL_ERRNO_TOO_LONG;
      return NULL;
    }

  strcpy(path_buf, dir_name);

  directory = opendir(dir_name);

  if(!directory)
    {
      cmpl_errno = errno;
      return NULL;
    }

  while((dirent_ptr = readdir(directory)) != NULL)
    {
      int entry_len = strlen(dirent_ptr->d_name);
      buffer_size += entry_len + 1;
      entry_count += 1;

      if(path_buf_len + entry_len + 2 >= MAXPATHLEN)
	{
	  cmpl_errno = CMPL_ERRNO_TOO_LONG;
 	  closedir(directory);
	  return NULL;
	}
    }

  sent->name_buffer = g_new(gchar, buffer_size);
  sent->entries = g_new(CompletionDirEntry, entry_count);
  sent->entry_count = entry_count;

  buffer_ptr = sent->name_buffer;

  rewinddir(directory);

  for(i = 0; i < entry_count; i += 1)
    {
      dirent_ptr = readdir(directory);

      if(!dirent_ptr)
	{
	  cmpl_errno = errno;
	  closedir(directory);
	  return NULL;
	}

      strcpy(buffer_ptr, dirent_ptr->d_name);
      sent->entries[i].entry_name = buffer_ptr;
      buffer_ptr += strlen(dirent_ptr->d_name);
      *buffer_ptr = 0;
      buffer_ptr += 1;

      path_buf[path_buf_len] = IM_DIR_SEP;
      strcpy(path_buf + path_buf_len + 1, dirent_ptr->d_name);

      if (stat_subdirs)
	{
	  if(stat(path_buf, &ent_sbuf) >= 0 && S_ISDIR(ent_sbuf.st_mode))
	    sent->entries[i].is_dir = 1;
	  else
	    /* stat may fail, and we don't mind, since it could be a
	     * dangling symlink. */
	    sent->entries[i].is_dir = 0;
	}
      else
	sent->entries[i].is_dir = 1;
    }

  qsort(sent->entries, sent->entry_count, sizeof(CompletionDirEntry), compare_cmpl_dir);

  closedir(directory);

  return sent;
}

/* win32 hates trailing slashes for stat :-(
 */
static int
my_stat( const char *path, struct stat *st )
{
  char buffer[MAXPATHLEN];
  char *last_slash;

  im_strncpy ( buffer, path, MAXPATHLEN );
  if ((last_slash = strrchr (buffer, IM_DIR_SEP)) && !last_slash[1])
  	*last_slash = '\0';

  return stat (buffer, st);
}

static gboolean
check_dir(const char *dir_name, struct stat *result, gboolean *stat_subdirs)
{
  /* A list of directories that we know only contain other directories.
   * Trying to stat every file in these directories would be very
   * expensive.
   */

  static struct {
    gchar *name;
    gboolean present;
    struct stat statbuf;
  } no_stat_dirs[] = {
    { IM_DIR_SEP_STR "afs", FALSE, { 0 } },
    { IM_DIR_SEP_STR "net", FALSE, { 0 } }
  };

  static const gint n_no_stat_dirs = sizeof(no_stat_dirs) / sizeof(no_stat_dirs[0]);
  static gboolean initialized = FALSE;

  gint i;

  if (!initialized)
    {
      initialized = TRUE;
      for (i = 0; i < n_no_stat_dirs; i++)
	{
	  if (my_stat (no_stat_dirs[i].name, &no_stat_dirs[i].statbuf) == 0)
	    no_stat_dirs[i].present = TRUE;
	}
    }

  if(stat(dir_name, result) < 0)
    {
      cmpl_errno = errno;
      return FALSE;
    }

  *stat_subdirs = TRUE;
  for (i=0; i<n_no_stat_dirs; i++)
    {
      if (no_stat_dirs[i].present &&
	  (no_stat_dirs[i].statbuf.st_dev == result->st_dev) &&
	  (no_stat_dirs[i].statbuf.st_ino == result->st_ino))
	{
	  *stat_subdirs = FALSE;
	  break;
	}
    }

  return TRUE;
}

/* open a directory by absolute pathname */
static CompletionDir*
open_dir(const char* dir_name, CompletionState* cmpl_state)
{
  struct stat sbuf;
  gboolean stat_subdirs;
  CompletionDirSent *sent;
  GList* cdsl;

  if (!check_dir (dir_name, &sbuf, &stat_subdirs))
    return NULL;

  cdsl = cmpl_state->directory_sent_storage;

  while (cdsl)
    {
      sent = cdsl->data;

      if(sent->inode == sbuf.st_ino &&
	 sent->mtime == sbuf.st_mtime &&
	 sent->device == sbuf.st_dev)
	return attach_dir(sent, dir_name, cmpl_state);

      cdsl = cdsl->next;
    }

  sent = open_new_dir(dir_name, &sbuf, stat_subdirs);

  if (sent) {
    cmpl_state->directory_sent_storage =
      g_list_prepend(cmpl_state->directory_sent_storage, sent);

    return attach_dir(sent, dir_name, cmpl_state);
  }

  return NULL;
}

static CompletionDir*
attach_dir(CompletionDirSent* sent, const char* dir_name, CompletionState *cmpl_state)
{
  CompletionDir* new_dir;

  new_dir = g_new(CompletionDir, 1);

  cmpl_state->directory_storage =
    g_list_prepend(cmpl_state->directory_storage, new_dir);

  new_dir->sent = sent;
  new_dir->fullname = g_strdup(dir_name);
  new_dir->fullname_len = strlen(dir_name);

  return new_dir;
}

static gint
correct_dir_fullname(CompletionDir* cmpl_dir)
{
  gint length = strlen (cmpl_dir->fullname);

#ifdef DEBUG
  printf( "correct_dir_fullname: cmpl_dir->fullname = \"%s\"\n",
  	cmpl_dir->fullname );
#endif /*DEBUG*/

  if (is_postfix (IM_DIR_SEP_STR ".", cmpl_dir->fullname) == 0)
    {
      if (strlen (g_path_skip_root (cmpl_dir->fullname)) == 1) 
	{
	  /* If we're at the top, don't remove the path root.
	   */
	  cmpl_dir->fullname[length - 1] = '\0';
	} 
      else 
        {
	  cmpl_dir->fullname[length - 2] = '\0';
	}
    }
  else if (is_postfix (IM_DIR_SEP_STR "." IM_DIR_SEP_STR, 
  	cmpl_dir->fullname) == 0)
    {
      cmpl_dir->fullname[length - 2] = '\0';
    }
  else if (is_postfix (IM_DIR_SEP_STR "..", cmpl_dir->fullname) == 0)
    {
      if (strlen (g_path_skip_root (cmpl_dir->fullname)) == 2) 
	{
          cmpl_dir->fullname[length - 2] = '\0';
	}
      else
        {
          cmpl_dir->fullname[length - 2] = '\0';

          if (!correct_parent (cmpl_dir))
	    return FALSE;
        }
    }
  else if (is_postfix (IM_DIR_SEP_STR ".." IM_DIR_SEP_STR, 
  	cmpl_dir->fullname) == 0)
    {
      if (strlen (g_path_skip_root (cmpl_dir->fullname)) == 3) 
	{
          cmpl_dir->fullname[length - 3] = '\0';
	}
      else
        {
          cmpl_dir->fullname[length - 3] = '\0';
    
          if (!correct_parent (cmpl_dir))
	    return FALSE;
        }
    }

  cmpl_dir->fullname_len = strlen (cmpl_dir->fullname);

  return TRUE;
}

static gint
correct_parent(CompletionDir* cmpl_dir)
{
  gchar *new_name;

#ifdef DEBUG
  printf( "correct_parent: cmpl_dir->fullname = \"%s\"\n", 
  	cmpl_dir->fullname );
#endif /*DEBUG*/

  new_name = find_parent_dir_fullname (cmpl_dir->fullname);
  if (!new_name)
    return FALSE;

  g_free (cmpl_dir->fullname);
  cmpl_dir->fullname = new_name;

  return TRUE;
}

static gchar*
find_parent_dir_fullname(gchar* dirname)
{
  gchar buffer[MAXPATHLEN];
  gchar buffer2[MAXPATHLEN];

#if defined(sun) && !defined(__SVR4)
  if (!getwd (buffer))
#else
  if (!getcwd (buffer, MAXPATHLEN))
#endif    
    {
#ifdef DEBUG
      printf( "find_parent_dir_fullname: fail #1\n" );
#endif /*DEBUG*/
      cmpl_errno = errno;
      return NULL;
    }

  if (chdir (dirname) || chdir (".."))
    {
#ifdef DEBUG
      printf( "find_parent_dir_fullname: fail #2\n" );
#endif /*DEBUG*/
      cmpl_errno = errno;
      return NULL;
    }

#if defined(sun) && !defined(__SVR4)
  if(!getwd(buffer2))
#else
  if(!getcwd(buffer2, MAXPATHLEN))
#endif
    {
#ifdef DEBUG
      printf( "find_parent_dir_fullname: fail #3\n" );
#endif /*DEBUG*/
      chdir (buffer);
      cmpl_errno = errno;

      return NULL;
    }

  if (chdir (buffer))
    {
#ifdef DEBUG
      printf( "find_parent_dir_fullname: fail #4\n" );
#endif /*DEBUG*/
      cmpl_errno = errno;
      return NULL;
    }

  return g_strdup (buffer2);
}

/**********************************************************************/
/*                        Completion Operations                       */
/**********************************************************************/

static PossibleCompletion*
attempt_homedir_completion(const char* text_to_complete,
			   CompletionState *cmpl_state)
{
  gint index;

  if (!cmpl_state->user_dir_name_buffer &&
      !get_pwdb(cmpl_state))
    return NULL;

  cmpl_state->user_completion_index += 1;

  while(cmpl_state->user_completion_index < cmpl_state->user_directories_len)
    {
      index = first_diff_index(text_to_complete + 1,
			       cmpl_state->user_directories
			       [cmpl_state->user_completion_index].login);

      switch(index)
	{
	case PATTERN_MATCH:
	  break;
	default:
	  if(cmpl_state->last_valid_char < (index + 1))
	    cmpl_state->last_valid_char = index + 1;
	  cmpl_state->user_completion_index += 1;
	  continue;
	}

      cmpl_state->the_completion.is_a_completion = 1;
      cmpl_state->the_completion.is_directory = 1;

      append_completion_text("~", cmpl_state);

      append_completion_text(cmpl_state->
			      user_directories[cmpl_state->user_completion_index].login,
			     cmpl_state);

      return append_completion_text(IM_DIR_SEP_STR, cmpl_state);
    }

  if(text_to_complete[1] ||
     cmpl_state->user_completion_index > cmpl_state->user_directories_len)
    {
      cmpl_state->user_completion_index = -1;
      return NULL;
    }
  else
    {
      cmpl_state->user_completion_index += 1;
      cmpl_state->the_completion.is_a_completion = 1;
      cmpl_state->the_completion.is_directory = 1;

      return append_completion_text("~" IM_DIR_SEP_STR, cmpl_state);
    }
}

/* returns the index (>= 0) of the first differing character,
 * PATTERN_MATCH if the completion matches. Used for scanning the environment,
 * so allow '=' as end-of-string in text */
static gint
env_first_diff_index(const char* pat, const char* text)
{
  gint diff = 0;

  while(*pat && (*text && *text != '=') && *text == *pat)
    {
      pat += 1;
      text += 1;
      diff += 1;
    }

  if(*pat)
    return diff;

  return PATTERN_MATCH;
}

static PossibleCompletion*
attempt_environment_completion(const char* text_to_complete,
			   CompletionState *cmpl_state)
{
  gint index;
  gint i;
  extern char** environ;

  for (i = cmpl_state->environment_completion_index + 1; environ[i]; i++)
    {
      index = env_first_diff_index(text_to_complete + 1, environ[i]);

      if (index == PATTERN_MATCH)
        {
	  gchar buf[MAXPATHLEN + 1];
	  gchar *p;

	  cmpl_state->the_completion.is_a_completion = 1;
	  cmpl_state->the_completion.is_directory = 1;
	  cmpl_state->environment_completion_index = i;

	  strncpy (buf, environ[i], MAXPATHLEN);
	  buf[MAXPATHLEN + 1] = '\0';
	  p = strchr (buf, '=');
	  if (p)
	    {
	      *p = '\0';
	    }

	  append_completion_text ("$", cmpl_state);
	  append_completion_text (buf, cmpl_state);

	  return append_completion_text (IM_DIR_SEP_STR, cmpl_state);
	}

      if (cmpl_state->last_valid_char < index + 1)
      	{
	  cmpl_state->last_valid_char = index + 1;
	}
    }

  cmpl_state->environment_completion_index = -1;
  return NULL;
}

/* returns the index (>= 0) of the first differing character,
 * PATTERN_MATCH if the completion matches 
 */
static gint
first_diff_index(const char* pat, const char* text)
{
  gint diff = 0;

  while(*pat && *text && *text == *pat)
    {
      pat += 1;
      text += 1;
      diff += 1;
    }

  if(*pat)
    return diff;

  return PATTERN_MATCH;
}

static PossibleCompletion*
append_completion_text(const char* text, CompletionState* cmpl_state)
{
  gint len, i = 1;

  if(!cmpl_state->the_completion.text)
    return NULL;

  len = strlen(text) + strlen(cmpl_state->the_completion.text) + 1;

  if(cmpl_state->the_completion.text_alloc > len)
    {
      strcat(cmpl_state->the_completion.text, text);
      return &cmpl_state->the_completion;
    }

  while(i < len) { i <<= 1; }

  cmpl_state->the_completion.text_alloc = i;

  cmpl_state->the_completion.text = (gchar*)g_realloc(cmpl_state->the_completion.text, i);

  if(!cmpl_state->the_completion.text)
    return NULL;
  else
    {
      strcat(cmpl_state->the_completion.text, text);
      return &cmpl_state->the_completion;
    }
}

static CompletionDir*
find_completion_dir(const char* text_to_complete,
		    const char** remaining_text,
		    CompletionState* cmpl_state)
{
  gchar* first_slash = strchr(text_to_complete, IM_DIR_SEP);
  CompletionDir* dir = cmpl_state->reference_dir;
  CompletionDir* next;
  *remaining_text = text_to_complete;

  while(first_slash)
    {
      gint len = first_slash - *remaining_text;
      gint found = 0;
      gchar *found_name = NULL;         /* Quiet gcc */
      gint i;
      gchar* pat_buf = g_new (gchar, len + 1);

      strncpy(pat_buf, *remaining_text, len);
      pat_buf[len] = 0;

      for(i = 0; i < dir->sent->entry_count; i += 1)
	{
	  if(dir->sent->entries[i].is_dir &&
	     FNMATCH(pat_buf, dir->sent->entries[i].entry_name))
	    {
	      if(found)
		{
		  g_free (pat_buf);
		  return dir;
		}
	      else
		{
		  found = 1;
		  found_name = dir->sent->entries[i].entry_name;
		}
	    }
	}

      if (!found)
	{
	  /* Perhaps we are trying to open an automount directory */
	  found_name = pat_buf;
	}

      next = open_relative_dir(found_name, dir, cmpl_state);
      
      if(!next)
	{
	  g_free (pat_buf);
	  return NULL;
	}
      
      next->cmpl_parent = dir;
      
      dir = next;
      
      if(!correct_dir_fullname(dir))
	{
	  g_free(pat_buf);
	  return NULL;
	}
      
      *remaining_text = first_slash + 1;
      first_slash = strchr(*remaining_text, IM_DIR_SEP);

      g_free (pat_buf);
    }

  return dir;
}

static void
update_cmpl(PossibleCompletion* poss, CompletionState* cmpl_state)
{
  gint cmpl_len;

  if(!poss || !cmpl_is_a_completion(poss))
    return;

  cmpl_len = strlen(cmpl_this_completion(poss));

  if(cmpl_state->updated_text_alloc < cmpl_len + 1)
    {
      cmpl_state->updated_text =
	(gchar*)g_realloc(cmpl_state->updated_text,
			  cmpl_state->updated_text_alloc);
      cmpl_state->updated_text_alloc = 2*cmpl_len;
    }

  if(cmpl_state->updated_text_len < 0)
    {
      strcpy(cmpl_state->updated_text, cmpl_this_completion(poss));
      cmpl_state->updated_text_len = cmpl_len;
      cmpl_state->re_complete = cmpl_is_directory(poss);
    }
  else if(cmpl_state->updated_text_len == 0)
    {
      cmpl_state->re_complete = FALSE;
    }
  else
    {
      gint first_diff =
	first_diff_index(cmpl_state->updated_text,
			 cmpl_this_completion(poss));

      cmpl_state->re_complete = FALSE;

      if(first_diff == PATTERN_MATCH)
	return;

      if(first_diff > cmpl_state->updated_text_len)
	strcpy(cmpl_state->updated_text, cmpl_this_completion(poss));

      cmpl_state->updated_text_len = first_diff;
      cmpl_state->updated_text[first_diff] = 0;
    }
}

static PossibleCompletion*
attempt_file_completion(CompletionState *cmpl_state)
{
  gchar *pat_buf, *first_slash;
  CompletionDir *dir = cmpl_state->active_completion_dir;

  dir->cmpl_index += 1;

  if(dir->cmpl_index == dir->sent->entry_count)
    {
      if(dir->cmpl_parent == NULL)
	{
	  cmpl_state->active_completion_dir = NULL;

	  return NULL;
	}
      else
	{
	  cmpl_state->active_completion_dir = dir->cmpl_parent;

	  return attempt_file_completion(cmpl_state);
	}
    }

  g_assert(dir->cmpl_text);

  first_slash = strchr(dir->cmpl_text, IM_DIR_SEP);

  if(first_slash)
    {
      gint len = first_slash - dir->cmpl_text;

      pat_buf = g_new (gchar, len + 1);
      strncpy(pat_buf, dir->cmpl_text, len);
      pat_buf[len] = 0;
    }
  else
    {
      gint len = strlen(dir->cmpl_text);

      pat_buf = g_new (gchar, len + 2);
      strcpy(pat_buf, dir->cmpl_text);
      strcpy(pat_buf + len, "*");
    }

  if(first_slash)
    {
      if(dir->sent->entries[dir->cmpl_index].is_dir)
	{
	  if(FNMATCH(pat_buf, dir->sent->entries[dir->cmpl_index].entry_name))
	    {
	      CompletionDir* new_dir;

	      new_dir = open_relative_dir(dir->sent->entries[dir->cmpl_index].entry_name,
					  dir, cmpl_state);

	      if(!new_dir)
		{
		  g_free (pat_buf);
		  return NULL;
		}

	      new_dir->cmpl_parent = dir;

	      new_dir->cmpl_index = -1;
	      new_dir->cmpl_text = first_slash + 1;

	      cmpl_state->active_completion_dir = new_dir;

	      g_free (pat_buf);
	      return attempt_file_completion(cmpl_state);
	    }
	  else
	    {
	      g_free (pat_buf);
	      return attempt_file_completion(cmpl_state);
	    }
	}
      else
	{
	  g_free (pat_buf);
	  return attempt_file_completion(cmpl_state);
	}
    }
  else
    {
      if(dir->cmpl_parent != NULL)
	{
	  append_completion_text(dir->fullname +
				 strlen(cmpl_state->completion_dir->fullname) + 1,
				 cmpl_state);
	  append_completion_text(IM_DIR_SEP_STR, cmpl_state);
	}

      append_completion_text(dir->sent->entries[dir->cmpl_index].entry_name, cmpl_state);

      cmpl_state->the_completion.is_a_completion =
	FNMATCH(pat_buf, dir->sent->entries[dir->cmpl_index].entry_name);

      if (cmpl_state->predicate && 
	  cmpl_state->the_completion.is_a_completion &&
	  !dir->sent->entries[dir->cmpl_index].is_dir)
        {
	  cmpl_state->the_completion.is_a_completion = (*cmpl_state->predicate) (
	               dir->sent->entries[dir->cmpl_index].entry_name,
		       cmpl_state->predicate_data);
	}

      cmpl_state->the_completion.is_directory = dir->sent->entries[dir->cmpl_index].is_dir;
      if(dir->sent->entries[dir->cmpl_index].is_dir)
	append_completion_text(IM_DIR_SEP_STR, cmpl_state);

      g_free (pat_buf);
      return &cmpl_state->the_completion;
    }
}


static gint
get_pwdb(CompletionState* cmpl_state)
{
#ifdef HAVE_GETPWENT
  struct passwd *pwd_ptr;
  gchar* buf_ptr;
  gint len = 0, i, count = 0;

  if(cmpl_state->user_dir_name_buffer)
    return TRUE;
  setpwent ();

  while ((pwd_ptr = getpwent()) != NULL)
    {
      len += strlen(pwd_ptr->pw_name);
      len += strlen(pwd_ptr->pw_dir);
      len += 2;
      count += 1;
    }

  setpwent ();

  cmpl_state->user_dir_name_buffer = g_new(gchar, len);
  cmpl_state->user_directories = g_new(CompletionUserDir, count);
  cmpl_state->user_directories_len = count;

  buf_ptr = cmpl_state->user_dir_name_buffer;

  for(i = 0; i < count; i += 1)
    {
      pwd_ptr = getpwent();
      if(!pwd_ptr)
	{
	  cmpl_errno = errno;
	  goto error;
	}

      strcpy(buf_ptr, pwd_ptr->pw_name);
      cmpl_state->user_directories[i].login = buf_ptr;
      buf_ptr += strlen(buf_ptr);
      buf_ptr += 1;
      strcpy(buf_ptr, pwd_ptr->pw_dir);
      cmpl_state->user_directories[i].homedir = buf_ptr;
      buf_ptr += strlen(buf_ptr);
      buf_ptr += 1;
    }

  qsort(cmpl_state->user_directories,
	cmpl_state->user_directories_len,
	sizeof(CompletionUserDir),
	compare_user_dir);

  endpwent();

  return TRUE;

error:

  if(cmpl_state->user_dir_name_buffer)
    g_free(cmpl_state->user_dir_name_buffer);
  if(cmpl_state->user_directories)
    g_free(cmpl_state->user_directories);

  cmpl_state->user_dir_name_buffer = NULL;
  cmpl_state->user_directories = NULL;

  return FALSE;
#else /*HAVE_GETPWENT*/
  return FALSE;
#endif /*HAVE_GETPWENT*/
}

static gint
compare_user_dir(const void* a, const void* b)
{
  return strcmp((((CompletionUserDir*)a))->login,
		(((CompletionUserDir*)b))->login);
}

static gint
compare_cmpl_dir(const void* a, const void* b)
{
  return strcmp((((CompletionDirEntry*)a))->entry_name,
		(((CompletionDirEntry*)b))->entry_name);
}

static gint
cmpl_state_okay(CompletionState* cmpl_state)
{
  return  cmpl_state && cmpl_state->reference_dir;
}

static const char*
cmpl_strerror(gint err)
{
  if(err == CMPL_ERRNO_TOO_LONG)
    return "Name too long";
  else
    return g_strerror (err);
}
