/* display an image in a window ... watching an Iimage model.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

/* Define to trace button press events.
#define EVENT
 */

#include "ip.h"

#define IMAGE_WINDOW_WIDTH \
	(watch_int_get( "IMAGE_WINDOW_WIDTH", 600 ))
#define IMAGE_WINDOW_HEIGHT \
	(watch_int_get( "IMAGE_WINDOW_HEIGHT", 650 ))

static iWindowClass *parent_class = NULL;

/* All the magnification menus we have.
 */
typedef struct _ImageviewMagmenu {
	const char *path;
	int mag;
} ImageviewMagmenu;

static const ImageviewMagmenu imageview_mags[] = {
	{ "/View/Magnification/6%", -16 },
	{ "/View/Magnification/12%", -8 },
	{ "/View/Magnification/25%", -4 },
	{ "/View/Magnification/50%", -2 },
	{ "/View/Magnification/100%", 1 },
	{ "/View/Magnification/200%", 2 },
	{ "/View/Magnification/400%", 4 },
	{ "/View/Magnification/800%", 8 },
	{ "/View/Magnification/1600%", 16 }
};

static void
imageview_destroy( GtkObject *object )
{
	Imageview *iv;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_IMAGEVIEW( object ) );

	iv = IMAGEVIEW( object );

	/* My instance destroy stuff.
	 */
	FREESID( iv->changed_sid, iv->iimage );
	FREESID( iv->destroy_sid, iv->iimage );

	FREEO( iv->conv );
	iv->conv_changed_sid = 0;

	FREESID( iv->ws_changed_sid, main_workspacegroup );

	iv->iimage->views = g_slist_remove( iv->iimage->views, iv );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

static void
imageview_class_init( ImageviewClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;

	parent_class = gtk_type_class( TYPE_IWINDOW );

	object_class->destroy = imageview_destroy;

	/* Create signals.
	 */

	/* Init methods.
	 */
}

static void
imageview_init( Imageview *iv )
{
	iv->iimage = NULL;
	iv->ip = NULL;
	iv->conv = NULL;

	iv->changed_sid = 0;
	iv->destroy_sid = 0;
	iv->conv_changed_sid = 0;
	iv->ws_changed_sid = 0;

	iv->cv = NULL;
	iv->ifac = NULL;
}

GtkType
imageview_get_type( void )
{
	static GtkType imageview_type = 0;

	if( !imageview_type ) {
		static const GtkTypeInfo sinfo = {
			"Imageview",
			sizeof( Imageview ),
			sizeof( ImageviewClass ),
			(GtkClassInitFunc) imageview_class_init,
			(GtkObjectInitFunc) imageview_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		imageview_type = gtk_type_unique( TYPE_IWINDOW, &sinfo );
	}

	return( imageview_type );
}

/* Our iimage has changed ... update.
 */
static void
imageview_changed_cb( Iimage *iimage, Imageview *iv )
{
	conversion_set_image( iv->conv, iimage->instance.image );
}

/* Our iimage has been destroyed ... destroy us in turn.
 */
static void
imageview_destroy_cb( Iimage *iimage, Imageview *iv )
{
	iwindow_kill( IWINDOW( iv ) );
}

static void
imageview_refresh_title( Imageview *iv )
{
	Imageinfo *image = iv->iimage->instance.image;
	BufInfo buf;
	char txt[512];

	buf_init_static( &buf, txt, 512 );
	buf_appendf( &buf, IP_NAME );

	buf_appendf( &buf, " - " );
	row_qualified_name_relative( 
		SYMBOL( main_workspacegroup->current ),
		HEAPMODEL( iv->iimage )->row, &buf );

	if( image && image->name )
		buf_appendf( &buf, " - %s", image->name );

	buf_appendf( &buf, " - %.0f%%", 
		100.0 * conversion_dmag( iv->conv->mag ) );

	iwindow_set_title( IWINDOW( iv ), "%s", buf_all( &buf ) );
}

/* The conversion has changed ... update our menus and titlebar.
 */
static void
imageview_conv_changed_cb( Conversion *conv, Imageview *iv )
{
	GtkWidget *item;
	int i;

	for( i = 0; i < IM_NUMBER( imageview_mags ); i++ )
		if( conv->mag == imageview_mags[i].mag ) {
			item = gtk_item_factory_get_widget( iv->ifac,
				imageview_mags[i].path );
			gtk_check_menu_item_set_active( 
				GTK_CHECK_MENU_ITEM( item ), TRUE );
		}

	item = gtk_item_factory_get_widget( iv->ifac, 
		"/View/Show display control bar" );
	gtk_check_menu_item_set_active( 
		GTK_CHECK_MENU_ITEM( item ), conv->enabled );

	item = gtk_item_factory_get_widget( iv->ifac, 
		"/View/Show status bar" );
	gtk_check_menu_item_set_active( 
		GTK_CHECK_MENU_ITEM( item ), conv->status );

	item = gtk_item_factory_get_widget( iv->ifac, 
		"/View/Show paint bar" );
	gtk_check_menu_item_set_active( 
		GTK_CHECK_MENU_ITEM( item ), conv->paintbox );

	imageview_refresh_title( iv );
}

static void
imageview_replace_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );

	classmodel_graphic_replace( GTK_WIDGET( iv ), 
		CLASSMODEL( iv->iimage ) );
}

static const char *imageview_region_name[] = {
	CLASS_POINT,
	CLASS_HGUIDE,
	CLASS_VGUIDE,
	CLASS_ARROW,
	CLASS_REGION,
	CLASS_AREA
};

static void
imageview_new_point_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	Conversion *conv = iv->conv;
	IMAGE *im = imageinfo_get( FALSE, conv->ii );
	iRegionType rt = (iRegionType) callback_action;
	int dx = conv->visible.left + conv->visible.width / 2;
	int dy = conv->visible.top + conv->visible.height / 2;

	char txt[NAMELEN];
	BufInfo buf;
	Symbol *sym;
	Column *col;
	int ix, iy;

	conversion_disp_to_im( iv->conv, dx, dy, &ix, &iy );
	ix -= im->Xoffset;
	iy -= im->Yoffset;

	buf_init_static( &buf, txt, NAMELEN );
	buf_appendf( &buf, "%s ", imageview_region_name[rt] );
	row_qualified_name_relative( 
		SYMBOL( main_workspacegroup->current ),
		HEAPMODEL( iv->iimage )->row, &buf );
	switch( rt ) {
	case IREGION_POINT:
		buf_appendf( &buf, " (%d) (%d)", ix, iy );
		break;

	case IREGION_HGUIDE:
		buf_appendf( &buf, " (%d)", iy );
		break;

	case IREGION_VGUIDE:
		buf_appendf( &buf, " (%d)", ix );
		break;

	default:
		assert( FALSE );
	}

	if( !(sym = mainw_def_new( buf_all( &buf ) )) ) {
		box_alert( GTK_WIDGET( iv ) );
		return;
	}

	col = sym->expr->row->top_col;
	model_scrollto( MODEL( col ) );
}

static void
imageview_new_region_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	Conversion *conv = iv->conv;
	IMAGE *im = imageinfo_get( FALSE, conv->ii );
	iRegionType rt = (iRegionType) callback_action;

	Rect dr, ir;
	char txt[NAMELEN];
	BufInfo buf;
	Symbol *sym;
	Column *col;

	dr.left = iv->conv->visible.left + iv->conv->visible.width / 4;
	dr.top = iv->conv->visible.top + iv->conv->visible.height / 4;
	dr.width = iv->conv->visible.width / 2;
	dr.height = iv->conv->visible.height / 2;
	conversion_disp_to_im_rect( iv->conv, &dr, &ir );
	ir.left -= im->Xoffset;
	ir.top -= im->Yoffset;

	buf_init_static( &buf, txt, NAMELEN );
	buf_appendf( &buf, "%s ", imageview_region_name[rt] );
	row_qualified_name_relative( 
		SYMBOL( main_workspacegroup->current ),
		HEAPMODEL( iv->iimage )->row, &buf );
	buf_appendf( &buf, " (%d) (%d) %d %d", 
		ir.left, ir.top, ir.width, ir.height );

	if( !(sym = mainw_def_new( buf_all( &buf ) )) ) {
		box_alert( GTK_WIDGET( iv ) );
		return;
	}

	col = sym->expr->row->top_col;
	model_scrollto( MODEL( col ) );
}

static void
imageview_save_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );

	classmodel_graphic_save( GTK_WIDGET( iv ), 
		CLASSMODEL( iv->iimage ) );
}

static void
imageview_header_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	GtkWidget *imageheader = imageheader_new( iv->conv );
	BufInfo buf;
	char txt[512];

	buf_init_static( &buf, txt, 512 );
	row_qualified_name_relative( 
		SYMBOL( main_workspacegroup->current ),
		HEAPMODEL( iv->iimage )->row, &buf );

	iwindow_set_title( IWINDOW( imageheader ), "Header for %s", 
		buf_all( &buf ) );
	idialog_set_callbacks( IDIALOG( imageheader ), 
		NULL, NULL, NULL, NULL, iv );
	idialog_add_ok( IDIALOG( imageheader ), iwindow_true_cb, "OK" );
	idialog_set_parent( IDIALOG( imageheader ), GTK_WIDGET( iv ) );
	iwindow_build( IWINDOW( imageheader ) );

	gtk_widget_show( imageheader );
}

static void
imageview_quit_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );

	iwindow_kill( IWINDOW( iv ) );
}

static void
imageview_recalc_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	Row *row = HEAPMODEL( iv->iimage )->row;
	Workspace *ws = row->top_col->ws;

        workspace_deselect_all( ws );
        row_select( row );
        if( !mainw_selected_recalc() )
                box_alert( NULL );
        workspace_deselect_all( ws );
}

static void
imageview_menu_mag_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	int mag = (int) callback_action;
	GtkCheckMenuItem *item = GTK_CHECK_MENU_ITEM( widget );

	if( item->active ) 
		imagedisplay_set_mag( iv->ip->id, mag );
}

static void
imageview_menu_mode_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	ImagepresentState state = (ImagepresentState) callback_action;
	GtkCheckMenuItem *item = GTK_CHECK_MENU_ITEM( widget );

	if( item->active ) 
		imagepresent_set_state( iv->ip, state );
}

static void
imageview_menu_fit_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );

	imagedisplay_set_mag( iv->ip->id, 0 );
}

static void
imageview_menu_show_rulers_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	GtkCheckMenuItem *item = GTK_CHECK_MENU_ITEM( widget );

	imagepresent_rulers( iv->ip, item->active );
}

static void
imageview_menu_show_status_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	GtkCheckMenuItem *item = GTK_CHECK_MENU_ITEM( widget );

	iv->conv->status = item->active;
	model_changed( MODEL( iv->conv ) );
}

void
imageview_set_paint( Imageview *iv, gboolean paint )
{
	if( paint ) {
		/* Just turn on paint mode ... will pop open the bar.
		 */
		imagepresent_set_state( iv->ip, 
			IMAGEPRESENT_STATE_PAINT );
	}
	else {
		/* Only turn off paint mode if we're in paint mode.
		 */
		if( iv->ip->state == IMAGEPRESENT_STATE_PAINT )
			imagepresent_set_state( iv->ip, 
				IMAGEPRESENT_STATE_SELECT );

		/* Hide explicitly.
		 */
		iv->conv->paintbox = FALSE;
		model_changed( MODEL( iv->conv ) );
	}
}

static void
imageview_menu_show_paintbox_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );

	imageview_set_paint( iv, GTK_CHECK_MENU_ITEM( widget )->active );
}

static void
imageview_menu_show_convert_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	GtkCheckMenuItem *item = GTK_CHECK_MENU_ITEM( widget );

	iv->conv->enabled = item->active;
	conversion_refresh( iv->conv );
}

/* Menu bar.
 */
static GtkItemFactoryEntry imageview_menu_items[] = {
	{ "/_File",            			NULL,         
		NULL,			0, "<Branch>" },
	{ "/File/_New", 			NULL, 
		NULL, 			0, "<Branch>" },
	{ "/File/New/_Point", 			NULL,
		imageview_new_point_cb,	IREGION_POINT },
	{ "/File/New/_HGuide", 			NULL,
		imageview_new_point_cb,	IREGION_HGUIDE },
	{ "/File/New/_VGuide", 			NULL,
		imageview_new_point_cb,	IREGION_VGUIDE },
	{ "/File/New/_Arrow", 			NULL,
		imageview_new_region_cb,IREGION_ARROW },
	{ "/File/New/_Region", 			NULL,
		imageview_new_region_cb,IREGION_REGION },
	{ "/File/sep3",        			NULL,         
		NULL,       		0, "<Separator>" },
	{ "/File/_Replace image ...",		"<control>R", 
		imageview_replace_cb,  	0 },
	{ "/File/_Save image ...",     		"<control>S", 
		imageview_save_cb,     	0 },
	{ "/File/sep1",        			NULL,         
		NULL,       		0, "<Separator>" },
	{ "/File/View _header ...",     	NULL, 
		imageview_header_cb,   	0 },
	{ "/File/sep3",        			NULL,         
		NULL,       		0, "<Separator>" },
	{ "/File/Re_calculate image",		"<control>C", 
		imageview_recalc_cb,   	0 },
	{ "/File/sep2",        			NULL,         
		NULL,       		0, "<Separator>" },
	{ "/File/_Quit",       			"<control>Q", 
		imageview_quit_cb,     	0 },

	{ "/_View",       			NULL,		
		NULL,			0, "<Branch>" },
	{ "/View/_Magnification",		"", 
		NULL,       		0, "<Branch>" },
	{ "/View/Magnification/6%",		"", 
		imageview_menu_mag_cb, 	-16, "<RadioItem>" },
	{ "/View/Magnification/12%",		"<control>8", 
		imageview_menu_mag_cb, 	-8, "/View/Magnification/6%" },
	{ "/View/Magnification/25%",		"<control>4", 
		imageview_menu_mag_cb,  -4, "/View/Magnification/6%" },
	{ "/View/Magnification/50%",		"<control>2", 
		imageview_menu_mag_cb,  -2, "/View/Magnification/6%" },
	{ "/View/Magnification/100%",		"1", 
		imageview_menu_mag_cb,  1, "/View/Magnification/6%" },
	{ "/View/Magnification/200%",		"2", 
		imageview_menu_mag_cb,  2, "/View/Magnification/6%" },
	{ "/View/Magnification/400%",		"4", 
		imageview_menu_mag_cb,  4, "/View/Magnification/6%" },
	{ "/View/Magnification/800%",		"8", 
		imageview_menu_mag_cb,  8, "/View/Magnification/6%" },
	{ "/View/Magnification/1600%",		"", 
		imageview_menu_mag_cb,  16, "/View/Magnification/6%" },
	{ "/View/M_ode",		"", 
		NULL,       		0, "<Branch>" },
	{ "/View/Mode/_Select",		"", 
		imageview_menu_mode_cb, 
			IMAGEPRESENT_STATE_SELECT, "<RadioItem>" },
	{ "/View/Mode/_Pan",		"", 
		imageview_menu_mode_cb, 
			IMAGEPRESENT_STATE_PAN, "/View/Mode/Select" },
	{ "/View/Mode/Zoom _in",		"", 
		imageview_menu_mode_cb, 
			IMAGEPRESENT_STATE_MAGIN, "/View/Mode/Select" },
	{ "/View/Mode/Zoom _out",		"", 
		imageview_menu_mode_cb, 
			IMAGEPRESENT_STATE_MAGOUT, "/View/Mode/Select" },
	{ "/View/Mode/P_aint",		"", 
		imageview_menu_mode_cb, 
			IMAGEPRESENT_STATE_PAINT, "/View/Mode/Select" },
	{ "/View/_Fit to window",		"0", 
		imageview_menu_fit_cb,  0 },
	{ "/View/sep1",        			NULL,         
		NULL,       		0, "<Separator>" },
	{ "/View/Show _rulers", 			"", 
		imageview_menu_show_rulers_cb,    	0, "<CheckItem>" },
	{ "/View/Show _status bar", 		"", 
		imageview_menu_show_status_cb,       	0, "<CheckItem>" },
	{ "/View/Show _display control bar", 	"", 
		imageview_menu_show_convert_cb,       	0, "<CheckItem>" },
	{ "/View/Show _paint bar", 		"", 
		imageview_menu_show_paintbox_cb,       	0, "<CheckItem>" },

	{ "/_Help",            			NULL,         
		NULL,			0, "<LastBranch>" },
	{ "/Help/_Image view",  		NULL,         
		box_help_cb,		GPOINTER_TO_UINT( "sec:view" ) },
	{ "/Help/_Paint bar",  			NULL,         
		box_help_cb,		GPOINTER_TO_UINT( "sec:paintbox" ) }
};

/* Spot mouse motion events, to update status bar.
 */
static gint
imageview_event( GtkWidget *widget, GdkEvent *event, Imageview *iv )
{
#ifdef EVENT
	if( event->type == GDK_BUTTON_PRESS )
		printf( "imageview_event: GDK_BUTTON_PRESS\n" );
#endif /*EVENT*/

	if( event->type == GDK_MOTION_NOTIFY ) { 
		int dx, dy;
		int ix, iy;

		imagedisplay_xev_to_disp( iv->ip->id, 
			event->button.x, event->button.y, &dx, &dy );
		conversion_disp_to_im( iv->conv, dx, dy, &ix, &iy );

		statusview_mouse( iv->sv, ix, iy );
	}

	return( FALSE );
}

/* Some ip child visibility has changed ... update our toggle menus.
 */
static void
imageview_ip_changed( Imagepresent *ip, Imageview *iv )
{
	GtkWidget *item;

	item = gtk_item_factory_get_widget( iv->ifac, "/View/Show rulers" );
	gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM( item ),
		ip->show_rulers );
}

/* The main workspacegroup has changed ... prolly new current workspace.
 */
static void
imageview_workspace_changed_cb( Workspacegroup *wsg, Imageview *iv )
{
	imageview_refresh_title( iv );
}

static gboolean
imageview_filedrop( Imageview *iv, const char *file )
{
	gboolean result;

	result = iimage_replace( iv->iimage, file );
	if( result )
		symbol_recalculate_all();

	return( result );
}

static void
imageview_build( Imageview *iv, GtkWidget *vbox, Iimage *iimage )
{
	GtkAccelGroup *accel_group;
	GtkWidget *mbar;
	GtkWidget *magmenu;
	GtkWidget *item;
	GSList *radio_group;

	int w, h; 

	iv->iimage = iimage;

	/* Make menu bar
	 */
	accel_group = gtk_accel_group_new();
	iv->ifac = gtk_item_factory_new( GTK_TYPE_MENU_BAR, 
		"<main>", accel_group );
	gtk_item_factory_create_items( iv->ifac, 
		IM_NUMBER( imageview_menu_items ), imageview_menu_items, iv );
	gtk_accel_group_attach( accel_group, GTK_OBJECT( iv ) );
	mbar = gtk_item_factory_get_widget( iv->ifac, "<main>" );
	gtk_box_pack_start( GTK_BOX( vbox ), mbar, FALSE, FALSE, 0 );
	gtk_widget_show( mbar );

	/* Converter.
	 */
	iv->conv = conversion_new( NULL );

	/* Status bar.
	 */
	iv->sv = statusview_new( iv->conv, iv->iimage );
	gtk_box_pack_start( GTK_BOX( vbox ), 
		GTK_WIDGET( iv->sv ), FALSE, FALSE, 0 );

	/* Conversion bar.
	 */
	iv->cv = conversionview_new( iv->conv );
	gtk_box_pack_start( GTK_BOX( vbox ), 
		GTK_WIDGET( iv->cv ), FALSE, FALSE, 0 );

	/* Paintbox bar.
	 */
	iv->pbv = paintboxview_new( iv );
	gtk_box_pack_start( GTK_BOX( vbox ), 
		GTK_WIDGET( iv->pbv ), FALSE, FALSE, 0 );

	/* Image area. 
	 */
	iv->ip = imagepresent_new( iimage, iv->conv );
	gtk_box_pack_start( GTK_BOX( vbox ), 
		GTK_WIDGET( iv->ip ), TRUE, TRUE, 0 );
	gtk_widget_show( GTK_WIDGET( iv->ip ) );
	gtk_signal_connect( GTK_OBJECT( iv->ip->id ), "event",
		GTK_SIGNAL_FUNC( imageview_event ), iv );
	gtk_signal_connect( GTK_OBJECT( iv->ip ), "changed",
		GTK_SIGNAL_FUNC( imageview_ip_changed ), iv );

	iv->changed_sid = gtk_signal_connect( GTK_OBJECT( iv->iimage ), 
		"changed", GTK_SIGNAL_FUNC( imageview_changed_cb ), iv );
	iv->destroy_sid = gtk_signal_connect( GTK_OBJECT( iv->iimage ), 
		"destroy", GTK_SIGNAL_FUNC( imageview_destroy_cb ), iv );
	iv->conv_changed_sid = gtk_signal_connect( GTK_OBJECT( iv->conv ), 
		"changed", GTK_SIGNAL_FUNC( imageview_conv_changed_cb ), iv );

	/* Turn off ugly visible toggles on mag menu.
	 */
	magmenu = gtk_item_factory_get_widget( iv->ifac, 
		"/View/Magnification/6%" );
	radio_group = gtk_radio_menu_item_group( 
		GTK_RADIO_MENU_ITEM( magmenu ) );
	while( radio_group ) {
		GtkWidget *item = GTK_WIDGET( radio_group->data );

		gtk_check_menu_item_set_show_toggle( 
			GTK_CHECK_MENU_ITEM( item ), FALSE );
		
		radio_group = radio_group->next;
	}

	/* Set start state for menus.
	 */
	item = gtk_item_factory_get_widget( iv->ifac, "/View/Show rulers" );
	gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM( item ),
		iv->ip->show_rulers );

	/* Install image.
	 */
	conversion_set_image( iv->conv, iimage->instance.image );

	/* Position and size to restore?
	 */
	if( iimage->window_width != -1 ) {
		gtk_widget_set_uposition( GTK_WIDGET( iv ), 
			iimage->window_x, iimage->window_y );
		gtk_window_set_default_size( GTK_WINDOW( iv ),
			iimage->window_width, iimage->window_height );

		imagedisplay_set_mag( iv->ip->id, 
			iimage->image_mag );
		imagedisplay_set_position( iv->ip->id, 
			iimage->image_left, iimage->image_top );

		iv->conv->status = iimage->show_status;
		iv->conv->paintbox = iimage->show_paintbox;
		iv->conv->enabled = iimage->show_convert;
		iv->conv->scale = iimage->scale;
		iv->conv->offset = iimage->offset;
		conversion_refresh( iv->conv );
		imagepresent_rulers( iv->ip, iimage->show_rulers );
	}
	else {
		/* Set initial size. This is really hard to do right :-( for
		 * now, just hack in some numbers that make it come out right
		 * for me ... gtk2 fixes this (I think).
		 */
		w = IM_MIN( IMAGE_WINDOW_WIDTH, iv->conv->image.width + 31 );
		h = IM_MIN( IMAGE_WINDOW_HEIGHT, iv->conv->image.height + 57 );
		gtk_window_set_default_size( GTK_WINDOW( iv ), w, h );
		imagedisplay_set_mag( iv->ip->id, 1 );
	}

	gtk_widget_grab_focus( GTK_WIDGET( iv->ip->id ) );

	/* Look for workspace switch and refresh title.
	 */
	iv->ws_changed_sid = gtk_signal_connect( 
		GTK_OBJECT( main_workspacegroup ), "changed", 
		GTK_SIGNAL_FUNC( imageview_workspace_changed_cb ), iv );

	/* Set as file drop destination 
	 */
	filedrop_register( GTK_WIDGET( iv ), 
		(FiledropFunc) imageview_filedrop, iv );
}

static void
imageview_popdown( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	Imageview *iv = IMAGEVIEW( iwnd );

#ifdef DEBUG
	printf( "imageview_popdown: popdown!\n" );
#endif /*DEBUG*/

	/* We have to note position/size in popdown rather than destroy, since
	 * the widgets have to all still be extant.
	 */

	/* Note position/size for later reuse.
	 */
	iv->iimage->window_width = GTK_WIDGET( iv )->allocation.width;
	iv->iimage->window_height = GTK_WIDGET( iv )->allocation.height;
	iv->iimage->image_mag = iv->conv->mag;
	gdk_window_get_root_origin( 
		gtk_widget_get_toplevel( GTK_WIDGET( iv ) )->window, 
		&iv->iimage->window_x, &iv->iimage->window_y );

	/* Memorise centre of screen, in image cods.
	 */
	iv->iimage->image_left = iv->ip->id->visible.left;
	iv->iimage->image_top = iv->ip->id->visible.top;

	iv->iimage->show_status = iv->conv->status;
	iv->iimage->show_paintbox = iv->conv->paintbox;
	iv->iimage->show_convert = iv->conv->enabled;
	iv->iimage->show_rulers = iv->ip->show_rulers;

	iv->iimage->scale = iv->conv->scale;
	iv->iimage->offset = iv->conv->offset;

	nfn( sys, IWINDOW_TRUE );
}

static void *
imageview_add_region( Classmodel *classmodel, Imageview *iv )
{
	if( MODEL( classmodel )->display ) {
		IregionInstance *instance = 
			classmodel_get_instance( classmodel );
		Regionview *regionview = regionview_new( classmodel, 
			&instance->area, iv->ip );
		PElement *root = &HEAPMODEL( classmodel )->row->expr->root;

		/* Look at the class we are drawing, set the display type.
		 */
		regionview_set_type( regionview, root );
	}

	return( NULL );
}

static void
imageview_link( Imageview *iv, Iimage *iimage )
{
	iwindow_set_build( IWINDOW( iv ), 
		(iWindowBuildFn) imageview_build, iimage, NULL, NULL );
	iwindow_set_popdown( IWINDOW( iv ), imageview_popdown, NULL );
	iwindow_build( IWINDOW( iv ) );
	iimage->views = g_slist_prepend( iimage->views, iv );

	slist_map( iimage->classmodels,
		(SListMapFn) imageview_add_region, iv );
}

Imageview *
imageview_new( Iimage *iimage )
{
	Imageview *iv = gtk_type_new( TYPE_IMAGEVIEW );

	imageview_link( iv, iimage );

	return( iv );
}

/* Make an imageview, and try to make area (image cods) visible. width/height
 * can be -ve
 */
Imageview *
imageview_new_area( Iimage *iimage, Rect *area )
{
	Imageview *iv = imageview_new( iimage );
	int shrink_x, shrink_y, shrink;

	/* Have to show it ... need window size.
	 */
	gtk_widget_show( GTK_WIDGET( iv ) );

	/* Calculate a shrink factor which should make all the region 
	 * visible ... don't zoom.
	 */
	shrink_x = (abs( area->width ) + iv->ip->id->screen.width) / 
		iv->ip->id->screen.width;
	shrink_y = (abs( area->height ) + iv->ip->id->screen.height) / 
		iv->ip->id->screen.height;
	shrink = -IM_MAX( 1, IM_MAX( shrink_x, shrink_y ) );
	if( shrink == -1 )
		shrink = 1;

	imagedisplay_set_mag_position( iv->ip->id, shrink, 
		area->left + area->width / 2, 
		area->top + area->height / 2 );

	return( iv );
}
