/* main processing window
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#include "ip.h"

/* 
#define DEBUG
 */

/* Startup window size.
 */
#define MAINW_WINDOW_WIDTH (watch_int_get( "MAINW_WINDOW_WIDTH", 640 ))
#define MAINW_WINDOW_HEIGHT (watch_int_get( "MAINW_WINDOW_HEIGHT", 480 ))

/* Main ip window.
 */
GtkWidget *mainw_window = NULL;			/* Main window */
GtkWidget *mainw_message = NULL;		/* Label for messages */

/* Views of toolkits and workspaces we use.
 */
Toolkitgroupview *mainw_kitgview = NULL;
Workspacegroupview *mainw_wsgview = NULL;

static GtkWidget *mainw_next_error = NULL;	/* Next error button */
static GtkWidget *mainw_space_free = NULL;	/* MB free message area */
static GtkWidget *mainw_space_free_eb = NULL;	/* Enclosing eb for the above */

static GtkWidget *mainw_timer_box = NULL;	/* %to go area */
static GtkWidget *mainw_togo = NULL;		/* togo scrollbar */

/* Display MB free in tmp, or cells free in heap.
 */
static gboolean mainw_free_type = TRUE;

static void
mainw_exit_sub( void *sys, iWindowResult result )
{
	iWindowSusp *susp = IWINDOW_SUSP( sys );

	if( result == IWINDOW_TRUE )
		/* No return from this.
		 */
		main_quit();

	iwindow_susp_return( susp, result );
}

/* Main window popdown callback.
 */
static void
mainw_popdown_cb( iWindow *iwnd, void *client, iWindowNotifyFn nfn, void *sys )
{
	iWindowSusp *susp = iwindow_susp_new( NULL, iwnd, client, nfn, sys );

	/* Iconised? Quit immediately.
	 */
	if( !GTK_WIDGET_MAPPED( mainw_window ) )
		main_quit();

	/* Close registered models.
	 */
	filemodel_inter_close_registered_cb( IWINDOW( mainw_window ), NULL,
		mainw_exit_sub, susp );
}

/* Quit ip menu item selected.
 */
static void
mainw_exit_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	mainw_popdown_cb( IWINDOW( mainw_window ), NULL,
		iwindow_notify_null, NULL );
}

/* Clone the selected object(s), or the bottom object.
 */
static void
mainw_clone_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Workspace *ws = main_workspacegroup->current;

	if( !workspace_selected_any( ws ) ) {
		/* Nothing selected -- select bottom object.
		 */
		Row *row = workspace_get_bottom( ws );

		if( !row ) {
			box_alert( NULL );
			return;
		}
		row_select( row );

		/* Clone that object.
		 */
		if( !workspace_clone_selected( ws ) ) {
			box_alert( NULL );
			workspace_deselect_all( ws );
			return;
		}
		workspace_deselect_all( ws );
		symbol_recalculate_all();
	}
	else {
		/* Clone selected symbols.
		 */
		set_hourglass();
		if( !workspace_clone_selected( ws ) ) { 
			box_alert( NULL );
			set_pointer();
			return;
		}
		symbol_recalculate_all();
		workspace_deselect_all( ws );
		set_pointer();
	}

	model_scrollto( MODEL( ws->current ) );
}

/* Sub fn of below ... add a new index expression.
 */
static void *
mainw_ungroup_add_index( PElement *head, int *n, Row *row )
{
	Symbol *new;

	static BufInfo buf;
	static char txt[200];

	buf_init_static( &buf, txt, 200 );
	row_qualified_name( row, &buf );
	buf_appendf( &buf, "?%d", *n );
	if( !(new = mainw_def_new( buf_all( &buf ) )) )
		return( sym );

	*n += 1;

	return( NULL );
}

/* Ungroup a row.
 */
void *
mainw_ungroup( Row *row )
{
	PElement *root = &row->expr->root;
	int n;

	if( !PEISLIST( root ) ) {
		ierrors( "you can only ungroup lists (comma-separated lists "
			"of things enclosed in square brackets)\n"
			"use Format=>Decompose to break compound objects into "
			"lists" );
		return( row );
	}

	n = 0;
	if( heap_map_list( root,
		(heap_map_list_fn) mainw_ungroup_add_index, &n, row ) )
		box_alert( NULL );

	return( NULL );
}

/* Ungroup the selected object(s), or the bottom object.
 */
static void
mainw_ungroup_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Workspace *ws = main_workspacegroup->current;

	if( !workspace_selected_any( ws ) ) {
		/* Nothing selected -- ungroup bottom object.
		 */
		Symbol *sym = workspace_get_bottom( ws )->sym;

		if( !sym ) {
			box_alert( NULL );
			return;
		}

		if( mainw_ungroup( sym->expr->row ) ) 
			box_alert( NULL );
		symbol_recalculate_all();
	}
	else {
		/* Ungroup selected symbols.
		 */
		set_hourglass();
		if( workspace_selected_map( ws,
			(row_map_fn) mainw_ungroup, NULL, NULL ) ) { 
			box_alert( NULL );
			symbol_recalculate_all();
			set_pointer();
			return;
		}
		symbol_recalculate_all();
		workspace_deselect_all( ws );
		set_pointer();
	}
}

/* Make a new symbol at the bottom of the current column in the current
 * workspace.
 */
Symbol *
mainw_def_new( const char *rhs )
{
	Workspace *ws = main_workspacegroup->current;
	Symbol *sym;

	if( !(sym = workspace_add_def( ws, rhs )) ) 
		return( NULL );
	if( sym->expr->err ) {	
		expr_error_get( sym->expr );

		/* Block changes to error_string ... symbol_destroy() can set
		 * this for compound objects.
		 */
		error_block();
		gtk_object_destroy( GTK_OBJECT( sym ) );
		error_unblock();

		mainw_next_error_refresh();

		/* Can't recompute without trashing error buffer.
		 */

		return( NULL );
	}

	model_scrollto( MODEL( ws->current ) );

	return( sym );
}

/* Group the selected object(s).
 */
static void
mainw_group_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Workspace *ws = main_workspacegroup->current;

	if( workspace_selected_any( ws ) ) {
		BufInfo buf;
		char txt[MAX_STRSIZE];

		/* Group selected symbols.
		 */
		buf_init_static( &buf, txt, MAX_STRSIZE );
		buf_appends( &buf, "[" );
		workspace_selected_names( ws, &buf, "," );
		buf_appends( &buf, "]" );

		if( !mainw_def_new( buf_all( &buf ) ) ) {
			box_alert( NULL );
			return;
		}
		workspace_deselect_all( ws );
	}
	else {
		ierrors( "No objects selected" );
		box_alert( NULL );
	}
}

/* Select all objects.
 */
static void
mainw_select_all_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Workspace *ws = main_workspacegroup->current;

	workspace_select_all( ws );
}

static void
mainw_find_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	ierrors( "find in workspace not implemented yet" );
	box_alert( NULL );
}

static void
mainw_find_again_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	ierrors( "find again in workspace not implemented yet" );
	box_alert( NULL );
}

static void *
mainw_row_dirty( Row *row, int serial )
{
	return( expr_dirty( row->expr, serial ) );
}

/* Recalculate selected items.
 */
gboolean
mainw_selected_recalc( void )
{
	Workspace *ws = main_workspacegroup->current;

	if( workspace_selected_map( ws,
		(row_map_fn) mainw_row_dirty, 
		(void *) link_serial_new(), NULL ) ) 
		return( FALSE );

	/* Recalc even if autorecomp is off.
	 */
	symbol_recalculate_all_force();

	workspace_deselect_all( ws );

	return( TRUE );
}

/* Callback for box_yesno in mainw_force_calc_cb. Recalc selected items.
 */
static void
mainw_selected_recalc_dia( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	if( mainw_selected_recalc() )
		nfn( sys, IWINDOW_TRUE );
	else
		nfn( sys, IWINDOW_ERROR );
}

/* If symbols are selected, make them very dirty and recalculate. If not, 
 * just recalculate symbols which are already dirty.
 */
static void
mainw_force_calc_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Workspace *ws = main_workspacegroup->current;

        if( workspace_selected_any( ws ) ) {
		BufInfo buf;
		char str[30];

		buf_init_static( &buf, str, 30 );
		workspace_selected_names( ws, &buf, ", " );

		box_yesno( NULL, 
			mainw_selected_recalc_dia, iwindow_true_cb, NULL, 
			iwindow_notify_null, NULL,
			"Completely recalculate",
			"are you sure you want to "
			"completely recalculate %s?", buf_all( &buf ) );
	}
	else 
		/* Recalculate.
		 */
		symbol_recalculate_all_force();
}

/* Callback from load browser.
 */
static void *
mainw_matrix_insert_file_fn( Filesel *filesel,
	const char *filename, void *a, void *b )
{
	char txt[NAMELEN];
	BufInfo buf;
	Symbol *sym;
	Column *col;

	buf_init_static( &buf, txt, NAMELEN );
	buf_appends( &buf, "Matrix_file \"" );
	buf_appendsc( &buf, filename );
	buf_appends( &buf, "\"" );
	if( !(sym = mainw_def_new( buf_all( &buf ) )) ) 
		return( filesel );

	col = sym->expr->row->top_col;
	model_scrollto( MODEL( col ) );

	return( NULL );
}

/* Callback from load browser.
 */
static void
mainw_matrix_insert_file_cb( iWindow *iwnd, 
	void *client, iWindowNotifyFn nfn, void *sys )
{
	Filesel *filesel = FILESEL( iwnd );

	if( filesel_map_filename_multi( filesel,
		mainw_matrix_insert_file_fn, NULL, NULL ) ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	nfn( sys, IWINDOW_TRUE );
}

static void
mainw_matrix_insert_file( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	GtkWidget *filesel = filesel_new();

	iwindow_set_title( IWINDOW( filesel ), "Insert matrix" );
	filesel_set_flags( FILESEL( filesel ), FALSE, FALSE );
	filesel_set_type( FILESEL( filesel ), filesel_type_matrix, 0 ); 
	idialog_set_parent( IDIALOG( filesel ), mainw_window );
	filesel_set_done( FILESEL( filesel ), 
		mainw_matrix_insert_file_cb, NULL );
	gtk_file_selection2_set_multi( GTK_FILE_SELECTION2( filesel ),
		TRUE );
	iwindow_build( IWINDOW( filesel ) );

	gtk_widget_show( GTK_WIDGET( filesel ) );
}

/* Callback from load browser for images.
 */
static void *
mainw_image_insert_file_fn( Filesel *filesel,
	const char *filename, void *a, void *b )
{
	char txt[NAMELEN];
	BufInfo buf;
	Symbol *sym;
	Column *col;

	buf_init_static( &buf, txt, NAMELEN );
	buf_appends( &buf, "Image_file \"" );
	buf_appendsc( &buf, filename );
	buf_appends( &buf, "\"" );
	if( !(sym = mainw_def_new( buf_all( &buf ) )) ) 
		return( filesel );

	col = sym->expr->row->top_col;
	model_scrollto( MODEL( col ) );

	return( NULL );
}

/* Callback from load browser.
 */
static void
mainw_image_insert_file_cb( iWindow *iwnd, 
	void *client, iWindowNotifyFn nfn, void *sys )
{
	Filesel *filesel = FILESEL( iwnd );

	if( filesel_map_filename_multi( filesel,
		mainw_image_insert_file_fn, NULL, NULL ) ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	nfn( sys, IWINDOW_TRUE );
}

/* Callback from load image.
 */
static void
mainw_image_insert_file( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	GtkWidget *filesel = filesel_new();

	iwindow_set_title( IWINDOW( filesel ), "Insert image" );
	filesel_set_flags( FILESEL( filesel ), TRUE, FALSE );
	filesel_set_type( FILESEL( filesel ), 
		filesel_type_image, IMAGE_FILE_TYPE );
	idialog_set_parent( IDIALOG( filesel ), mainw_window );
	filesel_set_done( FILESEL( filesel ), 
		mainw_image_insert_file_cb, NULL );
	gtk_file_selection2_set_multi( GTK_FILE_SELECTION2( filesel ),
		TRUE );
	iwindow_build( IWINDOW( filesel ) );

	gtk_widget_show( GTK_WIDGET( filesel ) );
}

/* Merge into current workspace.
 */
static void *
mainw_workspace_merge_fn( Filesel *filesel,
	const char *filename, void *a, void *b )
{
	Workspace *ws = main_workspacegroup->current;

	if( !workspace_merge_file( ws, filename ) ) 
		return( filesel );

	/* Process some events to make sure we rethink the layout and
	 * are able to get the append-at-RHS offsets.
	 */
	while( g_main_iteration( FALSE ) )
		;

	return( NULL );
}

/* Callback from load browser.
 */
static void
mainw_workspace_merge_cb( iWindow *iwnd, 
	void *client, iWindowNotifyFn nfn, void *sys )
{
	Filesel *filesel = FILESEL( iwnd );

	if( filesel_map_filename_multi( filesel,
		mainw_workspace_merge_fn, NULL, NULL ) ) {
		symbol_recalculate_all();
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	symbol_recalculate_all();

	nfn( sys, IWINDOW_TRUE );
}

/* Merge into current workspace.
 */
static void
mainw_workspace_merge( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	GtkWidget *filesel = filesel_new();

	iwindow_set_title( IWINDOW( filesel ), "Merge workspace" );
	filesel_set_flags( FILESEL( filesel ), FALSE, FALSE );
	filesel_set_type( FILESEL( filesel ), filesel_type_workspace, 0 ); 
	idialog_set_parent( IDIALOG( filesel ), mainw_window );
	filesel_set_done( FILESEL( filesel ), mainw_workspace_merge_cb, NULL );
	gtk_file_selection2_set_multi( GTK_FILE_SELECTION2( filesel ),
		TRUE );
	iwindow_build( IWINDOW( filesel ) );

	gtk_widget_show( GTK_WIDGET( filesel ) );
}

/* Callback for save workspace.
 */
static void
mainw_workspace_save( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Workspace *ws = main_workspacegroup->current;

	if( ws ) 
		filemodel_inter_save( IWINDOW( mainw_window ),
			FILEMODEL( ws ) );
}

/* Callback for "save as .." workspace.
 */
static void
mainw_workspace_saveas( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Workspace *ws = main_workspacegroup->current;

	if( ws ) 
		filemodel_inter_saveas( IWINDOW( mainw_window ),
			FILEMODEL( ws ) );
}

/* Load a workspace at the top level.
 */
static void *
mainw_workspace_open_fn( Filesel *filesel,
	const char *filename, void *a, void *b )
{
	Workspace *ws;

	if( !(ws = workspace_new_from_file( main_workspacegroup, filename )) ) 
		return( filesel );

	workspacegroup_set_current( main_workspacegroup, ws );

	return( NULL );
}

/* Callback from load browser.
 */
static void
mainw_workspace_open_cb( iWindow *iwnd, 
	void *client, iWindowNotifyFn nfn, void *sys )
{
	Filesel *filesel = FILESEL( iwnd );

	if( filesel_map_filename_multi( filesel,
		mainw_workspace_open_fn, NULL, NULL ) ) {
		symbol_recalculate_all();
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	symbol_recalculate_all();

	nfn( sys, IWINDOW_TRUE );
}

/* Load ws callback.
 */
static void
mainw_workspace_open( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	GtkWidget *filesel = filesel_new();

	iwindow_set_title( IWINDOW( filesel ), "Open workspace" );
	filesel_set_flags( FILESEL( filesel ), FALSE, FALSE );
	filesel_set_type( FILESEL( filesel ), filesel_type_workspace, 0 ); 
	idialog_set_parent( IDIALOG( filesel ), mainw_window );
	filesel_set_done( FILESEL( filesel ), mainw_workspace_open_cb, NULL );
	gtk_file_selection2_set_multi( GTK_FILE_SELECTION2( filesel ),
		TRUE );
	iwindow_build( IWINDOW( filesel ) );

	gtk_widget_show( GTK_WIDGET( filesel ) );
}

/* Callback for close workspace.
 */
static void
mainw_workspace_close( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Workspace *ws = main_workspacegroup->current;

	if( ws ) 
		filemodel_inter_savenclose( IWINDOW( mainw_window ),
			FILEMODEL( ws ) );
}

/* New ws callback.
 */
static void
mainw_workspace_new( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	workspacegroup_new_edit( main_workspacegroup );
}

/* Auto recover.
 */
static void
mainw_recover_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	if( !workspace_auto_recover() )
		box_alert( NULL );
}

/* Callback from program.
 */
static void
mainw_program_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	(void) program_new();
}

/* Callback from make new column.
 */
static void
mainw_column_new_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Workspace *ws = main_workspacegroup->current;

	workspace_new_column( ws );
}

static void *
mainw_selected_remove_sub3( Row *row, int *nsel )
{
	if( row->selected )
		*nsel += 1;

	return( NULL );
}

static void *
mainw_selected_remove_sub2( Column *col, GSList **cs )
{
	int nsel = 0;

	(void) column_map( col, 
		(row_map_fn) mainw_selected_remove_sub3, &nsel, NULL );
	if( nsel > 0 )
		*cs = g_slist_prepend( *cs, col );

	return( NULL );
}

static void *
mainw_selected_remove_sub4( Column *col )
{
	Subcolumn *scol = col->scol;
	int nmembers = g_slist_length( MODEL( scol )->children );

	if( nmembers > 0 ) 
		model_pos_renumber( MODEL( scol ) );
	else
		FREEO( col );

	return( NULL );
}

/* Callback for box_yesno in mainw_selected_remove_cb. Remove selected items.
 *
 * 0. check all objects to be destroyed are top level rows
 * 1. look for and note all columns containing items we are going to delete
 * 2. loop over selected items, and delete them one-by-one.
 * 3. loop over the columns we noted in 1 and delete empty ones
 * 4. renumber rows in all remaining colums

 	FIXME ... should be part of column.c?

 */
static void
mainw_selected_remove_sub( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	Workspace *ws = main_workspacegroup->current;
	GSList *cs = NULL;

	(void) workspace_map_column( ws, 
		(column_map_fn) mainw_selected_remove_sub2, &cs );
	(void) workspace_selected_map_sym( ws,
		(symbol_map_fn) object_destroy, NULL, NULL );
	(void) slist_map( cs, 
		(SListMapFn) mainw_selected_remove_sub4, NULL );

	FREEF( g_slist_free, cs );
	symbol_recalculate_all();

	nfn( sys, IWINDOW_TRUE );
}

static void *
mainw_selected_remove2( Row *row )
{
	if( row != row->top_row )
		return( row );

	return( NULL );
}

/* Remove selected items.
 */
void
mainw_selected_remove( void )
{
	Workspace *ws = main_workspacegroup->current;
	BufInfo buf;
	char str[30];
	Row *row;

	if( (row = (Row *) workspace_selected_map( ws,
		(row_map_fn) mainw_selected_remove2, NULL, NULL )) ) {
		box_error( NULL,
			"you can only remove top level rows\n"
			"not all selected objects are top level rows" );
		return;
	}

	buf_init_static( &buf, str, 30 );
	workspace_selected_names( ws, &buf, ", " );

	box_yesno( mainw_window, 
		mainw_selected_remove_sub, iwindow_true_cb, NULL, 
		iwindow_notify_null, NULL,
		"Remove",
		"are you sure you want to remove %s?", buf_all( &buf ) );
}

/* Remove selected items.
 */
static void
mainw_selected_remove_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Workspace *ws = main_workspacegroup->current;

        if( workspace_selected_any( ws ) ) 
		mainw_selected_remove();
	else
		box_info( NULL,
			"select object(s) with left-click, control-left-click "
			"and shift-left-click,\nthen select Remove "
			"Selected." );
}

/* Make a magic definition for the selected symbol.

	FIXME .. paste this back when magic is reinstated

static void
mainw_magic_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Workspace *ws = main_workspacegroup->current;
	Row *row = workspace_selected_one( ws );

	if( !row )
		box_info( NULL,
			"Select a single object with left-click,\n"
			"select Magic Definition." );
	else if( !magic_sym( row->sym ) )
		box_alert( NULL );
	else
		workspace_deselect_all( ws );
}
 */

/* Toggle of show values. Change col->sform for the current column.
 */
static void
mainw_show_formula_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Column *col = main_workspacegroup->current->current;

	if( col ) {
		if( col && !col->sform ) {
			/* Show formula on.
			 */
			col->sform = TRUE;
			model_reset( MODEL( col ) );
		}
		else if( col && col->sform ) {
			/* Show formula off.
			 */
			col->sform = FALSE;
			model_reset( MODEL( col ) );
		}
	}
}

/* Update the space remaining indicator. Can be called as a close callback, 
 * so return 0.
 */
int
mainw_free_update( void )
{
	Workspace *ws;
	Heap *heap = reduce_context->hi;
	double sz;
	char txt[NAMELEN];
	BufInfo buf;

	if( !mainw_space_free || !main_workspacegroup )
		return( 0 );

	ws = main_workspacegroup->current;
	if( ws && workspace_selected_any( ws ) )
		return( 0 );

	buf_init_static( &buf, txt, NAMELEN );
	sz = find_space( PATH_TMP );
	if( sz < 0 )
		buf_appendf( &buf, "No temp area" );
	else {
		to_size( &buf, sz );
		buf_appendf( &buf, " free" );
	}

	/* Out of space? Make sure we swap to cell display.
	 */
	if( !heap->free )
		mainw_free_type = FALSE;

	if( mainw_free_type ) 
		set_glabel( mainw_space_free, "%s", buf_all( &buf ) );
	else {
		/* How much we can still expand the heap by ... this can be
		 * -ve if we've closed a workspace, or changed the upper
		 * limit.
		 */
		int togo = IM_MAX( 0, (heap->mxb - heap->nb) * heap->rsz );

		set_glabel( mainw_space_free, "%d cells free", 
			heap->nfree + togo );
	}

        set_tooltip( mainw_space_free_eb, 
                "%s in \"%s\", "
		"%d cells in heap, %d cells free, %d cells maximum",
		buf_all( &buf ), PATH_TMP,
                heap->ncells, heap->nfree, heap->max_fn( heap ) );

	return( 0 );
}

/* Update the selection message.
 */
void
mainw_select_message_update()
{
	Workspace *ws = main_workspacegroup->current;

	if( !ws || !workspace_selected_any( ws ) )
		/* Display mbytes free instead.
		 */
		mainw_free_update();
	else {
		BufInfo buf;
		char str[40];

		buf_init_static( &buf, str, 40 );
		workspace_selected_names( ws, &buf, ", " );
		set_glabel( mainw_space_free, "Selected: %s", buf_all( &buf ) );
	}
}

/* Load up a Pixmap to use as the icon for the root window. 
 */
static GdkBitmap *image_icons[4] = { NULL };
static GdkBitmap *image_icon = NULL;
static int next_automatic_icon = 0;

#include "BITMAPS/automatic.xbm"
#include "BITMAPS/automatic1.xbm"
#include "BITMAPS/automatic2.xbm"
#include "BITMAPS/automatic3.xbm"

static void
mainw_icon_load( void )
{
	if( !image_icons[0] ) {
		GdkBitmap *px, *px1, *px2, *px3;

		px = gdk_bitmap_create_from_data( mainw_window->window,
			(char *) automatic_bits, 
			automatic_width, automatic_height );
		px1 = gdk_bitmap_create_from_data( mainw_window->window,
			(char *) automatic1_bits, 
			automatic1_width, automatic1_height );
		px2 = gdk_bitmap_create_from_data( mainw_window->window,
			(char *) automatic2_bits, 
			automatic2_width, automatic2_height );
		px3 = gdk_bitmap_create_from_data( mainw_window->window,
			(char *) automatic3_bits, 
			automatic3_width, automatic3_height );

		image_icons[0] = px1;
		image_icons[1] = px3;
		image_icons[2] = px1;
		image_icons[3] = px2;
		image_icon = px;
	}
}

/* Set and animate the icon.
 */
static void
mainw_icon_animate( int pc ) 
{
	char buf[256];

	if( !mainw_window )
		return;

	mainw_icon_load();
	gdk_window_set_icon( mainw_window->window, NULL,
		image_icons[next_automatic_icon], 
		image_icons[next_automatic_icon] );

	im_snprintf( buf, 256, IP_NAME " - %d%%", pc );
	gdk_window_set_icon_name( mainw_window->window, buf );

	next_automatic_icon = (next_automatic_icon + 1) % 4;
}

/* Set the non-animated state.
 */
static void
mainw_icon_restore( void ) 
{
	if( !mainw_window )
		return;

	mainw_icon_load();
	gdk_window_set_icon( mainw_window->window, NULL,
		image_icon, image_icon );
	gdk_window_set_icon_name( mainw_window->window, IP_NAME );
}

static void
mainw_countdown_show( void )
{
	if( !mainw_window )
		return;

	if( GTK_WIDGET_VISIBLE( mainw_message ) ) {
		set_hourglass();
		gtk_widget_show( mainw_timer_box );
		gtk_widget_hide( mainw_message );

		while( g_main_iteration( FALSE ) )
			;
	}
}

/* Set new values for the countdown widgets.
 */
static void
mainw_countdown_update( int percent )
{
	if( !mainw_window )
		return;

	if( percent == 99 ) {
		gtk_progress_set_activity_mode( GTK_PROGRESS( mainw_togo ), 1 );
	}
	else {
		gtk_progress_set_activity_mode( GTK_PROGRESS( mainw_togo ), 0 );
		gtk_progress_set_percentage( GTK_PROGRESS( mainw_togo ), 
			IM_CLIP( 0.0, (float)percent / 100, 1.0 ) );
	}

	mainw_free_update();
}

static int countdown_timeout = 0;

static gint
mainw_countdown_timeout( void *dummy )
{
	countdown_timeout = 0;
	
	if( GTK_WIDGET_VISIBLE( mainw_timer_box ) ) {
		gtk_widget_show( mainw_message );
		gtk_widget_hide( mainw_timer_box );
		mainw_select_message_update();
		mainw_icon_restore();
		mainw_free_update();
		set_pointer();
	}

	return( FALSE );
}

/* Set if we're currently computing something.
 */
static gboolean mainw_countdown_computing = FALSE;

/* Make sure we are not displaying the countdown timer. Called on evalend,
 * close etc. Don't kill immediately to stop too much flashing.
 */
int
mainw_countdown_end( void )
{
	mainw_countdown_computing = FALSE;

	if( !mainw_window )
		return( 0 );

	/* Reset any prior timeout.
	 */
	FREEFI( gtk_timeout_remove, countdown_timeout );

	countdown_timeout = gtk_timeout_add( 500, 
		(GtkFunction) mainw_countdown_timeout, NULL );

	return( 0 );
}

/* Has the stop button been pressed?
 */
static gboolean mainw_countdown_cancel_pushed = FALSE;

/* Try to cancel the current computation.
 */
void
mainw_countdown_cancel( void )
{
	mainw_countdown_cancel_pushed = TRUE;
}

/* Animate the countdown timer. Return non-zero for cancel. Delay switching to
 * countdown mode to reduce flashing.
 */
int
mainw_countdown_animate( int percent )
{
	/* Start this timer when we get the first mainw_countdown_animate(). 
	 * If we animate for more than start_threshold sec, display 
	 * the cancel widgets. Update display every update_threshold after
	 * that.
	 */
	static GTimer *mainw_countdown_timer = NULL;
	static double last_update;

	const double start_threshold = 0.5;
	const double update_threshold = 0.2;

	double elapsed_time;

	if( !mainw_window )
		return( 0 );

	if( !mainw_countdown_timer )
		mainw_countdown_timer = g_timer_new();

	/* Is this the first mainw_countdown_animate() since the last
	 * mainw_countdown_cancel()? Reset the timer if it is.
	 */
	if( !mainw_countdown_computing ) {
		mainw_countdown_computing = TRUE;
		mainw_countdown_cancel_pushed = FALSE;
		g_timer_reset( mainw_countdown_timer );
		last_update = 0.0;
	}

	/* How long have we been calculating? 
	 */
	elapsed_time = g_timer_elapsed( mainw_countdown_timer, NULL );

	/* Should we have a countdown display?
	 */
	if( elapsed_time > start_threshold ) 
		mainw_countdown_show();

	/* Should we update?
	 */
	if( mainw_timer_box && GTK_WIDGET_VISIBLE( mainw_timer_box ) && 
		elapsed_time - last_update > update_threshold ) {
		mainw_countdown_update( percent );
		mainw_icon_animate( percent );
		last_update = elapsed_time;

		while( g_main_iteration( FALSE ) )
			;
	}

	/* If cancel has been pushed, tell our caller.
	 */
	if( mainw_countdown_cancel_pushed ) {
		im_errormsg( "ip: operation cancelled by user" );
		return( -1 );
	}

	return( 0 );
}

/* Called as an eval callback.
 */
static int
mainw_eval_cb( IMAGE *im )
{
	static int old_run;

	if( old_run != im->time->run )
		if( mainw_countdown_animate( im->time->percent ) )
			return( -1 );
	old_run = im->time->run;

	return( 0 );
}

/* Add an eval callback to an image to animate the cancel widgets.
 */
gboolean
mainw_add_eval_callback( IMAGE *im )
{
	/* Calls to tidy up.
	 */
	if( im_add_close_callback( im, 
		(im_callback_fn) mainw_countdown_end, NULL, NULL ) ) {
		verrors( "unable to attach close callbacks" );
		return( FALSE );
	}
	if( im_add_evalend_callback( im, 
		(im_callback_fn) mainw_countdown_end, NULL, NULL ) ) {
		verrors( "unable to attach evalend callbacks" );
		return( FALSE );
	}

	/* Call to animate countdown.
	 */
	if( im_add_eval_callback( im, 
		(im_callback_fn) mainw_eval_cb, im, NULL ) )  {
		verrors( "unable to attach eval callbacks" );
		return( FALSE );
	}

	/* Make extra-certain mainw_countdown_cancel_pushed is clear.
	 */
	mainw_countdown_cancel_pushed = FALSE;

	return( TRUE );
}

/* The last row we visited with the 'next-error' button.
 */
static Row *mainw_row_last_error = NULL;

static Row *
mainw_test_error( Row *row, int *found )
{
	assert( row->err );

	/* Found next?
	 */
	if( *found )
		return( row );

	if( row == mainw_row_last_error ) {
		/* Found the last one ... return the next one.
		 */
		*found = 1;
		return( NULL );
	}

	return( NULL );
}

/* Callback for next-error button.
 */
static void
mainw_next_error_cb( GtkWidget *widget, gpointer dummy )
{
	Workspace *ws = main_workspacegroup->current;

	BufInfo buf;
	char str[MAX_LINELENGTH];

	int found;

	if( !ws->errors ) {
		box_info( mainw_window, "Workspace contains no errors" );
		return;
	}

	/* Search for the one after the last one.
	 */
	found = 0;
	mainw_row_last_error = (Row *) slist_map( ws->errors, 
		(SListMapFn) mainw_test_error, &found );

	/* NULL? We've hit end of table, start again.
	 */
	if( !mainw_row_last_error ) {
		found = 1;
		mainw_row_last_error = (Row *) slist_map( ws->errors, 
			(SListMapFn) mainw_test_error, &found );
	}

	/* *must* have one now.
	 */
	assert( mainw_row_last_error && mainw_row_last_error->err );

	model_scrollto( MODEL( mainw_row_last_error ) );

	buf_init_static( &buf, str, MAX_LINELENGTH );
	row_qualified_name( mainw_row_last_error->expr->row, &buf );
	buf_appends( &buf, ": " );
	buf_appendline( &buf, mainw_row_last_error->expr->errstr );
	set_glabel( mainw_message, "%s", buf_all( &buf ) );
}

/* Event in the "space free" display ... toggle mode on left click.
 */
static gint
mainw_space_free_event( GtkWidget *widget, GdkEvent *ev, gpointer dummy )
{
	if( ev->type == GDK_BUTTON_RELEASE ) {
		mainw_free_type = !mainw_free_type;
		mainw_free_update();
	}

	return( FALSE );
}

/* Refresh the colour of the next-error button.
 */
void
mainw_next_error_refresh( void )
{
	Workspace *ws = main_workspacegroup->current;

        if( !mainw_next_error || !ws )
                return;

        if( !ws->errors ) 
		set_name( mainw_next_error, "default_widget" );
        else 
		set_name( mainw_next_error, "error_widget" );
}

/* Go to home page.
 */
static void
mainw_home_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	box_url( mainw_window, VIPS_HOMEPAGE );
}

/* About... box.
 */
void
mainw_about_cb( gpointer callback_data, guint callback_action,
        GtkWidget *par )
{
	box_about( mainw_window );
}

/* Menu bar.
 */
static GtkItemFactoryEntry mainw_menu_items[] = {
	{ "/_File",				NULL,		
		NULL,			0, "<Branch>" },

	{ "/File/_New workspace ...", 		"<control>N", 
		mainw_workspace_new, 	0, "" },
	{ "/File/_Open workspace ...", 		"<control>O", 
		mainw_workspace_open, 	0, "" },

	{ "/File/sep3",        			NULL,         	
		NULL,    		0, "<Separator>" },

	{ "/File/_Close workspace", 			"", 
		mainw_workspace_close,	0, "" },
	{ "/File/_Save workspace",		"<control>S", 	
		mainw_workspace_save,	0, "" },
	{ "/File/_Save workspace as ...", 	NULL, 	
		mainw_workspace_saveas, 0, "" },

	{ "/File/sep1",        			NULL,         	
		NULL,    		0, "<Separator>" },

	{ "/File/_Program ...", 		NULL, 	
		mainw_program_cb,   	0, "" },

	{ "/File/sep1",        			NULL,         	
		NULL,    		0, "<Separator>" },

	{ "/File/Re_cover after crash ...", 	NULL, 
		mainw_recover_cb, 	0, "" },

	{ "/File/sep2",        			NULL,         	
		NULL,    		0, "<Separator>" },

	{ "/File/_Quit",       			"<control>Q", 	
		mainw_exit_cb, 		0, "" },

	{ "/_Edit",       			NULL,		
		NULL,			0, "<Branch>" },

	{ "/Edit/Select all",  			"<control>A", 	
		mainw_select_all_cb, 	0, "" },
	{ "/Edit/sep3",        			NULL,         	
		NULL,		    	0, "<Separator>" },
	{ "/Edit/_Find",       			"<control>F", 
		mainw_find_cb,     	0, "" },
	{ "/Edit/Find _again",       		"F3", 
		mainw_find_again_cb, 	0, "" },
	{ "/Edit/sep2",        			NULL,         	
		NULL,		    	0, "<Separator>" },
	{ "/Edit/_Group",      			"", 	
		mainw_group_cb, 	0, "" },
	{ "/Edit/U_ngroup",    			"", 	
		mainw_ungroup_cb, 	0, "" },
	{ "/Edit/sep4",        			NULL,         	
		NULL,    		0, "<Separator>" },
	{ "/Edit/_Recalculate", 		"", 	
		mainw_force_calc_cb, 	0, "" },
	{ "/Edit/C_lone",      			"<control>L", 	
		mainw_clone_cb,		0, "" },
	{ "/Edit/Re_move selected ...", 	"<control>R", 	
		mainw_selected_remove_cb,  	0, "" },
	{ "/Edit/sep5",        			NULL,         	
		NULL,    		0, "<Separator>" },
	{ "/Edit/Show formula", 		"", 	
		mainw_show_formula_cb,	0, "" },

	{ "/_Insert",       			NULL,		
		NULL,       		0, "<Branch>" },
	{ "/Insert/New c_olumn ...", 		NULL, 	
		mainw_column_new_cb, 	0, "" },
	{ "/Insert/sep6",      			NULL,         	
		NULL,    		0, "<Separator>" },
	{ "/Insert/Ima_ge from file ...", 	"<control>G", 
		mainw_image_insert_file,0, "" },
	{ "/Insert/Workspace from file ...", 	"", 
		mainw_workspace_merge, 	0, "" },
	{ "/Insert/Matrix from file ...", 	NULL, 	
		mainw_matrix_insert_file,0, "" },

	{ "/_Help",            			NULL,         
		NULL,           	0, "<LastBranch>" },
	{ "/Help/_About",      			NULL,         
		mainw_about_cb,    	0, "" },

	{ "/Help/sep7",        			NULL,         	
		NULL,    		0, "<Separator>" },

	{ "/Help/_This window",      		NULL,         
		box_help_cb,   		GPOINTER_TO_UINT( "sec:ipwindow" ) },
	{ "/Help/_Users guide",      	NULL,         
		box_url_cb,    		GPOINTER_TO_UINT( "index.html" ) },
	{ "/Help/_Quick tour",  		NULL,         
		box_help_cb,		GPOINTER_TO_UINT( "sec:quicktour" ) },
	{ "/Help/_Mosaic tour",  		NULL,         
		box_help_cb,		GPOINTER_TO_UINT( "sec:irtut" ) },
	{ "/Help/_Nerd tour",  			NULL,         
		box_help_cb,		GPOINTER_TO_UINT( "sec:nerdtour" ) },
	{ "/Help/M_enu reference",		NULL,         
		box_help_cb,		GPOINTER_TO_UINT( "sec:menus" ) },
	{ "/Help/_Configuration",		NULL,         
		box_help_cb,		GPOINTER_TO_UINT( "sec:config" ) },

	{ "/Help/sep6",        			NULL,         	
		NULL,    		0, "<Separator>" },

	{ "/Help/_Go to VIPS home page",  	NULL,         
		mainw_home_cb,    	0, "" },
};

/* Make the insides of the panel.
 */
static void
mainw_build_display( iWindow *iwnd, GtkWidget *vbox )
{
        GtkWidget *mbar;
        GtkAccelGroup *accel_group;
        GtkItemFactory *item_factory;
	GtkWidget *hbox;
	GtkWidget *but;

        /* Make main menu bar
         */
        accel_group = gtk_accel_group_new();
        item_factory = gtk_item_factory_new( GTK_TYPE_MENU_BAR,
                "<main>", accel_group );
        gtk_item_factory_create_items( item_factory,
                IM_NUMBER( mainw_menu_items ), mainw_menu_items, mainw_window );
        gtk_accel_group_attach( accel_group, GTK_OBJECT( iwnd ) );
        mbar = gtk_item_factory_get_widget( item_factory, "<main>" );
        gtk_widget_show( mbar );
        gtk_box_pack_start( GTK_BOX( vbox ), mbar, FALSE, FALSE, 0 );

	/* hbox for status bar etc.
	 */
        hbox = gtk_hbox_new( FALSE, 0 );
	gtk_container_border_width( GTK_CONTAINER( hbox ), 2 );
        gtk_box_pack_end( GTK_BOX( vbox ), hbox, FALSE, FALSE, 0 );
        gtk_widget_show( hbox );

	/* Next-error button.
	 */
	but = gtk_button_new();
	gtk_button_set_relief( GTK_BUTTON( but ), GTK_RELIEF_NONE );
        gtk_box_pack_end( GTK_BOX( hbox ), but, FALSE, FALSE, 0 );
        gtk_widget_show( but );
	set_tooltip( but, "Cycle through workspace errors" );
	mainw_next_error = gtk_arrow_new( GTK_ARROW_RIGHT, GTK_SHADOW_OUT );
	gtk_container_add( GTK_CONTAINER( but ), mainw_next_error );
        gtk_widget_show( mainw_next_error );
        gtk_signal_connect( GTK_OBJECT( but ), "clicked",
                GTK_SIGNAL_FUNC( mainw_next_error_cb ), NULL );
	mainw_next_error_refresh();

	/* Make space free label.
	 */
	mainw_space_free_eb = gtk_event_box_new();
        gtk_box_pack_end( GTK_BOX( hbox ), 
		mainw_space_free_eb, FALSE, FALSE, 2 );
	gtk_widget_show( mainw_space_free_eb );
	mainw_space_free = gtk_label_new( "space_free" );
        gtk_container_add( GTK_CONTAINER( mainw_space_free_eb ), 
		mainw_space_free );
        gtk_signal_connect( GTK_OBJECT( mainw_space_free_eb ), "event",
                GTK_SIGNAL_FUNC( mainw_space_free_event ), NULL );
        gtk_widget_show( mainw_space_free );

	/* Make message label.
	 */
	mainw_message = gtk_label_new( IP_COPYRIGHT );
	gtk_misc_set_alignment( GTK_MISC( mainw_message ), 0, 0.5 );
	gtk_misc_set_padding( GTK_MISC( mainw_message ), 3, 3 );
        gtk_box_pack_start( GTK_BOX( hbox ), mainw_message, FALSE, FALSE, 0 );
        gtk_widget_show( mainw_message );

	/* Make countdown/cancel stuff, but don't show it.
	 */
        mainw_timer_box = gtk_hbox_new( FALSE, 0 );
        gtk_box_pack_start( GTK_BOX( hbox ), mainw_timer_box, TRUE, TRUE, 2 );

	/* Make insides of countdown timer: scrollbar, label and button.
	 */
	but = gtk_button_new_with_label( "Cancel" );
        gtk_box_pack_start( GTK_BOX( mainw_timer_box ), but, FALSE, FALSE, 2 );
        gtk_widget_show( but );
	set_tooltip( but, "Click here to cancel computation" );
	gtk_signal_connect( GTK_OBJECT( but ), "clicked", 
		GTK_SIGNAL_FUNC( mainw_countdown_cancel ), NULL );

	mainw_togo = gtk_progress_bar_new();
        gtk_box_pack_start( GTK_BOX( mainw_timer_box ), 
		mainw_togo, TRUE, TRUE, 2 );
        gtk_widget_show( mainw_togo );
	set_tooltip( mainw_togo, "Computation progress indicator" );

	/* Make toolkit/workspace displays.
	 */
	mainw_kitgview = TOOLKITGROUPVIEW( 
		model_view_new( MODEL( main_toolkitgroup ) ) );
	view_link( VIEW( mainw_kitgview ), MODEL( main_toolkitgroup ), NULL );
	mainw_wsgview = WORKSPACEGROUPVIEW( 
		model_view_new( MODEL( main_workspacegroup ) ) );
	view_link( VIEW( mainw_wsgview ), MODEL( main_workspacegroup ), NULL );

        hbox = gtk_hbox_new( FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( vbox ), hbox, TRUE, TRUE, 0 );
	gtk_widget_show( hbox );

	gtk_box_pack_end( GTK_BOX( hbox ), 
		GTK_WIDGET( mainw_kitgview ), FALSE, FALSE, 0 );
	gtk_widget_show( GTK_WIDGET( mainw_kitgview ) );
	gtk_box_pack_start( GTK_BOX( hbox ), 
		GTK_WIDGET( mainw_wsgview ), TRUE, TRUE, 0 );
	gtk_widget_show( GTK_WIDGET( mainw_wsgview ) );

	/* Fill free-space indicator.
	 */
	(void) mainw_free_update();
}

/* Make the main ip window. Can only have one of these (for now).
 */
void
mainw_new( const char *name )
{
	assert( !mainw_window );

        mainw_window = iwindow_new( GTK_WINDOW_TOPLEVEL );

	gtk_window_set_default_size( GTK_WINDOW( mainw_window ), 
		MAINW_WINDOW_WIDTH, MAINW_WINDOW_HEIGHT );
        iwindow_set_build( IWINDOW( mainw_window ), 
		(iWindowBuildFn) mainw_build_display, NULL, NULL, NULL );
        iwindow_set_popdown( IWINDOW( mainw_window ), 
		(iWindowFn) mainw_popdown_cb, NULL );
	iwindow_build( IWINDOW( mainw_window ) ); 

	gtk_widget_show( GTK_WIDGET( mainw_window ) ); 

	mainw_icon_restore();
}

void
mainw_refresh_title( void )
{
	Workspace *ws = main_workspacegroup->current;
	BufInfo buf;
	char txt[512];

	buf_init_static( &buf, txt, 512 );
	buf_appendf( &buf, IP_NAME " - %s", NN( MODEL( ws )->name ) );
	if( FILEMODEL( ws )->filename )
		buf_appendf( &buf, " - %s", FILEMODEL( ws )->filename );
	if( FILEMODEL( ws )->modified )
		buf_appendf( &buf, " [modified]" ); 

	iwindow_set_title( IWINDOW( mainw_window ), buf_all( &buf ) );
}
