/* abstract base class for things which form the model of a model/view pair 
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/* How to rename symbols.
 */
typedef struct _ModelRename {
	char *old_name;
	char *new_name;
} ModelRename;

/* What we track during a load operation.
 */
typedef struct _ModelLoadState {
	char *filename;		/* Name we loaded from */
	xmlDoc *xdoc;		/* Document we load from */

	/* 

		FIXME ... a linked list? try a hash sometime
		see model_loadstate_rewrite_name()

		would probably only see a speedup for merging large
		workspaces, not something we do often

	 */
	GSList *renames;	/* Rename table for this load context */

	/* Log error messages here.
	 */
	char error_log_buffer[MAX_STRSIZE];
	BufInfo error_log;
} ModelLoadState;

#define TYPE_MODEL (model_get_type())
#define MODEL( obj ) (GTK_CHECK_CAST( (obj), TYPE_MODEL, Model ))
#define MODEL_CLASS( klass ) \
	(GTK_CHECK_CLASS_CAST( (klass), TYPE_MODEL, ModelClass ))
#define IS_MODEL( obj ) (GTK_CHECK_TYPE( (obj), TYPE_MODEL ))
#define IS_MODEL_CLASS( klass ) \
	(GTK_CHECK_CLASS_TYPE( (klass), TYPE_MODEL ))

/* Test for is C a child of M.
 */
#define MODEL_IS_CHILD( M, C ) \
	(g_slist_find( MODEL( M )->children, MODEL( C ) ) && \
	MODEL( C )->parent == MODEL( M ))

struct _Model {
	GtkObject object;

	/* My instance vars.
	 */
	GSList *children;	/* Models which are sub-this one */
	Model *parent;		/* Model we are inside */
	int pos;		/* Position in parent */

        char *name;             /* Model name */
        char *caption;          /* Comment of some sort */

	gboolean display;	/* This model should have a view */
};

typedef struct _ModelClass {
	GtkObjectClass parent_class;

	/* Build display methods.

		view_new	make a view for this model ... make the top
				view yourself, thereafter view will watch
				model_child_add etc. and manage subviews
				automatically ... use model->display to create
				and destroy views

	 */

	View *(*view_new)( void );

	/* Change methods

		changed 	model has changed signal ... views listen to
				this to decide when repaint

		edit		open an editor on the model

		scrollto	try to make views visible

		reset		signals views to reset ... eg. textview pops
				back to whatever the column says it should be
				displaying (value or formula)

		info		try to say some useful stuff 

	 */

	void (*changed)( Model * );
	void (*edit)( GtkWidget *, Model * );
	void (*scrollto)( Model * );
	void (*reset)( Model * );
	void (*info)( Model *, BufInfo * );

	/* Create/destroy methods.

		child_add	a child has been added to us

		child_remove	a child is about to be removed from us

		parent_add	parent has been attached

		parent_remove	parent is about to be removed

	 */
	 
	void (*child_add)( Model *, Model *, int );
	void (*child_remove)( Model *, Model * );
	void (*parent_add)( Model *, Model * );
	void (*parent_remove)( Model *, Model * );

	/* Load and save methods.

		save		write model as child of node

		save_test	predicate ... save model if save_test is 
				defined and true

		save_text	plain text save ... eg. for toolkits

		load		_init() model from xmlNode

		load_text	_init() from plain text ... eg. toolkit

		empty		remove contents of model

	 */
	xmlNode *(*save)( Model *, xmlNode * );
	gboolean (*save_test)( Model * );
	gboolean (*save_text)( Model *, iOpenFile * );
	gboolean (*load)( Model *model, 
		ModelLoadState *state, Model *parent, xmlNode *xnode );
	gboolean (*load_text)( Model *model, Model *parent, iOpenFile * );
	void (*empty)( Model * );
} ModelClass;

extern Model *model_base;

extern ModelLoadState *model_loadstate;

ModelRename *model_loadstate_rename_new( ModelLoadState *state, 
	const char *old_name, const char *new_name );
ModelLoadState *model_loadstate_new( const char *filename );
void model_loadstate_destroy( ModelLoadState *state );
char *model_loadstate_rewrite_name( char *name );
void model_loadstate_rewrite( ModelLoadState *state, 
	char *old_rhs, char *new_rhs );

void model_register_loadable( ModelClass *model_class );

View *model_view_new( Model *model );
void *model_changed( Model *model );
void model_scrollto( Model *model );
void model_reset( Model *model );
void *model_edit( GtkWidget *parent, Model *model );
void *model_info( Model *model, BufInfo *info );

void *model_save( Model *model, xmlNode * );
gboolean model_save_test( Model *model );
void *model_save_text( Model *model, iOpenFile * );
void *model_load( Model *model,
	ModelLoadState *state, Model *parent, xmlNode *xnode );
void *model_load_text( Model *model, Model *parent, iOpenFile * );
void *model_empty( Model *model );

void *model_test_name( Model *model, const char *name );

int model_get_n_children( Model *model );
void *model_map( Model *model, model_map_fn fn, void *a, void *b );
void *model_map3( Model *model, model_map3_fn fn, void *a, void *b, void *c );
void *model_map4( Model *model, 
	model_map4_fn fn, void *a, void *b, void *c, void *d );
void *model_map5( Model *model, 
	model_map5_fn fn, void *a, void *b, void *c, void *d, void *e ); 
void *model_map_rev( Model *model, model_map_fn fn, void *a, void *b );
void *model_map_all( Model *model, model_map_fn fn, void *a );
void *model_map2_all( Model *model, model_map_fn fn, void *a, void *b );
void *model_map3_all( Model *model, 
	model_map3_fn fn, void *a, void *b, void *c );
void *model_map4_all( Model *model, 
	model_map4_fn fn, void *a, void *b, void *c, void *d );
void *model_map_all_intrans( Model *model, model_map_fn fn, void *a );

void model_pos_sort( Model *model );
int model_pos_last( Model *model );
void model_pos_renumber( Model *model );
void model_name_sort( Model *model );

void model_child_add( Model *model, Model *child, int pos );
void model_child_add_before( Model *parent, Model *child, Model *before );
void model_child_move( Model *child, int pos );
void model_child_remove( Model *model, Model *child );
void model_destroy_children( Model *parent );

void model_set( Model *model, const char *name, const char *caption );

gboolean model_new_xml( ModelLoadState *state, Model *parent, xmlNode *xnode );

GtkType model_get_type( void );

void model_base_init( void );

View *model_build_display_all( Model *model, View *parent );

void model_check_destroy( GtkWidget *parent, Model *model );

gboolean model_set_display( Model *model, gboolean display );

void *model_clear_edited( Model *model );
