/* Graph reducer.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#include "ip.h"

/* 
#define DEBUG
 */

/* trace each regeneration
#define DEBUG_REGEN
 */

/* trace just member regeneration
#define DEBUG_REGEN_MEMBER
 */

/* Turn on WHNF tests.
#define WHNF_DEBUG
 */

/* If debug is on, make sure we enable WHNF testing.
 */
#ifdef DEBUG
#ifndef WHNF_DEBUG
#define WHNF_DEBUG
#endif /*WHNF_DEBUG*/
#endif /*DEBUG*/

#define MAX_HEAPSIZE (watch_int_get( "CALC_MAX_HEAP", 40000 ))

/* State of the reduction engine.
 */
Reduce *reduce_context;

/* Index with a CombinatorType, get the number of args that combinator takes.
        COMB_S = 0,		
        COMB_SL,	
        COMB_SR,
	COMB_I,
	COMB_K,
	COMB_GEN,
 */
static int nargs[] = {3, 3, 3, 1, 2, 3};

/* Recomps this time.
 */
int reduce_total_recomputations = 0;

/* Eval error here.
 */
void
everror( Reduce *rc, const char *fmt, ... )
{
	va_list ap;
 	char buf[4096];

        va_start( ap, fmt );
        (void) im_vsnprintf( buf, 4096, fmt, ap );
        va_end( ap );

#ifdef DEBUG
	printf( "reduce.everror: seen \"%s\"\n", buf );
#endif /*DEBUG*/

	ierrors( buf );
	if( !rc->running ) 
		printf( "panic: uncaught exception in everror! "
			"expect a crash soon\n" 
			"msg is: \"%s\"\n", buf );
	else
		longjmp( rc->error[--rc->running], -1 );
}

/* As above, but append a VIPS error message to the error string.
 */
void
veverror( Reduce *rc, const char *fmt, ... )
{	
	va_list ap;
	char buf[4096];

        va_start( ap, fmt );
        (void) im_vsnprintf( buf, 4096, fmt, ap );
        va_end( ap );

#ifdef DEBUG
	printf( "reduce.veverror: seen \"%s\"\n", buf );
#endif /*DEBUG*/

	verrors( buf );
	assert( rc->running );
	longjmp( rc->error[--rc->running], -1 );
}

/* Call a function, passing in a "safe" PElement ... ie. the PElement points
 * at a fresh element which will be safe from the GC.
 */
void *
reduce_safe_pointer( Reduce *rc, reduce_safe_pointer_fn fn, 
	void *a, void *b, void *c, void *d )
{
	Element e;
	PElement pe;
	void *result;

	e.type = ELEMENT_NOVAL;
	e.ele = NULL;
	PEPOINTE( &pe, &e );
	heap_register_element( rc->hi, &e );

	result = fn( rc, &pe, a, b, c, d );

	heap_unregister_element( rc->hi, &e );

	return( result );
}

/* Map over a heap list. Reduce the list spine as we go, don't reduce the
 * heads. Return base on error, or whatever the user function returns.
 */
void *
reduce_map_list( Reduce *rc, 
	PElement *base, reduce_map_list_fn fn, void *a, void *b )
{
	PElement e = *base;

	reduce_spine( rc, &e );

	if( !PEISLIST( &e ) ) 
		everror( rc, "not [*]" );

	while( !PEISELIST( &e ) ) {
		PElement head;
		void *res;

		/* Apply user function to the head.
		 */
		PEGETHD( &head, &e );
		if( (res = fn( rc, &head, a, b )) )
			return( res );

		/* Reduce the tail.
		 */
		PEGETTL( &e, &e );
		reduce_spine( rc, &e );
	}

	return( NULL );
}

static void *
reduce_clone_list_sub( Reduce *rc, PElement *data, PElement *out )
{
	PElement lhs;

	if( !heap_list_add( rc->hi, out, &lhs ) )
		everror( rc, "%s", error_string );
	PEPUTPE( &lhs, data );

	heap_list_next( out );

	return( NULL );
}

/* Clone a list ... just clone the spine, copy pointers to the heads. Reduce
 * the list as we go (strict shallow clone). We update out as we go, so that
 * on return it points to the tail (always []) of the cloned list.
 */
void
reduce_clone_list( Reduce *rc, PElement *base, PElement *out )
{
	heap_list_init( out );

	(void) reduce_map_list( rc, base, 
		(reduce_map_list_fn) reduce_clone_list_sub, out, NULL );
}

/* Sub-fn of below. Add a character to the buffer.
 */
static void *
reduce_add_char( Reduce *rc, PElement *base, char **buf, int *sz )
{
	/* Overflow?
	 */
	if( *sz == 0 ) 
		everror( rc, "[char] too long" );

	/* Reduce this list element.
	 */
	reduce_spine( rc, base );

	/* Should be a char.
	 */
	if( !PEISCHAR( base ) ) 
		everror( rc, "not [char]" );

	/* Add to buffer.
	 */
	**buf = PEGETCHAR( base );
	(*buf)++;
	(*sz)--;

	return( NULL );
}

/* Evaluate a PElement into a string buffer. Return the number of characters 
 * in string, not including '\0' terminator.
 */
int
reduce_get_string( Reduce *rc, PElement *base, char *buf, int n )
{
	int sz = n - 1;

	(void) reduce_map_list( rc, base, 
		(reduce_map_list_fn) reduce_add_char, &buf, &sz );

	/* Add '\0' terminator.
	 */
	*buf = '\0';

	return( n - sz - 1 );
}

static void *
reduce_get_lstring_sub( Reduce *rc, PElement *base, GSList **labels, int *n )
{
	char buf[MAX_STRSIZE];

	(void) reduce_get_string( rc, base, buf, MAX_STRSIZE );
	*labels = g_slist_append( *labels, g_strdup( buf ) );

	return( NULL );
}

/* Evaluate to [[char]]. Return the number of strings we read. Append to
 * anything that's in the list already.
 */
int
reduce_get_lstring( Reduce *rc, PElement *base, GSList **labels )
{
	int n;

	n = 0;
	(void) reduce_map_list( rc, base,
		(reduce_map_list_fn) reduce_get_lstring_sub, labels, &n );

	return( n );
}

/* Get an element as a boolean. 
 */
gboolean
reduce_get_bool( Reduce *rc, PElement *base )
{
	reduce_spine( rc, base );

	if( !PEISBOOL( base ) ) 
		everror( rc, "not bool" );

	return( PEGETBOOL( base ) );
}

/* Get an element as a real. 
 */
double
reduce_get_real( Reduce *rc, PElement *base )
{
	/* Reduce this element.
	 */
	reduce_spine( rc, base );

	/* Should be a real.
	 */
	if( !PEISREAL( base ) ) 
		everror( rc, "not real" );

	return( PEGETREAL( base ) );
}

/* Get an element as a class. 
 */
void
reduce_get_class( Reduce *rc, PElement *base )
{
	/* Reduce this element.
	 */
	reduce_spine( rc, base );

	/* Should be a class.
	 */
	if( !PEISCLASS( base ) ) 
		everror( rc, "not class" );
}

/* Get an element as an image. 
 */
Imageinfo *
reduce_get_image( Reduce *rc, PElement *base )
{
	/* Reduce this element.
	 */
	reduce_spine( rc, base );

	/* Should be an image.
	 */
	if( !PEISIMAGE( base ) ) 
		everror( rc, "not image" );

	/* Add to buffer.
	 */
	return( PEGETII( base ) );
}

/* Sub-fn of below. Add a real to the buffer.
 */
static void *
reduce_add_real( Reduce *rc, PElement *base, double **buf, int *sz )
{
	/* Overflow?
	 */
	if( *sz == 0 ) 
		everror( rc, "[real] too long" );

	/* Add to buffer.
	 */
	**buf = reduce_get_real( rc, base );
	(*buf)++;
	(*sz)--;

	return( NULL );
}

/* Get an element as a realvec. Return length of vector.
 */
int
reduce_get_realvec( Reduce *rc, PElement *base, double *buf, int n )
{
	int sz = n;

	(void) reduce_map_list( rc, base, 
		(reduce_map_list_fn) reduce_add_real, &buf, &sz );

	return( n - sz );
}

/* Test for 1st sz elements are reals. Init sz < 0 for unlimited test.
 */
static void *
reduce_test_real( Reduce *rc, PElement *base, int *sz )
{
	/* Tested enough?
	 */
	if( *sz == 0 ) 
		return( NULL );

	(void) reduce_get_real( rc, base );
	(*sz)--;

	return( NULL );
}

/* Sub fn ... get the length of a list of real.
 */
int
reduce_get_real_size( Reduce *rc, PElement *base )
{
	int n;

	n = -1;
	(void) reduce_map_list( rc, base, 
		(reduce_map_list_fn) reduce_test_real, &n, NULL );

	return( -1 - n );
}

/* Sub fn of below ... get the length of one line from a matrix.
 */
static void *
reduce_get_line_size( Reduce *rc, PElement *base, int *w, int *h )
{
	int l;

	l = reduce_get_real_size( rc, base );

	if( *w == 0 )
		*w = l;
	else if( *w != l )
		everror( rc, "matrix is not rectangular" );
	*h += 1;

	return( NULL );
}

/* Find the size of a matrix. Write xsize/ysize to args.
 */
void
reduce_get_matrix_size( Reduce *rc, 
	PElement *base, int *xsize, int *ysize )
{
	int w, h;

	w = 0; 
	h = 0;
	(void) reduce_map_list( rc, base, 
		(reduce_map_list_fn) reduce_get_line_size, &w, &h );

	*xsize = w;
	*ysize = h;
}

/* Track stuff during a get_matrix in one of these.
 */
typedef struct {
	double *buf;		/* Start of output buffer */
	int mx;			/* Size of output buffer */
	int w, h;		/* Size of matrix we have generated */
	int i;			/* Current write point */
} GetMatrixInfo;

/* Sub-fn of below ... get another line of the matrix.
 */
static void *
reduce_get_line( Reduce *rc, PElement *base, GetMatrixInfo *gmi )
{
	int l;
	int remain = gmi->mx - gmi->i;

	/* Read next line from matrix.
	 */
	l = reduce_get_realvec( rc, base, gmi->buf + gmi->i, remain );

	/* Overflow?
	 */
	if( l > remain ) 
		everror( rc, "matrix too large" );

	/* 1st line?
	 */
	if( gmi->h == 0 )
		gmi->w = l;
	else if( l != gmi->w ) 
		everror( rc, "matrix is not rectangular" );

	/* Move pointers on!
	 */
	gmi->h++;
	gmi->i += l;

	return( NULL );
}

/* Get an element as a matrix. Return length of buffer used. 
 * Write xsize/ysize to args.
 */
int
reduce_get_matrix( Reduce *rc, 
	PElement *base, double *buf, int n, int *xsize, int *ysize )
{
	GetMatrixInfo gmi;

	gmi.buf = buf;
	gmi.mx = n;
	gmi.w = gmi.h = 0;
	gmi.i = 0;

	(void) reduce_map_list( rc, base, 
		(reduce_map_list_fn) reduce_get_line, &gmi, NULL );

	*xsize = gmi.w;
	*ysize = gmi.h;

	return( gmi.i );
}

/* Sub-fn of below. Test for 1st sz elements are char.
 */
static void *
reduce_test_char( Reduce *rc, PElement *base, int *sz )
{
	/* Tested enough?
	 */
	if( *sz == 0 ) 
		return( NULL );

	/* Reduce this list element.
	 */
	reduce_spine( rc, base );

	/* Should be a char.
	 */
	if( !PEISCHAR( base ) ) 
		return( base );

	/* Move on.
	 */
	(*sz)--;

	return( NULL );
}

/* Test for object is the empty list.
 */
gboolean
reduce_iselist( Reduce *rc, PElement *base )
{
	reduce_spine( rc, base );
	if( PEISELIST( base ) ) 
		return( TRUE );

	return( FALSE );
}

/* Test the first n elements of a list are char. n < 0 means test all
 * elements.
 */
static gboolean
reduce_n_isstring( Reduce *rc, PElement *base, int sz )
{
	reduce_spine( rc, base );
	if( !PEISLIST( base ) ) 
		return( FALSE );

	if( reduce_map_list( rc, base, 
		(reduce_map_list_fn) reduce_test_char, &sz, NULL ) )
		return( FALSE );

	return( TRUE );
}

/* Test for object is string. Just test the first few elements, so we
 * allow infinite strings.
 */
gboolean
reduce_isstring( Reduce *rc, PElement *base )
{
	return( reduce_n_isstring( rc, base, 4 ) );
}

/* Test for list is a finite string. 
 */
gboolean
reduce_isfinitestring( Reduce *rc, PElement *base )
{
	return( reduce_n_isstring( rc, base, -1 ) );
}

/* Test for list is realvec.
 */
gboolean
reduce_isrealvec( Reduce *rc, PElement *base )
{
	int sz = 4;

	reduce_spine( rc, base );
	if( !PEISLIST( base ) ) 
		return( FALSE );

	if( reduce_map_list( rc, base, 
		(reduce_map_list_fn) reduce_test_real, &sz, NULL ) )
		return( FALSE );

	return( TRUE );
}

/* Sub-fn of below ... test another line of the matrix.
 */
static void *
reduce_test_line( Reduce *rc, PElement *base, int *w, int *h )
{
	/* Test next line from matrix.
	 */
	if( !reduce_isrealvec( rc, base ) )
		return( base );

	return( NULL );
}

/* Test for object is [[real]] .. don't test for rectangularness.
 */
gboolean
reduce_ismatrix( Reduce *rc, PElement *base )
{
	reduce_spine( rc, base );
	if( !PEISLIST( base ) ) 
		return( FALSE );

	if( reduce_map_list( rc, base, 
		(reduce_map_list_fn) reduce_test_line, NULL, NULL ) )
		return( FALSE );

	return( TRUE );
}

/* Test for object is a class.
 */
gboolean
reduce_isclass( Reduce *rc, PElement *klass )
{
	reduce_spine( rc, klass );
	if( PEISCLASS( klass ) )
		return( TRUE );

	return( FALSE );
}

/* Test for instance is an exact instance ... ie. no inheritance.

	FIXME ... yuk! strcmp()!!

 */
gboolean
reduce_isinstanceof_exact( Reduce *rc, 
	const char *name, PElement *klass )
{
	BufInfo buf;
	char txt[200];

	if( !reduce_isclass( rc, klass ) )
		return( FALSE );

	buf_init_static( &buf, txt, 200 );
	symbol_qualified_name( PEGETCLASSCOMPILE( klass )->sym, &buf );
	if( strcmp( name, buf_all( &buf ) ) == 0 )
		return( TRUE );

	return( FALSE );
}

/* Test for instance is an instance of the named class symbol.
 */
gboolean
reduce_isinstanceof( Reduce *rc, const char *name, PElement *klass )
{
	PElement super;

	reduce_spine( rc, klass );
	if( !PEISCLASS( klass ) )
		return( FALSE );
	if( reduce_isinstanceof_exact( rc, name, klass ) )
		return( TRUE );
	if( class_get_super( klass, &super ) && !PEISELIST( &super ) ) 
		return( reduce_isinstanceof( rc, name, &super ) );

	return( FALSE );
}

/* Find the length of a list.
 */
int
reduce_list_length( Reduce *rc, PElement *base )
{
	PElement p;
	int i;

	/* Reduce to first element.
	 */
	p = *base;
	reduce_spine( rc, &p );

	/* Does it look like the start of a list? 
	 */
	if( !PEISLIST( &p ) ) 
		everror( rc, "argument to list length is not list" );

	/* Loop down list.
	 */
	for( i = 0; PEISFLIST( &p ); i++ ) {
		HeapNode *hn;

		hn = PEGETVAL( &p );
		PEPOINTRIGHT( hn, &p );

		reduce_spine( rc, &p );
	}

	assert( PEISELIST( &p ) );

	return( i );
}

/* Point "out" at the nth element of a list. Index from 0.
 */
void
reduce_list_index( Reduce *rc, PElement *base, int n, PElement *out )
{
	PElement p;
	int i;
	HeapNode *hn;

	/* n OK?
	 */
	if( n < 0 ) 
		everror( rc, "list index should be >= 0" );

	/* Reduce to first element.
	 */
	p = *base;
	reduce_spine( rc, &p );

	/* Does it look like the start of a list? 
	 */
	if( !PEISLIST( &p ) ) 
		everror( rc, "argument to list subscript is not list" );

	/* Loop down list.
	 */
	for( i = n;; ) {
		/* At end? 
		 */ 
		if( PEISELIST( &p ) ) 
			everror( rc, "list too short for subscript %d", n );

		assert( PEISFLIST( &p ) );

		hn = PEGETVAL( &p );
		PEPOINTRIGHT( hn, &p );

		if( --i < 0 )
			break;

		reduce_spine( rc, &p );
	}

	if( trace_flags & TRACE_OPERATOR ) {
		BufInfo *buf = trace_push();

		trace_pelement( base );
		buf_appendf( buf, " \"?\" %d ->\n", n );
	}

	PEPOINTLEFT( hn, out );

	if( trace_flags & TRACE_OPERATOR ) {
		trace_result( TRACE_OPERATOR, out );
		trace_pop();
	}
}

/* No args allowed error.
 */
static void
argserror( Reduce *rc,  PElement *a )
{
	BufInfo buf;
	char txt[MAX_ERROR_FRAG];

	buf_init_static( &buf, txt, MAX_ERROR_FRAG );
	itext_decompile_ev( rc, &buf, a );

	everror( rc, "object \"%s\" should have no arguments",
		buf_all( &buf ) );
}

#ifdef WHNF_DEBUG
/* Test for PElement is in weak head-normal form.
 */
static gboolean
is_WHNF( PElement *out )
{
	PElement spine;
	int i;
	HeapNode *hn;
	Symbol *sym;
	Compile *compile;
	int na;

	/* Might be a base type ...
	 */
	if( PEISREAL( out ) || 
		PEISCOMPLEX( out ) || PEISNUM( out ) || PEISCHAR( out ) ||
		PEISBOOL( out ) || PEISTAG( out ) || PEISIMAGE( out ) ||
		PEISLIST( out ) || PEISCLASS( out ) || PEISSYMREF( out ) ||
		PEISCOMPILEREF( out ) || PEISNOVAL( out ) )
		return( TRUE );

	/* Must be a function ... loop down the spine, counting args.
	 */
	for( spine = *out, i = 0; PEGETTYPE( &spine ) == ELEMENT_NODE; i++ ) {
		hn = PEGETVAL( &spine );

		if( hn->type != TAG_APPL )
			break;

		PEPOINTLEFT( PEGETVAL( &spine ), &spine );
	}

	if( PEISBINOP( &spine ) ) {
		if( i > 1 )
			return( FALSE );
	}
	else if( PEISDOT( &spine ) || PEISUNOP( &spine ) ) {
		if( i > 0 )
			return( FALSE );
	}
	else if( PEISCOMB( &spine ) ) {
		if( i > nargs[ (int) PEGETCOMB( &spine ) ] - 1 )
			return( FALSE );
	}
	else if( PEISCONSTRUCTOR( &spine ) ) {
		compile = PEGETCOMPILE( &spine );
		na = compile->nparam + compile->nsecret;

		if( i > na ) {
			printf( "constructor %s with %d args ", 
				symbol_name( sym ), i );
			printf( "should have %d args\n", compile->nparam ); 
			return( FALSE );
		}
	}
	else if( PEISSYMBOL( &spine ) ) {
		/* If it's a VIPS or a builtin with too few args, it's OK.
		 */
		sym = SYMBOL( PEGETVAL( &spine ) );

		if( sym->type == SYM_EXTERNAL ) {
			if( i < sym->fn_nargs )
				return( TRUE );
		}
		else if( sym->type == SYM_BUILTIN ) {
			if( i < sym->builtin->nargs )
				return( TRUE );
		}

		/* Nope ... should have been reduced.
		 */
		return( FALSE );
	}
	else {
		return( FALSE );
	}

	return( TRUE );
}
#endif /*WHNF_DEBUG*/

/* Main reduction machine loop.
 */
void
reduce_spine( Reduce *rc, PElement *out )
{
	Heap *hi = rc->hi;
	PElement np;

	/*
	HeapNode *hn, *hn1, *hn2;
	HeapNode **arg;
	PElement np, rhs1, rhs2, rhs3;
	int na;
	 */

	/* Check for possible C stack overflow ... can't go over 2M on most
	 * systems if we're using (or any of our libs are using) threads.
	 */
	if( (char *) main_c_stack_base - (char *) &rc > 2000000 ) 
		everror( rc, "C stack overflow: expression too complicated" );

	/* Point node pointer at reduction start.
	 */
	np = *out;

	/* Start a new frame.
	 */
	RSPUSHFRAME( rc, out ); 

reduce_start:
	reduce_total_recomputations += 1;
	if( (reduce_total_recomputations % 1000) == 0 ) {
		if( mainw_countdown_animate( 99 ) )
			everror( rc, "interrupted" );
	}

	switch( PEGETTYPE( &np ) ) {
	case ELEMENT_CHAR:
	case ELEMENT_BOOL:
	case ELEMENT_ELIST:
	case ELEMENT_TAG:
	case ELEMENT_IMAGE:
	case ELEMENT_SYMREF:
	case ELEMENT_COMPILEREF:
		/* Base type .. no more reduction needed.
		 */

		/* Should have no args.
		 */
		if( RSFRAMESIZE( rc ) != 0 ) 
			argserror( rc, &np );

		break;

	case ELEMENT_CONSTRUCTOR:
	{
		Compile *compile;
		HeapNode **arg;
		PElement rhs1;
		int na;

		/* Class constructor.
		 */
		compile = PEGETCOMPILE( &np );
		if( !is_class( compile ) )
			everror( rc, "%s is not a class", 
				symbol_name( compile->sym ) );

		/* Class args ... real params, secret params.
		 */
		na = compile->nparam + compile->nsecret;

		if( na == 0 ) {
			/* Zero args ... just construct on top of the current
			 * node pointer.
			 */
			action_proc_construct( rc, compile, arg, &np );
			goto reduce_start;
		}

		/* Get args.
		 */
		if( !RSCHECKARGS( rc, na ) ) 
			break;
		arg = &RSGET( rc, na - 1 );

		/* Overwrite RHS of arg[0], make LHS into COMB_I.
		 */
		PEPOINTRIGHT( arg[0], &rhs1 ); 
		action_proc_construct( rc, compile, arg, &rhs1 );
		PPUTLEFT( arg[0], ELEMENT_COMB, COMB_I );

		RSPOP( rc, na );
		if( RSFRAMEEMPTY( rc ) )
			np = RSGETWB( rc ); 
		else
			PEPOINTLEFT( RSGET( rc, 0 ), &np );
		PEPUTP( &np, 
			GETRT( arg[0] ), GETRIGHT( arg[0] ) );

		goto reduce_start;
	}

	case ELEMENT_SYMBOL:
	{
		Symbol *sym = PEGETSYMBOL( &np );

		assert( sym );

		switch( sym->type ) {
		case SYM_VALUE:
		{
			PElement rhs1;

			/* Make sure it's clean and has a value ... we can get 
			 * links to dirty syms through dynamic dependencies.
			 */
			if( sym->dirty || PEISNOVAL( &sym->expr->root ) )
				everror( rc, "%s has no value", 
					symbol_name( sym ) );

			PEPUTPE( &np, &sym->expr->root );

			/* Is it a class with a typecheck member? Go through
			 * that.
			 */
			if( PEISCLASS( &np ) &&
				class_get_member( &np, MEMBER_CHECK, &rhs1 ) ) {
#ifdef DEBUG
				printf( "reduce: invoking arg checker\n" );
#endif 

				PEPUTPE( &np, &rhs1 );
			}

			goto reduce_start;
		}

		case SYM_PARAM:
			/* All params should be taken out by var abstract.
			 */
			everror( rc, "bug!! reference to parameter %s", 
				symbol_name( sym ) );
			assert( FALSE );
			break;

		case SYM_EXTERNAL:
		{
			HeapNode **arg;
			int na;

			/* A VIPS unction.
			 */
			na = sym->fn_nargs;

			/* Get args. 
			 */
			if( !RSCHECKARGS( rc, na ) ) 
				/* Not enough ... function result. 
				 */
				break;

			/* Run strictly.
			 */
			arg = &RSGET( rc, na - 1 );

			action_dispatch( rc, reduce_spine,
				-1, sym->function->name, FALSE,
				(ActionFn) vips_run, na, arg, 
				sym->function );

			/* Find output element.
			 */
			RSPOP( rc, na );

			if( RSFRAMEEMPTY( rc ) )
				np = RSGETWB( rc ); 
			else
				PEPOINTLEFT( RSGET( rc, 0 ), &np );

			/* Write to node above.
			 */
			PEPUTP( &np, 
				GETRT( arg[0] ), GETRIGHT( arg[0] ) );

			goto reduce_start;
		}

		case SYM_BUILTIN:
		{
			HeapNode **arg;
			int na;

			/* A builtin function.
			 */
			na = sym->builtin->nargs;

			/* Get args. 
			 */
			if( !RSCHECKARGS( rc, na ) ) 
				/* Not enough ... function result. 
				 */
				break;

			/* Run strictly.
			 */
			arg = &RSGET( rc, na - 1 );

			action_dispatch( rc, reduce_spine,
				-1, sym->builtin->name, sym->builtin->override,
				(ActionFn) builtin_run, 
				na, arg, sym->builtin );

			/* Find output element.
			 */
			RSPOP( rc, na );

			if( RSFRAMEEMPTY( rc ) )
				np = RSGETWB( rc ); 
			else
				PEPOINTLEFT( RSGET( rc, 0 ), &np );

			/* Write to node above.
			 */
			PEPUTP( &np, 
				GETRT( arg[0] ), GETRIGHT( arg[0] ) );

			goto reduce_start;
		}

		case SYM_ZOMBIE:
			everror( rc, "%s is not defined", symbol_name( sym ) );
			assert( FALSE );
			break;

		case SYM_ROOT:
		case SYM_WORKSPACE:
		case SYM_WORKSPACEGROUP:
			/* Becomes a symref ... base type.
			 */
			PEPUTP( &np, ELEMENT_SYMREF, sym );

			/* Should have no args.
			 */
			if( RSFRAMESIZE( rc ) != 0 ) 
				argserror( rc, &np );

			break;

		default:
			assert( FALSE );
		}

		break;
	}

	case ELEMENT_NODE:
	{
		HeapNode *hn;

		/* Get the node that np points to.
		 */
		hn = PEGETVAL( &np );

		switch( hn->type ) {
		case TAG_CONS:
		case TAG_DOUBLE:
		case TAG_COMPLEX:
		case TAG_CLASS:
			/* Base type ... reduction all done! We don't test
			 * that class's superclasses are base, as they aren't
			 * always for non-top-level base types ... see 
			 * reduce_pelement().
			 */

			/* Should have no args.
			 */
			if( RSFRAMESIZE( rc ) != 0 )
				argserror( rc, &np );

			break;

		case TAG_APPL:
			/* Function application ... push this node and loop
			 * down the LHS looking for a combinator.
			 */

			/* Push this node.
			 */
			RSPUSH( rc, hn );

			/* Move down left branch.
			 */
			PEPOINTLEFT( hn, &np );

			goto reduce_start;
		
		case TAG_GEN:
		{
			double d1;
			double d2;
			double d3 = 0.0;	/* keeps gcc happy */
			gboolean limit;
			HeapNode *hn1, *hn2;

			/* Extract next, step, final.
			 */
			d1 = GETLEFT( hn )->body.num;
			d2 = GETLEFT( GETRIGHT( hn ) )->body.num;
			limit = GETRT( GETRIGHT( hn ) ) != ELEMENT_ELIST;
			if( limit )
				d3 = GETRIGHT( GETRIGHT( hn ) )->body.num;

			if( trace_flags & TRACE_OPERATOR ) {
				BufInfo *buf = trace_push();

				if( limit )
					buf_appendf( buf, 
						"generator %g %g %g ->\n",
						d1, d2, d3 );
				else
					buf_appendf( buf, 
						"generator %g %g ->\n",
						d1, d2 );
			}

			/* At end?
			 */
			if( GETRT( GETRIGHT( hn ) ) != ELEMENT_ELIST &&
				((d2 > 0 && d1 > d3) || 
					(d2 < 0 && d1 < d3)) ) {
				/* Make I node for end.
				 */
				hn->type = TAG_APPL;
				PPUT( hn, 
					ELEMENT_COMB, COMB_I,
					ELEMENT_ELIST, NULL ); 

				/* Write back to node above.
				 */
				PEPUTP( &np, ELEMENT_ELIST, NULL );

				if( trace_flags & TRACE_OPERATOR ) {
					trace_result( TRACE_OPERATOR, &np );
					trace_pop();
				}

				/* All done!
				 */
				break;
			}

			/* Not at end, or no final. Generate new gen node.
			 */
			if( NEWNODE( hi, hn1 ) )
				everror( rc, "%s", error_string );
			*hn1 = *hn;

			/* Change hn into CONS node.
			 */
			hn->type = TAG_CONS;
			PPUTRIGHT( hn, ELEMENT_NODE, hn1 ); 

			/* Generate new number.
			 */
			if( NEWNODE( hi, hn2 ) )
				everror( rc, "%s", error_string );
			hn2->type = TAG_DOUBLE;
			hn2->body.num = d1 + d2;
			PPUTLEFT( hn1, 
				ELEMENT_NODE, hn2 ); 

			if( trace_flags & TRACE_OPERATOR ) {
				trace_result( TRACE_OPERATOR, &np );
				trace_pop();
			}

			/* And loop!
			 */
			goto reduce_start;
		}

		case TAG_DOT:
		{
			HeapNode **arg;
			PElement rhs1;

			/* '.' extract operator ... should have 1 arg.
			 */
			if( !RSCHECKARGS( rc, 1 ) )
				/* Not enough ... function result.
				 */
				break;

			arg = &RSGET( rc, 0 );

			/* We don't use action_dispatch(), since dot is a bit
			 * bizarre ... it's a family of uops, making tracing
			 * slightly different from usual.
			 */
			PEPOINTRIGHT( arg[0], &rhs1 );
			reduce_spine( rc, &rhs1 );

			if( trace_flags & TRACE_OPERATOR ) {
				BufInfo *buf = trace_push();
				PElement rhs2;

				buf_appendf( buf, "\"%s\" ", 
					OPERATOR_NAME( BI_DOT ) );
				PEPOINTLEFT( hn, &rhs2 );
				trace_pelement( &rhs2 ); 
				buf_appends( buf, " " );
				trace_args( arg, 1 );
			}

			action_proc_dot( rc, 
				BI_DOT, OPERATOR_NAME( BI_DOT ), 
				arg, &rhs1 );

			PPUTLEFT( arg[0], ELEMENT_COMB, COMB_I );

			if( trace_flags & TRACE_OPERATOR ) {
				trace_result( TRACE_OPERATOR, &rhs1 );
				trace_pop();
			}

			/* Find output element.
			 */
			RSPOP( rc, 1 );
			if( RSFRAMEEMPTY( rc ) )
				np = RSGETWB( rc ); 
			else
				PEPOINTLEFT( RSGET( rc, 0 ), &np );

			/* Write to above node.
			 */
			PEPUTP( &np, GETRT( arg[0] ), GETRIGHT( arg[0] ) );

			/* Loop again with new np.
			 */
			goto reduce_start;
		}

		case TAG_FREE:
			everror( rc, "panic: free node in heap!\n" );

		default:
			everror( rc, "panic: unknown node in heap!\n" );
		}

		break;
	}

	case ELEMENT_COMB:
	{
		CombinatorType comb = PEGETCOMB( &np );
		HeapNode *hn1, *hn2;
		HeapNode **arg;
		int na;

		na = nargs[ (int) comb ];

		/* Get args. 
		 */
		if( !RSCHECKARGS( rc, na ) ) 
			/* Not enough ... function result. 
			 */
			break;

		/* Extract args.
		 */
		arg = &RSGET( rc, na - 1 );

		switch( comb ) {
		case COMB_S:
			/* Rewrite graph for S a b c => (a c) (b c).
			 */

			/* Make (b c) appl node.
			 */
			if( NEWNODE( hi, hn1 ) )
				everror( rc, "%s", error_string );
			*hn1 = *arg[0];
			PPUTLEFT( hn1, 
				GETRT( arg[1] ), GETRIGHT( arg[1] ) ); 
			PPUTRIGHT( arg[0], 
				ELEMENT_NODE, hn1 );

			/* Make (a c) appl node.
			 */
			if( NEWNODE( hi, hn2 ) )
				everror( rc, "%s", error_string );
			*hn2 = *hn1;
			PPUTLEFT( hn2, 
				 GETRT( arg[2] ), GETRIGHT( arg[2] ) );
			PPUTLEFT( arg[0], 
				ELEMENT_NODE, hn2 ); 

			/* End of S ... now pop three, push 1 and loop.
			 */
			RSPOP( rc, 2 );
			PEPOINTLEFT( arg[0], &np );
			goto reduce_start;

		case COMB_SL:
			/* Rewrite graph for Sl a b c => (a c) b.
			 */

			/* Make (a c) appl node.
			 */
			if( NEWNODE( hi, hn1 ) )
				everror( rc, "%s", error_string );
			*hn1 = *arg[0];
			PPUTLEFT( hn1, 
				GETRT( arg[2] ), GETRIGHT( arg[2] ) );
			PPUT( arg[0], 
				ELEMENT_NODE, hn1,
				GETRT( arg[1] ), GETRIGHT( arg[1] ) );

			/* End of SL ... now pop three, push 1 and loop.
			 */
			RSPOP( rc, 2 );
			PEPOINTLEFT( arg[0], &np );
			goto reduce_start;

		case COMB_SR:
			/* Rewrite graph for Sr a b c => a (b c).
			 */

			/* Make (b c) appl node.
			 */
			if( NEWNODE( hi, hn1 ) )
				everror( rc, "%s", error_string );
			*hn1 = *arg[0];
			PPUTLEFT( hn1, 
				GETRT( arg[1] ), GETRIGHT( arg[1] ) );
			PPUT( arg[0],
				GETRT( arg[2] ), GETRIGHT( arg[2] ),
				ELEMENT_NODE, hn1 );

			/* End of SR ... now pop three, push 1 and loop.
			 */
			RSPOP( rc, 2 );
			PEPOINTLEFT( arg[0], &np );
			goto reduce_start;

		case COMB_I:
			/* No action necessary.
			 */
			break;

		case COMB_K:
			/* Make I node. 
			 */
			PPUT( arg[0], 
				ELEMENT_COMB, COMB_I,
				GETRT( arg[1] ), GETRIGHT( arg[1] ) );

			break;

		case COMB_GEN:
		{
			double d1;
			double d2 = 0.0;	/* Don't need to init, but */
			double d3 = 0.0;	/* keeps gcc happy */
			PElement rhs1, rhs2, rhs3;

			PEPOINTRIGHT( arg[2], &rhs1 );
			PEPOINTRIGHT( arg[1], &rhs2 );
			PEPOINTRIGHT( arg[0], &rhs3 );
			reduce_spine_strict( rc, &rhs1 );
			reduce_spine_strict( rc, &rhs2 );
			reduce_spine_strict( rc, &rhs3 );

			/* May have done ourselves in the process.
			 */
			if( arg[0]->type != TAG_APPL )
				break;

			/* Typecheck.
			 */
			if( !PEISREAL( &rhs1 ) ) 
				everror( rc, "start of generator is not real" );
			d1 = PEGETREAL( &rhs1 );

			if( !PEISELIST( &rhs2 ) && !PEISREAL( &rhs2 ) ) 
				everror( rc, "next of generator is not real" );
			if( PEISREAL( &rhs2 ) )
				d2 = PEGETREAL( &rhs2 ); 

			if( !PEISELIST( &rhs3 ) && !PEISREAL( &rhs3 ) ) 
				everror( rc, "final of generator is not real" );
			if( PEISREAL( &rhs3 ) )
				d3 = PEGETREAL( &rhs3 ); 

			if( trace_flags & TRACE_OPERATOR ) {
				BufInfo *buf = trace_push();

				buf_appends( buf, "generator constructor " );
				trace_args( arg, 3 );
			}

			/* If next is missing, set default.
			 */
			if( PEISREAL( &rhs2 ) ) 
				/* Next is there, calculate step.
				 */
				d2 = d2 - d1;
			else {
				/* If final is missing, default is 1.
				 */
				if( PEISELIST( &rhs3 ) ) 
					d2 = 1;
				else {
					/* Final is there, choose 1 or -1.
					 */
					if( d1 < d3 )
						d2 = 1;
					else
						d2 = -1;
				}
			}

			/* Make node for pairing next and final fields.
			 */
			if( NEWNODE( hi, hn1 ) )
				everror( rc, "%s", error_string );
			hn1->type = TAG_COMPLEX;
			PPUT( hn1, 
				GETRT( arg[1] ), GETRIGHT( arg[1] ),
				GETRT( arg[0] ), GETRIGHT( arg[0] ) );

			/* Link to old root, make gen node.
			 */
			arg[0]->type = TAG_GEN;
			PPUT( arg[0],
				GETRT( arg[2] ), GETRIGHT( arg[2] ),
				ELEMENT_NODE, hn1 );

			/* Make step node.
			 */
			if( NEWNODE( hi, hn2 ) )
				everror( rc, "%s", error_string );
			hn2->type = TAG_DOUBLE;
			hn2->body.num = d2;
			PPUTLEFT( hn1,
				ELEMENT_NODE, hn2 );

			if( trace_flags & TRACE_OPERATOR ) {
				BufInfo *buf = trace_current();

				buf_appends( buf, "    " ); 
				trace_node( arg[0] );
				buf_appends( buf, "\n" ); 

				trace_text( TRACE_OPERATOR, buf_all( buf ) ); 

				trace_pop();
			}

			/* Find output element.
			 */
			RSPOP( rc, 3 );
			if( RSFRAMEEMPTY( rc ) )
				np = RSGETWB( rc ); 
			else
				PEPOINTLEFT( RSGET( rc, 0 ), &np );

			/* Restart from there.
			 */
			goto reduce_start;
		}

		default:
			everror( rc, "panic: unknown combinator in heap!" );
		}

		/* Find output element.
		 */
		RSPOP( rc, na );
		if( RSFRAMEEMPTY( rc ) )
			np = RSGETWB( rc ); 
		else
			PEPOINTLEFT( RSGET( rc, 0 ), &np );

		/* Write to above node.
		 */
		PEPUTP( &np, 
			 GETRT( arg[0] ), GETRIGHT( arg[0] ) );

		/* Loop again with new np.
		 */
		goto reduce_start;
		/*NOTREACHED*/
	}

	case ELEMENT_BINOP:
	{
		BinOp bop = PEGETBINOP( &np );
		HeapNode **arg;
		PElement rhs1, rhs2;

		if( !RSCHECKARGS( rc, 2 ) )
			/* Not enough ... function result.
			 */
			break;

		/* Extract args.
		 */
		arg = &RSGET( rc, 1 );

		/* CONS is very, very lazy ... more like a combinator.
		 */
		if( bop == BI_CONS ) {
			PEPOINTRIGHT( arg[1], &rhs1 );

			if( trace_flags & TRACE_OPERATOR ) {
				trace_push();

				PEPOINTRIGHT( arg[0], &rhs2 );
				trace_binop( &rhs1, bop, &rhs2 );
			}

			arg[0]->type = TAG_CONS;
			PPUTLEFT( arg[0], 
				PEGETTYPE( &rhs1 ), PEGETVAL( &rhs1 ) );

			if( trace_flags & TRACE_OPERATOR ) {
				BufInfo *buf = trace_current();

				buf_appends( buf, "    " ); 
				trace_node( arg[0] );
				buf_appends( buf, "\n" ); 

				trace_text( TRACE_OPERATOR, buf_all( buf ) );

				trace_pop();
			}

			RSPOP( rc, 2 );

			break;
		}

		action_proc_bop( rc, bop, arg );

		/* Find output element.
		 */
		RSPOP( rc, 2 );

		if( RSFRAMEEMPTY( rc ) )
			np = RSGETWB( rc ); 
		else
			PEPOINTLEFT( RSGET( rc, 0 ), &np );

		/* Write to node above.
		 */
		PEPUTP( &np, 
			GETRT( arg[0] ), GETRIGHT( arg[0] ) );

		/* Loop again with new np.
		 */
		goto reduce_start;
	}

	case ELEMENT_UNOP:
	{
		HeapNode **arg;

		/* Some unary operator. These all take one
		 * arg, which much be evaled a bit.
		 */
		if( !RSCHECKARGS( rc, 1 ) )
			/* Not enough ... function result.
			 */
			break;

		/* Extract arg.
		 */
		arg = &RSGET( rc, 0 );

		action_dispatch( rc, reduce_spine,
			PEGETUNOP( &np ), OPERATOR_NAME( PEGETUNOP( &np ) ),
			TRUE, (ActionFn) action_proc_uop, 1, arg, NULL );

		/* Find output element.
		 */
		RSPOP( rc, 1 );
		if( RSFRAMEEMPTY( rc ) )
			np = RSGETWB( rc ); 
		else
			PEPOINTLEFT( RSGET( rc, 0 ), &np );

		/* Write to above node.
		 */
		PEPUTP( &np, 
			 GETRT( arg[0] ), GETRIGHT( arg[0] ) );

		/* Loop again with new np.
		 */
		goto reduce_start;
	}

	case ELEMENT_NOVAL:
		break;

	default:
		everror( rc, "panic: unknown element tag in heap!" );
	}

	/* Unwind stack, restore frame pointer.
	 */
	RSPOPFRAME( rc ); 

#ifdef WHNF_DEBUG
	/* Should now be in WHNF ... test!
	 */
	if( !is_WHNF( out ) ) {
		BufInfo buf;
		char txt[1000];

		buf_init_static( &buf, txt, 1000 );
		graph_pelement( hi, &buf, out, TRUE );
		printf( "*** internal error:\n" );
		printf( "result of reduce_spine not in WHNF: " );
		printf( "%s\n", buf_all( &buf ) );
		everror( rc, "not in WHNF!" );
	}
#endif /*WHNF_DEBUG*/
}

/* Strict reduction ... fully eval all lists etc.
 */
void
reduce_spine_strict( Reduce *rc, PElement *np )
{
	PElement rhs, lhs;

	/* Make sure this element is reduced.
	 */
	if( PEGETTYPE( np ) == ELEMENT_NODE ||
		PEGETTYPE( np ) == ELEMENT_SYMBOL ) 
		reduce_spine( rc, np );

	/* If it's a non-empty list, may need to reduce inside.
	 */
	if( PEISFLIST( np ) ) {
		/* Recurse for head and tail.
		 */
		HeapNode *hn = PEGETVAL( np );

		PEPOINTLEFT( hn, &lhs );
		PEPOINTRIGHT( hn, &rhs );
		reduce_spine_strict( rc, &lhs );
		reduce_spine_strict( rc, &rhs );
	}
}

/* Free a Reduce.
 */
void
reduce_destroy( Reduce *rc )
{
	heap_unregister_reduce( rc->hi, rc );
	FREEF( heap_destroy, rc->hi );
	FREE( rc );
}

/* Max cells function for main reduce engine. Read from Preferences, and scale
 * by the number of workspaces we have open.
 */
static int
reduce_heap_max_fn( Heap *heap )
{
	return( workspace_number() * MAX_HEAPSIZE );
}

/* Build a Reduce.
 */
Reduce *
reduce_new( void )
{
	/* Initial heap size.
	 */
	const int stsz = 20000;

	/* Heap increment..
	 */
	const int incr = 2000;

	Reduce *rc = IM_NEW( NULL, Reduce );

	if( !rc )
		return( NULL );
	rc->sp = 0;
	rc->fsp = 0;
	rc->hi = NULL;
	rc->running = 0;

	if( !(rc->hi = heap_new( NULL, reduce_heap_max_fn, stsz, incr )) ) { 
		reduce_destroy( rc );
		return( NULL );
	}
	heap_register_reduce( rc->hi, rc );
	SETSTR( rc->hi->name, "reduce-heap" );

	return( rc );
}

/* Reduce a PElement to a base type. Return TRUE/FALSE, no longjmp.
 */
gboolean
reduce_pelement( Reduce *rc, ReduceFunction fn, PElement *out )
{
	gboolean res = TRUE;

	trace_reset();
	REDUCE_CATCH_START( FALSE );
	fn( reduce_context, out );
	REDUCE_CATCH_STOP;
	trace_check();

	return( res );
}

/* Make sure a symbol's value is registered with the main GC.
 */
void
reduce_register( Symbol *sym )
{
	Reduce *rc = reduce_context;
	Heap *hi = rc->hi;

	heap_register_element( hi, &sym->base );
}

/* Make sure a symbol's value is not registered with the main GC.
 */
void
reduce_unregister( Symbol *sym )
{
	Reduce *rc = reduce_context;
	Heap *hi = rc->hi;

	heap_unregister_element( hi, &sym->base );
}

static gboolean update_graph( Heap *hi, Compile *compile, PElement *out );

/* Copy all sub-definitions into heap, then copy this definition.
 */
static void *
update_graph_sub( Symbol *sym, Heap *hi )
{
	if( sym->expr && sym->expr->compile ) 
		if( !update_graph( hi, sym->expr->compile, &sym->expr->root ) )
			return( sym );

	return( NULL );
}

static gboolean
update_graph( Heap *hi, Compile *compile, PElement *out )
{
	/* Copy any locals.
	 */
	if( stable_map( compile->locals, 
		(symbol_map_fn) update_graph_sub, hi, NULL, NULL ) )
		return( FALSE );

#ifdef DEBUG
	printf( "update_graph: copying code for " );
	compile_name_print( compile );
	printf( "\n" );
#endif /*DEBUG*/

	if( !heap_copy( hi, &compile->base, out ) )
		return( FALSE );

	return( TRUE );
}

/* Copy and evaluate compiled code into element pointed to by out.
 */
gboolean
reduce_regenerate( Expr *expr, PElement *out )
{
	Reduce *rc = reduce_context;
	Heap *hi = rc->hi;

	/* Clear any run state from old expr value.
	 */
	expr_error_clear( expr );
	if( slist_map( expr->dynamic_links, 
		(SListMapFn) link_expr_destroy, NULL ) )
		return( FALSE );

	/* Copy new code in.
	 */
	if( !update_graph( hi, expr->compile, out ) ) {
		expr_error_set( expr );
		return( FALSE );
	}

#ifdef DEBUG_REGEN
{
	BufInfo buf;
	char txt[1024];

	buf_init_static( &buf, txt, 1024 );
	graph_pelement( hi, &buf, out, TRUE );
	printf( "reduce_regenerate: reducing " );
	expr_name_print( expr );
	printf( "graph: %s\n", buf_all( &buf ) );
}
#endif /*DEBUG_REGEN*/

	/* Do initial reduction.
	 */
	if( !reduce_pelement( rc, reduce_spine, out ) ) {
		/* Failure! Junk the half-made value. 
		 */
		expr_error_set( expr );
		PEPUTP( out, ELEMENT_NOVAL, NULL );
		(void) heap_gc( hi );
		return( FALSE );
	}

#ifdef DEBUG_REGEN
{
	BufInfo buf;
	char txt[1024];

	/* Force immediate GC to pick up any stray pointers.
	 */
	if( !heap_gc( hi ) ) {
		expr_error_set( expr );
		return( FALSE );
	}

	buf_init_static( &buf, txt, 1024 );
	graph_pelement( hi, &buf, out, TRUE );
	printf( "reduce_regenerate: reduced " );
	expr_name_print( expr );
	printf( " to: %s\n", buf_all( &buf ) );
}
#endif /*DEBUG_REGEN*/

	return( TRUE );
}

/* Regenerate an (expr this) pair.
 */
gboolean
reduce_regenerate_member( Expr *expr, PElement *ths, PElement *out )
{
	Reduce *rc = reduce_context;
	Heap *hi = rc->hi;

	PElement e;
	HeapNode *apl;

	/* New (NULL this) pair.
	 */
	if( NEWNODE( hi, apl ) ) {
		expr_error_set( expr );
		return( FALSE );
	}
	apl->type = TAG_APPL;
	PPUT( apl, ELEMENT_NOVAL, NULL, PEGETTYPE( ths ), PEGETVAL( ths ) ); 
	PEPUTP( out, ELEMENT_NODE, apl );

	/* Link code to node.
	 */
	PEPOINTLEFT( apl, &e );
	if( !reduce_regenerate( expr, &e ) ) 
		return( FALSE );

#ifdef DEBUG_REGEN_MEMBER
{
	BufInfo buf;
	char txt[1024];

	buf_init_static( &buf, txt, 1024 );
	graph_pelement( hi, &buf, out, TRUE );
	printf( "reduce_regenerate_member: " );
	expr_name_print( expr );
	printf( " new code: %s\n", buf_all( &buf ) );
}
#endif /*DEBUG_REGEN_MEMBER*/

	/* Do initial reduction.
	 */
	if( !reduce_pelement( rc, reduce_spine, out ) ) {
		/* Failure! Junk the half-made value. 
		 */
		expr_error_set( expr );
		PEPUTP( out, ELEMENT_NOVAL, NULL );
		(void) heap_gc( hi );
		return( FALSE );
	}

	/* Special case: if this is a "super" row, we need to rebuild the
	 * class.
	 */
	if( is_super( expr->compile->sym ) ) {
		Compile *parent = compile_get_parent( expr->compile );
		PElement instance;

		PEPOINTE( &instance, &expr->row->scol->base );

		if( !class_new_super( hi, parent, ths, &instance ) )
			return( FALSE );
	}

	return( TRUE );
}
