/* run the displays for regions on images
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/* Verbose.
#define DEBUG
 */

/* Just trace create/destroy.
#define DEBUG_MAKE
 */

/* Trace grab/ungrab
#define DEBUG_GRAB
 */

/* Define this to do pointer grabs for drag/resize. Can make debugging tricky.
 * Without this, drags outside the window can get hosed.
 */
#define GRAB

/* Define this to make region drags default to no-update during drag/resize.
 */
#define NO_UPDATE

/* Define this to trace event propogation
#define EVENT
 */

#include "ip.h"

#ifdef NO_UPDATE
#define CALC_RECOMP_REGION (watch_bool_get( "CALC_RECOMP_REGION", FALSE ))
#else
#define CALC_RECOMP_REGION (watch_bool_get( "CALC_RECOMP_REGION", TRUE ))
#endif

static ViewClass *parent_class = NULL;

/* Region border width, without shadows.
 */
static const int regionview_border_width = 2;

/* Space around text in label.
 */
static const int regionview_label_border = 5;

/* Length of crosshair bars.
 */
static const int regionview_crosshair_length = 5;

/* How close you need to get to switch the type.
 */
static const int regionview_morph_threshold = 3;

/* Just one popup for all regions.
 */
static GtkWidget *regionview_popup_menu = NULL;

typedef void *(*regionview_rect_fn)( Regionview *, Rect *, void * );
typedef void (*regionview_paint_fn)( Regionview * );

/* Paint a rectangle. 
 */
void
regionview_paint_rect( GdkDrawable *draw, GdkGC *gc, Rect *r )
{	
	gdk_draw_rectangle( draw, gc, FALSE, 
		r->left, r->top, 
		IM_MAX( 0, r->width - 1 ), IM_MAX( 0, r->height - 1 ) );
}

/* Paint a thick rectangle. 
 */
void
regionview_paint_rect_thick( GdkDrawable *draw, GdkGC *gc, Rect *r, int n )
{
	Rect our_r;
	int i;

	our_r = *r;
	for( i = 0; i < n; i++ ) {
		regionview_paint_rect( draw, gc, &our_r );
		im_rect_marginadjust( &our_r, 1 );
	}
}

/* Paint a rect in 3D --- pass a GC for the top-left and a gc for the
 * bottom-right shadows.
 */
static void
regionview_paint_rect_3d( GdkDrawable *draw, GdkGC *tl, GdkGC *br, Rect *r )
{
	/* Bottom and right.
	 */
	gdk_draw_line( draw, br, 
		IM_RECT_RIGHT( r ) - 1, r->top, 
		IM_RECT_RIGHT( r ) - 1, IM_RECT_BOTTOM( r ) - 1 );
	gdk_draw_line( draw, br, 
		IM_RECT_RIGHT( r ) - 1, IM_RECT_BOTTOM( r ) - 1, 
		r->left, IM_RECT_BOTTOM( r ) - 1 );

	/* Top and left.
	 */
	gdk_draw_line( draw, tl, 
		r->left, IM_RECT_BOTTOM( r ) - 1, r->left, r->top );
	gdk_draw_line( draw, tl, 
		r->left, r->top, IM_RECT_RIGHT( r ) - 1, r->top );
}

/* Paint little ticks ... for marking the edges of the resize handles.
 */
static void
regionview_paint_vtick( GdkDrawable *draw, GdkGC *tl, GdkGC *br, 
	int x, int y, int n )
{
	gdk_draw_line( draw, br, x, y, x, y - n - 1 );
	gdk_draw_line( draw, tl, x + 1, y, x + 1, y - n - 1 );
}

static void
regionview_paint_htick( GdkDrawable *draw, GdkGC *tl, GdkGC *br, 
	int x, int y, int n )
{
	gdk_draw_line( draw, br, x, y, x - n - 1, y );
	gdk_draw_line( draw, tl, x, y + 1, x - n - 1, y + 1 );
}

/* Paint a region border, enclosing the pixels in r.
 */
static void
regionview_paint_border( GdkDrawable *draw, GdkGC *tl, GdkGC *bg, GdkGC *br,
	Rect *r, gboolean locked ) 
{
	int n = regionview_border_width;
	Rect our_r = *r;

	im_rect_marginadjust( &our_r, 1 );
	regionview_paint_rect_3d( draw, br, tl, &our_r );
	im_rect_marginadjust( &our_r, 1 );
	regionview_paint_rect_thick( draw, bg, &our_r, n );
	im_rect_marginadjust( &our_r, n );
	regionview_paint_rect_3d( draw, tl, br, &our_r );

	/* Add little tick marks for corner resizing. Don't bother for very 
	 * small rects, or for locked rects.
	 */
	if( !locked && r->width > 20 ) {
		/* Top edge.
		 */
		regionview_paint_vtick( draw, tl, br, 
			r->left + 10, r->top - 1, n );
		regionview_paint_vtick( draw, tl, br, 
			IM_RECT_RIGHT( r ) - 11, r->top - 1, n );

		/* Bottom edge.
		 */
		regionview_paint_vtick( draw, tl, br, 
			r->left + 10, IM_RECT_BOTTOM( r ) + n + 1, n );
		regionview_paint_vtick( draw, tl, br, 
			IM_RECT_RIGHT( r ) - 11, IM_RECT_BOTTOM( r ) + n + 1, 
			n );
	}

	if( !locked && r->height > 20 ) {
		/* Left edge.
		 */
		regionview_paint_htick( draw, tl, br, 
			r->left - 1, r->top + 10, n );
		regionview_paint_htick( draw, tl, br, 
			r->left - 1, IM_RECT_BOTTOM( r ) - 12, n );

		/* Right edge.
		 */
		regionview_paint_htick( draw, tl, br, 
			IM_RECT_RIGHT( r ) + n + 1, r->top + 10, n );
		regionview_paint_htick( draw, tl, br, 
			IM_RECT_RIGHT( r ) + n + 1, IM_RECT_BOTTOM( r ) - 12, 
			n );
	}
}

/* Paint a square area, with a beveled edge.
 */
static void
regionview_paint_area( GdkDrawable *draw, 
	GdkGC *tl, GdkGC *bg, GdkGC *br,
	Rect *r )
{
	gdk_draw_rectangle( draw, bg, TRUE,
		r->left, r->top, r->width, r->height );
	regionview_paint_rect_3d( draw, tl, br, r );
}

/* Paint a region label.
 */
static void
regionview_paint_label( Regionview *regionview, GdkDrawable *draw, 
	GdkGC *tl, GdkGC *fg, GdkGC *bg, GdkGC *br,
	Rect *r, int ascent, const char *txt )
{
	GdkFont *font = gtk_widget_get_style( GTK_WIDGET( regionview ) )->font;
	int n = regionview_label_border;

	regionview_paint_area( draw, tl, bg, br, r );
	gdk_draw_string( draw, font, fg, 
		r->left + n, r->top + n + ascent, txt );
}

/* Paint a crosshair, centered at x, y.
 */
static void
regionview_paint_crosshair( GdkDrawable *draw,
        GdkGC *tl, GdkGC *bg, GdkGC *br,
	int x, int y )
{
	const int bw = regionview_border_width / 2 + 1;
	const int l = regionview_crosshair_length + 2;

	Rect area;

	area.left = x - bw - 1 - l;
	area.top = y - bw;
	area.width = l;
	area.height = bw * 2;
	regionview_paint_area( draw, tl, bg, br, &area );

	area.left = x + bw + 1;
	regionview_paint_area( draw, tl, bg, br, &area );

	area.left = x - bw;
	area.top = y - bw - 1 - l;
	area.width = bw * 2;
	area.height = l;
	regionview_paint_area( draw, tl, bg, br, &area );

	area.top = y + bw + 1;
	regionview_paint_area( draw, tl, bg, br, &area );
}

/* Paint the dotted line connecting an arrow or a guide.
 */
static void
regionview_paint_arrow( GdkDrawable *draw, GdkGC *fg, Rect *r )
{
	gdk_draw_line( draw, fg, 
		r->left, r->top, IM_RECT_RIGHT( r ), IM_RECT_BOTTOM( r ) );
}

/* Apply a function to every rect in a crosshair positioned at (x, y).
 */
static void *
regionview_crosshair_foreach( Regionview *regionview, 
	int x, int y, regionview_rect_fn fn, void *data )
{
	const int n = regionview_border_width + 2;
	const int l = regionview_crosshair_length + 2;

	Rect area;
	void *res;

	area.left = x - n/2 - 1 - l;
	area.top = y - n/2;
	area.width = l;
	area.height = n;
	if( (res = fn( regionview, &area, data )) )
		return( res );

	area.left = x + n/2 + 1;
	if( (res = fn( regionview, &area, data )) )
		return( res );

	area.left = x - n/2;
	area.top = y - n/2 - 1 - l;
	area.width = n;
	area.height = l;
	if( (res = fn( regionview, &area, data )) )
		return( res );

	area.top = y + n/2 + 1;
	if( (res = fn( regionview, &area, data )) )
		return( res );

	return( NULL );
}

/* Apply a function to every rect in a region border positioned at border.
 */
static void *
regionview_border_foreach( Regionview *regionview, 
	Rect *border, regionview_rect_fn fn, void *data )
{
	const int n = regionview_border_width + 2;

	Rect area;
	void *res;

	area.left = border->left - n;
	area.top = border->top - n;
	area.width = border->width + 2*n;
	area.height = n;
	if( (res = fn( regionview, &area, data )) )
		return( res );

	area.top = IM_RECT_BOTTOM( border );
	if( (res = fn( regionview, &area, data )) )
		return( res );

	area.left = border->left - n;
	area.top = border->top;
	area.width = n;
	area.height = border->height;
	if( (res = fn( regionview, &area, data )) )
		return( res );

	area.left = IM_RECT_RIGHT( border );
	if( (res = fn( regionview, &area, data )) )
		return( res );

	return( NULL );
}

/* Repaint ... as a rect_foreach function.
 */
static void *
regionview_rect_repaint( Regionview *regionview, Rect *area )
{
	imagedisplay_repaint( regionview->ip->id, area );

	return( NULL );
}

/* Unpaint a region ... assume it's painted ATM, so do any xoring to unpaint
 * lines etc., then queue repaints for the drawn bits.
 */
static void
regionview_unpaint( Regionview *regionview )
{
	Imagedisplay *id = regionview->ip->id;
	Conversion *conv = id->conv;
	GdkDrawable *draw = GTK_WIDGET( id )->window;
	Rect *area = &regionview->area;

	Rect dr, er;
	int x, y;

	/* Stop async paints.
	 */
	regionview->unpainting = TRUE;

	switch( regionview->last_type ) {
	case REGIONVIEW_AREA:
	case REGIONVIEW_REGION:
		conversion_im_to_disp_rect( conv, area, &dr );
		(void) regionview_border_foreach( regionview, &dr, 
			(regionview_rect_fn) regionview_rect_repaint, NULL );
		break;

	case REGIONVIEW_POINT:
		conversion_im_to_disp( conv, area->left, area->top, &x, &y );
		(void) regionview_crosshair_foreach( regionview, x, y, 
			(regionview_rect_fn) regionview_rect_repaint, NULL );
		break;

	case REGIONVIEW_ARROW:
		conversion_im_to_disp_rect( conv, area, &dr );
		imagedisplay_disp_to_xev_rect( id, &dr, &er );
		regionview_paint_arrow( draw, id->xor_gc, &er );

		(void) regionview_crosshair_foreach( regionview, 
			dr.left, dr.top,
			(regionview_rect_fn) regionview_rect_repaint, NULL );
		(void) regionview_crosshair_foreach( regionview, 
			IM_RECT_RIGHT( &dr ), 
			IM_RECT_BOTTOM( &dr ),
			(regionview_rect_fn) regionview_rect_repaint, NULL );
		break;

	case REGIONVIEW_HGUIDE:
	case REGIONVIEW_VGUIDE:
	case REGIONVIEW_LINE:
		conversion_im_to_disp_rect( conv, area, &dr );
		imagedisplay_disp_to_xev_rect( id, &dr, &er );
		regionview_paint_arrow( draw, id->xor_gc, &er );
		break;

	case REGIONVIEW_BOX:
		conversion_im_to_disp_rect( conv, area, &dr );
		imagedisplay_disp_to_xev_rect( id, &dr, &er );
		im_rect_normalise( &er );
		regionview_paint_rect( draw, id->xor_gc, &er );
		break;

	default:
		assert( FALSE );
	}

	if( regionview->classmodel )
		imagedisplay_repaint( id, &regionview->label );
}

/* Paint a region ... assume the screen has only the background visible (ie.
 * we've nothing of this region visible). Clip paints against clip rect (in
 * xev coordinates) ... either the expose area, or the imagedisplay area.
 */
static void
regionview_paint( Regionview *regionview )
{
	Imagepresent *ip = regionview->ip;
	Imagedisplay *id = ip->id;
	Conversion *conv = id->conv;

	GtkStyle *style = gtk_widget_get_style( GTK_WIDGET( id ) );
	GdkDrawable *draw = GTK_WIDGET( id )->window;
	int state = regionview->last_paint_state;

	GdkGC *tl, *br, *fg, *bg;
	Rect dr, er;

	/* Async updates back on.
	 */
	regionview->unpainting = FALSE;

	tl = style->light_gc[state];
	br = style->dark_gc[state];
	fg = style->fg_gc[state];
	bg = style->bg_gc[state];

	conversion_im_to_disp_rect( conv, &regionview->area, &dr );
	imagedisplay_disp_to_xev_rect( id, &dr, &er );
	switch( regionview->last_type ) {
	case REGIONVIEW_REGION:
		regionview_paint_border( draw, tl, bg, br, &er, FALSE );
		break;

	case REGIONVIEW_AREA:
		regionview_paint_border( draw, tl, bg, br, &er, TRUE );
		break;

	case REGIONVIEW_POINT:
		regionview_paint_crosshair( draw, 
			tl, bg, br, er.left, er.top );
		break;

	case REGIONVIEW_ARROW:
		regionview_paint_arrow( draw, id->xor_gc, &er );

		regionview_paint_crosshair( draw, 
			tl, bg, br, er.left, er.top );
		regionview_paint_crosshair( draw, tl, bg, br, 
			IM_RECT_RIGHT( &er ), IM_RECT_BOTTOM( &er ) );
		break;

	case REGIONVIEW_HGUIDE:
	case REGIONVIEW_VGUIDE:
	case REGIONVIEW_LINE:
		regionview_paint_arrow( draw, id->xor_gc, &er );
		break;

	case REGIONVIEW_BOX:
		im_rect_normalise( &er );
		regionview_paint_rect( draw, id->xor_gc, &er );
		break;

	default:
		assert( FALSE );
	}

	if( regionview->classmodel ) {
		imagedisplay_disp_to_xev_rect( id, &regionview->label, &er );
		if( regionview->paint_state == GTK_STATE_ACTIVE )
			regionview_paint_label( 
				regionview,
				draw, br, fg, bg, tl, &er, 
				regionview->ascent,
				buf_all( &regionview->caption ) );
		else
			regionview_paint_label( 
				regionview,
				draw, tl, fg, bg, br, &er, 
				regionview->ascent,
				buf_all( &regionview->caption ) );
	}
}

/* Call a paint or unpaint fn, setting a clip.
 */
static void
regionview_paint_clipped( Regionview *regionview, 
	regionview_paint_fn fn, Rect *clip )
{
	static char dash_list[] = { 10, 10 };

	Imagepresent *ip = regionview->ip;
	Imagedisplay *id = ip->id;
	GtkStyle *style = gtk_widget_get_style( GTK_WIDGET( id ) );
	int state = regionview->last_paint_state;

	GdkGC *tl, *br, *fg, *bg;
	GdkRectangle gr;

	tl = style->light_gc[state];
	br = style->dark_gc[state];
	fg = style->fg_gc[state];
	bg = style->bg_gc[state];

	gr.x = clip->left;
	gr.y = clip->top;
	gr.width = clip->width;
	gr.height = clip->height;
	gdk_gc_set_clip_rectangle( tl, &gr );
	gdk_gc_set_clip_rectangle( br, &gr );
	gdk_gc_set_clip_rectangle( fg, &gr );
	gdk_gc_set_clip_rectangle( bg, &gr );
	gdk_gc_set_clip_rectangle( id->xor_gc, &gr );
	gdk_gc_set_dashes( id->xor_gc, regionview->dash_offset, dash_list, 2 );

	fn( regionview );

	gdk_gc_set_clip_rectangle( tl, NULL );
	gdk_gc_set_clip_rectangle( br, NULL );
	gdk_gc_set_clip_rectangle( fg, NULL );
	gdk_gc_set_clip_rectangle( bg, NULL );
	gdk_gc_set_clip_rectangle( id->xor_gc, NULL );
}

static void
regionview_ungrab( Regionview *regionview, guint32 time )
{
	if( regionview->grabbed ) {
#ifdef DEBUG_GRAB
		printf( "regionview_ungrab: ungrabbing\n" );
#endif /*DEBUG_GRAB*/

		regionview->grabbed = FALSE;

#ifdef GRAB
                gtk_grab_remove( GTK_WIDGET( regionview->ip->id ) );
		gdk_pointer_ungrab( time );
#else /*GRAB*/
		printf( "regionview_ungrab: true grabs disabled\n" );
#endif /*GRAB*/

		imagepresent_scroll_stop( regionview->ip );
	}
}

static void
regionview_destroy( GtkObject *object )
{
	Regionview *regionview;
	Imagedisplay *id;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_REGIONVIEW( object ) );

	regionview = REGIONVIEW( object );
	id = regionview->ip->id;

#ifdef DEBUG_MAKE
	printf( "regionview_destroy: 0x%x\n", (unsigned int) regionview );
#endif /*DEBUG_MAKE*/

	/* If have drawn, need to queue an undraw.
	 */
	if( !regionview->first && GTK_WIDGET_REALIZED( GTK_WIDGET( id ) ) ) 
		regionview_paint_clipped( regionview, 
			regionview_unpaint, &regionview->ip->id->screen );

	regionview_ungrab( regionview, 0 );

	FREESID( regionview->expose_start_sid, id ); 
	FREESID( regionview->repaint_sid, id ); 
	FREESID( regionview->destroy_sid, id ); 
	FREESID( regionview->event_sid, id ); 
	FREESID( regionview->changed_sid, id->conv ); 
	FREESID( regionview->conv_destroy_sid, id->conv ); 
	FREESID( regionview->ws_changed_sid, main_workspacegroup ); 
	FREESID( regionview->model_changed_sid, regionview->classmodel ); 
	FREEFI( gtk_timeout_remove, regionview->click );
	FREEFI( gtk_timeout_remove, regionview->dash_crawl );
	FREEF( iwindow_cursor_context_destroy, regionview->cntxt );
	buf_destroy( &regionview->caption );

	if( regionview->ip ) {
		regionview->ip->regionviews = 
			g_slist_remove( regionview->ip->regionviews, 
			regionview );
		regionview->ip = NULL;
	}

	if( regionview->classmodel ) {
		regionview->classmodel->views = g_slist_remove( 
			regionview->classmodel->views, regionview );
		regionview->classmodel = NULL;
	}

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

/* Compute the label geometry.
 */
static void
regionview_label_geo( Regionview *regionview )
{
	int n = regionview_label_border;
	int lbearing, rbearing, width, ascent, descent;
	const char *str = buf_all( &regionview->caption );
	GdkFont *font = gtk_widget_get_style( GTK_WIDGET( regionview ) )->font;

	gdk_text_extents( font, str, strlen( str ), 
		  &lbearing, &rbearing, &width, &ascent, &descent );
	regionview->label.width = rbearing + lbearing + 2 * n;
	regionview->label.height = ascent + descent + 2 * n;
	regionview->ascent = ascent;
}

static void
regionview_refresh_label( Regionview *regionview )
{
	if( regionview->classmodel ) {
		buf_rewind( &regionview->caption );
		row_qualified_name_relative( 
			SYMBOL( main_workspacegroup->current ),
			HEAPMODEL( regionview->classmodel )->row, 
			&regionview->caption );
		regionview_label_geo( regionview );
	}
}

/* Move label to try to keep it within the image, and away from the
 * selected pixels.
 */
static void
regionview_position_label( Regionview *regionview )
{
	Imagepresent *ip = regionview->ip;
	Conversion *conv = ip->id->conv;
	Rect *label = &regionview->label;
	const int b = regionview_border_width + 2;

	Rect dr, er;

	if( regionview->label_geo ) {
		regionview_refresh_label( regionview );
		regionview->label_geo = FALSE;
	}

	conversion_im_to_disp_rect( conv, &regionview->area, &dr );
	imagedisplay_disp_to_xev_rect( ip->id, &dr, &er );

	switch( regionview->type ) {
	case REGIONVIEW_REGION:
	case REGIONVIEW_AREA:
	case REGIONVIEW_BOX:
		if( er.top > label->height + b ) {
			/* Space above region for label.
			 */
			label->left = dr.left - b;
			label->top = dr.top - label->height - b;
		}
		else if( er.left > label->width + b ) {
			/* Space to left of region for label
			 */
			label->left = dr.left - label->width - b;
			label->top = dr.top - b;
		}
		else if( IM_RECT_RIGHT( &er ) < ip->id->screen.width - 
			label->width - b ) {
			/* Space at right.
			 */
			label->left = IM_RECT_RIGHT( &dr ) + b;
			label->top = dr.top - b;
		}
		else if( IM_RECT_BOTTOM( &er ) < ip->id->screen.height - 
			label->height - b ) {
			/* Space at bottom.
			 */
			label->left = dr.left - b;
			label->top = IM_RECT_BOTTOM( &dr ) + b;
		}
		else {
			/* Inside top left.
			 */
			label->left = dr.left;
			label->top = dr.top;
		}
		break;

	case REGIONVIEW_HGUIDE:
	case REGIONVIEW_VGUIDE:
	case REGIONVIEW_POINT:
	case REGIONVIEW_ARROW:
	case REGIONVIEW_LINE:
		/* Space above? 
		 */
		if( er.top > label->height + b/2 + 2 ) {
			if( er.left > ip->id->screen.width - 
				label->width - b/2 - 2 ) {
				/* Above left.
				 */
				label->left = dr.left - b/2 - 2 - label->width;
				label->top = dr.top - b/2 - 2 - label->height;
			}
			else {
				/* Above right.
				 */
				label->left = dr.left + b/2 + 2;
				label->top = dr.top - b/2 - 2 - label->height;
			}
		}
		else if( er.left > ip->id->screen.width - 
			label->width - b/2 - 2 ) {
			/* Below left.
			 */
			label->left = dr.left - b/2 - 2 - label->width;
			label->top = dr.top + b/2 + 2;
		}
		else {
			/* Below right.
			 */
			label->left = dr.left + b/2 + 2;
			label->top = dr.top + b/2 + 2;
		}
		break;

	default:
		assert( FALSE );
	}
}

static Rect *
regionview_get_model( Regionview *regionview )
{
	Classmodel *classmodel = regionview->classmodel;
	Rect *model_area;

        /* If we have a class, update from the inside of that.
         */
        if( classmodel ) {
                IregionInstance *instance =
                        classmodel_get_instance( classmodel );

                model_area = &instance->area;
        }
        else
                model_area = regionview->model_area;

	return( model_area );
}

/* Update our_area from the model. Translate to our cods too: we always have
 * x/y in 0 to xsize/ysize.
 */
static void
regionview_update_from_model( Regionview *regionview )
{
	Imagepresent *ip = regionview->ip;
	Conversion *conv = ip->id->conv;
	IMAGE *im = imageinfo_get( FALSE, conv->ii );
	Rect *model_area = regionview_get_model( regionview );

	regionview->our_area = *model_area;
	regionview->our_area.left += im->Xoffset;
	regionview->our_area.top += im->Yoffset;
}

/* Update the model from our_area.
 */
static void
regionview_model_update( Regionview *regionview )
{
	Classmodel *classmodel = regionview->classmodel;
	Imagepresent *ip = regionview->ip;
	Conversion *conv = ip->id->conv;
	IMAGE *im = imageinfo_get( FALSE, conv->ii );
	Rect *model_area = regionview_get_model( regionview );

	*model_area = regionview->our_area;
	model_area->left -= im->Xoffset;
	model_area->top -= im->Yoffset;

	if( classmodel ) {
		classmodel_update( classmodel );

		if( CALC_RECOMP_REGION )
			symbol_recalculate_all();
	}

	view_refresh( VIEW( regionview ) );
}

/* Our model has changed ... undraw in the old position, draw in the new
 * position.
 */
static void 
regionview_refresh( View *view )
{
	Regionview *regionview = REGIONVIEW( view );

#ifdef DEBUG
	printf( "regionview_refresh1: %dx%d size %dx%d\n",
		regionview->our_area.left, regionview->our_area.top, 
		regionview->our_area.width, regionview->our_area.height );
#endif /*DEBUG*/

	/* Update our_area from model.
	 */
	regionview_update_from_model( regionview );

#ifdef DEBUG
	printf( "regionview_refresh2: %dx%d size %dx%d\n",
		regionview->our_area.left, regionview->our_area.top, 
		regionview->our_area.width, regionview->our_area.height );
#endif /*DEBUG*/

	/* Mark the old area for repainting with the background.
	 */
	if( !regionview->first ) 
		regionview_paint_clipped( regionview, 
			regionview_unpaint, &regionview->ip->id->screen );
	regionview->first = FALSE;

	/* Set new position. 
	 */
	regionview->area = regionview->our_area;
	regionview->last_paint_state = regionview->paint_state;
	regionview->last_type = regionview->type;

	/* Choose a new label position.
	 */
	regionview_position_label( regionview );

	/* Draw in the new place, clip against imagedisplay draw area.
	 */
	regionview_paint_clipped( regionview, 
		regionview_paint, &regionview->ip->id->screen );

	VIEW_CLASS( parent_class )->refresh( view );
}

static void
regionview_edit_cb( GtkWidget *menu, Regionview *regionview, Imagepresent *ip )
{
	model_edit( GTK_WIDGET( ip ), MODEL( regionview->classmodel ) );
}

static void
regionview_clone_cb( GtkWidget *menu, Regionview *regionview, Imagepresent *ip )
{
	Row *row = HEAPMODEL( regionview->classmodel )->row;
	Workspace *ws = row->top_col->ws;

	if( row->top_row != row ) {
		box_info( GTK_WIDGET( regionview->ip ),
			"you can only clone top level regions" );
		return;
	}

        workspace_deselect_all( ws );
        row_select( row );
        if( !workspace_clone_selected( ws ) )
                box_alert( NULL );
        workspace_deselect_all( ws );

        symbol_recalculate_all();
}

static void
regionview_clear_edited_cb( GtkWidget *menu, 
	Regionview *regionview, Imagepresent *ip )
{
	(void) model_map_all( MODEL( regionview->classmodel ),
		(model_map_fn) model_clear_edited, NULL );
        symbol_recalculate_all();
}

static void
regionview_remove_yes( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	Regionview *regionview = REGIONVIEW( client );
	Row *row = HEAPMODEL( regionview->classmodel )->row;

	gtk_object_destroy( GTK_OBJECT( row->sym ) );

	nfn( sys, IWINDOW_TRUE );
}

static void
regionview_remove_cb( GtkWidget *menu, 
	Regionview *regionview, Imagepresent *ip )
{
	Row *row = HEAPMODEL( regionview->classmodel )->row;

	if( row->top_row != row ) {
		box_info( GTK_WIDGET( regionview->ip ),
			"you can only clone top level regions" );
		return;
	}

	box_yesno( GTK_WIDGET( ip ),
		regionview_remove_yes, iwindow_true_cb, regionview,
		iwindow_notify_null, NULL,
		"Remove",
		"are you sure you want to remove region \"%s\"?",
		buf_all( &regionview->caption ) );
}

static void
regionview_class_init( RegionviewClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	ViewClass *view_class = (ViewClass *) klass;

	GtkWidget *pane;

	parent_class = gtk_type_class( TYPE_VIEW );

	object_class->destroy = regionview_destroy;

	/* Create signals.
	 */

	/* Init methods.
	 */
	view_class->refresh = regionview_refresh;

        /* Other init.
         */
	pane = regionview_popup_menu = popup_build( "Region menu" );
	popup_add_but( pane, "Edit ...", regionview_edit_cb );
	popup_add_but( pane, "Clone", regionview_clone_cb );
	popup_add_but( pane, "Reset", regionview_clear_edited_cb );
	menu_add_sep( pane );
	popup_add_but( pane, "Remove ...", regionview_remove_cb );
}

static void
regionview_init( Regionview *regionview )
{
	static Rect empty_rect = { -1, -1, -1, -1 };

#ifdef DEBUG_MAKE
	printf( "regionview_init\n" );
#endif /*DEBUG_MAKE*/

	regionview->type = REGIONVIEW_POINT;
	regionview->frozen = TRUE;

	regionview->state = REGIONVIEW_WAIT;
	regionview->shape = IWINDOW_SHAPE_NONE;
	regionview->dx = -1;
	regionview->dy = -1;
	regionview->grabbed = FALSE;

	regionview->classmodel = NULL;
	regionview->ip = NULL;
	regionview->cntxt = NULL;
	regionview->expose_start_sid = 0;
	regionview->repaint_sid = 0;
	regionview->destroy_sid = 0;
	regionview->event_sid = 0;
	regionview->changed_sid = 0;
	regionview->conv_destroy_sid = 0;
	regionview->ws_changed_sid = 0;

	regionview->model_area = NULL;
	regionview->paint_state = GTK_STATE_NORMAL;

	regionview->unpainting = FALSE;
	regionview->area = empty_rect;
	regionview->label = empty_rect;
	regionview->click = 0;
	regionview->ascent = 0;
	regionview->dash_offset = 0;
	regionview->dash_crawl = 0;
	regionview->last_paint_state = (GtkStateType) -1;
	regionview->last_type = (RegionviewType) -1;
	regionview->first = TRUE;
	regionview->label_geo = TRUE;

	buf_init_dynamic( &regionview->caption, REGIONVIEW_LABEL_MAX );

	set_name( GTK_WIDGET( regionview ), "regionview_widget" );
}

GtkType
regionview_get_type( void )
{
	static GtkType regionview_type = 0;

	if( !regionview_type ) {
		static const GtkTypeInfo info = {
			"Regionview",
			sizeof( Regionview ),
			sizeof( RegionviewClass ),
			(GtkClassInitFunc) regionview_class_init,
			(GtkObjectInitFunc) regionview_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		regionview_type = gtk_type_unique( TYPE_VIEW, &info );
	}

	return( regionview_type );
}

/* Test for rect touches rect (non-empty intersection).
 */
static void *
regionview_rect_touching( Regionview *regionview, Rect *a, Rect *b )
{
	Rect overlap;

	im_rect_intersectrect( a, b, &overlap );

	if( !im_rect_isempty( &overlap ) )
		return( regionview );
	else
		return( NULL );
}

/* Test for rect intersects some part of region.
 */
static gboolean
regionview_rect_touches_region( Regionview *regionview, Rect *expose )
{
	Conversion *conv = regionview->ip->id->conv;

	Rect canvas_area, tiny;
	int x, y;

	if( regionview->classmodel && regionview_rect_touching( regionview, 
		&regionview->label, expose ) )
		return( TRUE );

	switch( regionview->type ) {
	case REGIONVIEW_REGION:
	case REGIONVIEW_AREA:
	case REGIONVIEW_BOX:
	case REGIONVIEW_LINE:
		conversion_im_to_disp_rect( conv, 
			&regionview->area, &canvas_area );
		if( regionview_border_foreach( regionview, &canvas_area, 
			(regionview_rect_fn) regionview_rect_touching, 
			expose ) )
			return( TRUE );
		break;

	case REGIONVIEW_POINT:
		conversion_im_to_disp( conv, 
			regionview->area.left, regionview->area.top, &x, &y );
		if( regionview_crosshair_foreach( regionview, x, y, 
			(regionview_rect_fn) regionview_rect_touching, 
			expose ) )
			return( TRUE );
		break;

	case REGIONVIEW_ARROW:
		conversion_im_to_disp_rect( conv, 
			&regionview->area, &canvas_area );
		if( regionview_crosshair_foreach( regionview, 
			canvas_area.left, canvas_area.top,
			(regionview_rect_fn) regionview_rect_touching, 
			expose ) )
			return( TRUE );
		if( regionview_crosshair_foreach( regionview, 
			IM_RECT_RIGHT( &canvas_area ), 
			IM_RECT_BOTTOM( &canvas_area ),
			(regionview_rect_fn) regionview_rect_touching, 
			expose ) )
			return( TRUE );

		/* Spot in main area too ... for the dotted line. Also avoid
		 * zero-width/height areas for h and v lines.
		 */
		im_rect_normalise( &canvas_area );
		im_rect_marginadjust( &canvas_area, 1 );
		if( regionview_rect_touching( regionview, 
			&canvas_area, expose ) )
			return( TRUE );

		/* ... and the centre of the crosshairs.
		 */
		tiny.left = canvas_area.left;
		tiny.top = canvas_area.top;
		tiny.width = 1;
		tiny.height = 1;
		im_rect_marginadjust( &tiny, regionview_border_width );
		if( regionview_rect_touching( regionview, &tiny, expose ) )
			return( TRUE );

		tiny.left = IM_RECT_RIGHT( &canvas_area );
		tiny.top = IM_RECT_BOTTOM( &canvas_area );
		tiny.width = 1;
		tiny.height = 1;
		im_rect_marginadjust( &tiny, regionview_border_width );
		if( regionview_rect_touching( regionview, &tiny, expose ) )
			return( TRUE );

		break;

	case REGIONVIEW_HGUIDE:
	case REGIONVIEW_VGUIDE:
		conversion_im_to_disp_rect( conv, 
			&regionview->area, &canvas_area );
		im_rect_marginadjust( &canvas_area, 1 );
		if( regionview_rect_touching( regionview, 
			&canvas_area, expose ) )
			return( TRUE );

		break;

	default:
		assert( FALSE );
	}

	return( FALSE );
}

/* Here from async.c repaint stuff.
 */
static void
regionview_expose( Regionview *regionview, Rect *expose )
{
	Imagedisplay *id = regionview->ip->id;
	Rect clip, clip2;

	assert( expose->width >= 0 && expose->height >= 0 );

	/* Async repaints disabled? (we're doing a sync update)
	 */
	if( regionview->unpainting )
		return;

	/* If the expose doesn't touch the region, don't bother painting.
	 */
	if( !regionview_rect_touches_region( regionview, expose ) )
		return;

	/* If we've not finished init, don't paint.
	 */
	if( regionview->first )
		return;

	/* Make sure we don't paint over the focus indicator etc.
	 */
	imagedisplay_disp_to_xev_rect( id, expose, &clip );
	im_rect_intersectrect( &clip, &id->screen, &clip2 );
	regionview_paint_clipped( regionview, regionview_paint, &clip2 );
}

static void
regionview_expose_start( Imagedisplay *id, Rect *expose, 
	Regionview *regionview )
{
	regionview_expose( regionview, expose );
}

static void
regionview_repaint( Imagedisplay *id, Rect *expose, 
	Regionview *regionview )
{
	regionview_expose( regionview, expose );
}

/* Test for point is in the grab area of a region border or label.
 */
static gboolean
regionview_point_in_region( Regionview *regionview, int x, int y )
{
	Rect r;

	r.left = x;
	r.top = y;
	r.width = 1;
	r.height = 1;
	return( regionview_rect_touches_region( regionview, &r ) );
}

/* Given a position, find the sort of cursor we should show.
 */
static iWindowShape
regionview_find_shape( Regionview *regionview, int x, int y )
{
	Conversion *conv = regionview->ip->id->conv;

	Rect canvas_area, tiny;
	int dx, dy; 

	if( im_rect_includespoint( &regionview->label, x, y ) )
		return( IWINDOW_SHAPE_EDIT );

	conversion_im_to_disp_rect( conv, &regionview->area, &canvas_area );
	dx = x - canvas_area.left;
	dy = y - canvas_area.top;

	switch( regionview->type ) {
	case REGIONVIEW_REGION:
		if( dx > canvas_area.width - 10 ) {
			if( dy > canvas_area.height - 10 )
				return( IWINDOW_SHAPE_BOTTOMRIGHT );
			else if( dy < 10 )
				return( IWINDOW_SHAPE_TOPRIGHT );
			else 
				return( IWINDOW_SHAPE_RIGHT );
		}
		else if( dx < 10 ) {
			if( dy > canvas_area.height - 10 )
				return( IWINDOW_SHAPE_BOTTOMLEFT );
			else if( dy < 10 )
				return( IWINDOW_SHAPE_TOPLEFT );
			else 
				return( IWINDOW_SHAPE_LEFT );
		}
		else {
			if( dy < canvas_area.height / 2 )
				return( IWINDOW_SHAPE_TOP );
			else
				return( IWINDOW_SHAPE_BOTTOM );
		}
		break;

	case REGIONVIEW_POINT:
	case REGIONVIEW_AREA:
		return( IWINDOW_SHAPE_MOVE );

	case REGIONVIEW_ARROW:
		tiny.left = x;
		tiny.top = y;
		tiny.width = 1;
		tiny.height = 1;
		if( regionview_crosshair_foreach( regionview,
			canvas_area.left, canvas_area.top,
			(regionview_rect_fn) regionview_rect_touching, &tiny ) )
			return( IWINDOW_SHAPE_TOPLEFT );
		if( regionview_crosshair_foreach( regionview,
			IM_RECT_RIGHT( &canvas_area ), 
			IM_RECT_BOTTOM( &canvas_area ),
			(regionview_rect_fn) regionview_rect_touching, &tiny ) )
			return( IWINDOW_SHAPE_BOTTOMRIGHT );

		/* Extra tests ... allow grabs in the centre of the crosshairs
		 * too.
		 */
		tiny.left = IM_RECT_RIGHT( &canvas_area );
		tiny.top = IM_RECT_BOTTOM( &canvas_area );
		tiny.width = 1;
		tiny.height = 1;
		im_rect_marginadjust( &tiny, regionview_border_width );
		if( im_rect_includespoint( &tiny, x, y ) )
			return( IWINDOW_SHAPE_BOTTOMRIGHT );

		tiny.left = canvas_area.left;
		tiny.top = canvas_area.top;
		tiny.width = 1;
		tiny.height = 1;
		im_rect_marginadjust( &tiny, regionview_border_width );
		if( im_rect_includespoint( &tiny, x, y ) )
			return( IWINDOW_SHAPE_TOPLEFT );

		break;

	case REGIONVIEW_HGUIDE:
		im_rect_marginadjust( &canvas_area, 1 );
		if( im_rect_includespoint( &canvas_area, x, y ) )
			return( IWINDOW_SHAPE_MOVE );

		break;

	case REGIONVIEW_VGUIDE:
		im_rect_marginadjust( &canvas_area, 1 );
		if( im_rect_includespoint( &canvas_area, x, y ) )
			return( IWINDOW_SHAPE_MOVE );

		break;

	case REGIONVIEW_BOX:
	case REGIONVIEW_LINE:
		break;

	default:
		assert( FALSE );
	}

	return( IWINDOW_SHAPE_NONE );
}

/* Right button press event.
 */
static gint
regionview_right_press( Regionview *regionview, GdkEvent *ev, int x, int y )
{
	if( im_rect_includespoint( &regionview->label, x, y ) ) {
		gtk_signal_emit_stop_by_name( 
			GTK_OBJECT( regionview->ip->id ), "event" );
		popup_show( GTK_WIDGET( regionview ), ev );

		return( TRUE );
	}

	return( FALSE );
}

/* Get ready to track this region. See imagepresent.c.
 */
void
regionview_attach( Regionview *regionview )
{
	switch( regionview->shape ) {
	case IWINDOW_SHAPE_NONE:
		regionview->shape = IWINDOW_SHAPE_BOTTOMRIGHT;
		regionview->state = REGIONVIEW_RESIZE;
		break;

	case IWINDOW_SHAPE_MOVE:
	case IWINDOW_SHAPE_EDIT:
		regionview->state = REGIONVIEW_MOVE;
		break;

	default:
		regionview->state = REGIONVIEW_RESIZE;
		break;
	}

	regionview->paint_state = GTK_STATE_ACTIVE;
}

/* Stop tracking.
 */
void
regionview_detach( Regionview *regionview )
{
	regionview->state = REGIONVIEW_WAIT;
	regionview->paint_state = GTK_STATE_PRELIGHT;
}

/* Lock the pointer to this regionview ... for tracking resize/move etc.
 */
static void
regionview_grab( Regionview *regionview, GdkEvent *ev )
{
	Conversion *conv = regionview->ip->id->conv;
	Rect *area = &regionview->area;
	int x, y;
	int dx, dy;

	assert( !regionview->grabbed );

#ifdef DEBUG_GRAB
	printf( "regionview_grab: grabbing\n" );
#endif /*DEBUG_GRAB*/

	iwindow_cursor_context_set_cursor( regionview->cntxt, 
		regionview->shape );

	imagedisplay_xev_to_disp( regionview->ip->id, 
		ev->button.x, ev->button.y, &x, &y );

	regionview->grabbed = TRUE;

	conversion_im_to_disp( conv, area->left, area->top, &dx, &dy );
	regionview->dx = dx - x;
	regionview->dy = dy - y;

#ifdef GRAB
	gdk_pointer_grab( GTK_WIDGET( regionview->ip->id )->window,
		FALSE,
		GDK_BUTTON1_MOTION_MASK |
		GDK_POINTER_MOTION_MASK |
		GDK_POINTER_MOTION_HINT_MASK |
		GDK_BUTTON_RELEASE_MASK,
		NULL, NULL, ev->button.time );
	gtk_grab_add( GTK_WIDGET( regionview->ip->id ) );
#else /*GRAB*/
	printf( "regionview_grab: true grabs disabled\n" );
#endif /*GRAB*/
}

static gint
regionview_doubleclick_cb( Regionview *regionview )
{
	regionview->click = 0;

	/* Stop timer.
	 */
	return( FALSE );
}

/* Left button press event.
 */
static gint
regionview_left_press( Regionview *regionview, GdkEvent *ev, int x, int y )
{
	if( !regionview_point_in_region( regionview, x, y ) ) 
		return( FALSE );

	switch( regionview->state ) {
	case REGIONVIEW_WAIT:
		regionview->shape = regionview_find_shape( regionview, x, y );

		if( regionview->shape != IWINDOW_SHAPE_NONE ) 
			regionview_attach( regionview );

		/* Spot doubleclick on label.
		 */
		if( regionview->shape == IWINDOW_SHAPE_EDIT ) {
			if( regionview->click ) {
				/* Already a timer running ... must be a
				 * doubleclick!
				 */
				FREEFI( gtk_timeout_remove, regionview->click );
				model_edit( GTK_WIDGET( regionview->ip ), 
					MODEL( regionview->classmodel ) );
			}

			regionview->click = gtk_timeout_add( 200, 
				(GtkFunction) regionview_doubleclick_cb, 
				regionview );
		}

		if( regionview->state != REGIONVIEW_WAIT ) {
			gtk_signal_emit_stop_by_name( 
				GTK_OBJECT( regionview->ip->id ), "event" );

			return( TRUE );
		}
		break;

	case REGIONVIEW_MOVE:
	case REGIONVIEW_RESIZE:
		break;

	default:
		assert( FALSE );
	}

	return( FALSE );
}

/* Left button release event.
 */
static gint
regionview_left_release( Regionview *regionview, GdkEvent *ev )
{
	switch( regionview->state ) {
	case REGIONVIEW_WAIT:
		break;

	case REGIONVIEW_MOVE:
	case REGIONVIEW_RESIZE:
		regionview_detach( regionview );

		if( !CALC_RECOMP_REGION )
			symbol_recalculate_all();

		break;
	}

	return( FALSE );
}

static void
regionview_resize_area( Regionview *regionview, int ix, int iy )
{
	Imagepresent *ip = regionview->ip;
	Conversion *conv = ip->id->conv;
	IMAGE *im = imageinfo_get( FALSE, conv->ii );
	Rect *our_area = &regionview->our_area;
	const int th = regionview_morph_threshold;

	int bot = our_area->top + our_area->height;
	int ri = our_area->left + our_area->width;
	int rx = ix - our_area->left;
	int ry = iy - our_area->top;

	/* If we're not frozen, do an unconstrained resize.
	 */
	if( !regionview->frozen ) {
		switch( regionview->shape ) {
		case IWINDOW_SHAPE_RIGHT:
			our_area->width = IM_CLIP( -our_area->left,
				rx, im->Xsize - our_area->left );
			break;

		case IWINDOW_SHAPE_BOTTOM:
			our_area->height = IM_CLIP( -our_area->top, 
				ry, im->Ysize - our_area->top );
			break;

		case IWINDOW_SHAPE_MOVE:
			/* Get this for POINT on create ... treat as 
			 * BOTTOMRIGHT.
			 */
		case IWINDOW_SHAPE_BOTTOMRIGHT:
			our_area->width = IM_CLIP( -our_area->left,
				rx, im->Xsize - our_area->left );
			our_area->height = IM_CLIP( -our_area->top, 
				ry, im->Ysize - our_area->top );
			break;

		case IWINDOW_SHAPE_LEFT:
			our_area->left = IM_CLIP( 0, ix, im->Xsize - 1 );
			our_area->width = ri - our_area->left;
			break;

		case IWINDOW_SHAPE_TOP:
			our_area->top = IM_CLIP( 0, iy, im->Ysize - 1 );
			our_area->height = bot - our_area->top;
			break;

		case IWINDOW_SHAPE_TOPLEFT:
			our_area->top = IM_CLIP( 0, iy, im->Ysize - 1 );
			our_area->left = IM_CLIP( 0, ix, im->Xsize - 1 );
			our_area->width = ri - our_area->left;
			our_area->height = bot - our_area->top;
			break;

		case IWINDOW_SHAPE_TOPRIGHT:
			our_area->top = IM_CLIP( 0, iy, im->Ysize - 1 );
			our_area->height = bot - our_area->top;
			our_area->width = IM_CLIP( -our_area->left,
				rx, im->Xsize - our_area->left );
			break;

		case IWINDOW_SHAPE_BOTTOMLEFT:
			our_area->left = IM_CLIP( 0, ix, im->Xsize - 1 );
			our_area->width = ri - our_area->left;
			our_area->height = IM_CLIP( -our_area->top, 
				ry, im->Ysize - our_area->top );
			break;

		default:
			assert( FALSE );
		}

		if( abs( our_area->width ) < th && 
			abs( our_area->height - im->Ysize ) < th )
			regionview->type = REGIONVIEW_VGUIDE;
		else if( abs( our_area->height ) < th && 
			abs( our_area->width - im->Xsize ) < th ) 
			regionview->type = REGIONVIEW_HGUIDE;
		else if( abs( our_area->width ) < th && 
			abs( our_area->height ) < th ) 
			regionview->type = REGIONVIEW_POINT;
		else if( our_area->width > 0 && 
			our_area->height > 0 ) 
			regionview->type = REGIONVIEW_REGION;
		else
			regionview->type = REGIONVIEW_ARROW;
	}
	else {
		/* We're frozen ... resize should be tightly constrained.
		 */
		switch( regionview->type ) {
		case REGIONVIEW_REGION:
			switch( regionview->shape ) {
			case IWINDOW_SHAPE_RIGHT:
				our_area->width = IM_CLIP( 1, rx, 
					im->Xsize - our_area->left );
				break;

			case IWINDOW_SHAPE_BOTTOM:
				our_area->height = IM_CLIP( 1, ry, 
					im->Ysize - our_area->top );
				break;

			case IWINDOW_SHAPE_BOTTOMRIGHT:
				our_area->width = IM_CLIP( 1, rx, 
					im->Xsize - our_area->left );
				our_area->height = IM_CLIP( 1, ry, 
					im->Ysize - our_area->top );
				break;

			case IWINDOW_SHAPE_TOP:
				our_area->top = IM_CLIP( 0, iy, bot - 1 );
				our_area->height = bot - our_area->top;
				break;

			case IWINDOW_SHAPE_LEFT:
				our_area->left = IM_CLIP( 0, ix, ri - 1 );
				our_area->width = ri - our_area->left;
				break;

			case IWINDOW_SHAPE_TOPLEFT:
				our_area->left = IM_CLIP( 0, ix, ri - 1 );
				our_area->width = ri - our_area->left;
				our_area->top = IM_CLIP( 0, iy, bot - 1 );
				our_area->height = bot - our_area->top;
				break;

			case IWINDOW_SHAPE_TOPRIGHT:
				our_area->top = IM_CLIP( 0, iy, bot - 1 );
				our_area->height = bot - our_area->top;
				our_area->width = IM_CLIP( 1, rx, 
					im->Xsize - our_area->left );
				break;

			case IWINDOW_SHAPE_BOTTOMLEFT:
				our_area->left = IM_CLIP( 0, ix, ri - 1 );
				our_area->width = ri - our_area->left;
				our_area->height = IM_CLIP( 1, ry, 
					im->Ysize - our_area->top );
				break;

			default:
				assert( FALSE );
			}
			break;

		case REGIONVIEW_ARROW:
		case REGIONVIEW_LINE:
		case REGIONVIEW_BOX:
			switch( regionview->shape ) {
			case IWINDOW_SHAPE_TOPLEFT:
				our_area->left = IM_CLIP( 0, ix, im->Xsize );
				our_area->width = ri - our_area->left;
				our_area->top = IM_CLIP( 0, iy, im->Ysize );
				our_area->height = bot - our_area->top;
				break;

			case IWINDOW_SHAPE_BOTTOMRIGHT:
				our_area->width = 
					IM_CLIP( -our_area->left, rx, 
						im->Xsize - our_area->left );
				our_area->height = 
					IM_CLIP( -our_area->top, ry, 
						im->Ysize - our_area->top );
				break;

			default:
				assert( FALSE );
			}
			break;

		case REGIONVIEW_POINT:
			our_area->left = IM_CLIP( 0, ix, im->Xsize - 1 );
			our_area->top = IM_CLIP( 0, iy, im->Ysize - 1 );
			our_area->width = 0;
			our_area->height = 0;
			break;

		case REGIONVIEW_HGUIDE:
			our_area->top = IM_CLIP( 0, iy, im->Ysize - 1 );
			break;

		case REGIONVIEW_VGUIDE:
			our_area->left = IM_CLIP( 0, ix, im->Xsize - 1 );
			break;

		default:
			assert( FALSE );
		}
	}
}

/* Change the state ... and refresh directly.
 */
static void
regionview_set_paint_state( Regionview *regionview, GtkStateType paint_state )
{
	if( regionview->paint_state != paint_state ) {
		regionview->paint_state = paint_state;
		view_refresh( VIEW( regionview ) );
	}
}

/* A motion event while we're grabbed.
 */
static void
regionview_motion_grab( Regionview *regionview, int x, int y )
{
	Imagepresent *ip = regionview->ip;
	Conversion *conv = ip->id->conv;
	Rect *visible = &ip->id->visible;
	Rect *our_area = &regionview->our_area;
	Rect snap;

	IMAGE *im;
	int ix, iy;

#ifdef DEBUG
	printf( "regionview_motion_grab:\n" );
	printf( "cods: %dx%d size %dx%d\n",
		regionview->model_area->left, regionview->model_area->top, 
		regionview->model_area->width, regionview->model_area->height );
#endif /*DEBUG*/

	switch( regionview->state ) {
	case REGIONVIEW_MOVE:
		conversion_disp_to_im( conv, 
			x + regionview->dx, y + regionview->dy, &ix, &iy );
		im = imageinfo_get( FALSE, conv->ii );

		switch( regionview->type ) {
		case REGIONVIEW_REGION:
		case REGIONVIEW_AREA:
			our_area->left = IM_CLIP( 0, ix, 
				im->Xsize - our_area->width );
			our_area->top = IM_CLIP( 0, iy, 
				im->Ysize - our_area->height );
			break;

		case REGIONVIEW_ARROW:
			our_area->left = IM_CLIP( 
				IM_MAX( 0, -our_area->width ), 
				ix, 
				IM_MIN( im->Xsize - 1, 
					im->Xsize - our_area->width ) );
			our_area->top = IM_CLIP( 
				IM_MAX( 0, -our_area->height ), 
				iy, 
				IM_MIN( im->Ysize - 1, 
					im->Ysize - our_area->height ) );
			break;

		case REGIONVIEW_POINT:
		case REGIONVIEW_HGUIDE:
		case REGIONVIEW_VGUIDE:
			our_area->left = IM_CLIP( 0, ix, 
				im->Xsize - our_area->width - 1 );
			our_area->top = IM_CLIP( 0, iy, 
				im->Ysize - our_area->height - 1 );
			break;

		case REGIONVIEW_LINE:
		case REGIONVIEW_BOX:
			our_area->left = ix; 
			our_area->top = iy;
			break;

		default:
			assert( FALSE );
		}

		snap = *our_area;
		conversion_im_to_disp_rect( conv, &snap, &snap );
		if( imagepresent_snap_rect( ip, &snap, &snap ) ) {
			conversion_disp_to_im_rect( conv, &snap, &snap );
			our_area->left = snap.left;
			our_area->top = snap.top;
		}

		regionview_model_update( regionview );

		break;

	case REGIONVIEW_RESIZE:
		imagepresent_snap_point( ip, x, y, &x, &y );
		conversion_disp_to_im( conv, x, y, &ix, &iy );
		regionview_resize_area( regionview, ix, iy );

		regionview_model_update( regionview );

		break;

	default:
		break;
	}

	if( !im_rect_includespoint( visible, x, y ) ) {
		int u, v;

		if( x < visible->left )
			u = -8;
		else if( x > IM_RECT_RIGHT( visible ) )
			u = 8;
		else
			u = 0;
		if( y < visible->top )
			v = -8;
		else if( y > IM_RECT_BOTTOM( visible ) )
			v = 8;
		else
			v = 0;

		imagepresent_scroll_start( regionview->ip, u, v );
	}
	else
		imagepresent_scroll_stop( regionview->ip );
}

/* Motion event.
 */
static gint
regionview_motion( Regionview *regionview, GdkEvent *ev, int x, int y )
{
	GdkWindow *win = GTK_WIDGET( regionview->ip->id )->window;

	iWindowShape shape;

	GdkEventMask old_mask;
	GdkModifierType mods;
	int ex, ey;

#ifdef EVENT
	printf( "regionview_motion: 0x%x\n", (unsigned int) regionview );
#endif /*EVENT*/

	switch( regionview->state ) {
	case REGIONVIEW_WAIT:
		if( regionview_point_in_region( regionview, x, y ) ) {
			shape = regionview_find_shape( regionview, x, y );
			iwindow_cursor_context_set_cursor( regionview->cntxt, 
				shape );
			regionview_set_paint_state( regionview, 
				GTK_STATE_PRELIGHT );
		}
		else {
			iwindow_cursor_context_set_cursor( regionview->cntxt, 
				IWINDOW_SHAPE_NONE );
			regionview_set_paint_state( regionview, 
				GTK_STATE_NORMAL );
		}

		break;

	case REGIONVIEW_MOVE:
	case REGIONVIEW_RESIZE:
		if( regionview->shape != IWINDOW_SHAPE_NONE && 
			!regionview->grabbed ) {
			int last_x = -1;
			int last_y = -1;

#ifdef DEBUG
			printf( "regionview_motion: 0x%x, starting track\n",
				(unsigned int) regionview );
#endif /*DEBUG*/

			old_mask = gdk_window_get_events( win );

			gdk_window_set_events( win, 
				GDK_POINTER_MOTION_HINT_MASK | 
				GDK_BUTTON_RELEASE_MASK ); 

			regionview_grab( regionview, ev );

			/* We want the regionview sticking around while we
			 * loop on it.
			 */
			gtk_object_ref( GTK_OBJECT( regionview ) );

			for(;;) {
				gdk_window_get_pointer( win, &ex, &ey, &mods );
				if( !(mods & GDK_BUTTON1_MASK) )
					break;

				if( ex != last_x || ey != last_y ) {
					last_x = ex;
					last_y = ey;

					imagedisplay_xev_to_disp( 
						regionview->ip->id, 
						ex, ey, &x, &y );

					regionview_motion_grab( regionview, 
						x, y );
				}

				while( g_main_iteration( FALSE ) )
					;

				/* We may miss a release + press event with
				 * the gdk_window_get_pointer() mods test
				 * above, which means we might have destroyed 
				 * the regionview.
				 */
				if( GTK_OBJECT_DESTROYED( regionview ) ) {
#ifdef DEBUG_MAKE
	printf( "regionview_motion: early exit!\n" );
#endif /*DEBUG_MAKE*/

					break;
				}
			}

			regionview_ungrab( regionview, ev->button.time );

			gdk_window_set_events( win, old_mask );

#ifdef DEBUG
			printf( "regionview_motion: 0x%x, done track\n",
				(unsigned int) regionview );
#endif /*DEBUG*/

			gtk_object_unref( GTK_OBJECT( regionview ) );
		}

		break;

	default:
		assert( FALSE );
	}

	return( FALSE );
}

/* Main event loop.
 */
static gint
regionview_event( GtkWidget *widget, GdkEvent *ev, Regionview *regionview )
{
	Imagepresent *ip = regionview->ip;
	Imagedisplay *id = ip->id;

	int x, y;

#ifdef EVENT
	if( ev->type == GDK_BUTTON_PRESS )
		printf( "regionview_event: GDK_BUTTON_PRESS\n" );
	if( ev->type == GDK_MOTION_NOTIFY )
		printf( "regionview_event: GDK_MOTION_NOTIFY\n" );
#endif /*EVENT*/

	/* Only manipulate regions if we're in SELECT mode ... don't want to
	 * drag regions while we're panning, for example. Exception ... we can
	 * drag/resize floating regions any time.
	 */
	if( ip->state != IMAGEPRESENT_STATE_SELECT && 
		regionview->classmodel )
		return( FALSE );

	switch( ev->type ) {
	case GDK_BUTTON_PRESS:
		imagedisplay_xev_to_disp( id, 
			ev->button.x, ev->button.y, &x, &y );

		switch( ev->button.button ) {
		case 1:
			return( regionview_left_press( regionview, ev, x, y ) );

		case 3:
			return( regionview_right_press( 
				regionview, ev, x, y ) );

		default:
			break;
		}

		break;

	case GDK_BUTTON_RELEASE:
		switch( ev->button.button ) {
		case 1:
			return( regionview_left_release( regionview, ev ) );

		default:
			break;
		}

		break;

	case GDK_MOTION_NOTIFY:
		imagedisplay_xev_to_disp( id, 
			ev->motion.x, ev->motion.y, &x, &y );

		return( regionview_motion( regionview, ev, x, y ) );

	default:
		break;
	}

	return( FALSE );
}

/* The conversion on our image has changed ... eg. on zoom in/out we need to
 * rethink the label position.
 */
static void
regionview_changed( Model *model, Regionview *regionview )
{
#ifdef DEBUG
	printf( "regionview_changed\n" );
#endif /*DEBUG*/

	view_refresh( VIEW( regionview ) );
}

/* The conversion on our image has been destroyed ... make sure we won't try
 * to disconnect when we go too.
 */
static void
regionview_conv_destroy( Model *model, Regionview *regionview )
{
	regionview->changed_sid = 0;
	regionview->conv_destroy_sid = 0;
}

/* The main workspacegroup has changed ... prolly new current workspace.
 */
static void
regionview_workspace_changed_cb( Workspacegroup *wsg, Regionview *regionview )
{
	regionview->label_geo = TRUE;
	view_refresh( VIEW( regionview ) );
}

static gint
regionview_dash_crawl_cb( Regionview *regionview )
{
	/* Don't dash crawl on winders, gdk does not support dashed lines and
	 * it flickers horribly.
	 */
#ifndef HAVE_WINDOWS_H
	/* Don't for regions, areas and points ... no lines in 'em.
	 */
	if( regionview->type != REGIONVIEW_REGION &&
		regionview->type != REGIONVIEW_POINT &&
		regionview->type != REGIONVIEW_AREA ) {
		regionview_paint_clipped( regionview, 
			regionview_unpaint, &regionview->ip->id->screen );
		regionview->dash_offset += 3;
		regionview_paint_clipped( regionview, 
			regionview_paint, &regionview->ip->id->screen );
	}
#endif /*HAVE_WINDOWS_H*/

	return( TRUE );
}

static void
regionview_setup( Regionview *regionview, 
	Classmodel *classmodel, Rect *model_area, Imagepresent *ip )
{
	iWindow *iwnd;

	regionview->classmodel = classmodel;
	regionview->ip = ip;
	regionview->model_area = model_area;
	regionview->our_area = *model_area;
	regionview->model_changed_sid = 0;
	ip->regionviews = g_slist_prepend( ip->regionviews, regionview );

	if( classmodel ) {
		classmodel->views = g_slist_prepend( classmodel->views, 
			regionview );

		regionview->model_changed_sid = gtk_signal_connect_object( 
			GTK_OBJECT( classmodel ), "changed",
			GTK_SIGNAL_FUNC( view_refresh ), 
			GTK_OBJECT( regionview ) );
	}

	regionview->expose_start_sid = gtk_signal_connect( 
		GTK_OBJECT( ip->id ), "expose_start", 
		GTK_SIGNAL_FUNC( regionview_expose_start ), regionview );
	regionview->repaint_sid = gtk_signal_connect( 
		GTK_OBJECT( ip->id ), "repaint", 
		GTK_SIGNAL_FUNC( regionview_repaint ), regionview );
	regionview->destroy_sid = gtk_signal_connect_object( 
		GTK_OBJECT( ip->id ), "destroy", 
		GTK_SIGNAL_FUNC( gtk_widget_destroy ), 
		GTK_OBJECT( regionview ) );
	regionview->event_sid = gtk_signal_connect( 
		GTK_OBJECT( ip->id ), "event",
		GTK_SIGNAL_FUNC( regionview_event ), regionview );
	regionview->changed_sid = gtk_signal_connect( 
		GTK_OBJECT( ip->id->conv ), "changed",
		GTK_SIGNAL_FUNC( regionview_changed ), regionview );
	regionview->conv_destroy_sid = gtk_signal_connect( 
		GTK_OBJECT( ip->id->conv ), "destroy",
		GTK_SIGNAL_FUNC( regionview_conv_destroy ), regionview );
	regionview->ws_changed_sid = gtk_signal_connect( 
		GTK_OBJECT( main_workspacegroup ), "changed", 
		GTK_SIGNAL_FUNC( regionview_workspace_changed_cb ), 
		regionview );

	iwnd = IWINDOW( gtk_widget_get_toplevel( GTK_WIDGET( ip ) ) );	
	regionview->cntxt = iwindow_cursor_context_new( iwnd );

	popup_link( GTK_WIDGET( regionview ), regionview_popup_menu, ip );

	regionview->dash_crawl = gtk_timeout_add( 200, 
		(GtkFunction) regionview_dash_crawl_cb, regionview );
}

Regionview *
regionview_new( Classmodel *classmodel, Rect *model_area, Imagepresent *ip )
{
	Regionview *regionview = gtk_type_new( TYPE_REGIONVIEW );

	regionview_setup( regionview, classmodel, model_area, ip );

#ifdef DEBUG
	printf( "regionview_new: %dx%d size %dx%d\n",
		model_area->left, model_area->top, 
		model_area->width, model_area->height );
#endif /*DEBUG*/

	return( regionview );
}

/* Type we display for each of the classes. Order is important!
 */
typedef struct {
	const char *name;
	RegionviewType type;
} RegionviewDisplay;

static RegionviewDisplay regionview_display_table[] = {
	{ CLASS_HGUIDE, REGIONVIEW_HGUIDE },
	{ CLASS_VGUIDE, REGIONVIEW_VGUIDE },
	{ CLASS_POINT, REGIONVIEW_POINT },
	{ CLASS_AREA, REGIONVIEW_AREA },
	{ CLASS_REGION, REGIONVIEW_REGION },
	{ CLASS_ARROW, REGIONVIEW_ARROW }
};

/* Look at the class we are drawing, set the display type.
 */
void 
regionview_set_type( Regionview *regionview, PElement *root )
{
	gboolean result;
	int i;

	if( heap_isclass( root, &result ) && result )
		for( i = 0; i < IM_NUMBER( regionview_display_table ); i++ ) {
			const char *name = regionview_display_table[i].name;

			if( !heap_isinstanceof( name, root, &result ) )
				continue;
			if( result ) {
				regionview->type =
					regionview_display_table[i].type;
				view_refresh_queue( VIEW( regionview ) );
				break;
			}
		}
}
