/* A row in a workspace ... not a widget, part of subcolumn
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/* Mad detail.
#define DEBUG
 */

/* Making and removing links between rows.
#define DEBUG_LINK
 */

/* Time each row recomp.
#define DEBUG_TIME
 */

/* Trace create/destroy.
#define DEBUG_NEW
 */

/* Dirty/clean/error stuff.
#define DEBUG_DIRTY
 */

#include "ip.h"

#define POPUP_NEW_ROWS (watch_bool_get( "POPUP_NEW_ROWS", FALSE ))

static HeapmodelClass *parent_class = NULL;

static void *
row_map_all_sub( Model *model, row_map_fn fn, void *a, void *b, void *c )
{
	if( IS_ROW( model ) )
		return( fn( ROW( model ), a, b, c ) );

	return( NULL );
}

static void *
row_map_all( Row *row, row_map_fn fn, void *a, void *b, void *c )
{
	return( model_map4_all( MODEL( row ),
		(model_map4_fn) row_map_all_sub, (void *) fn, a, b, c ) );
}

const char *
row_name( Row *row )
{
	if( row->sym )
		return( MODEL( row->sym )->name );
	else
		return( MODEL( row )->name );
}

static Row *
row_get_parent( Row *row )
{
	return( HEAPMODEL( row )->row );
}

/* Make a fully-qualified name for a row's symbol ... walk back up the tally
 * hierarchy. eg. "A1.fred.x" ... produce a name which will find this row from
 * a local of context.
 */
void
row_qualified_name_relative( Symbol *context, Row *row, BufInfo *buf )
{
	if( GTK_OBJECT_DESTROYED( GTK_OBJECT( row ) ) )
		buf_appendf( buf, "dead-row (0x%x)", (unsigned int) row );
	else if( !row_get_parent( row ) )
		symbol_qualified_name_relative( context, row->sym, buf );
	else {
		/* Qualify our parents, then do us.
		 */
		row_qualified_name_relative( context, 
			row_get_parent( row ), buf );
		buf_appends( buf, "." );
		buf_appends( buf, row_name( row ) );
	}
}

/* Make a fully-qualified name for a row's symbol ... walk back up the tally
 * hierarchy. eg. "A1.fred.x" ... produce a name which will find this row from
 * a local of context.
 */
void
row_qualified_name( Row *row, BufInfo *buf )
{
	row_qualified_name_relative( SYMBOL( row->ws ), row, buf );
}

/* Convenience ... print a row name out, identifying by tally heirarchy.
 */
void *
row_name_print( Row *row )
{
	if( row ) {
		BufInfo buf;
		char txt[100];

		buf_init_static( &buf, txt, 100 );
		row_qualified_name( row, &buf );
		printf( "\"%s\"", buf_all( &buf ) );
	}
	else
		printf( "(null)" );

	return( NULL );
}

/* Mark a row as containing an error ... called from expr_error_set()
 * ... don't call this directly.
 */
void
row_error_set( Row *row )
{
	if( !row->err ) {
		Workspace *ws = row->ws;

		ws->errors = g_slist_prepend( ws->errors, row );
		row->err = TRUE;

#ifdef DEBUG_DIRTY
		printf( "row_error_set: " );
		row_name_print( row );
		printf( "\n" );
#endif /*DEBUG_DIRTY*/

		model_changed( MODEL( row ) );
	}
}

/* Clear error state ... called from expr_error_clear() ... don't call this
 * directly.
 */
void
row_error_clear( Row *row )
{
	if( row->err ) {
		Workspace *ws = row->ws;

		ws->errors = g_slist_remove( ws->errors, row );
		row->err = FALSE;

#ifdef DEBUG_DIRTY
		printf( "row_error_clear: " );
		row_name_print( row );
		printf( "\n" );
#endif /*DEBUG_DIRTY*/

		/* Mark our text modified to make sure we reparse and compile.
		 * The code may contain pointers to dead symbols.
		 */
		if( row->child_rhs && row->child_rhs->itext )
			heapmodel_set_modified( 
				HEAPMODEL( row->child_rhs->itext ), TRUE );

		model_changed( MODEL( row ) );
	}
}

static void *
row_dirty_clear( Row *row )
{
#ifdef DEBUG_DIRTY
{
	Row *top_row = row->top_row;

	if( row->dirty )
		assert( g_slist_find( top_row->recomp, row ) );
}
#endif /*DEBUG_DIRTY*/

	if( row->dirty ) {
		Row *top_row = row->top_row;

		row->dirty = FALSE;
		top_row->recomp = g_slist_remove( top_row->recomp, row );

#ifdef DEBUG_DIRTY
		printf( "row_dirty_clear: " );
		row_name_print( row );
		printf( "\n" );
#endif /*DEBUG_DIRTY*/

		model_changed( MODEL( row ) );
	}

	return( NULL );
}

/* Set a single row dirty.
 */
static void *
row_dirty_set_single( Row *row )
{
#ifdef DEBUG_DIRTY
{
	Row *top_row = row->top_row;

	if( row->dirty )
		assert( g_slist_find( top_row->recomp, row ) );
	if( !row->dirty )
		assert( !g_slist_find( top_row->recomp, row ) );
}
#endif /*DEBUG_DIRTY*/

	if( !row->dirty ) {
		Row *top_row = row->top_row;

		row->dirty = TRUE;
		top_row->recomp = g_slist_prepend( top_row->recomp, row );

		model_changed( MODEL( row ) );

#ifdef DEBUG_DIRTY
		printf( "row_dirty_set_single: " );
		row_name_print( row );
		printf( "\n" );
#endif /*DEBUG_DIRTY*/
	}

	return( NULL );
}

static void *
row_dirty_set_sub( Model *model )
{
	if( IS_ROW( model ) ) {
		Row *row = ROW( model );
		Rhs *rhs = row->child_rhs;

		assert( !rhs || IS_RHS( rhs ) );

		if( rhs && rhs->itext && ITEXT( rhs->itext )->edited )
			row_dirty_set_single( row );
		else if( rhs && rhs->graphic && 
			CLASSMODEL( rhs->graphic )->edited )
			row_dirty_set_single( row );
	}

	return( NULL );
}

/* When we mark a row dirty, we need to mark any subrows with non-default
 * values dirty too so that they will get a chance to reapply their edits over
 * the top of the new value we will make for this row.
 */
void *
row_dirty_set( Row *row )
{
	row_dirty_set_single( row );

	return( model_map_all( MODEL( row ),
		(model_map_fn) row_dirty_set_sub, NULL ) );
}

/* Break a dependency.
 */
static void *
row_link_break( Row *parent, Row *child )
{
	/* Must be there.
	 */
	assert( g_slist_find( parent->children, child ) &&
	    g_slist_find( child->parents, parent ) );

	parent->children = g_slist_remove( parent->children, child );
	child->parents = g_slist_remove( child->parents, parent );

#ifdef DEBUG_LINK
	printf( "row_link_break: breaking link from " );
	row_name_print( parent );
	printf( " to " );
	row_name_print( child );
	printf( "\n" );
#endif /*DEBUG_LINK*/

	return( NULL );
}

static void *
row_link_break_rev( Row *child, Row *parent )
{
	return( row_link_break( parent, child ) );
}

static void
row_expr_destroy( Row *row )
{
	/* Unlink from expr.
	 */
	if( row->expr ) {
		row->expr->row = NULL;
		row->expr = NULL;
	}

	/* We don't want expr_destroy() to destroy us ...
	 * break the link before we call.
	 */
	if( row->expr_private ) {
		Expr *expr = row->expr_private;

		row->expr_private->row = NULL;
		row->expr_private = NULL;
		expr_destroy( expr );
	}
}

static void
row_destroy( GtkObject *object )
{
	Row *row;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_ROW( object ) );

	row = ROW( object );

#ifdef DEBUG_NEW
	/* Can't use row_name_print(), we may not have a parent.
	 */
	printf( "row_destroy: %s", NN( MODEL( row )->name ) );
	if( row->sym ) 
		printf( " (%s)", symbol_name( row->sym ) );
	printf( "\n" );
#endif /*DEBUG_NEW*/

	/* Reset state. Also see row_parent_remove().
	 */
	if( row->expr )
		expr_error_clear( row->expr );
	if( row->top_col && row->top_col->last_select == row )
		row->top_col->last_select = NULL;

	/* Break all recomp links.
	 */
	slist_map( row->parents, (SListMapFn) row_link_break, row );
	slist_map( row->children, (SListMapFn) row_link_break_rev, row );
	assert( !row->parents && !row->children );
	(void) slist_map( row->recomp, (SListMapFn) row_dirty_clear, NULL );
	if( row->top_row )
		row->top_row->recomp_save = 
			g_slist_remove( row->top_row->recomp_save, row );
	FREEF( g_slist_free, row->recomp_save );

	assert( !row->recomp );

	/* Junk the expr.
	 */
	row_expr_destroy( row );

	/* Is this a top-level row? Kill the symbol too.
	 */
	if( row->sym_private ) {
		Symbol *sym = row->sym_private;

		sym->expr->row = NULL;
		row->sym = NULL;

		FREEO( row->sym_private );
	}

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

static Rhs *
row_get_rhs( Row *row )
{
	assert( g_slist_length( MODEL( row )->children ) == 1 );

	return( RHS( MODEL( row )->children->data ) );
}

static void
row_child_add( Model *parent, Model *child, int pos )
{
	Row *row = ROW( parent );

	MODEL_CLASS( parent_class )->child_add( parent, child, pos );

	/* Update our context.
	 */
	row->child_rhs = row_get_rhs( row );
}

static Subcolumn *
row_get_subcolumn( Row *row )
{
	return( SUBCOLUMN( MODEL( row )->parent ) );
}

static Column *
row_get_column( Row *row )
{
	Subcolumn *scol = row_get_subcolumn( row );

	return( scol->top_col ); 
}

/* Search back up the widget hierarchy for the base row for this
 * row ... eg "A7"->expr->row.
 */
static Row *
row_get_root( Row *row )
{
	Row *enclosing = row_get_parent( row );

	if( !enclosing )
		return( row );
	else
		return( row_get_root( enclosing ) );
}

static Workspace *
row_get_workspace( Row *row )
{
	return( row_get_column( row )->ws );
}

static void
row_parent_add( Model *child, Model *parent )
{
	Row *row = ROW( child );

	assert( IS_SUBCOLUMN( parent ) );

	MODEL_CLASS( parent_class )->parent_add( child, parent );

	/* Update our context.
	 */
	row->scol = row_get_subcolumn( row );
	row->top_col = row_get_column( row );
	row->ws = row_get_workspace( row );
	row->top_row = row_get_root( row );
}

static void
row_parent_remove( Model *child, Model *parent )
{
	Row *row = ROW( child );

	/* Reset the parts of state which touch our parent.
	 */
	row_dirty_clear( row );
	row_deselect( row );

	/* Don't clear error ... we may no longer have the link to expr. See
	 * row_destroy() for that.
	 */

	MODEL_CLASS( parent_class )->parent_remove( child, parent );
}

static gboolean
row_load( Model *model, 
	ModelLoadState *state, Model *parent, xmlNode *xnode )
{
	Row *row = ROW( model );
	Subcolumn *scol = SUBCOLUMN( parent );

	char name[256];

	if( !IS_SUBCOLUMN( parent ) ) {
		ierrors( "row_load: can only add a row to a subcolumn" );
		return( FALSE );
	}

	/* "name" is loaded by model ... we need it here for the link to the
	 * sym, grab early.
	 */
	if( !get_sprop( xnode, "name", name, 256 ) )
		return( FALSE );

#ifdef DEBUG
	printf( "row_load: loading row %s (xmlNode 0x%x)\n", 
		name, (unsigned int) xnode );
#endif /*DEBUG*/

	/* Popup is optional (top level only)
	 */
	(void) get_bprop( xnode, "popup", &row->popup );

	if( scol->istop ) {
		Column *col = scol->top_col;
		Workspace *ws = col->ws;

		Symbol *sym;

		sym = symbol_new( SYMBOL( ws )->expr->compile->locals, name );
		symbol_user_init( sym );
		(void) compile_new_local( sym->expr );
		row_link_symbol( row, sym, NULL );

		/* We can't symbol_made() here, we've not parsed our value
		 * yet. See below ... we just make sure we're on the recomp
		 * lists.
		 */
	}
	else {
		/* For locals, note the name in the row and do 
		 * row_link_symbol() on 1st recomp. See
		 * subcolumn_class_update_model_sub().
		 */
		model_set( MODEL( row ), name, NULL );
	}

	if( !MODEL_CLASS( parent_class )->load( model, state, parent, xnode ) )
		return( FALSE );

	if( scol->istop ) {
		/* If we've loaded a complete row system, mark this row plus
		 * any edited subrows dirty, and make sure this sym is 
		 * dirty too.
		 */
		row_dirty_set( row );
		expr_dirty( row->sym->expr, link_serial_new() );
	}

	return( TRUE );
}

/* Should we display this row. Non-displayed rows don't have rhs, don't
 * appear on the screen, and aren't saved. They do have rows though, so their
 * dependencies are tracked.
 *
 * We work off sym rather than row so that we can work before the row is fully
 * built.
 */
static gboolean
row_is_displayable( Symbol *sym )
{
	if( is_system( sym ) )
		return( FALSE );
	if( sym->expr && sym->expr->compile && sym->expr->compile->nparam > 0 )
		return( FALSE );
	if( is_super( sym ) && sym->expr ) {
		Expr *expr = sym->expr;
		PElement *root = &expr->root;

		/* Empty superclass.
		 */
		if( PEISELIST( root ) )
			return( FALSE );
	}

	return( TRUE );
}

static xmlNode *
row_save( Model *model, xmlNode *xnode )
{
	Row *row = ROW( model );

	xmlNode *xthis;

	/* Don't save system rows, or empty superclasses.
	 */
	if( row->sym ) {
		if( !row_is_displayable( row->sym ) )
			/* Need to return non-NULL for abort with no error.
			 */
			return( (xmlNode *) -1 );
	}

	if( !(xthis = MODEL_CLASS( parent_class )->save( model, xnode )) )
		return( NULL );

	/* Top-level only.
	 */
	if( row->top_row == row ) 
		if( !set_prop( xthis, "popup", bool_to_char( row->popup ) ) )
			return( NULL );

	return( xthis );
}

static void *
row_clear_to_save( Model *model )
{
	if( IS_ROW( model ) ) 
		ROW( model )->to_save = FALSE;

	return( NULL );
}

static void *
row_set_to_save( Row *row )
{
	Row *enclosing;

	if( !row->to_save ) {
		row->to_save = TRUE;

		/* All peers must be saved. When we reload, we want to keep
		 * row ordering. If we just save modded row, they'll move to
		 * the front of the row list on reload, since they'll be made
		 * first.
		 */
		model_map( MODEL( row->scol ),
			(model_map_fn) row_set_to_save, NULL, NULL );

		/* All rows back up to the top level must also be saved.
		 */
		for( enclosing = row; enclosing != row->top_row; 
			enclosing = row_get_parent( enclosing ) )
			row_set_to_save( enclosing );
	}

	return( NULL );
}

static void *
row_calculate_to_save( Model *model )
{
	if( IS_ROW( model ) ) {
		Row *row = ROW( model );
		Rhs *rhs = row->child_rhs;

		if( row != row->top_row && rhs && !row->to_save ) {
			if( rhs->itext && ITEXT( rhs->itext )->edited )
				row_set_to_save( row );
			else if( rhs->graphic && 
				CLASSMODEL( rhs->graphic )->edited )
				row_set_to_save( row );
		}
	}

	return( NULL );
}

static gboolean
row_save_test( Model *model )
{
	Row *row = ROW( model );
	Workspace *ws = row->ws;
	gboolean save;

	if( row == row->top_row ) {
		/* This is a top-level row ... save unless we're in
		 * only-save-selected mode.
		 */
		if( ws->save_type == WORKSPACE_SAVE_SELECTED )
			save = row->selected;
		else
			save = TRUE;

		/* If we're going to save this row, clear all the to_save
		 * flags, then walk the tree working out which bits we will need
		 * to write.
		 */
		if( save ) {
			model_map_all( MODEL( row ),
				(model_map_fn) row_clear_to_save, NULL );
			model_map_all( MODEL( row ),
				(model_map_fn) row_calculate_to_save, NULL );
		}

	}
	else 
		save = row->to_save;

	return( save );
}

/* Make a private expr for this row.
 */
static void
row_expr_private( Row *row )
{
	Symbol *sym = row->sym;

#ifdef DEBUG
	printf( "row_expr_private: " );
	symbol_name_print( row->sym );
	printf( "\n" );
#endif /*DEBUG*/

	row_expr_destroy( row );

	row->expr = row->expr_private = expr_clone( row->sym );
	assert( row->expr );
	sym->ext_expr = g_slist_prepend( sym->ext_expr, row->expr );
	row->expr->row = row;
}

static void *
row_new_heap( Heapmodel *heapmodel, PElement *root )
{
	Row *row = ROW( heapmodel );
	Expr *expr = row->expr;

#ifdef DEBUG
	printf( "row_new_heap: " );
	row_name_print( row );
	printf( "\n" );

	printf( "row_new_heap: new value is " );
	pgraph( root );

	printf( "row_new_heap: top value is " );
	pgraph( &row->top_row->expr->root );
#endif /*DEBUG*/

	if( row_is_displayable( row->sym ) ) {
		if( is_super( row->sym ) && PEISCLASS( root ) &&
			*MODEL( PEGETCLASSCOMPILE( root )->sym )->name == '_' )
			model_set_display( MODEL( heapmodel ), FALSE );
		else
			model_set_display( MODEL( heapmodel ), TRUE );
	}

	/* New value ... reset error state.
	 */
	expr_error_clear( expr );
	expr->root = *root;
	expr_value_new( expr );

	if( row->child_rhs &&
		heapmodel_new_heap( HEAPMODEL( row->child_rhs ), root ) )
		return( heapmodel );

	/* Class display only for non-param classes.
	 */
	row->isclass = PEISCLASS( root ) && row->sym->type != SYM_PARAM;

	/* Set the default vis level.
	 */
	if( row->child_rhs && row->child_rhs->vislevel == -1 ) {
		PElement member;
		double value;
		gboolean isclass;

		if( !heap_isclass( root, &isclass ) )
			return( heapmodel );

		/* If it's a class with a vis hint, use that.
		 */
		if( isclass && 
			class_get_member( root, MEMBER_VISLEVEL, &member ) &&
			heap_get_real( &member, &value ) ) 
			rhs_set_vislevel( row->child_rhs, value );

		/* Non-parameter rows get higher vislevel, except for super. 
		 */
		else if( row->sym->type != SYM_PARAM && !is_super( row->sym ) )
			rhs_set_vislevel( row->child_rhs, 1 );
		else 
			rhs_set_vislevel( row->child_rhs, 0 );
	}

	return( HEAPMODEL_CLASS( parent_class )->new_heap( heapmodel, root ) ); }

static void *
row_update_model( Heapmodel *heapmodel )
{
	Row *row = ROW( heapmodel );

	if( row->expr )
		expr_value_new( row->expr );

	return( HEAPMODEL_CLASS( parent_class )->update_model( heapmodel ) );
}

static void
row_class_init( RowClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	ModelClass *model_class = (ModelClass *) klass;
	HeapmodelClass *heapmodel_class = (HeapmodelClass *) klass;

	parent_class = gtk_type_class( TYPE_HEAPMODEL );

	object_class->destroy = row_destroy;

	/* Create signals.
	 */

	/* Init methods.
	 */
	model_class->view_new = rowview_new;
	model_class->child_add = row_child_add;
	model_class->parent_add = row_parent_add;
	model_class->parent_remove = row_parent_remove;
	model_class->load = row_load;
	model_class->save = row_save;
	model_class->save_test = row_save_test;

	heapmodel_class->new_heap = row_new_heap;
	heapmodel_class->update_model = row_update_model;

	/* Static init.
	 */
	model_register_loadable( MODEL_CLASS( klass ) );
}

static void
row_init( Row *row )
{
#ifdef DEBUG
	printf( "row_init\n" );
#endif /*DEBUG*/

	row->scol = NULL;
	row->child_rhs = NULL;
	row->top_col = NULL;
	row->ws = NULL;
	row->top_row = NULL;

	row->sym = NULL;
	row->sym_private = NULL;

	row->expr = NULL;
	row->expr_private = NULL;
	row->err = FALSE;

	row->selected = FALSE;
	row->isclass = FALSE;
	row->popup = POPUP_NEW_ROWS;
	row->to_save = FALSE;

	/* Init recomp stuff.
	 */
	row->parents = NULL;
	row->children = NULL;
	row->dirty = FALSE;
	row->recomp = NULL;
	row->recomp_save = NULL;

	row->depend = FALSE;

	row->show = ROW_SHOW_NONE;
}

GtkType
row_get_type( void )
{
	static GtkType row_type = 0;

	if( !row_type ) {
		static const GtkTypeInfo info = {
			"Row",
			sizeof( Row ),
			sizeof( RowClass ),
			(GtkClassInitFunc) row_class_init,
			(GtkObjectInitFunc) row_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		row_type = gtk_type_unique( TYPE_HEAPMODEL, &info );
	}

	return( row_type );
}

/* After making a row and adding it to model tree ... attach the symbol and
 * value this row displays.
 */
void
row_link_symbol( Row *row, Symbol *sym, PElement *root )
{
	assert( !row->sym );
	assert( !sym->expr || !sym->expr->row );

	row->sym = sym;

	/* Code we display/update ... if this is a top-level row, we
	 * directly change the symbol's expr. If it's a sub-row, we need a
	 * cloned expr for us to fiddle with.
	 */
	if( is_top( sym ) ) {
		row->expr = sym->expr;
		row->expr_private = NULL;
		row->sym_private = sym;
		assert( !row->expr->row );
		row->expr->row = row;
	}
	else {
		row_expr_private( row );

		if( root ) {
			row->expr->root = *root;
			expr_value_new( row->expr );
		}
	}
}

Row *
row_new( Subcolumn *scol, Symbol *sym, PElement *root )
{
	Row *row;

#ifdef DEBUG_NEW
	printf( "row_new: " );
	dump_tiny( sym );
	printf( ", pos = %d\n", pos );
#endif /*DEBUG_NEW*/

	row = gtk_type_new( TYPE_ROW );

	/* Don't make a display or a RHS for invisible rows.
	 */
	if( !row_is_displayable( sym ) )
		MODEL( row )->display = FALSE;
	else 
		(void) rhs_new( row ); 

	model_set( MODEL( row ), MODEL( sym )->name, NULL );
	model_child_add( MODEL( scol ), MODEL( row ), -1 );

	row_link_symbol( row, sym, root );

	return( row );
}

/* Make a dependency. parent is displaying an expression which 
 * refers to the symbol being displayed by child.
 */
static void *
row_link_make( Row *parent, Row *child )
{
	/* Already a dependency? Don't make a second link.
	 */
	if( g_slist_find( parent->children, child ) ) 
		return( NULL );

	/* Don't link to self (harmless, but pointless too).
	 */
	if( parent == child )
		return( NULL );
	
	/* New link, each direction.
	 */
	parent->children = g_slist_prepend( parent->children, child );
	child->parents = g_slist_prepend( child->parents, parent );

#ifdef DEBUG_LINK
	printf( "row_link_make: " );
	row_name_print( parent );
	printf( " refers to " );
	row_name_print( child );
	printf( "\n" );
#endif /*DEBUG_LINK*/

	return( NULL );
}

static void *
row_link_build4( Expr *child_expr, Row *row )
{
	if( child_expr->row->top_row == row )
		return( child_expr->row );

	return( NULL );
}

/* Does child have a display in the same tally heirarchy as row? Make a link!
 */
static void *
row_link_build3( Symbol *child, Row *row )
{
	Row *child_row;

	if( (child_row = (Row *) slist_map( child->ext_expr,
		(SListMapFn) row_link_build4, row->top_row )) ) 
		(void) row_link_make( row, child_row );

	return( NULL );
}

static void *row_link_build2( Expr *expr, Row *row );

static void *
row_link_build2_sym( Symbol *sym, Row *row )
{
	if( sym->expr && row_link_build2( sym->expr, row ) )
		return( row );
	
	return( NULL );
}

static void *
row_link_build2( Expr *expr, Row *row )
{
	/* Make links to anything expr refers to in this tree.
	 */
	if( expr->compile &&
		slist_map( expr->compile->children, 
			(SListMapFn) row_link_build3, row ) )
		return( expr );

	/* Recurse for any locals of expr. 
	 * Exception: 
	 * 
	 *	f = class {
	 *		g = class {
	 *			a = 12;
	 *		}
	 *	}
	 *
	 * zero-arg local classes will have rows anyway, so we don't need to
	 * check inside them for locals, since we'll do them anyway at the top
	 * level.
	 */
	if( expr->compile && 
		!(is_class( expr->compile ) && expr->compile->nparam == 0) &&
		stable_map( expr->compile->locals,
			(symbol_map_fn) row_link_build2_sym, row, NULL, NULL ) )
		return( expr );

	return( NULL );
}

/* Scan a row, adding links for any dependencies we find.
 */
static void *
row_link_build( Row *row )
{
#ifdef DEBUG_LINK
	printf( "row_link_build: " );
	row_name_print( row );
	printf( "\n" );
#endif /*DEBUG_LINK*/

	/* Build new recomp list. Only for class displays.
	 */
	if( !row->scol->istop && row->expr && 
		row_link_build2( row->expr, row ) )
		return( row );

	return( NULL );
}

/* Remove any links on a row.
 */
static void *
row_link_destroy( Row *row )
{
	slist_map( row->children, 
		(SListMapFn) row_link_break_rev, row );

	return( NULL );
}

/* Do this row, and any that depend on it.
 */
static void *
row_dependant_mark( Row *row, row_map_fn fn, void *a )
{
	static void *row_dependant_map_sub( Row *row, row_map_fn fn, void *a );

	void *res;

	/* Done this one already?
	 */
	if( row->depend )
		return( NULL );

	row->depend = TRUE;
	if( (res = fn( row, a, NULL, NULL )) )
		return( res );

	return( row_dependant_map_sub( row, fn, a ) );
}

/* Find the root of this scrap of tree.
 */
static Row *
row_dependant_root( Row *row )
{
	/* Find our super.
	 */
	if( (row = row->scol->super) ) { 
		/* Up to the nearest non-super.
		 */
		do {
			row = row_get_parent( row );
		} while( row == row->scol->super );
	}

	return( row );
}

/* If this row depends on "this", mark it.
 */
static void *
row_dependant_this( Row *row, Row *root, row_map_fn fn, void *a )
{
	if( row_dependant_root( row ) == root ) {
		/* This row shares a this with the modified row ... if it
		 * explicitly refs this, we need to update it.
		 */
		if( g_slist_find( row->children, row->scol->this ) ) 
			return( row_dependant_mark( row, fn, a ) );
	}

	return( NULL );
}

/* Apply to all dependants of row.
 */
static void *
row_dependant_map_sub( Row *row, row_map_fn fn, void *a )
{
	Row *i;
	void *res;

	/* Things that refer to us.
	 */
	if( (res = slist_map2( row->parents, 
		(SListMap2Fn) row_dependant_mark, (void *) fn, a )) )
		return( res );

	/* Things that refer to our enclosing syms ... eg. if A1.fred.x 
	 * changes, we don't want to recalc A1.fred, we do want to recalc 
	 * anything that refers to A1.fred.
	 */
	for( i = row; (i = HEAPMODEL( i )->row); ) 
		if( (res = row_dependant_map_sub( i, fn, a )) )
			return( res );

	/* Yuk ... we need to recomp anything in the whole of the class which
	 * explicitly refers to the "this" that "row" is part of. "this" can be
	 * aliased, so in fact we want any ref to this along this class spine.
	 * Track up "super" to find the base of this chunk, then search down
	 * supers again looking at each member of each class for refs to this.
	 * Don't search into class rows (they'll have a different this).

	 	FIXME ... potentially a lot of stuff ... this is because we
		can't track refs on the RHS of DOT at compile time

		could maybe do something like the dynamic links things?

		we could precompute some of this ... link each "this" to the
		most enclosing this in the same class? we wouldn't need the
		_root() searches then

	 */
	if( (i = row_dependant_root( row )) )
		if( (res = row_map_all( i, 
			(row_map_fn) row_dependant_this, i, (void *) fn, a )) )
			return( res );

	return( NULL );
}

static void *
row_dependant_clear( Row *row )
{
	row->depend = FALSE;

	return( NULL );
}

/* Apply a function to all rows in this tree which depend on this row.
 */
void *
row_dependant_map( Row *row, row_map_fn fn, void *a )
{
	/* Clear the flags we use to spot loops.
	 */
	row_map_all( row->top_row,
		(row_map_fn) row_dependant_clear, NULL, NULL, NULL );

	return( row_dependant_map_sub( row, fn, a ) );
}

/* This row has changed ... mark all dependants (direct and indirect) 
 * dirty.
 */
void *
row_dirty( Row *row )
{
	(void) row_dirty_set( row );
	(void) row_dependant_map( row, (row_map_fn) row_dirty_set, NULL );

	return( NULL );
}

/* This tally has changed ... mark all dependants (but not this one!)
 * dirty.
 */
void *
row_dirty_intrans( Row *row )
{
	(void) row_dependant_map( row, (row_map_fn) row_dirty_set, NULL );

	return( NULL );
}

/* Find the 'depth' of a row ... 0 is top level.
 */
static int
row_recomp_depth( Row *row )
{
	if( row == row->top_row )
		return( 0 );

	return( 1 + row_recomp_depth( row_get_parent( row ) ) );
}

/* Compare func for row recomp sort.
 */
static int
row_recomp_sort_func( Row *a, Row *b )
{
#ifdef DEBUG
	printf( "row_recomp_sort_func: " );
#endif /*DEBUG*/

	/* If b depends on a, want a first.
	 */
	if( row_dependant_map( a, (row_map_fn) map_equal, b ) ) {
#ifdef DEBUG
		row_name_print( b );
		printf( " depends on " );
		row_name_print( a );
		printf( ", " );
		row_name_print( a );
		printf( " goes first\n" );
#endif /*DEBUG*/

		return( -1 );
	}
	else if( row_dependant_map( b, (row_map_fn) map_equal, a ) ) {
#ifdef DEBUG
		row_name_print( a );
		printf( " depends on " );
		row_name_print( b );
		printf( ", " );
		row_name_print( b );
		printf( " goes first\n" );
#endif /*DEBUG*/

		return( 1 );
	}
	else {
		int adepth = row_recomp_depth( a );
		int bdepth = row_recomp_depth( b );

#ifdef DEBUG
		row_name_print( a );
		printf( " depth = %d, ", adepth );
		row_name_print( b );
		printf( " depth = %d: lowest first\n", bdepth );
#endif /*DEBUG*/

		/* No dependency ... want shallower first.
		 */
		return( adepth - bdepth );
	}
}

static gboolean
row_regenerate( Row *row )
{
	Expr *expr = row->expr;
	PElement base;

	/* Regenerate any compiled code.
	 */
	if( expr->compile ) {
		PEPOINTE( &base, &expr->compile->base );

		if( !PEISNOVAL( &base ) ) {
			PElement *root = &expr->root;

			if( row == row->top_row ) {
				/* Recalcing base of tally display ... not a 
				 * class member, must be a sym with a value.
				 */
				gboolean res;

				res = reduce_regenerate( expr, root );
				expr_value_new( expr );

				if( !res )
					return( FALSE );
			}
			else {
				/* Recalcing a member somewhere inside ... 
				 * regen (member this) pair. Get the "this"
				 * for the enclosing class instance ... the
				 * top one won't always be right (eg. for
				 * local classes); the enclosing one should
				 * be the same as the most enclosing this.
				 */
				Row *this = row->scol->this;
				gboolean res;

				res = reduce_regenerate_member( expr, 
					&this->expr->root, root );
				expr_value_new( expr );

				if( !res )
					return( FALSE );
			}

			/* We may have made a new class instance ... all our 
			 * children need to update their heap pointers.
			 */
			if( heapmodel_new_heap( HEAPMODEL( row ), root ) ) 
				return( FALSE );
		}
	}

	return( TRUE );
}

/* Does this row have a dirty enclosing row?
 */
static gboolean 
row_is_enclosing_dirty( Row *row )
{
	Row *enclosing;

	if( row == row->top_row )
		return( FALSE );

	if( (enclosing = row_get_parent( row ))->dirty )
		return( TRUE );

	return( row_is_enclosing_dirty( enclosing ) );
}

static gboolean
row_recomp_row( Row *row )
{
	Rhs *rhs = row->child_rhs;

#ifdef DEBUG
	printf( "row_recomp_row: " );
	row_name_print( row );
	printf( "\n" );
#endif /*DEBUG*/

	/* Not much we can do. 
	 */
	if( !row->expr )
		return( TRUE );

	/* Clear old error state.
	 */
	expr_error_clear( row->expr );

	/* Parse and compile any changes to our text since we last came through.
	 */
	if( rhs && rhs->itext && 
		heapmodel_update_heap( HEAPMODEL( rhs->itext ) ) ) 
		return( FALSE );

	/* We're about to zap the graph: make sure this tree of rows has a
	 * private copy.
	 */
	if( !subcolumn_make_private( row->scol ) )
		return( FALSE );

	/* Regenerate from the expr.
	 */
	if( !row_regenerate( row ) ) 
		return( FALSE );

	/* Reapply any graphic mods.
	 */
	if( rhs && rhs->graphic ) {
		Classmodel *classmodel = CLASSMODEL( rhs->graphic );

		/* If the graphic is non-default, need to set modified to make
		 * sure we reapply the changes.
		 */
		if( classmodel->edited )
			heapmodel_set_modified( HEAPMODEL( classmodel ), TRUE );

		if( heapmodel_update_heap( HEAPMODEL( classmodel ) ) )
			return( FALSE );
	}

	return( TRUE );
}

static void
row_recomp_all( Row *top_row )
{
	/* Rebuild all dirty rows.
	 */
	while( !top_row->err && top_row->recomp ) {
		Row *dirty_row = ROW( top_row->recomp->data );

#ifdef DEBUG_TIME
		printf( "row_recomp_all: starting for row " );
		row_name_print( dirty_row );
		printf( "\n" );
#endif /*DEBUG_TIME*/

		if( !row_recomp_row( dirty_row ) ) {
			expr_error_set( dirty_row->expr );

			if( dirty_row != top_row ) {
				BufInfo buf;
				char txt[100];

				buf_init_static( &buf, txt, 100 );
				row_qualified_name( dirty_row, &buf );

				ierrors( "error in row %s: %s",	
					buf_all( &buf ),
					dirty_row->expr->errstr );

				expr_error_set( top_row->expr );
			}

			break;
		}

		row_dirty_clear( dirty_row );

#ifdef DEBUG_TIME
		/*
		printf( "row_recomp_all: after row recomp, top value now " );
		pgraph( &top_row->expr->root );
		 */
#endif /*DEBUG_TIME*/
	}
}

static void *
row_recomp_error_clear( Row *row )
{
	if( row->expr )
		expr_error_clear( row->expr );

	return( NULL );
}

void
row_recomp( Row *row )
{
	Row *top_row = row->top_row;

#ifdef DEBUG_TIME
	static GTimer *recomp_timer = NULL;

	if( !recomp_timer )
		recomp_timer = g_timer_new();

	g_timer_reset( recomp_timer );

	printf( "row_recomp: starting for dirties on " );
	row_name_print( top_row );
	printf( "\n" );
#endif /*DEBUG_TIME*/

	/* Sort dirties into recomp order.
	 */
	top_row->recomp = g_slist_sort( top_row->recomp, 
		(GCompareFunc) row_recomp_sort_func );

#ifdef DEBUG_DIRTY
	printf( "row_recomp: sorted dirties are: " );
	slist_map( top_row->recomp, (SListMapFn) row_name_print, NULL );
	printf( "\n" );
#endif /*DEBUG_DIRTY*/

	/* Take a copy of the recomp list for later testing.
	 */
	FREEF( g_slist_free, top_row->recomp_save );
	top_row->recomp_save = g_slist_copy( top_row->recomp );

	/* Remove all top-level dependencies.
	 */
	(void) symbol_link_destroy( top_row->sym );

	/* Remove any row recomp links we have.
	 */
	(void) row_map_all( top_row,
		(row_map_fn) row_link_destroy, NULL, NULL, NULL );

	/* Rebuild all dirty rows. This may add some dynamic top links.
	 */
	row_recomp_all( top_row );

	/* Add all static row links. Have to do this after any 
	 * parsing in row_recomp_all().
	 */
	(void) row_map_all( top_row,
		(row_map_fn) row_link_build, NULL, NULL, NULL );

	/* Remake all static top-level links.
	 */
	(void) symbol_link_build( top_row->sym );

	/* Now we know dependencies ... mark everything dirty again. This may
	 * pick up stuff we missed last time and may change the order we
	 * recomp rows in.
	 */
	slist_map( top_row->recomp_save, (SListMapFn) row_dirty, NULL );

	/* Is this topsym still a leaf? We may have discovered an external 
	 * reference to another dirty top-level sym. We can come back here
	 * later.
	 */
	if( top_row->sym->ndirtychildren != 0 ) {
#ifdef DEBUG_TIME
		printf( "row_recomp: delaying recomp of " );
		row_name_print( top_row );
		printf( " after %gs\n",  
			g_timer_elapsed( recomp_timer, NULL ) );
#endif /*DEBUG_TIME*/

		/* If we have discovered external references, those rows will
		 * now have errors. Clear all errors to make sure we recomp
		 * next time we try this row.
		 */
		slist_map( top_row->recomp_save,
			(SListMapFn) row_recomp_error_clear, NULL );

		FREEF( g_slist_free, top_row->recomp_save );

		return;
	}

	/* Sort dirties into recomp order.
	 */
	top_row->recomp = g_slist_sort( top_row->recomp, 
		(GCompareFunc) row_recomp_sort_func );

#ifdef DEBUG_DIRTY
	printf( "row_recomp: pass 2: sorted dirties are: " );
	slist_map( top_row->recomp, (SListMapFn) row_name_print, NULL );
	printf( "\n" );
#endif /*DEBUG_DIRTY*/

	/* Now: if the recomp list is the same as last time, we don't need to
	 * recalc again.
	 */
	if( slist_equal( top_row->recomp_save, top_row->recomp ) ) {
		/* Can just mark all rows clean.
		 */
		slist_map( top_row->recomp, 
			(SListMapFn) row_dirty_clear, NULL );
	}
	else {
#ifdef DEBUG_DIRTY
		printf( "row_recomp: recomp list has changed ... pass 2\n" );
#endif /*DEBUG_DIRTY*/

		/* Rebuild all dirty rows.
		 */
		row_recomp_all( top_row );
	}

	FREEF( g_slist_free, top_row->recomp_save );

	symbol_dirty_clear( top_row->sym );

	/* Now we're clean, all models can update from the heap. Only if this
	 * row has no errors, though.
	 */
	if( !top_row->err && model_map_all( MODEL( top_row ),
		(model_map_fn) heapmodel_update_model, NULL ) )
		expr_error_set( top_row->expr );

#ifdef DEBUG_TIME
	printf( "row_recomp: done for dirties of " );
	row_name_print( top_row );
	printf( " in %gs\n",  g_timer_elapsed( recomp_timer, NULL ) );
#endif /*DEBUG_TIME*/

#ifdef DEBUG
	printf( "row_recomp: value of " );
	row_name_print( top_row );
	printf( " is " );
	pgraph( &top_row->expr->root );
#endif /*DEBUG*/
}

/* Deselect a row. 
 */
void *
row_deselect( Row *row )
{
	Workspace *ws;

	if( !row->selected )
		return( NULL );

	ws = row->ws;

	assert( ws && IS_WORKSPACE( ws ) );
	assert( g_slist_find( ws->selected, row ) );

	ws->selected = g_slist_remove( ws->selected, row );
	row->selected = FALSE;

	model_changed( MODEL( row ) );
	mainw_select_message_update();

	return( NULL );
}

/* Select a row. 
 */
static void
row_select2( Row *row )
{
	if( !row->selected ) {
		Workspace *ws = row->ws;

		row->selected = TRUE;
		ws->selected = g_slist_append( ws->selected, row );

		model_changed( MODEL( row ) );
		mainw_select_message_update();
	}
}

/* Select a row, deselecting others first.
 */
void *
row_select( Row *row )
{
	Workspace *ws = row->ws;

	workspace_deselect_all( ws );
	row_select2( row );

	/* Note for extend select.
	 */
	row->top_col->last_select = row;

	return( NULL );
}

/* Extend the previous selection.
 */
void *
row_select_extend( Row *row )
{
	Column *col = row->top_col;
	Row *last_select = col->last_select;

	/* Range select if there was a previous selection, and it was in the
	 * same subcolumn.
	 */
	if( last_select && row->scol == last_select->scol ) {
		Subcolumn *scol = row->scol;
		GSList *rows = MODEL( scol )->children;
		int pos = g_slist_index( rows, row );
		int pos_last = g_slist_index( rows, last_select );
		int step = pos > pos_last ? 1 : -1;
		int i;

		assert( pos != -1 && pos_last != -1 );

		for( i = pos_last; i != pos + step; i += step )
			row_select2( ROW( g_slist_nth_data( rows, i ) ) );
	}
	else 
		row_select2( row );

	/* Note for extend select.
	 */
	col->last_select = row;

	return( NULL );
}

/* Toggle a selection.
 */
void *
row_select_toggle( Row *row )
{
	if( row->selected ) {
		row_deselect( row );
		row->top_col->last_select = NULL;
	}
	else {
		row_select2( row );
		row->top_col->last_select = row;
	}

	return( NULL );
}

/* Do a select action using a modifier.
 */
void 
row_select_modifier( Row *row, guint state )
{
	if( state & GDK_CONTROL_MASK ) 
		row_select_toggle( row );
	else if( state & GDK_SHIFT_MASK ) 
		row_select_extend( row );
	else 
		row_select( row );
}

void
row_set_show( Row *row, RowShowState show )
{
	if( row->show != show ) {
		row->show = show;
		model_changed( MODEL( row ) );
	}
}
