/* widgets for the status bar
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

#include "ip.h"

static GtkFrameClass *parent_class = NULL;

/* The popup menu.
 */
static GtkWidget *statusview_menu = NULL;

/* Sub. fn. of below. Junk the widgets inside a band display.
 */
static void *
statusviewband_destroy_sub( StatusviewBand *svb )
{	
	FREEW( svb->val );
	FREE( svb );

	return( NULL );
}

/* Junk the widgets inside a band display.
 */
static void
statusviewband_destroy( Statusview *sv )
{	
	slist_map( sv->bands, 
		(SListMapFn) statusviewband_destroy_sub, NULL );
	FREEF( g_slist_free, sv->bands );
}

static void
statusview_destroy( GtkObject *object )
{
	Statusview *sv;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_STATUSVIEW( object ) );

	sv = STATUSVIEW( object );

	/* My instance destroy stuff.
	 */
	FREESID( sv->conv_changed_sid, sv->conv );
	FREESID( sv->conv_destroy_sid, sv->conv );
	FREESID( sv->iimage_changed_sid, sv->iimage );
	FREESID( sv->iimage_destroy_sid, sv->iimage );
	statusviewband_destroy( sv );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

/* Hide this statusview.
 */
static void
statusview_hide_cb( GtkWidget *menu, GtkWidget *host, Statusview *sv )
{
	sv->conv->status = FALSE;
	conversion_refresh( sv->conv );
}

static void
statusview_class_init( StatusviewClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;

	GtkWidget *pane;

	parent_class = gtk_type_class( GTK_TYPE_FRAME );

	object_class->destroy = statusview_destroy;

	/* Create signals.
	 */

	/* Init methods.
	 */

	pane = statusview_menu = popup_build( "Status bar menu" );
	popup_add_but( pane, "Hide", statusview_hide_cb );
}

static void
statusview_init( Statusview *sv )
{
	GtkWidget *vb, *hb;
	GtkWidget *eb;

	sv->conv = NULL;

	sv->conv_changed_sid = 0;
	sv->conv_destroy_sid = 0;
	sv->iimage_changed_sid = 0;
	sv->iimage_destroy_sid = 0;
	sv->bands = NULL;

        gtk_frame_set_shadow_type( GTK_FRAME( sv ), GTK_SHADOW_OUT );

	eb = gtk_event_box_new();
        gtk_container_add( GTK_CONTAINER( sv ), eb );
        popup_attach( eb, statusview_menu, sv );

	vb = gtk_vbox_new( FALSE, 0 );
        gtk_container_set_border_width( GTK_CONTAINER( vb ), 1 );
        gtk_container_add( GTK_CONTAINER( eb ), vb );

	sv->top = gtk_label_new( "" );
        gtk_misc_set_alignment( GTK_MISC( sv->top ), 0.0, 0.5 );
        gtk_box_pack_start( GTK_BOX( vb ), sv->top, TRUE, TRUE, 0 );

	hb = gtk_hbox_new( FALSE, 5 );
        gtk_box_pack_start( GTK_BOX( vb ), hb, TRUE, TRUE, 0 );

	sv->pos = gtk_label_new( "" );
	set_fixed( sv->pos, strlen( "(88888,88888)" ) );
        gtk_misc_set_alignment( GTK_MISC( sv->pos ), 0.0, 0.5 );
        gtk_box_pack_start( GTK_BOX( hb ), sv->pos, FALSE, FALSE, 0 );

	sv->hb = gtk_hbox_new( FALSE, 5 );
        gtk_box_pack_start( GTK_BOX( hb ), sv->hb, TRUE, TRUE, 0 );

	sv->mag = gtk_label_new( "" );
        gtk_misc_set_alignment( GTK_MISC( sv->mag ), 0.0, 0.5 );
        gtk_box_pack_end( GTK_BOX( hb ), sv->mag, FALSE, FALSE, 0 );

	gtk_widget_show_all( eb );
}

GtkType
statusview_get_type( void )
{
	static GtkType statusview_type = 0;

	if( !statusview_type ) {
		static const GtkTypeInfo sinfo = {
			"Statusview",
			sizeof( Statusview ),
			sizeof( StatusviewClass ),
			(GtkClassInitFunc) statusview_class_init,
			(GtkObjectInitFunc) statusview_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		statusview_type = 
			gtk_type_unique( GTK_TYPE_FRAME, &sinfo );
	}

	return( statusview_type );
}

/* Our conversion has changed ... update.
 */
static void
statusview_conv_changed_cb( Conversion *conv, Statusview *sv )
{
	if( sv->conv->mag > 0 )
		set_glabel( sv->mag, "Mag %d:1", sv->conv->mag );
	else
		set_glabel( sv->mag, "Mag 1:%d", -sv->conv->mag );

	widget_visible( GTK_WIDGET( sv ), conv->status );
}

/* Our iimage has changed ... update.
 */
static void
statusview_iimage_changed_cb( Iimage *iimage, Statusview *sv )
{
	static char *sample[] = {
		/* Sample text for each BandFmt. Used to try to get
		 * the spacing right.
		 */
		"888",				/* uchar */
		"-888",				/* char */
		"88888",			/* ushort */
		"-88888",			/* short */
		"888888888",			/* int */
		"-888888888",			/* uint */
		"888888888",			/* float */
		"(88888888,888888888)",		/* complex */
		"88888888888",			/* double */
		"(8888888888,888888888)"	/* dpcomplex */
	};

	IMAGE *im = imageinfo_get( FALSE, iimage->instance.image );
	double size = (double) im->Ysize * IM_IMAGE_SIZEOF_LINE( im );
	int b;
	int fmt;
	BufInfo buf;
	char txt[MAX_LINELENGTH];

	buf_init_static( &buf, txt, MAX_LINELENGTH );

	buf_appendf( &buf, "%s, ", buf_all( &iimage->caption ) );
	if( im->Coding != IM_CODING_NONE )
		buf_appendf( &buf, "%s, ", im_Coding2char( im->Coding ) );
	to_size( &buf, size );
	buf_appendf( &buf, ", %.3gx%.3g p/mm", im->Xres, im->Yres );
	set_glabel( sv->top, "%s", buf_all( &buf ) );

	if( im->Coding == IM_CODING_LABQ ) {
		b = 3;
		fmt = 6;
	}
	else {
		b = im->Bands;
		fmt = im->BandFmt;
	}

	if( (int) g_slist_length( sv->bands ) != b || sv->fmt != fmt ) {
		/* Bands/fmt has changed ... rebuild band display widgets.
		 */
		int i;
		int width;

		statusviewband_destroy( sv );
		sv->fmt = fmt;
		if( sv->fmt >= 0 && sv->fmt < IM_NUMBER( sample ) )
			width = strlen( sample[sv->fmt] );
		else
			width = 10;

		for( i = 0; i < b; i++ ) {
			StatusviewBand *band = IM_NEW( NULL, StatusviewBand );

			band->sv = sv;
			band->bandno = i;
			band->val = gtk_label_new( "" );
			gtk_misc_set_alignment( GTK_MISC( band->val ), 
				0.0, 0.5 );
			set_fixed( band->val, width );
			gtk_box_pack_start( GTK_BOX( sv->hb ), 
				band->val, FALSE, FALSE, 0 );
			gtk_widget_show( band->val );

			sv->bands = g_slist_append( sv->bands, band );
		}
	}
}

/* Our conversion has been destroyed ... destroy us in turn.
 */
static void
statusview_destroy_cb( Conversion *conv, Statusview *sv )
{
	gtk_widget_destroy( GTK_WIDGET( sv ) );
}

static void
statusview_link( Statusview *sv, Conversion *conv, Iimage *iimage )
{
	assert( !sv->conv );

	sv->conv = conv;
	sv->iimage = iimage;

	sv->conv_changed_sid = gtk_signal_connect( GTK_OBJECT( sv->conv ), 
		"changed", GTK_SIGNAL_FUNC( statusview_conv_changed_cb ), sv );
	sv->conv_destroy_sid = gtk_signal_connect( GTK_OBJECT( sv->conv ), 
		"destroy", GTK_SIGNAL_FUNC( statusview_destroy_cb ), sv );
	sv->iimage_changed_sid = gtk_signal_connect( GTK_OBJECT( sv->iimage ), 
		"changed", 
		GTK_SIGNAL_FUNC( statusview_iimage_changed_cb ), sv );
	sv->iimage_destroy_sid = gtk_signal_connect( GTK_OBJECT( sv->iimage ), 
		"destroy", GTK_SIGNAL_FUNC( statusview_destroy_cb ), sv );

	/* Update for inital iimage.
	 */
	statusview_iimage_changed_cb( iimage, sv );
}

Statusview *
statusview_new( Conversion *conv, Iimage *iimage )
{
	Statusview *sv = gtk_type_new( TYPE_STATUSVIEW );

	statusview_link( sv, conv, iimage );

	return( sv );
}

/* Turn a IM_CODING_LABQ 4-band image into three floats.
 */
static void
statusview_mouse_LABPACK( Statusview *sv, int x, int y )
{
	GSList *bands = sv->bands;

	/* Three widgets we update.
	 */
	StatusviewBand *b1;
	StatusviewBand *b2;
	StatusviewBand *b3;

	unsigned char *e;
	unsigned int iL, ia, ib;
	float L, a, b;

	if( sv->conv->err )
		return;
	if( !(e = (unsigned char *) get_element( sv->conv->reg, x, y, 0 )) ) {
		conversion_error_set( sv->conv );
		return;
	}

	iL = (e[0] << 2) | (e[3] >> 6);
	L = 100.0 * iL / 1023.0;
	ia = ((signed char) e[1] << 3) | ((e[3] >> 3) & 0x7);
	a = 0.125 * ia;
	ib = ((signed char) e[2] << 3) | (e[3] & 0x7);
	b = 0.125 * ib;

	if( g_slist_length( sv->bands ) == 3 ) {
		b1 = (StatusviewBand *) bands->data;
		b2 = (StatusviewBand *) bands->next->data;
		b3 = (StatusviewBand *) bands->next->next->data;

		set_glabel( b1->val, "%g", L );
		set_glabel( b2->val, "%g", a );
		set_glabel( b3->val, "%g", b );
	}
}

/* Sub-fn of below. Remake a band in the bar.
 */
static void *
statusview_mouse_band( StatusviewBand *svb, void *e )
{
	REGION *reg = svb->sv->conv->reg;
	IMAGE *im = reg->im;

	/* Generate string for contents of band element.
	 */
	if( im->Coding == IM_CODING_NONE )
		switch( im->BandFmt ) {
		case IM_BANDFMT_UCHAR:
			set_glabel( svb->val, "%d", 
				((unsigned char *)e)[svb->bandno] );
			break;
			
		case IM_BANDFMT_CHAR:
			set_glabel( svb->val, "%d", 
				((char *)e)[svb->bandno] );
			break;
			
		case IM_BANDFMT_USHORT:
			set_glabel( svb->val, "%d", 
				((unsigned short *)e)[svb->bandno] );
			break;
			
		case IM_BANDFMT_SHORT:
			set_glabel( svb->val, "%d", 
				((short *)e)[svb->bandno] );
			break;
			
		case IM_BANDFMT_UINT:
			set_glabel( svb->val, "%d", 
				((unsigned int *)e)[svb->bandno] );
			break;
			
		case IM_BANDFMT_INT:
			set_glabel( svb->val, "%d", 
				((int *)e)[svb->bandno] );
			break;
			
		case IM_BANDFMT_FLOAT:
			set_glabel( svb->val, "%g", 
				((float *)e)[svb->bandno] );
			break;
			
		case IM_BANDFMT_COMPLEX:
			set_glabel( svb->val, "(%g,%g)", 
				((float *)e)[svb->bandno << 1], 
				((float *)e)[(svb->bandno << 1) + 1] );
			break;
			
		case IM_BANDFMT_DOUBLE:
			set_glabel( svb->val, "%g", 
				((double *)e)[svb->bandno] );
			break;
			
		case IM_BANDFMT_DPCOMPLEX:
			set_glabel( svb->val, "(%g,%g)", 
				((double *)e)[svb->bandno << 1], 
				((double *)e)[(svb->bandno << 1) + 1] );
			break;

		default:
			set_glabel( svb->val, "???" );
			break;
		}
	else
		set_glabel( svb->val, "???" );

	return( NULL );
}

void 
statusview_mouse( Statusview *sv, int x, int y )
{
	Conversion *conv = sv->conv;
	IMAGE *im = imageinfo_get( FALSE, conv->ii );
	REGION *reg = conv->reg;

	x = IM_CLIP( 0, x, sv->conv->underlay.width - 1 );
	y = IM_CLIP( 0, y, sv->conv->underlay.height - 1 );

	/* Use pixels per mm in horizontal ruler to convert position
	 * to display units.
	set_label( bar->xpos, "%d", 
		(int) ((float) x / get_ruler_ppm( bar->scr->hrule )) );
	set_label( bar->ypos, "%d", 
		(int) ((float) y / get_ruler_ppm( bar->scr->vrule )) );
	 */

	set_glabel( sv->pos, "(%5d, %5d)", x - im->Xoffset, y - im->Yoffset ); 

	/* Update value list.
	 */
	if( reg->im->Coding == IM_CODING_LABQ )
		statusview_mouse_LABPACK( sv, x, y );
	else if( !conv->err ) {
		unsigned char *e;

		if( !(e = (unsigned char *) get_element( reg, x, y, 0 )) ) {
			conversion_error_set( conv );
			return;
		}

		slist_map( sv->bands, 
			(SListMapFn) statusview_mouse_band, e );
	}
}
