/* Types for the symbol table.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#define TYPE_SYMBOL (symbol_get_type())
#define SYMBOL( obj ) (GTK_CHECK_CAST( (obj), TYPE_SYMBOL, Symbol ))
#define SYMBOL_CLASS( klass ) \
	(GTK_CHECK_CLASS_CAST( (klass), TYPE_SYMBOL, SymbolClass ))
#define IS_SYMBOL( obj ) (GTK_CHECK_TYPE( (obj), TYPE_SYMBOL ))
#define IS_SYMBOL_CLASS( klass ) \
	(GTK_CHECK_CLASS_TYPE( (klass), TYPE_SYMBOL ))

/* The types of symbol we can have.
 */
typedef enum {
	SYM_VALUE,		/* Symbol with a value attached */
	SYM_PARAM,		/* A parameter to a user function */
	SYM_ZOMBIE,		/* A referred to but not defined */
	SYM_WORKSPACE,		/* A loaded workspace */
	SYM_WORKSPACEGROUP,	/* A collection of workspaces */
	SYM_ROOT,		/* The root symbol */
	SYM_EXTERNAL,		/* A reference to an external function */
	SYM_BUILTIN		/* A reference to a built-in function */
} SymType;

/* A symbol.
 */
struct _Symbol {
	Filemodel parent_class;

	/* Our name, and where we are.
	 */
	SymTable *tab;		/* Symbol table we are within */
	SymType type; 		/* The type of this symbol */

	int dest;		/* >0 if being deleted */

	/* Track during parse. A list of pointers to pointers to this
	 * symbol which we need to patch if we resolve to an outer scope.
	 */
	GSList *patch;

	/* Main expression for this sym.
	 */
	Expr *expr;

	/* Other expressions at this scope.
	 */
	GSList *ext_expr;

	/* Base of graph for value of this symbol. Use sym->expr->root to get
	 * value though .. we just hold pointer for GC here. Expressions on
	 * ext_expr have their GC handle in Subcolumn.
	 */
	Element base;		/* Value for this expr */

	/* Recomputation links. Use these to work out what to build next.
	 */
	gboolean dirty;		/* True if this sym needs recalc */
	GSList *parents;	/* Compile which refer to this sym */

	GSList *topchildren;	/* For top syms, all top-level children */
	GSList *topparents;	/* For top syms, all top-level parents */
	int ndirtychildren;	/* Number of dirty top syms we refer to */
	gboolean leaf;		/* True for in recomp set */

	/* X-tras for definitions.
	 */
	Tool *tool;		/* Tool and toolkit defined in */

	/* X-tras for SYM_EXTERNAL ... our im_function.
	 */
	im_function *function;	/* Function we run */
	int fn_nargs;		/* Number of args fn needs from nip */

	/* X-tras for SYM_BUILTIN ... our function.
	 */
	BuiltinInfo *builtin;
};

typedef struct _SymbolClass {
	FilemodelClass parent_class;

	/* Methods.
	 */
} SymbolClass;

GtkType symbol_get_type( void );

/* All symbols come off this.
 */
extern Symbol *symbol_root;
extern Symbol *symbol_root_root;

Symbol *symbol_map_all( Symbol *sym, symbol_map_fn fn, void *a, void *b );

Symbol *symbol_get_parent( Symbol *sym );
Workspace *symbol_get_workspace( Symbol *sym );
Tool *symbol_get_tool( Symbol *sym );

void symbol_qualified_name( Symbol *sym, BufInfo *buf );
void symbol_qualified_name_relative( Symbol *context, 
	Symbol *sym, BufInfo *buf );
const char *symbol_name( Symbol *sym );
void symbol_name_print( Symbol *sym );

void *symbol_patch_add( void **pnt, Symbol *sym );
void symbol_patch_pointers( Symbol *nsym, Symbol *osym );

void symbol_root_init();

void symbol_link( Symbol *sym, SymTable *tab, const char *name );
Symbol *symbol_new( SymTable *tab, const char *name );
Symbol *symbol_new_defining( Compile *Compile, const char *name );
Symbol *symbol_new_reference( Compile *compile, const char *name );
void symbol_made( Symbol *sym );

void *symbol_link_break( Symbol *child, Compile *compile );

gboolean symbol_user_init( Symbol *sym );
gboolean symbol_parameter_init( Symbol *sym, Compile *compile );
gboolean symbol_parameter_builtin_init( Symbol *sym );

Symbol *symbol_leaf_next( void );

void *symbol_sanity( Symbol *sym );
void symbol_leaf_set_sanity( void );
void *symbol_strip( Symbol *sym );

void symbol_state_change( Symbol *sym );

void symbol_recalculate_all_force( void );
void symbol_recalculate_all( void );
void *symbol_recalculate( Symbol *sym );
void *symbol_recalculate_check( Symbol *sym );

void symbol_help( Symbol *sym, BufInfo *buf );
