/* Manage symbol tables.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

*/

#include "ip.h"

/* 
#define DEBUG
#define DEBUG_RESOLVE
 */

/* Map a function over a symbol table. 
 */
Symbol *
stable_map( SymTable *tab, symbol_map_fn fn, void *a, void *b, void *c )
{	
	return( (Symbol *) slist_map3( tab->traverse, 
		(SListMap3Fn) fn, a, b, c ) );
}

/* Map a function over a symbol table. 
 */
Symbol *
stable_map_rev( SymTable *tab, symbol_map_fn fn, void *a, void *b, void *c )
{	
	return( (Symbol *) slist_map3_rev( tab->traverse, 
		(SListMap3Fn) fn, a, b, c ) );
}

/* Print out the name of a table ... go back up the symbol hierarchy.
 */
void
stable_name_print( SymTable *tab )
{
	Symbol *sym = tab->compile->sym;

	if( sym->type == SYM_ROOT )
		printf( "(table 0x%x)root", (unsigned int) tab );
	else {
		stable_name_print( sym->tab );
		printf( ".%s", MODEL( sym )->name );
	}
}

/* Get the enclosing table ... NULL for none.
 */
SymTable *
stable_get_parent( SymTable *tab )
{
	return( tab->compile->sym->tab );
}

/* Free a symbol table.
 */
void 
stable_destroy( SymTable *tab )
{
	while( tab->traverse )
		gtk_object_destroy( GTK_OBJECT( tab->traverse->data ) ); 

	FREE( tab->array );
	FREE( tab );
}

/* Make a new symbol table. 
 */
SymTable *
stable_new( Compile *compile, int size )
{
	SymTable *tab = IM_NEW( NULL, SymTable );
	int i;

	tab->compile = compile;
	tab->size = size;
	tab->sofar = 0;
	tab->array = NULL;
	tab->traverse = NULL;

	if( !(tab->array = IM_ARRAY( NULL, size, GSList * )) ) {
		stable_destroy( tab );
		return( NULL );
	}
	for( i = 0; i < size; i++ )
		tab->array[i] = NULL;

	return( tab );
}

/* Does a symbol have the right name? NULL if not, symbol if yes.
 */
static Symbol *
stable_find_sub( Symbol *sym, const char *name )
{	
	if( strcmp( MODEL( sym )->name, name ) == 0 )
		return( sym );
	else
		return( NULL );
}

/* Look up a symbol in a table. NULL for not found.
 */
Symbol *
stable_find( SymTable *tab, const char *name )
{
	int i = hash_name( name ) % tab->size;

	return( (Symbol *) slist_map( tab->array[i], 
		(SListMapFn) stable_find_sub, (void *) name ) );
}

/* Add a new symbol to a table. 
 */
void
stable_add( SymTable *tab, Symbol *sym )
{
	int i = hash_name( MODEL( sym )->name ) % tab->size;

	assert( !sym->tab && !stable_find( tab, MODEL( sym )->name ) );

	tab->array[i] = g_slist_prepend( tab->array[i], sym );
	tab->traverse = g_slist_append( tab->traverse, sym );
	sym->tab = tab;

	gtk_object_ref( GTK_OBJECT( sym ) );
	gtk_object_sink( GTK_OBJECT( sym ) );

#ifdef DEBUG
	printf( "stable_add: adding " );
	symbol_name_print( sym );
	printf( " to table " );
	stable_name_print( tab );
	printf( "\n" );
#endif /*DEBUG*/
}

/* Move a symbol to the end of the traverse order. Do this when we hit a
 * defining occurence of a previously referenced sym, so that traverse keeps
 * define order rather than reference order.
 */
void
stable_to_end( Symbol *sym )
{
	SymTable *tab = sym->tab;

	tab->traverse = g_slist_remove( tab->traverse, sym );
	tab->traverse = g_slist_append( tab->traverse, sym );
}

/* Unlink a symbol from a table. Called for symbol delete, and symbol scope
 * move.
 */
void
stable_unlink( Symbol *sym )
{	
	SymTable *tab = sym->tab;
	int i;

	/* root-most sym has no enclosing sym ... plus sometimes we are called
	 * for syms which haven't yet been linked.
	 */
	if( !tab ) 
		return;

	i = hash_name( MODEL( sym )->name ) % tab->size;

	tab->array[i] = g_slist_remove( tab->array[i], sym );
	tab->traverse = g_slist_remove( tab->traverse, sym );
	sym->tab = NULL;

	gtk_object_unref( GTK_OBJECT( sym ) );

#ifdef DEBUG
	printf( "stable_unlink: removing " );
	symbol_name_print( sym );
	printf( " from table " );
	stable_name_print( tab );
	printf( "\n" );
#endif /*DEBUG*/
}

/* Sub fn of below.
 */
static void *
stable_resolve_sub3( Compile *pnt, Symbol *sym )
{
	if( !g_slist_find( sym->parents, pnt ) ) 
		sym->parents = g_slist_prepend( sym->parents, pnt );

	return( NULL );
}

/* Sub fn 2 of below.
 */
static void *
stable_resolve_sub2( Compile *compile )
{
	return( symbol_fix_counts( compile->sym ) );
}

/* We've found a symbol which is the true definition of an unresolved symbol.
 * We fiddle references to zombie to refer to sym instead.
 */
static void
stable_resolve( Symbol *sym, Symbol *zombie )
{
#ifdef DEBUG_RESOLVE
	printf( "stable_resolve: resolving zombie " );
	symbol_name_print( zombie );
	printf( " to sym on table " );
	stable_name_print( sym->tab );
	printf( "\n" );
#endif /*DEBUG_RESOLVE*/

	/* Symbol on outer table. Patch pointers to zombie to point to
	 * sym instead.
	 */
	symbol_patch_pointers( sym, zombie );

	/* Also unresolved in outer scope?
	 */
	if( sym->type == SYM_ZOMBIE )
		/* We may need to move it again - so add the patch
		 * pointers we have just used to the patch list on
		 * sym.
		 */
		(void) slist_map( zombie->patch, 
			(SListMapFn) symbol_patch_add, sym );

	/* Add other information the ZOMBIE has picked up. We only
	 * need to make the link one way: the patching will make the
	 * other half for us.
	 */
	(void) slist_map( zombie->parents, 
		(SListMapFn) stable_resolve_sub3, sym );

	/* Make sure the dirty counts are set correctly. We have
	 * changed dep (maybe), so need a fiddle.
	 */
	(void) slist_map( zombie->parents, 
		(SListMapFn) stable_resolve_sub2, NULL );

	/* No one refers to the zombie now.
	 */
	FREEF( g_slist_free, zombie->parents );

	gtk_object_destroy( GTK_OBJECT( zombie ) );
}

/* Sub-function of below.
 */
static void *
stable_resolve_names_sub( Symbol *sym, SymTable *outer )
{
	const char *name = MODEL( sym )->name;
	Symbol *old;

	/* Is it the sort of thing we are looking for? ZOMBIEs only, please.
	 */
	if( sym->type != SYM_ZOMBIE )
		return( NULL );

	if( (old = stable_find( outer, name )) ) 
		stable_resolve( old, sym );
	else {
		/* Nothing on the outer table of that name. Can just move the
		 * symbol across.
		 */
		gtk_object_ref( GTK_OBJECT( sym ) );
		stable_unlink( sym );
		stable_add( outer, sym );
		gtk_object_unref( GTK_OBJECT( sym ) );
	}

	return( NULL );
}

/* End of definition parse: we search the symbol table we have built for this
 * definition, looking for unresolved names (ZOMBIEs). If we find any, we move
 * the zombie to the enclosing symbol table, since the name may be
 * resolved one level up. If we find a symbol on the enclosing table of the
 * same name, we have to patch pointers to our inner ZOMBIE to point to this
 * new symbol. Nasty!
 */
void
stable_resolve_names( SymTable *inner, SymTable *outer )
{
	(void) stable_map( inner, 
		(symbol_map_fn) stable_resolve_names_sub, outer, NULL, NULL );
}

/* Sub fn of below ... move refs to "root" out.
 */
static void *
stable_resolve_root( Symbol *sym )
{
	if( sym->type == SYM_ZOMBIE && 
		strcmp( MODEL( sym )->name, MODEL( symbol_root )->name ) == 0 )
		stable_resolve_names_sub( sym, 
			symbol_root_root->expr->compile->locals );

	return( NULL );
}

/* End of all scopes ... resolve everything out to top-level.
 */
void
stable_resolve_names_top( SymTable *inner )
{
	SymTable *top = symbol_root->expr->compile->locals;
	SymTable *tab;
	SymTable *outer;

	for( tab = inner; tab != top; tab = outer ) {
		outer = stable_get_parent( tab );

		stable_resolve_names( tab, outer );
	}

	/* Special case: references to "root" on symbol_root->expr->locals
	 * get resolved out one more level.
	 */
	(void) stable_map( top, 
		(symbol_map_fn) stable_resolve_root, NULL, NULL, NULL );
}


/* Search outwards for this sym.
 */
static void *
stable_resolve_dynamic_sub( Symbol *sym, SymTable *context )
{
	SymTable *tab;

	if( sym->type != SYM_ZOMBIE )
		return( NULL );

	for( tab = context; tab; tab = stable_get_parent( tab ) ) {
		Symbol *def = stable_find( tab, MODEL( sym )->name );

		if( def && def->type != SYM_ZOMBIE ) {
			/* We've found a non-zombie! Bind and we're done.
			 */
			stable_resolve( def, sym );
			break;
		}
	}

	return( NULL );
}

/* Resolve ZOMBIEs in tab by searching outwards from context. We only move 
 * and patch if we find a match ... otherwise we leave the zombie where is is.
 *
 * This is used for dynamic exprs in the tally display: we don't care about
 * fwd refs, but we want to be able to handle multiple scope contexts.
 */
void 
stable_resolve_dynamic( SymTable *tab, SymTable *context )
{
	(void) stable_map( tab, 
		(symbol_map_fn) stable_resolve_dynamic_sub, context, NULL,
		NULL );
}
