/* Manage toolkits and their display.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

#include "ip.h"

static FilemodelClass *parent_class = NULL;

Toolkit *
toolkit_map( toolkit_map_fn fn, void *a, void *b )
{
	return( (Toolkit *) model_map( MODEL( main_toolkitgroup ), 
		(model_map_fn) fn, a, b ) );
}

/* Remove a toolkit.
 */
static void 
toolkit_destroy( GtkObject *object )
{
	Toolkit *kit;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_TOOLKIT( object ) );

	kit = TOOLKIT( object );

#ifdef DEBUG
	printf( "toolkit_destroy %s\n", MODEL( kit )->name );
#endif /*DEBUG*/

	/* Kill all the symbols we contain.
	 */
	program_freeze();
	model_map( MODEL( kit ), 
		(model_map_fn) object_destroy, NULL, NULL );
	program_thaw();

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

static void
toolkit_changed( Model *model )
{
	/* If we change, signal change on our parent too (toolkitgroup) ...
	 * things like Program which need to spot any change to any kit can 
	 * connect to that, rather than having to connect to all kits
	 * independantly.
	 */
	if( model->parent )
		model_changed( model->parent );
}

static void
toolkit_info( Model *model, BufInfo *buf )
{
	Toolkit *kit = TOOLKIT( model );

	MODEL_CLASS( parent_class )->info( model, buf );

	buf_appendf( buf, "group = \"%s\"\n", MODEL( kit->kitg )->name );
}

static gboolean
toolkit_save_text( Model *model, iOpenFile *of )
{
	if( model_map( model, 
		(model_map_fn) model_save_text, of, NULL ) )
		return( FALSE );

	return( TRUE );
}

/* Load from an iOpenFile.
 */
static gboolean
toolkit_load_text( Model *model, Model *parent, iOpenFile *of )
{
	Toolkit *kit = TOOLKIT( model );
	int pos = model_pos_last( MODEL( model ) ) + 1;
	gboolean res;

	/* Load up definitions.
	 */
	program_freeze();
	filemodel_set_filename( FILEMODEL( model ), of->fname_real );
	attach_input_file( of );
	if( !(res = parse_toplevel( kit, pos )) ) 
		ierrors( "\"%s\"(%d): %s\n", FILEMODEL( kit )->filename, 
			input_state.lineno, error_string );
	program_thaw();

	model_changed( MODEL( kit ) );

#ifdef DEBUG
	(void) dump_kit( kit );
#endif /*DEBUG*/

	return( res );
}

static void 
toolkit_empty( Model *model )
{
	model_destroy_children( MODEL( model ) );
}

static void
toolkit_class_init( ToolkitClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	ModelClass *model_class = (ModelClass *) klass;
	FilemodelClass *filemodel_class = (FilemodelClass *) klass;

	parent_class = gtk_type_class( TYPE_FILEMODEL );

	object_class->destroy = toolkit_destroy;

	/* Create signals.
	 */

	/* Init methods.
	 */
	model_class->view_new = toolkitview_new;
	model_class->changed = toolkit_changed;
	model_class->info = toolkit_info;
	model_class->save_text = toolkit_save_text;
	model_class->load_text = toolkit_load_text;
	model_class->empty = toolkit_empty;

	filemodel_class->filetype = filesel_type_definition;
}

static void
toolkit_init( Toolkit *kit )
{
	kit->kitg = NULL;
	kit->pseudo = FALSE;
}

GtkType
toolkit_get_type( void )
{
	static GtkType toolkit_type = 0;

	if( !toolkit_type ) {
		static const GtkTypeInfo info = {
			"Toolkit",
			sizeof( Toolkit ),
			sizeof( ToolkitClass ),
			(GtkClassInitFunc) toolkit_class_init,
			(GtkObjectInitFunc) toolkit_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		toolkit_type = gtk_type_unique( TYPE_FILEMODEL, &info );
	}

	return( toolkit_type );
}

static void
toolkit_link( Toolkit *kit, Toolkitgroup *kitg, const char *name )
{
	model_set( MODEL( kit ), name, NULL );
	model_child_add( MODEL( kitg ), MODEL( kit ), -1 );
	kit->kitg = kitg;
	filemodel_register( FILEMODEL( kit ) );
	if( name[0] == '_' )
		MODEL( kit )->display = FALSE;
	toolkitgroup_sort( kitg );
}

/* Find a kit by kit name.
 */
Toolkit *
toolkit_find( Toolkitgroup *kitg, const char *name )
{	
	return( (Toolkit *) model_map( MODEL( kitg ), 
		(model_map_fn) model_test_name, (char *) name, NULL ) );
}

Toolkit *
toolkit_new( Toolkitgroup *kitg, const char *name )
{	
	Toolkit *kit;

	/* Exists already?
	 */
	if( (kit = toolkit_find( kitg, name )) ) 
		FREEO( kit );

	/* Make a new kit.
	 */
	kit = gtk_type_new( TYPE_TOOLKIT );
	toolkit_link( kit, kitg, name );

	return( kit );
}

Toolkit *
toolkit_new_filename( Toolkitgroup *kitg, const char *filename )
{
	char name[NAMELEN];
	Toolkit *kit;

	name_from_filename( filename, name );
	kit = toolkit_new( kitg, name );
	filemodel_set_filename( FILEMODEL( kit ), filename );

	return( kit );
}

/* Load a file as a toolkit.
 */
Toolkit *
toolkit_new_from_file( Toolkitgroup *kitg, const char *filename )
{
	Toolkit *kit = toolkit_new_filename( kitg, filename );

	if( !filemodel_load_all( FILEMODEL( kit ), MODEL( kitg ), filename ) ) 
		/* We want to leave the kit loaded, so the user can try to fix
		 * the problem.
		 */
		return( NULL );

	filemodel_set_modified( FILEMODEL( kit ), FALSE );

	return( kit );
}

/* Look up a toolkit, make an empty one if not there.
 */
Toolkit *
toolkit_by_name( const char *name )
{
	Toolkit *kit;

	if( !(kit = toolkit_find( main_toolkitgroup, name )) ) {
		char file[PATHLEN];

		im_snprintf( file, PATHLEN,
			"$HOME/.$PACKAGE-$VERSION/start/%s.def", name );
		kit = toolkit_new_filename( main_toolkitgroup, file );
	}

	return( kit );
}

static void *
toolkit_linkreport_sym_sym( Symbol *child, 
	Symbol *parent, BufInfo *buf, gboolean *bad_links )
{
	if( child->type == SYM_ZOMBIE ) {
		Tool *tool = symbol_get_tool( parent );

		buf_appendf( buf, "\"" );
		symbol_qualified_name( parent, buf );
		buf_appendf( buf, "\", tool \"%s\", toolkit \"%s\", "
			"refers to undefined symbol \"",
			MODEL( tool )->name, MODEL( tool->kit )->name );
		symbol_qualified_name( child, buf );
		buf_appendf( buf, "\"\n" );

		*bad_links = TRUE;
	}

	return( NULL );
}

static void *
toolkit_linkreport_sym( Symbol *sym, BufInfo *buf, gboolean *bad_links )
{
	if( sym->expr )
		return( slist_map3( sym->expr->compile->children,
			(SListMap3Fn) toolkit_linkreport_sym_sym, 
			sym, buf, bad_links ) );

	return( NULL );
}

static void *
toolkit_linkreport_tool( Tool *tool, BufInfo *buf, gboolean *bad_links )
{
	Symbol *sym;

	if( tool->type != TOOL_SYM )
		return( NULL );
	sym = tool->sym;

	return( symbol_map_all( tool->sym, 
		(symbol_map_fn) toolkit_linkreport_sym, buf, bad_links ) );
}

void *
toolkit_linkreport( Toolkit *kit, BufInfo *buf, gboolean *bad_links )
{
	return( model_map( MODEL( kit ), 
		(model_map_fn) toolkit_linkreport_tool, buf, bad_links ) );
}


