/* trace window
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

#include "ip.h"

/* Disassemble functions.
 */
#define TRACE_FUNCTIONS (watch_bool_get( "CALC_TRACE_FUNCTIONS", FALSE ))

/* OR of the flags in all the trace windows.
 */
TraceFlags trace_flags = (TraceFlags) 0;

static iWindowClass *parent_class = NULL;

/* All trace windows.
 */
static GSList *trace_all = NULL;

/* Trace buffer stack.
 */
static BufInfo trace_buffer_stack[SPINE_SIZE];
static int trace_buffer_stack_p = 0;

/* Number of active trace blocks.
 */
static int trace_block_count = 0;

void
trace_block( void )
{
	trace_block_count += 1;
}

void
trace_unblock( void )
{
	trace_block_count -= 1;

	assert( trace_block_count >= 0 );
}

void
trace_reset( void )
{
	int i;

	for( i = 0; i < trace_buffer_stack_p; i++ )
		buf_destroy( &trace_buffer_stack[i] );

	trace_buffer_stack_p = 0;
}

void
trace_check( void )
{
	assert( trace_buffer_stack_p == 0 );
}

BufInfo *
trace_push( void )
{
	int i;

	if( trace_buffer_stack_p >= SPINE_SIZE )
		everror( reduce_context, "trace buffer stack overflow" );

	i = trace_buffer_stack_p++;
	buf_init_dynamic( &trace_buffer_stack[i], MAX_TRACE );

	return( &trace_buffer_stack[i] );
}

void
trace_pop( void )
{
	int i;

	assert( trace_buffer_stack_p > 0 );

	i = --trace_buffer_stack_p;
	buf_destroy( &trace_buffer_stack[i] );
}

BufInfo *
trace_current( void )
{
	assert( trace_buffer_stack_p > 0 );

	return( &trace_buffer_stack[trace_buffer_stack_p - 1] );
}

int
trace_get_mark( void )
{
	return( trace_buffer_stack_p );
}

void
trace_pop_to( int n )
{
	assert( n >= 0 && n <= trace_buffer_stack_p );

	while( trace_buffer_stack_p > n ) 
		trace_pop();
}

static void *
trace_global_rethink_sub( Trace *trace )
{
	trace_flags |= trace->flags;

	return( NULL );
}

/* Rethink the global trace_flags.
 */
static void
trace_global_rethink( void )
{
	trace_flags = 0;

	slist_map( trace_all, (SListMapFn) trace_global_rethink_sub, NULL );
}

static void
trace_destroy( GtkObject *object )
{
	Trace *trace;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_TRACE( object ) );

	trace = TRACE( object );

	/* My instance destroy stuff.
	 */

	trace_all = g_slist_remove( trace_all, trace );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );

	trace_global_rethink();
}

static void
trace_class_init( TraceClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;

	parent_class = gtk_type_class( TYPE_IWINDOW );

	object_class->destroy = trace_destroy;

	/* Create signals.
	 */

	/* Init methods.
	 */
}

static void
trace_init( Trace *trace )
{
	trace->flags = 0;
}

GtkType
trace_get_type( void )
{
	static GtkType trace_type = 0;

	if( !trace_type ) {
		static const GtkTypeInfo info = {
			"Trace",
			sizeof( Trace ),
			sizeof( TraceClass ),
			(GtkClassInitFunc) trace_class_init,
			(GtkObjectInitFunc) trace_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		trace_type = gtk_type_unique( TYPE_IWINDOW, &info );
	}

	return( trace_type );
}

static void
trace_clear_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Trace *trace = TRACE( callback_data );

	gtk_editable_delete_text( GTK_EDITABLE( trace->text ), 0, -1 );
}

static void
trace_quit_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Trace *trace = TRACE( callback_data );

	iwindow_kill( IWINDOW( trace ) );
}

static void
trace_view_cb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Trace *trace = TRACE( callback_data );
	TraceFlags flag = (TraceFlags) callback_action;
	GtkCheckMenuItem *item = GTK_CHECK_MENU_ITEM( widget );

	if( item->active ) 
		trace->flags |= flag;
	else 
		trace->flags &= flag ^ ((TraceFlags) -1);

	trace_global_rethink();
}

/* Menu bar.
 */
static GtkItemFactoryEntry trace_menu_items[] = {
	{ "/_File",            			NULL,         
		NULL,			0, "<Branch>" },
	{ "/File/_Clear",       		NULL, 
		trace_clear_cb,     	0 },
	{ "/File/sep1",        			NULL,         
		NULL,       		0, "<Separator>" },
	{ "/File/_Quit",       			"<control>Q", 
		trace_quit_cb,     	0 },

	{ "/_View",       			NULL,		
		NULL,			0, "<Branch>" },
	{ "/View/Operators",			"", 
		trace_view_cb, 		TRACE_OPERATOR, "<CheckItem>" },
	{ "/View/Built in functions",		"", 
		trace_view_cb, 		TRACE_BUILTIN, "<CheckItem>" },
	{ "/View/Class construction",		"", 
		trace_view_cb, 		TRACE_CLASS_NEW, "<CheckItem>" },

	{ "/_Help",            			NULL,         
		NULL,			0, "<LastBranch>" },
	{ "/Help/_Trace ...",  			NULL,         
		box_help_cb,            GPOINTER_TO_UINT( "sec:trace" ) }
};

static void
trace_build( Trace *trace, GtkWidget *vbox )
{
	GtkAccelGroup *accel_group;
	GtkWidget *mbar;
	GtkWidget *swin;
	GtkAdjustment *hadj;
	GtkAdjustment *vadj;

	/* Make menu bar
	 */
	accel_group = gtk_accel_group_new();
	trace->ifac = gtk_item_factory_new( GTK_TYPE_MENU_BAR, 
		"<main>", accel_group );
	gtk_item_factory_create_items( trace->ifac, 
		IM_NUMBER( trace_menu_items ), trace_menu_items, trace );
	gtk_accel_group_attach( accel_group, GTK_OBJECT( trace ) );
	mbar = gtk_item_factory_get_widget( trace->ifac, "<main>" );
	gtk_box_pack_start( GTK_BOX( vbox ), mbar, FALSE, FALSE, 0 );
	gtk_widget_show( mbar );

	swin = gtk_scrolled_window_new( NULL, NULL );
	gtk_box_pack_start( GTK_BOX( vbox ), swin, TRUE, TRUE, 0 );
	hadj = gtk_scrolled_window_get_hadjustment( 
		GTK_SCROLLED_WINDOW( swin ) );
	vadj = gtk_scrolled_window_get_vadjustment( 
		GTK_SCROLLED_WINDOW( swin ) );
	gtk_widget_show( swin );

	trace->text = gtk_text_new( hadj, vadj );
	gtk_container_add( GTK_CONTAINER( swin ), trace->text );
	gtk_widget_show( trace->text );
}

static void
trace_link( Trace *trace )
{
        iwindow_set_title( IWINDOW( trace ), IP_NAME " - trace" );
	gtk_window_set_default_size( GTK_WINDOW( trace ), 640, 480 );
	iwindow_set_build( IWINDOW( trace ), 
		(iWindowBuildFn) trace_build, NULL, NULL, NULL );
	iwindow_build( IWINDOW( trace ) );
	trace_all = g_slist_prepend( trace_all, trace );

	gtk_widget_show( GTK_WIDGET( trace ) ); 
}

Trace *
trace_new( void )
{
	Trace *trace = gtk_type_new( TYPE_TRACE );

	trace_link( trace );

	return( trace );
}

static void *
trace_text_sub( Trace *trace, const char *buf, TraceFlags flags )
{
	if( !trace_block_count && trace->flags & flags ) {
		gtk_text_set_point( GTK_TEXT( trace->text ), 
			gtk_text_get_length( GTK_TEXT( trace->text ) ) );
		gtk_text_insert( GTK_TEXT( trace->text ),
			NULL, NULL, NULL, buf, strlen( buf ) );
	}

	return( NULL );
}

void
trace_text( TraceFlags flags, const char *fmt, ... )
{
	va_list ap;
 	char buf[MAX_STRSIZE];

	if( !(trace_flags & flags) )
		return;

        va_start( ap, fmt );
        (void) im_vsnprintf( buf, MAX_STRSIZE, fmt, ap );
        va_end( ap );

	slist_map2( trace_all, 
		(SListMap2Fn) trace_text_sub, buf, (void *) flags );
}

void
trace_pelement( PElement *pe )
{
	BufInfo *buf = trace_current();
	Heap *hi = reduce_context->hi;

	graph_pelement( hi, buf, pe, TRACE_FUNCTIONS );
}

void
trace_node( HeapNode *node )
{
	Element e;
	PElement pe;

	PEPOINTE( &pe, &e );
	PEPUTP( &pe, ELEMENT_NODE, node );
	trace_pelement( &pe );
}

void
trace_args( HeapNode **arg, int n )
{
	BufInfo *buf = trace_current();
	int i;

	for( i = 0; i < n; i++ ) {
		PElement rhs;

		PEPOINTRIGHT( arg[i], &rhs );
		trace_pelement( &rhs ); 
		buf_appends( buf, " " ); 
	}

	buf_appendf( buf, "->\n" ); 
}

void
trace_binop( PElement *left, BinOp bop, PElement *right )
{
	BufInfo *buf = trace_current();

	buf_appendf( buf, "\"%s\" ", decode_BinOp( bop ) );
	trace_pelement( left );
	buf_appendf( buf, " ", decode_BinOp( bop ) );
	trace_pelement( right );
	buf_appends( buf, " ->\n" ); 
}

void
trace_uop( UnOp uop, PElement *arg )
{
	BufInfo *buf = trace_current();

	buf_appendf( buf, "\"%s\" ", decode_UnOp( uop ) );
	trace_pelement( arg );
	buf_appends( buf, " ->\n" ); 
}

void
trace_result( TraceFlags flags, PElement *out )
{
	BufInfo *buf = trace_current();

	buf_appendf( buf, "    " ); 
	trace_pelement( out );
	buf_appends( buf, "\n" ); 

	trace_text( flags, buf_all( buf ) ); 
}
