/* Load and save workspace files.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

#include "ip.h"

#define AUTO_WS_SAVE (watch_bool_get( "CALC_AUTO_WS_SAVE", TRUE ))

static SymbolClass *parent_class = NULL;

static GSList *workspace_all = NULL;

Workspacegroup *
workspace_get_workspacegroup( Workspace *ws )
{
	return( WORKSPACEGROUP( MODEL( ws )->parent ) );
}

/* Over all workspaces.
 */
void *
workspace_map( workspace_map_fn fn, void *a, void *b )
{
	return( model_map( MODEL( main_workspacegroup ), 
		(model_map_fn) fn, a, b ) );
}

/* Map across the columns in a workspace.
 */
void *
workspace_map_column( Workspace *ws, column_map_fn fn, void *a )
{
	return( model_map( MODEL( ws ), (model_map_fn) fn, a, NULL ) );
}

/* Map across a Workspace, applying to the symbols of the top-level rows.
 */
void *
workspace_map_symbol( Workspace *ws, symbol_map_fn fn, void *a )
{
	return( model_map( MODEL( ws ), 
		(model_map_fn) column_map_symbol, (void *) fn, a ) );
}

/* Map a function over all selected rows in a workspace.
 */
void *
workspace_selected_map( Workspace *ws, row_map_fn fn, void *a, void *b )
{
	return( slist_map2( ws->selected, (SListMap2Fn) fn, a, b ) );
}

static void *
workspace_selected_map_sym_sub( Row *row, symbol_map_fn fn, void *a )
{
	return( fn( row->sym, a, NULL, NULL ) );
}

/* Map a function over all selected symbols in a workspace.
 */
void *
workspace_selected_map_sym( Workspace *ws, 
	symbol_map_fn fn, void *a, void *b )
{
	return( workspace_selected_map( ws,
		(row_map_fn) workspace_selected_map_sym_sub, (void *) fn, a ) );
}

/* Are there any selected rows?
 */
gboolean
workspace_selected_any( Workspace *ws )
{
	return( ws->selected != NULL );
}

/* Number of selected rows.
 */
int
workspace_selected_num( Workspace *ws )
{
	return( g_slist_length( ws->selected ) );
}

static void *
workspace_selected_sym_sub( Row *row, Symbol *sym )
{
	if( row->sym == sym )
		return( row );

	return( NULL );
}

/* Is sym selected?
 */
gboolean
workspace_selected_sym( Workspace *ws, Symbol *sym )
{
	return( workspace_selected_map( ws,
		(row_map_fn) workspace_selected_sym_sub, sym, NULL ) != NULL  );
}

/* Is just one row selected? If yes, return it.
 */
Row *
workspace_selected_one( Workspace *ws )
{
	int len = g_slist_length( ws->selected );

	if( len == 1 )
		return( (Row *)(ws->selected->data) );
	else if( len == 0 ) {
		ierrors( "No objects selected" );
		return( NULL );
	}
	else {
		ierrors( "More than one object selected" );
		return( NULL );
	}
}

static void *
workspace_deselect_all_sub( Column *col )
{
	col->last_select = NULL;

	return( NULL );
}

/* Deselect all rows.
 */
void
workspace_deselect_all( Workspace *ws )
{
	(void) workspace_selected_map( ws, 
		(row_map_fn) row_deselect, NULL, NULL );
	(void) workspace_map_column( ws, 
		(column_map_fn) workspace_deselect_all_sub, NULL );
}

/* Track this while we build a names list.
 */
typedef struct {
	Workspace *ws;
	BufInfo *buf;
	char *separator;
	gboolean first;
} NamesInfo;

/* Add a name to a string for a symbol.
 */
static void *
workspace_selected_names_sub( Row *row, NamesInfo *names )
{
	if( !names->first )
		buf_appends( names->buf, names->separator );
	row_qualified_name( row, names->buf );
	names->first = FALSE;

	return( NULL );
}

/* Add a list of selected symbol names to a string. 
 */
void
workspace_selected_names( Workspace *ws, BufInfo *buf, char *separator )
{
	NamesInfo names;

        names.ws = ws;
        names.buf = buf;
        names.separator = separator;
        names.first = TRUE;

	(void) workspace_selected_map( ws,
		(row_map_fn) workspace_selected_names_sub, &names, NULL );
}

/* Select all objects in all columns.
 */
void
workspace_select_all( Workspace *ws )
{
	(void) model_map( MODEL( ws ), 
		(model_map_fn) column_select_symbols, NULL, NULL );
}

/* Is there just one column, and is it empty? 
 */
Column *
workspace_is_one_empty( Workspace *ws )
{
	GSList *children = MODEL( ws )->children;
	Column *col;

	if( g_slist_length( children ) != 1 ) 
		return( NULL );

	col = COLUMN( children->data );
	if( !column_isempty( col ) )
		return( NULL );

	return( col );
}

/* Test the name field of a Column.
 */
static void *
workspace_column_name_test( Column *col, const char *str )
{
	if( strcmp( MODEL( col )->name, str ) == 0 )
		return( (void *) col );
	else
		return( NULL );
}

/* Search for a column by name.
 */
Column *
workspace_column_find( Workspace *ws, const char *name )
{
	Model *model;

	if( !(model = model_map( MODEL( ws ), 
		(model_map_fn) workspace_column_name_test, 
			(void *) name, NULL )) )
		return( NULL );

	return( COLUMN( model ) );
}

/* Return the column for a name ... an existing column, or a new one.
 */
Column *
workspace_column_get( Workspace *ws, const char *name )
{
	Column *col;

	/* Exists?
	 */
	if( (col = workspace_column_find( ws, name )) ) 
		return( col );

	/* No - build new column and return a pointer to that.
	 */
	return( column_new( ws, name ) );
}

/* Make up a new column name. Check for not already in workspace, and not in 
 * xml file (if columns non-NULL).
 */
char *
workspace_column_name_new( Workspace *ws, xmlNode *columns )
{
	char buf[ 256 ];

	/* Search for one not in use.
	 */
	for(;;) {
		number_to_string( ws->next++, buf );

		if( workspace_column_find( ws, buf ) ) 
			continue;
		if( columns ) {
			xmlNode *i;

			for( i = columns; i; i = i->next ) {
				char name[NAMELEN];

				if( strcmp( i->name, "Column" ) == 0 &&
					get_sprop( i, "name", name, NAMELEN ) )
					if( strcmp( name, buf ) == 0 )
						break;
			}

			if( i )
				continue;
		}

		return( im_strdup( NULL, buf ) );
	}
}

/* Make sure we have a column selected ... pick one of the existing columns; if 
 * there are none, make a column.
 */
Column *
workspace_column_pick( Workspace *ws )
{
	Column *col;

	if( ws->current )
		return( ws->current );

	if( MODEL( ws )->children ) {
		col = COLUMN( MODEL( ws )->children->data );
		workspace_column_select( ws, col );

		return( col );
	}

	/* Make an empty column.
	 */
	col = column_new( ws, "A" );
	workspace_column_select( ws, col );

	return( col );
}

/* Select a column. Can select NULL for no current col in this ws.
 */
void
workspace_column_select( Workspace *ws, Column *col )
{
	assert( !col || MODEL_IS_CHILD( ws, col ) ); 

	if( col && col == ws->current )
		return;

	if( ws->current ) {
		ws->current->selected = FALSE;
		model_changed( MODEL( ws->current ) );
	}

	ws->current = col;

	if( col ) {
		col->selected = TRUE;
		model_changed( MODEL( col ) );
	}
}

/* Make a new symbol, part of the current column.
 */
Symbol *
workspace_add_symbol( Workspace *ws )
{
	Column *col = workspace_column_pick( ws );
	Symbol *sym;
	char *name;

	name = column_name_new( col );
	sym = symbol_new( SYMBOL( ws )->expr->compile->locals, name );
	FREE( name );

	workspace_set_modified( sym );

	return( sym );
}

/* Make up a new definition.
 */
Symbol *
workspace_add_def( Workspace *ws, const char *str )
{
	Column *col = workspace_column_pick( ws );
	Symbol *sym;

        if( !str || strspn( str, " \t\b\n" ) == strlen( str ) )
		return( NULL );
	if( !(sym = workspace_add_symbol( ws )) )
		return( NULL );
	attach_input_string( str );
	if( !symbol_user_init( sym ) || 
		!parse_rhs( sym->expr, PARSE_RHS, FALSE ) )
		return( sym );
	(void) row_new( col->scol, sym, &sym->expr->root );
	symbol_made( sym );

	if( symbol_recalculate( sym ) ) 
		return( sym );

	return( sym );
}

/* Save just the selected objects.
 */
gboolean
workspace_selected_save( Workspace *ws, const char *filename )
{
	WorkspaceSaveType save = ws->save_type;

	ws->save_type = WORKSPACE_SAVE_SELECTED;
	if( !filemodel_save_all( FILEMODEL( ws ), filename ) ) {
		ws->save_type = save;
		unlink( filename );

		return( FALSE );
	}
	ws->save_type = save;

	return( TRUE );
}

/* Clone all selected symbols.
 */
gboolean 
workspace_clone_selected( Workspace *ws )
{
	char filename[FILENAME_MAX];

	/* Make a name for our clone file.
	 */
	if( !temp_name( filename, "ws" ) )
		return( FALSE );

	/* Save selected objects.
	 */
	if( !workspace_selected_save( ws, filename ) ) 
		return( FALSE );

	/* Try to load the clone file back again.
	 */
        set_hourglass();
	if( !workspace_merge_column_file( ws, filename ) ) {
		set_pointer();
		unlink( filename );

		return( FALSE );
	}
	set_pointer();
	unlink( filename );

	return( TRUE );
}

/* Keep the last WS_RETAIN workspaces as ipfl*.ws files.
 */
#define WS_RETAIN (10)

/* Array of names of workspace files we are keeping.
 */
static char *retain_files[ WS_RETAIN ] = { NULL };

/* The next one we allocate.
 */
static int retain_next = 0;

/* Save the workspace to one of our temp files.
 */
static int
workspace_checkmark_timeout( Workspace *ws )
{
	ws->auto_save_timeout = 0;

	if( !AUTO_WS_SAVE )
		return( FALSE );

	/* Do we have a name for this retain file?
	 */
	if( !retain_files[retain_next] ) {
		char filename[FILENAME_MAX];

		/* No name yet - make one up.
		 */
		if( !temp_name( filename, "ws" ) )
			return( FALSE );
		retain_files[retain_next] = im_strdup( NULL, filename );
	}
 
	if( !filemodel_save_all( FILEMODEL( ws ), retain_files[retain_next] ) )
		return( FALSE );

	retain_next = (retain_next + 1) % WS_RETAIN;

	return( FALSE );
}

/* Save the workspace to one of our temp files. Don't save directly (pretty
 * slow), instead set a timeout and save when we're quiet for >1s.
 */
void
workspace_checkmark( Workspace *ws )
{
	if( !AUTO_WS_SAVE )
		return;

	FREEFI( gtk_timeout_remove, ws->auto_save_timeout );

	ws->auto_save_timeout = gtk_timeout_add( 1000, 
		(GtkFunction) workspace_checkmark_timeout, ws );
}

/* On safe exit, remove all ws checkmarks.
 */
void
workspace_retain_clean( void )
{
	int i;

	for( i = 0; i < WS_RETAIN; i++ ) {
		if( retain_files[i] ) {
			unlink( retain_files[i] );
			FREE( retain_files[i] );
		}
	}
}

/* Track best-so-far file date here during search.
 */
static time_t date_sofar;

/* This file any better than the previous best candidate? Subfn of below.
 */
static char *
workspace_test_file( char *name, char *name_sofar )
{
	char buf[PATHLEN];
	struct stat st;
	int i;

	(void) expand_variables( name, buf );
	for( i = 0; i < WS_RETAIN; i++ )
		if( retain_files[i] && 
			strcmp( buf, retain_files[i] ) == 0 )
			return( NULL );
	if( stat( buf, &st ) == -1 )
		return( NULL );
#ifdef HAVE_GETEUID
	if( st.st_uid != geteuid() )
		return( NULL );
#endif /*HAVE_GETEUID*/
	if( st.st_size == 0 )
		return( NULL );
	if( date_sofar > 0 && st.st_mtime < date_sofar )
		return( NULL );
	
	strcpy( name_sofar, name );
	date_sofar = st.st_mtime;

	return( NULL );
}

/* Load a workspace, called from a yesno dialog.
 */
static void
workspace_auto_recover_load( iWindow *iwnd, 
	void *client, iWindowNotifyFn nfn, void *sys )
{
	char *filename = (char *) client;
	Workspace *ws;

	/* Load ws file.
	 */
	if( !(ws = workspace_new_from_file( main_workspacegroup, filename )) ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}
	workspacegroup_set_current( main_workspacegroup, ws );
	filemodel_set_filename( FILEMODEL( ws ), filename );
	symbol_recalculate_all();

	nfn( sys, IWINDOW_TRUE );
}

/* Do an auto-recover ... search for and load the most recent "ipfl*.ws" file 
 * in the tmp area owned by us, with a size > 0, that's not in our
 * retain_files[] set.
 */
gboolean
workspace_auto_recover( void )
{
	char *p;
	char *name;
	char buf[FILENAME_MAX];
	char buf2[FILENAME_MAX];

	/* Find the dir we are saving temp files to.
	 */
	if( !temp_name( buf, "ws" ) )
		return( FALSE );

	if( (p = strrchr( buf, IM_DIR_SEP )) )
		*p = '\0';

	date_sofar = -1;
	(void) path_map_dir( buf, "*.ws", 
		(path_map_fn) workspace_test_file, buf2 );
	if( date_sofar == -1 ) {
		ierrors( "no suitable workspace save files found in \"%s\"",
			buf );

		if( !AUTO_WS_SAVE )
			box_info( mainw_window,
				"you need to enable \"Auto workspace save\" in "
				"Preferences\n"
				"before automatic recovery works" );

		return( FALSE );
	}

	/* Tricksy ... free str with notify callack from yesno.
	 */
	name = im_strdupn( buf2 );
	box_yesno( mainw_window, 
		workspace_auto_recover_load, iwindow_true_cb, name, 
		(iWindowNotifyFn) im_free, name,
		"Load auto-saved workspace",
		"found workspace backup \"%s\"\n"
		"dated %s\n\n"
		"do you want to recover this workspace?",
		name, ctime( &date_sofar ) );

	return( TRUE );
}

static void
workspace_destroy( GtkObject *object )
{
	Workspace *ws;

#ifdef DEBUG
	printf( "workspace_destroy\n" );
#endif /*DEBUG*/

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_WORKSPACE( object ) );

	ws = WORKSPACE( object );

	ws->last_symbol = NULL;
	ws->last_column = NULL;

	workspace_all = g_slist_remove( workspace_all, ws );

	FREEFI( gtk_timeout_remove, ws->auto_save_timeout );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

static void
workspace_child_add( Model *parent, Model *child, int pos )
{
	Workspace *ws = WORKSPACE( parent );
	Column *col = COLUMN( child );

	MODEL_CLASS( parent_class )->child_add( parent, child, pos );

	if( col->selected )
		workspace_column_select( ws, col );
}

static void
workspace_child_remove( Model *parent, Model *child )
{
	Workspace *ws = WORKSPACE( parent );
	Column *col = COLUMN( child );

	/* Will we remove the current column? If yes, make sure
	 * current_column is NULL.
	 */
	if( col == ws->current )
		workspace_column_select( ws, NULL );

	filemodel_set_modified( FILEMODEL( ws ), TRUE );

	MODEL_CLASS( parent_class )->child_remove( parent, child );
}

static gboolean
workspace_load( Model *model, 
	ModelLoadState *state, Model *parent, xmlNode *xnode )
{
	Workspace *ws = WORKSPACE( model );
	Symbol *sym = SYMBOL( ws );

	if( !IS_WORKSPACEGROUP( parent ) ) {
		ierrors( "workspace_load: can only add a workspace to a "
			"workspacegroup" );
		return( FALSE );
	}

	sym->type = SYM_WORKSPACE;
	sym->expr = expr_new( sym );
	(void) compile_new_root( sym->expr );

	if( !MODEL_CLASS( parent_class )->load( model, state, parent, xnode ) )
		return( FALSE );

	return( TRUE );
}

static xmlNode *
workspace_save( Model *model, xmlNode *xnode )
{
	xmlNode *xthis;

	if( !(xthis = MODEL_CLASS( parent_class )->save( model, xnode )) )
		return( NULL );

	return( xthis );
}

/* Rename an XML node, if necessary, and add to global rename list. Pass in
 * the context in which we should check for conflicts.
 */
static void
workspace_rename_symbol_node( xmlNode *xnode, 
	SymTable *tab, ModelLoadState *state )
{
	char name[NAMELEN];
	char old_name[NAMELEN];

	if( !get_sprop( xnode, "name", name, NAMELEN ) )
		return;
	im_strncpy( old_name, name, NAMELEN );

	while( stable_find( tab, name ) )
		increment_name( name );

	(void) set_prop( xnode, "name", name );
	(void) model_loadstate_rename_new( state, old_name, name );
}

static void
workspace_rename_row_node( ModelLoadState *state, Column *col, xmlNode *xnode )
{
	char name[NAMELEN];

	if( strcmp( xnode->name, "Row" ) == 0 &&
		get_sprop( xnode, "name", name, NAMELEN ) ) {
		char *new_name;

		new_name = column_name_new( col );
		(void) set_prop( xnode, "name", new_name );
		(void) model_loadstate_rename_new( state, name, new_name );
		FREE( new_name );
	}
}

/* Rename column if there's one of that name in workspace. 
 */
static void
workspace_rename_column_node( Workspace *ws, 
	ModelLoadState *state, xmlNode *xnode, xmlNode *columns )
{
	char name[NAMELEN];

	if( strcmp( xnode->name, "Column" ) == 0 &&
		get_sprop( xnode, "name", name, NAMELEN ) &&
		model_map( MODEL( ws ), 
			(model_map_fn) model_test_name, name, NULL ) ) {
		char *new_name;
		Column *col;
		xmlNode *i;

		/* Exists already ... rename this column.
		 */
		new_name = workspace_column_name_new( ws, columns );
		col = column_new( ws, new_name );

#ifdef DEBUG
		printf( "workspace_rename_column_node: renaming column "
			"%s to %s\n", 
			name, new_name );
#endif /*DEBUG*/

		(void) set_prop( xnode, "name", new_name );
		FREE( new_name );

		/* And allocate new names for all rows in the subcolumn.
		 */
		for( i = get_node( xnode, "Subcolumn" )->children; 
			i; i = i->next ) 
			workspace_rename_row_node( state, col, i );

		FREEO( col );
	}
}

/* If there's a "name" property, reset from filename.
 */
static void
workspace_set_name_from_filename( xmlNode *xnode, const char *filename )
{
	char name[NAMELEN];

	if( get_sprop( xnode, "name", name, NAMELEN ) ) {
		name_from_filename( filename, name );
		(void) set_prop( xnode, "name", name );
	}
}

static gboolean
workspace_top_load( Filemodel *filemodel,
	ModelLoadState *state, Model *parent, xmlNode *xnode )
{
	Workspace *ws = WORKSPACE( filemodel );
	Workspace *current_ws;
	Column *current_col;
	xmlNode *i, *j, *k;

	switch( ws->load_type ) {
	case WORKSPACE_LOAD_TOP:
		/* Easy ... ws is a blank Workspace we are loading into. No
		 * renaming needed, except for the ws.
		 */

		/* Set the workspace name from the filename, ignoring the name
		 * saved in the file.
		 */
		workspace_set_name_from_filename( xnode, state->filename );
		workspace_rename_symbol_node( xnode, 
			SYMBOL( parent )->expr->compile->locals, state );

		if( model_load( MODEL( ws ), state, parent, xnode ) )
			return( FALSE );

		break;

	case WORKSPACE_LOAD_COLUMNS:
		current_ws = main_workspacegroup->current;

		/* Load at column level ... rename columns which clash with 
		 * columns in the current workspace. Also look out for clashes
		 * with columns we will load.
		 */
		for( i = xnode->children; i; i = i->next ) 
			workspace_rename_column_node( current_ws, 
				state, i, xnode->children );

		/* Load those columns.
		 */
		for( i = xnode->children; i; i = i->next ) 
			if( !model_new_xml( state, MODEL( current_ws ), i ) )
				return( FALSE );

		break;

	case WORKSPACE_LOAD_ROWS:
		current_ws = main_workspacegroup->current;
		current_col = workspace_column_pick( current_ws );

		/* Rename all rows into current column ... loop over column,
		 * subcolumns, rows.
		 */
		for( i = xnode->children; i; i = i->next ) 
			for( j = i->children; j; j = j->next ) 
				for( k = j->children; k; k = k->next ) 
					workspace_rename_row_node( state, 
						current_col, k );

		/* And load rows.
		 */
		for( i = xnode->children; i; i = i->next ) 
			for( j = i->children; j; j = j->next ) 
				for( k = j->children; k; k = k->next ) 
					if( !model_new_xml( state, 
						MODEL( current_col->scol ), 
						k ) )
						return( FALSE );

		break;

	default:
		assert( FALSE );
	}

	return( FILEMODEL_CLASS( parent_class )->top_load( filemodel, 
		state, parent, xnode ) );
}

static void
workspace_class_init( WorkspaceClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	ModelClass *model_class = (ModelClass *) klass;
	FilemodelClass *filemodel_class = (FilemodelClass *) klass;

	parent_class = gtk_type_class( TYPE_SYMBOL );

	object_class->destroy = workspace_destroy;

	/* Create signals.
	 */

	/* Init methods.
	 */
	model_class->view_new = workspaceview_new;
	model_class->child_add = workspace_child_add;
	model_class->child_remove = workspace_child_remove;
	model_class->load = workspace_load;
	model_class->save = workspace_save;

	filemodel_class->top_load = workspace_top_load;
	filemodel_class->filetype = filesel_type_workspace;

	/* Static init.
	 */
	model_register_loadable( MODEL_CLASS( klass ) );
}

static void
workspace_init( Workspace *ws )
{
	ws->next = 0;
	ws->current = NULL;
	ws->selected = NULL;
	ws->errors = NULL;
	ws->ontop = FALSE;

	ws->load_type = WORKSPACE_LOAD_TOP;
	ws->save_type = WORKSPACE_SAVE_ALL;

	ws->last_symbol = NULL;
	ws->last_column = NULL;

	ws->area.left = 0;
	ws->area.top = 0;
	ws->area.width = 0;
	ws->area.height = 0;
	ws->vp = ws->area;

	ws->auto_save_timeout = 0;

	filemodel_register( FILEMODEL( ws ) );

	workspace_all = g_slist_prepend( workspace_all, ws );
}

GtkType
workspace_get_type( void )
{
	static GtkType workspace_type = 0;

	if( !workspace_type ) {
		static const GtkTypeInfo info = {
			"Workspace",
			sizeof( Workspace ),
			sizeof( WorkspaceClass ),
			(GtkClassInitFunc) workspace_class_init,
			(GtkObjectInitFunc) workspace_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		workspace_type = gtk_type_unique( TYPE_SYMBOL, &info );
	}

	return( workspace_type );
}

Workspace *
workspace_new( Workspacegroup *wsg, const char *name )
{
	Workspace *ws;
	Symbol *sym;

	if( stable_find( SYMBOL( wsg )->expr->compile->locals, name ) ) {
		ierrors( "workspace \"%s\" already exists", name );
		return( NULL );
	}

	ws = gtk_type_new( TYPE_WORKSPACE );
	sym = SYMBOL( ws );

	symbol_link( sym, SYMBOL( wsg )->expr->compile->locals, name );
	sym->type = SYM_WORKSPACE;
	sym->expr = expr_new( sym );
	(void) compile_new_root( sym->expr );
	model_set( MODEL( ws ), name, NULL );
	model_child_add( MODEL( wsg ), MODEL( ws ), -1 );
	symbol_made( sym );

	(void) workspace_column_pick( ws );

	return( ws );
}

/* New workspace from a file.
 */
Workspace *
workspace_new_from_file( Workspacegroup *wsg, const char *filename )
{
	Workspace *ws;

	ws = gtk_type_new( TYPE_WORKSPACE );
	ws->load_type = WORKSPACE_LOAD_TOP;
	if( !filemodel_load_all( FILEMODEL( ws ), 
		MODEL( wsg ), filename ) ) {
		gtk_object_destroy( GTK_OBJECT( ws ) );
		return( NULL );
	}

	filemodel_set_modified( FILEMODEL( ws ), FALSE );
	filemodel_set_filename( FILEMODEL( ws ), filename );

	return( ws );
}

/* Merge file into this workspace.
 */
gboolean
workspace_merge_file( Workspace *ws, const char *filename )
{
	ws->load_type = WORKSPACE_LOAD_COLUMNS;
	column_set_offset( IM_RECT_RIGHT( &ws->area ), 0 );
	if( !filemodel_load_all( FILEMODEL( ws ), 
		MODEL( ws )->parent, filename ) ) 
		return( FALSE );

	return( TRUE );
}

/* Merge file into the current column of this workspace.
 */
gboolean
workspace_merge_column_file( Workspace *ws, const char *filename )
{
	ws->load_type = WORKSPACE_LOAD_ROWS;
	column_set_offset( IM_RECT_RIGHT( &ws->area ), 
		IM_RECT_BOTTOM( &ws->area ) );
	if( !filemodel_load_all( FILEMODEL( ws ), 
		MODEL( ws )->parent, filename ) ) 
		return( FALSE );

	return( TRUE );
}

/* A symbol has been edited ... set 'dirty' on the workspace if this change
 * will affect the save file.
 */
void 
workspace_set_modified( Symbol *sym )
{
	Workspace *ws;

	if( is_top( sym ) && (ws = symbol_get_workspace( sym )) )
		filemodel_set_modified( FILEMODEL( ws ), TRUE );
}

/* Get the bottom row from the current column.
 */
Row *
workspace_get_bottom( Workspace *ws )
{
	return( column_get_bottom( workspace_column_pick( ws ) ) );
}

void
workspace_symbol_action( Workspace *ws, Symbol *sym )
{
	Column *col = workspace_column_pick( ws );
	int nparam = sym->expr->compile->nparam;

	char str[ MAX_STRSIZE ];
	BufInfo buf;

	/* Are there any selected symbols?
	 */
	buf_init_static( &buf, str, MAX_STRSIZE );
	if( nparam > 0 && workspace_selected_any( ws ) ) {
		symbol_qualified_name( sym, &buf );

		if( nparam != workspace_selected_num( ws ) ) {
			ierrors( "\"%s\" needs %d arguments\n"
				"You have selected %d", 
				buf_all( &buf ), nparam,
				workspace_selected_num( ws ) );
			box_alert( NULL );
			return;
		}

		buf_appends( &buf, " " );
		workspace_selected_names( ws, &buf, " " );
		if( buf_isfull( &buf ) ) {
			ierrors( "Too many names selected" );
			box_alert( NULL );
			return;
		}

		if( !mainw_def_new( buf_all( &buf ) ) ) {
			box_alert( NULL );
			return;
		}
		workspace_deselect_all( ws );
	}
	else {
		/* Try to use the previous n items in this column as the
		 * arguments. 
		 */
		symbol_qualified_name( sym, &buf );
		if( !column_add_n_names( col, &buf, nparam ) || 
			!mainw_def_new( buf_all( &buf ) ) ) {
			box_alert( NULL );
			return;
		}
	}
}

/* Done button hit.
 */
static void
workspace_done_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	static int left = 0;
	static int top = 0;

	Workspace *ws = WORKSPACE( client );
	Stringset *ss = STRINGSET( iwnd );
	StringsetChild *name = stringset_child_get( ss, "Name" );
	StringsetChild *caption = stringset_child_get( ss, "Caption" );

	Column *col;

	char name_text[1024];
	char caption_text[1024];

	if( !get_geditable_name( name->entry, name_text, 1024 ) ||
		!get_geditable_string( caption->entry, caption_text, 1024 ) ||
		!(col = column_new( ws, name_text )) ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}
	model_set( MODEL( col ), NULL, caption_text );

	/* Pick a position for it.
	 */
	left += 10;
	top += 10;
	if( top > 200 )
		top = 0;
	col->x = left + ws->vp.left;
	col->y = top + ws->vp.top;

	workspace_column_select( ws, col );

	nfn( sys, IWINDOW_TRUE );
}

/* Pop up a new column editor.
 */
void
workspace_new_column( Workspace *ws )
{
	GtkWidget *ss = stringset_new();
	char *name;

	name = workspace_column_name_new( ws, NULL );
	stringset_child_new( STRINGSET( ss ), 
		"Name", name, "Set column name here" );
	FREE( name );
	stringset_child_new( STRINGSET( ss ), 
		"Caption", "", "Set column caption here" );

	iwindow_set_title( IWINDOW( ss ), "New column" );
	idialog_set_callbacks( IDIALOG( ss ), 
		iwindow_true_cb, NULL, NULL, NULL, ws );
	idialog_add_ok( IDIALOG( ss ), workspace_done_cb, "Create" );
	idialog_set_parent( IDIALOG( ss ), mainw_window );
	iwindow_build( IWINDOW( ss ) );

	gtk_widget_show( ss );
}

int
workspace_number( void )
{
	return( g_slist_length( workspace_all ) );
}
