/* a workspaceview button in a workspace
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/* 
#define DEBUG
 */

#include "ip.h"

/* Define to do true pointer grabs on workspace drag scroll.
 */
#define GRAB

/* Little quark for linking notebook pages back to workspace symbols.
 */
GQuark workspaceview_quark = 0;

static GtkWidget *workspaceview_popup_menu = NULL;

static ViewClass *parent_class = NULL;

/* Scroll by an amount horizontally and vertically.
 */
static void
workspaceview_scroll_to( Workspaceview *wview, int x, int y )
{
	GtkAdjustment *hadj = gtk_scrolled_window_get_hadjustment( 
		GTK_SCROLLED_WINDOW( wview->window ) );
	GtkAdjustment *vadj = gtk_scrolled_window_get_vadjustment( 
		GTK_SCROLLED_WINDOW( wview->window ) );
        int nx, ny;

        nx = IM_CLIP( 0, x, wview->width - wview->vp.width );
        ny = IM_CLIP( 0, y, wview->height - wview->vp.height );

	adjustments_set_value( hadj, vadj, nx, ny );
}

/* Scroll by an amount horizontally and vertically.
 */
static void
workspaceview_displace( Workspaceview *wview, int u, int v )
{
	workspaceview_scroll_to( wview, wview->vp.left + u, wview->vp.top + v );
}

/* Scroll to make an xywh area visible. If the area is larger than the
 * viewport, position the view at the bottom left if the xywh area ... 
 * this is usually right for workspaces.
 */
void
workspaceview_scroll( Workspaceview *wview, int x, int y, int w, int h )
{
	const int margin = 64; 		/* overscroll by this much */
	GtkAdjustment *hadj = gtk_scrolled_window_get_hadjustment( 
		GTK_SCROLLED_WINDOW( wview->window ) );
	GtkAdjustment *vadj = gtk_scrolled_window_get_vadjustment( 
		GTK_SCROLLED_WINDOW( wview->window ) );
	float nx, ny;

#ifdef DEBUG
	printf( "workspaceview_scroll: x=%d, y=%d, w=%d, h=%d\n", x, y, w, h );
#endif /*DEBUG*/

	nx = hadj->value;
	if( x + w < wview->vp.left + margin )
                nx = IM_MAX( 0, x - margin );
	else if( x > IM_RECT_RIGHT( &wview->vp ) - margin )
                nx = IM_MIN( IM_MAX( 0, x - margin ), 
			x + w - wview->vp.width + margin );

	ny = vadj->value;
	if( y + h < wview->vp.top + margin )
                ny = IM_MAX( 0, y - margin );
	else if( y > IM_RECT_BOTTOM( &wview->vp ) - margin )
                ny = IM_MIN( IM_MAX( 0, y - margin ),
			y + h - wview->vp.height + margin );

	adjustments_set_value( hadj, vadj, nx, ny );
}

/* Update our geometry from the fixed widget.
 */
static void
workspaceview_scroll_update( Workspaceview *wview )
{
	Workspace *ws = WORKSPACE( VIEW( wview )->model );
	GtkAdjustment *hadj = gtk_scrolled_window_get_hadjustment( 
		GTK_SCROLLED_WINDOW( wview->window ) );
	GtkAdjustment *vadj = gtk_scrolled_window_get_vadjustment( 
		GTK_SCROLLED_WINDOW( wview->window ) );

        wview->vp.left = hadj->value;
        wview->vp.top = vadj->value;
        wview->vp.width = hadj->page_size;
        wview->vp.height = vadj->page_size;

        wview->width = hadj->upper;
        wview->height = vadj->upper;

	/* Update vp hint in model too.
	 */
	ws->vp = wview->vp;

#ifdef DEBUG
	printf( "workspaceview_scroll_update:\n" );
	printf( "  wview->vp: l=%d, t=%d, w=%d, h=%d; fixed w=%d; h=%d\n",
		wview->vp.left, wview->vp.top, 
		wview->vp.width, wview->vp.height,
		wview->width, wview->height );
#endif /*DEBUG*/
}

static void
workspaceview_realize_cb( GtkWidget *wid )
{
	GdkEventMask mask;

	assert( wid->window );

	mask = gdk_window_get_events( wid->window );
	mask |= GDK_BUTTON_PRESS_MASK;
	gdk_window_set_events( wid->window, mask );
}

void
workspaceview_set_cursor( Workspaceview *wview, iWindowShape shape )
{
	if( !wview->context ) 
		wview->context = iwindow_cursor_context_new( 
			IWINDOW( view_get_toplevel( VIEW( wview ) ) ) );

	iwindow_cursor_context_set_cursor( wview->context, shape );
}

/* Event in workspaceview fixed.
 */
static gboolean
workspaceview_fixed_event_cb( GtkWidget *widget, 
	GdkEvent *ev, Workspaceview *wview )
{
	gboolean used = FALSE;

        switch( ev->type ) {
        case GDK_BUTTON_PRESS:
                if( ev->button.button == 1 ) {
			Workspace *ws = WORKSPACE( VIEW( wview )->model );

			workspace_deselect_all( ws );
                        used = TRUE;
                }
		else if( ev->button.button == 2 ) {
			wview->drag_x = ev->button.x_root + wview->vp.left;
			wview->drag_y = ev->button.y_root + wview->vp.top;
			workspaceview_set_cursor( wview, IWINDOW_SHAPE_MOVE );
			wview->dragging = TRUE;

#ifdef GRAB
                        gdk_pointer_grab( GTK_WIDGET( wview->fixed )->window,
                                FALSE,
                                GDK_BUTTON2_MOTION_MASK |
                                GDK_BUTTON_RELEASE_MASK,
                                NULL, NULL, ev->button.time );
                        gtk_grab_add( GTK_WIDGET( wview->fixed ) );
#endif /*GRAB*/

                        used = TRUE;
		}

                break;

        case GDK_BUTTON_RELEASE:
                if( ev->button.button == 2 ) {
#ifdef GRAB
			gtk_grab_remove( GTK_WIDGET( wview->fixed ) );
			gdk_pointer_ungrab( ev->button.time );
#endif /*GRAB*/

			workspaceview_set_cursor( wview, IWINDOW_SHAPE_NONE );
			wview->dragging = FALSE;
                        used = TRUE;
                }

                break;

        case GDK_MOTION_NOTIFY:
                if( ev->motion.state & GDK_BUTTON2_MASK ) {
			workspaceview_scroll_to( wview, 
				wview->drag_x - ev->motion.x_root,
				wview->drag_y - ev->motion.y_root );

                        used = TRUE;
                }

                break;

        default:
                break;
        }

	if( used ) 
		gtk_signal_emit_stop_by_name( GTK_OBJECT( widget ), "event" );

        return( used );
}

static void
workspaceview_scroll_adjustment_cb( GtkAdjustment *adj, Workspaceview *wview )
{
	workspaceview_scroll_update( wview );
}

/* Timer callback for background scroll.
 */
/*ARGSUSED*/
static gint
workspaceview_scroll_time_cb( Workspaceview *wview )
{
	/* Perform scroll.
	 */
	workspaceview_scroll_update( wview );
	if( wview->u != 0 || wview->v != 0 ) 
		workspaceview_displace( wview, wview->u, wview->v );

	/* Start timer again.
	 */
	return( TRUE );
}

/* Stop the tally_scroll timer.
 */
static void
workspaceview_scroll_stop( Workspaceview *wview )
{	
        FREEFI( gtk_timeout_remove, wview->timer );
}

/* Start the tally_scroll timer.
 */
static void
workspaceview_scroll_start( Workspaceview *wview )
{
	workspaceview_scroll_stop( wview );
        wview->timer = gtk_timeout_add( 30,
                (GtkFunction) workspaceview_scroll_time_cb, wview );
}

/* Set a background scroll. Pass both zero to stop scroll.
 */
void
workspaceview_scroll_background( Workspaceview *wview, int u, int v )
{
	wview->u = u;
	wview->v = v;

	if( u == 0 && v == 0 )
		workspaceview_scroll_stop( wview );
	else
		workspaceview_scroll_start( wview );
}

static void
workspaceview_destroy( GtkObject *object )
{
	Workspaceview *wview;
	GtkNotebook *nb;

#ifdef DEBUG
	printf( "workspaceview_destroy\n" );
#endif /*DEBUG*/

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_WORKSPACEVIEW( object ) );

	wview = WORKSPACEVIEW( object );
	nb = GTK_NOTEBOOK( wview->wgview->notebook );

	/* Instance destroy.
	 */
	workspaceview_scroll_stop( wview );
	gtk_notebook_remove_page( nb, 
		gtk_notebook_page_num( nb, wview->frame ) );

	FREEF( iwindow_cursor_context_destroy, wview->context );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

static void
workspaceview_link( View *view, Model *model, View *parent )
{
	Workspaceview *wview = WORKSPACEVIEW( view ); 
	Workspacegroupview *wgview = WORKSPACEGROUPVIEW( parent );

	GtkNotebook *nb = GTK_NOTEBOOK( wgview->notebook );

	VIEW_CLASS( parent_class )->link( view, model, parent );

	wview->wgview = wgview;

	/* Make sure we don't trigger workspacegroupview_switch_page_cb()
	 */
	gtk_signal_handler_block_by_data( GTK_OBJECT( nb ), wgview );
	gtk_notebook_append_page( nb, wview->frame, wview->tab );
	gtk_signal_handler_unblock_by_data( GTK_OBJECT( nb ), wgview );
}

static void *
workspaceview_child_size_sub( Columnview *cview, Rect *area )
{
	int x, y, w, h;
	Rect col;

	columnview_get_position( cview, &x, &y, &w, &h );

	col.left = x;
	col.top = y;
	col.width = w;
	col.height = h;

	im_rect_unionrect( area, &col, area );

	return( NULL );
}

static void
workspaceview_child_size_cb( Columnview *cview, 
	GtkAllocation *allocation, Workspaceview *wview )
{
	Workspace *ws = WORKSPACE( VIEW( wview )->model );

	int right, bottom;

	assert( IS_WORKSPACEVIEW( wview ) );

	/* Compute a new boundung box for our children.
	 */
	wview->bounding.left = 0;
	wview->bounding.top = 0;
	wview->bounding.width = 0;
	wview->bounding.height = 0;

	(void) view_map( VIEW( wview ),
		(view_map_fn) workspaceview_child_size_sub, 
		&wview->bounding, NULL );

	wview->bounding.width += 3;
	wview->bounding.height += 30;

#ifdef DEBUG
	printf( "workspaceview_child_size_cb: "
		"bb left=%d, top=%d, width=%d, height=%d\n",
		wview->bounding.left,
		wview->bounding.top,
		wview->bounding.width,
		wview->bounding.height );
#endif /*DEBUG*/

	/* Resize our fixed if necessary.
	 */
	right = IM_RECT_RIGHT( &wview->bounding );
	bottom = IM_RECT_BOTTOM( &wview->bounding );
	if( right != wview->width || bottom != wview->height ) {
		/* Update the model hints ... it uses bounding to position
		 * loads and saves.
		 */
		ws->area = wview->bounding;
		filemodel_set_offset( FILEMODEL( ws ), 
			ws->area.left, ws->area.top );
	}
}

static void
workspaceview_child_add( View *parent, View *child )
{
	assert( IS_WORKSPACEVIEW( parent ) );
	assert( IS_COLUMNVIEW( child ) );

	gtk_signal_connect( GTK_OBJECT( child ), "size_allocate", 
		GTK_SIGNAL_FUNC( workspaceview_child_size_cb ), parent );

	VIEW_CLASS( parent_class )->child_add( parent, child );
}

static void 
workspaceview_refresh( View *view )
{
	Workspaceview *wview = WORKSPACEVIEW( view );
	Workspace *ws = WORKSPACE( VIEW( wview )->model );
	GtkNotebook *nb = GTK_NOTEBOOK( wview->wgview->notebook );

#ifdef DEBUG
	printf( "workspaceview_refresh\n" );
#endif /*DEBUG*/

	if( MODEL( ws )->caption )
		set_tooltip( wview->tab, "%s", MODEL( ws )->caption );
	set_glabel( wview->label, "%s", MODEL( ws )->name );

	if( ws->ontop ) {
		gint page = gtk_notebook_page_num( nb, wview->frame );

		gtk_notebook_set_page( nb, page );
		mainw_select_message_update();
	}

	if( ws->ontop ) 
		mainw_refresh_title();

	if( ws->ontop )
		gtk_widget_set_state( GTK_WIDGET( wview->tab ),
			GTK_STATE_NORMAL );
	else
		gtk_widget_set_state( GTK_WIDGET( wview->tab ),
			GTK_STATE_ACTIVE );

	VIEW_CLASS( parent_class )->refresh( view );
}

static void
workspaceview_clone_cb( GtkWidget *menu, GtkWidget *host, View *view )
{
	Workspaceview *wview = WORKSPACEVIEW( view ); 
	Workspace *ws = WORKSPACE( VIEW( wview )->model );
	Workspace *nws;
	char filename[4096];

	/* Make a name for our clone file.
	 */
	if( !temp_name( filename, "ws" ) ||
		!filemodel_save_all( FILEMODEL( ws ), filename ) ) {
		box_alert( NULL );
		return;
	}

	/* Try to load the clone file back again.
	 */
        set_hourglass();
	if( !(nws = workspace_new_from_file( main_workspacegroup, 
		filename )) ) {
		set_pointer();
		unlink( filename );
		box_alert( NULL );
		return;
	}
	unlink( filename );
	workspacegroup_set_current( main_workspacegroup, nws );
	set_pointer();
	symbol_recalculate_all();

	/* Get rid of the crazy "nip-x-xxxx.ws" filename.
	 */
	filemodel_set_filename( FILEMODEL( nws ), NULL );
}

static void
workspaceview_select_all_cb( GtkWidget *menu, GtkWidget *host, View *view )
{
	Workspaceview *wview = WORKSPACEVIEW( view ); 
	Workspace *ws = WORKSPACE( VIEW( wview )->model );

	workspace_select_all( ws );
}

static void
workspaceview_class_init( WorkspaceviewClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	ViewClass *view_class = (ViewClass *) klass;
	GtkWidget *pane;

	parent_class = gtk_type_class( TYPE_VIEW );

	object_class->destroy = workspaceview_destroy;

	/* Create signals.
	 */

	/* Set methods.
	 */
	view_class->link = workspaceview_link;
	view_class->child_add = workspaceview_child_add;
	view_class->refresh = workspaceview_refresh;

	pane = workspaceview_popup_menu = popup_build( "Workspace menu" );
	popup_add_but( pane, "Edit caption ...", view_not_implemented_cb );
	popup_add_but( pane, "Select all items in workspace", 
		workspaceview_select_all_cb );
	popup_add_but( pane, "Clone workspace", workspaceview_clone_cb );
	popup_add_but( pane, "Save workspace as ...", view_save_as_cb );
	menu_add_sep( pane );
	popup_add_but( pane, "Close workspace", view_close_cb );

	workspaceview_quark = 
		g_quark_from_static_string( "workspaceview_quark" );
}

/* Event in workspaceview tab.
 */
static gboolean
workspaceview_event_cb( GtkWidget *widget, GdkEvent *ev, Workspaceview *wview )
{
        if( ev->type == GDK_BUTTON_PRESS && ev->button.button == 1 ) {
		Workspace *ws = WORKSPACE( VIEW( wview )->model );
		Workspacegroup *wsg = workspace_get_workspacegroup( ws );

		workspacegroup_set_current( wsg, ws );

		return( TRUE );
	}

        return( FALSE );
}

static void
workspaceview_init( Workspaceview *wview )
{
	GtkAdjustment *hadj;
	GtkAdjustment *vadj;

	wview->wgview = NULL;

	wview->label = NULL;
	wview->tab = NULL;
	wview->frame = NULL;
	wview->fixed = NULL;
	wview->window = NULL;

	wview->timer = 0;
	wview->u = 0;
	wview->v = 0;

	wview->dragging = FALSE;
	wview->drag_x = 0;
	wview->drag_y = 0;

	wview->vp.left = 0;
	wview->vp.top = 0;
	wview->vp.width = 0;
	wview->vp.height = 0;
	wview->width = -1;
	wview->height = -1;
	wview->bounding.left = 0;
	wview->bounding.top = 0;
	wview->bounding.width = 0;
	wview->bounding.height = 0;

	wview->next_x = 3;
	wview->next_y = 3;

	wview->front = NULL;

	wview->context = NULL; 

	/* Size set in refresh.
	 */
	wview->fixed = gtk_fixed_new();
	wview->window = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( wview->window ), 
		GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC );
	gtk_scrolled_window_add_with_viewport( 
		GTK_SCROLLED_WINDOW( wview->window ), wview->fixed );
	gtk_viewport_set_shadow_type( 
		GTK_VIEWPORT( GTK_BIN( wview->window)->child ), 
		GTK_SHADOW_NONE );
	gtk_signal_connect( GTK_OBJECT( wview->fixed ), "realize", 
		GTK_SIGNAL_FUNC( workspaceview_realize_cb ), NULL );
        gtk_signal_connect( GTK_OBJECT( wview->fixed ), "event",
                GTK_SIGNAL_FUNC( workspaceview_fixed_event_cb ), wview );
#ifndef GRAB
	/* If we're not doing true grabs, we need to enable MOTION events
	 * always.
	 */
	gtk_widget_set_events( GTK_WIDGET( wview->fixed ), 
		GDK_POINTER_MOTION_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK ); 
#endif /*!GRAB*/

	hadj = gtk_scrolled_window_get_hadjustment( 
		GTK_SCROLLED_WINDOW( wview->window ) );
	vadj = gtk_scrolled_window_get_vadjustment( 
		GTK_SCROLLED_WINDOW( wview->window ) );
	gtk_signal_connect( GTK_OBJECT( hadj ), "value_changed", 
		GTK_SIGNAL_FUNC( workspaceview_scroll_adjustment_cb ), wview );
	gtk_signal_connect( GTK_OBJECT( hadj ), "changed", 
		GTK_SIGNAL_FUNC( workspaceview_scroll_adjustment_cb ), wview );
	gtk_signal_connect( GTK_OBJECT( vadj ), "value_changed", 
		GTK_SIGNAL_FUNC( workspaceview_scroll_adjustment_cb ), wview );
	gtk_signal_connect( GTK_OBJECT( vadj ), "changed", 
		GTK_SIGNAL_FUNC( workspaceview_scroll_adjustment_cb ), wview );

	/*

		FIXME ... starts jumping about strangely :-( 

	gtk_container_set_focus_hadjustment( GTK_CONTAINER( wview->fixed ),
		hadj );
	gtk_container_set_focus_vadjustment( GTK_CONTAINER( wview->fixed ),
		vadj );
	 */

	wview->frame = gtk_frame_new( NULL );
	gtk_frame_set_shadow_type( GTK_FRAME( wview->frame ), GTK_SHADOW_NONE );
	gtk_container_set_border_width( GTK_CONTAINER( wview->frame ), 1 );
	gtk_container_add( GTK_CONTAINER( wview->frame ), wview->window );
	gtk_object_set_data_by_id( GTK_OBJECT( wview->frame ), 
		workspaceview_quark, (void *) wview );

	wview->tab = gtk_event_box_new();
	wview->label = gtk_label_new( "" );
        gtk_container_add( GTK_CONTAINER( wview->tab ), wview->label );
        gtk_signal_connect( GTK_OBJECT( wview->tab ), "event",
                GTK_SIGNAL_FUNC( workspaceview_event_cb ), wview );
        popup_attach( wview->tab, workspaceview_popup_menu, wview );

	gtk_widget_show_all( wview->tab );
	gtk_widget_show_all( wview->frame );
}

GtkType
workspaceview_get_type( void )
{
	static GtkType workspaceview_type = 0;

	if (!workspaceview_type) {
		static const GtkTypeInfo wviewt = {
			"Workspaceview",
			sizeof( Workspaceview ),
			sizeof( WorkspaceviewClass ),
			(GtkClassInitFunc) workspaceview_class_init,
			(GtkObjectInitFunc) workspaceview_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		workspaceview_type = gtk_type_unique( TYPE_VIEW, &wviewt );
	}

	return( workspaceview_type );
}

View *
workspaceview_new( void )
{
	Workspaceview *wview = gtk_type_new( TYPE_WORKSPACEVIEW );

	return( VIEW( wview ) );
}

/* Pick an xy position for the next column.
 */
void
workspaceview_pick_xy( Workspaceview *wview, int *x, int *y )
{
	/* Position already set? No change.
	 */
	if( *x >= 0 )
		return;

	/* Set this position.
	 */
	*x = wview->next_x + wview->vp.left;
	*y = wview->next_y + wview->vp.top;

	/* And move on.
	 */
	wview->next_x += 30;
	wview->next_y += 30;
	if( wview->next_x > 300 )
		wview->next_x = 3;
	if( wview->next_y > 200 )
		wview->next_y = 3;
}
