/* an image class object in a workspace
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

#include "ip.h"

static ClassmodelClass *parent_class = NULL;

static void
iimage_instance_init( iImageInstance *instance, Classmodel *classmodel )
{
	instance->ii = NULL;
	instance->file_changed_sid = 0;
	instance->classmodel = classmodel;
}

static void
iimage_instance_destroy( iImageInstance *instance )
{
	FREESID( instance->file_changed_sid, instance->ii );
	FREEF( imageinfo_destroy_nonheap, instance->ii );
}

static void
iimage_instance_file_changed( Imageinfo *imageinfo, iImageInstance *instance )
{
	Row *row = HEAPMODEL( instance->classmodel )->row;

#ifdef DEBUG
	printf( "iimage_instance_file_changed: " );
	iobject_print( IOBJECT( instance->classmodel ) );
#endif /*DEBUG*/

	if( CALC_RELOAD ) {
		(void) expr_dirty( row->expr, link_serial_new() );
		symbol_recalculate_all();
	}
}

static gboolean
iimage_instance_update( iImageInstance *instance, PElement *root )
{
	Imageinfo *ii;

	if( !class_get_member_image( root, MEMBER_VALUE, &ii ) )
		return( FALSE );

	iimage_instance_destroy( instance );

	instance->ii = ii;

	if( ii ) {
		imageinfo_dup_nonheap( ii );
		instance->file_changed_sid = g_signal_connect( 
			G_OBJECT( instance->ii ), "file_changed", 
			G_CALLBACK( iimage_instance_file_changed ), instance );
	}

#ifdef DEBUG
	printf( "iimage_instance_update: ii = 0x%x\n", (unsigned int) ii );
#endif /*DEBUG*/

	return( TRUE );
}

static void
iimage_dispose( GObject *gobject )
{
	iImage *iimage;

#ifdef DEBUG
	printf( "iimage_dispose\n" );
#endif /*DEBUG*/

	g_return_if_fail( gobject != NULL );
	g_return_if_fail( IS_IIMAGE( gobject ) );

	iimage = IIMAGE( gobject );

	slist_map( iimage->classmodels, 
		(SListMapFn) classmodel_iimage_unlink, iimage );
	assert( !iimage->classmodels );

	G_OBJECT_CLASS( parent_class )->dispose( gobject );
}

static void
iimage_finalize( GObject *gobject )
{
	iImage *iimage;

#ifdef DEBUG
	printf( "iimage_finalize\n" );
#endif /*DEBUG*/

	g_return_if_fail( gobject != NULL );
	g_return_if_fail( IS_IIMAGE( gobject ) );

	iimage = IIMAGE( gobject );

	/* My instance finalize stuff.
	 */
	iimage_instance_destroy( &iimage->instance );
	FREEF( g_slist_free, iimage->views );
	buf_destroy( &iimage->caption_buffer );

	G_OBJECT_CLASS( parent_class )->finalize( gobject );
}

static const char *
iimage_generate_caption( iObject *iobject ) 
{
	iImage *iimage = IIMAGE( iobject );
	Row *row = HEAPMODEL( iimage )->row;
	Expr *expr;

	if( !HEAPMODEL( iimage )->row ||
		!(expr = HEAPMODEL( iimage )->row->expr) ) 
		return( _( "No image" ) );

	buf_rewind( &iimage->caption_buffer );

	/* Show the filename, if we know it, otherwise something from
	 * the class.
	 */
	if( CLASSMODEL( iimage )->filename )
		buf_appends( &iimage->caption_buffer, 
			im_skip_dir( CLASSMODEL( iimage )->filename ) );
	else if( expr && PEISCLASS( &expr->root ) ) {
		Symbol *sym = PEGETCLASSCOMPILE( &expr->root )->sym;
		Toolitem *toolitem;

		/* If this is an action member, we should be able to look up
		 * it's sym and get a descriptive label.
		 */
		if( (toolitem = toolitem_lookup( row->ws->kitg, sym )) )
			buf_appends( &iimage->caption_buffer, toolitem->name );
		else 
			symbol_qualified_name_relative( row->ws->sym,
				sym, &iimage->caption_buffer );
	}
	else
		buf_appends( &iimage->caption_buffer, CLASS_IMAGE );
	buf_appends( &iimage->caption_buffer, ", " );

	if( iimage->instance.ii )
		imageinfo_info( iimage->instance.ii, &iimage->caption_buffer );

	return( buf_all( &iimage->caption_buffer ) );
}

static void
iimage_edit( GtkWidget *parent, Model *model )
{
        iImage *iimage = IIMAGE( model );

	if( iimage->instance.ii ) 
		imageview_new( iimage );
}

static xmlNode *
iimage_save( Model *model, xmlNode *xnode )
{
        iImage *iimage = IIMAGE( model );
	xmlNode *xthis;

	if( !(xthis = MODEL_CLASS( parent_class )->save( model, xnode )) )
		return( NULL );

	/* We always rebuild the value from the expr ... don't save.
	 */
	if( !set_prop( xthis, "window_x", "%d", iimage->window_x ) ||
		!set_prop( xthis, "window_y", "%d", iimage->window_y ) ||
		!set_prop( xthis, "window_width", 
			"%d", iimage->window_width ) ||
		!set_prop( xthis, "window_height", 
			"%d", iimage->window_height ) ||
		!set_prop( xthis, "image_left", "%d", iimage->image_left ) ||
		!set_prop( xthis, "image_top", "%d", iimage->image_top ) ||
		!set_prop( xthis, "image_mag", "%d", iimage->image_mag ) ||
		!set_prop( xthis, "show_status", 
			bool_to_char( iimage->show_status ) ) ||
		!set_prop( xthis, "show_paintbox", 
			bool_to_char( iimage->show_paintbox ) ) ||
		!set_prop( xthis, "show_convert", 
			bool_to_char( iimage->show_convert ) ) ||
		!set_prop( xthis, "show_rulers", 
			bool_to_char( iimage->show_rulers ) ) ||
		!set_prop( xthis, "scale", "%g", iimage->scale ) ||
		!set_prop( xthis, "offset", "%g", iimage->offset ) )
		return( NULL );

	return( xthis );
}

static gboolean
iimage_load( Model *model, 
	ModelLoadState *state, Model *parent, xmlNode *xnode )
{
        iImage *iimage = IIMAGE( model );

	assert( IS_RHS( parent ) );

	(void) get_iprop( xnode, "window_x", &iimage->window_x );
	(void) get_iprop( xnode, "window_y", &iimage->window_y );
	(void) get_iprop( xnode, "window_width", &iimage->window_width );
	(void) get_iprop( xnode, "window_height", &iimage->window_height );
	(void) get_iprop( xnode, "image_left", &iimage->image_left );
	(void) get_iprop( xnode, "image_top", &iimage->image_top );
	(void) get_iprop( xnode, "image_mag", &iimage->image_mag );
	(void) get_bprop( xnode, "show_status", &iimage->show_status );
	(void) get_bprop( xnode, "show_paintbox", &iimage->show_paintbox );
	(void) get_bprop( xnode, "show_convert", &iimage->show_convert );
	(void) get_bprop( xnode, "show_rulers", &iimage->show_rulers );
	(void) get_dprop( xnode, "scale", &iimage->scale );
	(void) get_dprop( xnode, "offset", &iimage->offset );

	return( MODEL_CLASS( parent_class )->load( model, 
		state, parent, xnode ) );
}

/* Need to implement _update_heap(), as not all model fields are directly
 * editable ... some are set only from expr. See also iregion.c.
 */
static void *
iimage_update_heap( Heapmodel *heapmodel )
{
	Expr *expr = heapmodel->row->expr;
        iImage *iimage = IIMAGE( heapmodel );
	iImageInstance *instance = &iimage->instance;

	PElement pe;

#ifdef DEBUG
	printf( "iimage_update_heap: " );
	row_name_print( HEAPMODEL( iimage )->row );
	printf( "\n" );
#endif /*DEBUG*/

	/* Read the heap into the model, over the top of the unapplied edits.
	 */
	if( !class_get_exact( &expr->root, IOBJECT( heapmodel )->name, &pe ) )
		return( FALSE );
	if( !iimage_instance_update( instance, &pe ) )
		return( heapmodel );

	FREE( CLASSMODEL( iimage )->filename );
        if( instance->ii && instance->ii->from_file &&
                IOBJECT( instance->ii )->name )
                SETSTR( CLASSMODEL( iimage )->filename, 
                        IOBJECT( instance->ii )->name );

	/* Classmodel _update_heap() will do _instance_new() from the fixed up
	 * model.
	 */
	return( HEAPMODEL_CLASS( parent_class )->update_heap( heapmodel ) );
}

/* Update iImage from heap.
 */
static gboolean
iimage_class_get( Classmodel *classmodel, PElement *root )
{
        iImage *iimage = IIMAGE( classmodel );
	iImageInstance *instance = &iimage->instance;

#ifdef DEBUG
	printf( "iimage_class_get: " );
	row_name_print( HEAPMODEL( iimage )->row );
	printf( "\n" );
#endif /*DEBUG*/

	if( !iimage_instance_update( instance, root ) )
		return( FALSE );

	FREE( classmodel->filename );
        if( instance->ii && instance->ii->from_file &&
                IOBJECT( instance->ii )->name )
                SETSTR( classmodel->filename, 
                        IOBJECT( instance->ii )->name );

	return( CLASSMODEL_CLASS( parent_class )->class_get( 
		classmodel, root ) );
}

/* Make a new "fn value" application.
 */
static gboolean
iimage_class_new( Classmodel *classmodel, PElement *fn, PElement *out )
{
	Heap *heap = reduce_context->heap;
        iImage *iimage = IIMAGE( classmodel );
	iImageInstance *instance = &iimage->instance;

	PElement rhs;

#ifdef DEBUG
	printf( "iimage_class_new: " );
	row_name_print( HEAPMODEL( iimage )->row );
	printf( "\n" );
#endif /*DEBUG*/

	/* Make application nodes.
	 */
	heap_appl_init( out, fn );
	if( !heap_appl_add( heap, out, &rhs ) ||
		!heap_image_new( heap, instance->ii, &rhs ) )
		return( FALSE );

	return( TRUE );
}

static gboolean
iimage_graphic_save( Classmodel *classmodel, 
	GtkWidget *parent, const char *filename )
{
	iImage *iimage = IIMAGE( classmodel );
	iImageInstance *instance = &iimage->instance;
	char buf[MAX_STRSIZE];

	strcpy( buf, filename );
	filesel_add_mode( buf );

	if( !imageinfo_write( instance->ii, parent, buf ) )
		return( FALSE );

	mainw_recent_add( &mainw_recent_image, filename );

	return( TRUE );
}

gboolean
iimage_replace( iImage *iimage, const char *filename )
{
	Row *row = HEAPMODEL( iimage )->row;
	iText *itext = ITEXT( HEAPMODEL( iimage )->rhs->itext );
	char txt[MAX_STRSIZE];
	BufInfo buf;

	buf_init_static( &buf, txt, MAX_STRSIZE );
	buf_appends( &buf, "Image_file \"" );
	buf_appendsc( &buf, TRUE, filename );
	buf_appends( &buf, "\"" );

	itext_set_formula( itext, buf_all( &buf ) );
	itext_set_edited( itext, TRUE );
	(void) expr_dirty( row->expr, link_serial_new() );

	mainw_recent_add( &mainw_recent_image, filename );

	return( TRUE );
}

static gboolean
iimage_graphic_replace( Classmodel *classmodel, 
	GtkWidget *parent, const char *filename )
{
	return( iimage_replace( IIMAGE( classmodel ), filename ) );
}

static void
iimage_class_init( iImageClass *class )
{
	GObjectClass *gobject_class = (GObjectClass *) class;
	iObjectClass *iobject_class = (iObjectClass *) class;
	ModelClass *model_class = (ModelClass *) class;
	HeapmodelClass *heapmodel_class = (HeapmodelClass *) class;
	ClassmodelClass *classmodel_class = (ClassmodelClass *) class;

	parent_class = g_type_class_peek_parent( class );

	/* Create signals.
	 */

	/* Init methods.
	 */
	gobject_class->dispose = iimage_dispose;
	gobject_class->finalize = iimage_finalize;

	iobject_class->generate_caption = iimage_generate_caption;

	model_class->view_new = iimageview_new;
	model_class->edit = iimage_edit;
	model_class->save = iimage_save;
	model_class->load = iimage_load;

	heapmodel_class->update_heap = iimage_update_heap;

	classmodel_class->class_get = iimage_class_get;
	classmodel_class->class_new = iimage_class_new;

	classmodel_class->graphic_save = iimage_graphic_save;
	classmodel_class->graphic_replace = iimage_graphic_replace;

	classmodel_class->filetype = filesel_type_image;
	classmodel_class->default_filetype = IMAGE_FILE_TYPE;

	/* Static init.
	 */
	model_register_loadable( MODEL_CLASS( class ) );
}

static void
iimage_init( iImage *iimage )
{
	iimage_instance_init( &iimage->instance, CLASSMODEL( iimage ) );

	iimage->classmodels = NULL;

	iimage->views = NULL;

	iimage->window_x = -1;
	iimage->window_y = -1;
	iimage->window_width = -1;
	iimage->window_height = -1;
	iimage->image_left = -1;
	iimage->image_top = -1;
	iimage->image_mag = -1;

	iimage->show_status = FALSE;
	iimage->show_paintbox = FALSE;
	iimage->show_convert = FALSE;
	iimage->show_rulers = FALSE;

	iimage->scale = -1.0;
	iimage->offset = -1.0;

	buf_init_dynamic( &iimage->caption_buffer, MAX_LINELENGTH );

	iobject_set( IOBJECT( iimage ), CLASS_IMAGE, NULL );
}

GtkType
iimage_get_type( void )
{
	static GType type = 0;

	if( !type ) {
		static const GTypeInfo info = {
			sizeof( iImageClass ),
			NULL,           /* base_init */
			NULL,           /* base_finalize */
			(GClassInitFunc) iimage_class_init,
			NULL,           /* class_finalize */
			NULL,           /* class_data */
			sizeof( iImage ),
			32,             /* n_preallocs */
			(GInstanceInitFunc) iimage_init,
		};

		type = g_type_register_static( TYPE_CLASSMODEL, 
			"iImage", &info, 0 );
	}

	return( type );
}

Classmodel *
iimage_new( Rhs *rhs )
{
	iImage *iimage;

	iimage = IIMAGE( g_object_new( TYPE_IIMAGE, NULL ) );
	icontainer_child_add( ICONTAINER( rhs ), ICONTAINER( iimage ), -1 );

	return( CLASSMODEL( iimage ) );
}
