#include <fe/njamd.h>
#include <fe/util/err.h>
#include <fe/util/libcwrap.h>

int __njAccept(int s, struct sockaddr* addr, size_t* addrlen,
		const char* fn, const char* file, int line)
{
	int ret = accept(s, addr, addrlen);

	if (ret == -1)
		perrfn(EXIT_FAILURE, fn, file, line, "%s", strerror(errno));
	
	return ret;
}

int __njBind(int s, struct sockaddr* addr, size_t addrlen, const char* fn,
		const char* file, int line)
{
	int ret = bind(s, addr, addrlen);

	if (ret == -1)
		perrfn(EXIT_FAILURE, fn, file, line, "%s", strerror(errno));

	return ret;
}

int __njClose(int fd, const char* fn, const char* file, int line)
{
	int ret = close(fd);

/*	if (ret == -1)
		warnfn(fn, file, line, "%s", strerror(errno)); */

	return ret;
}

int __njDup2(int oldfd, int newfd, const char* fn, const char* file, int line)
{
	int ret = dup2(oldfd, newfd);

	if (ret == -1)
		perrfn(EXIT_FAILURE, fn, file, line, "%s", strerror(errno));

	return ret;
}

int __njExecvp(const char* exe, char *const argv[], const char* fn,
		const char* file, int line)
{
	int ret = execvp(exe, argv);

	if (ret == -1)
		perrfn(EXIT_FAILURE, fn, file, line, "%s", strerror(errno));

	return ret;
}

pid_t __njFork(const char* fn, const char* file, int line)
{
	int ret = fork();

	if (ret == -1)
		perrfn(EXIT_FAILURE, fn, file, line, "%s", strerror(errno));

	return ret;
}

int __njGetSockName(int s, struct sockaddr* addr, size_t* addrlen,
		const char* fn, const char* file, int line)
{
	int ret = getsockname(s, addr, addrlen);

	if (ret == -1)
		perrfn(EXIT_FAILURE, fn, file, line, "%s", strerror(errno));

	return ret;
}

int __njListen(int s, int backlog, const char* fn, const char* file, int line)
{
	int ret = listen(s, backlog);

	if (ret == -1)
		perrfn(EXIT_FAILURE, fn, file, line, "%s", strerror(errno));

	return ret;
}

void* __njMalloc(size_t size, const char* fn, const char* file, int line)
{
	void* ret = malloc(size);

	if (ret == NULL)
		perrfn(EXIT_FAILURE, fn, file, line, "%s", strerror(errno));

	return ret;
}

int __njPipe(int filedes[2], const char* fn, const char* file, int line)
{
	int ret = pipe(filedes);

	if (ret == -1)
		perrfn(EXIT_FAILURE, fn, file, line, "%s", strerror(errno));

	return ret;
}

ssize_t __njRead(int fd, void* buf, size_t count, const char* fn,
		const char* file, int line)
{
	ssize_t ret = read(fd, buf, count);

	if (ret == -1)
		perrfn(EXIT_FAILURE, fn, file, line, "%s", strerror(errno));

	return ret;
}

int __njSelect(int n, fd_set* readfds, fd_set* writefds, fd_set* exceptfds,
		struct timeval* timeout, const char* fn, const char* file, int line)
{
	int ret = select(n, readfds, writefds, exceptfds, timeout);

	if (ret == -1)
		perrfn(EXIT_FAILURE, fn, file, line, "%s", strerror(errno));

	return ret;
}

int __njSetenv(const char* name, const char* value, int overwrite,
		 const char* fn, const char* file, int line)
{
	char buf[1024];
	int ret;
	if(!overwrite && getenv(name))
	    return 0;
	
	snprintf(buf, 1024, "%s=%s", name, value);

	ret = putenv(buf);

	if (ret == -1)
		perrfn(EXIT_FAILURE, fn, file, line, "%s", strerror(errno));

	return ret;
}

void (*__njSignal(int signo, void (*handler)(int), const char* fn,
		const char* file, int line))(int)
{
	struct sigaction act, oact;

	act.sa_handler = handler;
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;
	if (signo == SIGALRM) {
#ifdef SA_INTERRUPT
		act.sa_flags |= SA_INTERRUPT;    /* SunOS 4.x */
#endif
	} else {
#ifdef SA_RESTART
		act.sa_flags |= SA_RESTART;      /* SVR4, 4.4BSD */
#endif
	}

	if (sigaction(signo, &act, &oact) < 0)
		perrfn(EXIT_FAILURE, fn, file, line, "%s", strerror(errno));
	
	return oact.sa_handler;
}

int __njSocket(int domain, int type, int protocol, const char* fn,
		const char* file, int line)
{
	int ret = socket(domain, type, protocol);

	if (ret == -1)
		perrfn(EXIT_FAILURE, fn, file, line, "%s", strerror(errno));

	return ret;
}

char* __njStrdup(const char* s, const char* fn, const char* file, int line)
{
	char* ret = strdup(s);

	if (ret == NULL)
		perrfn(EXIT_FAILURE, fn, file, line, "%s", strerror(errno));

	return ret;
}

ssize_t __njWrite(int fd, const void* buf, size_t count, const char* fn,
		const char* file, int line)
{
	ssize_t ret = write(fd, buf, count);

	if (ret == -1)
		perrfn(EXIT_FAILURE, fn, file, line, "%s", strerror(errno));

	return ret;
}
