/** 
 * @file allocator.c Allocation Routines
 *
 * These functions return user chunks for the public interface
 * 
 * Copyright (C) 2000 by Mike Perry.
 * Distributed WITHOUT WARRANTY under the GPL. See COPYING for details.
 * 
 */
#include <lib/portability.h>
#include <lib/allocator.h>
#include <lib/user_chunk.h>
#include <lib/block.h>
#include <sys/mman.h>
#include <string.h>


/**
 * Starts the allocator with enough info to give us segments
 *
 * @param allocator The allocator to start.
 */ 
void __nj_allocator_bootstrap_init(struct nj_allocator *allocator)
{
	allocator->first_sys_alloc = __nj_memory_pool_bootstrap_init(&allocator->mem_pool);
	__nj_entry_pool_bootstrap_init(&allocator->entry_pool);
}

/**
 * Initializes the allocator for the user, once the prefs are available
 * 
 * @param allocator The allocator to init
 * @param syms The libc symbols object, to get the libc shit we need for realloc
 * @param prefs The user prefrences 
 */
void __nj_allocator_user_init(struct nj_allocator *allocator, 
		struct nj_libc_syms *syms,
		struct nj_prefs *prefs)
{
	__nj_entry_pool_user_init(&allocator->entry_pool, prefs);

	/* Get the last system alloc address so we can tell when the bootstrap 
	 * allocator stopped being used */
	allocator->last_sys_alloc = __nj_memory_pool_user_init(&allocator->mem_pool, syms, prefs);

	allocator->fixed_alloc = !prefs->stat.mutable_alloc;
	
	allocator->libc_realloc = __nj_libc_syms_resolve_libc(syms, "realloc");
}

/** 
 * Ends the allocator
 *
 * @param allocator The allocator
 */
void __nj_allocator_fini(struct nj_allocator *allocator)
{
	__nj_memory_pool_fini(&allocator->mem_pool);
	__nj_entry_pool_fini(&allocator->entry_pool);
}

/**
 * Converts an allocation type to a string for error reporting
 *
 * @param alloc_type The allocation type
 * @returns A constant string with the type 
 */
const char *__nj_allocator_type_to_string(int alloc_type)
{
	switch(alloc_type)
	{
		case NJ_PROT_NONE:
			return "NJ_PROT_NONE";
		case NJ_PROT_OVER:
			return "NJ_PROT_OVER";
		case NJ_PROT_UNDER:
			return "NJ_PROT_UNDER";
		case NJ_PROT_SUNDER:
			return "NJ_PROT_SUNDER";
		default:
			return "PROT UNKNOWN!";
	}
}

/**
 * Request a completed block from the allocator
 *
 * Obtains a properly protected block from the memory pool, then performs
 * alignment on it, as well as creating a heap entry and callstack
 * for it.
 * 
 * @param allocator The allocator
 * @param len The length
 * @param prefs The user prefrences
 * 
 * @returns A user pointer 
 */
nj_addr_t __nj_allocator_request_user_chunk(struct nj_allocator *allocator, 
		size_t user_len, struct nj_dynamic_prefs prefs)
{
	nj_addr_t block_addr;
	struct nj_block block;
	size_t block_size;

	block_size = __nj_block_calc_size(user_len, prefs.align, prefs.alloc_type);

	block_addr = __nj_memory_pool_request_block(&allocator->mem_pool, 
			block_size, prefs);

	__nj_block_init(&block, block_addr, block_size, user_len, prefs);
	
	*block.idx = __nj_entry_pool_request_index(&allocator->entry_pool);

	__nj_entry_pool_index_init(&allocator->entry_pool, *block.idx, block_addr, user_len, prefs);
	
	/* The other block types are protected in the memory pool */	
	if(prefs.alloc_type == NJ_PROT_SUNDER)
		mprotect((void *)block_addr, NJ_PAGE_SIZE, __nj_prot);
	
	/** @TODO Stats */

	return block.user_chunk;
}

/**
 * Implemmentation of realloc
 *
 * This is horriblly corrupt in terms of design because of the optimizations 
 * we can do by bypassing the caches and whatnot in certain cases. Oh well, 
 * the price you pay.
 *
 * @param allocator The allocator object itself
 * @param old_ret The old return address
 * @param new_len the new length to alloc
 * @param prefs The dynamic, changable user prefrences
 * @returns A user pointer, ready and willing to be used
 */
nj_addr_t __nj_allocator_resize_user_chunk(struct nj_allocator *allocator, 
		nj_addr_t old_ret, size_t new_len, struct nj_dynamic_prefs prefs)
{
	nj_addr_t old_block, new_block;
	struct nj_heap_entry *entry;
	nj_entry_index_t old_idx;
	int old_block_size, new_block_size;
	int old_alloc_type, new_alloc_type;
	int old_align, new_align;
	struct nj_block new_block_desc;
	size_t old_len;

	if(allocator->fixed_alloc)
	{
		if(NJ_ALLOCATOR_CHUNK_FROM_BOOTSTRAP(allocator, old_ret))
			entry = __nj_user_chunk_get_entry(old_ret, &allocator->entry_pool, NJ_ALLOCATOR_BOOTSTRAP_ALLOC_TYPE, &old_idx);
		else
			entry = __nj_user_chunk_get_entry(old_ret, &allocator->entry_pool, prefs.alloc_type, &old_idx);
	}
	else
	{
		entry = __nj_user_chunk_find_entry(old_ret, &allocator->entry_pool, &old_idx);
	}
	
	old_alloc_type = entry->alloc_type;
	old_len = entry->user_len;
	old_block = entry->block;
	old_align = 1<<entry->align_shift;
	old_block_size = __nj_block_calc_size(old_len, old_align, old_alloc_type);
	
	new_align = prefs.align;
	new_alloc_type = prefs.alloc_type;
	new_block_size = __nj_block_calc_size(new_len, new_align, new_alloc_type);
	
	/** @TODO Stats */

	/** 
	 * Optimization is the root of all evil.. Look what it did to my nice 
	 * design.. Normally this would go into memory_pool... but we can't do that
	 * because we don't know where to copy the shit between the two
	 * segments.
	 */
	if(old_alloc_type == NJ_PROT_NONE && new_alloc_type == NJ_PROT_NONE)
	{
		new_block = (nj_addr_t)allocator->libc_realloc((void*)old_block, new_block_size);

		if(old_block == new_block)
			__nj_block_renew(&new_block_desc, new_block, new_block_size, new_len, 
					memmove, old_ret, old_len, prefs);
		else
			__nj_block_renew(&new_block_desc, new_block, new_block_size, new_len, 
					memcpy, old_ret, old_len, prefs);
	}
	else
	{
		if(prefs.free_type == NJ_CHK_FREE_NONE && 
		   new_alloc_type == old_alloc_type && 
		   new_block_size == old_block_size)
		{
			new_block = old_block;
			
			__nj_block_renew(&new_block_desc, new_block, new_block_size, new_len, 
					memmove, old_ret, old_len, prefs);
		}
		else
		{
			new_block = __nj_memory_pool_request_block(&allocator->mem_pool, 
					new_block_size, prefs);
			
			__nj_block_renew(&new_block_desc, new_block, new_block_size, new_len, 
					memcpy, old_ret, old_len, prefs);
	
			/* Now release the block */
			__nj_memory_pool_release_block(&allocator->mem_pool, old_block, 
					old_block_size, old_alloc_type, prefs.free_type);
		}
	}

	/* RENEW! RENEW! ;) */
	*new_block_desc.idx = __nj_entry_pool_renew_index(&allocator->entry_pool, old_idx, new_block, new_len, prefs);

	/* The other block types are protected in the memory pool */	
	if(prefs.alloc_type == NJ_PROT_SUNDER)
		mprotect((void *)new_block, NJ_PAGE_SIZE, PROT_NONE);

	return new_block_desc.user_chunk;
}

/**
 * Implementation of free
 * 
 * Frees a user segment by returning it to the memory pools and 
 * caching or storing its index
 * 
 * @param allocator The NJAMD allocator object
 * @param user_chunk The user's pointer
 * @param prefs The dynamic prefrences object
 */
void __nj_allocator_release_user_chunk(struct nj_allocator *allocator, 
		nj_addr_t user_chunk, struct nj_dynamic_prefs prefs)
{
	struct nj_heap_entry *entry;
	nj_entry_index_t idx;
	nj_addr_t block_addr;
	int block_size;

	/** @TODO Make race condition lists to check for races on all addresses 
	 * being freed and realloced at the present time */
	
	/** @TODO Usage Stats */
	if(allocator->fixed_alloc)
	{
		if(NJ_ALLOCATOR_CHUNK_FROM_BOOTSTRAP(allocator, user_chunk))
			entry = __nj_user_chunk_get_entry(user_chunk, &allocator->entry_pool, NJ_ALLOCATOR_BOOTSTRAP_ALLOC_TYPE, &idx);
		else
			entry = __nj_user_chunk_get_entry(user_chunk, &allocator->entry_pool, prefs.alloc_type, &idx);
	}
	else
		entry = __nj_user_chunk_find_entry(user_chunk, &allocator->entry_pool, &idx);

	block_size = __nj_block_calc_size(entry->user_len, 1<<entry->align_shift, entry->alloc_type);
	
	block_addr = entry->block;

	__nj_memory_pool_release_block(&allocator->mem_pool, block_addr,
			block_size, entry->alloc_type, prefs.free_type);

	__nj_entry_pool_index_fini(&allocator->entry_pool, idx, prefs);
}

// vim:ts=4
