---
layout: layout
title: Asserts
---

# Asserts

The `Test` object has a collection of assertion methods, many of which
are given several synonyms for compatibility with other test runners
and the vagaries of human expectations and spelling.  When a synonym
is multi-word in `camelCase` the corresponding lower case and
`snake_case` versions are also created as synonyms.

All assertion methods take optional `message` and `extra` arguments as
the last two params.  The `message` is the name of the test.  The
`extra` argument can contain any arbitrary data about the test, but
the following fields are "special".

* `todo` Set to boolean `true` or a String to mark this as pending
* `skip` Set to boolean `true` or a String to mark this as skipped
* `diagnostic` Set to boolean `true` to show a yaml diagnostic block
  even if the test point passes.  (Failing asserts always show yaml
  diagnostics.)
* `at` Generated by the framework.  The location where the assertion
  was called.  Do not set this field unless you know what you are
  doing.
* `stack` Generated by the framework.  The stack trace to the point
  where the assertion was called.  Do not set this field unless you
  know what you are doing.

**Note**: There's no requirement that you use tap's built-in
assertions.  You can also use any Error-throwing assertion library,
including Node.js's built-in `assert` module.  A throw fails a test,
so not-throwing passes.  That does, however, mean that you won't
generate a test point in your output for each assertion run.  You do
you.

## t.ok(obj, message, extra)

Verifies that the object is truthy.

Synonyms: `t.true`, `t.assert`

## t.notOk(obj, message, extra)

Verifies that the object is not truthy.

Synonyms: `t.false`, `t.assertNot`

## t.error(obj, message, extra)

If the object is an error, then the assertion fails.

Note: if an error is encountered unexpectedly, it's often better to
simply throw it.  The Test object will handle this as a failure.

Synonyms: `t.ifErr`, `t.ifError`

## t.rejects(promise | fn, [expectedError], message, extra)

Verifies that the promise (or promise-returning function) rejects.  If
an expected error is provided, then also verify that the rejection
matches the expected error.

Note: since promises always reject and resolve asynchronously, this
assertion is actually implemented using a subtest.  As such, it does
not return a boolean to indicate its passing status.  Instead, it
returns a Promise that resolves when it is completed.

## t.resolves(promise | fn, message, extra)

Verifies that the promise (or promise-returning function) resolves,
making no expectation about the value that the promise resolves to.

Note: since promises always reject and resolve asynchronously, this
assertion is actually implemented using a subtest.  As such, it does
not return a boolean to indicate its passing status.  Instead, it
returns a Promise that resolves when it is completed.

## t.resolveMatch (promise | fn, wanted, message, extra)

Verifies that the promise (or promise-returning function) resolves,
and furthermore that the value of the promise matches the `wanted`
pattern using `t.match`.

Note: since promises always reject and resolve asynchronously, this
assertion is actually implemented using a subtest.  As such, it does
not return a boolean to indicate its passing status.  Instead, it
returns a Promise that resolves when it is completed.


## t.throws(fn, [expectedError], message, extra)

Expect the function to throw an error.  If an expected error is
provided, then also verify that the thrown error matches the expected
error.

If the expected error is an object, then it's matched against the
thrown error using `t.match(er, expectedError)`.  If it's a function,
then the error is asserted to be a member of that class.

If the function has a name, and the message is not provided, then the
function name will be used as the message.

If the function is not provided, then this will be treated as a `todo`
test.

Caveat: if you pass a `extra` object to t.throws, then you MUST also
pass in an expected error, or else it will read the diag object as the
expected error, and get upset when your thrown error doesn't match
`{skip:true}` or whatever.

For example, this will not work as expected:

```javascript
// anti-example, do not use!
t.throws(function() {throw new Error('x')}, { skip: true })
```

But this is fine:

```javascript
// this example is ok to use.
// note the empty 'expected error' object.
// since it has no fields, it'll only verify that the thrown thing is
// an object, not the value of any properties
t.throws(function() {throw new Error('x')}, {}, { skip: true })
```

The expected error is tested against the throw error using `t.match`,
so regular expressions and the like are fine.  If the expected error
is an `Error` object, then the `stack` field is ignored, since that
will generally never match.

Synonyms: `t.throw`

## t.doesNotThrow(fn, message, extra)

Verify that the provided function does not throw.

If the function has a name, and the message is not provided, then the
function name will be used as the message.

If the function is not provided, then this will be treated as a `todo`
test.

Note: if an error is encountered unexpectedly, it's often better to
simply throw it.  The Test object will handle this as a failure.

Synonyms: `t.notThrow`

## t.equal(found, wanted, message, extra)

Verify that the object found is exactly the same (that is, `===`) to
the object that is wanted.

Synonyms: `t.equals`, `t.isEqual`, `t.is`, `t.strictEqual`,
`t.strictEquals`, `t.strictIs`, `t.isStrict`, `t.isStrictly`

## t.notEqual(found, notWanted, message, extra)

Inverse of `t.equal()`.

Verify that the object found is not exactly the same (that is, `!==`) as
the object that is wanted.

Synonyms: `t.inequal`, `t.notEqual`, `t.notEquals`,
`t.notStrictEqual`, `t.notStrictEquals`, `t.isNotEqual`, `t.isNot`,
`t.doesNotEqual`, `t.isInequal`

## t.same(found, wanted, message, extra)

Verify that the found object is deeply equivalent to the wanted
object.  Use non-strict equality for scalars (ie, `==`).  See:
[tsame](http://npm.im/tsame)

Synonyms: `t.equivalent`, `t.looseEqual`, `t.looseEquals`,
`t.deepEqual`, `t.deepEquals`, `t.isLoose`, `t.looseIs`

## t.notSame(found, notWanted, message, extra)

Inverse of `t.same()`.

Verify that the found object is not deeply equivalent to the
unwanted object.  Uses non-strict inequality (ie, `!=`) for scalars.

Synonyms: `t.inequivalent`, `t.looseInequal`, `t.notDeep`,
`t.deepInequal`, `t.notLoose`, `t.looseNot`

## t.strictSame(found, wanted, message, extra)

Strict version of `t.same()`.

Verify that the found object is deeply equivalent to the wanted
object.  Use strict equality for scalars (ie, `===`).

Synonyms: `t.strictEquivalent`, `t.strictDeepEqual`, `t.sameStrict`,
`t.deepIs`, `t.isDeeply`, `t.isDeep`, `t.strictDeepEquals`

## t.strictNotSame(found, notWanted, message, extra)

Inverse of `t.strictSame()`.

Verify that the found object is not deeply equivalent to the unwanted
object.  Use strict equality for scalars (ie, `===`).

Synonyms: `t.strictInequivalent`, `t.strictDeepInequal`,
`t.notSameStrict`, `t.deepNot`, `t.notDeeply`, `t.strictDeepInequals`,
`t.notStrictSame`

## t.match(found, pattern, message, extra)

Verify that the found object matches the pattern provided.

If pattern is a regular expression, and found is a string, then verify
that the string matches the pattern.

If the pattern is a string, and found is a string, then verify that
the pattern occurs within the string somewhere.

If pattern is an object, then verify that all of the (enumerable)
fields in the pattern match the corresponding fields in the object
using this same algorithm.  For example, the pattern `{x:/a[sdf]{3}/}`
would successfully match `{x:'asdf',y:'z'}`.

This is useful when you want to verify that an object has a certain
set of required fields, but additional fields are ok.

See [tmatch](http://npm.im/tmatch) for the full details on how this
works.

Synonyms: `t.has`, `t.hasFields`, `t.matches`, `t.similar`, `t.like`,
`t.isLike`, `t.includes`, `t.include`, `t.contains`

## t.notMatch(found, pattern, message, extra)

Interse of `match()`

Verify that the found object does not match the pattern provided.

Synonyms: `t.dissimilar`, `t.unsimilar`, `t.notSimilar`, `t.unlike`,
`t.isUnlike`, `t.notLike`, `t.isNotLike`, `t.doesNotHave`,
`t.isNotSimilar`, `t.isDissimilar`

## t.type(object, type, message, extra)

Verify that the object is of the type provided.

Type can be a string that matches the `typeof` value of the object, or
the string name of any constructor in the object's prototype chain, or
a constructor function in the object's prototype chain.

For example, all the following will pass:

```javascript
t.type(new Date(), 'object')
t.type(new Date(), 'Date')
t.type(new Date(), Date)
```

Synonyms: `t.isa`, `t.isA`
