/*
 *  Copyright (C) 2004 Mathias Andre <mathias@openbrookes.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <glade/glade.h>
#include <gconf/gconf-client.h>

#include "nb_global.h"
#include "ui/interface.h"

#define _(text) gettext(text)

global *
nb_global_new ()
{
	global * g = g_malloc (sizeof (global));
	gchar * notefile = NULL;
	gchar * default_browser = NULL;

	g->current_note = nb_note_new_empty ();
	g->inserted_note = NEW;
	g->modified_note = CLEAN;

	g->store = gtk_list_store_new (N_COLUMNS,
				       G_TYPE_STRING,
				       G_TYPE_STRING,
				       G_TYPE_STRING,
				       G_TYPE_STRING,
				       G_TYPE_STRING,
				       G_TYPE_POINTER);

	set_sort_func (g->store);

	notefile = nb_global_parse_config (g);

	g->nl = nb_notelist_new (notefile);

	return (g);
}

void
nb_global_new_note (global * g)
{
	if  ( g->modified_note == EDITED )
	{
		if ( (g->inserted_note == INSERTED) || (is_empty_note () == 0) )
		{
			GtkWidget *dialog;
			int button;

			dialog = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
					(gchar *) _("Do you want to save note?"));

			g_object_set (dialog, "title", _("Save note?"), "allow_grow", TRUE, NULL);
			button = gtk_dialog_run ((GtkDialog *) dialog);
			gtk_widget_destroy (dialog);

			if (button == GTK_RESPONSE_NO)
				return ;

			if ( nb_global_save_note (g) )
				push_statusbar ((gchar *) _("Successfully saved note to file."));
			else
				return;
		}
		else
			return;

	}
	else
		push_statusbar ((gchar *) _("Empty note created."));

	/* empty the current note text to gain memory */
	nb_note_clear_text (g->current_note);

	/* create a new blank note */
	g->current_note = nb_note_new_empty ();
	g->inserted_note = NEW;
	g->modified_note = CLEAN;

	/*reset the option menus status and type */
	reset_option_menu ();

	/* blank the fields */
	blank_fields ();
}

void
nb_global_delete_note (global * g)
{
	extern GladeXML * xml;
	GtkWidget * treeview = glade_xml_get_widget (xml, "treeview");
	GtkTreeSelection * selection = gtk_tree_view_get_selection ( GTK_TREE_VIEW (treeview));
	int count = gtk_tree_selection_count_selected_rows (selection);

	if ( count < 1 )
	{
		GtkWidget *dialog;

		dialog = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
				(gchar *) _("No note selected"));

		gtk_dialog_run ((GtkDialog *) dialog);
		gtk_widget_destroy (dialog);

		return;
	}

	GtkWidget *dialog;
	int button;

	if ( count == 1 )
		dialog = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
			                         (gchar *) _("Do you want to delete current note?"));
	else
		dialog = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
			                         (gchar *) _("Do you want to delete all selected notes?"));

	g_object_set (dialog, "title", _("Delete note?"), "allow_grow", TRUE, NULL);
	button = gtk_dialog_run ((GtkDialog *) dialog);
	gtk_widget_destroy (dialog);

	if (button == GTK_RESPONSE_YES)
	{
		GList *rr_list = NULL;    /* list of GtkTreeRowReferences to remove */
		GList *node;

		/* delete note(s) from file  and get list of notes' pointers */
		gtk_tree_selection_selected_foreach (selection, (GtkTreeSelectionForeachFunc) nb_global_delete_note_from_file, &rr_list);
		

		/* delete note from treeview */
		for ( node = rr_list;  node != NULL;  node = node->next )
		{
			GtkTreePath *path;

			path = gtk_tree_row_reference_get_path((GtkTreeRowReference*)node->data);

			if (path)
			{
				GtkTreeIter  iter;

				if (gtk_tree_model_get_iter(GTK_TREE_MODEL(g->store), &iter, path))
				{
					gtk_list_store_remove(g->store, &iter);
				}
			}
		}

		blank_fields ();
		g->current_note = nb_note_new_empty ();
		g->inserted_note = NEW;
		g->modified_note = CLEAN;

		if ( count == 1 )
			push_statusbar ((gchar *) _("Note successfully deleted"));
		else
			push_statusbar (g_strdup_printf ((gchar *) _("%d notes successfully deleted"), count));

	}
}

gboolean
nb_global_delete_note_from_file (GtkTreeModel * model, GtkTreePath  * path, GtkTreeIter * iter, GList ** rowref_list)
/* This Function delete the note from the file and add its pointer to a list so it can be deleted later on in the parent function */
{
	note * n;
	extern global * g; 

	g_assert ( rowref_list != NULL );

	gtk_tree_model_get (model, iter, NOTE_POINTER_COLUMN, &n, -1);

	if ( !nb_notelist_update_file (g->nl, n, DEL_NOTE) )
	{
		GtkWidget *dialog;

		dialog = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
				(gchar *) _("Failed to delete note from file"));

		gtk_dialog_run ((GtkDialog *) dialog);
		gtk_widget_destroy (dialog);
	}
	else
	{
		GtkTreeRowReference  *rowref;

		rowref = gtk_tree_row_reference_new(model, path);

		*rowref_list = g_list_append(*rowref_list, rowref);

		/* Delete the note from the notelist */
		nb_notelist_remove (g->nl, n);
	}

	return FALSE;
}

void 
nb_global_update_note (global * g)
{
	/* save the note if it is not an empty one */

	if (is_empty_note ()) 
		return;

	if ( nb_global_save_note (g) )
		push_statusbar ((gchar *) _("Successfully saved note to file."));
}

gboolean 
nb_global_save_note (global * g)
{
	extern GladeXML * xml;
	GtkTextIter start;
	GtkTextIter end;

	/* get the text and title from fields */
	GtkWidget * entrySubject = glade_xml_get_widget (xml, "entrysubject");
	GtkWidget * textview = glade_xml_get_widget (xml, "textview");
	GtkTextBuffer * buffer =  gtk_text_view_get_buffer ( GTK_TEXT_VIEW (textview) );
	gtk_text_buffer_get_start_iter(buffer,&start);
	gtk_text_buffer_get_end_iter(buffer,&end);

	if (g->inserted_note == INSERTED)
		g_free (g->current_note->title);

	g->current_note->title = gtk_editable_get_chars( GTK_EDITABLE( entrySubject ), 0, -1 );
	g->current_note->text = gtk_text_buffer_get_text(buffer,&start, &end,TRUE);
	
	/* if the subject is empty prompt the user */
	if (strlen (g->current_note->title) == 0)
	{
		GtkWidget *dialog;

		dialog = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
				(gchar *) _("The note has no subject, please find one"));

		gtk_dialog_run ((GtkDialog *) dialog);
		gtk_widget_destroy (dialog);
		return FALSE;
	}

	/* update the lastUpdate field */
	g_date_set_time (g->current_note->lastUpdate, time (NULL));

	if ( g->inserted_note == NEW )
	{
		if ( !nb_notelist_update_file (g->nl, g->current_note, ADD_NOTE) )
		{
			GtkWidget *dialog;

			dialog = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
					(gchar *) _("Failed to save note in file"));

			gtk_dialog_run ((GtkDialog *) dialog);
			gtk_widget_destroy (dialog);
			return FALSE;
		}
	}
	else
	{
		if ( !nb_notelist_update_file (g->nl, g->current_note, UPDATE_NOTE) )
		{
			GtkWidget *dialog;

			dialog = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
					(gchar *) _("Failed to save note in file"));

			gtk_dialog_run ((GtkDialog *) dialog);
			gtk_widget_destroy (dialog);
			return FALSE;

		}
	}

	/* if the note is new then add it to the list */
	if (g->inserted_note == NEW)
	{
		g->inserted_note = INSERTED;
		nb_notelist_add (g->nl, g->current_note);
		nb_global_add_note_to_tree (g);
	}
	else
	{
		/* update the last_update field */
		g_date_set_time (g->current_note->lastUpdate, time (NULL));
		nb_global_update_tree_with_current (g);
	}

	g->modified_note = CLEAN;
	
	return TRUE;
}

void
nb_global_add_note_to_tree (global * g)
{
	gchar * type;
	gchar * status;
	gchar * date;
	gchar * lastupdate;

	if (g->current_note->type == NOTE)
		type = "Note";
	else if (g->current_note->type == PERSONAL)
		type = "Personal";
	else if (g->current_note->type == PROBLEM)
		type = "Problem";
	else if (g->current_note->type == TRICK)
		type = "Trick";
	else if (g->current_note->type == BOOKMARK)
		type = "Bookmark";

	if (g->current_note->status == NONE)
		status = "None";
	else if (g->current_note->status == OPEN)
		status = "Open";
	else if (g->current_note->status == SOLVED)
		status = "Solved";

	/* convert the date and lastUpdate to String */
	date = nb_note_get_string_from_date (g->current_note->date);
	lastupdate = nb_note_get_string_from_date (g->current_note->lastUpdate);

	add_note_to_tree (g->current_note->title,
			  type,
			  status,
			  date,
			  lastupdate,
			  g->current_note,
			  TRUE);

	g_free (date);
	g_free (lastupdate);
}

void
nb_global_update_tree_with_current (global * g)
{
	gchar * type;
	gchar * status;
	gchar * lastupdate;

	if (g->current_note->type == NOTE)
		type = (gchar *) _("Note");
	else if (g->current_note->type == PERSONAL)
		type = (gchar *) _("Personal");
	else if (g->current_note->type == PROBLEM)
		type = (gchar *) _("Problem");
	else if (g->current_note->type == TRICK)
		type = (gchar *) _("Trick");
	else if (g->current_note->type == BOOKMARK)
		type = (gchar *) _("Bookmark");

	if (g->current_note->status == NONE)
		status = (gchar *) _("None");
	else if (g->current_note->status == OPEN)
		status = (gchar *) _("Open");
	else if (g->current_note->status == SOLVED)
		status = (gchar *) _("Solved");

	lastupdate = nb_note_get_string_from_date (g->current_note->lastUpdate);

	update_tree_with_current (g->current_note->title,
				  type,
				  status,
				  lastupdate);

	g_free (lastupdate);
}

gchar *
nb_global_parse_config (global * g)
{
	GConfClient * client;
	gchar * notefile = NULL;
	gchar * browser = NULL;
	gchar * configpath = NULL;

	client = gconf_client_get_default ();

	notefile = gconf_client_get_string (client, "/apps/notebook/config/notes_file", NULL);
	browser = gconf_client_get_string (client, "/desktop/gnome/url-handlers/http/command", NULL);

	if ( browser == NULL )
		browser = gconf_client_get_string (client, "/apps/notebook/config/browser", NULL);
	
	if ( notefile == NULL)
	{
		/* try to create the config directory if it doesn't already exist */
		configpath = g_strdup_printf ("%s/%s", g_get_home_dir (), DEFAULT_CONFIG_DIR);

		if ( mkdir ( configpath, 0777) == -1 )
		{
			if ( errno != EEXIST )
			{
				perror ("Can't create directory: ");
				exit (EXIT_FAILURE);
			}
		}

		notefile = g_strdup_printf ("%s/%s/%s", g_get_home_dir (), DEFAULT_CONFIG_DIR, DEFAULT_NOTES_FILE);
		gconf_client_set_string (client, "/apps/notebook/config/notes_file", notefile, NULL);
	}

	if ( browser == NULL)
	{
		browser = g_strdup_printf ("mozilla-firefox");
		gconf_client_set_string (client, "/apps/notebook/config/browser", browser, NULL);
	}
	
	g->default_browser = browser;

	/* define callback on browser change */
	gconf_client_add_dir (client, "/desktop/gnome/url-handlers/http", GCONF_CLIENT_PRELOAD_NONE, NULL);
	gconf_client_notify_add(client, "/desktop/gnome/url-handlers/http/command", nb_global_browser_changed_callback, g, NULL, NULL);

	return (notefile);
}

void
nb_global_browser_changed_callback (GConfClient *client, guint cnxn_id, GConfEntry *entry, gpointer user_data)
{
	extern global * g;

	if ( (entry->value != NULL) && (entry->value->type == GCONF_VALUE_STRING) )
	{
		g->default_browser = (gchar * ) gconf_value_get_string(entry->value);
	}
}


void
nb_global_import_notelist (global * g, const gchar * filename)
{
	int i;
	note * n = g_malloc (sizeof (note));
	notelist * nl = nb_notelist_new (filename);

	if ( !nb_notelist_load_from_file (nl) )
		return;

	for ( i = 0 ; i < nl->notes->len ; i++ )
	{
		/* get the note */
		n = g_ptr_array_index (nl->notes, i);

		/* get note text from file */
		nb_note_set_text (n, nb_notelist_get_text_from_index (nl, i));

		/* update the file */
		if ( !nb_notelist_update_file (g->nl, n, ADD_NOTE) )
		{
			GtkWidget *dialog;

			dialog = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
					(gchar *) _("Failed to save note in file"));

			gtk_dialog_run ((GtkDialog *) dialog);
			gtk_widget_destroy (dialog);
			return ;
		}

		/* add the note to the list */
		nb_notelist_add (g->nl, n);
	}

	/* update the tree view */
	update_tree_with_result ("");
	push_statusbar ( g_strdup_printf ((gchar *) _("Successfully imported %d notes."), nl->notes->len));

	//FIXME: we probably need to free nl here
}

void
nb_global_export_notelist (global * g, exporttype * etype, const gchar * filename)
{
	int i, nb = 0;
	note * n = g_malloc (sizeof (note));
	notelist * nl = nb_notelist_new (filename);

	for ( i = 0 ; i < g->nl->notes->len ; i++ )
	{
		/* get the note */
		n = g_ptr_array_index (g->nl->notes, i);

		if ( nb_note_matches_type (n, etype))
		{
			/* get note text from file */
			nb_note_set_text (n, nb_notelist_get_text_from_index (g->nl, i));

			/* update the file */
			if ( !nb_notelist_update_file (nl, n, ADD_NOTE) )
			{
				GtkWidget *dialog;

				dialog = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
						(gchar *) _("Failed to save note in file"));

				gtk_dialog_run ((GtkDialog *) dialog);
				gtk_widget_destroy (dialog);

				nb_notelist_free (nl);
				return ;
			}
			nb++;
		}
	}

	push_statusbar ( g_strdup_printf ((gchar *) _("Successfully exported %d notes."), nb));
	nb_notelist_free (nl);
}

gboolean
nb_global_quit (global * g)
{
	if  ( g->modified_note == EDITED )
	{
		if ( (g->inserted_note == INSERTED) || (is_empty_note () == 0) )
		{
			GtkWidget *dialog;
			int button;

			dialog = gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
					(gchar *) _("Do you want to save note?"));

			g_object_set (dialog, "title", _("Save note?"), "allow_grow", TRUE, NULL);
			button = gtk_dialog_run ((GtkDialog *) dialog);
			gtk_widget_destroy (dialog);

			if (button == GTK_RESPONSE_NO)
				return TRUE;

			if ( !nb_global_save_note (g) )
				return FALSE;
		}
	}

	return TRUE;
}
