/*
 *  Copyright (C) 2004 Mathias Andre <mathias@openbrookes.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */


#include <glib.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <stdio.h>
#include <libxml/xmlmemory.h>
#include <libxml/parser.h>

#include "nb_note.h"

note *
nb_note_new (gchar * title, gchar * text, int type, int status, GDate * date, GDate * lastupdate)
{
	note * n = g_malloc (sizeof (note));
	n->title = title;
	n->text = text;
	n->date = date;
	n->lastUpdate = lastupdate;
	n->type = type;
	n->status = status;

	return (n);
}

note *
nb_note_new_empty ()
{
	note * n = g_malloc (sizeof (note));
	n->title = NULL;
	n->text = NULL;
	n->date = g_date_new ();
	n->lastUpdate = g_date_new ();
	g_date_set_time (n->date, time (NULL));
	n->lastUpdate = g_memdup (n->date, sizeof (GDate));
	n->type = NOTE;
	n->status = NONE;

	return (n);
}

note *
nb_note_load_from_file (xmlDocPtr doc, xmlNodePtr cur)
{
	note * n = NULL;
	gboolean invalid = FALSE;
	xmlChar * title = NULL;
	xmlChar * ctype = NULL;
	xmlChar * cstatus = NULL;
	int type;
	int status;
	GDate * date = NULL;
	GDate * lastupdate = NULL;

	/* set default values */
	type = NOTE;
	status = NONE;

	/* get the type */
	ctype = xmlGetProp (cur, "type");

	if ( !xmlStrcmp (ctype, (const xmlChar *) "personal") )
		type = PERSONAL;
	else if ( !xmlStrcmp (ctype, (const xmlChar *) "problem") )
		type = PROBLEM;
	else if ( !xmlStrcmp (ctype, (const xmlChar *) "trick") )
		type = TRICK;
	else if ( !xmlStrcmp (ctype, (const xmlChar *) "bookmark") )
		type = BOOKMARK;

	/* get the status */
	cstatus = xmlGetProp (cur, "status");

	if ( !xmlStrcmp (cstatus, (const xmlChar *) "open") )
		status = OPEN;
	else if ( !xmlStrcmp (cstatus, (const xmlChar *) "closed") )
		status = SOLVED;


	cur = cur->xmlChildrenNode;
	

	while ( cur != NULL )
	{
		/* get the title */
		if ( !xmlStrcmp (cur->name, (const xmlChar *) "title") )
		{
			title = xmlNodeListGetString (doc, cur->xmlChildrenNode, 1);
		}


		/* get the date */
		if ( !xmlStrcmp (cur->name, (const xmlChar *) "date") )
		{
			date = nb_note_load_date_from_file (doc, cur);

			if ( date == NULL ) 
				invalid = TRUE;
		}

		/* get the lastupdate */
		if ( !xmlStrcmp (cur->name, (const xmlChar *) "lastupdate") )
		{
			lastupdate = nb_note_load_date_from_file (doc, cur);

			if ( lastupdate == NULL )
				invalid = TRUE;
		}

		cur = cur->next;
	}

	if ( lastupdate == NULL )
		lastupdate = g_memdup (date, sizeof (GDate));

	if (invalid)
	{
		if ( title != NULL )
			xmlFree (title);

		g_date_free (date);
		g_date_free (lastupdate);

		return NULL;
	}
	else
	{
		n = nb_note_new (title, NULL, type, status, date, lastupdate);
		return n;
	}
}

GDate *
nb_note_load_date_from_file (xmlDocPtr doc, xmlNodePtr cur)
{
	GDate * date;
	int day;
	int month;
	int year;

	cur = cur->xmlChildrenNode;

	while ( cur != NULL )
	{
		if ( !xmlStrcmp (cur->name, (const xmlChar *) "day") )
			day = atoi (xmlNodeListGetString (doc, cur->xmlChildrenNode, 1));

		if ( !xmlStrcmp (cur->name, (const xmlChar *) "month") )
			month = atoi (xmlNodeListGetString (doc, cur->xmlChildrenNode, 1));

		if ( !xmlStrcmp (cur->name, (const xmlChar *) "year") )
			year = atoi (xmlNodeListGetString (doc, cur->xmlChildrenNode, 1));

		cur = cur->next;
	}

	date = g_date_new_dmy (day, month, year);

	if ( g_date_valid (date) )
		return date;

	return NULL;
}


void
nb_note_clear_text (note * n)
{
	g_free (n->text);
	n->text = NULL;
}

gchar *
nb_note_get_text_from_file (xmlDocPtr doc, xmlNodePtr cur)
{
	cur = cur->xmlChildrenNode;

	xmlChar * text = NULL;

	while ( cur != NULL )
	{
		if ( !xmlStrcmp (cur->name, (const xmlChar *) "content") )
		{
			text = xmlNodeListGetString (doc, cur->xmlChildrenNode, 1);
			break;
		}

		cur = cur->next;
	}

	return text;
}

void 
nb_note_set_title (note * n, gchar * title)
{
	g_free (n->title);
	n->title = title;
}

void 
nb_note_set_text (note * n, gchar * text)
{
	g_free (n->text);
	n->text = text;
}

void
nb_note_set_type (note * n, int type)
{
	n->type = type;
}

void
nb_note_set_status (note * n, int status)
{
	n->status = status;
}

xmlNodePtr
nb_note_get_new_pointer	(note * n, xmlNsPtr ns, int index)
{
	xmlNodePtr cur;
	xmlNodePtr date;
	xmlNodePtr lastupdate;
	xmlChar * type;
	xmlChar * status;
	xmlChar cindex[5];
	xmlChar cday[3];
	xmlChar cmonth[3];
	xmlChar cyear[5];
	xmlChar cluday[3];
	xmlChar clumonth[3];
	xmlChar cluyear[5];


	/* create a new note node */
	cur = xmlNewNode (ns, "note");

	/* add the index as a node attribute */
	snprintf (cindex, 4, "%d", index);
	xmlNewProp (cur, "index", cindex);

	/* add the type as a node attribute */
	if ( n->type == NOTE )
		type = "note";
	else if ( n->type == PERSONAL )
		type = "personal";
	else if ( n->type == PROBLEM )
		type = "problem";
	else if ( n->type == TRICK )
		type = "trick";
	else 
		type = "bookmark";

	xmlNewProp (cur,"type", type);

	/* add the status as a node attribute */
	if ( n->status == NONE )
		status = "none";
	else if ( n->status == OPEN )
		status = "open";
	else
		status = "solved";

	xmlNewProp (cur, "status", status);

	/* set the children nodes */
	xmlNewTextChild (cur, ns, "title", n->title);
	xmlNewTextChild (cur, ns, "content", n->text);

	date = xmlNewTextChild (cur, ns, "date", NULL);

	snprintf (cday, 3, "%d", g_date_get_day (n->date));
	snprintf (cmonth, 3, "%d", g_date_get_month (n->date));
	snprintf (cyear, 5, "%d", g_date_get_year (n->date));

	xmlNewTextChild (date, ns, "day", cday);
	xmlNewTextChild (date, ns, "month", cmonth);
	xmlNewTextChild (date, ns, "year", cyear);

	lastupdate = xmlNewTextChild (cur, ns, "lastupdate", NULL);

	snprintf (cluday, 3, "%d", g_date_get_day (n->lastUpdate));
	snprintf (clumonth, 3, "%d", g_date_get_month (n->lastUpdate));
	snprintf (cluyear, 5, "%d", g_date_get_year (n->lastUpdate));

	xmlNewTextChild (lastupdate, ns, "day", cluday);
	xmlNewTextChild (lastupdate, ns, "month", clumonth);
	xmlNewTextChild (lastupdate, ns, "year", cluyear);

	return cur;
}

gboolean nb_note_matches_query (note * n, gchar * query)
{
	if ( g_strrstr ( g_utf8_casefold (n->title, -1), query) != NULL )
		return TRUE;
	else if ( ( n->text != NULL ) && ( g_strrstr (g_utf8_casefold (n->text, -1), query) != NULL ) )
		return TRUE;

	return FALSE;
}

gboolean
nb_note_matches_type (note * n, exporttype * etype)
{
	if ( (n->type == NOTE) && (etype->note) )
		return TRUE;
	if ( (n->type == PERSONAL) && (etype->personal) )
		return TRUE;
	if ( (n->type == PROBLEM) && (etype->problem) )
		return TRUE;
	if ( (n->type == TRICK) && (etype->trick) )
		return TRUE;
	if ( (n->type == BOOKMARK) && (etype->bookmark) )
		return TRUE;

	return FALSE;
}

gchar *
nb_note_get_string_from_date	(GDate * date)
{
	gchar * cdate = NULL;

	/* not very clean, should try to make it dynamic */
	cdate = g_malloc (128 * sizeof (gchar));
	g_date_strftime (cdate, 127, "%A %d %b", date);

	return cdate;
}
