////////////////////////////////////////////////////////////////////////////
// NoteCase notes manager project <http://notecase.sf.net>
//
// This code is licensed under BSD license.See "license.txt" for more details.
//
// File: Class that stores/manipulates NoteCase document contents 
////////////////////////////////////////////////////////////////////////////

#ifndef NOTEDOCUMENT_H__
#define NOTEDOCUMENT_H__

//
// Document contains a list of nodes that can be accessed in more than one way:
//  - by using index of the node in the node list (not unique, since on node deletion index is reused)
//  - by using node ID (garaneteed to be unique during the lifetime of the document)
//
// Hierarchy of the nodes within the document is described with node (NodeNode class) properties:
//  - node parent ID (m_nParentID)
//     indicates node's parent node, or equals -1 if the node has no parent (root node)
//  - node sibling index (m_nSiblingIdx)
//     defines ordering of the nodes having the same parent. Node with lower sibling index
//     comes before the node with higher sibling index and having the same parent node
// 
// Several document formats are supported (all are implemented by inheriting from NoteFormatIO base class).
// Some formats are used only for importing and have no write support.
// Basic .ncd format is HTML based, and inherits HtmlParser class (overriding some parser event handlers)
//

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

//TOFIX add current node idx inside ? 

#include "NoteFormatIO.h"
#include "NoteNode.h"
#include <vector>
#include <string>

class NoteFormatIO;

typedef const char *(* PASS_CALLBACK)(const char *);

//define Load operation results
#define DOC_LOAD_OK             0
#define DOC_LOAD_ABORTED        1	//aborted loading new document
#define DOC_LOAD_NOT_FOUND      2	//file not found
#define DOC_LOAD_WRONG_PASSWORD 3	//invalid password
#define DOC_LOAD_WRONG_FORMAT   4	//unsuported format
#define DOC_LOAD_FORMAT_ERROR   5	//parse error
#define DOC_LOAD_ERROR          6	//other error


class NoteDocument  
{
	friend class DocumentIterator;

public:
	NoteDocument();
	virtual ~NoteDocument();

	NoteDocument(const NoteDocument &doc);
	void operator = (const NoteDocument &doc);

	void AssignSubtree(NoteDocument &doc, int nIdx);

	int  Load(const char *szPath);
	bool Save(const char *szPath, bool bRemberPath = true, const char *szPassword = NULL);
	void Close();
	void Dump();	//debugging helper

	void Clear();

	bool IsOpen();
	bool IsEmpty();
	bool IsModified();
	void SetModified(bool bModified);
	bool IsEncrypted();
	const char *GetPassword(){ return m_strPassword.c_str(); }

	void SetPassCallback(PASS_CALLBACK pfn){ m_pfnPassCallback = pfn; }

	int  GetFormat();
	std::string	 GetPath(){ return m_strPath; };	//if attached to file

	bool NodeInsert(int nParentID, int nPreviousID);
	bool NodeRename(int nID, const char *szTitle);
	bool NodeDelete(int nID);
	bool NodeMove(int nID, int nNewParentID, int nSiblingIdx);

	int  GetNodeCount() const;
	NoteNode &GetNodeByIdx(int nIdx);
	
	int GetIdxFromID(int nID);
	bool Merge(NoteDocument &doc, int nParentID = -1, int nSiblingIdx = -1, bool bKeepIDs = false);

	int  m_nCurrentNode;	//initialy active node

protected:
	void NodeDeleteRecursive(int nID);

protected:
	NoteFormatIO	*m_pObjLoader;
	std::vector<NoteNode> m_lstNodes;
	bool m_bModified;	
	int  m_nIDGenerator;
	std::string	m_strPath;
	std::string	m_strPassword;
	PASS_CALLBACK m_pfnPassCallback;
};

#endif // NOTEDOCUMENT_H__
