////////////////////////////////////////////////////////////////////////////
// NoteCase notes manager project <http://notecase.sf.net>
//
// This code is licensed under BSD license.See "license.txt" for more details.
//
// File: Implements file operations with large file support (64-bit size)
////////////////////////////////////////////////////////////////////////////

#include "File64.h"
#include "debug.h"

#ifdef _WIN32
#else
 #include <sys/types.h>
 #include <sys/stat.h>
 #include <fcntl.h>
 #include <unistd.h>
 #include <errno.h>
#endif

File64::File64()
{
#ifdef _WIN32
	m_hFile = INVALID_HANDLE_VALUE;
#else
	m_nFile	= -1;
#endif
}

File64::~File64()
{
	Close();
}

bool File64::Open(const char *szPath, unsigned long nFlags)
{
#ifdef _WIN32
	DWORD dwDesiredAccess = 0;
	if(nFlags & F64_READ)
		dwDesiredAccess |= GENERIC_READ;
	if(nFlags & F64_WRITE)
		dwDesiredAccess |= GENERIC_WRITE;

	DWORD dwShareMode = 0;
	if(nFlags & F64_SHARE_READ)
		dwShareMode |= FILE_SHARE_READ;
	if(nFlags & F64_SHARE_WRITE)
		dwShareMode |= FILE_SHARE_WRITE;

	DWORD dwCreationDistribution = 0;
	if(nFlags & F64_OPEN_NEW)
		dwCreationDistribution |= CREATE_ALWAYS;
	if(nFlags & F64_OPEN_EXISTING)
		dwCreationDistribution |= OPEN_EXISTING;
	if((nFlags & (F64_OPEN_NEW|F64_OPEN_EXISTING)) == (F64_OPEN_NEW|F64_OPEN_EXISTING))
		dwCreationDistribution = OPEN_ALWAYS;

	ASSERT(dwCreationDistribution != 0);

	HANDLE hFile = ::CreateFile(szPath,	
								dwDesiredAccess,
								dwShareMode,
								NULL,	//lpSecurityAttributes,
								dwCreationDistribution,	
								0,		//dwFlagsAndAttributes,
								NULL);	//hTemplateFile);
	
	bool bSuccess = false;
	if (hFile != INVALID_HANDLE_VALUE)
	{
		bSuccess = true;
		m_hFile = hFile;
	}
	
	return bSuccess;
#else
	int nOpenFlags = 0;
	if(nFlags & F64_READ)
		nOpenFlags |= O_RDONLY;
	if(nFlags & F64_WRITE)
		nOpenFlags |= O_WRONLY;
	if((nFlags & (F64_READ|F64_WRITE)) == (F64_READ|F64_WRITE))
		nOpenFlags = O_RDWR;
	if(nFlags & F64_OPEN_NEW)
		nOpenFlags |= O_CREAT;
	if((nFlags & (F64_OPEN_NEW|F64_OPEN_EXISTING)) == (F64_OPEN_NEW|F64_OPEN_EXISTING))
		nOpenFlags |= O_TRUNC;

 #ifdef O_LARGEFILE
	nOpenFlags |= O_LARGEFILE;
 #endif
	
	m_nFile	= open(szPath, nOpenFlags, 0600);	//default permissions are set to 0600
	return (m_nFile != -1);
#endif
}

void File64::Close()
{
#ifdef _WIN32
	if (IsOpen())
		::CloseHandle(m_hFile);
	m_hFile = INVALID_HANDLE_VALUE;
#else
	if(IsOpen())
		close(m_nFile);
	m_nFile = -1;
#endif
}

bool File64::IsOpen() const
{
#ifdef _WIN32
	return (m_hFile != INVALID_HANDLE_VALUE); 
#else
	return (m_nFile != -1);
#endif
}

void File64::Flush()
{
	ASSERT(IsOpen());

#ifdef _WIN32	
	::FlushFileBuffers(m_hFile);
#else
	fsync(m_nFile);
#endif
}

int	 File64::Read(char *szBuffer, int nLen)
{
	ASSERT(IsOpen());

#ifdef _WIN32
	DWORD dwRead;
	if (!::ReadFile(m_hFile, (void *)szBuffer, nLen, &dwRead, NULL))
		return -1;

	return dwRead;
#else
	return read(m_nFile, (void *)szBuffer, nLen);
#endif
}

int	File64::Write(const char *szBuffer, int nLen)
{
	ASSERT(IsOpen());

#ifdef _WIN32
	unsigned long nWritten;
	if (!::WriteFile(m_hFile, (void *)szBuffer, nLen, &nWritten, NULL))
		return -1;

	if (nWritten != (unsigned int)nLen)
		return -1;

	return nWritten;
#else
	int nWritten;

	nWritten = write(m_nFile, szBuffer, nLen);
	while(-1 == nWritten && EINTR == errno)
		nWritten = write(m_nFile, szBuffer, nLen);	// retry if interrupted

	return nWritten;
#endif
}

INT64 File64::Seek(INT64 offset)
{
	ASSERT(IsOpen());

#ifdef _WIN32
	LARGE_INTEGER li;
	li.QuadPart = offset;
	li.LowPart  = ::SetFilePointer(m_hFile, li.LowPart, &li.HighPart, FILE_BEGIN);

	DWORD dw = GetLastError();
	if ((li.LowPart == 0xFFFFFFFF) && (dw != NO_ERROR))
		return -1;

	return li.QuadPart;
#else
	return lseek(m_nFile, offset, SEEK_SET);
#endif
}

INT64 File64::SeekEnd()
{
	ASSERT(IsOpen());

#ifdef _WIN32
	LARGE_INTEGER li;
	li.QuadPart = 0;
	li.LowPart  = ::SetFilePointer(m_hFile, li.LowPart, &li.HighPart, FILE_END);

	DWORD dw = GetLastError();
	if ((li.LowPart == 0xFFFFFFFF) && (dw != NO_ERROR))
		return -1;

	return li.QuadPart;
#else
	return lseek(m_nFile, 0, SEEK_END);
#endif
}

INT64 File64::GetPosition() const
{
	ASSERT(IsOpen());

#ifdef _WIN32
	LARGE_INTEGER li;

	li.HighPart = 0;
	li.LowPart  = ::SetFilePointer(m_hFile, 0, (LONG*)&li.HighPart, FILE_CURRENT);
	if (li.LowPart  == 0xFFFFFFFF)
		return -1;

	return li.QuadPart;
#else

	return lseek(m_nFile, 0, SEEK_CUR);	//just return pos
#endif
}

int File64::ReadLine(char *szBuffer, int nLen)
{
	INT64 nPos = GetPosition();

	int nRead = Read(szBuffer, nLen);
	if(nRead > 0)
	{
		szBuffer[nLen-1] = '\0';

		char *szPos = strchr(szBuffer, '\n');
		if(szPos){
			int nBufPos = szPos - szBuffer;
			if(nBufPos < nLen-1){
				if(nBufPos > 0 && szBuffer[nBufPos-1] == '\r')
					*(szPos-1) = '\0';
				else
					*(szPos-2) = '\0';
			}
			Seek(nPos + nBufPos + 1);
		}
	}

	return nRead;
}
