/*
   group.c - group entry lookup routines
   Parts of this file were part of the nss_ldap library (as ldap-grp.c) which
   has been forked into the nss-ldapd library.

   Copyright (C) 1997-2006 Luke Howard
   Copyright (C) 2006 West Consulting
   Copyright (C) 2006, 2007 Arthur de Jong

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
   02110-1301 USA
*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
/* for gid_t */
#include <grp.h>

#include "common.h"
#include "log.h"
#include "myldap.h"
#include "cfg.h"
#include "attmap.h"

/* ( nisSchema.2.2 NAME 'posixGroup' SUP top STRUCTURAL
 *   DESC 'Abstraction of a group of accounts'
 *   MUST ( cn $ gidNumber )
 *   MAY ( userPassword $ memberUid $ description ) )
 *
 * apart from that the above the uniqueMember attributes may be
 * supported in a coming release (they map to DNs, which is an extra
 * lookup step)
 *
 * using nested groups (groups that are member of a group) is currently
 * not supported, this may be added in a later release
 */

/* the search base for searches */
const char *group_base = NULL;

/* the search scope for searches */
int group_scope = LDAP_SCOPE_DEFAULT;

/* the basic search filter for searches */
const char *group_filter = "(objectClass=posixGroup)";

/* the attributes to request with searches */
const char *attmap_group_cn            = "cn";
const char *attmap_group_userPassword  = "userPassword";
const char *attmap_group_gidNumber     = "gidNumber";
const char *attmap_group_memberUid     = "memberUid";
/*
const char *attmap_group_uniqueMember  = "uniqueMember";
const char *attmap_group_memberOf      = "memberOf";
*/

/* default values for attributes */
static const char *default_group_userPassword     = "*"; /* unmatchable */


/* the attribute list to request with searches */
static const char *group_attrs[6];

/* create a search filter for searching a group entry
   by name, return -1 on errors */
static int mkfilter_group_byname(const char *name,
                                 char *buffer,size_t buflen)
{
  char buf2[1024];
  /* escape attribute */
  if(myldap_escape(name,buf2,sizeof(buf2)))
    return -1;
  /* build filter */
  return mysnprintf(buffer,buflen,
                    "(&%s(%s=%s))",
                    group_filter,
                    attmap_group_cn,buf2);
}

/* create a search filter for searching a group entry
   by gid, return -1 on errors */
static int mkfilter_group_bygid(gid_t gid,
                                char *buffer,size_t buflen)
{
  return mysnprintf(buffer,buflen,
                    "(&%s(%s=%d))",
                    group_filter,
                    attmap_group_gidNumber,gid);
}

/* create a search filter for searching a group entry
   by member uid, return -1 on errors */
static int mkfilter_group_bymember(const char *uid,
                                   char *buffer,size_t buflen)
{
  return mysnprintf(buffer,buflen,
                    "(&%s(%s=%s))",
                    group_filter,
                    attmap_group_memberUid,uid);
}

static void group_init(void)
{
  /* set up base */
  if (group_base==NULL)
    group_base=nslcd_cfg->ldc_base;
  /* set up scope */
  if (group_scope==LDAP_SCOPE_DEFAULT)
    group_scope=nslcd_cfg->ldc_scope;
  /* set up attribute list */
  group_attrs[0]=attmap_group_cn;
  group_attrs[1]=attmap_group_userPassword;
  group_attrs[2]=attmap_group_memberUid;
  group_attrs[3]=attmap_group_gidNumber;
  group_attrs[4]=NULL;
/* group_attrs[4]=attmap_group_uniqueMember; */
}

/* the maximum number of gidNumber attributes per entry */
#define MAXGIDS_PER_ENTRY 5

static int write_group(TFILE *fp,MYLDAP_ENTRY *entry,const char *reqname,
                       const gid_t *reqgid,int wantmembers)
{
  int32_t tmpint32,tmp2int32,tmp3int32;
  const char *tmparr[2];
  const char **names,**gidvalues;
  const char *passwd;
  const char **memberuidvalues;
  gid_t gids[MAXGIDS_PER_ENTRY];
  int numgids;
  char *tmp;
  int i,j;
  /* get group name (cn) */
  if (reqname!=NULL)
  {
    names=tmparr;
    names[0]=reqname;
    names[1]=NULL;
  }
  else
  {
    names=myldap_get_values(entry,attmap_group_cn);
    if ((names==NULL)||(names[0]==NULL))
    {
      log_log(LOG_WARNING,"alias entry %s does not contain %s value",
                          myldap_get_dn(entry),attmap_group_cn);
      return 0;
    }
  }
  /* get the group id(s) */
  if (reqgid!=NULL)
  {
    gids[0]=*reqgid;
    numgids=1;
  }
  else
  {
    gidvalues=myldap_get_values(entry,attmap_group_gidNumber);
    if ((gidvalues==NULL)||(gidvalues[0]==NULL))
    {
      log_log(LOG_WARNING,"alias entry %s does not contain %s value",
                          myldap_get_dn(entry),attmap_group_gidNumber);
      return 0;
    }
    for (numgids=0;(gidvalues[numgids]!=NULL)&&(numgids<=MAXGIDS_PER_ENTRY);numgids++)
    {
      gids[numgids]=(gid_t)strtol(gidvalues[numgids],&tmp,0);
      if ((*(gidvalues[numgids])=='\0')||(*tmp!='\0'))
      {
        log_log(LOG_WARNING,"group entry %s contains non-numeric %s value",
                            myldap_get_dn(entry),attmap_group_gidNumber);
        return 0;
      }
    }
  }
  /* get group passwd (userPassword) (use only first entry) */
  passwd=get_userpassword(entry,attmap_group_userPassword);
  if (passwd==NULL)
    passwd=default_group_userPassword;
  /* TODO: translate passwd value into something returnable */
  /* get group memebers (memberUid) */
  if (wantmembers)
    memberuidvalues=myldap_get_values(entry,attmap_group_memberUid);
  else
    memberuidvalues=NULL;
  /* write entries for all names and gids */
  for (i=0;names[i]!=NULL;i++)
    for (j=0;j<numgids;j++)
    {
      WRITE_INT32(fp,NSLCD_RESULT_SUCCESS);
      WRITE_STRING(fp,names[i]);
      WRITE_STRING(fp,passwd);
      WRITE_TYPE(fp,gids[j],gid_t);
      if (memberuidvalues!=NULL)
        { WRITE_STRINGLIST(fp,memberuidvalues); }
      else
        { WRITE_INT32(fp,0); }
    }
  return 0;
}

NSLCD_HANDLE(
  group,byname,
  char name[256];
  char filter[1024];
  READ_STRING_BUF2(fp,name,sizeof(name));,
  log_log(LOG_DEBUG,"nslcd_group_byname(%s)",name);,
  NSLCD_ACTION_GROUP_BYNAME,
  mkfilter_group_byname(name,filter,sizeof(filter)),
  write_group(fp,entry,name,NULL,1)
)

NSLCD_HANDLE(
  group,bygid,
  gid_t gid;
  char filter[1024];
  READ_TYPE(fp,gid,gid_t);,
  log_log(LOG_DEBUG,"nslcd_group_bygid(%d)",(int)gid);,
  NSLCD_ACTION_GROUP_BYGID,
  mkfilter_group_bygid(gid,filter,sizeof(filter)),
  write_group(fp,entry,NULL,&gid,1)
)

NSLCD_HANDLE(
  group,bymember,
  char name[256];
  char filter[1024];
  READ_STRING_BUF2(fp,name,sizeof(name)),
  log_log(LOG_DEBUG,"nslcd_group_bymember(%s)",name);,
  NSLCD_ACTION_GROUP_BYMEMBER,
  mkfilter_group_bymember(name,filter,sizeof(filter)),
  write_group(fp,entry,NULL,NULL,0)
)

NSLCD_HANDLE(
  group,all,
  const char *filter;
  /* no parameters to read */,
  log_log(LOG_DEBUG,"nslcd_group_all()");,
  NSLCD_ACTION_GROUP_ALL,
  (filter=group_filter,0),
  write_group(fp,entry,NULL,NULL,1)
)

