/****************************************************************************************/
/*											*/
/* This program is free software; you can redistribute it and/or modify it under the	*/
/* terms of the GNU General Public License as published by the Free Software		*/
/* Foundation; either version 2 of the License, or (at your option) any later version.	*/
/*											*/
/* This program is distributed in the hope that it will be useful, but WITHOUT ANY	*/
/* WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A	*/
/* PARTICULAR PURPOSE. See the GNU General Public License for more details.		*/
/*											*/
/* You should have received a copy of the GNU General Public License along with this	*/
/* program; (See "COPYING"). If not, If not, see <http://www.gnu.org/licenses/>.        */
/*											*/
/*--------------------------------------------------------------------------------------*/
/*											*/
/*  Copyright   Joerg Anders, TU Chemnitz, Fakultaet fuer Informatik, GERMANY           */
/*		ja@informatik.tu-chemnitz.de						*/
/*											*/
/*											*/
/****************************************************************************************/

#include <math.h>
#include <string.h>
#include "chordorrest.h"
#include "voice.h"
#include "note.h"
#include "staff.h"
#include "system.h"
#include "page.h"
#include "beam.h"
#include "tuplet.h"
#include "system.h"
#include "mainwindow.h"
#include "commandlist.h"
#include "commandhistory.h"
#include "ereasechordorrestcommand.h"
#include "ereasenotecommand.h"
#include "newlyricscommand.h"
#include "deletelyricscommand.h"


#define X_POS_PAGE_REL(p) ((getPage()->getContetXpos() + (p)) * zoom_factor - leftx)
#define Y_POS_STAFF_REL(p) (((p) + getSystem()->getYPos() + getStaff()->getBottomPos()) * zoom_factor - topy)
#define X_POS_INVERS_PAGE_REL(p) ((leftx + (p)) / zoom_factor - getPage()->getContetXpos())
#define Y_POS_INVERS_ABS(p) ((topy + (p)) / zoom_factor)
#define X_PS_POS(p) ((DEFAULT_BORDER + LEFT_RIGHT_BORDER + (p)) * PS_ZOOM)
#define Y_PS_POS_STAFF_REL(p) ((height - ((p) + getSystem()->getYPos() + getStaff()->getBottomPos())) * PS_ZOOM)

NedChordOrRest::NedChordOrRest(NedVoice *voice, int type, bool is_hidden, int line, int dot_count, unsigned int length, int head, unsigned int status, unsigned long long time) :
m_notes(NULL), m_beam(NULL), m_tuplet_ptr(NULL), m_tuplet_val(0), m_type(type), m_is_hidden(is_hidden), m_length(length), m_time(time), m_dot_count(dot_count), m_status(status), m_line(line), m_voice(voice), m_midi_time(time),
m_tmp_xpos(-1.0), m_all_shifted(FALSE), m_art_count(0), m_up_art_count(0)
{
	for (int i = 0; i < MAX_LYRICS_LINES; i++) {
		m_lyrics[i] = NULL;
		m_lyrics_extends[i] = NULL;
	}
	if (m_length < NOTE_64) {
		m_type = TYPE_GRACE;
		m_dot_count = 0;
		m_status = 0;
	}
	if (m_type == TYPE_REST) {
		m_ypos = - 3 * LINE_DIST/2.0;
		if (m_length == WHOLE_NOTE) {
			if (m_voice != NULL && m_voice->getVoiceNumber() > 0) {
				m_is_hidden = TRUE;
			}
		}
	}
	else {
		NedNote *note = new NedNote(this, m_line, head, m_status);
		m_notes = g_list_append(m_notes, note);
#ifdef TTT
		static int zz;
		zz++;
		if ((zz % 3) == 0) {
			createLyrics(0, "Хěвел-");
		}
		else if ((zz % 3) == 1) {
			createLyrics(0, "ухăç");
		}
		else {
			//createLyrics(0, "округě");
			createLyrics(0, "Ärger");
		}
#endif
		if (m_voice != NULL) {
			reConfigure();
		}
		xPositNotes();
		m_ypos = 0.0;
	}
	if (m_voice != NULL) {
		computeBbox();
	}
}

NedChordOrRest::NedChordOrRest(NedVoice *voice, int type, unsigned int clef_number, int octave_shift, unsigned long long time, bool dummy) : // clef
m_notes(NULL), m_beam(NULL), m_tuplet_ptr(NULL), m_tuplet_val(0), m_type(type), m_is_hidden(false), m_length(clef_number), m_ypos(0.0), m_time(time), m_dot_count(octave_shift), m_status(0), m_line(3), m_voice(voice), m_midi_time(time),
m_tmp_xpos(-1.0), m_all_shifted(FALSE)
{

	for (int i = 0; i < MAX_LYRICS_LINES; i++) {
		m_lyrics[i] = NULL;
		m_lyrics_extends[i] = NULL;
	}
	if (m_voice != NULL) {
		computeBbox();
	}
}

NedChordOrRest::NedChordOrRest(NedVoice *voice, int type, int keysig, int last_keysig, unsigned long long time) : // keysig
m_notes(NULL), m_beam(NULL), m_tuplet_ptr(NULL), m_tuplet_val(0), m_type(type), m_is_hidden(false), m_length(keysig), m_ypos(0.0), m_time(time), m_dot_count(last_keysig), m_status(0), m_line(3), m_voice(voice), m_midi_time(time),
m_tmp_xpos(-1.0), m_all_shifted(FALSE)
{
	for (int i = 0; i < MAX_LYRICS_LINES; i++) {
		m_lyrics[i] = NULL;
		m_lyrics_extends[i] = NULL;
	}
	if (m_voice != NULL) {
		computeBbox();
	}
}


NedChordOrRest::~NedChordOrRest() {
	GList *lptr;

	for (lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
		delete ((NedNote *) lptr->data);
	}

	g_list_free(m_notes);
	m_notes = NULL;
}

NedChordOrRest *NedChordOrRest::clone(struct addr_ref_str **addrlist, bool keep_beam /* = false */) {
	GList *lptr;
	NedChordOrRest *cl = new NedChordOrRest(*this);
	NedNote *note;
	int i;
	struct addr_ref_str *adr_ptr;

	cl->m_notes = NULL;

	for (lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
		note = new NedNote(*((NedNote *) lptr->data));
		note->setChord(cl);
		if ((adr_ptr = (struct addr_ref_str *) malloc(sizeof(struct addr_ref_str))) == NULL) {
			NedResource::Abort("NedChordOrRest::clone");
		}
		adr_ptr->ori = lptr->data;
		adr_ptr->copy = (void *) note;
		adr_ptr->next = *addrlist;
		*addrlist = adr_ptr;
		cl->m_notes = g_list_append(cl->m_notes, note);
		if (!keep_beam) {
			cl->m_beam = NULL;
		}
	}
	for (i = 0; i < MAX_LYRICS_LINES; i++) {
		if (m_lyrics[i] != NULL) {
			cl->m_lyrics[i] = strdup(m_lyrics[i]);
		}
	}
	return cl;
}

void NedChordOrRest::setLyrics(int line, char *text) {
	if (text[0] == '\0') return;
	if (g_utf8_strlen(text, -1) > MAX_TEXT_LENGTH) return;
	if ((m_lyrics[line] = (char *) malloc(sizeof(gunichar) * MAX_TEXT_LENGTH + 1)) == NULL) {
		NedResource::Abort("NedChordOrRest::setLyrics");
	}
	strcpy(m_lyrics[line], text);
}

void NedChordOrRest::computeLyricsParams() {
	int line;
	cairo_t *cr = NULL;

	for (line = 0; line < MAX_LYRICS_LINES; line++) {
		if (m_lyrics[line] == 0) continue;
		if ((m_lyrics_extends[line] = (cairo_text_extents_t *) malloc(sizeof(cairo_text_extents_t))) == NULL) {
			NedResource::Abort("NedChordOrRest::createLyrics(3)");
		}
		if (cr == NULL) {
			cr = gdk_cairo_create(getMainWindow()->getDrawingArea()->window);
		}
		cairo_select_font_face(cr, LYRICS_FONT, LYRICS_FONT_SLANT, LYRICS_FONT_WEIGHT);
		cairo_set_font_size(cr, 10.0 * 1.2);
		cairo_text_extents(cr, m_lyrics[line], m_lyrics_extends[line]);
	}
	if (cr != NULL)  {
		cairo_destroy (cr);
	}
	m_cursorpos = 0;
}

void NedChordOrRest::createLyrics(int line, char *text) {
	cairo_t *cr;
	if (line < 0 || line >= MAX_LYRICS_LINES) {
		NedResource::Abort("NedChordOrRest::createLyrics(1)");
	}
	if (m_type != TYPE_NOTE) return;
	if (m_voice->getVoiceNumber() > 0) return;
	if (g_utf8_strlen(text, -1) > MAX_TEXT_LENGTH) return;
	if ((m_lyrics[line] = (char *) malloc(sizeof(gunichar) * MAX_TEXT_LENGTH + 1)) == NULL) {
		NedResource::Abort("NedChordOrRest::createLyrics(2)");
	}
	strcpy(m_lyrics[line], text);
	if ((m_lyrics_extends[line] = (cairo_text_extents_t *) malloc(sizeof(cairo_text_extents_t))) == NULL) {
		NedResource::Abort("NedChordOrRest::createLyrics(3)");
	}
	cr = gdk_cairo_create(getMainWindow()->getDrawingArea()->window);
	cairo_select_font_face(cr, LYRICS_FONT, LYRICS_FONT_SLANT, LYRICS_FONT_WEIGHT);
	cairo_set_font_size(cr, 10.0 * 1.2);
	cairo_text_extents(cr, m_lyrics[line], m_lyrics_extends[line]);
	cairo_destroy (cr);
	m_cursorpos = 0;
}

void NedChordOrRest::appendToLyrics(NedCommandList *command_list, int line, char *text) {
	gchar *cptr;
	gchar str1[sizeof(gunichar) * MAX_TEXT_LENGTH + 1];
	gchar str2[sizeof(gunichar) * MAX_TEXT_LENGTH + 1];
	if (line < 0 || line >= MAX_LYRICS_LINES) {
		NedResource::Abort("NedChordOrRest::appendToLyrics(1)");
	}
	
	if (m_voice->getVoiceNumber() > 0) return;
	if (m_type != TYPE_NOTE) return;
	if (m_lyrics[line] == NULL) {
		if (g_utf8_strlen(text, -1) > MAX_TEXT_LENGTH) return;
		if ((m_lyrics[line] = (char *) malloc(sizeof(gunichar) * MAX_TEXT_LENGTH + 1)) == NULL) {
			NedResource::Abort("NedChordOrRest::appendToLyrics(2)");
		}
		if ((m_lyrics_extends[line] = (cairo_text_extents_t *) malloc(sizeof(cairo_text_extents_t))) == NULL) {
			NedResource::Abort("NedChordOrRest::appendToLyrics(3)");
		}
		strcpy(m_lyrics[line], text);
		//strcpy(m_lyrics[line], "Хěве");
		command_list->addCommand(new NedNewLyricsCommand(this, line));
	}
	else {
		if (m_cursorpos < 1) {
			str1[0] = '\0';
		}
		else {
			g_utf8_strncpy(str1, m_lyrics[line], m_cursorpos);
		}
		strcat(str1, text);
		if (m_cursorpos < g_utf8_strlen(m_lyrics[line], -1)) {
			cptr = g_utf8_offset_to_pointer(m_lyrics[line], m_cursorpos);
			strcpy(str2, cptr);
			strcat(str1, str2);
		}
		strcpy(m_lyrics[line], str1);
	}
	m_cursorpos++;
	computeFontExtentions(line);
	computeBbox();
}

bool NedChordOrRest::deleteLyricsLetter(NedCommandList *command_list, int line) {
	bool ret = TRUE;
	gchar *cptr;
	gchar str1[sizeof(gunichar) * MAX_TEXT_LENGTH + 1];
	gchar str2[sizeof(gunichar) * MAX_TEXT_LENGTH + 1];
	if (line < 0 || line >= MAX_LYRICS_LINES) {
		NedResource::Abort("NedChordOrRest::deleteLyricsLetter(1)");
	}
	if (m_type != TYPE_NOTE || m_lyrics[line] == NULL || m_cursorpos < 1) return TRUE;
	if (g_utf8_strlen(m_lyrics[line], -1) < 2) {
		command_list->addCommand(new NedDeleteLyricsCommand(this, line));
		m_lyrics[line] = NULL;
		ret = FALSE;
	}
	else {
		if (m_cursorpos < g_utf8_strlen(m_lyrics[line], -1)) {
			cptr = g_utf8_offset_to_pointer(m_lyrics[line], m_cursorpos);
			strcpy(str2, cptr);
		}
		else {
			str2[0] = '\0';
		}
		if (m_cursorpos > 1) {
			g_utf8_strncpy(str1, m_lyrics[line], m_cursorpos - 1);
		}
		else {
			str1[0] = '\0';
		}
		strcpy(m_lyrics[line], str1);
		strcat(m_lyrics[line], str2);
		m_cursorpos--;
		computeFontExtentions(line);
	}
	computeBbox();
	return ret;
}

void NedChordOrRest::computeFontExtentions(int line) {
	cairo_t *cr;
	if (line < 0 || line >= MAX_LYRICS_LINES) {
		NedResource::Abort("NedChordOrRest::computeFontExtentions");
	}
	if (m_lyrics[line] == NULL) return;
	if (m_cursorpos > strlen(m_lyrics[line])) {
		m_cursorpos = strlen(m_lyrics[line]);
	}
	if (m_cursorpos < 0) {
		m_cursorpos = 0;
	}
	cr = gdk_cairo_create(getMainWindow()->getDrawingArea()->window);
	cairo_select_font_face(cr, LYRICS_FONT, LYRICS_FONT_SLANT, LYRICS_FONT_WEIGHT);
	cairo_set_font_size(cr, 10.0 * 1.2);
	cairo_text_extents(cr, m_lyrics[line], m_lyrics_extends[line]);
	cairo_destroy (cr);
}

bool NedChordOrRest::setLyricsCursor(int line, int pos) {
	if (line < 0 || line >= MAX_LYRICS_LINES) {
		NedResource::Abort("NedChordOrRest::setLyricsCursor");
	}
	switch(pos) {
		case LYR_CURSOR_RIGHT:
			if (m_lyrics[line] == NULL) return false;
			if (m_cursorpos >= g_utf8_strlen(m_lyrics[line], -1)) return false;
			m_cursorpos++;
			break;
		case LYR_CURSOR_LEFT: if (m_cursorpos <= 0) return false;
			m_cursorpos--;
			break;
		case LYR_CURSOR_END:
			if (m_lyrics[line] == NULL) return false;
			m_cursorpos = g_utf8_strlen(m_lyrics[line], -1);
			break;
		case LYR_CURSOR_START: m_cursorpos = 0;
			break;
	}
	return true;
}


void NedChordOrRest::selectNextChord(int line, bool lyrics_required) {
	return m_voice->selectNextChord(this, line, lyrics_required);
}

void NedChordOrRest::selectPreviousChord(int line, bool lyrics_required) {
	return m_voice->selectPreviousChord(this, line, lyrics_required);
}


NedMainWindow *NedChordOrRest::getMainWindow() {return m_voice->getStaff()->getSystem()->getPage()->getMainWindow();}
NedPage *NedChordOrRest::getPage() {return m_voice->getStaff()->getSystem()->getPage();}
NedSystem *NedChordOrRest::getSystem() {return m_voice->getStaff()->getSystem();}
NedStaff *NedChordOrRest::getStaff() {return m_voice->getStaff();}

void NedChordOrRest::setToRest(bool toRest) {
	if (m_type != TYPE_REST && toRest) {
		if (m_beam != NULL) {
			delete m_beam; // includes unbeaming other notes and unregister on voice
		}
	}
	if (toRest) {
		m_type = TYPE_REST;
	}
	else {
		m_type = TYPE_NOTE;
	}
	
	if (m_type == TYPE_REST) {
		m_ypos = - 3 * LINE_DIST/2.0;
	}
	else {
		m_ypos = 0.0;
		reConfigure();
		xPositNotes();
	}
	computeBbox();
}

void NedChordOrRest::reposit(unsigned int pos) {
	m_xpos = 1.0 + 1.0 * pos;
}

int NedChordOrRest::getPitchOfFirstNote() {
	int clef, keysig, octave_shift;
	getStaff()->getCurrentClefAndKeysig(m_midi_time, &clef, &keysig, &octave_shift);
	return ((NedNote *) g_list_first(m_notes)->data)->getPitch(clef, keysig, octave_shift);
}


void NedChordOrRest::insertNoteAt(NedNote *note, bool echo /* = FALSE */) {
	staff_context_str *context;
	int clef, keysig, octave_shift;
	note->setChord(this);
	if (m_voice != NULL) {
		context = &(getMainWindow()->m_staff_contexts[getStaff()->getStaffNumber()]);
		getStaff()->getCurrentClefAndKeysig(m_midi_time, &clef, &keysig, &octave_shift);

		if (echo && NedResource::m_midi_echo) {
			NedResource::playImmediately(context->m_midi_channel, context->m_midi_program, note->getPitch(clef, keysig, octave_shift), context->m_midi_volume);
		}
		xPositNotes();
	}
	m_notes = g_list_insert_sorted(m_notes, note, (gint (*)(const void*, const void*)) NedNote::compare);
}



bool NedChordOrRest::removeNoteAt(int line) {
	GList *lptr;
	for (lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
		if (((NedNote *) lptr->data)->getLine() == line) {
			m_notes = g_list_delete_link(m_notes, lptr);
			xPositNotes();
			return g_list_length(m_notes) > 0;
		}
	}
	NedResource::Abort("NedChordOrRest::removeNoteAt");
	return FALSE;
}

NedChordOrRest *NedChordOrRest::cloneWithDifferentLength(unsigned int newlen, int dotcount) {
	int line, head;
	if (m_type == TYPE_REST) {
		line = 3;
	}
	else {
		NedNote *note = (NedNote *) g_list_first(m_notes)->data;
		line = note->getLine();
	}
	if (m_notes != NULL) {
		head = ((NedNote *) g_list_first(m_notes)->data)->getNoteHead();
	}
	else {
		head = NORMAL_NOTE;
	}
	NedChordOrRest *newChord = new NedChordOrRest(m_voice, m_type, m_is_hidden, line, dotcount, newlen, head, m_status, 0);
	if (m_type != TYPE_REST) {
		GList *lptr = g_list_first(m_notes);
		for (lptr = g_list_next(lptr); lptr; lptr = g_list_next(lptr)) {
			newChord->m_notes = g_list_append(newChord->m_notes, new NedNote(newChord, ((NedNote *) lptr->data)->getLine(), ((NedNote *) lptr->data)->getNoteHead(), 0));
		}
	}
	return newChord;
}

void NedChordOrRest::tieCompleteTo(NedChordOrRest *other) {
	GList *lptr1, *lptr2;
	if (m_type == TYPE_REST || other->m_type == TYPE_REST) return;

	for (lptr1 = g_list_first(m_notes), lptr2 = g_list_first(other->m_notes); lptr1 && lptr2; lptr1 = g_list_next(lptr1), lptr2 = g_list_next(lptr2)) {
		((NedNote *) lptr1->data)->setTieSimple((NedNote *) lptr2->data);
	}
}

void NedChordOrRest::removeAllBackwardTies() {
	GList *lptr;
	if (m_type == TYPE_REST) return;

	for (lptr= g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
		((NedNote *) lptr->data)->removeBackwardTie();
	}
}

bool NedChordOrRest::tieBackward(NedChordOrRest *prev, NedNote *note) {
	GList *lptr;
	int line = note->getLine();

	if (g_list_find(m_notes, note) == NULL) {
		NedResource::Abort("NedChordOrRest::tieBackward");
	}

	for (lptr = g_list_first(prev->m_notes); lptr; lptr = g_list_next(lptr)) {
		if (((NedNote *) lptr->data)->getLine() == line) {
			((NedNote *) lptr->data)->setTies((NedNote *) lptr->data, note);
			return true;
		}
	}
	return false;
}


bool NedChordOrRest::testForTiesToDelete(NedCommandList *command_list, unsigned int dir /* = BREAK_TIE_BACKWARD */, bool execute /* = false */) {
	bool deleted_ties = FALSE, delties;
	for (GList *lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
		delties = ((NedNote *) lptr->data)->testForTiesToDelete(command_list, dir, execute);
		deleted_ties = deleted_ties || delties;
	}
	return deleted_ties;
}

void NedChordOrRest::correctTiesForward() {
	GList *lptr;
	for (lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
		((NedNote *) lptr->data)->correctTieForward();
	}
}

void NedChordOrRest::tieComplete(NedChordOrRest *other_chord) {
	GList *lptr1, *lptr2;
	for (lptr1 = g_list_first(m_notes), lptr2 = g_list_first(other_chord->m_notes); lptr1 && lptr2;
		lptr1 = g_list_next(lptr1), lptr2 = g_list_next(lptr2)) {
		((NedNote *) lptr1->data)->setTieSimple((NedNote *) lptr2->data);
	}
}


bool NedChordOrRest::noConflict(int line) {
	if (m_type == TYPE_REST) {
		NedResource::Abort("NedChordOrRest::noConflict");
	}
	GList *lptr;
	for (lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
		if (((NedNote *) lptr->data)->getLine() == line) {
			return FALSE;
		}
	}
	return TRUE;
}

int NedChordOrRest::getSorting(NedChordOrRest *other_chord) {
	NedVoice *other_voice;
	NedChordOrRest *c0, *c1;
	int sorting;
	if (other_chord == this) return SORTING_NONE;

	if (getPage() != other_chord->getPage()) {
		sorting = getMainWindow()->getSorting(getPage(), getStaff(), other_chord->getPage(), other_chord->getStaff());
		switch (sorting) {
			case SORTING_NONE: return SORTING_NONE;
			case SORTING_LESS: c0 = other_chord; c1 = this; break;
			default: c0 = this; c1 = other_chord; break;
		}
		if (!c0->getPage()->isLast(c0->getSystem())) return SORTING_NONE;
		if (!c1->getPage()->isFirst(c1->getSystem())) return SORTING_NONE;

		if (!c0->m_voice->isLast(c0)) return SORTING_NONE;
		if (!c1->m_voice->isFirst(c1)) return SORTING_NONE;
		return sorting;
	}

	other_voice = other_chord->m_voice;
	if (m_voice != other_voice) {
		sorting = getPage()->getSorting(getStaff(), getSystem(), other_chord->getStaff(), other_chord->getSystem());
		switch (sorting) {
			case SORTING_NONE: return SORTING_NONE;
			case SORTING_LESS: c0 = other_chord; c1 = this; break;
			default: c0 = this; c1 = other_chord; break;
		}

		if (!c0->m_voice->isLast(c0)) return SORTING_NONE;
		if (!c1->m_voice->isFirst(c1)) return SORTING_NONE;
		return sorting;
	}
	return m_voice->getSorting(this, other_chord);
}


void NedChordOrRest::reConfigure() {
	int line_sum = 0;
	unsigned int oldstemdir;
	if (m_type == TYPE_REST) {
		if (m_voice->getStemDir() == STEM_DIR_NONE) {
			m_ypos = - 3 * LINE_DIST/2.0;
		}
		return;
	}
	m_notes = g_list_sort(m_notes,  (gint (*)(const void*, const void*)) NedNote::compare);
	GList *lptr;
	for (lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
		line_sum += ((NedNote *) lptr->data)->getLine();
	}
	oldstemdir = (m_status & STAT_UP);
	switch (m_voice->getStemDir()) {
		case STEM_DIR_UP: m_status |= STAT_UP; if (!oldstemdir) {xPositNotes(); computeBbox();} break;
		case STEM_DIR_DOWN: m_status &= (~(STAT_UP)); if (oldstemdir) {xPositNotes(); computeBbox();} break;
		default:
		int c = 5 * g_list_length(m_notes);
		if (line_sum < c || m_length ==  WHOLE_NOTE) {
			m_status |= STAT_UP;
			if (!oldstemdir) {
				xPositNotes();
				computeBbox();
			}
		}
		else  {
			m_status &= (~(STAT_UP));
			if (oldstemdir) {
				xPositNotes();
				computeBbox();
			}
		}
		break;
	}
	if (m_beam != NULL) {
		if (m_length >= NOTE_4) {
			m_beam->removeChord(this);
			m_beam = NULL;
		}
		else {
			m_beam->computeBeam(getVoice()->getStemDir());
		}
	}
}

void NedChordOrRest::setLength(int length) {
	m_length = length;
	unsigned int upbeat;
	unsigned int measure_length;
	if (m_length == WHOLE_NOTE) {
		if (m_voice->getVoiceNumber() > 0) {
			m_is_hidden = TRUE;
		}
	}
	else if (m_midi_time == 0 && getMainWindow()->getUpBeatInverse() > 0) {
		measure_length = getMainWindow()->getNumerator() * WHOLE_NOTE / getMainWindow()->getDenominator();
		upbeat = measure_length - getMainWindow()->getUpBeatInverse();
		if (m_voice->getVoiceNumber() > 0 && getDuration() == measure_length) {
			m_is_hidden = TRUE;
		}
	}
		
}

bool NedChordOrRest::ReleaseTmpValue() {
	if (m_tmp_xpos < 0.0) return FALSE;
	bool ret = m_voice->setNewXpos(this, m_tmp_xpos);
	m_tmp_xpos = -1.0;
	if (!ret) {
		getMainWindow()->reposit(NULL, getPage());
	}
	return ret;
}

void NedChordOrRest::changeDuration(unsigned int new_duration, int tuplet_val) {
	if (m_tuplet_val != 0) {
		new_duration *= tuplet_val;
		new_duration /= (tuplet_val - 1);
	}
	compute_fitting_duration(new_duration, &m_length, &m_dot_count);
}

void NedChordOrRest::compute_fitting_duration(unsigned int sum, unsigned int *length, int *dot_count) {
	unsigned int dur = NOTE_64;
	while (dur <= sum && dur <= (WHOLE_NOTE >> 1)) {
		dur <<= 1;
	}
	dur >>= 1;
	*length = dur;
	*dot_count = (*length / 2 * 3 <= sum) ? 1 : 0;
}

bool NedChordOrRest::testXShift(double x) {
	double leftx = getMainWindow()->getLeftX();
	double zoom_factor = getMainWindow()->getCurrentZoomFactor();
	double xx = X_POS_INVERS_PAGE_REL(x);
	double xxdiff = xx - m_xpos;

	if (xxdiff < 0.0) xxdiff = -xxdiff;
	if (xxdiff > MIN_SHIFT_X_DIFF) {
		m_tmp_xpos = xx;
		return TRUE;
	}
	return FALSE;
}


void NedChordOrRest::computeBbox() {
	double minx = 10000.0, miny = 10000.0;
	double maxx = -10000.0, maxy = -10000.0;
	double zoom_factor;
	int zoom_level;
	cairo_text_extents_t *extention;
	NedNote *stem_start_note;
	int dir, i;
	double yy;
	int fac, ll, lll;
	double extry_y_offs = 0, off1;
	double maxlyricslength;

	switch (m_type) {
	case TYPE_CLEF:
		zoom_factor = getMainWindow()->getCurrentZoomFactor();
		zoom_level = getMainWindow()->getCurrentZoomLevel();
		extention = &(NedResource::fontextentions[zoom_level][3]);
		m_bbox.x = extention->x_bearing / zoom_factor;
		m_bbox.y = (extention->y_bearing) / zoom_factor + (m_ypos -3 * LINE_DIST/2.0);
		m_bbox.width =  extention->width / zoom_factor;
		m_bbox.height = extention->height / zoom_factor;
		break;
	case TYPE_KEYSIG:
		lll = m_dot_count; // keysig so far
		if (lll < 0) lll = -lll;
		ll = (int) m_length; // "new" keysig
		off1 = (ll > 0) ? (0.5 * LINE_DIST) : 0.0;
		if (ll < 0) ll = -ll;
		m_bbox.x = 0.0;
		m_bbox.y = - 6 * LINE_DIST;
		m_bbox.width = (ll + lll) * SIG_X_SHIFT + off1;
		m_bbox.height = 7 * LINE_DIST;
		break;
	case TYPE_REST:
		zoom_factor = getMainWindow()->getCurrentZoomFactor();
		zoom_level = getMainWindow()->getCurrentZoomLevel();
		switch(m_length) {
		case WHOLE_NOTE:
			extry_y_offs = WHOLE_NOTE_Y_OFFS;
			extention = &(NedResource::fontextentions[zoom_level][9]); break;
		case NOTE_2:
			extry_y_offs = HALF_NOTE_Y_OFFS;
			extention = &(NedResource::fontextentions[zoom_level][10]); break;
		case NOTE_4:
			extention = &(NedResource::fontextentions[zoom_level][11]); break;
		case NOTE_8:
			extention = &(NedResource::fontextentions[zoom_level][12]); break;
		case NOTE_16:
			extention = &(NedResource::fontextentions[zoom_level][13]); break;
		case NOTE_32:
			extention = &(NedResource::fontextentions[zoom_level][16]); break;
		case NOTE_64:
			extention = &(NedResource::fontextentions[zoom_level][15]); break;
		}
		m_bbox.x = extention->x_bearing / zoom_factor;
		m_bbox.y = (extention->y_bearing) / zoom_factor + m_ypos;
		m_bbox.width =  extention->width / zoom_factor;
		m_bbox.height = extention->height / zoom_factor;
		switch (m_dot_count) {
			case 1: m_bbox.width += m_dot_count * DOT1POS + extry_y_offs; break;
			case 2: m_bbox.width += m_dot_count * DOT2POS + extry_y_offs; break;
		}
		break;
	case TYPE_NOTE:
		zoom_factor = getMainWindow()->getCurrentZoomFactor();
		zoom_level = getMainWindow()->getCurrentZoomLevel();
		GList *lptr;
		for (lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
			((NedNote *) lptr->data)->computeBounds(m_length, &minx, &miny, &maxx, &maxy);
		}
		if (m_length < NOTE_4 && m_beam == NULL) {
			if (NedResource::getFlagWidth() > maxx) maxx = NedResource::getFlagWidth();
		}
		if (m_beam != NULL) {
			if (m_beam->isBeamUp()) {
				dir = -1;
				stem_start_note = (NedNote *) g_list_last(m_notes)->data;
			}
			else {
				dir = 1;
				stem_start_note = (NedNote *) g_list_first(m_notes)->data;
			}
			fac = 0;
		}
		else {
			dir = (m_status & STAT_UP) ? -1 : 1;
			if (m_status & STAT_UP) {
				stem_start_note = (NedNote *) g_list_last(m_notes)->data;
			}
			else {
				stem_start_note = (NedNote *) g_list_first(m_notes)->data;
			}
		  	switch(m_length) {
		  	case NOTE_32:
		  		fac = 1; break;
		  	case NOTE_64:
		  		fac = 2; break;
			default: fac = 0; break;
		  	}
		}
		yy = stem_start_note->getYpos() + dir * (STEM_HEIGHT + fac * FLAG_DIST);
		if (yy > maxy) maxy = yy;
		if (yy < miny) miny = yy;
		m_bbox.x = minx; m_bbox.y = miny;
		m_bbox.width = maxx - minx;
		maxlyricslength = 0.0;
		for (i = 0; i < MAX_LYRICS_LINES; i++) {
			if (m_lyrics[i] != NULL) {
				if (m_lyrics_extends[i]->width > maxlyricslength) {
					maxlyricslength = m_lyrics_extends[i]->width;
				}
			}
		}
		if (maxlyricslength > 0.0) {
			if (maxlyricslength * LYRICS_FONT_SIZE / 10.0 > m_bbox.width) {
				m_bbox.width = maxlyricslength * LYRICS_FONT_SIZE / 10.0;
				m_bbox.x = -m_bbox.width / 2.0;
			}
		}
		m_bbox.height = maxy - miny;
		break;
	case TYPE_GRACE:
		zoom_factor = getMainWindow()->getCurrentZoomFactor();
		zoom_level = getMainWindow()->getCurrentZoomLevel();
		extention = &(NedResource::fontextentions[zoom_level][18]); 
		m_bbox.x = extention->x_bearing / zoom_factor;
		m_bbox.y = (extention->y_bearing) / zoom_factor - ((NedNote *) g_list_first(m_notes)->data)->getLine() * LINE_DIST / 2.0;
		m_bbox.width =  extention->width / zoom_factor;
		m_bbox.height = extention->height / zoom_factor;
		break;
	}
}

double NedChordOrRest::compute_microshift(NedChordOrRest *left_chord) {
	if (left_chord->m_bbox.y + TOLERANCE> m_bbox.y +  m_bbox.height) return -1.0;
	if (m_bbox.y + TOLERANCE > left_chord->m_bbox.y +  left_chord->m_bbox.height) return -1.0;
	if (left_chord->m_xpos + left_chord->m_bbox.x > m_xpos + m_bbox.x + m_bbox.width) return -1.0;
	if (left_chord->m_xpos + left_chord->m_bbox.x + left_chord->m_bbox.width < m_xpos + m_bbox.x) return -1.0;
	return m_bbox.x + m_bbox.width - (left_chord->m_bbox.x) + MIN_VOICE_ELEMS_DIST;
}

double NedChordOrRest::compute_y_shift(NedChordOrRest *other_chord, int dir) {
	if (other_chord->m_bbox.y > MIN_REST_Y_VOICE_DIST + m_bbox.y +  m_bbox.height) return -1.0;
	if (m_bbox.y > MIN_REST_Y_VOICE_DIST + other_chord->m_bbox.y +  other_chord->m_bbox.height) return -1.0;
	if (other_chord->m_xpos + other_chord->m_bbox.x > m_xpos + m_bbox.x + m_bbox.width) return -1.0;
	if (other_chord->m_xpos + other_chord->m_bbox.x + other_chord->m_bbox.width < m_xpos + m_bbox.x) return -1.0;
	if (dir == STEM_DIR_DOWN) {
		return m_bbox.y + m_bbox.height - other_chord->m_bbox.y + MIN_REST_Y_VOICE_DIST;
	}
	return other_chord->m_bbox.y + other_chord->m_bbox.height - m_bbox.y + MIN_REST_Y_VOICE_DIST;
}

void NedChordOrRest::sortNotes() {
	m_notes = g_list_sort(m_notes, (gint (*)(const void*, const void*)) NedNote::compare);
}

void NedChordOrRest::setMidiTime(unsigned long long time, bool resetRestYPos /* = FALSE */) {
	m_midi_time = time;
	if (m_type == TYPE_REST && resetRestYPos) {
		m_ypos = - 3 * LINE_DIST/2.0;
		computeBbox();
	}
}

bool NedChordOrRest::hasUpDir() {
	if (m_beam != NULL) {
		return m_beam->isBeamUp();
	}
	return ((m_status & STAT_UP) != 0);
}

void NedChordOrRest::draw(cairo_t *cr, bool *lyrics_present) {
	cairo_glyph_t glyph;
	double leftx = getMainWindow()->getLeftX();
	double topy = getMainWindow()->getTopY();
	double zoom_factor = getMainWindow()->getCurrentZoomFactor();
	int dir, i, ll;
	double sig_xpos, yl;
	int flag_glyph, clef_type, keysig_type, octave_shift;
	double real_xpos = m_tmp_xpos < 0.0 ? m_xpos : m_tmp_xpos;
	double y_start, y_stem_start;
	double oct_shift_y_offs, oct_shift_x_offs = 0.0;
	double clef_y_offs = 0.0;
	bool oct_shift = false;
	int art_nr = 0, up_art_nr = 0;
	int noteline, line;
	switch (m_type) {
	case TYPE_CLEF:
		switch (m_length) { // kind
			case TREBLE_CLEF: glyph.index = BASE + 2;
				  	switch (m_dot_count) { // octave shift
				  		case  12: oct_shift_y_offs = OCTUP_Y_TREBLE_DIST_UP; oct_shift = true; break;
				  		case -12: oct_shift_y_offs = OCTUP_Y_TREBLE_DIST_DOWN; oct_shift = true; break;
				  	}
					break;
			case ALTO_CLEF: glyph.index = BASE + 1;
				  	switch (m_dot_count) {
				  		case  12: oct_shift_y_offs = OCTUP_Y_ALTO_DIST_UP;
						 	oct_shift_x_offs = OCTUP_X_ALTO_DIST;
						 	oct_shift = true; break;
				  		case -12: oct_shift_y_offs = OCTUP_Y_ALTO_DIST_DOWN;
						 	oct_shift_x_offs = OCTUP_X_ALTO_DIST;
						 	oct_shift = true; break;
				  	}
					break;
			case SOPRAN_CLEF: glyph.index = BASE + 1;
				  	switch (m_dot_count) {
				  		case  12: oct_shift_y_offs = OCTUP_Y_ALTO_DIST_UP;
						 	oct_shift_x_offs = OCTUP_X_ALTO_DIST;
						 	oct_shift = true; break;
				  		case -12: oct_shift_y_offs = OCTUP_Y_ALTO_DIST_DOWN;
						 	oct_shift_x_offs = OCTUP_X_ALTO_DIST;
						 	oct_shift = true; break;
				  	}
				  	clef_y_offs = 2 * LINE_DIST;
					break;
			case TENOR_CLEF: glyph.index = BASE + 1;
				  	switch (m_dot_count) {
				  		case  12: oct_shift_y_offs = OCTUP_Y_ALTO_DIST_UP;
						 	oct_shift_x_offs = OCTUP_X_ALTO_DIST;
						 	oct_shift = true; break;
				  		case -12: oct_shift_y_offs = OCTUP_Y_ALTO_DIST_DOWN;
						 	oct_shift_x_offs = OCTUP_X_ALTO_DIST;
						 	oct_shift = true; break;
				  	}
					clef_y_offs = -LINE_DIST;break;
			case NEUTRAL_CLEF1: glyph.index = BASE + 27;
					break;
			case NEUTRAL_CLEF2: glyph.index = BASE + 27;
					break;
			default: glyph.index = BASE + 3;
				  	switch (m_dot_count) {
				  		case  12: oct_shift_y_offs = OCTUP_Y_BASS_DIST_UP; oct_shift = true; break;
				  		case -12: oct_shift_y_offs = OCTUP_Y_BASS_DIST_DOWN; oct_shift = true; break;
				  	}
					break;
		}
		glyph.x = X_POS_PAGE_REL(real_xpos);
		glyph.y = Y_POS_STAFF_REL(m_ypos + clef_y_offs + -3 * LINE_DIST/2.0);
		if (getMainWindow()->m_selected_chord_or_rest == this) {
			cairo_set_source_rgb (cr, 1.0, 0.0, 0.0);
		}
		cairo_show_glyphs(cr, &glyph, 1);
		if (oct_shift) {
			glyph.index = BASE + 28;
			glyph.x = X_POS_PAGE_REL(real_xpos + oct_shift_x_offs);
			glyph.y = Y_POS_STAFF_REL(m_ypos -3 * LINE_DIST/2.0 + clef_y_offs + oct_shift_y_offs);
			cairo_show_glyphs(cr, &glyph, 1);
		}
		cairo_stroke(cr);
		if (getMainWindow()->m_selected_chord_or_rest == this) {
			cairo_set_source_rgb (cr, 0.0, 0.0, 0.0);
		}
		break;
	case TYPE_KEYSIG:
		ll = m_dot_count;
		getStaff()->getCurrentClefAndKeysig(m_time, &clef_type, &keysig_type, &octave_shift);
		sig_xpos = real_xpos;
		if (getMainWindow()->m_selected_chord_or_rest == this) {
			cairo_set_source_rgb (cr, 1.0, 0.0, 0.0);
		}
		if (ll > 0) { // keysig type so far
			cairo_new_path(cr);
			glyph.index = BASE + 17;
			for (i = 0; i < ll; i++) {
				glyph.x = X_POS_PAGE_REL(sig_xpos);
				glyph.y = Y_POS_STAFF_REL(-NedStaff::m_sharpPos[clef_type][i] * LINE_DIST/2.0);
				sig_xpos += SIG_X_SHIFT;
				cairo_show_glyphs(cr, &glyph, 1);
			}
			cairo_stroke(cr);
		}
		else if (ll < 0) {
			cairo_new_path(cr);
			glyph.index = BASE + 17;
			for (i = 0; i > ll; i--) {
				glyph.x = X_POS_PAGE_REL(sig_xpos);
				glyph.y = Y_POS_STAFF_REL(-NedStaff::m_flatPos[clef_type][-i] * LINE_DIST/2.0);
				sig_xpos += SIG_X_SHIFT;
				cairo_show_glyphs(cr, &glyph, 1);
			}
			cairo_stroke(cr);
		}
		ll = (int) m_length;
		if (ll > 0) { // keysig type
			cairo_new_path(cr);
			glyph.index = BASE + 0;
			for (i = 0; i < ll; i++) {
				glyph.x = X_POS_PAGE_REL(sig_xpos);
				glyph.y = Y_POS_STAFF_REL(-NedStaff::m_sharpPos[clef_type][i] * LINE_DIST/2.0);
				sig_xpos += SIG_X_SHIFT;
				cairo_show_glyphs(cr, &glyph, 1);
			}
			cairo_stroke(cr);
		}
		else if (ll < 0) {
			cairo_new_path(cr);
			glyph.index = BASE + 16;
			for (i = 0; i > ll; i--) {
				glyph.x = X_POS_PAGE_REL(sig_xpos);
				glyph.y = Y_POS_STAFF_REL(-NedStaff::m_flatPos[clef_type][-i] * LINE_DIST/2.0);
				sig_xpos += SIG_X_SHIFT;
				cairo_show_glyphs(cr, &glyph, 1);
			}
			cairo_stroke(cr);
		}
		if (getMainWindow()->m_selected_chord_or_rest == this) {
			cairo_set_source_rgb (cr, 0.0, 0.0, 0.0);
		}
		break;
	case TYPE_REST:
#ifndef SHOW_HIDDEN
		if (m_is_hidden) return;
#endif
		glyph.x = X_POS_PAGE_REL(real_xpos);
		glyph.y = Y_POS_STAFF_REL(m_ypos);
#ifdef SHOW_HIDDEN
		if (m_is_hidden) {
			glyph.y = Y_POS_STAFF_REL(m_ypos + LINE_DIST / 2 * m_voice->getVoiceNumber());
		}
#endif
		switch(m_length) {
		case WHOLE_NOTE:
			glyph.index = BASE +  9; break;
		case NOTE_2:
			glyph.index = BASE + 10; break;
		case NOTE_4:
			glyph.index = BASE + 11; break;
		case NOTE_8:
			glyph.index = BASE + 12; break;
		case NOTE_16:
			glyph.index = BASE + 13; break;
		case NOTE_32:
			glyph.index = BASE + 14; break;
		case NOTE_64:
			glyph.index = BASE + 15; break;
		}
		if (getMainWindow()->m_selected_chord_or_rest == this) {
			if (getMainWindow()->doPaintColored()) {
				switch(getVoice()->getVoiceNumber()) {
					case 0: cairo_set_source_rgb (cr, DSCOLOR(V1RED), DSCOLOR(V1GREEN), DSCOLOR(V1BLUE)); break;
					case 1: cairo_set_source_rgb (cr, DSCOLOR(V2RED), DSCOLOR(V2GREEN), DSCOLOR(V2BLUE)); break;
					case 2: cairo_set_source_rgb (cr, DSCOLOR(V3RED), DSCOLOR(V3GREEN), DSCOLOR(V3BLUE)); break;
					default: cairo_set_source_rgb (cr, DSCOLOR(V4RED), DSCOLOR(V4GREEN), DSCOLOR(V4BLUE)); break;
				}
			}
			else {
				cairo_set_source_rgb (cr, 1.0, 0.0, 0.0);
			}
		}
		else if (getMainWindow()->doPaintColored()) {
			switch(getVoice()->getVoiceNumber()) {
			case 0: cairo_set_source_rgb (cr, DCOLOR(V1RED), DCOLOR(V1GREEN), DCOLOR(V1BLUE)); break;
			case 1: cairo_set_source_rgb (cr, DCOLOR(V2RED), DCOLOR(V2GREEN), DCOLOR(V2BLUE)); break;
			case 2: cairo_set_source_rgb (cr, DCOLOR(V3RED), DCOLOR(V3GREEN), DCOLOR(V3BLUE)); break;
			default: cairo_set_source_rgb (cr, DCOLOR(V4RED), DCOLOR(V4GREEN), DCOLOR(V4BLUE)); break;
			}
		}
		cairo_show_glyphs(cr, &glyph, 1);
		if (m_dot_count > 0) {
			double extra_x_offs, extry_y_offs;
			switch (m_length) {
				case WHOLE_NOTE: extra_x_offs = WHOLE_NOTE_X_OFFS; 
						 extry_y_offs = WHOLE_NOTE_Y_OFFS;
						 break;
				case NOTE_2:  extra_x_offs = HALF_NOTE_X_OFFS; 
						 extry_y_offs = HALF_NOTE_Y_OFFS;
						 break;
				default:	 extra_x_offs = 0;
						 extry_y_offs = 0;
						 break;
			}
				
			cairo_new_path(cr);
			cairo_arc(cr, X_POS_PAGE_REL(real_xpos + DOT1_REST_POS +  extra_x_offs), Y_POS_STAFF_REL(m_ypos + extry_y_offs),
					zoom_factor * DOT_RAD, 0, 2.0 * M_PI);
			cairo_fill(cr);
			if (m_dot_count > 1) {
				cairo_arc(cr, X_POS_PAGE_REL(real_xpos + DOT2_REST_POS +  extra_x_offs), Y_POS_STAFF_REL(m_ypos + extry_y_offs),
						zoom_factor * DOT_RAD, 0, 2.0 * M_PI);
				cairo_fill(cr);
			}
		}
			
		if (getMainWindow()->m_selected_chord_or_rest == this || getMainWindow()->doPaintColored()) {
			cairo_set_source_rgb (cr, 0.0, 0.0, 0.0);
		}
		break;
	case TYPE_NOTE:
		NedNote *start_note, *stem_start_note;
	
	
		if (m_beam != NULL) {
			if (m_beam->isBeamUp()) {
				dir = -1;
				start_note = (NedNote *) g_list_first(m_notes)->data;
				stem_start_note = (NedNote *) g_list_last(m_notes)->data;
			}
			else {
				dir = 1;
				if (m_all_shifted) {
					real_xpos -= HEAD_THICK;
				}
				stem_start_note = (NedNote *) g_list_first(m_notes)->data;
				start_note = (NedNote *) g_list_last(m_notes)->data;
			}
		}
		else {
			dir = (m_status & STAT_UP) ? -1 : 1;
			flag_glyph = (m_status & STAT_UP) ? BASE + 5 : BASE + 8;
			if (m_status & STAT_UP) {
				start_note = (NedNote *) g_list_first(m_notes)->data;
				stem_start_note = (NedNote *) g_list_last(m_notes)->data;
			}
			else {
				if (m_all_shifted) {
					real_xpos -= HEAD_THICK;
				}
				stem_start_note = (NedNote *) g_list_first(m_notes)->data;
				start_note = (NedNote *) g_list_last(m_notes)->data;
			}
		}
		y_start = start_note->getYpos();
		y_stem_start = stem_start_note->getYpos();
		if (m_beam == NULL) {
		  switch(m_length) {
		  case NOTE_2:
		  case NOTE_4:
			cairo_set_line_width(cr, zoom_factor * STEM_THICK);
			cairo_new_path(cr);
			cairo_move_to(cr, X_POS_PAGE_REL(real_xpos), Y_POS_STAFF_REL(y_start));
			cairo_line_to(cr, X_POS_PAGE_REL(real_xpos), Y_POS_STAFF_REL(y_stem_start + dir * STEM_HEIGHT));
			cairo_stroke(cr);
			break;
		  case NOTE_8:
			glyph.index = flag_glyph;
			glyph.x = X_POS_PAGE_REL(real_xpos);
			glyph.y = Y_POS_STAFF_REL(y_stem_start + dir * STEM_HEIGHT);
			cairo_show_glyphs(cr, &glyph, 1);
			cairo_set_line_width(cr, zoom_factor * STEM_THICK);
			cairo_new_path(cr);
			cairo_move_to(cr, X_POS_PAGE_REL(real_xpos), Y_POS_STAFF_REL(y_start));
			cairo_line_to(cr, X_POS_PAGE_REL(real_xpos), Y_POS_STAFF_REL(y_stem_start + dir * STEM_HEIGHT));
			cairo_stroke(cr);
			break;
		  case NOTE_16:
			glyph.index = flag_glyph;
			glyph.x = X_POS_PAGE_REL(real_xpos);
			glyph.y = Y_POS_STAFF_REL(y_stem_start + dir * STEM_HEIGHT);
			cairo_show_glyphs(cr, &glyph, 1);
			glyph.y = Y_POS_STAFF_REL(y_stem_start + dir * (STEM_HEIGHT - FLAG_DIST));
			cairo_show_glyphs(cr, &glyph, 1);
			cairo_set_line_width(cr, zoom_factor * STEM_THICK);
			cairo_new_path(cr);
			cairo_move_to(cr, X_POS_PAGE_REL(real_xpos), Y_POS_STAFF_REL(y_start));
			cairo_line_to(cr, X_POS_PAGE_REL(real_xpos), Y_POS_STAFF_REL(y_stem_start + dir * STEM_HEIGHT));
			cairo_stroke(cr);
			break;
		  case NOTE_32:
			glyph.index = flag_glyph;
			glyph.x = X_POS_PAGE_REL(real_xpos);
			glyph.y = Y_POS_STAFF_REL(y_stem_start + dir * (STEM_HEIGHT + FLAG_DIST));
			cairo_show_glyphs(cr, &glyph, 1);
			glyph.y = Y_POS_STAFF_REL(y_stem_start + dir * STEM_HEIGHT);
			cairo_show_glyphs(cr, &glyph, 1);
			glyph.y = Y_POS_STAFF_REL(y_stem_start + dir * (STEM_HEIGHT - FLAG_DIST));
			cairo_show_glyphs(cr, &glyph, 1);
			cairo_new_path(cr);
			cairo_move_to(cr, X_POS_PAGE_REL(real_xpos), Y_POS_STAFF_REL(y_start));
			cairo_line_to(cr, X_POS_PAGE_REL(real_xpos), Y_POS_STAFF_REL(y_stem_start + dir * (STEM_HEIGHT + FLAG_DIST)));
			cairo_stroke(cr);
			break;
		  case NOTE_64:
			glyph.index = flag_glyph;
			glyph.x = X_POS_PAGE_REL(real_xpos);
			glyph.y = Y_POS_STAFF_REL(y_stem_start + dir * (STEM_HEIGHT + 2 * FLAG_DIST));
			cairo_show_glyphs(cr, &glyph, 1);
			glyph.x = X_POS_PAGE_REL(real_xpos);
			glyph.y = Y_POS_STAFF_REL(y_stem_start + dir * (STEM_HEIGHT + FLAG_DIST));
			cairo_show_glyphs(cr, &glyph, 1);
			glyph.y = Y_POS_STAFF_REL(y_stem_start + dir * STEM_HEIGHT);
			cairo_show_glyphs(cr, &glyph, 1);
			glyph.y = Y_POS_STAFF_REL(y_stem_start + dir * (STEM_HEIGHT - FLAG_DIST));
			cairo_show_glyphs(cr, &glyph, 1);
			cairo_set_line_width(cr, zoom_factor * STEM_THICK);
			cairo_new_path(cr);
			cairo_move_to(cr, X_POS_PAGE_REL(real_xpos), Y_POS_STAFF_REL(y_start));
			cairo_line_to(cr, X_POS_PAGE_REL(real_xpos), Y_POS_STAFF_REL(y_stem_start + dir * (STEM_HEIGHT + 2 * FLAG_DIST)));
			cairo_stroke(cr);
			break;
		  }
		}
		else {
			cairo_set_line_width(cr, zoom_factor * STEM_THICK);
			cairo_new_path(cr);
			cairo_move_to(cr, X_POS_PAGE_REL(real_xpos), Y_POS_STAFF_REL(y_start));
			cairo_line_to(cr, X_POS_PAGE_REL(real_xpos), Y_POS_STAFF_REL(m_beam->getYPosAt(real_xpos)));
			cairo_stroke(cr);
		}
#define STACC_DOT_X_DIST 0.13
		if (m_status & STAT_STACC) {
			cairo_new_path(cr);
			cairo_arc(cr, X_POS_PAGE_REL(real_xpos + dir * STACC_DOT_X_DIST), Y_POS_STAFF_REL(m_art_y_pos[art_nr++]),
					zoom_factor * DOT_RAD, 0, 2.0 * M_PI);
			cairo_fill(cr);
		}
#define TENUTO_X_DIST 0.13
#define TENUTO_LONG 0.2
#define TENUTO_LONG_2 (TENUTO_LONG / 2.0)
#define TENUTO_THICK 0.04
		if (m_status & STAT_TENUTO) {
			cairo_new_path(cr);
			cairo_set_line_width(cr, zoom_factor * TENUTO_THICK);
			cairo_move_to(cr, X_POS_PAGE_REL(real_xpos + dir * TENUTO_X_DIST - TENUTO_LONG_2), Y_POS_STAFF_REL(m_art_y_pos[art_nr]));
			cairo_line_to(cr, X_POS_PAGE_REL(real_xpos + dir * TENUTO_X_DIST + TENUTO_LONG_2), Y_POS_STAFF_REL(m_art_y_pos[art_nr++]));
			cairo_stroke(cr);
		}
		if (m_status & STAT_SFORZANDO) {
#define SFORZANDO_X_DIST 0.13
#define SFORZANDO_LONG 0.2
#define SFORZANDO_LONG_2 (SFORZANDO_LONG / 2.0)
#define SFORZANDO_HIGHT 0.15
#define SFORZANDO_HIGHT_2 (SFORZANDO_HIGHT / 2.0)
#define SFORZANDO_THICK 0.04
			cairo_new_path(cr);
			cairo_set_line_width(cr, zoom_factor * SFORZANDO_THICK);
			cairo_move_to(cr, X_POS_PAGE_REL(real_xpos + dir * SFORZANDO_X_DIST - SFORZANDO_LONG_2), Y_POS_STAFF_REL(m_art_y_pos[art_nr]));
			cairo_line_to(cr, X_POS_PAGE_REL(real_xpos + dir * SFORZANDO_X_DIST + SFORZANDO_LONG_2), Y_POS_STAFF_REL(m_art_y_pos[art_nr] - dir * SFORZANDO_HIGHT_2));
			cairo_line_to(cr, X_POS_PAGE_REL(real_xpos + dir * SFORZANDO_X_DIST - SFORZANDO_LONG_2), Y_POS_STAFF_REL(m_art_y_pos[art_nr++] - dir * SFORZANDO_HIGHT));
			cairo_stroke(cr);
		}
		if (m_status & STAT_SFORZATO) {
#define SFORZATO_X_DIST 0.13
#define SFORZATO_LONG 0.15
#define SFORZATO_LONG_2 (SFORZATO_LONG / 2.0)
#define SFORZATO_HIGHT 0.2
#define SFORZATO_THICK1 0.08
#define SFORZATO_THICK2 0.04
			cairo_new_path(cr);
			if (dir > 0) {
				cairo_set_line_width(cr, zoom_factor * SFORZATO_THICK1);
				cairo_move_to(cr, X_POS_PAGE_REL(real_xpos + dir * SFORZATO_X_DIST - SFORZATO_LONG_2), Y_POS_STAFF_REL(m_art_y_pos[art_nr]));
				cairo_line_to(cr, X_POS_PAGE_REL(real_xpos + dir * SFORZATO_X_DIST), Y_POS_STAFF_REL(m_art_y_pos[art_nr] - SFORZATO_HIGHT));
				cairo_stroke(cr);
				cairo_new_path(cr);
				cairo_set_line_width(cr, zoom_factor * SFORZATO_THICK2);
				cairo_move_to(cr, X_POS_PAGE_REL(real_xpos + dir * SFORZATO_X_DIST), Y_POS_STAFF_REL(m_art_y_pos[art_nr] - SFORZATO_HIGHT));
				cairo_line_to(cr, X_POS_PAGE_REL(real_xpos + dir * SFORZATO_X_DIST + SFORZATO_LONG_2), Y_POS_STAFF_REL(m_art_y_pos[art_nr++]));
				cairo_stroke(cr);
			}
			else {
				cairo_set_line_width(cr, zoom_factor * SFORZATO_THICK2);
				cairo_move_to(cr, X_POS_PAGE_REL(real_xpos + dir * SFORZATO_X_DIST - SFORZATO_LONG_2), Y_POS_STAFF_REL(m_art_y_pos[art_nr]));
				cairo_line_to(cr, X_POS_PAGE_REL(real_xpos + dir * SFORZATO_X_DIST), Y_POS_STAFF_REL(m_art_y_pos[art_nr] + SFORZATO_HIGHT));
				cairo_stroke(cr);
				cairo_new_path(cr);
				cairo_set_line_width(cr, zoom_factor * SFORZATO_THICK1);
				cairo_move_to(cr, X_POS_PAGE_REL(real_xpos + dir * SFORZATO_X_DIST), Y_POS_STAFF_REL(m_art_y_pos[art_nr] + SFORZATO_HIGHT));
				cairo_line_to(cr, X_POS_PAGE_REL(real_xpos + dir * SFORZATO_X_DIST + SFORZATO_LONG_2), Y_POS_STAFF_REL(m_art_y_pos[art_nr++]));
				cairo_stroke(cr);
			}
		}
		if (m_status & STAT_BOW_UP) {
#define BOW_UP_X_DIST 0.13
#define BOW_UP_LONG 0.2
#define BOW_UP_HIGHT 0.3
#define BOW_UP_LONG_2 (BOW_UP_LONG / 2.0)
#define BOW_UP_THICK 0.04
			cairo_new_path(cr);
			cairo_set_line_width(cr, zoom_factor * BOW_UP_THICK);
			cairo_move_to(cr, X_POS_PAGE_REL(real_xpos + dir * BOW_UP_X_DIST - BOW_UP_LONG_2), Y_POS_STAFF_REL(m_art_up_y_pos[up_art_nr] - BOW_UP_HIGHT));
			cairo_line_to(cr, X_POS_PAGE_REL(real_xpos + dir * BOW_UP_X_DIST), Y_POS_STAFF_REL(m_art_up_y_pos[up_art_nr]));
			cairo_line_to(cr, X_POS_PAGE_REL(real_xpos + dir * BOW_UP_X_DIST + BOW_UP_LONG_2), Y_POS_STAFF_REL(m_art_up_y_pos[up_art_nr++] - BOW_UP_HIGHT));
			cairo_stroke(cr);
		}
		if (m_status & STAT_BOW_DOWN) {
#define BOW_DOWN_X_DIST 0.13
#define BOW_DOWN_LONG 0.25
#define BOW_DOWN_HIGHT 0.2
#define BOW_DOWN_LONG_2 (BOW_DOWN_LONG / 2.0)
#define BOW_DOWN_THICK1 0.04
#define BOW_DOWN_THICK2 0.08
			cairo_new_path(cr);
			cairo_set_line_width(cr, zoom_factor * BOW_DOWN_THICK1);
			cairo_move_to(cr, X_POS_PAGE_REL(real_xpos + dir * BOW_DOWN_X_DIST - BOW_DOWN_LONG_2), Y_POS_STAFF_REL(m_art_up_y_pos[up_art_nr]));
			cairo_line_to(cr, X_POS_PAGE_REL(real_xpos + dir * BOW_DOWN_X_DIST - BOW_DOWN_LONG_2), Y_POS_STAFF_REL(m_art_up_y_pos[up_art_nr] - BOW_DOWN_HIGHT));
			cairo_move_to(cr, X_POS_PAGE_REL(real_xpos + dir * BOW_DOWN_X_DIST + BOW_DOWN_LONG_2), Y_POS_STAFF_REL(m_art_up_y_pos[up_art_nr]));
			cairo_line_to(cr, X_POS_PAGE_REL(real_xpos + dir * BOW_DOWN_X_DIST + BOW_DOWN_LONG_2), Y_POS_STAFF_REL(m_art_up_y_pos[up_art_nr] - BOW_DOWN_HIGHT));
			cairo_stroke(cr);
			cairo_new_path(cr);
			cairo_set_line_width(cr, zoom_factor * BOW_DOWN_THICK2);
			cairo_line_to(cr, X_POS_PAGE_REL(real_xpos + dir * BOW_DOWN_X_DIST - BOW_DOWN_LONG_2), Y_POS_STAFF_REL(m_art_up_y_pos[up_art_nr] - BOW_DOWN_HIGHT));
			cairo_line_to(cr, X_POS_PAGE_REL(real_xpos + dir * BOW_DOWN_X_DIST + BOW_DOWN_LONG_2), Y_POS_STAFF_REL(m_art_up_y_pos[up_art_nr++] - BOW_DOWN_HIGHT));
			cairo_stroke(cr);
		}
		GList *lptr;
		for (lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
			((NedNote *) lptr->data)->draw(cr);
		}
		for (i = 0; !(*lyrics_present) && i  < MAX_LYRICS_LINES; i++) {
			if (m_lyrics[i] != NULL) *lyrics_present = true;
		}
		break;
	case TYPE_GRACE:
		if (getMainWindow()->m_selected_chord_or_rest == this) {
			if (getMainWindow()->doPaintColored()) {
				switch(getVoice()->getVoiceNumber()) {
					case 0: cairo_set_source_rgb (cr, DSCOLOR(V1RED), DSCOLOR(V1GREEN), DSCOLOR(V1BLUE)); break;
					case 1: cairo_set_source_rgb (cr, DSCOLOR(V2RED), DSCOLOR(V2GREEN), DSCOLOR(V2BLUE)); break;
					case 2: cairo_set_source_rgb (cr, DSCOLOR(V3RED), DSCOLOR(V3GREEN), DSCOLOR(V3BLUE)); break;
					default: cairo_set_source_rgb (cr, DSCOLOR(V4RED), DSCOLOR(V4GREEN), DSCOLOR(V4BLUE)); break;
				}
			}
			else {
				cairo_set_source_rgb (cr, 1.0, 0.0, 0.0);
			}
		}
		else if (getMainWindow()->doPaintColored()) {
			switch(getVoice()->getVoiceNumber()) {
				case 0: cairo_set_source_rgb (cr, DCOLOR(V1RED), DCOLOR(V1GREEN), DCOLOR(V1BLUE)); break;
				case 1: cairo_set_source_rgb (cr, DCOLOR(V2RED), DCOLOR(V2GREEN), DCOLOR(V2BLUE)); break;
				case 2: cairo_set_source_rgb (cr, DCOLOR(V3RED), DCOLOR(V3GREEN), DCOLOR(V3BLUE)); break;
				default: cairo_set_source_rgb (cr, DCOLOR(V4RED), DCOLOR(V4GREEN), DCOLOR(V4BLUE)); break;
			}
		}
		stem_start_note = (NedNote *) g_list_first(m_notes)->data;
		y_stem_start = stem_start_note->getYpos();
		switch(m_length) {
			case STROKEN_GRACE: glyph.index =  BASE + 18; break;
			case GRACE_8:
			case GRACE_16: glyph.index =  BASE + 19; break;
		}
		glyph.x = X_POS_PAGE_REL(real_xpos);
		glyph.y = Y_POS_STAFF_REL(y_stem_start);
		cairo_show_glyphs(cr, &glyph, 1);
		if (m_beam == NULL) {
			switch(m_length) {
				case GRACE_16: glyph.index =  BASE + 20;
					glyph.y = Y_POS_STAFF_REL(y_stem_start + GRACE_HEIGHT2);
					cairo_show_glyphs(cr, &glyph, 1);
				case GRACE_8: glyph.index =  BASE + 20;
					glyph.y = Y_POS_STAFF_REL(y_stem_start + GRACE_HEIGHT1);
					cairo_show_glyphs(cr, &glyph, 1);
					cairo_new_path(cr);
					cairo_set_line_width(cr, zoom_factor * GRACE_STEM_THICK);
					cairo_move_to(cr, X_POS_PAGE_REL(real_xpos + GRACE_HEAD_THICK_HALF), Y_POS_STAFF_REL(y_stem_start));
					cairo_line_to(cr, X_POS_PAGE_REL(real_xpos + GRACE_HEAD_THICK_HALF), Y_POS_STAFF_REL(y_stem_start - GRACE_HEIGHT3));
					cairo_stroke(cr);
					break;
			}
		}
		else {
			cairo_set_line_width(cr, zoom_factor * GRACE_STEM_THICK);
			cairo_new_path(cr);
			cairo_move_to(cr, X_POS_PAGE_REL(real_xpos + GRACE_HEAD_THICK_HALF), Y_POS_STAFF_REL(y_stem_start));
			cairo_line_to(cr, X_POS_PAGE_REL(real_xpos + GRACE_HEAD_THICK_HALF), Y_POS_STAFF_REL(m_beam->getYPosAt(real_xpos + GRACE_HEAD_THICK_HALF)));
			cairo_stroke(cr);
		}
		if (getMainWindow()->m_selected_chord_or_rest == this || getMainWindow()->doPaintColored()) {
			cairo_set_source_rgb (cr, 0.0, 0.0, 0.0);
		}
		noteline = ((NedNote *) g_list_first(m_notes)->data)->getLine();
		if (noteline < -1) {
			cairo_new_path(cr);
			cairo_set_line_width(cr, zoom_factor * AUX_LINE_THICK);
			line = noteline / 2;
			line *= 2;
			for (; line < -1; line += 2) {
				yl = - line * LINE_DIST/2.0;
				cairo_move_to(cr, X_POS_PAGE_REL(m_xpos - HEAD_THICK_2 / 2.0 - AUX_LINE_OVER),
					Y_POS_STAFF_REL(yl));
				cairo_line_to(cr, X_POS_PAGE_REL(m_xpos + HEAD_THICK_2 / 2.0 + AUX_LINE_OVER),
					Y_POS_STAFF_REL(yl));
			}
			cairo_stroke(cr);
		}
		if (noteline > 9) {
			cairo_new_path(cr);
			cairo_set_line_width(cr, zoom_factor * AUX_LINE_THICK);
			line = noteline / 2;
			line *= 2;
			for (; line > 9; line -= 2) {
				yl = - line * LINE_DIST/2.0;
				cairo_move_to(cr, X_POS_PAGE_REL(m_xpos - HEAD_THICK_2 / 2.0 - AUX_LINE_OVER),
					Y_POS_STAFF_REL(yl));
				cairo_line_to(cr, X_POS_PAGE_REL(m_xpos + HEAD_THICK_2 / 2.0 + AUX_LINE_OVER),
					Y_POS_STAFF_REL(yl));
			}
			cairo_stroke(cr);
		}
		break;
	}
//#define SHOW_BBOX
#ifdef SHOW_BBOX
	cairo_set_source_rgb (cr, 1.0, 0.0, 1.0);
	cairo_new_path(cr);
	cairo_set_line_width(cr, zoom_factor * STEM_THICK);
	cairo_rectangle (cr, X_POS_PAGE_REL(m_xpos + m_bbox.x), Y_POS_STAFF_REL(m_bbox.y), zoom_factor * (m_bbox.width), zoom_factor * m_bbox.height);
	cairo_stroke(cr);
	cairo_set_source_rgb (cr, 0.0, 0.0, 0.0);
#endif
}

void NedChordOrRest::drawlyrics(cairo_t *cr) {
	int line, i;
	double leftx = getMainWindow()->getLeftX();
	double topy = getMainWindow()->getTopY();
	double zoom_factor = getMainWindow()->getCurrentZoomFactor();
	char str[4 * MAX_TEXT_LENGTH];
	double real_xpos = m_tmp_xpos < 0.0 ? m_xpos : m_tmp_xpos;
	if (m_type != TYPE_NOTE) return;
	cairo_text_extents_t extends;
	bool lyrics_present = false;
	for (i = 0; i < MAX_LYRICS_LINES; i++) {
		if (m_lyrics[i] != NULL) {
			lyrics_present = true;
			cairo_move_to(cr, X_POS_PAGE_REL(real_xpos - m_bbox.width / 2.0),
				Y_POS_STAFF_REL(getStaff()->getBottomYBorder() + TEXT_Y_OFFS + i * LYRICS_HEIGHT));
			cairo_show_text(cr, m_lyrics[i]);
		}
	}
	if (!getMainWindow()->doDrawPostscript() && lyrics_present && ((line = getMainWindow()->getLyricsMode()) != -1) && getMainWindow()->m_selected_chord_or_rest == this) {
		if (m_lyrics[line] != NULL) {
			g_utf8_strncpy(str, m_lyrics[line], m_cursorpos);
			cairo_text_extents(cr, str, &extends);
			cairo_new_path(cr);
			cairo_move_to(cr, X_POS_PAGE_REL(real_xpos - m_bbox.width / 2.0) + extends.width, Y_POS_STAFF_REL(getStaff()->getBottomYBorder()
				+ TEXT_Y_OFFS + (line - 1) * LYRICS_HEIGHT));
			cairo_line_to(cr, X_POS_PAGE_REL(real_xpos - m_bbox.width / 2.0) + extends.width, Y_POS_STAFF_REL(getStaff()->getBottomYBorder()
				+ TEXT_Y_OFFS + line * LYRICS_HEIGHT));
			cairo_stroke(cr);
		}
	}
}

void NedChordOrRest::xPositNotes() {
	int line_num, last_line_num;
	bool shifted = false;
	GList *lptr;
	int dir;
	double y_stem_start, art_offs = 0.0, up_art_offs = 0.0;
	int stac_line;
	int acc_places[4];
	int i = 0;
	m_art_count = 0; m_up_art_count = 0;
	NedNote *start_note, *stem_start_note;
	if (m_beam) {
		dir = m_beam->isBeamUp() ? -1 : 1;
	}
	else {
		dir = ((m_status & STAT_UP) != 0) ? -1 : 1;
	}
	if (dir < 0) {
		last_line_num = -100;
		for (lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr), i++) {
			line_num = ((NedNote *) lptr->data)->getLine();
			if (!shifted && line_num - last_line_num < 2) {
				((NedNote *) lptr->data)->shiftNote(true);
				shifted = true;
			}
			else {
				((NedNote *) lptr->data)->shiftNote(false);
				m_all_shifted = false;
				shifted = false;
			}
			last_line_num = line_num;
		}
		start_note = (NedNote *) g_list_first(m_notes)->data;
		stem_start_note = (NedNote *) g_list_last(m_notes)->data;
	}
	else {
		m_all_shifted = true;
		last_line_num = 100;
		i =  g_list_length(m_notes) - 1;
		for (lptr = g_list_last(m_notes); lptr; lptr = g_list_previous(lptr), i--) {
			line_num = ((NedNote *) lptr->data)->getLine();
			if (!shifted && last_line_num - line_num < 2) {
				((NedNote *) lptr->data)->shiftNote(false);
				shifted = true;
				m_all_shifted = false;
			}
			else {
				((NedNote *) lptr->data)->shiftNote(true);
				shifted = false;
			}
			last_line_num = line_num;
		}
		if (m_all_shifted) {
			for (lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
				((NedNote *) lptr->data)->shiftNote(false);
			}
		}
		stem_start_note = (NedNote *) g_list_first(m_notes)->data;
		start_note = (NedNote *) g_list_last(m_notes)->data;
			
	}
	acc_places[0] = acc_places[1] = acc_places[2] = acc_places[3] = -10000;
	for (lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
		((NedNote *) lptr->data)->placeAccidental(FALSE /*all_shifted*/, acc_places);
	}
	y_stem_start = stem_start_note->getYpos();
	if (m_status & STAT_STACC) {
		stac_line = start_note->getLine() + 2 * dir;
		switch (stac_line) {
			case 0: case 2: case 4: case 6: case 8: stac_line += dir; break;
		}
		m_art_y_pos[m_art_count++] = -stac_line * LINE_DIST/2.0;
		art_offs += LINE_DIST;
	}
	if (m_status & STAT_TENUTO) {
		stac_line = start_note->getLine() + 2 * dir;
		switch (stac_line) {
			case 0: case 2: case 4: case 6: case 8: stac_line += dir; break;
		}
		m_art_y_pos[m_art_count++] = -stac_line * LINE_DIST/2.0 - dir * art_offs;
		art_offs += LINE_DIST;
	}
	if (m_status & STAT_SFORZANDO) {
		stac_line = start_note->getLine() + 2 * dir;
		switch (stac_line) {
			case 0: case 2: case 4: case 6: case 8: stac_line += dir; break;
		}
		m_art_y_pos[m_art_count++] = -stac_line * LINE_DIST/2.0 - dir * art_offs;
		art_offs += LINE_DIST;
	}
	if (m_status & STAT_SFORZATO) {
		stac_line = start_note->getLine() + 2 * dir;
		switch (stac_line) {
			case 0: case 2: case 4: case 6: case 8: stac_line += dir; break;
		}
		m_art_y_pos[m_art_count++] = -stac_line * LINE_DIST/2.0 - dir * art_offs;
		art_offs += LINE_DIST;
	}
	if (m_status & STAT_BOW_UP) {
		if (dir < 0) {
			m_art_up_y_pos[m_up_art_count] = y_stem_start - STEM_HEIGHT - LINE_DIST - up_art_offs;
		}
		else {
			up_art_offs = art_offs;
			stac_line = start_note->getLine() + 2 * dir;
			m_art_up_y_pos[m_up_art_count] = -stac_line * LINE_DIST/2.0 - up_art_offs;
		}
		if (m_art_up_y_pos[m_up_art_count] > -5 * LINE_DIST) m_art_up_y_pos[m_up_art_count] =  -5 * LINE_DIST - up_art_offs;
		up_art_offs += 2*LINE_DIST;
		m_up_art_count++;
	}
	if (m_status & STAT_BOW_DOWN) {
		if (dir < 0) {
			m_art_up_y_pos[m_up_art_count] = y_stem_start - STEM_HEIGHT - LINE_DIST - up_art_offs;
		}
		else {
			if (up_art_offs == 0.0) {
				up_art_offs = art_offs;
			}
			stac_line = start_note->getLine() + 2 * dir;
			m_art_up_y_pos[m_up_art_count] = -stac_line * LINE_DIST/2.0 - up_art_offs;
		}
		if (m_art_up_y_pos[m_up_art_count] > -5 * LINE_DIST) m_art_up_y_pos[m_up_art_count] =  -5 * LINE_DIST - up_art_offs;
		up_art_offs += LINE_DIST;
		m_up_art_count++;
	}
		
}

double NedChordOrRest::getBeamXPos() {
	if (m_type == TYPE_GRACE) return m_xpos + GRACE_HEAD_THICK_HALF;
	if (m_all_shifted) return m_xpos - HEAD_THICK;
	return m_xpos;
}


double NedChordOrRest::getStemTop() {
	NedNote *stem_start_note;
	stem_start_note = (NedNote *) g_list_last(m_notes)->data;
	double y_stem_start = stem_start_note->getYpos();
	if (m_type == TYPE_GRACE) {
		return y_stem_start - GRACE_HEIGHT3;
	}
	switch(m_length) {
		case NOTE_32:
			return y_stem_start - STEM_HEIGHT - FLAG_DIST;
		case NOTE_64:
			return y_stem_start - STEM_HEIGHT - 2 * FLAG_DIST;
	}
	return y_stem_start - STEM_HEIGHT;
}

double NedChordOrRest::getTopOfChordOrRest() {
	if (m_type == TYPE_REST) {
		return -3 * LINE_DIST / 2.0;
	}
	NedNote *stem_start_note;
	stem_start_note = (NedNote *) g_list_last(m_notes)->data;
	double y_stem_start = stem_start_note->getYpos();
	if (!(m_status & STAT_UP)) {
		return y_stem_start - 0.1;
	}
	if (m_beam != NULL) {
		return m_beam->getTopOfBeam();
	}
	switch(m_length) {
		case NOTE_32:
			return y_stem_start - STEM_HEIGHT - FLAG_DIST;
		case NOTE_64:
			return y_stem_start - STEM_HEIGHT - 2 * FLAG_DIST;
	}
	return y_stem_start - STEM_HEIGHT;
}

double NedChordOrRest::getBottomOfChordOrRest() {
	if (m_type == TYPE_REST) {
		return -3 * LINE_DIST / 2.0;
	}
	NedNote *stem_start_note;
	stem_start_note = (NedNote *) g_list_first(m_notes)->data;
	double y_stem_start = stem_start_note->getYpos();
	if (m_status & STAT_UP) {
		return y_stem_start + 0.1;
	}
	if (m_beam != NULL) {
		return m_beam->getBottomOfBeam();
	}
	switch(m_length) {
		case NOTE_32:
			return y_stem_start + STEM_HEIGHT + FLAG_DIST;
		case NOTE_64:
			return y_stem_start + STEM_HEIGHT + 2 * FLAG_DIST;
	}
	return y_stem_start + STEM_HEIGHT;
}

double NedChordOrRest::getStemBottom() {
	NedNote *stem_start_note;
	stem_start_note = (NedNote *) g_list_first(m_notes)->data;
	double y_stem_start = stem_start_note->getYpos();
	switch(m_length) {
		case NOTE_32:
			return y_stem_start + STEM_HEIGHT + FLAG_DIST;
		case NOTE_64:
			return y_stem_start + STEM_HEIGHT + 2 * FLAG_DIST;
	}
	return y_stem_start + STEM_HEIGHT;
}

/*
double getTopPos() {
	if (m_status & STAT_UP) {
		return getStemTop();
	}
	return getStemBottom();
}
*/

double NedChordOrRest::getStemYStart() {
	NedNote *start_note;
	if (m_status & STAT_UP) {
		start_note = (NedNote *) g_list_first(m_notes)->data;
	}
	else {
		start_note = (NedNote *) g_list_last(m_notes)->data;
	}
	return start_note->getYpos();
}

void NedChordOrRest::getTopBotY(int *lyrics_lines, double *topy, double *boty) {
	int dir;
	NedNote *start_note, *stem_start_note;
	double endposoffs;
	double y_start;
	int i;

	if (m_type == TYPE_REST) {
		if (m_is_hidden) {
			*topy = *boty = 3 * LINE_DIST / 2.0;
		}
		else {
			*topy = m_bbox.y + 4 * LINE_DIST;
			*boty = m_bbox.y + m_bbox.height + 4 * LINE_DIST;
		}
		return;
	}
	for (i = 0; i < MAX_LYRICS_LINES; i++) {
		if (m_lyrics[i] != NULL && *lyrics_lines < (i + 1)) {
			*lyrics_lines = (i + 1);
		}
	}
	if (m_beam != NULL) {
		if (m_beam->isBeamUp()) {
			dir = -1;
		}
		else {
			dir = 1;
		}
	}
	else {
		dir = (m_status & STAT_UP) ? -1 : 1;
	}
	if (dir == -1) {
		start_note = (NedNote *) g_list_first(m_notes)->data;
		stem_start_note = (NedNote *) g_list_last(m_notes)->data;
		y_start = start_note->getYpos() + HEAD_THICK_HALF;
	}
	else {
		stem_start_note = (NedNote *) g_list_first(m_notes)->data;
		start_note = (NedNote *) g_list_last(m_notes)->data;
		y_start = start_note->getYpos() - HEAD_THICK_HALF;
	}
	if (m_length > NOTE_2) {
		*boty = 4 * LINE_DIST + y_start;
		*topy = 4 * LINE_DIST + stem_start_note->getYpos();
	}
	else if (m_beam != NULL) {
		if (dir == -1) {
			*boty = 4 * LINE_DIST + y_start;
			*topy = 4 * LINE_DIST + m_beam->getYPosAt(m_xpos);
		}
		else {
			*topy = 4 * LINE_DIST + y_start;
			*boty = 4 * LINE_DIST + m_beam->getYPosAt(m_xpos);
		}
	}
	else {
		switch(m_length) {
		  	case NOTE_2:
		  	case NOTE_4:
		  	case NOTE_8:
		  	case NOTE_16:
				endposoffs = STEM_HEIGHT;
				break;
		  	case NOTE_32:
				endposoffs = STEM_HEIGHT + FLAG_DIST;
				break;
		  	case NOTE_64:
				endposoffs = STEM_HEIGHT + 2 * FLAG_DIST;
				break;
		}
	}
	double y_stem_start = stem_start_note->getYpos();
	if (dir == -1) {
		*topy = 4 * LINE_DIST + y_stem_start - endposoffs;
		*boty = 4 * LINE_DIST + y_start;
		if (m_up_art_count > 0 && m_art_up_y_pos[m_up_art_count-1] + 3 * LINE_DIST < *topy) *topy  = m_art_up_y_pos[m_up_art_count-1] + 3 * LINE_DIST;
		if (m_art_count > 0 &&  m_art_y_pos[m_art_count-1] + 5 * LINE_DIST > *boty) *boty = m_art_y_pos[m_art_count-1] + 5 * LINE_DIST ;
	}
	else {
		*boty = 4 * LINE_DIST + (y_stem_start + endposoffs);
		*topy = 4 * LINE_DIST + y_start;
		if (m_up_art_count > 0 && m_art_up_y_pos[m_up_art_count-1] + 3 * LINE_DIST < *topy)
			 *topy = m_art_up_y_pos[m_up_art_count-1] + 3 * LINE_DIST;
		if (m_art_count > 0 && m_art_y_pos[m_art_count-1] + 3 * LINE_DIST < *topy)
			 *topy = m_art_y_pos[m_art_count-1] + 3 * LINE_DIST;
	}
}

bool NedChordOrRest::isUp() {
	if (m_beam != 0) {
		return m_beam->isBeamUp();
	}
	return (m_status & STAT_UP);
}


double NedChordOrRest::computeStemDist(bool thisup, NedChordOrRest *other_chord, bool otherup) {
	double d, dist = 0.0;
	int dir;
	double thisypos, otherypos;
	NedNote *this_stem_start_note, *other_stem_start_note;

	if (m_type == TYPE_REST || other_chord->m_type == TYPE_REST) return 0.0;

	dir = thisup ? 1 : -1;
	if (dir > 0) {
		this_stem_start_note = (NedNote *) g_list_first(m_notes)->data;
	}
	else {
		this_stem_start_note = (NedNote *) g_list_last(m_notes)->data;
	}
	thisypos = this_stem_start_note->getYpos() + dir * STEM_HEIGHT;

	dir = otherup ? 1 : -1;
	if (dir > 0) {
		other_stem_start_note = (NedNote *) g_list_first(other_chord->m_notes)->data;
	}
	else {
		other_stem_start_note = (NedNote *) g_list_last(other_chord->m_notes)->data;
	}
	otherypos = other_stem_start_note->getYpos() + dir * STEM_HEIGHT;
	d = otherypos - this_stem_start_note->getYpos();
	if (d < 0.0) d = -d;
	dist += d;
	d = thisypos - other_stem_start_note->getYpos();
	if (d < 0.0) d = -d;
	dist += d;
	return dist;
}

int NedChordOrRest::compareMidiTimes(NedChordOrRest *c1, NedChordOrRest *c2) {
	if (c1->m_midi_time < c2->m_midi_time) return -1;
	if (c1->m_midi_time == c2->m_midi_time) return 0;
	return 1;
}

void NedChordOrRest::shiftY(double y) {
	m_ypos += y;
	computeBbox();
}

void NedChordOrRest::changeStemDir(int dir) {
	unsigned int upbeat;
	unsigned int measure_length;
	if (m_type == TYPE_REST) {
		if (m_length != WHOLE_NOTE) {
			if (getSystem()->getSystemNumber() > 0) {
				m_is_hidden = FALSE;
				return;
			}
			if (getMainWindow()->getUpBeatInverse() == 0) {
				m_is_hidden = FALSE;
				return;
			}
			if (m_midi_time  != getMainWindow()->getUpBeatInverse()) {
				m_is_hidden = FALSE;
				return;
			}
			measure_length = getMainWindow()->getNumerator() * WHOLE_NOTE / getMainWindow()->getDenominator();
			upbeat = measure_length - getMainWindow()->getUpBeatInverse();
			if (getDuration() != upbeat) {
				m_is_hidden = FALSE;
			}
		}
		return;
	}
	bool changed = FALSE;
	switch(dir) {
		case STEM_DIR_UP: if (m_status & STAT_UP) break; changed = TRUE; m_status |= STAT_UP; break;
		case STEM_DIR_DOWN: if (!(m_status & STAT_UP)) break; changed = TRUE; m_status &= (~(STAT_UP)); break;
		default:  reConfigure(); changed = TRUE; break;
	}
	if (changed) {
		xPositNotes();
		computeBbox();
	}
}



bool NedChordOrRest::trySelect(double x, double y) {
	int zoom_level;
	double leftx, topy;
	double zoom_factor;
	double xl, yl;
	GList *lptr;
	cairo_text_extents_t *extention;
	switch (m_type) {
	case TYPE_CLEF:
		zoom_level = getMainWindow()->getCurrentZoomLevel();
		leftx = getMainWindow()->getLeftX();
		topy = getMainWindow()->getTopY();
		zoom_factor = getMainWindow()->getCurrentZoomFactor();
		xl = X_POS_INVERS_PAGE_REL(x);
		yl = Y_POS_INVERS_ABS(y);
		switch (m_length) {
			case TREBLE_CLEF:
				extention = &(NedResource::fontextentions[zoom_level][2]); break;
			case BASS_CLEF:
				extention = &(NedResource::fontextentions[zoom_level][3]); break;
			case ALTO_CLEF:
				extention = &(NedResource::fontextentions[zoom_level][1]); break;
		}
		if (getSystem()->getYPos() + getStaff()->getBottomPos() + m_ypos + extention->y_bearing / zoom_factor < yl &&
			getSystem()->getYPos() + getStaff()->getBottomPos() + m_ypos + (extention->y_bearing  +  extention->height) / zoom_factor > yl &&
		    	m_xpos + extention->x_bearing / zoom_factor < xl &&
		    	m_xpos + (extention->x_bearing  +  extention->width) / zoom_factor > xl) {
				getMainWindow()->m_selected_chord_or_rest = this;
				return TRUE;
		}
		return FALSE;
	case TYPE_KEYSIG:
		zoom_level = getMainWindow()->getCurrentZoomLevel();
		leftx = getMainWindow()->getLeftX();
		topy = getMainWindow()->getTopY();
		zoom_factor = getMainWindow()->getCurrentZoomFactor();
		xl = X_POS_INVERS_PAGE_REL(x);
		yl = Y_POS_INVERS_ABS(y);
		if (getSystem()->getYPos() + getStaff()->getBottomPos() + m_ypos - m_bbox.height < yl && getSystem()->getYPos() + getStaff()->getBottomPos() + m_ypos > yl &&
			m_xpos < xl && m_xpos + m_bbox.width > xl) {
				getMainWindow()->m_selected_chord_or_rest = this;
				return TRUE;
		}
		return FALSE;
	case TYPE_REST:
		zoom_level = getMainWindow()->getCurrentZoomLevel();
		leftx = getMainWindow()->getLeftX();
		topy = getMainWindow()->getTopY();
		zoom_factor = getMainWindow()->getCurrentZoomFactor();
		xl = X_POS_INVERS_PAGE_REL(x);
		yl = Y_POS_INVERS_ABS(y);
		switch(m_length) {
		case WHOLE_NOTE:
			extention = &(NedResource::fontextentions[zoom_level][9]); break;
		case NOTE_2:
			extention = &(NedResource::fontextentions[zoom_level][10]); break;
		case NOTE_4:
			extention = &(NedResource::fontextentions[zoom_level][11]); break;
		case NOTE_8:
			extention = &(NedResource::fontextentions[zoom_level][12]); break;
		case NOTE_16:
			extention = &(NedResource::fontextentions[zoom_level][13]); break;
		case NOTE_32:
			extention = &(NedResource::fontextentions[zoom_level][14]); break;
		case NOTE_64:
			extention = &(NedResource::fontextentions[zoom_level][15]); break;
		}
		if (getSystem()->getYPos() + getStaff()->getBottomPos() + m_ypos + extention->y_bearing / zoom_factor < yl &&
		    getSystem()->getYPos() + getStaff()->getBottomPos() + m_ypos + (extention->y_bearing  +  extention->height) / zoom_factor > yl &&
		    m_xpos + extention->x_bearing / zoom_factor < xl &&
		    m_xpos + (extention->x_bearing  +  extention->width) / zoom_factor > xl) {
			getMainWindow()->m_selected_chord_or_rest = this;
			return TRUE;
		}
		return FALSE;
	case TYPE_GRACE:
		zoom_level = getMainWindow()->getCurrentZoomLevel();
		leftx = getMainWindow()->getLeftX();
		topy = getMainWindow()->getTopY();
		zoom_factor = getMainWindow()->getCurrentZoomFactor();
		xl = X_POS_INVERS_PAGE_REL(x);
		yl = Y_POS_INVERS_ABS(y);
		if (getSystem()->getYPos() + getStaff()->getBottomPos() + m_bbox.y < yl && getSystem()->getYPos() + getStaff()->getBottomPos() + m_bbox.y + m_bbox.height > yl &&
			m_xpos + m_bbox.x < xl && m_xpos + m_bbox.x + m_bbox.width > xl) {
				getMainWindow()->m_selected_chord_or_rest = this;
				getMainWindow()->m_selected_note = (NedNote *) g_list_first(m_notes)->data;
				return TRUE;
		}
		return FALSE;
	default:
		for (lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
			if (((NedNote *) lptr->data)->trySelect(x, y)) {
				getMainWindow()->m_selected_chord_or_rest = this;
				return TRUE;
			}
		}
		return FALSE;
	}
}

void NedChordOrRest::decrMidiTime(unsigned long long incr) {
	if (incr > m_midi_time) {
		printf("\this = 0x%x, m_midi_time = %llu, incr = %llu\n", this, m_midi_time, incr) ;fflush(stdout);
		NedResource::Abort("NedChordOrRest::decrMidiTime, incr > m_midi_time");
	}
	m_midi_time -= incr;
}


unsigned int NedChordOrRest::getDuration(unsigned int measure_duration /* = -1 */) {
	int denom;
	if (m_type & (TYPE_CLEF | TYPE_GRACE)) return 0;
	if (m_type == TYPE_REST && m_length == WHOLE_NOTE) {
		if (measure_duration != -1) {
			return measure_duration;
		}
		NedMeasure *measure = getSystem()->getMeasure(m_midi_time);
		if (measure == NULL) {
			return getMainWindow()->getNumerator() * WHOLE_NOTE / getMainWindow()->getDenominator();
		}
		denom = getMainWindow()->getDenominator(measure->getMeasureNumber());
		if (denom == 0) {
			return getMainWindow()->getNumerator() * WHOLE_NOTE / getMainWindow()->getDenominator();
		}
		return getMainWindow()->getNumerator(measure->getMeasureNumber()) * WHOLE_NOTE / denom;
	}
	return computeDuration(m_length, m_dot_count, m_tuplet_val);
}

unsigned int NedChordOrRest::computeDuration(unsigned int length, int dotcount, int tuplet_val) {
	unsigned int ret;
	if (dotcount == 1) {
		ret = length / 2 * 3;
		if (tuplet_val != 0) {
			ret = ret * NedResource::m_tuplet_tab[tuplet_val & TUPLET_VAR_MASK];
			return ret / (tuplet_val & TUPLET_VAR_MASK);
		}
		return ret;
	}
	if (dotcount == 2) {
		ret = length / 4 * 7;
		if (tuplet_val != 0) {
			ret = ret * NedResource::m_tuplet_tab[tuplet_val & TUPLET_VAR_MASK];
			return ret / (tuplet_val & TUPLET_VAR_MASK);
		}
		return ret;
	}
	if (tuplet_val != 0) {
		ret = length * NedResource::m_tuplet_tab[tuplet_val & TUPLET_VAR_MASK];
		return ret / (tuplet_val & TUPLET_VAR_MASK);
	}
	return length;
}


bool NedChordOrRest::tryErease(double x, double y, bool *removed) {
	NedCommandList *command_list;
	GList *lptr;
	*removed = false;
	if (m_type == TYPE_REST) return FALSE;
	if (g_list_length(m_notes) < 2) {
		for (lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
			if (((NedNote *) lptr->data)->trySelect(x, y)) {
				*removed = true;
				return TRUE;
			}
		}
	}
	else {
		for (lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
			if (((NedNote *) lptr->data)->trySelect(x, y)) {
				command_list = new NedCommandList(getMainWindow(), getSystem());
				testForTiesToDelete(command_list);
				command_list->addCommand(new NedEreaseNoteCommand(&m_notes, lptr));
				getMainWindow()->getCommandHistory()->addCommandList(command_list);
				command_list->execute();
				*removed = true;
				return FALSE;
			}
		}
		return FALSE;
	}
	return FALSE;
}

unsigned int NedChordOrRest::getStopTime() {
	return m_time + m_length;
}

double NedChordOrRest::getNeededSpace() {
	return m_bbox.width;
}

void NedChordOrRest::saveChordOrRest(FILE *fp) {
	GList *lptr;
	int marker;
	int last_font_line;
	int i;
	
	fprintf(fp, "(");
	if (m_beam != NULL || m_tuplet_ptr != NULL) {
		marker = NedResource::addAddr(this);
		fprintf(fp, "<%d>", marker);
	}
	if (m_type == TYPE_REST) {
		fprintf(fp, "r");
		if (m_is_hidden) {
			fprintf(fp, "h");
		}
	}
	else {
		fprintf(fp, "n");
		if ((m_status & STAT_UP) != 0) {
			fprintf(fp, "u");
		}
		else {
			fprintf(fp, "d");
		}
	}
	fprintf(fp, " %d, %x ", m_length / FACTOR, m_status & (~(ACCIDENTAL_MASK)));

	if (m_dot_count > 0) {
		fprintf(fp, " D %d ", m_dot_count);
	}

	if (m_type == TYPE_REST) {
		fprintf(fp, ")");
		return;
	}

	fprintf(fp, " { ");
	for (lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
		((NedNote *) lptr->data)->saveNote(fp);
		if (lptr != g_list_last(m_notes)) {
			fprintf(fp, " , ");
		}
	}
	fprintf(fp, " } ");
	last_font_line = -1;
	for (i = 0; i < MAX_LYRICS_LINES; i++) {
		if (m_lyrics[i] != NULL && g_utf8_strlen(m_lyrics[i], -1) > 0) {
			last_font_line = i;
		}
	}
	if (m_type == TYPE_NOTE && last_font_line >= 0 > 0) {
		fprintf(fp, "[");
		for (i = 0; i <= last_font_line; i++) {
			if (m_lyrics[i] != NULL && g_utf8_strlen(m_lyrics[i], -1) > 0) {
				fprintf(fp, " %s ", m_lyrics[i]);
			}
			if (i < last_font_line) {
				fprintf(fp, " , ");
			}
		}
		fprintf(fp, " ] ");
	}
	fprintf(fp, ")");
}

void NedChordOrRest::saveTies(FILE *fp, bool *ties_written) {
	GList *lptr;

	for (lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
		((NedNote *) lptr->data)->saveTies(fp, ties_written);
	}
}

NedNote* NedChordOrRest::getFirstNote() {
	return (NedNote *)g_list_first(m_notes)->data;
}

NedChordOrRest *NedChordOrRest::getNextChordOrRest() {return m_voice->getNextChordOrRest(this);}
NedChordOrRest *NedChordOrRest::getPreviousChordOrRest() {return m_voice->getPreviousChordOrRest(this);}

void NedChordOrRest::prepareReplay(int clef, int keysig, int octave_shift, int grace_time) {
	GList *lptr;

	for (lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
		((NedNote *) lptr->data)->prepareReplay(clef, keysig, octave_shift, grace_time, m_type == TYPE_GRACE, (m_status & STAT_STACC) != 0);
	}
}

void NedChordOrRest::removeUnneededAccidentals(int clef, int keysig, int octave_shift) {
	GList *lptr;

	for (lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
		((NedNote *) lptr->data)->removeUnneededAccidental(clef, keysig, octave_shift);
	}
}

void NedChordOrRest::setOffset(char offs_array[115]) {
	GList *lptr;

	for (lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
		((NedNote *) lptr->data)->setOffset(offs_array);
	}
}

NedChordOrRest *NedChordOrRest::restoreChordOrRest(FILE *fp, NedVoice *voice) {
	char buffer[128], note_descr[128];
	NedChordOrRest *chord_or_rest;
	NedNote *note;
	int length;
	int dotcount = 0;
	int marker, chord_marker;
	bool marker_read = FALSE, chord_marker_read = FALSE;
	unsigned int status = 0;
	int line, head;

	if (!NedResource::readWord(fp, buffer)) {
		NedResource::m_error_message = "Note or address expected";
		return NULL;
	}
	if (!strcmp(buffer, "<")) {
		if (!NedResource::readInt(fp, &chord_marker)) {
			NedResource::m_error_message = "Hex number expected";
			return NULL;
		}
		if (!NedResource::readWord(fp, buffer) || strcmp(buffer, ">")) {
			NedResource::m_error_message = "> expected";
			return NULL;
		}
		if (!NedResource::readWord(fp, buffer)) {
			NedResource::m_error_message = "Note expected";
			return NULL;
		}
		chord_marker_read = TRUE;
	}
	strcpy(note_descr, buffer);
	if (!NedResource::readInt(fp, &length)) {
		NedResource::m_error_message = "Note length expected";
		return NULL;
	}
	if (voice->getMainWindow()->getFileVersion() > 5) {
		if (!NedResource::readWord(fp, buffer) || strcmp(buffer, ",")) {
			NedResource::m_error_message = ", expected(0)";
			return NULL;
		}
		if (!NedResource::readHex(fp, (int *) &status)) {
			NedResource::m_error_message = "status expected";
			return NULL;
		}
	}
	if (!NedResource::readWord(fp, buffer)) {
		NedResource::m_error_message = "{ or D expected";
		return NULL;
	}
	if (!strcmp(buffer, "D")) {
		if (!NedResource::readInt(fp, &dotcount)) {
			NedResource::m_error_message = "dotcount expected";
			return NULL;
		}
		if (!NedResource::readWord(fp, buffer)) {
			NedResource::m_error_message = "{ expected";
			return NULL;
		}
	}
	if (note_descr[0] == 'r') {
		if (note_descr[1] != '\0') {
			if (note_descr[1] != 'h') {
				NedResource::m_error_message = "rh expected";
				return NULL;
			}
			chord_or_rest = new NedChordOrRest(voice, TYPE_REST, TRUE, 3, dotcount, length * FACTOR, NORMAL_NOTE, status, 0);
			if (chord_marker_read) {
				NedResource::addAddr(chord_marker, chord_or_rest);
			}
			return chord_or_rest;
		}
		chord_or_rest = new NedChordOrRest(voice, TYPE_REST, FALSE, 3, dotcount, length * FACTOR, NORMAL_NOTE, status, 0);
		if (chord_marker_read) {
			NedResource::addAddr(chord_marker, chord_or_rest);
		}
		return chord_or_rest;
	}
	if (note_descr[0] != 'n') {
		NedResource::m_error_message = "n or r expected";
		return NULL;
	}
	if (strcmp(buffer, "{")) {
		NedResource::m_error_message = "{ expected";
		return NULL;
	}

	if (!NedResource::readWord(fp, buffer) || buffer[1] != '\0') {
		NedResource::m_error_message = " | _, b , #, p, c, or = expected(1)";
		return NULL;
	}
	if (buffer[0] == '|') {
		if (!NedResource::readInt(fp, &marker)) {
			NedResource::m_error_message = "addr expected";
			return NULL;
		}
		if (!NedResource::readWord(fp, buffer) || buffer[0] != '|' || buffer[1] != '\0') {
			NedResource::m_error_message = " | expected";
			return NULL;
		}
		if (!NedResource::readWord(fp, buffer) || buffer[1] != '\0') {
			NedResource::m_error_message = " _, b , #, p, c, or = expected";
			return NULL;
		}
		marker_read = TRUE;
	}
	switch (buffer[0]) {
		case '_': break;
		case 'b': status |= STAT_FLAT; break;
		case 'p': status |= STAT_DFLAT; break;
		case '#': status |= STAT_SHARP; break;
		case 'c': status |= STAT_DSHARP; break;
		case '=': status |= STAT_NATURAL; break;
		default: NedResource::m_error_message = "_, b , #, p, c, or = expected"; return NULL;
	}
	if (!NedResource::readInt(fp, &line) || line < -20 || line > 20) {
		NedResource::m_error_message = "line expected";
		return NULL;
	}
	head = NORMAL_NOTE;
	if (!NedResource::readWord(fp, buffer)) {
		NedResource::m_error_message = " [ or } or , expected";
		return NULL;
	} 
	if (strcmp(buffer, "[")) {
		NedResource::unreadWord(buffer);
	}
	else {
		if (!NedResource::readWord(fp, buffer) || strcmp(buffer, "head")) {
			NedResource::m_error_message = "head expected";
			return NULL;
		}
		if (!NedResource::readWord(fp, buffer) || strcmp(buffer, ":")) {
			NedResource::m_error_message = ": expected";
			return NULL;
		}
		if (!NedResource::readInt(fp, &head) || head < CROSS_NOTE1 || head > TRIAG_NOTE2) {
			NedResource::m_error_message = "head value expected";
			return NULL;
		}
		if (!NedResource::readWord(fp, buffer) || strcmp(buffer, "]")) {
			NedResource::m_error_message = "] expected";
			return NULL;
		}
	}

	chord_or_rest = new NedChordOrRest(voice, TYPE_NOTE, FALSE, line, dotcount, length * FACTOR, head, status, 0);
	if (chord_marker_read) {
		NedResource::addAddr(chord_marker, chord_or_rest);
	}
	if (marker_read) {
		NedResource::addAddr(marker, g_list_first(chord_or_rest->m_notes)->data);
	}
			
	if (!NedResource::readWord(fp, buffer) || buffer[1] != '\0') {
		NedResource::m_error_message = " } or ,  expected";
		return NULL;
	}

	while (buffer[0] != '}') {
		marker_read = FALSE;
		if (buffer[0] != ',') {
			printf("gelesen(1): *%s*\n", buffer); fflush(stdout);
			NedResource::m_error_message = " ,  expected";
			return NULL;
		}
		if (!NedResource::readWord(fp, buffer) || buffer[1] != '\0') {
			printf("buffer = %s\n", buffer);
			NedResource::m_error_message = " | _, b , #, p, c, or = expected(2)";
			return NULL;
		}
		if (buffer[0] == '|') {
			if (!NedResource::readInt(fp, &marker)) {
				NedResource::m_error_message = "addr expected";
				return NULL;
			}
			if (!NedResource::readWord(fp, buffer) || buffer[0] != '|' || buffer[1] != '\0') {
				NedResource::m_error_message = " | expected";
				return NULL;
			}
			if (!NedResource::readWord(fp, buffer) || buffer[1] != '\0') {
				NedResource::m_error_message = " _, b , #, p, c, or = expected";
				return NULL;
			}
			marker_read = TRUE;
		}
		status = 0;
		switch (buffer[0]) {
			case '_': break;
			case 'p': status |= STAT_DFLAT; break;
			case 'b': status |= STAT_FLAT; break;
			case '#': status |= STAT_SHARP; break;
			case 'c': status |= STAT_DSHARP; break;
			case '=': status |= STAT_NATURAL; break;
			default: NedResource::m_error_message = "_, b , #, p, c or = expected"; return NULL;
		}
		if (!NedResource::readInt(fp, &line) || line < -20 || line > 20) {
			NedResource::m_error_message = "line expected";
			return NULL;
		}
		if (!NedResource::readWord(fp, buffer)) {
			NedResource::m_error_message = " }, [ or ,  expected";
			return NULL;
		}
		head = NORMAL_NOTE;
		if (strcmp(buffer, "[")) {
			NedResource::unreadWord(buffer);
		}
		else {
			if (!NedResource::readWord(fp, buffer) || strcmp(buffer, "head")) {
				NedResource::m_error_message = "head expected";
				return NULL;
			}
			if (!NedResource::readWord(fp, buffer) || strcmp(buffer, ":")) {
				NedResource::m_error_message = ": expected";
				return NULL;
			}
			if (!NedResource::readInt(fp, &head) || head < CROSS_NOTE1 || head > TRIAG_NOTE2) {
				NedResource::m_error_message = "head value expected";
				return NULL;
			}
			if (!NedResource::readWord(fp, buffer) || strcmp(buffer, "]")) {
				NedResource::m_error_message = "] expected";
				return NULL;
			}
		}
		note = new NedNote(chord_or_rest, line, head, status);
		void *addr = (void *) note;
		chord_or_rest->insertNoteAt(note, FALSE);
		if (marker_read) {
			NedResource::addAddr(marker, addr);
		}
		if (!NedResource::readWord(fp, buffer)) {
			printf("gelesen(2): *%s*\n", buffer); fflush(stdout);
			NedResource::m_error_message = " } or ,  expected";
			return NULL;
		}
	}
	if (!NedResource::readWord(fp, buffer) || buffer[1] != '\0') {
		NedResource::m_error_message = " ) or [ expected";
		return NULL;
	}
	if (buffer[0] == '[') {
		line = 0;
		while (buffer[0] != ']') {
			if (!NedResource::readString(fp, buffer)) {
				NedResource::m_error_message = "lyrics expected";
				return NULL;
			}
			if (buffer[0] == ',' && buffer[1] == '\0') {
				line++;
				continue;
			}
			chord_or_rest->createLyrics(line, buffer);
			if (!NedResource::readWord(fp, buffer) || (buffer[0] != ']' && buffer[0] != ',') || buffer[1] != '\0') {
				NedResource::m_error_message = ",  or ]  expected";
				return NULL;
			}
			line++;
		}
		if (!NedResource::readWord(fp, buffer) || buffer[0] != ')' || buffer[1] != '\0') {
			NedResource::m_error_message = " )  expected";
			return NULL;
		}
	}
	else if (buffer[0] != ')') {
		NedResource::m_error_message = " )  expected";
		return NULL;
	}
	chord_or_rest->xPositNotes();
	return chord_or_rest;
}

void NedChordOrRest::adjust_pointers(struct addr_ref_str *addrlist) {
	GList *lptr;

	for (lptr = g_list_first(m_notes); lptr; lptr = g_list_next(lptr)) {
		((NedNote *) lptr->data)->adjust_pointers(addrlist);
	}
}

