/****************************************************************************************/
/*											*/
/* This program is free software; you can redistribute it and/or modify it under the	*/
/* terms of the GNU General Public License as published by the Free Software		*/
/* Foundation; either version 2 of the License, or (at your option) any later version.	*/
/*											*/
/* This program is distributed in the hope that it will be useful, but WITHOUT ANY	*/
/* WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A	*/
/* PARTICULAR PURPOSE. See the GNU General Public License for more details.		*/
/*											*/
/* You should have received a copy of the GNU General Public License along with this	*/
/* program; (See "COPYING"). If not, If not, see <http://www.gnu.org/licenses/>.        */
/*											*/
/*--------------------------------------------------------------------------------------*/
/*											*/
/*  Copyright   Joerg Anders, TU Chemnitz, Fakultaet fuer Informatik, GERMANY           */
/*		ja@informatik.tu-chemnitz.de						*/
/*											*/
/*											*/
/****************************************************************************************/

#include <string.h>
#include "page.h"
#include "system.h"
#include "mainwindow.h"
#include "clipboard.h"
#include "staff.h"
#include "deletesystemcommand.h"
#include "appendsystemcommand.h"
#include "movesystemcommand.h"
#include "removepagecommand.h"
#include "getsystemfromnextpagecmmand.h"
#include "commandlist.h"
#include "scoreinfodialog.h"


#define X_POS_INVERS(p) ((leftx + (p)) / zoom_factor - m_xpos)
#define Y_POS_INVERS(p) ((topy + (p)) / zoom_factor)

#define DEFAULT_SYSTEM_DIST (2 * 5 * LINE_DIST)
NedPage::NedPage(NedMainWindow *main_window, double width, double height, int nr, unsigned int start_measure_number, bool start) :
m_systems(NULL), m_xpos(nr * (width + DEFAULT_BORDER)),
m_width(width), m_height(height), default_border(DEFAULT_BORDER), m_page_number(nr), m_main_window(main_window)
{
	int i;
	double system_pos = TOP_BOTTOM_BORDER;
	NedSystem *system;
	GList *lptr;

	if (start) {
		i = 0;
		m_system_diff = 100.0;
		do {
			m_systems = g_list_append(m_systems, system = new NedSystem(this, system_pos,
				m_width - 2 * (LEFT_RIGHT_BORDER - DEFAULT_BORDER), i, start_measure_number, TRUE));
			system_pos += system->getHeight() + DEFAULT_SYSTEM_DIST;
			i++;
			placeStaffs(0);
		}
		while (m_system_diff > MIN_SYSTEM_Y_DIST);
		if (g_list_length(m_systems) > 1) {
			lptr = g_list_last(m_systems);
			delete ((NedSystem *) lptr->data);
			m_systems = g_list_delete_link(m_systems, lptr);
			placeStaffs(0);
		}
	}
}

NedPage::~NedPage() {
	GList *lptr;

	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		delete ((NedSystem *) lptr->data);
	}
	g_list_free(m_systems);
	m_systems = NULL;
}

double NedPage::getTopPos() {return (2 * default_border +  m_height) * m_main_window->getCurrentZoomFactor();}
double NedPage::getLeftPos() {return (( m_xpos + 2*default_border +  m_width)) * m_main_window->getCurrentZoomFactor();}

NedSystem *NedPage::appendSystem() {
	int system_nr;
	NedSystem *system;

	system_nr = g_list_length(m_systems);
	m_systems = g_list_append(m_systems, system = new NedSystem(this, 0,
	                                m_width - 2 * (LEFT_RIGHT_BORDER - DEFAULT_BORDER), system_nr, 1, TRUE));

	placeStaffs(0);
	return system;
}

void NedPage::removeSystem(NedSystem *system) {
	GList *lptr;
	int system_nr;

	if ((lptr = g_list_find(m_systems, system)) == NULL) {
		NedResource::Abort("NedPage::removeSystem");
	}
	m_systems = g_list_delete_link(m_systems, lptr);

	for (system_nr = 0, lptr = g_list_first(m_systems); lptr; system_nr++, lptr = g_list_next(lptr)) {
		((NedSystem *) lptr->data)->setSystemNumber(system_nr);
	}
}

void NedPage::insertSystem(NedSystem *system) {
	
	m_systems = g_list_prepend(m_systems, system);
	system->changePageInfo(this);

}

bool NedPage::isPageOverflow() {
	if (g_list_length(m_systems) < 2) return false;
	do_place_staffs(0.0);
	return m_system_diff <= MIN_SYSTEM_Y_DIST;
}

void NedPage::fill_up () {
	int system_nr;
	GList *lptr;

	system_nr = g_list_length(m_systems);
	do {
		m_systems = g_list_append(m_systems, new NedSystem(this, 0,
	                                m_width - 2 * (LEFT_RIGHT_BORDER - DEFAULT_BORDER), system_nr, 1, TRUE));
		system_nr++;
		placeStaffs(0);
	}
	while (m_system_diff > MIN_SYSTEM_Y_DIST);
	if (g_list_length(m_systems) > 1) {
		lptr = g_list_last(m_systems);
		delete (NedSystem *) lptr->data;
		m_systems = g_list_delete_link(m_systems, lptr);
	}
	for (system_nr = 0, lptr = g_list_first(m_systems); lptr; system_nr++, lptr = g_list_next(lptr)) {
		((NedSystem *) lptr->data)->setSystemNumber(system_nr);
	}
}

unsigned int NedPage::getNumberOfLastMeasure() {
	GList *lptr;
	lptr = g_list_last(m_systems);
	return ((NedSystem *) g_list_last(m_systems)->data)->getNumberOfLastMeasure();
}

int NedPage::getSorting(NedStaff *this_staff, NedSystem *this_system, NedStaff *other_staff, NedSystem *other_system) {
	int pos0, pos1;
	if (this_staff->getStaffNumber() != other_staff->getStaffNumber()) {
		return SORTING_NONE;
	}
	if ((pos0 = g_list_index(m_systems, this_system)) < 0) {
		NedResource::Warning("NedPage::getSorting: didn't foind system 0");
		return SORTING_NONE;
	}
	if ((pos1 = g_list_index(m_systems, other_system)) < 0) {
		NedResource::Warning("dNedPage::getSorting: idn't foind system 1");
		return SORTING_NONE;
	}
	if (pos0 + 1 == pos1) {
		return SORTING_GREAT;
	}
	if (pos1 + 1 == pos0) {
		return SORTING_LESS;
	}
	return SORTING_NONE;
}

bool NedPage::isFirst(NedSystem *system) {
	GList *lptr;
	if ((lptr = g_list_find(m_systems, system)) == NULL) {
		 NedResource::Abort("NedPage::isFirst");
	}
	return (lptr == g_list_first(m_systems));
}

bool NedPage::isLast(NedSystem *system) {
	GList *lptr;
	if ((lptr = g_list_find(m_systems, system)) == NULL) {
		 NedResource::Abort("NedPage::isLast");
	}
	return (lptr == g_list_last(m_systems));
}



void NedPage::draw(cairo_t *cr, double main_width, double main_height) {
	double leftx = m_main_window->getLeftX();
	double topy = m_main_window->getTopY();
	double zoom_factor = m_main_window->getCurrentZoomFactor();
	ScoreInfo *score_info = NULL;
	bool score_info_present;
	double xp, yp;
	bool first = true;

	if (!m_main_window->doDrawPostscript()) {
		if (m_xpos * zoom_factor - leftx > main_width) return;
		if ((m_xpos + (DEFAULT_BORDER + m_width)) * zoom_factor - leftx  < 0) return;
	
	
		cairo_set_source_rgb (cr, 0.3, 0.3, 0.3);
		cairo_rectangle (cr,  m_xpos * zoom_factor - leftx, -topy,
			(2 * DEFAULT_BORDER + m_width) * zoom_factor,
			(2 * DEFAULT_BORDER + m_height) * zoom_factor);
	
		cairo_fill(cr);
		cairo_set_source_rgb (cr, 1.0, 1.0, 1.0);
		cairo_rectangle (cr,  (m_xpos + DEFAULT_BORDER) * zoom_factor -leftx,
			DEFAULT_BORDER * zoom_factor - topy,
		 	m_width * zoom_factor, m_height * zoom_factor);
		cairo_fill(cr);
	

		cairo_set_source_rgb (cr, 0.0, 0.0, 0.0);
		xp = m_xpos;
	}
	else {
		cairo_set_source_rgb (cr, 1.0, 1.0, 1.0);
		cairo_rectangle (cr,  (DEFAULT_BORDER) * zoom_factor,
			DEFAULT_BORDER * zoom_factor,
		 	m_width * zoom_factor, m_height * zoom_factor);
		cairo_fill(cr);
		xp = 0.0;
	}
	GList *lptr;
	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		((NedSystem *) lptr->data)->draw(cr, m_page_number == 0 && first);
		first = false;
	}
	if (m_page_number == 0) {
		yp = 0.0;
		score_info = getMainWindow()->getScoreInfo();
		if (m_main_window->getScoreInfo()->title != NULL) {
			score_info_present = true;
			cairo_new_path(cr);
			cairo_select_font_face(cr, TITLE_FONT, TITLE_FONT_SLANT, TITLE_FONT_WEIGHT);
			cairo_set_font_size(cr, TITLE_FONT_SIZE * zoom_factor);
			cairo_move_to(cr, (xp + DEFAULT_BORDER + (m_width - score_info->title_extends.width) / 2.0) * zoom_factor -leftx,
				(STAFF_TOP_DIST + score_info->title_extends.height) * zoom_factor - topy);
			cairo_show_text(cr, score_info->title);
			cairo_stroke(cr);
			yp += score_info->title_extends.height + SCORE_INFO_EXTRA_DIST;
		}
		if (m_main_window->getScoreInfo()->subject != NULL) {
			score_info_present = true;
			cairo_new_path(cr);
			cairo_select_font_face(cr, SUBJECT_FONT, SUBJECT_FONT_SLANT, SUBJECT_FONT_WEIGHT);
			cairo_set_font_size(cr, SUBJECT_FONT_SIZE * zoom_factor);
			cairo_move_to(cr, (xp + DEFAULT_BORDER + (m_width - score_info->subject_extends.width) / 2.0) * zoom_factor -leftx,
				(STAFF_TOP_DIST + yp + score_info->subject_extends.height) * zoom_factor - topy);
			cairo_show_text(cr, score_info->subject);
			cairo_stroke(cr);
			yp += score_info->subject_extends.height;
		}
		if (m_main_window->getScoreInfo()->composer != NULL) {
			score_info_present = true;
			cairo_new_path(cr);
			cairo_select_font_face(cr, COMPOSER_FONT, COMPOSER_FONT_SLANT, COMPOSER_FONT_WEIGHT);
			cairo_set_font_size(cr, COMPOSER_FONT_SIZE * zoom_factor);
			cairo_move_to(cr, (xp + DEFAULT_BORDER + (m_width - score_info->composer_extends.width - DEFAULT_BORDER - LEFT_RIGHT_BORDER)) * zoom_factor -leftx,
				(STAFF_TOP_DIST + yp + score_info->composer_extends.height) * zoom_factor - topy);
			cairo_show_text(cr, score_info->composer);
			cairo_stroke(cr);
			yp += score_info->composer_extends.height + SCORE_INFO_EXTRA_DIST;
		}
		if (m_main_window->getScoreInfo()->arranger != NULL) {
			score_info_present = true;
			cairo_new_path(cr);
			cairo_select_font_face(cr, ARRANGER_FONT, ARRANGER_FONT_SLANT, ARRANGER_FONT_WEIGHT);
			cairo_set_font_size(cr, ARRANGER_FONT_SIZE * zoom_factor);
			cairo_move_to(cr, (xp + DEFAULT_BORDER + (m_width - score_info->arranger_extends.width - DEFAULT_BORDER - LEFT_RIGHT_BORDER)) * zoom_factor -leftx,
				(STAFF_TOP_DIST + yp + score_info->arranger_extends.height) * zoom_factor - topy);
			cairo_show_text(cr, score_info->arranger);
			cairo_stroke(cr);
			yp += score_info->arranger_extends.height + SCORE_INFO_EXTRA_DIST;
		}
		if (m_main_window->getScoreInfo()->copyright != NULL) {
			score_info_present = true;
			cairo_new_path(cr);
			cairo_select_font_face(cr, COPYRIGHT_FONT, COPYRIGHT_FONT_SLANT, COPYRIGHT_FONT_WEIGHT);
			cairo_set_font_size(cr, COPYRIGHT_FONT_SIZE * zoom_factor);
			cairo_move_to(cr, (xp + DEFAULT_BORDER + (m_width - score_info->copyright_extends.width - DEFAULT_BORDER - LEFT_RIGHT_BORDER)) * zoom_factor -leftx,
				(STAFF_TOP_DIST + yp + score_info->copyright_extends.height) * zoom_factor - topy);
			cairo_show_text(cr, score_info->copyright);
			cairo_stroke(cr);
		}
		if (score_info_present) {
			cairo_set_font_face(cr, NedResource::getFontFace());
			cairo_set_font_matrix(cr,  NedResource::getFontMatrix(getMainWindow()->getCurrentZoomLevel()));
		}
	}
	cairo_show_page(cr);
};


int NedPage::getNumberOfStaffs() {
	return ((NedSystem *) g_list_first(m_systems)->data)->getNumberOfStaffs();
}

double  NedPage::getContetXpos() {
	if (m_main_window->doDrawPostscript()) {
		return LEFT_RIGHT_BORDER;
	}
	return m_xpos + LEFT_RIGHT_BORDER;
}

bool NedPage::reposit(int pass, NedCommandList *command_list /* = NULL */) {
	GList *lptr;
	bool ch, changes = FALSE;
	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		ch = ((NedSystem *) lptr->data)->reposit(pass, -1, command_list);
		changes = changes || ch;
	}
	return changes;
}

void NedPage::setAndUpdateClefTypeAndKeySig(int *clef_and_key_array) {
	GList *lptr;
	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		((NedSystem *) lptr->data)->setAndUpdateClefTypeAndKeySig(clef_and_key_array);
	}
}

void NedPage::handleStaffElements() {
	GList *lptr;
	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		((NedSystem *) lptr->data)->handleStaffElements();
	}
}

void NedPage::removeUnneededAccidentals() {
	GList *lptr;
	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		((NedSystem *) lptr->data)->removeUnneededAccidentals();
	}
}


void NedPage::prepareReplay() {
	GList *lptr;

	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		((NedSystem *) lptr->data)->prepareReplay();
		NedResource::increaseSystemStartTime(((NedSystem *) lptr->data)->getSystemEndTime());

	}
}

void NedPage::reconfig_paper(double width, double height) {
	GList *lptr;

	m_xpos = m_page_number * (width + DEFAULT_BORDER);
	m_width = width; m_height = height;
	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		((NedSystem *) lptr->data)->setWidth(m_width - 2 * (LEFT_RIGHT_BORDER - DEFAULT_BORDER));
		((NedSystem *) lptr->data)->resetUntouched();
	}

}

void NedPage::savePage(FILE *fp) {
	GList *lptr;
	fprintf(fp, "PAGE %d\n", m_page_number);
	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		((NedSystem *) lptr->data)->saveSystem(fp);
	}
}

void NedPage::restorePage(FILE *fp, unsigned int start_measure_number) {
	int i, system_nr;
	char buffer[128];
	NedSystem *system;
	double system_pos = TOP_BOTTOM_BORDER;

	i = 0;
	do {
		if (!NedResource::readWord(fp, buffer)) {
			if (feof(fp)) return;
			NedResource::m_error_message = "(1)SYSTEM or PAGE expected";
			return;
		}
		if (strcmp(buffer, "SYSTEM")) {
			if (!strcmp(buffer, "PAGE")) {
				NedResource::unreadWord(buffer);
				return;
			}
			printf("buffer = %s\n", buffer);
			NedResource::m_error_message = "(2)SYSTEM or PAGE expected";
			return;
		}
	
		if (!NedResource::readInt(fp, &system_nr)) {
			NedResource::m_error_message = "SYSTEM number expected";
			return;
		}
		if (system_nr != i) {
			NedResource::m_error_message = "bad system_nr";
			return;
		}
		if (!NedResource::readWord(fp, buffer) || strcmp(buffer, ":")) {
			NedResource::m_error_message = ": expected";
			return;
		}

		system = new NedSystem(this, system_pos, m_width - 2 * (LEFT_RIGHT_BORDER - DEFAULT_BORDER), system_nr, start_measure_number, FALSE);
		m_systems = g_list_append(m_systems, system);

		system->restoreSystem(fp);

		i++;
		system_pos += system->getHeight() + DEFAULT_SYSTEM_DIST;

	}
	while (NedResource::m_error_message == NULL);
}



bool NedPage::tryChangeLength(NedChordOrRest *chord_or_rest) {
	GList *lptr;
	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		if (((NedSystem *) lptr->data)->tryChangeLength(chord_or_rest)) {
			return TRUE;
		}
	}
	return FALSE;
}


bool NedPage::trySelect(double x, double y) {
	double zoom_factor = m_main_window->getCurrentZoomFactor();
	double leftx = m_main_window->getLeftX();

	double xl = X_POS_INVERS(x);


	if (xl < 0.0 || xl > m_width) return FALSE;

	GList *lptr;
	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		if (((NedSystem *) lptr->data)->trySelect(x, y)) {
			return TRUE;
		}
	}
	return FALSE;
}

bool NedPage::findTimeOfMeasure(int meas_num, unsigned long long *meas_time, unsigned long long *system_offs) {
	GList *lptr;
	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		if (((NedSystem *) lptr->data)->findTimeOfMeasure(meas_num, meas_time)) {
			return true;
		}
		*system_offs += ((NedSystem *) lptr->data)->getSystemEndTime();
	}
	return false;
}

void NedPage::collectSelectionRectangleElements(NedBbox *sel_rect, GList **sel_group, NedSystem *first_selected_system, NedSystem *last_selected_system,
	NedPage *first_selected_page, NedPage *last_selected_page) {
	double xp = sel_rect->x - m_xpos;
	if (xp + sel_rect->width < 0.0 || m_width < xp) return;
	GList *lptr;
	xp -= LEFT_RIGHT_BORDER;
	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		((NedSystem *) lptr->data)->collectSelectionRectangleElements(xp, sel_rect, sel_group, first_selected_system, last_selected_system,
		first_selected_page == this, last_selected_page == this);
	}
}

void NedPage::findSelectedFirstsLasts(NedBbox *sel_rect, int *number_of_first_selected_staff, int *number_of_last_selected_staff,
		NedSystem **first_selected_system, NedSystem **last_selected_system, NedPage **first_selected_page, NedPage **last_selected_page) {
	double xp = sel_rect->x - m_xpos;
	if (xp +  sel_rect->width < 0.0 || m_width < xp) return;
	GList *lptr;
	xp -= LEFT_RIGHT_BORDER;
	if (*first_selected_page == NULL) {
		*first_selected_page = *last_selected_page = this;
	}
	else {
		*last_selected_page = this;
	}
	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		((NedSystem *) lptr->data)->findSelectedSystems(sel_rect, number_of_first_selected_staff, number_of_last_selected_staff, first_selected_system, last_selected_system);
	}
}

void NedPage::findFromTo(GList *clipboard, NedPage **min_page, NedPage **max_page, NedSystem **min_sys, NedSystem **max_sys) {
	GList *lptr;

	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		if (((NedSystem *) lptr->data)->findFromTo(clipboard, min_sys, max_sys)) {
			if (*min_page == NULL) {
				*min_page = *max_page = this;
			}
			else {
				*max_page = this;
			}
		}
	}
}

void NedPage::deleteItemsFromTo(NedCommandList *command_list, bool is_first_page, bool is_last_page,
		NedSystem *min_sys, NedSystem *max_sys, unsigned long long start_midi, unsigned long long end_midi) {
	NedSystem *system;
	GList *lptr;

	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		system = (NedSystem *) lptr->data;
		if (is_first_page && system->getSystemNumber() < min_sys->getSystemNumber()) continue;
		if (is_last_page && system->getSystemNumber() > max_sys->getSystemNumber()) break;
		system->deleteItemsFromTo(command_list, system == min_sys, system == max_sys, start_midi, end_midi);
	}
}

void NedPage::removeNotesFromTo(NedCommandList *command_list, GList *items, bool is_first_page, bool is_last_page,
			NedSystem *min_sys, NedSystem *max_sys) {
	GList *lptr;
	NedSystem *system;

	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		system = (NedSystem *) lptr->data;
		if (is_first_page && system->getSystemNumber() < min_sys->getSystemNumber()) continue;
		if (is_last_page && system->getSystemNumber() > max_sys->getSystemNumber()) break;
		system->removeNotesFromTo(command_list, items, system == min_sys, system == max_sys);
	}
}

void NedPage::testForPageBackwardTies(NedCommandList *command_list) {
	((NedSystem *) g_list_first(m_systems)->data)->testForPageBackwardTies(command_list);
}

void NedPage::checkForElementsToSplit(NedCommandList *command_list) {
	GList *lptr;
	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		((NedSystem *) lptr->data)->checkForElementsToSplit(command_list);
	}
}
	


NedStaff *NedPage::findStaff(double x, double y, NedMeasure **measure) {
	double zoom_factor = m_main_window->getCurrentZoomFactor();
	double leftx = m_main_window->getLeftX();
	NedStaff *staff;

	double xl = X_POS_INVERS(x);


	if (xl < 0.0 || xl > m_width) return NULL;

	GList *lptr;
	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		if ((staff = ((NedSystem *) lptr->data)->findStaff(xl + m_xpos, y, measure)) != NULL) {
			return staff;
		}
	}
	return NULL;
}

bool NedPage::tryInsertOrErease(double x, double y) {
	double zoom_factor = m_main_window->getCurrentZoomFactor();
	double leftx = m_main_window->getLeftX();

	double xl = X_POS_INVERS(x);


	if (xl < 0.0 || xl > m_width) return FALSE;

	GList *lptr;
	double mindist, d;
	lptr = g_list_first(m_systems);
	NedSystem *nearest_system = (NedSystem *) lptr->data;
	mindist = nearest_system->computeMidDist(y);
	for (lptr = g_list_next(lptr); lptr; lptr = g_list_next(lptr)) {
		if ((d = ((NedSystem *) lptr->data)->computeMidDist(y)) < mindist) {
			nearest_system = (NedSystem *) lptr->data;
			mindist = d;
		}
	}
	if (nearest_system->tryInsertOrErease(x, y)) {
		return TRUE;
	}
	return FALSE;
}


bool NedPage::findLine(double x, double y, int *ypos, int *line, int *bottom) {
	double zoom_factor = m_main_window->getCurrentZoomFactor();
	double leftx = m_main_window->getLeftX();

	double xl = X_POS_INVERS(x);

	if (xl < 0.0 || xl > m_width) return FALSE;
	GList *lptr;
	double mindist, d;
	lptr = g_list_first(m_systems);
	NedSystem *nearest_system = (NedSystem *) lptr->data;
	mindist = nearest_system->computeMidDist(y);
	for (lptr = g_list_next(lptr); lptr; lptr = g_list_next(lptr)) {
		if ((d = ((NedSystem *) lptr->data)->computeMidDist(y)) < mindist) {
			nearest_system = (NedSystem *) lptr->data;
			mindist = d;
		}
	}
	if (nearest_system->findLine(x, y, ypos, line, bottom)) {
		return TRUE;
	}
	return FALSE;
}

void NedPage::renumberSystems() {
	int i;
	GList *lptr;

	for (i = 0, lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr), i++) {
		((NedSystem *) lptr->data)->setSystemNumber(i);
	}
}

void NedPage::findLastTouchedSystem(NedSystem **start_of_systems) {
	GList *lptr;

	for (lptr = g_list_last(m_systems); lptr && *start_of_systems == NULL; lptr = g_list_previous(lptr)) {
		if (!((NedSystem *) lptr->data)->isUntouched()) {
			*start_of_systems = (NedSystem *) lptr->data;
			return;
		}
	}
}

bool NedPage::placeStaffs(int pass, NedCommandList *command_list /* = NULL */) {
	GList *lptr;
	NedPage *other_page;
	double offs = STAFF_TOP_DIST;
	NedRemovePageCommand *rem_last_page_cmd;
	do_place_staffs(offs);
	bool changed = FALSE;

	if (m_page_number == 0) {
		offs += getMainWindow()->getFirstPageYOffs();
	}

	if (pass == 0) m_ping_pong = FALSE;
	do_place_staffs(offs);
	if (pass == 0 && m_system_diff < MIN_SYSTEM_Y_DIST) {
		if (command_list != NULL && g_list_length(m_systems) > 1) {
			lptr = g_list_last(m_systems);
			if (((NedSystem *) lptr->data)->isUntouched()) {
				NedDeleteSystemCommand *command = new NedDeleteSystemCommand(this, &m_systems, lptr);
				command_list->addCommand(command);
				command->execute(); // the commandlist is not executed
			}
			else {
				other_page = m_main_window->getNextPage(this, command_list);
				NedMoveSystemCommand *move_system_command = new NedMoveSystemCommand(lptr, this, &m_systems, other_page, &(other_page->m_systems));
				move_system_command->execute(); // the commandlist is not executed
				command_list->addCommand(move_system_command);
				//other_page->placeStaffs(command_list);
			}
			changed = TRUE;
			do_place_staffs(offs);
		}
	}
	if (pass == 1 && !isEmpty() && !m_ping_pong && command_list != NULL && m_system_diff > MAX_SYSTEM_Y_DIST) {
		if ((other_page = m_main_window->getNextPage(this)) == NULL || other_page->isEmpty()) {
			NedAppendSystemCommand *command = new NedAppendSystemCommand(this, m_width, &m_systems, getNumberOfLastMeasure() + 1);
			command->execute(); // the commandlist is not executed ; performs placeStaffs();
			if (m_system_diff < MIN_SYSTEM_Y_DIST) {
				command->unexecute();
				m_ping_pong = TRUE;
				delete command;
			}
			else {
				command_list->addCommand(command);
			}
		}
		else {
			if ((lptr = g_list_first(other_page->m_systems)) == NULL) {
				NedResource::Abort("NedPage::placeStaffs");
			}
			NedGetSystemFromNextPageCommand *get_system_from_next_page_command =
				new NedGetSystemFromNextPageCommand(lptr, this, other_page);
			get_system_from_next_page_command->execute(); // the commandlist is not executed; performs placeStaffs()
			if (m_system_diff < MIN_SYSTEM_Y_DIST) {
				m_ping_pong = TRUE;
				get_system_from_next_page_command->unexecute();
				delete get_system_from_next_page_command;
			}
			else {
				command_list->addCommand(get_system_from_next_page_command);
				//other_page->placeStaffs(command_list);
				changed = TRUE;
				/*
				if (other_page->isEmpty()) {
					rem_last_page_cmd = new NedRemovePageCommand(m_main_window, other_page);
					command_list->addCommand(rem_last_page_cmd);
				}
				*/
			}
			if (!isEmpty()) {
				do_place_staffs(offs);
			}
		}
	}
	return changed;
}

void NedPage::do_place_staffs(double offs) {
	GList *lptr;
	double staffpos = TOP_BOTTOM_BORDER;

	int system_count = g_list_length(m_systems);

	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		staffpos = ((NedSystem *) lptr->data)->placeStaffs(staffpos);
	}
	m_system_diff = m_height - 2 * TOP_BOTTOM_BORDER - offs + STAFF_TOP_DIST - staffpos;
	if (m_system_diff > 0.0 && g_list_length(m_systems) > 0 ) {
		if (system_count > 1) {
			m_system_diff /= system_count - 1;
		}
		for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
			((NedSystem *) lptr->data)->shiftY(offs);
			offs += m_system_diff;
		}
	}
}

NedSystem *NedPage::getNextSystem(NedSystem *system, NedCommandList *command_list /* = NULL */ ) {
	GList *lptr;

	if (isEmpty()) return NULL;


	if ((lptr = g_list_find(m_systems, system)) == NULL) {
		NedResource::Abort("NedPage::getNextSystem");
	}

	if ((lptr = g_list_next(lptr)) == NULL) {
		return m_main_window->getNextSystem(this, command_list);
	}
	return (NedSystem *) lptr->data;
}

NedSystem *NedPage::getPreviousSystem(NedSystem *system) {
	GList *lptr;
	if ((lptr = g_list_find(m_systems, system)) == NULL) {
		NedResource::Abort("NedPage::getPreviousSystem");
	}
	if ((lptr = g_list_previous(lptr)) == NULL) {
		return NULL;
	}
	return (NedSystem *) lptr->data;
}

NedSystem *NedPage::getFirstSystem() {
	GList *lptr;

	if ((lptr = g_list_first(m_systems)) == NULL) {
		NedResource::Abort("NedPage::getFirstSystem");
	}

	return (NedSystem *) lptr->data;
}

void NedPage::appendStaff(NedCommandList *command_list, int p_staff_nr /* = -1 */) {
	GList *lptr;

	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		((NedSystem *) lptr->data)->appendStaff(command_list, p_staff_nr);
	}
}

void NedPage::removeLastStaff() {
	GList *lptr;

	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		((NedSystem *) lptr->data)->removeLastStaff();
	}
}

void NedPage::deleteStaff(int staff_number) {
	GList *lptr;

	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		((NedSystem *) lptr->data)->deleteStaff(staff_number);
	}
}

void NedPage::restoreStaff(int staff_number) {
	GList *lptr;

	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		((NedSystem *) lptr->data)->restoreStaff(staff_number);
	}
}

void NedPage::shiftStaff(int staff_number, int position) {
	GList *lptr;

	for (lptr = g_list_first(m_systems); lptr; lptr = g_list_next(lptr)) {
		((NedSystem *) lptr->data)->shiftStaff(staff_number, position);
	}
}
