/****************************************************************************************/
/*											*/
/* This program is free software; you can redistribute it and/or modify it under the	*/
/* terms of the GNU General Public License as published by the Free Software		*/
/* Foundation; either version 2 of the License, or (at your option) any later version.	*/
/*											*/
/* This program is distributed in the hope that it will be useful, but WITHOUT ANY	*/
/* WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A	*/
/* PARTICULAR PURPOSE. See the GNU General Public License for more details.		*/
/*											*/
/* You should have received a copy of the GNU General Public License along with this	*/
/* program; (See "COPYING"). If not, If not, see <http://www.gnu.org/licenses/>.        */
/*											*/
/*--------------------------------------------------------------------------------------*/
/*											*/
/*  Copyright   Joerg Anders, TU Chemnitz, Fakultaet fuer Informatik, GERMANY           */
/*		ja@informatik.tu-chemnitz.de						*/
/*											*/
/*											*/
/****************************************************************************************/

#include <string.h>
#include "voice.h"
#include "beaming.h"
#include "staff.h"
#include "chordorrest.h"
#include "system.h"
#include "page.h"
#include "resource.h"
#include "system.h"
#include "mainwindow.h"
#include "commandlist.h"
#include "clipboard.h"
#include "beam.h"
#include "tuplet.h"
#include "note.h"
#include "commandhistory.h"
#include "insertchordorrestcommand.h"
#include "ereasechordorrestcommand.h"
#include "changechordorrestlength.h"
#include "addnoteatcommand.h"
#include "deletechordorrestgroupcommand.h"
#include "insertchordorrestgroupcommand.h"
#include "deletechordcommand.h"
#include "changexpositioncommand.h"
#include "inserttiedelementcommand.h"
#include "settotupletcommand.h"
#include "insertstaffelemcommand.h"
#include "musicxmlimport.h"



#define DEFAULT_STAFF_DIST (4 * LINE_DIST)

#define X_POS_INVERS_PAGE_REL(p) ((leftx + (p)) / zoom_factor - getPage()->getContetXpos())
#define Y_POS_INVERS_STAFF_REL(p) ((topy + (p)) / zoom_factor - (getSystem()->getYPos() + getStaff()->getBottomPos()))

#define X_POS_PAGE_REL(p) ((getPage()->getContetXpos() + (p)) * zoom_factor - leftx)
#define Y_POS_STAFF_REL(p) (((p) + getSystem()->getYPos() + getStaff()->getBottomPos()) * zoom_factor - topy)

#define X_PS_POS(p) ((DEFAULT_BORDER + LEFT_RIGHT_BORDER + (p)) * PS_ZOOM)
#define Y_PS_POS_STAFF_REL(p) ((height - ((p) + getSystem()->getYPos() + getStaff()->getBottomPos())) * PS_ZOOM)


NedVoice::NedVoice(NedStaff *staff, int nr, bool start) :
m_chord_or_rests(NULL), m_beam_list(NULL), m_tuplet_list(NULL), m_voice_nr(nr), m_staff(staff), m_stem_direction(STEM_DIR_NONE), m_start_of_last_imported(NULL) {
#ifdef XXX
	unsigned int rest_time = 0;
	int i;

	if (start) {
		for (i = 0; i < START_MEASURE_COUNT; i++) {
			NedChordOrRest *rest = new NedChordOrRest(this, TYPE_REST, nr > 0, 3, 0, WHOLE_NOTE, getMainWindow->getCurrentNoteHead(), 0, rest_time);
			m_chord_or_rests = g_list_append(m_chord_or_rests, rest);
			rest_time = rest->getStopTime();
		}
	}
#endif
}

void NedVoice::appendWholeRest(NedCommandList *command_list) {
	NedChordOrRest *rest = new NedChordOrRest(this, TYPE_REST, m_voice_nr > 0, 3, 0, WHOLE_NOTE, NORMAL_NOTE, 0, 0);
	if (command_list == NULL) {
		m_chord_or_rests = g_list_append(m_chord_or_rests, rest);
		return;
	}
	NedInsertChordOrRestCommand *command = new NedInsertChordOrRestCommand(this, -1, rest);
	command->resetAdjustable(); // avoid ajjustment to filling wholes if undo/redo
	command->execute();
	command_list->addCommand(command);
}

void NedVoice::appendAppropriateWholes() {
	int i, meas_num = getSystem()->getMeasureCount();
	for (i = 0; i < meas_num; i++) {
		NedChordOrRest *rest = new NedChordOrRest(this, TYPE_REST, m_voice_nr > 0, 3, 0, WHOLE_NOTE, NORMAL_NOTE, 0, 0);
		m_chord_or_rests = g_list_append(m_chord_or_rests, rest);
	}
}


NedVoice::~NedVoice() {
	GList *lptr;

	for(lptr = g_list_first(m_beam_list); lptr; lptr = g_list_next(lptr)) {
		// important: must be first
		delete ((NedBeam *) lptr->data);
	}
	g_list_free(m_beam_list);
	m_beam_list = NULL;

	for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		delete ((NedChordOrRest *) lptr->data);
	}

	g_list_free(m_chord_or_rests);
	m_chord_or_rests = NULL;
}

int NedVoice::getNumElements() {
	return g_list_length(m_chord_or_rests);
}

void NedVoice::empty() {
	GList *lptr;

	for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		delete (NedChordOrRest *) lptr->data;
	}
	g_list_free(m_chord_or_rests);
	m_chord_or_rests = NULL;
}


NedMainWindow *NedVoice::getMainWindow() {return m_staff->getSystem()->getPage()->getMainWindow();}
NedPage *NedVoice::getPage() {return m_staff->getSystem()->getPage();}
NedSystem *NedVoice::getSystem() {return m_staff->getSystem();}

void NedVoice::draw(cairo_t *cr) {
	GList *lptr;
	bool lyrics_present = false;
	for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		((NedChordOrRest *) lptr->data)->draw(cr, &lyrics_present);
	}
	for(lptr = g_list_first(m_beam_list); lptr; lptr = g_list_next(lptr)) {
		((NedBeam *) lptr->data)->draw(cr);
	}
	for(lptr = g_list_first(m_tuplet_list); lptr; lptr = g_list_next(lptr)) {
		((NedTuplet *) lptr->data)->draw(cr);
	}
	if (lyrics_present) {
		cairo_new_path(cr);
		cairo_select_font_face(cr, LYRICS_FONT, LYRICS_FONT_SLANT, LYRICS_FONT_WEIGHT);
		cairo_set_font_size(cr, LYRICS_FONT_SIZE * getMainWindow()->getCurrentZoomFactor());
		for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
			((NedChordOrRest *) lptr->data)->drawlyrics(cr);
		}
		cairo_stroke(cr);
		cairo_set_font_face(cr, NedResource::getFontFace());
		cairo_set_font_matrix(cr,  NedResource::getFontMatrix(getMainWindow()->getCurrentZoomLevel()));
	}
}

bool NedVoice::trySelect(double x, double y) {
	GList *lptr;
	for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		if (((NedChordOrRest *) lptr->data)->trySelect(x, y)) {
			return TRUE;
		}
	}
	return FALSE;
}

void NedVoice::appendElementsOfMeasureLength(part *part_ptr, unsigned int meas_duration) {
	GList *lptr;
	NedChordOrRest *chord_or_rest;
	int pos = 0, dotcount;
	unsigned int len2;
	bool first = true;
	unsigned int duration = 0;

	if (part_ptr->eof[m_voice_nr]) {
		chord_or_rest = new NedChordOrRest(this, TYPE_REST, m_voice_nr > 0, 3, 0, WHOLE_NOTE, NORMAL_NOTE, 0, 0);
		m_chord_or_rests = g_list_append(m_chord_or_rests, chord_or_rest);
		m_start_of_last_imported = g_list_find(m_chord_or_rests, chord_or_rest);
		return;
	}

	if (part_ptr->start_of_last_taken[m_voice_nr] == NULL) {
		lptr = g_list_first(part_ptr->voices[m_voice_nr]);
	}
	else {
		lptr = g_list_next(part_ptr->end_of_last_taken[m_voice_nr]);
	}
	if (lptr == NULL) {
		part_ptr->eof[m_voice_nr] = true;
		chord_or_rest = new NedChordOrRest(this, TYPE_REST, m_voice_nr > 0, 3, 0, WHOLE_NOTE, NORMAL_NOTE, 0, 0);
		m_chord_or_rests = g_list_append(m_chord_or_rests, chord_or_rest);
		m_start_of_last_imported = g_list_find(m_chord_or_rests, chord_or_rest);
		return;
	}

	if (m_start_of_last_imported == NULL) {
		pos = 0;
	}
	else {
		if ((pos = g_list_position(m_chord_or_rests, m_end_of_last_imported)) < 0) {
			NedResource::Abort("NedVoice::appendElementsOfMeasureLength(1)");
		}
		pos++;
	}

	for (;duration < meas_duration && lptr; lptr = g_list_next(lptr)) {
		chord_or_rest = (NedChordOrRest *) lptr->data;
		m_chord_or_rests = g_list_insert(m_chord_or_rests, chord_or_rest, pos++);
		if (first) {
			first = false;
			if ((m_start_of_last_imported = g_list_nth(m_chord_or_rests, pos - 1)) == NULL) {
				NedResource::Abort("NedVoice::appendElementsOfMeasureLength(4)");
			}
			part_ptr->start_of_last_taken[m_voice_nr] = lptr;
		}
		part_ptr->end_of_last_taken[m_voice_nr] = lptr;
		chord_or_rest->setVoice(this);
		chord_or_rest->computeLyricsParams();
		chord_or_rest->xPositNotes();
		chord_or_rest->computeBbox();
		if ((chord_or_rest->getType() & (TYPE_NOTE | TYPE_REST)) != 0) {
			duration += chord_or_rest->getDuration();
		}
		chord_or_rest->computeBbox();
		chord_or_rest->reConfigure();
	}
	if (duration > meas_duration) {
		fprintf(stderr, "Overflow\n"); fflush(stderr);
	}
	if (duration < meas_duration) {
		NedResource::setLengthField(meas_duration - duration);
		while ((len2 = NedResource::getPartLength(&dotcount)) > 0) {
			m_chord_or_rests = g_list_insert(m_chord_or_rests, new NedChordOrRest(this, TYPE_REST, len2 == WHOLE_NOTE && m_voice_nr > 0, 3,
					dotcount, len2, NORMAL_NOTE, 0, 0), pos++);
		}
	}
	if ((m_end_of_last_imported = g_list_nth(m_chord_or_rests, pos - 1)) == NULL) {
		NedResource::Abort("NedVoice::appendElementsOfMeasureLength(5)");
	}
	if (first) {
		m_start_of_last_imported = m_end_of_last_imported;
	}
}

void NedVoice::removeLastImported() {
	GList *lptr;
	GList *dellist = NULL;

	if (m_start_of_last_imported == NULL) {
		printf("staff = %d, voi = %d\n", m_staff->getStaffNumber(), m_voice_nr); fflush(stdout);
		NedResource::Abort("NedVoice::removeLastImported");
	}
	for (lptr = m_start_of_last_imported; lptr; lptr = g_list_next(lptr)) {
		dellist = g_list_append(dellist, lptr);
	}
	for (lptr = g_list_first(dellist); lptr; lptr = g_list_next(lptr)) {
		m_chord_or_rests = g_list_delete_link(m_chord_or_rests, (GList *) lptr->data);
	}
	g_list_free(dellist);
}

bool NedVoice::handleImportedTuplets() {
	GList *lptr;
	unsigned int  duration = 0;
	NedChordOrRest *chord_or_rest, *last_tupletted_chord_or_rest = NULL;
	int tupletval = -1, tupval;

	for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		chord_or_rest = (NedChordOrRest *) lptr->data;
		tupval = chord_or_rest->getTupletVal();
		if (tupval != 0) {
			duration += chord_or_rest->getDuration();
			if (tupval != 0) {
				if (tupletval < 0) {
					tupletval = tupval;
				}
				else if (tupletval != tupval) {
					return false;
				}
				if (NedResource::fittingDuration(duration)) {
					chord_or_rest->setLastTupletFlag(true);
					duration = 0;
					tupletval = -1;
				}
				last_tupletted_chord_or_rest = chord_or_rest;
			}
		}
		else if (tupletval > 0) {
			if (!NedResource::fittingDuration(duration)) {printf("Stelle 1: tupletval = %d, duration = %u NOTE_4 = %u\n", tupletval, duration, NOTE_4); fflush(stdout); return false;}
			last_tupletted_chord_or_rest->setLastTupletFlag(true);
			tupletval = -1;
			duration = 0;
			last_tupletted_chord_or_rest = NULL;
		}
	}
	if (tupletval > 0) {
		if (!NedResource::fittingDuration(duration)) {printf("Stelle 2: duration = %u NOTE_4 = %u\n", duration, NOTE_4); fflush(stdout);return false;}
		last_tupletted_chord_or_rest->setLastTupletFlag(true);
	}
	return true;
}

bool NedVoice::truncateAtStart(NedCommandList *command_list, unsigned long long midi_time) {
	unsigned int duration = 0;
	GList *list_of_elems_to_delete = NULL;
	NedDeleteChordOrRestGroupCommand *del_cmd;
	GList *lptr;
	GList *lptr2;
	NedChordOrRest *element;
	int pos;

	for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		element = (NedChordOrRest *) lptr->data;
		if ((element->getType() & (TYPE_NOTE | TYPE_REST)) == 0) continue;
		if (element->getTupletVal() != 0) return false;
		duration += element->getDuration();
		list_of_elems_to_delete = g_list_append(list_of_elems_to_delete, element);
		if (duration >= midi_time) {
			element->testForTiesToDelete(command_list, BREAK_TIE_FORWARD);
			del_cmd = new NedDeleteChordOrRestGroupCommand(this, list_of_elems_to_delete);
			command_list->addCommand(del_cmd);
			pos = 0;
			if (duration > midi_time) {
				handleGapAfterInsertion(command_list, &pos, duration - midi_time, element, 0, false, false);
			}

			return true;
		}
	}
	return false;
}

void NedVoice::handleStaffElements() {
	GList *lptr;
	bool changed;
	NedChordOrRest *chord_or_rest;

	resetPositionPointer(false);

	do {
		changed = false;
		for (lptr = g_list_first(m_chord_or_rests); !changed && lptr; lptr = g_list_next(lptr)) {
			chord_or_rest = (NedChordOrRest *) lptr->data;
			switch (chord_or_rest->getType()) {
				case TYPE_CLEF:
				case TYPE_KEYSIG: m_staff->insertStaffElement(chord_or_rest, false);
						  m_chord_or_rests = g_list_delete_link(m_chord_or_rests, lptr);
						  changed = true;
						  break;
			}
		}
	}
	while (changed);
}

bool NedVoice::tryInsertOrErease(double x, double y) {
	double topy = getMainWindow()->getTopY();
	double zoom_factor = getMainWindow()->getCurrentZoomFactor();
	GList *lptr, *lptr2;
	NedChordOrRest *chord;
	NedChordOrRest *last_tupleted_chord_or_rest;
	NedTuplet *tuplet_ptr;
	bool removed = FALSE;
	staff_context_str *context;
	int tuplet_val, tuplet_val_raw;
	NedChordOrRest *chord_or_rest;
	unsigned long long end_time;
	unsigned long long new_chords_midi_end_time;

	if (getMainWindow()->getCurrentLength() < NOTE_64) {
		return insertGrace(x, y);
	}

	for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		chord = (NedChordOrRest *) lptr->data;
		if (chord->tryErease(x, y, &removed)) {
			NedCommandList *command_list = new NedCommandList(getMainWindow(), getSystem());
			chord->testForTiesToDelete(command_list, BREAK_TIE_FORWARD | BREAK_TIE_BACKWARD);
			command_list->addCommand(new NedDeleteChordCommand(chord));
			getSystem()->resetUntouched();
			command_list->execute();
			getMainWindow()->getCommandHistory()->addCommandList(command_list);
			return TRUE;
		}
		if (removed) {
			getSystem()->resetUntouched();
			return TRUE;
		}
	}

	double yl = Y_POS_INVERS_STAFF_REL(y);
	if (MAX_OVER_UNDER_LINES * LINE_DIST/2.0 < yl ||  -((MAX_OVER_UNDER_LINES + 8) * LINE_DIST/2.0) > yl) return FALSE;
	double leftx = getMainWindow()->getLeftX();
	double xl = X_POS_INVERS_PAGE_REL(x);
	int line = (int) ((- yl) / (LINE_DIST/2.0));
	double min = 1000000.0;
	double diff;
	unsigned int v, sv;
	int minpos = -1;
	int num_elements_to_delete = 1;
	int clef, keysig_so_far, octave_shift;
	GList *min_pos_ptr = NULL;
	for(lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		diff = ((NedChordOrRest *) lptr->data)->getXPos() - xl;
		if (diff < 0) diff = -diff;
		if (diff < min) {
			min = diff;
			min_pos_ptr = lptr;
			minpos = g_list_position(m_chord_or_rests, lptr);
		}
	}
	if (minpos < 0) {
		return FALSE;
	}
	tuplet_val = ((NedChordOrRest *) min_pos_ptr->data)->getTupletVal();
	tuplet_val_raw = ((NedChordOrRest *) min_pos_ptr->data)->getTupletValRaw();
	if (getMainWindow()->getSpecialType() < 0 && !((NedChordOrRest *) min_pos_ptr->data)->isRest() && !getMainWindow()->getRestMode()) {
		if (((NedChordOrRest *) min_pos_ptr->data)->noConflict(line)) {
			NedCommandList *command_list = new NedCommandList(getMainWindow(), getSystem());
			NedAddNoteAtCommand *add_note_at_command = new NedAddNoteAtCommand(this, ((NedChordOrRest *) min_pos_ptr->data), line, getMainWindow()->getCurrentNoteHead(), getMainWindow()->getStatus());
			command_list->addCommand(add_note_at_command);
			getMainWindow()->getCommandHistory()->addCommandList(command_list);
			getMainWindow()->setSelected((NedChordOrRest *) min_pos_ptr->data, add_note_at_command->getNote());
			getSystem()->resetUntouched();
			command_list->execute();
			return TRUE;
		}
		return FALSE;
	}
/*
	if (((NedChordOrRest *) min_pos_ptr->data)->isRest() && (((NedChordOrRest *) min_pos_ptr->data)->isHidden())) {
		return FALSE;
	}
*/
	NedCommandList *command_list = new NedCommandList(getMainWindow(), getSystem());
	if (getMainWindow()->getSpecialType() >= 0) {
		switch (getMainWindow()->getSpecialType() & 0xffff) {
			case TYPE_CLEF:
				v = getMainWindow()->getSpecialSubType();
				sv = (v >> 16) & 0xffff;
				sv -= 100;
				chord_or_rest = new NedChordOrRest(this, getMainWindow()->getSpecialType(), v & 0xffff, 
					sv, ((NedChordOrRest *) min_pos_ptr->data)->getMidiTime(), true);
				break;
			case TYPE_KEYSIG:
				getStaff()->getCurrentClefAndKeysig(((NedChordOrRest *) min_pos_ptr->data)->getMidiTime(), &clef, &keysig_so_far, &octave_shift);
				if (keysig_so_far == getMainWindow()->getSpecialSubType()) {
					getMainWindow()->resetSpecialType();
					return TRUE;
				}
				chord_or_rest = new NedChordOrRest(this, getMainWindow()->getSpecialType(), getMainWindow()->getSpecialSubType(), 
						keysig_so_far, ((NedChordOrRest *) min_pos_ptr->data)->getMidiTime());
				break;
		}
		getMainWindow()->resetSpecialType();
		NedInsertStaffElemCommand *ins_staff_elem_cmd = new NedInsertStaffElemCommand(chord_or_rest, m_staff);
		ins_staff_elem_cmd->execute();
		command_list->addCommand(ins_staff_elem_cmd);
		getMainWindow()->getCommandHistory()->addCommandList(command_list);
		getSystem()->resetUntouched();
		return TRUE;
	}
	else {
		chord_or_rest = new NedChordOrRest(this, getMainWindow()->getRestMode() ? TYPE_REST : TYPE_NOTE, FALSE, line,
		getMainWindow()->getDotCount(), getMainWindow()->getCurrentLength(), getMainWindow()->getCurrentNoteHead(), getMainWindow()->getStatus(), 0);
		chord_or_rest->setTupletVal(tuplet_val);
	}
	if (!chord_or_rest->isRest()) {
		getMainWindow()->setSelected(chord_or_rest, chord_or_rest->getFirstNote());
	}
	unsigned long long min_pos_midi_time = ((NedChordOrRest *) min_pos_ptr->data)->getMidiTime();
	if ((tuplet_ptr = ((NedChordOrRest *) min_pos_ptr->data)->getTupletPtr()) != NULL)  {
		end_time = tuplet_ptr->getEndTime();
	}
	else {
		NedMeasure *measure = getSystem()->getMeasure(((NedChordOrRest *) min_pos_ptr->data)->getMidiTime());
		end_time = measure->midi_end;
		testForTupletEnd(min_pos_midi_time, chord_or_rest->getDuration(), &end_time);
	}
	new_chords_midi_end_time = min_pos_midi_time + chord_or_rest->getDuration();
	if (end_time > new_chords_midi_end_time) end_time = new_chords_midi_end_time;
	unsigned long long sum_of_chords_and_rests_to_delete = 0;
	GList *chords_and_rests_to_delete = NULL;
	for (lptr2 = min_pos_ptr; lptr2 && ((NedChordOrRest *) lptr2->data)->getMidiTime() < end_time;
		lptr2 = g_list_next(lptr2)) {
		chords_and_rests_to_delete = g_list_append(chords_and_rests_to_delete, lptr2->data);
		((NedChordOrRest *) lptr2->data)->testForTiesToDelete(command_list);
		sum_of_chords_and_rests_to_delete += ((NedChordOrRest *) lptr2->data)->getDuration();
	}
	if (sum_of_chords_and_rests_to_delete < chord_or_rest->getDuration()) {
		chord_or_rest->changeDuration(sum_of_chords_and_rests_to_delete, tuplet_val);	
	}
	if (tuplet_ptr != NULL) {
		if (tuplet_ptr->getMidiStartTime() == min_pos_midi_time && chord_or_rest->getDuration() == tuplet_ptr->getDuration()) {
			delete command_list;
			return FALSE;
		}
	}
	sum_of_chords_and_rests_to_delete -= chord_or_rest->getDuration();
	if (chords_and_rests_to_delete != NULL) {
		command_list->addCommand(new NedDeleteChordOrRestGroupCommand(this, chords_and_rests_to_delete));
		num_elements_to_delete = g_list_length(chords_and_rests_to_delete);
	}
	command_list->addCommand(new NedInsertChordOrRestCommand(this, minpos++, chord_or_rest));
	last_tupleted_chord_or_rest = chord_or_rest;
	if  (sum_of_chords_and_rests_to_delete > 0) {
		NedChordOrRest *chord_or_rest_to_distribute = (NedChordOrRest *) (g_list_last(chords_and_rests_to_delete)->data);
		last_tupleted_chord_or_rest = handleGapAfterInsertion(command_list, &minpos, sum_of_chords_and_rests_to_delete, chord_or_rest_to_distribute, tuplet_val, false, false);
	}
	if (tuplet_val_raw & LAST_TUPLET_FLAG) {
		last_tupleted_chord_or_rest->setLastTupletFlag(TRUE);
	}
	getMainWindow()->getCommandHistory()->addCommandList(command_list);
	getSystem()->resetUntouched();
	command_list->execute();
	getMainWindow()->reposit(command_list, getPage(), getSystem());
	getSystem()->resetUntouched();
	context = &(getMainWindow()->m_staff_contexts[getStaff()->getStaffNumber()]);
	if (!chord_or_rest->isRest() && NedResource::m_midi_echo) {
		NedResource::playImmediately(context->m_midi_channel, context->m_midi_program, chord_or_rest->getPitchOfFirstNote(), context->m_midi_volume);
	}
	return TRUE;
}

bool NedVoice::insertGrace(double x, double y) {
	GList *lptr;
	NedChordOrRest *chord_or_rest;

	double topy = getMainWindow()->getTopY();
	double zoom_factor = getMainWindow()->getCurrentZoomFactor();
	double yl = Y_POS_INVERS_STAFF_REL(y);
	if (MAX_OVER_UNDER_LINES * LINE_DIST/2.0 < yl ||  -((MAX_OVER_UNDER_LINES + 8) * LINE_DIST/2.0) > yl) return FALSE;
	double leftx = getMainWindow()->getLeftX();
	double xl = X_POS_INVERS_PAGE_REL(x);
	int line = (int) ((- yl) / (LINE_DIST/2.0));
	double min = 1000000.0;
	double diff;
	int minpos = -1;
	GList *min_pos_ptr = NULL;
	for(lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		diff = ((NedChordOrRest *) lptr->data)->getXPos() - xl;
		if (diff < 0) diff = -diff;
		if (diff < min) {
			min = diff;
			min_pos_ptr = lptr;
			minpos = g_list_position(m_chord_or_rests, lptr);
		}
	}
	if (minpos < 0) {
		return FALSE;
	}

	chord_or_rest = new NedChordOrRest(this, TYPE_NOTE, FALSE, line,
		0, getMainWindow()->getCurrentLength(), NORMAL_NOTE, 0, 0);
	NedCommandList *command_list = new NedCommandList(getMainWindow(), getSystem());
	command_list->addCommand(new NedInsertChordOrRestCommand(this, minpos, chord_or_rest));
	getMainWindow()->getCommandHistory()->addCommandList(command_list);
	getSystem()->resetUntouched();
	command_list->execute();
	getMainWindow()->reposit(command_list, getPage(), getSystem());
	getSystem()->resetUntouched();
	return TRUE;
}

void NedVoice::collectSelectionRectangleElements(double xp, double yp, NedBbox *sel_rect, GList **sel_group,
		bool is_first_selected, bool is_last_selected) {
	GList *lptr;
	for(lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		if (is_first_selected && ((NedChordOrRest *) lptr->data)->getXPos() < xp) continue;
		if (is_last_selected && ((NedChordOrRest *) lptr->data)->getXPos() > xp + sel_rect->width) continue;
		*sel_group = g_list_append(*sel_group, lptr->data);
	}
}

bool NedVoice::hasTupletConflict(unsigned int meas_duration, GList **elements, int from_staff, unsigned int long long midi_time) {
	GList *lptr, *lptr2;
	GList *tested_elements = NULL;
	NedChordOrRest *element;
	NedTuplet *tuplet;
	unsigned long long current_midi_time;

	for (lptr2 = g_list_first(m_chord_or_rests); lptr2; lptr2 = g_list_next(lptr2)) {
		if (((NedChordOrRest *) lptr2->data)->getMidiTime() >= midi_time) {
			break;
		}
	}
	if (lptr2 == NULL) return FALSE;
	current_midi_time = ((NedChordOrRest *) lptr2->data)->getMidiTime();
	for (lptr = g_list_first(*elements); lptr; lptr = g_list_next(lptr)) {
		element = (NedChordOrRest *) lptr->data;
		if (element->getStaff()->getStaffNumber() != from_staff) continue;
		if (element->getVoice()->m_voice_nr != m_voice_nr) continue;
		tested_elements = g_list_append(tested_elements, element);
		current_midi_time += element->getDuration();
		if (element->getTupletVal() != 0) {
			if ((tuplet = element->getTupletPtr()) == NULL) {
				NedResource::Abort("NedVoice::hasTupletConflict(1)");
			}
			if (element->hasLastTupletFlag()) {
				/* -1 is very tricky but if it ends exactly at next measure it still belongs to this measure */
				if ((current_midi_time - tuplet->getDuration()) / meas_duration != (current_midi_time - 1) / meas_duration) {
					/*
					printf("current_midi_time = %llu, tuplet->getDuration() = %u, diff = %llu, meas_duration = %u, current_midi_time = %llu, d1 = %llu, f2 = %llu\n",
					current_midi_time, tuplet->getDuration(), current_midi_time - tuplet->getDuration(), meas_duration, current_midi_time,
					(current_midi_time - tuplet->getDuration()) / meas_duration,  current_midi_time / meas_duration);
					*/
					return TRUE;
				}
			}
		}
	}
	for (lptr = g_list_first(tested_elements); lptr; lptr = g_list_next(lptr)) {
		if ((lptr2 = g_list_find(*elements, lptr->data)) == NULL) {
			NedResource::Abort("NedVoice::hasTupletConflict(2)");
		}
		*elements = g_list_delete_link(*elements, lptr2);
	}
	g_list_free(tested_elements);
	return FALSE;
}

void NedVoice::pasteElements(NedCommandList *command_list, GList **elements, int from_staff, unsigned int long long midi_time) {
	GList *lptr, *lptr2;
	GList *copied_elements = NULL;
	GList *chordsorrests_to_delete = NULL;
	GList *chordsorrests_to_insert = NULL;
	NedChordOrRest *element;
	int idx, diff, pos;
	unsigned long long duration = 0, deleted_duration;
	unsigned long long current_midi_time;
	unsigned int last_duration = 0, first_duration = -1;

	for (lptr = g_list_first(*elements); lptr; lptr = g_list_next(lptr)) {
		element = (NedChordOrRest *) lptr->data;
		if (element->getStaff()->getStaffNumber() != from_staff) continue;
		if (element->getVoice()->m_voice_nr != m_voice_nr) continue;
		last_duration = element->getDuration();
		if (first_duration < 0) {
			first_duration =  element->getDuration();
		}
		duration += last_duration;
	}
	do {
		for (lptr2 = g_list_first(m_chord_or_rests); lptr2; lptr2 = g_list_next(lptr2)) {
			if (((NedChordOrRest *) lptr2->data)->getMidiTime() >= midi_time) {
				break;
			}
		}
		if (lptr2 == NULL || duration == 0) return;
		if ((idx = g_list_position(m_chord_or_rests, lptr2)) < 0) {
			NedResource::Abort("NedVoice::pasteElements(1)");
		}
		element = (NedChordOrRest *) lptr2->data;
	}
	while (searchForBreakableTuplet(command_list, element->getMidiTime()));
	searchForBreakableTies(command_list, BREAK_TIE_FORWARD | BREAK_TIE_BACKWARD, element->getMidiTime(), element->getMidiTime() + element->getDuration());
	current_midi_time = ((NedChordOrRest *) lptr2->data)->getMidiTime();
	searchForBreakableTuplet(command_list, current_midi_time + duration);
	searchForBreakableTies(command_list, BREAK_TIE_FORWARD | BREAK_TIE_BACKWARD, current_midi_time + duration - last_duration, current_midi_time + duration);
	resetPositionPointer(false); // recompute midi times
	for (lptr2 = g_list_first(m_chord_or_rests); lptr2; lptr2 = g_list_next(lptr2)) { // test again with removed tuplets
		if (((NedChordOrRest *) lptr2->data)->getMidiTime() >= midi_time) {
			if (((NedChordOrRest *) lptr2->data)->getMidiTime() > midi_time) {
				if ((lptr2 = g_list_previous(lptr2)) == NULL) {
					NedResource::Abort("NedVoice::pasteElements(2)");
				}
				if ((pos = g_list_position(m_chord_or_rests, lptr2)) < 0) {
					NedResource::Abort("NedVoice::pasteElements(3)");
				}
				NedChordOrRest *chord_or_rest = ((NedChordOrRest *) lptr2->data);
				int duration = chord_or_rest->getDuration();
				unsigned long long start_midi_time = chord_or_rest->getMidiTime();
				GList *list = NULL;
				list = g_list_append(list, chord_or_rest);
				NedDeleteChordOrRestGroupCommand *delete_chord_command = new NedDeleteChordOrRestGroupCommand(this, list);
				delete_chord_command->execute();
				int dur1 = midi_time - start_midi_time;
				command_list->addCommand(delete_chord_command);
				handleGapAfterInsertion(command_list, &pos, dur1, chord_or_rest, 0 , false, true);
				int dur2 = duration - dur1;
				handleGapAfterInsertion(command_list, &pos, dur2, NULL, 0, false, true);
				command_list->setBreak();
				idx = pos;
				if ((lptr2 = g_list_nth(m_chord_or_rests, idx)) == NULL) {
					NedResource::Abort("NedVoice::pasteElements(4)");
				}
				resetPositionPointer(false); // recompute midi times
				current_midi_time = ((NedChordOrRest *) lptr2->data)->getMidiTime();
			}
			break;
		}
	}
	deleted_duration = 0;
	while (deleted_duration < duration && lptr2) {
		deleted_duration += ((NedChordOrRest *) lptr2->data)->getDuration();
		chordsorrests_to_delete = g_list_append(chordsorrests_to_delete, lptr2->data);
		lptr2 = g_list_next(lptr2);
	}
	diff = deleted_duration - duration;
	for (lptr = g_list_first(*elements); deleted_duration > 0 && lptr; lptr = g_list_next(lptr)) {
		element = (NedChordOrRest *) lptr->data;
		if (element->getStaff()->getStaffNumber() != from_staff) continue;
		if (element->getVoice()->m_voice_nr != m_voice_nr) continue;
		unsigned long long midi_end_time = current_midi_time + element->getDuration();
		NedMeasure *measure = getSystem()->getMeasure(current_midi_time);
		if (current_midi_time < measure->midi_end && midi_end_time > measure->midi_end) {
			unsigned long long t1, t2;
			t1 = measure->midi_end - current_midi_time; t2 = midi_end_time - measure->midi_end;
			NedResource::split_element(elements, lptr, t1, t2);
			current_midi_time += t1;
		}
		else {
			current_midi_time = midi_end_time;
		}
		deleted_duration -= element->getDuration();
		chordsorrests_to_insert = g_list_append(chordsorrests_to_insert, element);
		copied_elements = g_list_append(copied_elements, element);
	}

	NedDeleteChordOrRestGroupCommand *delete_command = new NedDeleteChordOrRestGroupCommand(this, chordsorrests_to_delete);
	delete_command->execute();
	command_list->addCommand(delete_command);
	NedInsertChordOrRestGroupCommand *insert_command = new NedInsertChordOrRestGroupCommand(this, chordsorrests_to_insert, idx);
	insert_command->execute();
	command_list->addCommandAndSetBreak(insert_command);
	if (diff > 0) {
		NedChordOrRest *chord_or_rest_to_distribute = (NedChordOrRest *) g_list_last(chordsorrests_to_delete)->data;
		if ((pos = g_list_index(m_chord_or_rests, g_list_last(chordsorrests_to_insert)->data)) < 0) {
			NedResource::Abort("NedVoice::pasteElements(8)");
		}
		pos++;
		handleGapAfterInsertion(command_list, &pos, diff, chord_or_rest_to_distribute, 0, false, true);
	}

	for (lptr = g_list_first(copied_elements); lptr; lptr = g_list_next(lptr)) {
		if ((lptr2 = g_list_find(*elements, lptr->data)) == NULL) {
			NedResource::Abort("NedVoice::pasteElements(9)");
		}
		*elements = g_list_delete_link(*elements, lptr2);
	}
	g_list_free(copied_elements);
}

bool NedVoice::findFromTo(GList *clipboard) {
	GList *lptr;
	NedSystem *the_system = getSystem();
	NedPage *the_page = getPage();

	for (lptr = g_list_first(clipboard); lptr; lptr = g_list_next(lptr)) {
		if (((NedChordOrRest *) lptr->data)->getPage() != the_page) continue;
		if (((NedChordOrRest *) lptr->data)->getSystem() != the_system) continue;
		if (((NedChordOrRest *) lptr->data)->getStaff() != m_staff) continue;
		if (((NedChordOrRest *) lptr->data)->getVoice() != this) continue;
		return TRUE;
	}
	return FALSE;
}

bool NedVoice::findStartMeasureLimits(GList *clipboard, unsigned long long *start_midi) {
	GList *start_ptr;
	NedChordOrRest *element;
	NedSystem *the_system = getSystem();
	NedPage *the_page = getPage();
	NedMeasure *measure;

	for (start_ptr = g_list_first(clipboard); start_ptr; start_ptr = g_list_next(start_ptr)) {
		element = (NedChordOrRest *) start_ptr->data;
		if (element->isRest() && element->isHidden()) continue;
		if (element->getPage() != the_page) continue;
		if (element->getSystem() != the_system) continue;
		if (element->getStaff() != m_staff) continue;
		if (element->getVoice() != this) continue;
		measure = getSystem()->getMeasure(element->getMidiTime());
		if (element->getMidiTime() != measure->midi_start) continue;
		if (*start_midi > element->getMidiTime()) {
			*start_midi = element->getMidiTime();
			return TRUE;
		}
	}
	return FALSE;
}

bool NedVoice::findEndMeasureLimits(GList *clipboard, unsigned long long *end_midi) {
	GList *end_ptr = NULL;
	NedChordOrRest *element;
	NedSystem *the_system = getSystem();
	NedPage *the_page = getPage();
	NedMeasure *measure;

	for (end_ptr = g_list_last(clipboard); end_ptr; end_ptr = g_list_previous(end_ptr)) {
		element = (NedChordOrRest *) end_ptr->data;
		if (element->isRest() && element->isHidden()) continue;
		if (element->getPage() != the_page) continue;
		if (element->getSystem() != the_system) continue;
		if (element->getStaff() != m_staff) continue;
		if (element->getVoice() != this) continue;
		measure = getSystem()->getMeasure(element->getMidiTime());
		if (element->getMidiTime() + element->getDuration() != measure->midi_end) continue;
		if (*end_midi < measure->midi_end) {
			*end_midi = measure->midi_end;
			return TRUE;
		}
	}
	return FALSE;
}

void NedVoice::deleteItemsFromTo(NedCommandList *command_list, bool is_first, bool is_last, unsigned long long start_midi, unsigned long long end_midi) {
	GList *elements_to_delete = NULL;
	GList *start_ptr, *end_ptr = NULL;
	NedChordOrRest *element;

	start_ptr = g_list_first(m_chord_or_rests);

	if (is_first) {
		for (; start_ptr; start_ptr = g_list_next(start_ptr)) {
			if (((NedChordOrRest *) start_ptr->data)->getMidiTime() < start_midi) continue;
			break;
		}
	}
	if (start_ptr == NULL) return;
	if (is_last) {
		for (end_ptr = g_list_last(m_chord_or_rests); end_ptr; end_ptr = g_list_previous(end_ptr)) {
			element = (NedChordOrRest *) end_ptr->data;
			if (element->getMidiTime() + element->getDuration() > end_midi) continue;
			break;
		}
	}
	if (end_ptr == NULL) return;
	for (; start_ptr; start_ptr = g_list_next(start_ptr)) {
		elements_to_delete = g_list_append(elements_to_delete, start_ptr->data);
		if (start_ptr == end_ptr) break;
	}
	if (g_list_length(elements_to_delete) > 0) {
		((NedChordOrRest *) g_list_first(elements_to_delete)->data)->testForTiesToDelete(command_list, BREAK_TIE_BACKWARD, true);
		((NedChordOrRest *) g_list_last(elements_to_delete)->data)->testForTiesToDelete(command_list, BREAK_TIE_FORWARD, true);
		NedDeleteChordOrRestGroupCommand *delete_cmd = new NedDeleteChordOrRestGroupCommand(this, elements_to_delete);
		delete_cmd->execute();
		command_list->addCommand(delete_cmd);
	}
}

void NedVoice::removeNotesFromTo(NedCommandList *command_list, GList *items, bool is_first, bool is_last) {
	GList *elements_to_delete = NULL;
	GList *all_elements_to_delete = NULL;
	GList *start_ptr, *end_ptr = NULL;
	NedChordOrRest *element;
	NedSystem *the_system = getSystem();
	NedPage *the_page = getPage();
	int pos;
	unsigned int duration;
	NedDeleteChordOrRestGroupCommand *command;
	NedMeasure *measure;
	bool only_rests = true;
	bool measure_start_seen = false;

	start_ptr = g_list_first(items);

	if (is_first) {
		for (start_ptr = g_list_first(items); start_ptr; start_ptr = g_list_next(start_ptr)) {
			element = (NedChordOrRest *) start_ptr->data;
			if (element->isRest() && element->isHidden()) continue;
			if (element->getPage() != the_page) continue;
			if (element->getSystem() != the_system) continue;
			if (element->getStaff() != m_staff) continue;
			if (element->getVoice() != this) continue;
			break;
		}
	}
	if (start_ptr == NULL) return;
	if (is_last) {
		for (end_ptr = g_list_last(items); end_ptr; end_ptr = g_list_previous(end_ptr)) {
			element = (NedChordOrRest *) end_ptr->data;
			if (element->isRest() && element->isHidden()) continue;
			if (element->getPage() != the_page) continue;
			if (element->getSystem() != the_system) continue;
			if (element->getStaff() != m_staff) continue;
			if (element->getVoice() != this) continue;
			break;
		}
		if (end_ptr == NULL) return;
	}
	duration = 0;
	/*
	measure = getSystem()->getMeasure(((NedChordOrRest *) start_ptr->data)->getMidiTime());
	if (((NedChordOrRest *) start_ptr->data)->getMidiTime() == measure->midi_start) {
		measure_start_seen = true;
	}
	*/
	for (; start_ptr; start_ptr = g_list_next(start_ptr)) {
		element = (NedChordOrRest *) start_ptr->data;
		if (element->isRest() && element->isHidden()) continue;
		if (element->getPage() != the_page) continue;
		if (element->getSystem() != the_system) continue;
		if (element->getStaff() != m_staff) continue;
		if (element->getVoice() != this) continue;
		if (!element->isRest())  only_rests = false;
		measure = getSystem()->getMeasure(element->getMidiTime());
		if (g_list_length(elements_to_delete) > 0 && element->getMidiTime() == measure->midi_start && g_list_length(elements_to_delete) > 0) {
			if ((pos = g_list_index(m_chord_or_rests, g_list_first(elements_to_delete)->data)) < 0) {
				NedResource::Abort("NedVoice::removeNotesFromTo(1)");
			}
			command = new NedDeleteChordOrRestGroupCommand(this, elements_to_delete);
			command->execute();
			command_list->addCommand(command);
			handleGapAfterInsertion(command_list, &pos, duration, NULL, 0, measure_start_seen, true);
			measure_start_seen = true;
			elements_to_delete = NULL;
			duration = 0;
		}
		duration += element->getDuration();
		elements_to_delete = g_list_append(elements_to_delete, element);
		all_elements_to_delete = g_list_append(all_elements_to_delete, element);
		if (start_ptr == end_ptr) break;
	}
	if (g_list_length(elements_to_delete) > 0) {
		if ((pos = g_list_index(m_chord_or_rests, g_list_first(elements_to_delete)->data)) < 0) {
			NedResource::Abort("NedVoice::removeNotesFromTo(2)");
		}
		element = (NedChordOrRest *) g_list_last(elements_to_delete)->data;
		if (measure_start_seen) {
			measure_start_seen = ((element->getMidiTime() + element->getDuration()) == measure->midi_end);
		}
		command = new NedDeleteChordOrRestGroupCommand(this, elements_to_delete);
		command->execute();
		command_list->addCommand(command);
		handleGapAfterInsertion(command_list, &pos, duration, NULL, 0, measure_start_seen /* ??? false*/, true);
	}
	if (g_list_length(elements_to_delete) > 0) {
		((NedChordOrRest *) g_list_first(all_elements_to_delete)->data)->testForTiesToDelete(command_list, BREAK_TIE_BACKWARD, true);
		((NedChordOrRest *) g_list_last(all_elements_to_delete)->data)->testForTiesToDelete(command_list, BREAK_TIE_FORWARD, true);
	}
	g_list_free(all_elements_to_delete);
}

void NedVoice::insertBlocks(NedCommandList *command_list, int blockcount, unsigned long long midi_time) {
	int i;
	GList *lptr;
	GList *items_to_insert = NULL;
	int pos;

	for (lptr = g_list_first(m_chord_or_rests); lptr && ((NedChordOrRest *) lptr->data)->getMidiTime() < midi_time; lptr = g_list_next(lptr));
	if (lptr) {
		if ((pos = g_list_position(m_chord_or_rests, lptr)) < 0) {
			NedResource::Abort("NedVoice::insertBlocks");
		}
		((NedChordOrRest *) lptr->data)->testForTiesToDelete(command_list, BREAK_TIE_BACKWARD, true);
	}
	else {
		((NedChordOrRest *) g_list_last(m_chord_or_rests)->data)->testForTiesToDelete(command_list, BREAK_TIE_BACKWARD, true);
		pos = g_list_length(m_chord_or_rests);
	}

	for (i = 0; i < blockcount; i++) {
		items_to_insert = g_list_append(items_to_insert, new NedChordOrRest(this, TYPE_REST, m_voice_nr != 0, 3 /*dummy*/, 0, WHOLE_NOTE, NORMAL_NOTE, 0, 0 /* dummy */));
	}
	NedInsertChordOrRestGroupCommand *command = new NedInsertChordOrRestGroupCommand(this, items_to_insert, pos);
	command->execute();
	command_list->addCommand(command);
}

void NedVoice::testForPageBackwardTies(NedCommandList *command_list) {
	((NedChordOrRest *) g_list_first(m_chord_or_rests)->data)->testForTiesToDelete(command_list, BREAK_TIE_BACKWARD);
}


bool NedVoice::tryChangeLength(NedChordOrRest *chord_or_rest) {
	GList *lptr, *lptr2;
	NedChordOrRest *chord_or_rest_to_distribute;
	NedChordOrRest *last_tupleted_chord_or_rest;
	unsigned long long new_duration;
	unsigned long long change_chords_midi_end_time;
	unsigned long long end_time;
	bool was_last_in_tuplet;
	NedTuplet *tuplet_ptr;
	int tuplet_val, tuplet_val_raw;
	int new_dot_count = getMainWindow()->getDotCount();
	lptr = g_list_find(m_chord_or_rests, chord_or_rest);
	if (lptr == NULL) {
		return FALSE;
	}
	tuplet_val = chord_or_rest->getTupletVal();
	tuplet_val_raw = chord_or_rest->getTupletValRaw();
	was_last_in_tuplet = chord_or_rest->hasLastTupletFlag();
	chord_or_rest->setLastTupletFlag(FALSE);
	new_duration = NedChordOrRest::computeDuration(getMainWindow()->getCurrentLength(), getMainWindow()->getDotCount(), chord_or_rest->getTupletVal());
	if (chord_or_rest->getDuration() == new_duration) return TRUE;
	int pos = g_list_position(m_chord_or_rests, lptr);
	unsigned long long change_pos_midi_time = ((NedChordOrRest *) lptr->data)->getMidiTime();
	if ((tuplet_ptr = chord_or_rest->getTupletPtr()) != NULL)  {
		end_time = tuplet_ptr->getEndTime();
		change_chords_midi_end_time = change_pos_midi_time + new_duration;
	}
	else {
		NedMeasure *measure = getSystem()->getMeasure(((NedChordOrRest *) lptr->data)->getMidiTime());
		change_chords_midi_end_time = change_pos_midi_time + new_duration;
		end_time = measure->midi_end;
		if (getMainWindow()->getCurrentLength() == WHOLE_NOTE && chord_or_rest->getMidiTime() == change_pos_midi_time) {
			new_duration = getMainWindow()->getMeasureDuration(measure->getMeasureNumber());
		}
	}
	change_chords_midi_end_time = change_pos_midi_time + new_duration;
	if (end_time > change_chords_midi_end_time) end_time = change_chords_midi_end_time;
	if (tuplet_ptr != NULL && tuplet_ptr->getMidiStartTime() == change_pos_midi_time &&  change_chords_midi_end_time - change_pos_midi_time == tuplet_ptr->getDuration()) {
		return FALSE;
	}
	unsigned int sum_of_chords_and_rests_to_delete = 0;
	GList *chords_and_rests_to_delete = NULL;
	NedCommandList *command_list = new NedCommandList(getMainWindow(), getSystem());
	if (tuplet_ptr == NULL) {
		searchForBreakableTuplet(command_list, end_time);
	}
	for (lptr2 = g_list_next(lptr); lptr2 && ((NedChordOrRest *) lptr2->data)->getMidiTime() < end_time;
		lptr2 = g_list_next(lptr2)) {
		chords_and_rests_to_delete = g_list_append(chords_and_rests_to_delete, lptr2->data);
		((NedChordOrRest *) lptr2->data)->testForTiesToDelete(command_list);
		sum_of_chords_and_rests_to_delete += ((NedChordOrRest *) lptr2->data)->getDuration();
	}
	if (chords_and_rests_to_delete != NULL) {
		command_list->addCommand(new NedDeleteChordOrRestGroupCommand(this, chords_and_rests_to_delete));
	}
	unsigned int length = getMainWindow()->getCurrentLength();
	if (new_duration > chord_or_rest->getDuration()) {
		if (new_duration - chord_or_rest->getDuration() > sum_of_chords_and_rests_to_delete) {
			if (tuplet_val > 0) {
				unsigned int d = sum_of_chords_and_rests_to_delete + chord_or_rest->getDuration();
				d *= tuplet_val;
				d /= NedResource::m_tuplet_tab[tuplet_val];
				NedChordOrRest::compute_fitting_duration(d, &length, &new_dot_count);
			}
			else {
				NedChordOrRest::compute_fitting_duration(sum_of_chords_and_rests_to_delete + chord_or_rest->getDuration(), &length, &new_dot_count);
			}
			new_duration = NedChordOrRest::computeDuration(length, new_dot_count, chord_or_rest->getTupletVal());
		}
	}
	else {
		sum_of_chords_and_rests_to_delete = chord_or_rest->getDuration() - new_duration;
	}
	if (new_duration > chord_or_rest->getDuration()) {
		sum_of_chords_and_rests_to_delete -= new_duration - chord_or_rest->getDuration();
	}

	last_tupleted_chord_or_rest = chord_or_rest;
	command_list->addCommand(new NedChangeChordOrRestLengthCommand(chord_or_rest, length, new_dot_count));
	if (chords_and_rests_to_delete == NULL) {
		chord_or_rest_to_distribute = NULL;
	}
	else {
		chord_or_rest_to_distribute = (NedChordOrRest *) (g_list_last(chords_and_rests_to_delete)->data);
	}
	if (sum_of_chords_and_rests_to_delete > 0) {
		pos++;
		handleGapAfterInsertion(command_list, &pos, sum_of_chords_and_rests_to_delete, chord_or_rest_to_distribute, tuplet_val, false, false);
	}
	if (was_last_in_tuplet) {
		last_tupleted_chord_or_rest->setLastTupletFlag(TRUE);
	}
	getMainWindow()->getCommandHistory()->addCommandList(command_list);
	command_list->executeAfterBreak(); // because of searchForBreakableTuplet; don't repeat actions before break
	getMainWindow()->reposit(command_list, getPage(), getSystem());
	return TRUE;
}


bool NedVoice::tryConvertToTuplet(int method, int tuplet_val, NedChordOrRest *templ) {
	GList *lptr2;
	NedChordOrRest *chord_or_rest;
	NedChordOrRest *last_tupleted_chord_or_rest;
	int minpos;
	GList *min_pos_ptr;
	int num_elements_to_delete;
	unsigned long long new_chords_midi_end_time;
	unsigned long long  ref_duration;
	int num_additional_rests;


	if (templ->getTupletVal() != 0 || templ->getLength() < NOTE_32) {
		return FALSE;
	}
	if (method == 1 && ((templ->getDuration() % 2) != 0) || tuplet_val < 3) {
		return FALSE;
	}
	ref_duration = (method == 1) ? templ->getDuration() / 2 : templ->getDuration();
	if ((min_pos_ptr = g_list_find(m_chord_or_rests, templ)) < 0) {
		NedResource::Abort("NedVoice::tryConvertToTuplet(1)");
	}
	if ((minpos = g_list_index(m_chord_or_rests, templ)) < 0) {
		NedResource::Abort("NedVoice::tryConvertToTuplet(2)");
	}
	if (((NedChordOrRest *) min_pos_ptr->data)->getTupletVal() != 0) {
		return false;
	}
	minpos++;
	NedMeasure *measure = getSystem()->getMeasure(templ->getMidiTime());
	unsigned long long min_pos_midi_time = templ->getMidiTime();
	new_chords_midi_end_time = min_pos_midi_time + NedResource::m_tuplet_tab[tuplet_val] * ref_duration;
	unsigned int end_time = measure->midi_end;
	if (new_chords_midi_end_time > end_time) {
		return FALSE;
	}
	unsigned int sum_of_chords_and_rests_to_delete = 0;
	GList *chords_and_rests_to_delete = NULL;
	NedCommandList *command_list = new NedCommandList(getMainWindow(), getSystem());
	for (lptr2 = g_list_next(min_pos_ptr); lptr2 && ((NedChordOrRest *) lptr2->data)->getMidiTime() < new_chords_midi_end_time;
		lptr2 = g_list_next(lptr2)) {
		chords_and_rests_to_delete = g_list_append(chords_and_rests_to_delete, lptr2->data);
		((NedChordOrRest *) lptr2->data)->testForTiesToDelete(command_list);
		sum_of_chords_and_rests_to_delete += ((NedChordOrRest *) lptr2->data)->getDuration();
	}
	/*
	if (sum_of_chords_and_rests_to_delete <  templ->getDuration()) {
		return FALSE;
	}
	*/
	if ((NedResource::m_tuplet_tab[tuplet_val] - 1) * ref_duration <= sum_of_chords_and_rests_to_delete) {
		sum_of_chords_and_rests_to_delete -= (NedResource::m_tuplet_tab[tuplet_val] - 1) * ref_duration;
	}
	if (chords_and_rests_to_delete != NULL) {
		command_list->addCommand(new NedDeleteChordOrRestGroupCommand(this, chords_and_rests_to_delete));
		num_elements_to_delete = g_list_length(chords_and_rests_to_delete);
	}
	command_list->addCommand(new NedSetToTupletCommand(templ, tuplet_val));
	num_additional_rests = (method == 1) ? tuplet_val - 2 : tuplet_val - 1;
	for (int i = 0; i < num_additional_rests; i++) {
		chord_or_rest = new NedChordOrRest(this, TYPE_REST, FALSE, 0, 0, ref_duration, NORMAL_NOTE, 0, 0);
		chord_or_rest->setTupletVal(tuplet_val);
		command_list->addCommand(new NedInsertChordOrRestCommand(this, minpos++, chord_or_rest));
	}
	last_tupleted_chord_or_rest = chord_or_rest;
	if (sum_of_chords_and_rests_to_delete > 0) {
		NedChordOrRest *chord_or_rest_to_distribute = (NedChordOrRest *) (g_list_last(chords_and_rests_to_delete)->data);
		handleGapAfterInsertion(command_list, &minpos, sum_of_chords_and_rests_to_delete, chord_or_rest_to_distribute, 0, false, false); 
	}

	last_tupleted_chord_or_rest->setLastTupletFlag(TRUE);
	getMainWindow()->getCommandHistory()->addCommandList(command_list);
	command_list->execute();
	getMainWindow()->reposit(command_list, getPage(), getSystem());
	getSystem()->resetUntouched();
	return TRUE;
}

void NedVoice::removeTuplet(NedChordOrRest *element, NedCommandList *cmd_list) {
	NedTuplet *tuplet = element->getTupletPtr();
	NedCommandList *command_list;
	NedDeleteChordOrRestGroupCommand *del_chord;
	unsigned int tuplet_time;
	int pos;

	if (tuplet == NULL) {
		NedResource::Abort("NedVoice::removeTuplet(1)");
	}
	tuplet_time = tuplet->getDuration();
	pos = g_list_index(m_chord_or_rests, tuplet->getFirstElement());
	if (pos < 0) {
		NedResource::Abort("NedVoice::removeTuplet(2)");
	}
	if (cmd_list == NULL) {
		command_list = new NedCommandList(getMainWindow(), getSystem());
	}
	else {
		command_list = cmd_list;
	}
	tuplet->getFirstElement()->testForTiesToDelete(command_list, BREAK_TIE_BACKWARD, cmd_list != NULL);
	tuplet->getLastElement()->testForTiesToDelete(command_list, BREAK_TIE_FORWARD, cmd_list != NULL);
	command_list->addCommand(del_chord = new NedDeleteChordOrRestGroupCommand(this, tuplet->getElementList()));
	if (cmd_list != NULL) {
		del_chord->execute();
	}
	if (tuplet_time > 0) {
		handleGapAfterInsertion(command_list, &pos, tuplet_time, NULL, 0, false, cmd_list != NULL);
	}

	if (cmd_list == NULL) {
		getMainWindow()->getCommandHistory()->addCommandList(command_list);
		command_list->execute();
		getMainWindow()->reposit(command_list, getPage(), getSystem());
	}
	getSystem()->resetUntouched();
}

void NedVoice::checkForElementsToSplit(NedCommandList *command_list) {
	GList *lptr, *lptr2;
	GList *elements_to_delete = NULL;
	unsigned long long  t1;
	unsigned int len;
	bool in_upbeat;
	NedMeasure *measure;
	NedTuplet *tuplet;
	NedChordOrRest *element_to_distribute = NULL;
	unsigned int overflow = 0;
	NedChordOrRest *element;
	NedChangeChordOrRestLengthCommand *change_length_command;
	NedInsertTiedElementCommand *insert_element_command;
	NedDeleteChordOrRestGroupCommand *del_group;
	int pos = 0, dotcount;

	for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		element = (NedChordOrRest *) lptr->data;
		measure = getSystem()->getMeasure(element->getMidiTime());
		if (measure == NULL) continue;
		if (element->getTupletVal() != 0) {
			tuplet = element->getTupletPtr();
			if (tuplet->getMidiStartTime() < measure->midi_start && 
				tuplet->getMidiStartTime() + tuplet->getDuration() > measure->midi_start) {
				removeTuplet(element, command_list);
				lptr = g_list_first(m_chord_or_rests); 
				resetPositionPointer(false); // recompute midi times
			}
		}
	}
	in_upbeat = getPage()->getPageNumber() == 0 && getMainWindow()->getMidiTempoInverse() > 0.0 &&
			getSystem()->getSystemNumber() == 0;

	for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		element = (NedChordOrRest *) lptr->data;
		measure = getSystem()->getMeasure(element->getMidiTime());
		if (measure == NULL) {
			elements_to_delete = g_list_append(elements_to_delete, element);
			continue;
		}
		if (!(in_upbeat && measure->getMeasureNumber() == 0)  && overflow == 0 || (element_to_distribute != NULL && element_to_distribute->isRest())) {
			if (element->getLength() == WHOLE_NOTE && element->getMidiTime() == measure->midi_start && 
					element->getMidiTime() + element->getDuration() == measure->midi_end) {
						overflow = 0;
						continue;
			}
		}
		/*
		if (m_voice_nr == 0 && getSystem()->getSystemNumber() == 0) {
			printf("m_num = %d, midi_start = %llu, midi_end = %llu, rest = %d, element->getMidiTime() = %llu, dur = %u (0x%x) --> %llu\n",
				measure->getMeasureNumber(), measure->midi_start / NOTE_8, measure->midi_end / NOTE_8, element->isRest(),
				element->getMidiTime() / NOTE_8,  element->getDuration() / NOTE_8, element, (element->getMidiTime() + element->getDuration()) / NOTE_8); fflush(stdout);
		}
		*/
		if (overflow > getMainWindow()->getMeasureDuration(measure->getMeasureNumber())) {
			overflow = 0;
		}
		if (overflow > 0) {
			NedResource::setLengthField(overflow);
			len = NedResource::getPartLength(&dotcount);
			while (len > 0) {
				insert_element_command = new NedInsertTiedElementCommand(this, element_to_distribute, len, dotcount);
				insert_element_command->execute();
				command_list->addCommand(insert_element_command);
				pos++;
				if ((lptr2 = g_list_nth(m_chord_or_rests, pos)) == NULL) {
					NedResource::Abort("checkForElementsToSplit(1)");
				}
				element_to_distribute = (NedChordOrRest *) lptr2->data;
				len = NedResource::getPartLength(&dotcount);
			}
			getSystem()->reposit(0, -1);
			overflow = 0;
			element_to_distribute = NULL;
			if ((lptr = g_list_nth(m_chord_or_rests, pos)) == NULL) {
				NedResource::Abort("checkForElementsToSplit(2)");
			}
			/*
			printf("pos(1) = %d, element->dur = %u, isrest = %d\n", pos, ((NedChordOrRest *) lptr->data)->getDuration() / NOTE_8, 
				((NedChordOrRest *) lptr->data)->isRest());
				*/
		}

		overflow = 0;
				
		if (element->getMidiTime() < measure->midi_end && element->getMidiTime() + element->getDuration() > measure->midi_end) {
		/*
			printf("getMidiTime = %llu, isrest = %d, dur = %u, midi_end = %llu\n",
				element->getMidiTime() / NOTE_8, element->isRest(), element->getDuration() / NOTE_8,  measure->midi_end / NOTE_8);
				*/

			if ((pos = g_list_position(m_chord_or_rests, lptr)) < 0) {
				NedResource::Abort("checkForElementsToSplit(3)");
			}
			t1 = measure->midi_end - element->getMidiTime();
			if (!(in_upbeat && measure->getMeasureNumber() == 0) || element->getLength() != WHOLE_NOTE) {
				overflow = element->getMidiTime() + element->getDuration() - measure->midi_end;
			}
			NedResource::setLengthField(t1);
			len = NedResource::getPartLength(&dotcount);
			change_length_command = new NedChangeChordOrRestLengthCommand(element, len, dotcount);
			change_length_command->execute();
			command_list->addCommand(change_length_command);
			len = NedResource::getPartLength(&dotcount);
			element_to_distribute = element;
			while (len > 0) {
				insert_element_command = new NedInsertTiedElementCommand(this, element_to_distribute, len, dotcount);
				insert_element_command->execute();
				pos++;
				if ((lptr2 = g_list_nth(m_chord_or_rests, pos)) == NULL) {
					NedResource::Abort("checkForElementsToSplit(4)");
				}
				element_to_distribute = (NedChordOrRest *) lptr2->data;
				len = NedResource::getPartLength(&dotcount);
			}
				
			//printf("Duration davor = %u --> ", element->getDuration() / NOTE_8);
			//NedResource::split_element(&m_chord_or_rests, lptr, t1, t2);
			/*
			printf("Duration danach = %u --> %u (0x%x)\n",
				element->getDuration() / NOTE_8, ((NedChordOrRest *) lptr->data)->getDuration() / NOTE_8, lptr->data);
				*/

			getSystem()->reposit(0, -1);
			if ((lptr = g_list_nth(m_chord_or_rests, pos)) == NULL) {
				NedResource::Abort("checkForElementsToSplit(5)");
			}
			/*
			printf("pos(2) = %d, element->dur = %u, isrest = %d\n", pos, ((NedChordOrRest *) lptr->data)->getDuration() / NOTE_8, 
				((NedChordOrRest *) lptr->data)->isRest());
				*/
		}
	}
	if (elements_to_delete != NULL) {
		del_group = new NedDeleteChordOrRestGroupCommand(this, elements_to_delete);
		del_group->execute();
		command_list->addCommand(del_group);
	}
	testAllMeasuresForFillup(command_list);
}

GList *NedVoice::getFirstChordOrRest(int lyrics_line, bool lyrics_required) {
	GList *lptr;

	for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		if (((NedChordOrRest *) lptr->data)->getType() != TYPE_NOTE) continue;
		if (!lyrics_required) return lptr;
		if (((NedChordOrRest *) lptr->data)->getLyrics(lyrics_line) != NULL) {
			return lptr;
		}
	}
	return NULL;
}

GList *NedVoice::getLastChordOrRest(int lyrics_line, bool lyrics_required) {
	GList *lptr;

	for (lptr = g_list_last(m_chord_or_rests); lptr; lptr = g_list_previous(lptr)) {
		if (((NedChordOrRest *) lptr->data)->getType() != TYPE_NOTE) continue;
		if (!lyrics_required) return lptr;
		if (((NedChordOrRest *) lptr->data)->getLyrics(lyrics_line) != NULL) {
			return lptr;
		}
	}
	return NULL;
}


void NedVoice::testAllMeasuresForFillup(NedCommandList *command_list) {
	int last_measure_number;
	NedChordOrRest *last_element;
	GList *appended_wholes = NULL;
	NedMeasure *measure;
	int meas_number;
	unsigned int diff;
	int pos;
	NedInsertChordOrRestGroupCommand *insert_chord_or_rest_group_command;

	last_measure_number = getSystem()->getNumberOfLastMeasure();
	last_element = (NedChordOrRest *) g_list_last(m_chord_or_rests)->data;
	measure = getSystem()->getMeasure(last_element->getMidiTime());
	if (measure->midi_end > last_element->getMidiTime() + last_element->getDuration()) {
		diff = measure->midi_end - (last_element->getMidiTime() + last_element->getDuration());
		if ((pos = g_list_index(m_chord_or_rests, last_element)) < 0) {
			NedResource::Abort("NedVoice::testAllMeasuresForFillup");
		}
		pos++;
		handleGapAfterInsertion(command_list, &pos, diff, NULL, 0, false, true);
	}
	for (meas_number = measure->getMeasureNumber(); meas_number < last_measure_number; meas_number++) {
		appended_wholes = g_list_append(appended_wholes, new NedChordOrRest(this, TYPE_REST, m_voice_nr > 0, 3 /* dummy */, 0, WHOLE_NOTE, NORMAL_NOTE, 0, 0));
	}
	if (appended_wholes != NULL) {
		insert_chord_or_rest_group_command = new NedInsertChordOrRestGroupCommand(this, appended_wholes, pos);
		insert_chord_or_rest_group_command->execute();
		command_list->addCommand(insert_chord_or_rest_group_command);
	}
}

		



NedChordOrRest *NedVoice::handleGapAfterInsertion(NedCommandList *command_list, int *pos /* in/out */,
			int sum_of_chords_and_rests_to_delete, NedChordOrRest *chord_or_rest_to_distribute,
			int tuplet_val, bool force_wholes, bool do_execute) {
	unsigned int len2 = 0;
	int dotcount = 0;
	NedInsertChordOrRestCommand *insert_command;
	NedChordOrRest *chord_or_rest, *last_tupleted_chord_or_rest, *distributed_new_sym;
	if (tuplet_val > 0) {
		sum_of_chords_and_rests_to_delete *= tuplet_val;
		sum_of_chords_and_rests_to_delete /= NedResource::m_tuplet_tab[tuplet_val];
	}
	/*
	if (sum_of_chords_and_rests_to_delete >= WHOLE_NOTE) {
		len2 = WHOLE_NOTE;
	}
	else */
	if (force_wholes) {
		len2 = WHOLE_NOTE;
		last_tupleted_chord_or_rest = chord_or_rest = new NedChordOrRest(this, TYPE_REST, FALSE, 3, dotcount, len2, NORMAL_NOTE, 0, 0);
		insert_command = new NedInsertChordOrRestCommand(this, *pos, chord_or_rest);
		command_list->addCommand(insert_command);
		if (do_execute) {
			insert_command->execute();
		}
	}
	else {
		NedResource::setLengthField(sum_of_chords_and_rests_to_delete);
		len2 = NedResource::getPartLength(&dotcount);
		while (len2 > 0) {
			if (chord_or_rest_to_distribute == NULL) {
				last_tupleted_chord_or_rest = chord_or_rest = new NedChordOrRest(this, TYPE_REST, FALSE, 3, dotcount, len2, NORMAL_NOTE, 0, 0);
				chord_or_rest->setTupletVal(tuplet_val);
				insert_command = new NedInsertChordOrRestCommand(this, *pos, chord_or_rest);
				command_list->addCommand(insert_command);
			}
			else {
				last_tupleted_chord_or_rest = distributed_new_sym = chord_or_rest_to_distribute->cloneWithDifferentLength(len2, dotcount);
				distributed_new_sym->setTupletVal(tuplet_val);
				insert_command = new NedInsertChordOrRestCommand(this, *pos, distributed_new_sym);
				command_list->addCommand(insert_command);
			}
			if (do_execute) {
				insert_command->execute();
			}
			len2 = NedResource::getPartLength(&dotcount);
			(*pos)++;
		}
	}
	return last_tupleted_chord_or_rest;
}

bool NedVoice::searchForBreakableTuplet(NedCommandList *command_list, unsigned long long critical_time) {
	GList *lptr;
	bool tuplet_broken;
	bool was_broken = FALSE;
	NedTuplet *tuplet;
	NedChordOrRest *chord_or_rest;
	int pos;


	do {
		tuplet_broken = FALSE;
		for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
			chord_or_rest = (NedChordOrRest *) lptr->data;
			if (chord_or_rest->getTupletVal() == 0) continue;
			tuplet = chord_or_rest->getTupletPtr();
			if (tuplet == NULL) {
				NedResource::Abort("NedVoice::searchForBreakableTuplet(1)");
			}
			if (tuplet->getMidiStartTime() < critical_time && tuplet->getEndTime() > critical_time) {
				tuplet_broken = TRUE;
				pos = g_list_index(m_chord_or_rests, tuplet->getFirstElement());
				if (pos < 0) {
					NedResource::Abort("NedVoice::searchForBreakableTuplet(2)");
				}
				command_list->addCommand(new NedDeleteChordOrRestGroupCommand(this, tuplet->getElementList()));
				pos++;
				handleGapAfterInsertion(command_list, &pos, tuplet->getDuration(), NULL, 0, false, false);
				command_list->executeAfterBreakAndSetBreak();
				was_broken = TRUE;
				resetPositionPointer(false);
				break;
			}
		}
	}
	while (tuplet_broken);
	return was_broken;
}

void NedVoice::searchForBreakableTies(NedCommandList *command_list, unsigned int dir, unsigned long long start_time, unsigned long long end_time) {
	GList *lptr;
	NedChordOrRest *chord_or_rest;
	unsigned long long chord_start_time;

	for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		chord_or_rest = (NedChordOrRest *) lptr->data;
		chord_start_time = chord_or_rest->getMidiTime();
		if (chord_start_time + chord_or_rest->getDuration() <= start_time) continue;
		if (chord_start_time >= end_time) break;
		if (chord_or_rest->testForTiesToDelete(command_list, dir)) {
			command_list->executeAfterBreakAndSetBreak();
		}
	}
}

void NedVoice::testForTupletEnd(unsigned long long start_time, unsigned int duration, unsigned long long *end_time) {
	GList *lptr;
	NedChordOrRest *chord_or_rest;
	unsigned long long chord_start_time;
	unsigned long long chord_end_time;
	NedTuplet *tuplet;

	chord_end_time = start_time + duration;
	for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		chord_or_rest = (NedChordOrRest *) lptr->data;
		chord_start_time = chord_or_rest->getMidiTime();
		if (chord_start_time + chord_or_rest->getDuration() <= start_time) continue;
		if (chord_start_time >= *end_time) break;
		if (chord_or_rest->getTupletVal() != 0) {
			tuplet = chord_or_rest->getTupletPtr();
			if (tuplet == NULL) {
				NedResource::Abort("NedVoice::testForTupletEnd");
			}
			if (chord_end_time <= tuplet->getMidiStartTime()) continue;
			if (chord_end_time >= tuplet->getMidiStartTime() + tuplet->getDuration()) continue;
			*end_time = tuplet->getMidiStartTime();
			break;
		}

	}
}


bool NedVoice::setNewXpos(NedChordOrRest *element, double newpos) {
	GList *lptr1, *lptr2;
	NedTuplet *tuplet_ptr1, *tuplet_ptr2;
	int tuplet_val;


	lptr1 = g_list_find(m_chord_or_rests, element);
	if (lptr1 == NULL) {
		NedResource::Abort("NedVoice::setNewXpos");
	}
	tuplet_val = ((NedChordOrRest *) lptr1->data)->getTupletVal();
	if (tuplet_val == 0) {
	  NedMeasure *meas_info1 = getSystem()->getMeasure(((NedChordOrRest *) lptr1->data)->getMidiTime());
	  for (lptr2 = g_list_first(m_chord_or_rests); lptr2; lptr2 = g_list_next(lptr2)) {
		if (((NedChordOrRest *) lptr2->data)->getRealXpos() > newpos) {
			NedMeasure *meas_info2 =  getSystem()->getMeasure(((NedChordOrRest *) lptr2->data)->getMidiTime());
			if (meas_info1 != meas_info2) {
				if (meas_info2 - meas_info1 == 1) {
					break;
				}
				return FALSE;
			}
			if (lptr1 == lptr2) return FALSE;
			NedCommandList *command_list = new NedCommandList(getMainWindow(), getSystem());
			((NedChordOrRest *) lptr1->data)->testForTiesToDelete(command_list);
			((NedChordOrRest *) lptr2->data)->testForTiesToDelete(command_list);
			command_list->addCommand(new NedChangeXPositionCommand(&m_chord_or_rests, (NedChordOrRest *) lptr2->data, (NedChordOrRest *) lptr1->data, TRUE));
			command_list->execute();
			getMainWindow()->getCommandHistory()->addCommandList(command_list);
			return TRUE;
		}
	  }
	  for (lptr2 = g_list_last(m_chord_or_rests); lptr2; lptr2 = g_list_previous(lptr2)) {
		if (((NedChordOrRest *) lptr2->data)->getRealXpos() < newpos) {
			NedMeasure *meas_info2 =  getSystem()->getMeasure(((NedChordOrRest *) lptr2->data)->getMidiTime());
			if (meas_info1 != meas_info2) return FALSE;
			if (lptr1 == lptr2) return FALSE;
			NedCommandList *command_list = new NedCommandList(getMainWindow(), getSystem());
			((NedChordOrRest *) lptr1->data)->testForTiesToDelete(command_list);
			((NedChordOrRest *) lptr2->data)->testForTiesToDelete(command_list);
			command_list->addCommand(new NedChangeXPositionCommand(&m_chord_or_rests, (NedChordOrRest *) lptr2->data, (NedChordOrRest *) lptr1->data, FALSE));
			command_list->execute();
			getMainWindow()->getCommandHistory()->addCommandList(command_list);
			return TRUE;
		}
	  }
	}
	else {
	  tuplet_ptr1 = ((NedChordOrRest *) lptr1->data)->getTupletPtr();
	  for (lptr2 = g_list_first(m_chord_or_rests); lptr2; lptr2 = g_list_next(lptr2)) {
		if (((NedChordOrRest *) lptr2->data)->getRealXpos() > newpos) {
			tuplet_ptr2 = ((NedChordOrRest *) lptr2->data)->getTupletPtr();
			if (lptr1 == lptr2 || tuplet_ptr1 != tuplet_ptr2) {
				break;
			}
			NedCommandList *command_list = new NedCommandList(getMainWindow(), getSystem());
			((NedChordOrRest *) lptr1->data)->testForTiesToDelete(command_list);
			((NedChordOrRest *) lptr2->data)->testForTiesToDelete(command_list);
			command_list->addCommand(new NedChangeXPositionCommand(&m_chord_or_rests, (NedChordOrRest *) lptr2->data, (NedChordOrRest *) lptr1->data, TRUE));
			command_list->execute();
			getMainWindow()->getCommandHistory()->addCommandList(command_list);
			return TRUE;
		}
	  }
	  for (lptr2 = g_list_last(m_chord_or_rests); lptr2; lptr2 = g_list_previous(lptr2)) {
		if (((NedChordOrRest *) lptr2->data)->getRealXpos() < newpos) {
			NedCommandList *command_list = new NedCommandList(getMainWindow(), getSystem());
			tuplet_ptr2 = ((NedChordOrRest *) lptr2->data)->getTupletPtr();
			if (lptr1 == lptr2 || tuplet_ptr1 != tuplet_ptr2) {
				return FALSE;
			}
			((NedChordOrRest *) lptr1->data)->testForTiesToDelete(command_list);
			((NedChordOrRest *) lptr2->data)->testForTiesToDelete(command_list);
			command_list->addCommand(new NedChangeXPositionCommand(&m_chord_or_rests, (NedChordOrRest *) lptr2->data, (NedChordOrRest *) lptr1->data, FALSE));
			command_list->execute();
			getMainWindow()->getCommandHistory()->addCommandList(command_list);
			return TRUE;
		}
	  }
	}
	return FALSE;
}

void NedVoice::searchForBeamGroups(unsigned int midi_start_time, NedChordOrRest *new_chord_or_rest /* = NULL */) {
	GList *lptr, *beamlist = NULL, *lptr3 = NULL;
	unsigned int beats = midi_start_time;
	bool split = false;
	NedMeasure *measure = NULL;
	bool grace_mode = false;
	unsigned int meas_duration;

	while ((lptr =  g_list_first(m_beam_list)) != NULL) {
		// the beam removes itself from m_beam_list if deleted
		// therfore we must restart from list begin in every loop
		delete ((NedBeam *) lptr->data);
	}
	g_list_free(m_beam_list);
	m_beam_list = NULL;

	/*
	lptr = g_list_first(m_beam_list);
	while (lptr) {
		NedBeam *b = (NedBeam *) lptr->data;
		delete b; // includes NULL pointer in chords and removal from m_beam_list
		lptr = g_list_first(m_beam_list);
	}

	for (lptr = g_list_first(m_beam_list); lptr; lptr = g_list_next(lptr)) {
		((NedBeam *) lptr->data)->m_needed = FALSE;
	}
	*/
	meas_duration = getMainWindow()->getNumerator() * WHOLE_NOTE / getMainWindow()->getDenominator();
	for(lptr = g_list_first(m_chord_or_rests), lptr3 = NULL; lptr; lptr3 = lptr, lptr = g_list_next(lptr)) {
		NedChordOrRest *chord_or_rest = (NedChordOrRest *) lptr->data;
		measure = getSystem()->getMeasure(chord_or_rest->getMidiTime());
		split = m_staff->hasStaffElem(chord_or_rest->getMidiTime());
		if (measure == NULL) {
			if (meas_duration <= beats) {
				split = true;
				beats = 0;

			}
		}
		else {
			if (meas_duration <= beats) {
				split = true;
				beats = 0;
			}
		}
		/*
		if (!split) {
			split = beamEndRequired(beamlist, beats, measure) || ((NedChordOrRest *) lptr->data)->hasLastTupletFlag();
		}
		*/
		if (!split) {
			split = beamEndRequired(beamlist, beats, measure);
		}
		if (!split && lptr3 != NULL) {
			split = (chord_or_rest->getTupletVal() != ((NedChordOrRest *) lptr3->data)->getTupletVal()) || ((NedChordOrRest *) lptr3->data)->hasLastTupletFlag();
		}
		/*
		if (lptr3 != NULL) {
			split = split || (chord_or_rest->getTupletVal() != ((NedChordOrRest *) lptr3->data)->getTupletVal());
		}
		*/
		if (chord_or_rest->getType() == TYPE_GRACE) {
			if (grace_mode) {
				split = false;
			}
			else {
				split = true;
			}
			grace_mode = true;
		}
		else if (grace_mode) {
			grace_mode = false;
			split = true;
		}

		if ((split || (chord_or_rest->getType() & (TYPE_NOTE | TYPE_GRACE)) == 0 ||  chord_or_rest->getLength() >=  NOTE_4) && beamlist != NULL) {
			if (g_list_length(beamlist) > 1) {
			/*
				inserted = FALSE;
				for (zz = 0, lptr2 = g_list_first(m_beam_list); lptr2; lptr2 = g_list_next(lptr2), zz++) {
					beam = (NedBeam *) lptr2->data;
					beam->m_needed = beam->isEqual(beamlist, new_chord_or_rest, &newpos);
					printf("Beam %d --> %d\n", zz, beam->m_needed); fflush(stdout);
					if (new_chord_or_rest != NULL && beam->m_needed) {
						beam->insertChord(new_chord_or_rest, newpos);
						inserted = TRUE;
					}
					else if (beam->m_needed) {
						inserted = TRUE;
					}
				}
				if (inserted) {
					g_list_free(beamlist);
				}
				else {
				*/
					NedBeam *bb = new NedBeam(beamlist);
					m_beam_list = g_list_append(m_beam_list, bb);
					/*
				}
				*/
			}
			else {
				g_list_free(beamlist);
			}
			beamlist = NULL;
		}
		if ((chord_or_rest->getType() & (TYPE_NOTE | TYPE_GRACE)) != 0 && chord_or_rest->getLength() != STROKEN_GRACE && chord_or_rest->getLength() < NOTE_4) {
			beamlist = g_list_append(beamlist, chord_or_rest);
		}
		beats += chord_or_rest->getDuration();
		measure = getSystem()->getMeasure(chord_or_rest->getMidiTime());
		if (measure == NULL) {
			meas_duration = getMainWindow()->getNumerator() * WHOLE_NOTE / getMainWindow()->getDenominator();
		}
		else {
			meas_duration = getMainWindow()->getMeasureDuration(measure->getMeasureNumber());
		}
	}
	if (beamlist != NULL) {
		if (g_list_length(beamlist) > 1) {
			NedBeam *bb = new NedBeam(beamlist);
			m_beam_list = g_list_append(m_beam_list, bb);
		}
		else {
			g_list_free(beamlist);
			beamlist = NULL;
		}
	}
	/*
	lptr = g_list_first(m_beam_list);
	while (lptr) {
		NedBeam *b = (NedBeam *) lptr->data;
		if (b->getChordCount() < 1 || !b->m_needed)  {
			delete b; // includes NULL pointer in chords/rests and removal from m_beam_list
			lptr = g_list_first(m_beam_list);
		}
		else {
			lptr = g_list_next(lptr);
		}
	}
	*/
}

void NedVoice::collectChordsAndRests(NedClipBoard *board, unsigned long long midi_start, unsigned long long  midi_end) {
	GList *lptr;
	NedClipBoardElement *element = new NedClipBoardElement();
	element->from_voice = this;
	for(lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		if (((NedChordOrRest *) lptr->data)->getMidiTime() < midi_start  || ((NedChordOrRest *) lptr->data)->getMidiTime() >= midi_end) continue;
		element->chords_or_rests_to_move = g_list_append(element->chords_or_rests_to_move, lptr->data);
	}
	board->m_elements = g_list_append(board->m_elements, element);
}

void NedVoice::collectDestinationVoices(NedClipBoard *board) {
	NedClipBoardElement *element;
	if (board->element_counter == NULL) {
		NedResource::Abort("NedVoice::collectDestinationVoices(1)");
	}
	element = ((NedClipBoardElement *) board->element_counter->data);
	if (element->from_voice == NULL) {
		NedResource::Abort("NedVoice::collectDestinationVoices(2)");
	}
	element->to_voice = this;
	board->element_counter = g_list_next(board->element_counter);
}

	

bool NedVoice::beamEndRequired(GList *beamlist_so_far, unsigned int beats, NedMeasure *measure) {
	int shortestNote = WHOLE_NOTE, dur;
	GList *lptr;
	struct rule_str *wild_ptr = NULL, *rule_ptr = NULL, *ptr;
	int best_match = -1;
	int num, denom;

	for (lptr = g_list_first(beamlist_so_far); lptr; lptr = g_list_next(lptr)) {
		dur = ((NedChordOrRest *) lptr->data)->getDuration();
		if (dur < shortestNote) shortestNote = dur;
	}
	if (measure == NULL) {
		num = getMainWindow()->getNumerator(); 
		denom = getMainWindow()->getDenominator();
	}
	else {
		num = getMainWindow()->getNumerator(measure->getMeasureNumber());
		denom = getMainWindow()->getDenominator(measure->getMeasureNumber());
	}
	for (ptr = beam_rules_tab__; ptr->function != END_OF_TABLE; ptr++) {
		if (ptr->function!= FUNC_END) continue;
		if (num == ptr->time_num && denom == ptr->time_denom) {
			if (ptr->notelen < 0) {
				wild_ptr = ptr;
			}
			else if (ptr->notelen >= shortestNote) {
				if (best_match < 0) {
					best_match = ptr->notelen;
					rule_ptr = ptr;
				}
				else if (best_match > ptr->notelen) {
					best_match = ptr->notelen;
					rule_ptr = ptr;
				}
			}
		}
	}
	if ((best_match != shortestNote || rule_ptr == NULL) && wild_ptr != NULL) {
		rule_ptr = wild_ptr;
	}
	if (rule_ptr == NULL) {
		return false;
	}
	return ((beats % rule_ptr->duration) == 0);
}

int NedVoice::getSorting(NedChordOrRest *chord0, NedChordOrRest *chord1) {
	int pos0, pos1;

	if ((pos0 = g_list_index(m_chord_or_rests, chord0)) < 0) {
		NedResource::Abort("NedVoice::getSorting 1");
	}
	if ((pos1 = g_list_index(m_chord_or_rests, chord1)) < 0) {
		NedResource::Abort("NedVoice::getSorting 2");
	}
	if (pos0 == pos1) {
		NedResource::Abort("NedVoice::getSorting 3");
	}
	if (pos0 + 1 == pos1) {
		return SORTING_GREAT;
	}
	if (pos1 + 1 == pos0) {
		return SORTING_LESS;
	}
	return SORTING_NONE;
}

bool NedVoice::isFirst(NedChordOrRest *chord) {
	GList *lptr;
	if ((lptr = g_list_find(m_chord_or_rests, chord)) == NULL) {
		 NedResource::Abort("NedVoice::isFirst");
	}
	return (lptr == g_list_first(m_chord_or_rests));
}

bool NedVoice::isLast(NedChordOrRest *chord) {
	GList *lptr;
	if ((lptr = g_list_find(m_chord_or_rests, chord)) == NULL) {
		 NedResource::Abort("NedVoice::isLast");
	}
	return (lptr == g_list_last(m_chord_or_rests));
}

int NedVoice::getBeamCount() {
	return g_list_length(m_beam_list);
}

NedChordOrRest *NedVoice::getNextChordOrRest(NedChordOrRest *element) {
	GList *lptr;

	lptr = g_list_find(m_chord_or_rests, element);
	if ((lptr = g_list_next(lptr)) == NULL) {
		return NULL;
	}
	return (NedChordOrRest *) lptr->data;
}

void NedVoice::selectNextChord(NedChordOrRest *chord, int line, bool lyrics_required) {
	GList *lptr;
	NedSystem *other_system;

	getMainWindow()->m_selected_chord_or_rest = NULL;
	getMainWindow()->m_selected_note = NULL;
	if ((lptr = g_list_find(m_chord_or_rests, chord)) == NULL) {
		NedResource::Abort("NedVoice::selectNextChord");
	}
	for (lptr = g_list_next(lptr); lptr; lptr = g_list_next(lptr)) {
		if (((NedChordOrRest *) lptr->data)->getType() == TYPE_NOTE) {
			if (!lyrics_required || ((NedChordOrRest *) lptr->data)->getLyrics(line) != NULL) {
				break;
			}
		}
	}
	if (lptr == NULL) {
		other_system = getPage()->getNextSystem(getSystem());
		if (other_system != NULL) {
			lptr = other_system->getFirstChordOrRest(m_staff->getStaffNumber(), m_voice_nr, line, lyrics_required);
		}
	}
	if (lptr == NULL) {
		return;
	}
	getMainWindow()->m_selected_chord_or_rest = (NedChordOrRest *) lptr->data;
	getMainWindow()->m_selected_note = ((NedChordOrRest *) lptr->data)->getFirstNote();
}

void NedVoice::selectPreviousChord(NedChordOrRest *chord, int line, bool lyrics_required) {
	GList *lptr;
	NedSystem *other_system;

	getMainWindow()->m_selected_chord_or_rest = NULL;
	getMainWindow()->m_selected_note = NULL;
	if ((lptr = g_list_find(m_chord_or_rests, chord)) == NULL) {
		NedResource::Abort("NedVoice::selectPreviousChord");
	}
	for (lptr = g_list_previous(lptr); lptr; lptr = g_list_previous(lptr)) {
		if (((NedChordOrRest *) lptr->data)->getType() == TYPE_NOTE) {
			if (!lyrics_required || ((NedChordOrRest *) lptr->data)->getLyrics(line) != NULL) {
				break;
			}
		}
	}
	if (lptr == NULL) {
		other_system = getPage()->getPreviousSystem(getSystem());
		if (other_system != NULL) {
			lptr = other_system->getLastChordOrRest(m_staff->getStaffNumber(), m_voice_nr, line, lyrics_required);
		}
	}
	if (lptr == NULL) {
		return;
	}
	getMainWindow()->m_selected_chord_or_rest = (NedChordOrRest *) lptr->data;
	getMainWindow()->m_selected_note = ((NedChordOrRest *) lptr->data)->getFirstNote();
}

NedChordOrRest *NedVoice::getPreviousChordOrRest(NedChordOrRest *element) {
	GList *lptr;

	lptr = g_list_find(m_chord_or_rests, element);
	if ((lptr = g_list_previous(lptr)) == NULL) {
		return NULL;
	}
	return (NedChordOrRest *) lptr->data;
}

void NedVoice::removeBeam(NedBeam *beam) {
	GList *lptr;
	if ((lptr = g_list_find(m_beam_list, beam)) == NULL) {
		NedResource::Abort("NedVoice::removeBeam");
	}
	m_beam_list = g_list_delete_link(m_beam_list, lptr);
}

void NedVoice::addBeam(NedBeam *beam) {
	m_beam_list = g_list_append(m_beam_list, beam);
}

double NedVoice::findTimePos(unsigned int time) {
	GList *lptr;
	for(lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		if (((NedChordOrRest *) lptr->data)->getTime() == time) {
			return ((NedChordOrRest *) lptr->data)->getXPos();
		}
		else if (((NedChordOrRest *) lptr->data)->getTime() > time) {
			lptr = g_list_previous(lptr);
			if (lptr != NULL) {
				return ((NedChordOrRest *) lptr->data)->getXPos();
			}
			else {
				return 0;
			}
		}
	}
	return -1;
}

void NedVoice::resetPositionPointer(bool reset_multiple_flag, bool resetRestYPos /* = FALSE */) {
	GList *lptr;
	unsigned long long midi_time = (getPage()->getPageNumber() == 0 && getSystem()->getSystemNumber() == 0) ? getMainWindow()->getUpBeatInverse() : 0;
	for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		((NedChordOrRest *) lptr->data)->setMidiTime(midi_time, resetRestYPos);
		midi_time += ((NedChordOrRest *) lptr->data)->getDuration();
	}
	m_position_ptr = g_list_first(m_chord_or_rests);
	if (reset_multiple_flag) {
		m_stem_direction = STEM_DIR_NONE;
	}
}
		

NedChordOrRest *NedVoice::getNextElementAfter(unsigned long long midi_time) {
	for (; m_position_ptr && ((NedChordOrRest *) m_position_ptr->data)->getMidiTime() < midi_time; m_position_ptr = g_list_next(m_position_ptr));
	if (m_position_ptr == NULL) return NULL;
	if (((NedChordOrRest *) m_position_ptr->data)->getMidiTime() >= midi_time) {
		return (NedChordOrRest *) m_position_ptr->data;
	}
	return NULL;
}

void NedVoice::getElementAtPosition(int pc, bool with_rests, NedChordOrRest **chords, int *numchords, unsigned int *has_chords, unsigned int marker) {
	for (;m_position_ptr && ((NedChordOrRest *) m_position_ptr->data)->m_position < pc; m_position_ptr = g_list_next(m_position_ptr));
	if (m_position_ptr == NULL) return;
	if (((NedChordOrRest *) m_position_ptr->data)->m_position == pc) {
		if (((!with_rests || with_rests && ((NedChordOrRest *) m_position_ptr->data)->isHidden())) && ((NedChordOrRest *) m_position_ptr->data)->isRest()) {
			m_position_ptr = g_list_next(m_position_ptr);
			return;
		}
		if (*numchords >= VOICE_COUNT) {
			NedResource::Abort("NedVoice::getElementAtPosition");
		}
		chords[*numchords] = (NedChordOrRest *) m_position_ptr->data;
		(*numchords)++;
		(*has_chords) |= marker;
		m_position_ptr = g_list_next(m_position_ptr);
	}
}


void NedVoice::changeDirs() {
	GList *lptr;
	for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		((NedChordOrRest *) lptr->data)->changeStemDir(m_stem_direction);
	}
	for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		((NedChordOrRest *) lptr->data)->changeStemDir(m_stem_direction);
	}
	for (lptr = g_list_first(m_beam_list); lptr; lptr = g_list_next(lptr)) {
		((NedBeam *) lptr->data)->changeStemDir(m_stem_direction);
	}
}

void NedVoice::placeCurrentElementAt(double xpos) {
	if (m_position_ptr == NULL) {
		NedResource::Abort(" NedVoice::placeCurrentElementAt");
	}
	((NedChordOrRest *) m_position_ptr->data)->setXPos(xpos);
	m_position_ptr = g_list_next(m_position_ptr);
}

void NedVoice::incrElementPointer() {
	m_position_ptr = g_list_next(m_position_ptr);
}

void NedVoice::shiftX(int pc, double shift) {
	if (m_position_ptr == NULL) return;
	//if (((NedChordOrRest *) m_position_ptr->data)->m_position < pc) {printf("SKIP!!!\n"); fflush(stdout);}
	if (((NedChordOrRest *) m_position_ptr->data)->m_position != pc) return;
	((NedChordOrRest *) m_position_ptr->data)->shiftX(shift);
	m_position_ptr = g_list_next(m_position_ptr);
}


void NedVoice::insertIntoSystem(GList *chords_or_rests_to_move) {
	GList *lptr;
	NedBeam *b;
	int pos = 0;
	for (lptr = g_list_first(chords_or_rests_to_move); lptr; lptr = g_list_next(lptr)) {
		if ((b = ((NedChordOrRest *) lptr->data)->getBeam()) != NULL) {
			delete b;
		}
		((NedChordOrRest *) lptr->data)->setVoice(this);
		m_chord_or_rests = g_list_insert(m_chord_or_rests, lptr->data, pos++);
	}
}

void NedVoice::appendAtSystem(GList *chords_or_rests_to_move) {
	GList *lptr;
	NedBeam *b;
	for (lptr = g_list_first(chords_or_rests_to_move); lptr; lptr = g_list_next(lptr)) {
		if ((b = ((NedChordOrRest *) lptr->data)->getBeam()) != NULL) {
			delete b;
		}
		((NedChordOrRest *) lptr->data)->setVoice(this);
		m_chord_or_rests = g_list_append(m_chord_or_rests, lptr->data);
	}
}


void NedVoice::deleteNoteGroup(GList *chords_or_rests_to_move) {
	GList *lptr, *lptr2;
	for(lptr = g_list_first(chords_or_rests_to_move); lptr; lptr = g_list_next(lptr)) {
		if ((lptr2 = g_list_find(m_chord_or_rests, lptr->data)) == NULL) {
			NedResource::Abort("NedVoice::deleteNoteGroup");
		}
		m_chord_or_rests = g_list_delete_link(m_chord_or_rests, lptr2);
	}
}


void NedVoice::computeBeams(int *lyrics_lines, double *topy, double *boty) {
	GList *lptr;
	double ty, by;
	for(lptr = g_list_first(m_beam_list); lptr; lptr = g_list_next(lptr)) {
		((NedBeam *) lptr->data)->computeBeam(m_stem_direction);
		
	}
	for(lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		((NedChordOrRest *) lptr->data)->getTopBotY(lyrics_lines, &ty, &by);
		if (ty < *topy) *topy = ty;
		if (by > *boty) *boty = by;
	}
}

void NedVoice::computeTuplets(double *topy, double *boty) {
	GList *lptr;
	GList *tuplet_list = NULL;
	int tuplet_val = 0;
	NedTuplet *tuplet;
	//double ty, by;
	for(lptr = g_list_first(m_tuplet_list); lptr; lptr = g_list_next(lptr)) {
		delete ((NedTuplet *) lptr->data);
	}
	g_list_free(m_tuplet_list);
	m_tuplet_list = NULL;
	for(lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		if (((NedChordOrRest *) lptr->data)->getTupletVal() > 0) {
			if (tuplet_val == 0) {
				tuplet_val = ((NedChordOrRest *) lptr->data)->getTupletVal();
				tuplet_list = g_list_append(tuplet_list, lptr->data);
			}
			else if (((NedChordOrRest *) lptr->data)->getTupletVal() == tuplet_val) {
				tuplet_list = g_list_append(tuplet_list, lptr->data);
				if (((NedChordOrRest *) lptr->data)->hasLastTupletFlag()) {
					tuplet = new NedTuplet(tuplet_val, tuplet_list, topy, boty);
					 m_tuplet_list = g_list_append(m_tuplet_list, tuplet);
					 tuplet_val = 0;
					 tuplet_list = NULL;
				}
			}
			else {
				tuplet = new NedTuplet(tuplet_val, tuplet_list, topy, boty);
				m_tuplet_list = g_list_append(m_tuplet_list, tuplet);
				tuplet_val = 0;
				tuplet_list = NULL;
			}
		}
		else if (tuplet_val > 0) {
			tuplet = new NedTuplet(tuplet_val, tuplet_list, topy, boty);
			m_tuplet_list = g_list_append(m_tuplet_list, tuplet);
			tuplet_val = 0;
			tuplet_list = NULL;
		}
	}
			
	/*
	for(lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		((NedChordOrRest *) lptr->data)->getTopBotY(&ty, &by);
		if (ty < *topy) *topy = ty;
		if (by > *boty) *boty = by;
	}
	*/
}

void NedVoice::prepareReplay() {
	GList *lptr, *lptr2;
	int clef, keysig, octave_shift;
	int grace_time = 0;
	NedChordOrRest *chord_or_rest;
	int measurenumber = -1;
	for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		chord_or_rest = (NedChordOrRest *) lptr->data;
		NedMeasure *measure = getSystem()->getMeasure(chord_or_rest->getMidiTime());
		if (measure->getMeasureNumber() != measurenumber) {
			measurenumber = measure->getMeasureNumber();
			NedResource::addMeasureEntry(measure);
		}
		if (chord_or_rest->getType() == TYPE_NOTE) {
			m_staff->getCurrentClefAndKeysig(chord_or_rest->getMidiTime(), &clef, &keysig, &octave_shift);
			grace_time = 0;
			for (lptr2 = g_list_next(lptr); lptr2; lptr2 = g_list_next(lptr2)) {
				if (((NedChordOrRest *) lptr2->data)->getType() == TYPE_GRACE) {
					grace_time += GRACE_NOTE_TIME;
					continue;
				}
				break;
			}
			if (grace_time > chord_or_rest->getDuration() / 2) {
				grace_time = chord_or_rest->getDuration() / 2;
			}
			chord_or_rest->prepareReplay(clef, keysig, octave_shift, grace_time);
		}
		else if (chord_or_rest->getType() == TYPE_GRACE) {
			if (grace_time <= 0) {
				continue;
			}
			m_staff->getCurrentClefAndKeysig(chord_or_rest->getMidiTime(), &clef, &keysig, &octave_shift);
			chord_or_rest->prepareReplay(clef, keysig, octave_shift, grace_time);
			grace_time -= GRACE_NOTE_TIME;
		}
		if (chord_or_rest->getType() == TYPE_REST) {
			grace_time = 0;
			for (lptr2 = g_list_next(lptr); lptr2; lptr2 = g_list_next(lptr2)) {
				if (((NedChordOrRest *) lptr2->data)->getType() == TYPE_GRACE) {
					grace_time += GRACE_NOTE_TIME;
					continue;
				}
				break;
			}
			if (grace_time > chord_or_rest->getDuration() / 2) {
				grace_time = chord_or_rest->getDuration() / 2;
			}
		}
	}
}

void NedVoice::removeUnneededAccidentals() {
	GList *lptr;
	int clef, keysig, octave_shift;
	NedChordOrRest *chord_or_rest;
	for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		chord_or_rest = (NedChordOrRest *) lptr->data;
		if (chord_or_rest->getType() == TYPE_NOTE) {
			m_staff->getCurrentClefAndKeysig(chord_or_rest->getMidiTime(), &clef, &keysig, &octave_shift);
			chord_or_rest->removeUnneededAccidentals(clef, keysig, octave_shift);
		}
	}
}

void NedVoice::findAccidentals(char offs_array[115], NedMeasure *meas_info, unsigned long long midi_time) {
	GList *lptr;

	for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		if (((NedChordOrRest *) lptr->data)->getMidiTime() < meas_info->midi_start) continue;
		if (((NedChordOrRest *) lptr->data)->getMidiTime() >= midi_time) break;
		((NedChordOrRest *) lptr->data)->setOffset(offs_array);
	}
}

void NedVoice::saveVoice(FILE *fp) {
	GList *lptr;
	bool BEAMS_written = FALSE;
	bool TIES_written = FALSE;
	bool TUPLETS_written = FALSE;

	for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		((NedChordOrRest *) lptr->data)->saveChordOrRest(fp);
	}
	fprintf(fp, "\n");
	for(lptr = g_list_first(m_beam_list); lptr; lptr = g_list_next(lptr)) {
		if (!BEAMS_written) {
			fprintf(fp, "BEAMS\n");
			BEAMS_written = TRUE;
		}
		((NedBeam *) lptr->data)->saveBeam(fp);
	}
	for(lptr = g_list_first(m_tuplet_list); lptr; lptr = g_list_next(lptr)) {
		if (!TUPLETS_written) {
			fprintf(fp, "TUPLETS\n");
			TUPLETS_written = TRUE;
		}
		((NedTuplet *) lptr->data)->saveTuplet(fp);
	}
	for (lptr = g_list_first(m_chord_or_rests); lptr; lptr = g_list_next(lptr)) {
		((NedChordOrRest *) lptr->data)->saveTies(fp, &TIES_written);
	}
}

void NedVoice::restoreVoice(FILE *fp) {
	char buffer[128];
	bool beams_read = FALSE;
	bool ties_read = FALSE;
	bool tuplets_read = FALSE;
	NedChordOrRest *chord_or_rest;

	while (NedResource::m_error_message == NULL) {
		if (!NedResource::readWord(fp, buffer)) {
			if (feof(fp)) return;
			NedResource::m_error_message = "( TUPLETS, BEAMS, TIES, VOICE, STAFF SYSTEM or PAGE expected";
			return;
		}
		if (!strcmp(buffer, "VOICE") || !strcmp(buffer, "STAFF") || !strcmp(buffer, "SYSTEM") || !strcmp(buffer, "PAGE")) {
			NedResource::unreadWord(buffer);
			return;
		}
		if (!strcmp(buffer, "BEAMS")) {
			if (beams_read) {
				NedResource::m_error_message = "BEAMS twice";
				return;
			}
			if (ties_read) {
				NedResource::m_error_message = "BEAMS after TIES";
				return;
			}
			beams_read = TRUE;
			restoreBeams(fp);
			if (NedResource::m_error_message != NULL) {
				return;
			}
			continue;
		}
		if (!strcmp(buffer, "TIES")) {
			if (ties_read) {
				NedResource::m_error_message = "TIES twice";
				return;
			}
			ties_read = TRUE;
			restoreTies(fp);
			if (NedResource::m_error_message != NULL) {
				return;
			}
			continue;
		}
		if (!strcmp(buffer, "TUPLETS")) {
			if (tuplets_read) {
				NedResource::m_error_message = "TUPLETS twice";
				return;
			}
			tuplets_read = TRUE;
			restoreTuplets(fp);
			if (NedResource::m_error_message != NULL) {
				return;
			}
			continue;
		}
		if (!strcmp(buffer, "(")) {
			if (beams_read) {
				NedResource::m_error_message = "Notes after BEAMS";
				return;
			}
			if (ties_read) {
				NedResource::m_error_message = "Notes after TIES";
				return;
			}
			if ((chord_or_rest = NedChordOrRest::restoreChordOrRest(fp, this)) != NULL) {
				m_chord_or_rests = g_list_append(m_chord_or_rests, chord_or_rest);
				getSystem()->resetUntouched();
			}
			else {
				return;
			}
		}
		else {
			NedResource::m_error_message = "( BEAMS, TIES, VOICE, STAFF SYSTEM or PAGE expected";
			return;
		}
	}
}
				
void NedVoice::restoreBeams(FILE *fp) {
	char buffer[128];
	int marker;

	if (!NedResource::readWord(fp, buffer)) {
		if (feof(fp)) return;
		NedResource::m_error_message = "( TIES VOICE STAFF SYSTEM or PAGE expected";
		return;
	}
	if (!strcmp(buffer, "TUPLETS") || !strcmp(buffer, "TIES") || !strcmp(buffer, "VOICE") || !strcmp(buffer, "STAFF") || !strcmp(buffer, "SYSTEM") || !strcmp(buffer, "PAGE")) {
		NedResource::unreadWord(buffer);
		return;
	}
	if (strcmp(buffer, "(")) {
		NedResource::m_error_message = "( TIES VOICE STAFF SYSTEM or PAGE expected";
		return;
	}
	while (!strcmp(buffer, "(")) {
		if (!NedResource::readWord(fp, buffer)) {
			NedResource::m_error_message = "u or d expected";
			return;
		}
		if (!NedResource::readInt(fp, &marker)) {
			NedResource::m_error_message = "marker expected(1)";
			return;
		}
		if (!NedResource::readWord(fp, buffer) || buffer[1] != '\0') {
			NedResource::m_error_message = ", or ) expected";
			return;
		}
		while (strcmp(buffer, ")")) {
			if (buffer[0] != ',') {
				NedResource::m_error_message = ", or ) expected";
				return;
			}
			if (!NedResource::readInt(fp, &marker)) {
				NedResource::m_error_message = "marker expected(2)";
				return;
			}
			if (!NedResource::readWord(fp, buffer) || buffer[1] != '\0') {
				NedResource::m_error_message = ", or ) expected";
				return;
			}
		}
		if (strcmp(buffer, ")")) {
			NedResource::m_error_message = ", or ) expected";
			return;
		}
		if (!NedResource::readWord(fp, buffer)) {
			if (feof(fp)) return;
			NedResource::m_error_message = "( TIES VOICE STAFF SYSTEM or PAGE expected";
			return;
		}
		if (!strcmp(buffer, "TUPLETS") || !strcmp(buffer, "TIES") || !strcmp(buffer, "VOICE") || !strcmp(buffer, "STAFF") || !strcmp(buffer, "SYSTEM") || !strcmp(buffer, "PAGE")) {
			NedResource::unreadWord(buffer);
			return;
		}
		else if (strcmp(buffer, "(")) {
			NedResource::m_error_message = "( TIES VOICE STAFF SYSTEM or PAGE expected";
			return;
		}
	}
}

void NedVoice::restoreTuplets(FILE *fp) {
	char buffer[128];
	int marker;
	int tuplet_val;
	NedChordOrRest *chord_or_rest;

	if (!NedResource::readWord(fp, buffer)) {
		if (feof(fp)) return;
		NedResource::m_error_message = "( TIES VOICE STAFF SYSTEM or PAGE expected";
		return;
	}
	if (!strcmp(buffer, "TIES") || !strcmp(buffer, "VOICE") || !strcmp(buffer, "STAFF") || !strcmp(buffer, "SYSTEM") || !strcmp(buffer, "PAGE")) {
		NedResource::unreadWord(buffer);
		return;
	}
	if (strcmp(buffer, "(")) {
		NedResource::m_error_message = "( TIES VOICE STAFF SYSTEM or PAGE expected";
		return;
	}

	while (!strcmp(buffer, "(")) {
		if (!NedResource::readInt(fp, &tuplet_val) || tuplet_val < 2 || tuplet_val > 13) {
			NedResource::m_error_message = "tuplet val expected";
			return;
		}
		if (!NedResource::readWord(fp, buffer) || buffer[0] != ';' || buffer[1] != '\0') {
			NedResource::m_error_message = "; expected";
			return;
		}
		if (!NedResource::readInt(fp, &marker)) {
			NedResource::m_error_message = "marker expected(1)";
			return;
		}
		chord_or_rest = (NedChordOrRest *) NedResource::getAdressOfMarker(marker);
		chord_or_rest->setTupletVal(tuplet_val);
		if (!NedResource::readWord(fp, buffer) || buffer[1] != '\0') {
			NedResource::m_error_message = ", or ) expected";
			return;
		}
		while (strcmp(buffer, ")")) {
			if (buffer[0] != ',') {
				NedResource::m_error_message = ", or ) expected";
				return;
			}
			if (!NedResource::readInt(fp, &marker)) {
				NedResource::m_error_message = "marker expected(2)";
				return;
			}
			chord_or_rest = (NedChordOrRest *) NedResource::getAdressOfMarker(marker);
			chord_or_rest->setTupletVal(tuplet_val);
			if (!NedResource::readWord(fp, buffer) || buffer[1] != '\0') {
				NedResource::m_error_message = ", or ) expected";
				return;
			}
		}
		chord_or_rest->setLastTupletFlag(TRUE);
		if (strcmp(buffer, ")")) {
			NedResource::m_error_message = ", or ) expected";
			return;
		}
		if (!NedResource::readWord(fp, buffer)) {
			if (feof(fp)) return;
			NedResource::m_error_message = "( TIES VOICE STAFF SYSTEM or PAGE expected";
			return;
		}
		if (!strcmp(buffer, "TIES") || !strcmp(buffer, "VOICE") || !strcmp(buffer, "STAFF") || !strcmp(buffer, "SYSTEM") || !strcmp(buffer, "PAGE")) {
			NedResource::unreadWord(buffer);
			return;
		}
		else if (strcmp(buffer, "(")) {
			NedResource::m_error_message = "( TIES VOICE STAFF SYSTEM or PAGE expected";
			return;
		}
	}
}

void NedVoice::restoreTies(FILE *fp) {
	char buffer[128];
	int marker1, marker2;
	NedNote *note1, *note2;

	if (!NedResource::readWord(fp, buffer)) {
		if (feof(fp)) return;
		NedResource::m_error_message = "( VOICE STAFF SYSTEM or PAGE expected";
		return;
	}
	if (!strcmp(buffer, "TUPLETS") ||!strcmp(buffer, "VOICE") || !strcmp(buffer, "STAFF") || !strcmp(buffer, "SYSTEM") || !strcmp(buffer, "PAGE")) {
		NedResource::unreadWord(buffer);
		return;
	}
	if (strcmp(buffer, "(")) {
		NedResource::m_error_message = "( VOICE STAFF SYSTEM or PAGE expected";
		return;
	}
	while (!strcmp(buffer, "(")) {
		if (!NedResource::readInt(fp, &marker1)) {
			NedResource::m_error_message = "address expected";
			return;
		}
		if (!NedResource::readWord(fp, buffer) || buffer[0] != ',' || buffer[1] != '\0') {
			NedResource::m_error_message = ", expected";
			return;
		}
		if (!NedResource::readInt(fp, &marker2)) {
			NedResource::m_error_message = "address expected";
			return;
		}
		if (!NedResource::readWord(fp, buffer) || buffer[0] != ')' || buffer[1] != '\0') {
			NedResource::m_error_message = ") expected";
			return;
		}
		note1 = (NedNote *) NedResource::getAdressOfMarker(marker1);
		note2 = (NedNote *) NedResource::getAdressOfMarker(marker2);
		NedNote::setTies(note1, note2);
		if (!NedResource::readWord(fp, buffer)) {
			if (feof(fp)) return;
			NedResource::m_error_message = "( VOICE STAFF SYSTEM or PAGE expected";
			return;
		}
		if (!strcmp(buffer, "VOICE") || !strcmp(buffer, "STAFF") || !strcmp(buffer, "SYSTEM") || !strcmp(buffer, "PAGE")) {
			NedResource::unreadWord(buffer);
			return;
		}
		if (strcmp(buffer, "(")) {
			NedResource::m_error_message = "( VOICE STAFF SYSTEM or PAGE expected";
			return;
		}
	}
}
