/****************************************************************************************/
/*											*/
/* This program is free software; you can redistribute it and/or modify it under the	*/
/* terms of the GNU General Public License as published by the Free Software		*/
/* Foundation; either version 2 of the License, or (at your option) any later version.	*/
/*											*/
/* This program is distributed in the hope that it will be useful, but WITHOUT ANY	*/
/* WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A	*/
/* PARTICULAR PURPOSE. See the GNU General Public License for more details.		*/
/*											*/
/* You should have received a copy of the GNU General Public License along with this	*/
/* program; (See "COPYING"). If not, If not, see <http://www.gnu.org/licenses/>.        */
/*											*/
/*--------------------------------------------------------------------------------------*/
/*											*/
/*  Copyright   Joerg Anders, TU Chemnitz, Fakultaet fuer Informatik, GERMANY           */
/*		ja@informatik.tu-chemnitz.de						*/
/*											*/
/*											*/
/****************************************************************************************/

#include "instrumenttable.h"
#include "resource.h"
#include "chordorrest.h"
#include "note.h"
#include "mainwindow.h"
#include "temposign.h"
#include "volumesign.h"

#define DEFAULT_FREE_X_DIST -0.3
#define DEFAULT_FREE_Y_DIST -1.3

int NedTableInstrument::m_splits[5] = {0, 25, 60, 90, 128};
int NedTableInstrument::m_piano_splits[3] = {25, 60, 90};

bool NedTableTuplet::overlaps(unsigned long long t0, unsigned long long t2) {
	if (t0 >= m_t2) return false;
	if (t2 <= m_t0) return false;
	return true;
}

int NedTableTimesig::compare(NedTableTimesig *t1, NedTableTimesig *t2) {
	if (t1->m_midi_time < t2->m_midi_time) return -1;
	if (t1->m_midi_time > t2->m_midi_time) return  1;
	return 0;
}
unsigned int NedTableTimesig::getDuration() {
	return m_numerator * WHOLE_NOTE / m_denominator;
}

int NedTableTemposig::compare(NedTableTemposig *t1, NedTableTemposig *t2) {
	if (t1->m_midi_time < t2->m_midi_time) return -1;
	if (t1->m_midi_time > t2->m_midi_time) return  1;
	return 0;
}

int NedTableNote::compare(NedTableNote *n1, NedTableNote *n2) {
	if (n1->m_start > n2->m_start) return 1;
	if (n1->m_start < n2->m_start) return -1;
	return 0;
}

unsigned int NedTableNote::getDuration() {
	return m_stop - m_start;
}


NedTableNote::NedTableNote(unsigned long long start, unsigned long long stop, int pitch, int volume) :
			m_tie_forward(NULL), m_tie_backward(NULL), m_start(start), m_stop(stop), m_ori_start(start), m_ori_stop(stop), m_triplet_start(start),
			m_triplet_stop(stop), m_midi_pitch(pitch), m_volume(volume), m_line(0), m_state(0),
			m_start_fix(false), m_end_fix(false), m_is_triplet(false), m_tr_start(false), m_tr_stop(false),
			m_takenForTuplet(false), m_handledAsTuplet(false), m_triplet_split_note(false), m_is_rest(false),
			m_tuplet_ptr(NULL), m_chord(NULL), m_closed(false), m_is_chord_member(false) {}

NedTableNote::NedTableNote(bool is_rest, unsigned long long start, unsigned long long stop, int pitch, int volume) :
			m_tie_forward(NULL), m_tie_backward(NULL), m_start(start), m_stop(stop), m_ori_start(start), m_ori_stop(stop), m_triplet_start(start),
			m_triplet_stop(stop), m_midi_pitch(pitch), m_volume(volume), m_line(0), m_state(0),
			m_start_fix(false), m_end_fix(false), m_is_triplet(false), m_tr_start(false), m_tr_stop(false),
			m_takenForTuplet(false), m_handledAsTuplet(false), m_triplet_split_note(false), m_is_rest(is_rest),
			m_tuplet_ptr(NULL), m_chord(NULL), m_closed(false), m_is_chord_member(false) {}

NedTableNote *NedTableNote::clone(unsigned long long start) {
	NedTableNote *c = new NedTableNote(m_is_rest, start, m_stop, m_midi_pitch, m_volume);
	c->m_line = m_line;
	c->m_state = m_state;
	c->m_start_fix = true;
	c->m_closed = true;

	return c;
}

GList *NedTableNote::my_insert_sorted(GList *list, NedTableNote *nnote, gint (*cfunc)(const void*, const void*), bool allow_doubles) {
	NedTableNote *tableNote;
	int npitch, opitch;
	GList *lptr;

	npitch = nnote->m_is_rest ? 111111 : nnote->m_midi_pitch;


	for (lptr = g_list_first(list); lptr; lptr = g_list_next(lptr)) {
		tableNote = (NedTableNote *) lptr->data;
		opitch = tableNote->m_is_rest ? 111111 : tableNote->m_midi_pitch;
		if (npitch == opitch &&
			tableNote->m_start == nnote->m_start &&
			tableNote->m_stop == nnote->m_stop) {
			if (allow_doubles) {
				if (nnote->m_tie_forward) {
					nnote->m_tie_forward->m_tie_backward = NULL;
					nnote->m_tie_forward = NULL;
				} 
				if (nnote->m_tie_backward) {
					nnote->m_tie_backward->m_tie_forward = NULL;
					nnote->m_tie_backward = NULL;
				} 
				return list;
			}
			if (!nnote->m_is_rest || !nnote->m_is_rest) {
				
			NedResource::DbgMsg(DBG_TESTING, "tableNote(0x%x):start = %llu(%llu)(%llu), stop= %llu(%llu)(%llu), rest = %d, triplet = %d\n",
				tableNote, tableNote->m_start, tableNote->m_start / NOTE_64, tableNote->m_start / (NOTE_4 / 3),
				tableNote->m_stop, tableNote->m_stop / NOTE_64, tableNote->m_stop / (NOTE_4 / 3), tableNote->m_is_rest,
				tableNote->m_is_triplet);
			NedResource::DbgMsg(DBG_TESTING, "nnote(0x%x):start = %llu(%llu)(%llu), stop= %llu(%llu)(%llu), rest = %d, triplet = %d\n",
				nnote, nnote->m_start, nnote->m_start / NOTE_64, nnote->m_start / (NOTE_4 / 3),
				nnote->m_stop, nnote->m_stop / NOTE_64, nnote->m_stop / (NOTE_4 / 3), nnote->m_is_rest,
				nnote->m_is_triplet);
				NedResource::DbgMsg(DBG_TESTING, "double note\n");
	
			}
		}
	}
	return g_list_insert_sorted(list, nnote, cfunc);
}


	
	

NedTableNote *NedTableNote::clone_with_different_length(unsigned long long start, unsigned long long end) {
        NedTableNote *c = new NedTableNote(m_is_rest, start, end, m_midi_pitch, m_volume);
	c->m_line = m_line;
	c->m_state = m_state;
	c->m_closed = true;
	c->m_measure_start = m_measure_start;
	c->m_measure_end = m_measure_end;

        return c;
}

void NedTableNote::clone_chord(NedTableNote *pattern) {
	GList *lptr;
	NedTableNote *cn;
	NedTableNote *tableNote2;

        for (lptr = g_list_first(pattern->m_chord); lptr; lptr = g_list_next(lptr)) {
                tableNote2 = (NedTableNote *) lptr->data;
                cn = new NedTableNote(m_is_rest, m_start, m_stop, tableNote2->m_midi_pitch, m_volume);
		cn->m_line = tableNote2->m_line;
		cn->m_state = tableNote2->m_state;
		cn->m_closed = true;
                m_chord = g_list_append(m_chord, cn);
        }
}

NedTableNote *NedTableNote::snapNoteToMeasure(unsigned long long measure_start, unsigned long long measure_end,
		bool *changed) {
	unsigned long long duration;
	unsigned long long precision;
	unsigned long long dist;
	NedTableNote *c;
	*changed = false;

	m_measure_start = measure_start;
	m_measure_end = measure_end;
	duration = m_stop - m_start;
	if (duration >= WHOLE_NOTE) {
		precision = NOTE_2;
	}
	else if (duration >= NOTE_2) {
		precision = NOTE_4;
	}
	else if (duration >= NOTE_4) {
		precision = NOTE_8;
	}
	else if (duration >= NOTE_16) {
		precision = NOTE_32;
	}
	else {
		precision = NOTE_64;
	}
	if (m_start < measure_end && m_stop > m_measure_end) {
		dist = measure_end - m_start;
		if (dist < 0) dist = -dist;
		precision = NOTE_8;
		if (dist < precision && m_stop > measure_end + precision + NOTE_64) {
			m_start = measure_end;
			m_start_fix = true;
			*changed = true;
			return NULL;
		}
		if (m_stop - measure_end < 2 * precision) {
			m_stop = measure_end;
			return NULL;
		}
		c = clone(measure_end);
		m_stop = measure_end;
		if (!m_is_rest) {
			m_tie_forward = c;
			c->m_tie_backward = this;
		}
		m_end_fix = true;
		return c;
	}
	dist = measure_start - m_start;
	if (dist < 0) dist = -dist;
	if (dist < precision) {
		m_start = measure_start;
		m_start_fix = true;
	}
	dist = measure_end - m_stop;
	if (dist < 0) dist = -dist;
	if (dist < precision) {
		m_stop = measure_end;
		m_end_fix = true;
	}
	return NULL;
}


void NedTableNote::snapNote(unsigned long long next_start) {
	unsigned long long duration, dur2;
	unsigned long long precision;
	int measure_dist1, measure_dist2, fac;

	if (m_is_triplet) return;

	duration = m_stop - m_start;
	if (duration < NOTE_4) {
		if (m_end_fix) {
			dur2 = m_measure_end - m_start;
		}
		else {
			dur2 = next_start - m_start;
		}
		if (dur2 > duration) {
			if (dur2 / 2 > duration) {
				duration = dur2 / 2;
				m_stop = m_start + dur2 / 2;
			}
		}
	}
	if (duration >= WHOLE_NOTE) {
		precision = WHOLE_NOTE;
	}
	else if (duration >= NOTE_2) {
		precision = NOTE_2;
	}
	else if (duration >= NOTE_4) {
		precision = NOTE_4;
	}
	else if (duration >= NOTE_8) {
		precision = NOTE_8;
	}
	else if (duration >= NOTE_16) {
		precision = NOTE_16;
	}
	else if (duration >= NOTE_32) {
		precision = NOTE_32;
	}
	else {
		precision = NOTE_64;
	}

	if (!m_start_fix) {
		measure_dist1 = m_start - m_measure_start;
		fac = (measure_dist1 + precision / 2) / precision;
		measure_dist1 = fac * precision;
		m_start = m_measure_start + measure_dist1;
		if (m_start < m_measure_start) m_start = m_measure_start;
	}
	if (!m_end_fix) {
		measure_dist2 = m_stop - m_measure_start;
		fac = (measure_dist2 + precision / 2) / precision;
		measure_dist2 = fac * precision;
		m_stop = m_measure_start + measure_dist2;
		if (m_stop <= m_start && precision < NOTE_16) m_stop = m_start + NOTE_64;
		if (m_stop > m_measure_end ) m_stop = m_measure_end;
	}
}

bool NedTableNote::snapWouldFit() {
	unsigned long long duration;
	unsigned long long precision;
	unsigned long long start, stop;
	unsigned long long dist;

	duration = m_stop - m_start;

	if (duration >= WHOLE_NOTE) {
		precision = NOTE_4;
	}
	else if (duration >= NOTE_2) {
		precision = NOTE_8;
	}
	else if (duration >= NOTE_4) {
		precision = NOTE_16;
	}
	else if (duration >= NOTE_16) {
		precision = NOTE_32;
	}
	else {
		precision = NOTE_64;
	}

	start = (m_start + precision / 2) / precision;
	start *= precision;
	stop = (m_stop + precision / 2) / precision;
	stop *= precision;
	dist = start - m_start;
	if (dist < 0) dist = -dist;
	if (16 * dist > duration) return false;
	dist = stop - m_stop;
	if (dist < 0) dist = -dist;
	if (16 * dist > duration) return false;
	return true;
}

void NedTableNote::tripletRecognition(unsigned long long  base) {
	unsigned long long duration;
	unsigned long long f, next12;
	unsigned long long dist;

	if (m_is_triplet) return;
	if (snapWouldFit()) return;
	duration = m_stop - m_start;
	if (duration < base / 3) return;
	f = (m_start + base / 12) / (base / 6);
	next12 = f * (base / 6);
	if (next12 % (base / 2) != 0) {
		dist = m_start - next12;
		if (dist < 0) dist = -dist;
		if (dist < base / 12) {
			m_triplet_start = next12;
			m_tr_start = true;
			NedResource::DbgMsg(DBG_TESTING, "base = %llu markiere start von 0x%x als Triplet\n", base / NOTE_4, this);
		}
	}
	f = (m_stop + base / 12) / (base / 6);
	next12 = f * (base / 6);
	if (next12 % (base / 2) != 0) {
		dist = m_stop - next12;
		if (dist < 0) dist = -dist;
		if (dist < base / 12) {
			m_triplet_stop = next12;
			NedResource::DbgMsg(DBG_TESTING, "base = %llu markiere stop von 0x%x als Triplet\n", base / NOTE_4, this);
			m_tr_stop = true;
		}
	}
}

bool NedTableNote::overlapping(NedTableNote *other) {
	if (m_start == other->m_start && m_stop == other->m_stop) return false;
	if (m_start >= other->m_stop) return false;
	if (m_stop <= other->m_start) return false;
	return true;
}

GList *NedTableNote::getGroup(int *av_midi_pitch) {
	GList *group = NULL;
	GList *lptr, *lptr2;
	NedTableNote *tableNote;
	bool inserted;
	double av_pitch;

	group = g_list_append(group, this);
	NedResource::DbgMsg(DBG_TESTING, "(0) Fuege ein: 0x%x\n", this);

	inserted = true;
	while (inserted) {
		inserted = false;
		for (lptr = g_list_first(group); lptr; lptr = g_list_next(lptr)) {
			tableNote = (NedTableNote *) lptr->data;
			if (tableNote->m_tie_forward != NULL) {
				if (g_list_find(group, tableNote->m_tie_forward) == NULL && !tableNote->m_tie_forward->m_is_chord_member) {
					NedResource::DbgMsg(DBG_TESTING, "(1) Fuege in  0x%x ein: 0x%x, m_is_triplet = %d, m_is_rest = %d\n", tableNote, tableNote->m_tie_forward,
						tableNote->m_is_triplet, tableNote->m_is_rest);
					group = g_list_append(group, tableNote->m_tie_forward);
					inserted = true;
				}
			}
			if (tableNote->m_tie_backward != NULL) {
				if (g_list_find(group, tableNote->m_tie_backward) == NULL && !tableNote->m_tie_backward->m_is_chord_member) {
					NedResource::DbgMsg(DBG_TESTING, "(2) Fuege ein: 0x%x\n", tableNote->m_tie_backward);
					group = g_list_append(group, tableNote->m_tie_backward);
					inserted = true;
				}
			}
			if (tableNote->m_tuplet_ptr != NULL) {
				NedResource::DbgMsg(DBG_TESTING, "tableNote(0x%x)->m_tuplet_ptr 0x%x\n", tableNote, tableNote->m_tuplet_ptr); fflush(stdout);
				for (lptr2 = g_list_first(tableNote->m_tuplet_ptr->m_members); lptr2; lptr2 = g_list_next(lptr2)) {
					if (g_list_find(group, lptr2->data) == NULL && !((NedTableNote *) lptr2->data)->m_is_chord_member) {
						NedResource::DbgMsg(DBG_TESTING, "(1) Fuege ein: 0x%x\n", lptr2->data);
						group = g_list_append(group, lptr2->data);
						inserted = true;
					}
				}
			}
		}
	}

	av_pitch = 0.0;
	for (lptr = g_list_first(group); lptr; lptr = g_list_next(lptr)) {
		av_pitch+= ((NedTableNote *) lptr->data)->m_midi_pitch;
	}
	*av_midi_pitch = (int) ((av_pitch / (double) g_list_length(group)) + 0.5);
	return group;
}

void NedTableNote::print() {
	NedResource::DbgMsg(DBG_TESTING, "pitch = %d, vol = %d, start = %llu(%llu), stop = %llu(%llu)\n",
		m_midi_pitch, m_volume, m_start, m_start / NOTE_4, m_stop, m_stop / NOTE_4);
}

NedTableInstrument::NedTableInstrument(int midi_nr) :
	m_tuplet_list(NULL), m_midi_channel(0), m_midi_volume(64), m_keysig(0), m_note_list(NULL), m_midi_nr(midi_nr), m_min_pitch(10000), m_max_pitch(0),
	m_clef(TREBLE_CLEF), m_octave_shift(0), m_staves(0), m_av_pitch(60) {
	int i;

	for (i = 0; i < 4; m_slots[i++] = false);
	for (i = 0; i < 2; m_piano_slots[i++] = false);
	m_instrument_name[0] = '\0';
}

NedTableInstrument::~NedTableInstrument() {
	GList *track, *lptr, *lptr2;

	for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		track = (GList *) lptr->data;	
		if (track == NULL) continue;
		for (lptr2 = g_list_first(track); lptr2; lptr2 = g_list_next(lptr2)) {
			delete (NedTableNote *) lptr2->data;
		}
		g_list_free(track);
	}
	if (m_note_list != NULL) {
		g_list_free(m_note_list);
	}
	m_note_list = NULL;
}

void NedTableInstrument::addNote(NedTableNote *note) {
	GList *track;
	if (m_note_list == NULL) {
		track = NULL;
		track = g_list_append(track, note);
		m_note_list = g_list_append(m_note_list, track);
	}
	else {
		track = (GList *) g_list_first(m_note_list)->data;	
		track = NedTableNote::my_insert_sorted(track, note,
			 (gint (*)(const void*, const void*)) NedTableNote::compare, true);
		g_list_first(m_note_list)->data = (void *) track;
	}
}

bool NedTableInstrument::closeNote(int pitch, unsigned long long end_time) {
	GList *track, *lptr2;

	//for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		track = (GList *) g_list_first(m_note_list)->data;	
		if (track == NULL) return false;
		for (lptr2 = g_list_first(track); lptr2; lptr2 = g_list_next(lptr2)) {
			if (((NedTableNote *) lptr2->data)->m_midi_pitch == pitch && 
				((NedTableNote *) lptr2->data)->m_stop == 0) {
				((NedTableNote *) lptr2->data)->m_ori_stop =
				((NedTableNote *) lptr2->data)->m_stop = end_time;
				((NedTableNote *) lptr2->data)->m_closed = true;
				return true;
			}
		}
		g_list_first(m_note_list)->data = (void *) track;
	//}
	return false;
}

void NedTableInstrument::print() {
	GList *track, *lptr, *lptr2;

	for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		track = (GList *) lptr->data;	
		if (track == NULL) continue;
		for (lptr2 = g_list_first(track); lptr2; lptr2 = g_list_next(lptr2)) {
			((NedTableNote *) lptr2->data)->print();
		}
	}
}

void NedTableInstrument::resort() {
	GList *track;

	//for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		if (g_list_first(m_note_list) == NULL) return;
		track = (GList *) g_list_first(m_note_list)->data;	
		if (track == NULL) return;
		track = g_list_sort(track, (gint (*)(const void*, const void*)) NedTableNote::compare);
		g_list_first(m_note_list)->data = (void *) track;
	//}
}

void NedTableInstrument::snapNotesToMeasures(bool onlyRest, NedInstrumentTable *table) {
	GList *track, *lptr2;
	unsigned long long measure_start, measure_end;
	NedTableNote *table_note, *c;
	bool changed;

	//for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		if (g_list_first(m_note_list) == NULL) return;
		track = (GList *) g_list_first(m_note_list)->data;	
		if (track == NULL) return;
		lptr2 = g_list_first(track);
		while(lptr2) {
			table_note = (NedTableNote *) lptr2->data;
			if (onlyRest && !table_note->m_is_rest) continue;
			table->getMeasureDurationAt(table_note->m_start, &measure_start, &measure_end);
			c = table_note->snapNoteToMeasure(measure_start, measure_end, &changed);
			if (c != NULL) {
				track = NedTableNote::my_insert_sorted(track, c,
			 		(gint (*)(const void*, const void*)) NedTableNote::compare, true);
				if ((lptr2 = g_list_find(track, table_note)) == NULL) {
					NedResource::Abort("NedTableInstrument::snapNotesToMeasures");
				}
			}
			else if (!changed) {
				lptr2 = g_list_next(lptr2);
			}
			else {
				track = g_list_sort(track, (gint (*)(const void*, const void*)) NedTableNote::compare);
				lptr2 = g_list_first(track);
			}
		}	
		g_list_first(m_note_list)->data = (void *) track;
	//}
}

void NedTableInstrument::tripletRecognition(unsigned long long  base) {
	GList *lptr2;
	GList *track;

	//for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		if (g_list_first(m_note_list) == NULL) return;
		track = (GList *) g_list_first(m_note_list)->data;	
		if (track == NULL) return;
		for (lptr2 = g_list_first(track); lptr2; lptr2 = g_list_next(lptr2)) {
			((NedTableNote *) lptr2->data)->tripletRecognition(base);
		}
	//}
}


void NedTableInstrument::combinePossibleTripletChords() {
	GList *lptr2, *lptr3;
	GList *track;
	NedTableNote *tableNote1, *tableNote2;
	bool chord_found;

	//for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		if (g_list_first(m_note_list) == NULL) return;
		track = (GList *) g_list_first(m_note_list)->data;	
		if (track == NULL) return;
		for (lptr2 = g_list_first(track); lptr2; lptr2 = g_list_next(lptr2)) {
			tableNote1 = (NedTableNote *) lptr2->data;
			if (tableNote1->m_handledAsTuplet) continue;
			chord_found = false;
			for (lptr3 = g_list_next(lptr2); lptr3; lptr3 = g_list_next(lptr3)) {
				tableNote2 = (NedTableNote *) lptr3->data;
				if (tableNote2->m_handledAsTuplet) continue;
				if (tableNote2->m_triplet_start >= tableNote1->m_triplet_stop + 2 * WHOLE_NOTE) break;
				if (tableNote1->m_tr_start && tableNote1->m_tr_stop) {
					if (!tableNote2->m_tr_start || !tableNote2->m_tr_stop) continue;
					if (tableNote1->m_triplet_start != tableNote2->m_triplet_start) continue;
					if (tableNote1->m_triplet_stop != tableNote2->m_triplet_stop) continue;
					tableNote1->m_chord = g_list_append(tableNote1->m_chord, tableNote2);
					chord_found = true;
				}
				else if (tableNote1->m_tr_start) {
					if (tableNote2->m_handledAsTuplet) continue;
					if (!tableNote2->m_tr_start) continue;
					if (tableNote1->m_triplet_start != tableNote2->m_triplet_start) continue;
					tableNote1->m_chord = g_list_append(tableNote1->m_chord, tableNote2);
					chord_found = true;
				}
				else if (tableNote1->m_tr_stop) {
					if (tableNote2->m_handledAsTuplet) continue;
					if (!tableNote2->m_tr_stop) continue;
					if (tableNote1->m_triplet_stop != tableNote2->m_triplet_stop) continue;
					tableNote1->m_chord = g_list_append(tableNote1->m_chord, tableNote2);
					chord_found = true;
				}
			}
			if (chord_found) {
				for (lptr3 = g_list_first(tableNote1->m_chord); lptr3; lptr3 = g_list_next(lptr3)) {
					((NedTableNote *)lptr3->data)->m_is_chord_member = true;
					track = g_list_remove(track, lptr3->data);
				}
				if ((lptr2 = g_list_find(track, tableNote1)) == NULL) {
					NedResource::Abort("NedTableInstrument::combinePossibleTripletChords");
				}
				NedResource::DbgMsg(DBG_TESTING, "\n");
			}
		}
		g_list_first(m_note_list)->data = (void *) track;
	//}
}


void NedTableInstrument::buildTriplets(unsigned long long base, bool allow_single_notes) {
	GList *track, *lptr2, *lptr3, *lptr4;
	NedTableNote *tableNote;
	NedTableTuplet *table_tuplet;
	GList *tmembers;
	unsigned long long t0 = 0, t3 = 0;
	bool avoid_tuplet, av_tuplet;


	//for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		if (g_list_first(m_note_list) == NULL) return;
		track = (GList *) g_list_first(m_note_list)->data;	
		if (track == NULL) return;
		for (lptr2 = g_list_first(track); lptr2; lptr2 = g_list_next(lptr2)) {
			tableNote = (NedTableNote *) lptr2->data;
			if ((tableNote->m_tr_start || tableNote->m_tr_stop) && !tableNote->m_takenForTuplet) {
				av_tuplet = false;
				tmembers = NULL;
				tmembers = g_list_append(tmembers, tableNote);
				tableNote->m_takenForTuplet = true;
				if (tableNote->m_tr_start && tableNote->m_tr_stop) {
					if (base / 3 > tableNote->m_triplet_start) av_tuplet = true;
					t0 = tableNote->m_triplet_start - base / 3;
					t3 = tableNote->m_triplet_stop + base / 3;
				}
				else if (tableNote->m_tr_start) {
					t3 = tableNote->m_triplet_start + base / 3;
					if ((t3 % (base / 2)) != 0) t3 += base / 3;
					if (base > t3) av_tuplet = true;
					t0 = t3 - base;
				}
				else if (tableNote->m_tr_stop) {
					if (base / 3 > tableNote->m_triplet_stop) av_tuplet = true;
					t0 = tableNote->m_triplet_stop - base / 3;
					if (base / 3 > t0) av_tuplet = true;
					if ((t0 % (base / 2)) != 0) t0 -= base / 3;
					t3 = t0 + base;
				}
				else {
					NedResource::Abort("NedTableInstrument::buildTriplets");
				}
				while (searchForTupletMembers(&tmembers, lptr2, t0 + base / 3, t3 - base / 3, tableNote->m_measure_start));
				if (allow_single_notes) {
					avoid_tuplet = ((NedTableNote *) g_list_first(tmembers)->data)->getDuration() < (base / 6) + (base / 24);
					if (g_list_length(tmembers) < 2) {
						if (!((NedTableNote *) g_list_first(tmembers)->data)->m_tr_start) avoid_tuplet = true;
					}
				}
				else {
					avoid_tuplet = g_list_length(tmembers) < 2;
				}
				if (t0 < tableNote->m_measure_start) {
					avoid_tuplet = true;
				}
				if (t3 > tableNote->m_measure_end) {
					avoid_tuplet = true;
				}
				for (lptr4 = g_list_first(m_tuplet_list); lptr4; lptr4 = g_list_next(lptr4)) {
					if (((NedTableTuplet *) lptr4->data)->overlaps(t0, t3)) {
						avoid_tuplet = true;
						break;
					}
				}
				if (av_tuplet) {
					avoid_tuplet = true;
				}
				if (!avoid_tuplet) {
					table_tuplet = new NedTableTuplet(tmembers, t0, t3);
					m_tuplet_list = g_list_append(m_tuplet_list, table_tuplet);
					NedResource::DbgMsg(DBG_TESTING, "new NedTableTuplet tuplet = 0x%x, members = 0x%x\n", table_tuplet, tmembers);
					for (lptr3 = g_list_first(tmembers); lptr3; lptr3 = g_list_next(lptr3)) {
						tableNote = (NedTableNote *) lptr3->data;
						
						tableNote->m_tuplet_ptr = table_tuplet;
						NedResource::DbgMsg(DBG_TESTING, "Stelle A: tableNote->m_tuplet_ptr = 0x%x\n", tableNote->m_tuplet_ptr);
						tableNote->m_is_triplet = true;

						if (tableNote->m_tr_start) {
							tableNote->m_start = tableNote->m_triplet_start;
						}
						if (tableNote->m_tr_stop) {
							tableNote->m_stop = tableNote->m_triplet_stop;
						}
					}
				}
				else {
					for (lptr3 = g_list_first(tmembers); lptr3; lptr3 = g_list_next(lptr3)) {
						((NedTableNote *) lptr3->data)->m_takenForTuplet = false;
						if (((NedTableNote *) lptr3->data)->m_chord != NULL) {
							for (lptr4 = g_list_first(((NedTableNote *) lptr3->data)->m_chord); lptr4; lptr4 = g_list_next(lptr4)) {
								((NedTableNote *) lptr4->data)->m_takenForTuplet = false;
								((NedTableNote *) lptr4->data)->m_is_chord_member = false;
								track = NedTableNote::my_insert_sorted(track, (NedTableNote *) lptr4->data, 
									(gint (*)(const void*, const void*)) NedTableNote::compare, false);
							}
							g_list_free(((NedTableNote *) lptr3->data)->m_chord);
							((NedTableNote *) lptr3->data)->m_chord = NULL;
						}
					}
				}
			}
		}
		g_list_first(m_note_list)->data = (void *) track;
	//}
}

#ifdef ORI
bool NedTableInstrument::searchForTupletMembers(GList **table_tuplet, GList *lptr, unsigned long long t1, 
				unsigned long long t2, int meas_num) {
	GList *lptr2;
	NedTableNote *tableNote;
	bool ret = false;

	for (lptr2 = g_list_previous(lptr); lptr2; lptr2 = g_list_previous(lptr2)) {
		tableNote = (NedTableNote *) lptr2->data;
		if (tableNote->m_measure_number != meas_num) break;
		if (!tableNote->m_tr_start && !tableNote->m_tr_stop) continue;
		if (tableNote->m_takenForTuplet) continue;
		if (((tableNote->m_tr_start && (tableNote->m_triplet_start == t1 || tableNote->m_triplet_start == t2)) ||
		    (tableNote->m_tr_stop && (tableNote->m_triplet_stop == t1 || tableNote->m_triplet_stop == t2))) &&
		    	!overlapping(*table_tuplet, tableNote)) {
			*table_tuplet = NedTableNote::my_insert_sorted(*table_tuplet, tableNote,
				(gint (*)(const void*, const void*)) NedTableNote::compare, false);
			tableNote->m_takenForTuplet = true;
			ret = true;
		}
	}
	if (!ret) {
		for (lptr2 = g_list_next(lptr); lptr2; lptr2 = g_list_next(lptr2)) {
			tableNote = (NedTableNote *) lptr2->data;
			if (tableNote->m_measure_number != meas_num) break;
			if (!tableNote->m_tr_start && !tableNote->m_tr_stop) continue;
			if (tableNote->m_takenForTuplet) continue;
			if (((tableNote->m_tr_start && (tableNote->m_triplet_start == t1 || tableNote->m_triplet_start == t2)) ||
		    	(tableNote->m_tr_stop && (tableNote->m_triplet_stop == t1 || tableNote->m_triplet_stop == t2))) &&
		    		!overlapping(*table_tuplet, tableNote)) {
				*table_tuplet = NedTableNote::my_insert_sorted(*table_tuplet, tableNote,
					(gint (*)(const void*, const void*)) NedTableNote::compare, false);
				tableNote->m_takenForTuplet = true;
				ret = true;
			}
		}
	}
	return ret;
}
#else
bool NedTableInstrument::searchForTupletMembers(GList **table_tuplet, GList *lptr, unsigned long long t1, 
				unsigned long long t2, unsigned long long measure_start) {
	GList *lptr1, *lptr2;
	NedTableNote *tableNote1, *tableNote2;
	bool ret = false;

	for (lptr1 = g_list_first(*table_tuplet); lptr1; lptr1 = g_list_next(lptr1)) {
		tableNote1 = (NedTableNote *) lptr1->data;
		if (tableNote1->m_tr_start) {
			for (lptr2 = g_list_previous(lptr); lptr2; lptr2 = g_list_previous(lptr2)) {
				tableNote2 = (NedTableNote *) lptr2->data;
				if (tableNote2->m_measure_start != tableNote1->m_measure_start) break;
				if (!tableNote2->m_tr_start && !tableNote2->m_tr_stop) continue;
				if (tableNote2->m_takenForTuplet) continue;
				if (!tableNote2->m_tr_stop || tableNote2->m_triplet_stop != tableNote1->m_triplet_start) continue;
				*table_tuplet = NedTableNote::my_insert_sorted(*table_tuplet, tableNote2,
						(gint (*)(const void*, const void*)) NedTableNote::compare, false);
				tableNote2->m_takenForTuplet = true;
				ret = true;
			}
		}
		if (tableNote1->m_tr_stop) {
			for (lptr2 = g_list_next(lptr); lptr2; lptr2 = g_list_next(lptr2)) {
				tableNote2 = (NedTableNote *) lptr2->data;
				if (tableNote2->m_measure_start != tableNote1->m_measure_start) break;
				if (!tableNote2->m_tr_start && !tableNote2->m_tr_stop) continue;
				if (tableNote2->m_takenForTuplet) continue;
				if (!tableNote2->m_tr_start || tableNote2->m_triplet_start != tableNote1->m_triplet_stop) continue;
				*table_tuplet = NedTableNote::my_insert_sorted(*table_tuplet, tableNote2,
						(gint (*)(const void*, const void*)) NedTableNote::compare, false);
				tableNote2->m_takenForTuplet = true;
				ret = true;
			}
		}
	}
	return ret;
}
#endif

bool NedTableInstrument::overlapping(GList *table_tuplet, NedTableNote *tabletNote) {
	GList *lptr;
	NedTableNote *tabletNote2;


	for (lptr = g_list_first(table_tuplet); lptr; lptr = g_list_next(lptr)) {
		tabletNote2 = (NedTableNote *) lptr->data;
		if (tabletNote->overlapping(tabletNote2)) return true;
	}

	return false;
}

void NedTableInstrument::handleTuplets(unsigned long long  base) {
	GList *track, *lptr2, *lptr3;
	NedTableNote *tableNote, *next;
	NedTableTuplet *table_tuplet;
	unsigned long long start_time, end_time, j;
	NedTableNote *first_in_tuplet, *last_in_tuplet;
	NedTableNote *note, *rest;
	unsigned long long dist;

	//for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		if (g_list_first(m_note_list) == NULL) return;
		track = (GList *) g_list_first(m_note_list)->data;	
		if (track == NULL) return;
		for (lptr2 = g_list_first(track); lptr2; lptr2 = g_list_next(lptr2)) {
			next = tableNote = (NedTableNote *) lptr2->data;
			if (!tableNote->m_is_triplet) continue;
			if (tableNote->m_handledAsTuplet) continue;
			NedResource::DbgMsg(DBG_TESTING, "handleTuplets(1): tableNote = 0x%x\n", tableNote);
			table_tuplet = tableNote->m_tuplet_ptr;
			first_in_tuplet = (NedTableNote *) g_list_first(table_tuplet->m_members)->data;
			start_time = table_tuplet->m_t0;
			j = (start_time + base / 4) / (base / 2);
			start_time = j * (base / 2);
			dist = start_time - first_in_tuplet->m_start;
			if (dist < 0) dist = -dist;
			if (dist < (base / 4)) {
				first_in_tuplet->m_start = start_time;
			}
			if (start_time < first_in_tuplet->m_start) {
				rest = new NedTableNote(true, start_time, first_in_tuplet->m_start, 0 /* dummy */, 0 /* dummy */);
				NedResource::DbgMsg(DBG_TESTING, "handleTuplets(2):  rest = 0x%x\n", rest);
				table_tuplet->m_members = NedTableNote::my_insert_sorted(table_tuplet->m_members, rest, 
								(gint (*)(const void*, const void*)) NedTableNote::compare, false);
				track = NedTableNote::my_insert_sorted(track, rest, (gint (*)(const void*, const void*)) NedTableNote::compare, false);
			}
			else if (start_time > first_in_tuplet->m_start) {
				note = first_in_tuplet->clone_with_different_length(first_in_tuplet->m_start, start_time);
				NedResource::DbgMsg(DBG_TESTING, "handleTuplets(3):  note = 0x%x\n", note);
				first_in_tuplet->m_start = start_time;
				if (first_in_tuplet->m_start >= first_in_tuplet->m_stop) {
					table_tuplet->m_bad = true;
					NedResource::DbgMsg(DBG_TESTING, "Bad Tuplet note(2)\n");
				}
				track = NedTableNote::my_insert_sorted(track, note, (gint (*)(const void*, const void*)) NedTableNote::compare, false);
				if (first_in_tuplet->m_tie_backward != NULL) {
					note->m_tie_backward = first_in_tuplet->m_tie_backward;
					first_in_tuplet->m_tie_backward->m_tie_forward = note;
				}
				note->m_tie_forward = first_in_tuplet;
				note->m_triplet_split_note = true;
				NedResource::DbgMsg(DBG_TESTING, "handleTuplets(3a):  note->m_tie_forward =0x%x\n", note->m_tie_forward);
				first_in_tuplet->m_tie_backward = note;
				NedResource::DbgMsg(DBG_TESTING, "Stelle Y: m_tie_forward = 0x%x\n", note->m_tie_forward);
			}
			last_in_tuplet = (NedTableNote *) g_list_last(table_tuplet->m_members)->data;
			end_time = table_tuplet->m_t2;
			j = (end_time + base / 4) / (base / 2);
			end_time = j * (base / 2);
			dist = end_time - last_in_tuplet->m_stop;
			if (dist < 0) dist = -dist;
			if (dist < base / 4) {
				last_in_tuplet->m_stop = end_time;
			}
			if (end_time > last_in_tuplet->m_stop) {
				rest = new NedTableNote(true, last_in_tuplet->m_stop, end_time, 0 /* dummy */, 0 /* dummy */);
				NedResource::DbgMsg(DBG_TESTING, "handleTuplets(4):  rest = 0x%x\n", rest);
				table_tuplet->m_members = NedTableNote::my_insert_sorted(table_tuplet->m_members, rest, 
								(gint (*)(const void*, const void*)) NedTableNote::compare, false);
				track = NedTableNote::my_insert_sorted(track, rest, (gint (*)(const void*, const void*)) NedTableNote::compare, false);
			}
			else if (end_time < last_in_tuplet->m_stop) {
				note = last_in_tuplet->clone_with_different_length(end_time, last_in_tuplet->m_stop);
				NedResource::DbgMsg(DBG_TESTING, "handleTuplets(5):  note = 0x%x\n", note);
				last_in_tuplet->m_stop = end_time;
				if (last_in_tuplet->m_start >= last_in_tuplet->m_stop) {
					table_tuplet->m_bad = true;
					NedResource::DbgMsg(DBG_TESTING, "Bad Tuplet note(3), 0x%x m_handledAsTuplet = %d\n", last_in_tuplet, last_in_tuplet->m_handledAsTuplet);
				}
				track = NedTableNote::my_insert_sorted(track, note, (gint (*)(const void*, const void*)) NedTableNote::compare, false);
				if (last_in_tuplet->m_tie_forward != NULL) {
					note->m_tie_forward = last_in_tuplet->m_tie_forward;
					last_in_tuplet->m_tie_forward->m_tie_backward = note;
				}
				note->m_tie_backward = last_in_tuplet;
				note->m_triplet_split_note = true;
				last_in_tuplet->m_tie_forward = note;
				NedResource::DbgMsg(DBG_TESTING, "Stelle Z: m_tie_forward = 0x%x\n", last_in_tuplet->m_tie_forward);
			}
			if (g_list_length(table_tuplet->m_members) > 1) {
				NedResource::DbgMsg(DBG_TESTING, "handleTuplets(6) members:\n");
				for (lptr3 = g_list_first(table_tuplet->m_members); lptr3; lptr3 = g_list_next(lptr3)) {
					tableNote = (NedTableNote *) lptr3->data;
					tableNote->m_is_triplet = true;
					tableNote->m_handledAsTuplet = true;
					tableNote->m_tuplet_ptr = table_tuplet;
					if (tableNote->m_start >= tableNote->m_stop) {
						NedResource::DbgMsg(DBG_TESTING, "Bad Tuplet note(4)\n");
					}
					NedResource::DbgMsg(DBG_TESTING, "Stelle A: tableNote->m_tuplet_ptr = 0x%x\n", tableNote->m_tuplet_ptr);
					NedResource::DbgMsg(DBG_TESTING, "handleTuplets(7): 0x%x, m_start = %llu(%llu)(%llu), m_stop = %llu(%llu)(%llu), m_is_rest = %d, m_is_triplet = %d\n",
						lptr3->data, tableNote->m_start, tableNote->m_start / NOTE_64, tableNote->m_start / (NOTE_4 / 3),
						tableNote->m_stop, tableNote->m_stop / NOTE_64, tableNote->m_stop / (NOTE_4 / 3), tableNote->m_is_rest, tableNote->m_is_triplet);
				}
				NedResource::DbgMsg(DBG_TESTING, "handleTuplets(8)\n");
				if ((lptr2 = g_list_find(track, next)) == NULL) {
					NedResource::Abort("NedTableInstrument::handleTuplets");
				}
			}
		}
		g_list_first(m_note_list)->data = (void *) track;
	//}
}

void NedTableInstrument::freeUnhandledTuplets() {
	GList *track, *lptr2, *lptr3;
	NedTableNote *tableNote1, *tableNote2;
	bool bad;

	//for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		if (g_list_first(m_note_list) == NULL) return;
		track = (GList *) g_list_first(m_note_list)->data;	
		if (track == NULL) return;
		for (lptr2 = g_list_first(track); lptr2; lptr2 = g_list_next(lptr2)) {
			bad = false;
			tableNote1 = (NedTableNote *) lptr2->data;
			if (tableNote1->m_start >= tableNote1->m_stop) {
				NedResource::DbgMsg(DBG_TESTING, "freeUnhandledTuplets(0x%x): m_start(2) = %llu, m_stop = %llu, m_is_triplet = %d, m_handledAsTuplet = %d\n",
								tableNote1, tableNote1->m_start / NOTE_4, tableNote1->m_stop/ NOTE_4, tableNote1->m_is_triplet, tableNote1->m_handledAsTuplet);
				bad = true;
			}
			if (!tableNote1->m_is_triplet) continue;
			if (tableNote1->m_handledAsTuplet && !tableNote1->m_tuplet_ptr->m_bad) continue;
			if (tableNote1->m_triplet_split_note) {
				if (tableNote1->m_tie_backward != NULL) {
					tableNote1->m_tie_backward->m_tie_forward = NULL;
					tableNote1->m_tie_backward = NULL;
				}
				if (tableNote1->m_tie_forward != NULL) {
					tableNote1->m_tie_forward->m_tie_backward = NULL;
					tableNote1->m_tie_forward = NULL;
				}
				lptr2 = g_list_previous(lptr2);
				track = g_list_remove(track, tableNote1);
				if (lptr2 == NULL) {
					lptr2 = g_list_first(track);
				}
				continue;
			}
			if (tableNote1->m_is_rest) {
				lptr2 = g_list_previous(lptr2);
				track = g_list_remove(track, tableNote1);
				if (lptr2 == NULL) {
					lptr2 = g_list_first(track);
				}
				continue;
			}
			tableNote1->m_is_triplet = 
			tableNote1->m_tr_stop =
			tableNote1->m_takenForTuplet = 
			tableNote1->m_tr_start = false;
			tableNote1->m_chord = NULL;
			tableNote1->m_start = tableNote1->m_ori_start;
			tableNote1->m_stop = tableNote1->m_ori_stop;
			NedResource::DbgMsg(DBG_TESTING, "De-Markiere 0x%x\n", tableNote1);
			if (tableNote1->m_chord != NULL) {
				for (lptr3 = g_list_first(tableNote1->m_chord); lptr3; lptr3 = g_list_next(lptr3)) {
					tableNote2 = (NedTableNote *) lptr3->data;
					if (tableNote2->m_is_rest) continue;
					if (tableNote2->m_triplet_split_note) {
						if (tableNote2->m_tie_backward != NULL) {
							tableNote2->m_tie_backward->m_tie_forward = NULL;
							tableNote2->m_tie_backward = NULL;
						}
						if (tableNote2->m_tie_forward != NULL) {
							tableNote2->m_tie_forward->m_tie_backward = NULL;
							tableNote2->m_tie_forward = NULL;
						}
						continue;
					}
					tableNote2->m_is_triplet =
					tableNote2->m_is_chord_member =
					tableNote2->m_takenForTuplet = 
					tableNote2->m_tr_start =
					tableNote2->m_tr_stop = false;
					tableNote2->m_start = tableNote2->m_ori_start;
					tableNote2->m_stop = tableNote2->m_ori_stop;
					track = NedTableNote::my_insert_sorted(track, tableNote2,
						(gint (*)(const void*, const void*)) NedTableNote::compare, false);
					if ((lptr2 = g_list_find(track, tableNote1)) == NULL) {
						NedResource::Abort("NedTableInstrument::freeUnhandledTuplets");
					}
				}
				g_list_free(tableNote1->m_chord);
				tableNote1->m_chord = NULL;
			}
		}
		g_list_first(m_note_list)->data = (void *) track;
	//}
}

void NedTableInstrument::insertRests() {
	GList *lptr, *lptr2, *lptr3;
	GList *track;
	NedTableNote *tableNote2, *tableNote3;
	NedTableNote *rest;

	for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		track = (GList *) lptr->data;
		if (track == NULL) continue;
		lptr2 = g_list_first(track);
		if (lptr2 == NULL) continue;
		lptr3 = g_list_next(lptr2);
		tableNote2 = (NedTableNote *) lptr2->data;
		if (tableNote2->m_start >= NOTE_64) {
			rest = new NedTableNote(true, 0, tableNote2->m_start, 0 /* dummy */, 0 /* dummy */);
			if (tableNote2->m_tr_start) {
				rest->m_is_triplet = true;
				rest->m_tr_start = true;
				rest->m_takenForTuplet = rest->m_handledAsTuplet = true;
			}
			track = NedTableNote::my_insert_sorted(track, rest, (gint (*)(const void*, const void*)) NedTableNote::compare, false);
			if ((lptr3 = g_list_find(track, rest)) == NULL) {
				NedResource::Abort("NedTableInstrument::insertRests");
			}
			lptr3 = g_list_next(lptr3);
		}
		while (lptr3 != NULL) {
		//for (;lptr3; lptr2 = lptr3, lptr3 = g_list_next(lptr3)) {
			tableNote2 = (NedTableNote *) lptr2->data;
			tableNote3 = (NedTableNote *) lptr3->data;
			/*
			NedResource::DbgMsg(DBG_TESTING, "lptr2 = 0x%x, tableNote2(0x%x)->m_start = %llu(%llu), tableNote2->m_stop = %llu(%llu), rest = %d\n",
				lptr2, tableNote2, tableNote2->m_start, tableNote2->m_start / NOTE_4, tableNote2->m_stop, tableNote2->m_stop / NOTE_4, tableNote2->m_is_rest);
			NedResource::DbgMsg(DBG_TESTING, "lptr3 = 0x%x, tableNote3(0x%x)->m_start = %llu(%llu), tableNote3->m_stop = %llu(%llu), rest = %d\n",
				lptr3, tableNote3, tableNote3->m_start, tableNote3->m_start / NOTE_4, tableNote3->m_stop, tableNote3->m_stop / NOTE_4, tableNote3->m_is_rest);
				*/
			if (tableNote2->m_stop + NOTE_64 <= tableNote3->m_start) {
				rest = new NedTableNote(true, tableNote2->m_stop, tableNote3->m_start, 0 /* dummy */, 0 /* dummy */);
				//NedResource::DbgMsg(DBG_TESTING, "0x%x created\n", rest);
				if (tableNote2->m_tr_stop) {
					rest->m_is_triplet = true;
					rest->m_tr_start = true;
					rest->m_takenForTuplet = rest->m_handledAsTuplet = true;
				}
				if (tableNote3->m_tr_start) {
					rest->m_is_triplet = true;
					rest->m_tr_stop = true;
					rest->m_takenForTuplet = rest->m_handledAsTuplet = true;
				}
				track = NedTableNote::my_insert_sorted(track, rest, (gint (*)(const void*, const void*)) NedTableNote::compare, false);
				if ((lptr3 = g_list_find(track, tableNote3)) == NULL) {
					NedResource::Abort("NedTableInstrument::insertRests");
				}
				//NedResource::DbgMsg(DBG_TESTING, "eingefuegt(0x%x) an %llu\n", rest, tableNote2->getDuration());
			}
			lptr2 = lptr3;
			//NedResource::DbgMsg(DBG_TESTING, "davir: lptr3 = 0x%x\n", lptr3);
			lptr3 = g_list_next(lptr3);
			//NedResource::DbgMsg(DBG_TESTING, "danach: lptr3 = 0x%x\n", lptr3);

		}
		lptr->data = (void *) track;
	}
}

void NedTableInstrument::buildChords() {
	GList *lptr2, *lptr3, *lptr4;
	GList *track, *newchord, *double_note_list;
	unsigned int longest;
	NedTableNote *tableNote2, *tableNote3, *longestNote;
	bool chord_found, double_note;
	unsigned long long dist1, dist2;

	//for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		if (g_list_first(m_note_list) == NULL) return;
		track = (GList *) g_list_first(m_note_list)->data;
		if (track == NULL) return;
		for (lptr2 = g_list_first(track); lptr2; lptr2 = g_list_next(lptr2)) {
			tableNote2 = (NedTableNote *) lptr2->data;
			if (tableNote2->m_is_triplet) continue;
			chord_found = false;
			newchord = NULL;
			double_note_list = NULL;
			longest = tableNote2->getDuration();
			longestNote = tableNote2;
			for (lptr3 = g_list_next(lptr2); lptr3; lptr3 = g_list_next(lptr3)) {
				tableNote3 = (NedTableNote *) lptr3->data;
				if (tableNote3->m_is_triplet) continue;
				if (tableNote3->m_start > tableNote2->m_stop + WHOLE_NOTE) break;
				/*
				dist1 = tableNote2->m_start - tableNote3->m_start;
				if (dist1 < 0) dist1 = -dist1;
				dist2 = tableNote2->m_stop - tableNote3->m_stop;
				if (dist2 < 0) dist2 = -dist2;
				*/
				dist1 = (tableNote2->m_start > tableNote3->m_start) ? tableNote2->m_start  - tableNote3->m_start :
							tableNote3->m_start - tableNote2->m_start;
				dist2 = (tableNote2->m_stop > tableNote3->m_stop) ? tableNote2->m_stop  - tableNote3->m_stop :
							tableNote3->m_stop - tableNote2->m_stop;
				NedResource::DbgMsg(DBG_TESTING, "tableNote2(0x%x)->m_start = %llu(%llu)(%llu), tableNote2->m_stop = %llu(%llu)(%llu), tableNote3(0x%x)->m_start = %llu(%llu)(%llu), tableNote3->m_stop = %llu(%llu)(%llu) -->",
						tableNote2, tableNote2->m_start, tableNote2->m_start / NOTE_64, tableNote2->m_start / (NOTE_4 / 3),
						tableNote2->m_stop, tableNote2->m_stop / NOTE_64, tableNote2->m_stop / (NOTE_4 / 3),
						tableNote3, tableNote3->m_start, tableNote3->m_start / NOTE_64, tableNote3->m_start / (NOTE_4 / 3),
						tableNote3->m_stop, tableNote3->m_stop / NOTE_64, tableNote3->m_stop / (NOTE_4 / 3));
				if (dist1 < tableNote2->getDuration() / 4  && dist2 < tableNote2->getDuration() /* || tableNote2->m_start == tableNote3->m_start  || (tableNote2->getDuration() < NOTE_2 && tableNote2->m_stop == tableNote3->m_stop)*/) {
					double_note = false;
					for (lptr4 = g_list_first(newchord); lptr4; lptr4 = g_list_next(lptr4)) {
						if (((NedTableNote *) lptr4->data)->m_midi_pitch == tableNote3->m_midi_pitch) {
							double_note = true;
							NedResource::DbgMsg(DBG_TESTING, "CHORD (doubleNote)\n");
							break;
						}
					}
					NedResource::DbgMsg(DBG_TESTING, "CHORD\n");
					if (tableNote2->m_midi_pitch == tableNote3->m_midi_pitch) {
						double_note = true;
					}
					if (double_note) {
						double_note_list = g_list_append(double_note_list, tableNote3);
					}
					else {
						newchord = g_list_append(newchord, tableNote3);
						if (tableNote3->getDuration() > longest) {
							longest = tableNote3->getDuration();
							longestNote = tableNote3;
						}
						chord_found = true;
					}
				}
				else {
					NedResource::DbgMsg(DBG_TESTING, "KEIN CHORD\n");
				}
	
			}
			if (chord_found) {
				longestNote = tableNote2;
				if (longestNote != tableNote2) {
					newchord = g_list_remove(newchord, longestNote);
					newchord = g_list_append(newchord, tableNote2);
				}
				longestNote->m_chord = newchord;
					
				for (lptr3 = g_list_first(longestNote->m_chord); lptr3; lptr3 = g_list_next(lptr3)) {
					((NedTableNote *) lptr3->data)->m_is_chord_member = true;
					if ((lptr4 = g_list_find(track, lptr3->data)) == NULL) {
						NedResource::Abort("NedTableInstrument::buildChords(1)");
					}
					NedResource::DbgMsg(DBG_TESTING, "entferne 0x%x\n", lptr4->data);
					track = g_list_delete_link(track, lptr4);
				}
				if ((lptr2 = g_list_find(track, longestNote)) == NULL) {
					NedResource::Abort("NedTableInstrument::buildChords(2)");
				}
			}
			for (lptr3 = g_list_first(double_note_list); lptr3; lptr3 = g_list_next(lptr3)) {
				((NedTableNote *) lptr3->data)->m_is_chord_member = true;
				if ((lptr4 = g_list_find(track, lptr3->data)) == NULL) {
					NedResource::Abort("NedTableInstrument::buildChords(1b)");
				}
				NedResource::DbgMsg(DBG_TESTING, "entferne(d) 0x%x  == 0x%x\n", lptr4->data, lptr3->data);
				track = g_list_delete_link(track, lptr4);
				if (g_list_find(track, lptr3->data) != NULL) {
					NedResource::Abort("NedTableInstrument::buildChords(1c)");
				}
			}
		}
		g_list_first(m_note_list)->data = (void *) track;
	//}
}

void NedTableInstrument::fillUpUnwantedRests() {
	GList *lptr, *lptr2, *lptr3;
	GList *track;
	NedTableNote *tableNote2, *tableNote3;
	unsigned long long delta;
	unsigned int duration;
	int measure_dist1, measure_dist2, stop1;
	unsigned int prec;
	bool found;
	for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		track = (GList *) lptr->data;
		if (track == NULL) continue;
		lptr2 = g_list_first(track);
		if (lptr2 == NULL) continue;
		lptr3 = g_list_next(lptr2);
		while (lptr3) {
			tableNote2 = (NedTableNote *) lptr2->data;
			tableNote3 = (NedTableNote *) lptr3->data;
			if (!tableNote2->m_is_triplet && !tableNote3->m_is_triplet && tableNote2->getDuration() < NOTE_4) {
				if (tableNote3->m_start >= tableNote2->m_measure_end) {
					delta = tableNote2->m_measure_end - tableNote2->m_stop;
					if (delta < NOTE_4) {
						tableNote2->m_stop = tableNote2->m_measure_end;
					}
				}
				else {
					duration = tableNote2->getDuration();
					measure_dist1 = tableNote2->m_stop - tableNote2->m_measure_start;
					measure_dist2 = tableNote3->m_start - tableNote2->m_measure_start;
					found = false;
					for (prec = WHOLE_NOTE; prec >= NOTE_16; prec >>= 2) {
						for (stop1 = measure_dist2; stop1 > measure_dist1; stop1 -= prec) {
							if (measure_dist1 < stop1 && stop1 <= measure_dist2 
								/* && stop1 - measure_dist1 < duration / 2 */)  {
								tableNote2->m_stop = tableNote2->m_measure_start + stop1;
								found =	 true;
								break;
							}
						}
					}
				}
			}
			lptr2 = lptr3;
			lptr3 = g_list_next(lptr3);
		}
	}
}

int NedTableInstrument::getMidiPgm() {
	if (m_midi_nr == -1) return 0;
	return m_midi_nr;
}


void NedTableInstrument::determineLines() {
	GList *track, *lptr, *lptr2, *lptr3;
	char offs_array[115];
	int offs;
	NedTableNote *tableNote;
	unsigned long long current_measure_start;



	memset(offs_array, 0, sizeof(offs_array));

	for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		track = (GList *) lptr->data;	
		if (track == NULL) continue;
		current_measure_start = 0;
		for (lptr2 = g_list_first(track); lptr2; lptr2 = g_list_next(lptr2)) {
			tableNote = (NedTableNote *) lptr2->data;
			if (tableNote->m_is_rest) continue;
			if (tableNote->m_measure_start != current_measure_start) {
				current_measure_start = tableNote->m_measure_start;
				memset(offs_array, 0, sizeof(offs_array));
			}
			tableNote->m_line = NedNote::pitchToLine(tableNote->m_midi_pitch, m_clef, m_keysig, m_octave_shift, &offs);
			tableNote->m_state = NedNote::determineState(tableNote->m_midi_pitch, tableNote->m_line, offs, offs_array, m_keysig);
			for (lptr3 = g_list_first(tableNote->m_chord); lptr3; lptr3 = g_list_next(lptr3)) {
				tableNote = (NedTableNote *) lptr3->data;
				if (tableNote->m_is_rest) continue;
				tableNote->m_line = NedNote::pitchToLine(tableNote->m_midi_pitch, m_clef, m_keysig, m_octave_shift, &offs);
				tableNote->m_state = NedNote::determineState(tableNote->m_midi_pitch, tableNote->m_line, offs, offs_array, m_keysig);
			}
					
		}
	}
}

void NedTableInstrument::snapNotes() {
	GList *track, *lptr, *lptr2, *lptr3;
	NedTableNote *tableNote2, *tableNote3;

	for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		track = (GList *) lptr->data;	
		if (track == NULL) continue;
		lptr2 = g_list_first(track);
		if (lptr2 == NULL) continue;
		lptr3 = g_list_next(lptr2);
		for (; lptr2; lptr2 = lptr3, lptr3 = g_list_next(lptr3)) {
			tableNote2 = (NedTableNote *) lptr2->data;
			if (lptr3 == NULL) {
				tableNote2->snapNote(tableNote2->m_stop);
			}
			else {
				tableNote3 = (NedTableNote *) lptr3->data;
				if (tableNote2->m_measure_start == tableNote3->m_measure_start) {
					tableNote2->snapNote(tableNote3->m_start);
				}
				else {
					tableNote2->snapNote(tableNote2->m_measure_end);
				}
			}
		}
	}
}

void NedTableInstrument::setInstrumentName(char *name) {
	strcpy(m_instrument_name, name);
}

char *NedTableInstrument::getInstrumentName() {
	if (m_instrument_name[0] == '\0') return NULL;
	return m_instrument_name;
}

void NedTableInstrument::distributeOverlapping(int voice_num) {
	GList *lptr1, *lptr2, *lptr3, *lptr4;
	NedTableNote *tableNote1, *tableNote2;
	GList *new_voices[VOICE_COUNT - 1];
	GList **current_voice, **next_voice;
	GList *group1, *group2, *group;
	GList *track;
	int avp1, avp2;
	int i, next_voice_idx;
	bool overlapping, test_next;

	if (g_list_first(m_note_list) == NULL) return;
	for (i = 0; i < VOICE_COUNT - 1; new_voices[i++] = NULL);
	track = (GList *) g_list_first(m_note_list)->data;
	current_voice = &track;
	next_voice = &(new_voices[0]);
	next_voice_idx = 0;

	do {
		lptr1 = g_list_first(*current_voice);
		overlapping = false;
		while (lptr1) {
			tableNote1 = (NedTableNote *) lptr1->data;
			test_next = true;
			for (lptr2 = g_list_next(lptr1); lptr2; lptr2 = g_list_next(lptr2)) {
				tableNote2 = (NedTableNote *) lptr2->data;
				if (tableNote2->m_start > tableNote1->m_stop + WHOLE_NOTE) break;
				if (!tableNote1->overlapping(tableNote2)) continue;
				overlapping = true;
				group1 = tableNote1->getGroup(&avp1);
				group2 = tableNote2->getGroup(&avp2);
				group = avp1 > avp2 ? group2 : group1;
				for (lptr3 = g_list_first(group); lptr3; lptr3 = g_list_next(lptr3)) {
					lptr4 = g_list_find(*current_voice, lptr3->data);
					if (lptr4 != NULL) {
						*current_voice = g_list_delete_link(*current_voice, lptr4);
					}
					/*
					else if (!((NedTableNote *) lptr3->data)->m_is_triplet) {
						NedResource::Abort("NedTableInstrument::distributeOverlapping");
					}
					*/
					if (next_voice_idx < VOICE_COUNT - 1 && next_voice_idx < voice_num - 1) {
						*next_voice = NedTableNote::my_insert_sorted(*next_voice, (NedTableNote *) lptr3->data,
							(gint (*)(const void*, const void*)) NedTableNote::compare, false);
					}
					else {
						//delete (NedTableNote *) lptr3->data;
					}
				}
				lptr1 = g_list_first(*current_voice);
				test_next = false;
				break;
			}
			if (test_next) {
				lptr1 = g_list_next(lptr1);
			}
		}
		if (overlapping) {
			if (next_voice_idx == 0) {
				g_list_first(m_note_list)->data = (void *) *current_voice;
			}
			else {
				new_voices[next_voice_idx - 1] = *current_voice;
			}
			if (next_voice_idx < VOICE_COUNT - 1 && next_voice_idx < voice_num - 1) {
				new_voices[next_voice_idx] = *next_voice;
			}
			next_voice_idx++;
			if (next_voice_idx < VOICE_COUNT && next_voice_idx < voice_num) {
				current_voice = &(new_voices[next_voice_idx - 1]);
			}
			if (next_voice_idx < VOICE_COUNT - 1 && next_voice_idx < voice_num - 1) {
				next_voice = &(new_voices[next_voice_idx]);
			}
		}
	}
	while (overlapping && next_voice_idx < VOICE_COUNT);
	for (i = 0; i < VOICE_COUNT - 1 && i < voice_num - 1; i++) {
		if (new_voices[i] != NULL) {
			m_note_list = g_list_append(m_note_list, new_voices[i]);
		}
	}
}

void NedTableInstrument::testNewVolume(NedChordOrRest *chord, int volume, int *current_volume, int volume_change_density) {
	NedVolumeSign *volsign;
	int volume_dist = volume - *current_volume;
	int kind;

#define VOLUME_SIGN_Y_DIST 1.3


	if (volume_dist < 0) volume_dist =- volume_dist;
	if (100.0 * (double) volume_dist / (double) (*current_volume + 1) < volume_change_density) return;
	if (volume < 18) {
		kind = VOL_PPP;
	}
	else if (volume < 36) {
		kind = VOL_PP;
	}
	else if (volume < 54) {
		kind = VOL_P;
	}
	else if (volume < 72) {
		kind = VOL_MF;
	}
	else if (volume < 91) {
		kind = VOL_F;
	}
	else if (volume < 108) {
		kind = VOL_FF;
	}
	else {
		kind = VOL_FFF;
	}
	volsign = new NedVolumeSign(kind, chord, DEFAULT_FREE_X_DIST, VOLUME_SIGN_Y_DIST, true);
	volsign->m_midi_volume = volume;
	*current_volume = volume;
}




GList *NedTableInstrument::convertToNtEdNotes(int voice_nr, NedInstrumentTable *table, bool place_tempo_sigs) {
	GList *track, *lptr, *lptr2, *trackptr;
	NedNote *note;
	int dotcount, len2;
	NedChordOrRest *chord_or_rest = NULL, *chord_or_rest2;
	NedTableNote *table_note = NULL;
	GList *ned_notes = NULL;
	GList *pitches =  NULL;
	unsigned long long delta_time, d1, current_midi_time = 0;
	unsigned long long measure_start, measure_end;
	bool has_tie_backward;
	NedChordOrRest *this_first_chord = NULL, *this_last_chord = NULL, *previous_last_chord = NULL;
	NedTableNote *this_table_note = NULL, *last_table_note = NULL;
	int mnum = 1;
	NedTableTemposig *temposig;
	int current_volume = m_midi_volume;
	int current_tempo = table->getStartTempo();

	table->getMeasureDurationAt(0, &measure_start, &measure_end);
/*
	for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		track = (GList *) g_list_first(m_note_list)->data;	
		for (lptr2 = g_list_first(track); lptr2; lptr2 = g_list_next(lptr2)) {
			((NedTableNote *) lptr2->data)->print();
		}
	}
*/


	if (voice_nr < 0 || voice_nr >= VOICE_COUNT) {
		NedResource::Abort("NedTableInstrument::convertToNtEdNotes");
	}
	NedResource::DbgMsg(DBG_TESTING, "Start new voice --------------\n");
	if ((trackptr = g_list_nth(m_note_list, voice_nr)) == NULL) {
		return NULL;
	}
	track = (GList *) trackptr->data;
	delta_time = ((NedTableNote *) g_list_first(track)->data)->m_start;
	this_table_note = (NedTableNote *) g_list_first(track)->data;
			
	while (current_midi_time + delta_time >= measure_end) {
		NedResource::DbgMsg(DBG_TESTING, "LengthField = %llu(%llu)\n", measure_end - measure_start, (measure_end - measure_start) / NOTE_64);
		NedResource::setLengthField(measure_end - measure_start);
		len2 = NedResource::getPartLength(&dotcount, true, true, measure_end - measure_start);
		while (len2) {
			NedResource::DbgMsg(DBG_TESTING, "start Rest: len2 = %u(%u), measure_start = %llu(%llu), measure_end = %llu(%llu), delta_time = %llu(%llu)\n",
				len2, len2 / NOTE_64, measure_start, measure_start / NOTE_64, measure_end, measure_end / NOTE_64, delta_time, delta_time / NOTE_64);
					

			this_last_chord = chord_or_rest = new NedChordOrRest(NULL, TYPE_REST, false, 3 /* dummy */, dotcount, len2, NORMAL_NOTE, 0, 0);
			ned_notes = g_list_append(ned_notes, chord_or_rest);
			len2 = NedResource::getPartLength(&dotcount);
			current_midi_time += chord_or_rest->getDuration(measure_end - measure_start);
			delta_time -= chord_or_rest->getDuration(measure_end - measure_start);
		}
		if (current_midi_time != measure_end) {
			NedResource::Abort("NedTableInstrument::convertToNtEdNotes(A)");
		}
		NedResource::DbgMsg(DBG_TESTING, "current_midi_time = %llu(%llu) mnum = %d\n", current_midi_time, current_midi_time / NOTE_64, mnum);
		table->getMeasureDurationAt(measure_end, &measure_start, &measure_end);
		mnum++;
		NedResource::DbgMsg(DBG_TESTING, "Stelle OO: measure_start = %llu(%llu), measure_end = %llu(%llu), delta_time = %llu(%llu)\n",
				measure_start, measure_start / NOTE_64, measure_end, measure_end / NOTE_64, delta_time, delta_time / NOTE_64);
	}
	if (delta_time > NOTE_64) {
		NedResource::setLengthField(delta_time);
		len2 = NedResource::getPartLength(&dotcount);
		while (len2) {
			NedResource::DbgMsg(DBG_TESTING, "STelle VV Rest: %u(%u), delta_time = %llu(%llu)\n",
					len2, len2 / NOTE_64, delta_time, delta_time / NOTE_64);
			this_last_chord = chord_or_rest = new NedChordOrRest(NULL, TYPE_REST, false, 3 /* dummy */, dotcount, len2, NORMAL_NOTE, 0, 0);
			ned_notes = g_list_append(ned_notes, chord_or_rest);
			len2 = NedResource::getPartLength(&dotcount);
		}
	}
	current_midi_time = ((NedTableNote *) g_list_first(track)->data)->m_start;
	table->getMeasureDurationAt(current_midi_time, &measure_start, &measure_end);
	for (lptr = g_list_first(track); lptr; lptr = g_list_next(lptr)) {
		this_table_note = table_note = (NedTableNote *) lptr->data;
		if (current_midi_time > table_note->m_start) {
			NedResource::DbgMsg(DBG_TESTING, "Stalla A: %s - %s : current_midi_time = %llu(%llu)(%llu), table_note->m_start(0x%x) = %llu(%llu)(%llu), table_note->m_stop = %llu(%llu)(%llu)\n",
				table_note->m_is_triplet ? "Triplet" : "Standard", table_note->m_is_rest ? "Rest" : "Note", 
				current_midi_time, current_midi_time / NOTE_64, current_midi_time / (NOTE_4 / 3),
				table_note, table_note->m_start, table_note->m_start/ NOTE_64, table_note->m_start/ (NOTE_4 / 3),
				table_note->m_stop, table_note->m_stop/ NOTE_64, table_note->m_stop/ (NOTE_4 / 3));
			NedResource::Abort("NedTableInstrument::convertToNtEdNotes: current_midi_time > table_note->m_start");
		}
		delta_time = table_note->m_start - current_midi_time;
		NedResource::DbgMsg(DBG_TESTING, "delta_time = %llu(%llu)(%llu), table_note(0x%x)->m_start = %llu(%llu)(%llu), table_note->m_stop = %llu(%llu)(%llu)\n",
			delta_time, delta_time / NOTE_64, delta_time / (NOTE_4 / 3), 
			table_note, table_note->m_start, table_note->m_start / NOTE_64, table_note->m_start / (NOTE_4 / 3),
			table_note->m_stop, table_note->m_stop / NOTE_64, table_note->m_stop / (NOTE_4 / 3));
		if (current_midi_time + delta_time >= measure_end) {
			d1 = measure_end - current_midi_time;
			NedResource::DbgMsg(DBG_TESTING, "Stelle A: d1 = %llu(%llu), measure_end = %llu(%llu), current_midi_time = %llu(%llu), measure_start = %llu(%llu)\n",
				d1, d1 / NOTE_64, measure_end, measure_end / NOTE_64, current_midi_time, current_midi_time / NOTE_64, measure_start, measure_start / NOTE_64);
			NedResource::setLengthField(d1);
			len2 = NedResource::getPartLength(&dotcount, true, current_midi_time == measure_start, measure_end - measure_start);
			while (len2) {
				this_first_chord = this_last_chord = chord_or_rest = new NedChordOrRest(NULL, TYPE_REST, false, 3 /* dummy */, dotcount, len2, NORMAL_NOTE, 0, 0);
				NedResource::DbgMsg(DBG_TESTING, "current_midi_time = %llu(%llu) --> ", current_midi_time, current_midi_time / NOTE_64);
				current_midi_time += chord_or_rest->getDuration(measure_end - measure_start);
				NedResource::DbgMsg(DBG_TESTING, "current_midi_time = %llu(%llu) \n", current_midi_time, current_midi_time / NOTE_64);
				delta_time -= chord_or_rest->getDuration(measure_end - measure_start);
				NedResource::DbgMsg(DBG_TESTING, "Rest(2): d1 = %llu(%llu), len2 = %u(%u), duration = %u(%u), delta_time = %llu, current_midi_time = %llu, measure_start = %llu(%llu), measure_end = %llu(%llu)\n",
					d1, d1 / NOTE_64, len2, len2 / NOTE_64, chord_or_rest->getDuration(measure_end - measure_start),
					chord_or_rest->getDuration(measure_end - measure_start) / NOTE_64, delta_time / NOTE_64, current_midi_time / NOTE_64,
					measure_start, measure_start / NOTE_64, measure_end, measure_end / NOTE_64);
				ned_notes = g_list_append(ned_notes, chord_or_rest);
				len2 = NedResource::getPartLength(&dotcount);
			}
			if (current_midi_time != measure_end) {
				NedResource::Abort("NedTableInstrument::convertToNtEdNotes(B)");
			}
			table->getMeasureDurationAt(measure_end, &measure_start, &measure_end);
		}
		while (delta_time >= measure_end - measure_start) {
			NedResource::setLengthField(measure_end - measure_start);
			len2 = NedResource::getPartLength(&dotcount, true, current_midi_time == measure_start, measure_end - measure_start);
			while (len2) {
				this_first_chord = this_last_chord = chord_or_rest = new NedChordOrRest(NULL, TYPE_REST, false, 3 /* dummy */, dotcount, len2, NORMAL_NOTE, 0, 0);
				current_midi_time += chord_or_rest->getDuration(measure_end - measure_start);
				delta_time -= chord_or_rest->getDuration(measure_end - measure_start);
				NedResource::DbgMsg(DBG_TESTING, "Rest(2): len2 = %u(%u), delta_time = %llu, current_midi_time = %llu\n",
					len2, len2 / NOTE_64, chord_or_rest->getDuration(measure_end - measure_start) / NOTE_64, delta_time / NOTE_64, current_midi_time / NOTE_64);
				ned_notes = g_list_append(ned_notes, chord_or_rest);
				len2 = NedResource::getPartLength(&dotcount);
			}
			if (current_midi_time != measure_end) {
				NedResource::Abort("NedTableInstrument::convertToNtEdNotes(C)");
			}
			table->getMeasureDurationAt(measure_end, &measure_start, &measure_end);
		}
		NedResource::setLengthField(delta_time),
		len2 = NedResource::getPartLength(&dotcount);
		while (len2) {
			this_first_chord = this_last_chord = chord_or_rest = new NedChordOrRest(NULL, TYPE_REST, false, 3 /* dummy */, dotcount, len2, NORMAL_NOTE, 0, 0);
			current_midi_time += chord_or_rest->getDuration();
			NedResource::DbgMsg(DBG_TESTING, "Rest(3): %u, current_midi_time = %llu\n", chord_or_rest->getDuration() / NOTE_64, current_midi_time / NOTE_64);
			if (current_midi_time > measure_end) {
				NedResource::DbgMsg(DBG_TESTING, "current_midi_time = %llu(%llu), measure_start = %llu(%llu), measure_end = %llu(%llu)\n",
					current_midi_time, current_midi_time / NOTE_64, measure_start, measure_start / NOTE_64, measure_end, measure_end / NOTE_64);
				NedResource::Abort("NedTableInstrument::convertToNtEdNotes: current_midi_time > measure_end");
			}
			if (current_midi_time == measure_end) {
				NedResource::DbgMsg(DBG_TESTING, "NEW(4): measure_end = %llu\n", measure_end / NOTE_64);
				table->getMeasureDurationAt(measure_end, &measure_start, &measure_end);
			}
			delta_time -= chord_or_rest->getDuration();
			ned_notes = g_list_append(ned_notes, chord_or_rest);
			len2 = NedResource::getPartLength(&dotcount);
		}
		if (table_note->m_is_triplet) {
			NedResource::setLengthField(table_note->getDuration() * 3 / 2);
		}
		else {
			NedResource::setLengthField(table_note->getDuration());
		}
		len2 = NedResource::getPartLength(&dotcount, table_note->m_is_rest, !table_note->m_is_rest || current_midi_time == measure_start, measure_end - measure_start);
		if (len2) {
			if (table_note->m_is_rest) {
				this_first_chord = this_last_chord = chord_or_rest = new NedChordOrRest(NULL, TYPE_REST, false, 3 /* dummy */, dotcount, len2, NORMAL_NOTE, 0, 0);
				if (table_note->m_is_triplet) {
					 chord_or_rest->setTupletVal(3);
				}
				NedResource::DbgMsg(DBG_TESTING, "Stelle B: %s, Rest(4)(0x%x): %u(%u)(%u), ", table_note->m_is_triplet ? "Triplet" : "Standard",
				table_note, chord_or_rest->getDuration(measure_end - measure_start), chord_or_rest->getDuration(measure_end - measure_start) / NOTE_64,
				chord_or_rest->getDuration(measure_end - measure_start) / (NOTE_4 / 3));
			}
			else {
				this_first_chord = this_last_chord = chord_or_rest = new NedChordOrRest(NULL, TYPE_NOTE, false, table_note->m_line, dotcount, len2, NORMAL_NOTE, table_note->m_state, current_midi_time);
				if (table_note->m_start != current_midi_time) {
					NedResource::DbgMsg(DBG_TESTING, "table_note = 0x%x, table_note->m_measure_start = %llu(%llu), table_note->m_start = %llu(%llu), table_note->m_stop = %llu(%llu), current_midi_time = %llu(%llu), m_is_triplet = %d, m_is_rest = %d\n",
						table_note, table_note->m_measure_start, table_note->m_measure_start / NOTE_64,
						table_note->m_start, table_note->m_start / NOTE_64,
						table_note->m_stop, table_note->m_stop / NOTE_64,
						current_midi_time, current_midi_time / NOTE_64, table_note->m_is_triplet, table_note->m_is_rest);
					NedResource::Abort("NedTableInstrument::convertToNtEdNotes: table_note->m_start != current_midi_time");
				}
				if (pitches != NULL) {
					g_list_free(pitches);
					pitches = NULL;
				}
				pitches = g_list_append(pitches, table_note);
				if (table_note->m_is_triplet) {
					 chord_or_rest->setTupletVal(3);
				}
				NedResource::DbgMsg(DBG_TESTING, "%s Note(4): %u(%u)(%u), current_midi_time = %llu(%llu)(%llu), table_note->m_start = %llu(%llu)(%llu), table_note->m_stop = %llu(%llu)(%llu)  ",
					table_note->m_is_triplet ? "Triplet" : "Standard", 
					chord_or_rest->getDuration(measure_end - measure_start), chord_or_rest->getDuration(measure_end - measure_start) / NOTE_64, chord_or_rest->getDuration(measure_end - measure_start) / (NOTE_4 / 3),
					current_midi_time, current_midi_time / NOTE_64, current_midi_time / (NOTE_4 / 3),
					table_note->m_start, table_note->m_start / NOTE_64, table_note->m_start / (NOTE_4 / 3),
					table_note->m_stop, table_note->m_stop / NOTE_64, table_note->m_stop / (NOTE_4 / 3));
			}
			if (table_note->m_is_triplet) {
				chord_or_rest->setTupletVal(3);
			}
			if (!table_note->m_is_rest) {
				for (lptr2 = g_list_first(table_note->m_chord); lptr2; lptr2 = g_list_next(lptr2)) {
					//table_note->clone_chord((NedTableNote *) lptr2->data);
					note = new NedNote(chord_or_rest, ((NedTableNote *) lptr2->data)->m_line, NORMAL_NOTE, ((NedTableNote *) lptr2->data)->m_state);
					chord_or_rest->insertNoteAt(note);
				}
				if (place_tempo_sigs && (temposig = table->getCurrentTempo(&current_tempo, table_note->m_start)) != NULL) {
					new NedTempoSign(NOTE_4, temposig->m_tempo, chord_or_rest, DEFAULT_FREE_X_DIST, DEFAULT_FREE_Y_DIST, true);
				}
				if (voice_nr == 0) {
					testNewVolume(chord_or_rest, table_note->m_volume, &current_volume, table->m_volume_change_density);
				}
			}
			ned_notes = g_list_append(ned_notes, chord_or_rest);
			current_midi_time += chord_or_rest->getDuration(measure_end - measure_start);
			NedResource::DbgMsg(DBG_TESTING, "current_midi_time = %llu(%llu)(%llu)\n", current_midi_time, current_midi_time /NOTE_64, current_midi_time / (NOTE_4 /  3));
			if (current_midi_time > measure_end) {
				NedResource::DbgMsg(DBG_TESTING, "current_midi_time = %llu, measure_end = %llu\n",
				current_midi_time / NOTE_64, measure_end / NOTE_64);
				NedResource::Abort("NedTableInstrument::convertToNtEdNotes: current_midi_time > measure_end(2)");
			}
			if (current_midi_time == measure_end) {
				NedResource::DbgMsg(DBG_TESTING, "NEW(5): measure_end = %llu\n", measure_end / NOTE_64);
				table->getMeasureDurationAt(measure_end, &measure_start, &measure_end);
			}
		}
		len2 = NedResource::getPartLength(&dotcount);
		while (len2) {
			this_last_chord = chord_or_rest2 = chord_or_rest->cloneWithDifferentLength(len2, dotcount); 
			chord_or_rest2->setMidiTime(current_midi_time);
			if (table_note->m_is_triplet) {
				chord_or_rest2->setTupletVal(3);
			}
			if (!table_note->m_is_rest) {
				chord_or_rest->tieCompleteTo(chord_or_rest2);
			}
			ned_notes = g_list_append(ned_notes, chord_or_rest2);
			current_midi_time += chord_or_rest2->getDuration();
			NedResource::DbgMsg(DBG_TESTING, "Rest(5): %u, current_midi_time = %llu\n", chord_or_rest2->getDuration() / NOTE_64, current_midi_time /NOTE_64);
			if (current_midi_time > measure_end) {
				NedResource::Abort("NedTableInstrument::convertToNtEdNotes: current_midi_time > measure_end(3)");
			}
			if (current_midi_time == measure_end) {
				NedResource::DbgMsg(DBG_TESTING, "NEW(6): measure_end = %llu\n", measure_end / NOTE_64);
				table->getMeasureDurationAt(measure_end, &measure_start, &measure_end);
			}
			chord_or_rest2 = chord_or_rest;
			len2 = NedResource::getPartLength(&dotcount);
		}
		has_tie_backward = (previous_last_chord != NULL && !table_note->m_is_rest && this_first_chord != NULL && !this_first_chord->isRest() && table_note->m_tie_backward != NULL &&
				!table_note->m_tie_backward->m_is_rest && table_note->m_tie_backward->m_tie_forward == table_note);
		/*
		for (lptr2 = g_list_first(table_note->m_chord); !has_tie_backward && lptr2; lptr2 = g_list_next(lptr2)) {
			if (((NedTableNote *) lptr2->data)->m_tie_backward != NULL) {
				has_tie_backward = true;
			}
		}
		*/
		if (has_tie_backward) {
			previous_last_chord->tieCompleteTo(this_first_chord);
		}
		previous_last_chord = this_last_chord;
		last_table_note = this_table_note;
	}
	return ned_notes;
}

NedInstrumentTable::NedInstrumentTable(bool with_triplets, bool with_single_triplets, bool with_alto_clef, int voice_num,
	int tempo_change_density, int volume_change_density, bool force_piano, bool force_one_staff) :
m_tempo_inverse(START_TEMPO_INVERSE), m_tempo_change_density(tempo_change_density), m_volume_change_density(volume_change_density), m_force_piano(force_piano), m_force_one_staff(force_one_staff), 
m_instrument_list(NULL), timesig_list(NULL), temposig_list(NULL), m_pending_keysig(-111), m_last_inserted(NULL),
m_with_triplets(with_triplets), m_with_single_triplets(with_single_triplets), m_with_alto_clef(with_alto_clef), m_voice_num(voice_num) {
	m_pending_instrument_name[0] = '\0';
}

NedInstrumentTable::~NedInstrumentTable() {
	GList *lptr;

	for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
		delete (NedTableInstrument *) lptr->data;
	}
	if (m_instrument_list != NULL) {
		g_list_free(m_instrument_list);
	}
	m_instrument_list = NULL;
}

void NedInstrumentTable::addNote(NedTableNote *note, int midi_nr) {
	GList *lptr;
	NedTableInstrument *inst = NULL;

	for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
		if (((NedTableInstrument *) lptr->data)->getMidiNr() == midi_nr) {
			m_last_inserted = inst = (NedTableInstrument *) lptr->data;
			if (m_pending_keysig != -111) {
				m_last_inserted->addKeySig(m_pending_keysig);
				m_pending_keysig = -111;
			}
			if (m_pending_instrument_name[0] != '\0') {
				m_last_inserted->setInstrumentName(m_pending_instrument_name);
				m_pending_instrument_name[0] = '\0';
			}
		}
	}

	if (inst == NULL) {
		inst = new NedTableInstrument(midi_nr);
		m_instrument_list = g_list_append(m_instrument_list, inst);
	}

	inst->addNote(note);
}

void NedInstrumentTable::addKeySig(unsigned long long midi_time, int keysig) {
	if (m_last_inserted != NULL) {
		m_last_inserted->addKeySig(keysig);
	}
	else {
		m_pending_keysig = keysig;
	}
}

void NedInstrumentTable::addTimeSig(NedTableTimesig *timesig) {
	timesig_list = g_list_insert_sorted(timesig_list, timesig,
		 (gint (*)(const void*, const void*)) NedTableTimesig::compare);
}

void NedInstrumentTable::addTempoSig(NedTableTemposig *temposig) {
	temposig_list = g_list_insert_sorted(temposig_list, temposig,
		 (gint (*)(const void*, const void*)) NedTableTemposig::compare);
}


void NedInstrumentTable::setInstrumentName(char *name) {
	if (m_last_inserted != NULL) {
		m_last_inserted->setInstrumentName(name);
	}
	else {
		strcpy(m_pending_instrument_name, name);
	}
}

void NedInstrumentTable::closeNote(int pitch, unsigned long long end_time, int midi_nr) {
	GList *lptr;

	for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
		if (((NedTableInstrument *) lptr->data)->getMidiNr() == midi_nr) {
			if (((NedTableInstrument *) lptr->data)->closeNote(pitch, end_time)) {
				return;
			}
		}
	}

	NedResource::DbgMsg(DBG_TESTING, "NedInstrumentTable::closeNote: not found\n"); 
}

void NedInstrumentTable::print() {
	GList *lptr;
	int i;

	for (i = 0, lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr), i++) {
		NedResource::DbgMsg(DBG_TESTING,"Instrument %d:\n", i);
		((NedTableInstrument *) lptr->data)->print();
	}
}

void NedTableInstrument::determineRange(bool determine_volume /* = false */) {
	GList *track, *lptr, *lptr2;
	NedTableNote *table_note;
	int i;
	double pitch_sum = 0.0;
	int num_notes = 0;
	double volume_sum = 0.0;

	for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		track = (GList *) g_list_first(m_note_list)->data;	
		for (lptr2 = g_list_first(track); lptr2; lptr2 = g_list_next(lptr2)) {
			table_note = (NedTableNote *) lptr2->data;
			if (table_note->m_midi_pitch < m_min_pitch) m_min_pitch = table_note->m_midi_pitch;
			if (table_note->m_midi_pitch > m_max_pitch) m_max_pitch = table_note->m_midi_pitch;
			pitch_sum += table_note->m_midi_pitch;
			volume_sum += table_note->m_volume;
			num_notes++;
			for (i = 0; i < 4; i++) {
				if (m_splits[i] <= table_note->m_midi_pitch && table_note->m_midi_pitch < m_splits[i+1]) {
					m_slots[i] = true;
				}
			}
			for (i = 0; i < 2; i++) {
				if (m_piano_splits[i] <= table_note->m_midi_pitch && table_note->m_midi_pitch < m_piano_splits[i+1]) {
					m_piano_slots[i] = true;
				}
			}
		}
	}
	if (num_notes > 1) {
		m_av_pitch = (int) (pitch_sum / (double) num_notes + 0.5);
		if (determine_volume) {
			m_midi_volume = (int)(volume_sum / (double) num_notes + 0.5);
		}
	}
}

void NedTableInstrument::determineBestClef(bool with_alto_clef) {
	determineRange();
	static int clef_mids[] = { 50 - 12, 50, 60, 67, 67 + 12};
	int i, min_idx = 3;
	int dist, min_dist = 100000;


	for (i = 0; i < 5; i++) {
		dist = m_av_pitch - clef_mids[i];
		if (dist < 0) dist = -dist;
		if (dist < min_dist) {
			min_dist = dist;
			min_idx = i;
		}
	}

	switch (min_idx) {
		case 0: m_clef = BASS_CLEF; m_octave_shift = -1; break;
		case 1: m_clef = BASS_CLEF; m_octave_shift = 0; break;
		case 2: if (with_alto_clef) {
				m_clef = ALTO_CLEF; m_octave_shift = 0; 
				break;
			}
		case 3: m_clef = TREBLE_CLEF; m_octave_shift = 0; break;
		case 4: m_clef = TREBLE_CLEF; m_octave_shift = 1; break;
	}
}


void NedTableInstrument::testUnclosed(int track_nr) {
	GList *track, *lptr2;

	//for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		track = (GList *) g_list_first(m_note_list)->data;	
		for (lptr2 = g_list_first(track); lptr2; lptr2 = g_list_next(lptr2)) {
			if (!((NedTableNote *) lptr2->data)->m_closed) {
				NedResource::DbgMsg(DBG_TESTING, "0x%x unclosed: start = %llu(%llu), track = %d\n", lptr2->data,
					((NedTableNote *) lptr2->data)->m_start, ((NedTableNote *) lptr2->data)->m_start / NOTE_4, track_nr);
				lptr2 = g_list_previous(lptr2);
				track = g_list_delete_link(track, lptr2);
				if (lptr2 == NULL) {
					lptr2 = g_list_first(track);
				}
			}
		}
		g_list_first(m_note_list)->data = (void *) track;
	//}
}

void NedTableInstrument::detectClashes() {
	GList *track, *lptr2;

	//for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		if (g_list_first(m_note_list) == NULL) return;
		track = (GList *) g_list_first(m_note_list)->data;	
		if (track == NULL) return;
		for (lptr2 = g_list_first(track); lptr2; lptr2 = g_list_next(lptr2)) {
			if (((NedTableNote *) lptr2->data)->m_is_triplet || ((NedTableNote *) lptr2->data)->m_is_triplet) continue;
			if (((NedTableNote *) lptr2->data)->m_start >= ((NedTableNote *) lptr2->data)->m_stop) {
				NedResource::DbgMsg(DBG_TESTING, "0x%x clash start = %llu(%llu), stop = %llu(%llu)\n", lptr2->data,
					((NedTableNote *) lptr2->data)->m_start, ((NedTableNote *) lptr2->data)->m_start / NOTE_4,
					((NedTableNote *) lptr2->data)->m_stop, ((NedTableNote *) lptr2->data)->m_stop / NOTE_4);
				if (((NedTableNote *) lptr2->data)->m_tie_backward != NULL) {
					((NedTableNote *) lptr2->data)->m_tie_backward->m_tie_forward = NULL;
					((NedTableNote *) lptr2->data)->m_tie_backward = NULL;
				}
				if (((NedTableNote *) lptr2->data)->m_tie_forward != NULL) {
					((NedTableNote *) lptr2->data)->m_tie_forward->m_tie_backward = NULL;
					((NedTableNote *) lptr2->data)->m_tie_forward = NULL;
				}
				NedResource::DbgMsg(DBG_TESTING, "clash: loesche 0x%x m_stop = %llu(%llu)\n", lptr2->data, ((NedTableNote *) lptr2->data)->m_stop,
					((NedTableNote *) lptr2->data)->m_stop / NOTE_64);
				track = g_list_delete_link(track, lptr2);
				lptr2 = g_list_first(track);
			}
		
		}
		g_list_first(m_note_list)->data = (void *)track;
	//}
}

void NedTableInstrument::detectClashes2(char *m) {
	GList *track, *lptr, *lptr2;

	for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		track = (GList *) lptr->data;	
		if (track == NULL) continue;
		for (lptr2 = g_list_first(track); lptr2; lptr2 = g_list_next(lptr2)) {
			if (((NedTableNote *) lptr2->data)->m_start >= ((NedTableNote *) lptr2->data)->m_stop) {
				NedResource::DbgMsg(DBG_TESTING, "%s, 0x%x clash2 start = %llu(%llu), stop = %llu(%llu)\n", m, lptr2->data,
					((NedTableNote *) lptr2->data)->m_start, ((NedTableNote *) lptr2->data)->m_start / NOTE_4,
					((NedTableNote *) lptr2->data)->m_stop, ((NedTableNote *) lptr2->data)->m_stop / NOTE_4);
			}
		
		}
		g_list_first(m_note_list)->data = (void *)track;
	}
}

void NedTableInstrument::testForMeasureCrossingNotes(NedInstrumentTable *table) {
	GList *track, *lptr, *lptr2;
	unsigned long long measure_start, measure_end;
	NedTableNote *table_note;

	for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		track = (GList *) lptr->data;	
		if (track == NULL) continue;
		for (lptr2 = g_list_first(track); lptr2; lptr2 = g_list_next(lptr2)) {
			table_note = (NedTableNote *) lptr2->data;
			table->getMeasureDurationAt(table_note->m_start, &measure_start, &measure_end);
			if (table_note->m_stop > measure_end) {
				NedResource::DbgMsg(DBG_TESTING, "overlapping meas: %s\n", table_note->m_is_rest ? "Rest" : "Note");
			}
		}	
	}
}

void NedTableInstrument::testForRests() {
	GList *track, *lptr2;
	NedTableNote * table_note;

	//for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
		track = (GList *) g_list_first(m_note_list)->data;	
		if (track == NULL) return;
		for (lptr2 = g_list_first(track); lptr2; lptr2 = g_list_next(lptr2)) {
			table_note = (NedTableNote *) lptr2->data;
			if (table_note->m_is_rest) {
				NedResource::DbgMsg(DBG_TESTING, "RRRest\n");
			}
		}	
	//}
}

GList *NedTableInstrument::splitRange(bool force_piano, bool with_alto_clef) {
	GList *new_instruments = NULL;
	NedTableInstrument *instrument[3];
	GList *track, *lptr2;
	NedTableNote *table_note;
	int split[4][2], i;
	bool piano = false;
	bool do_continue;

	determineRange(true);
	if (m_max_pitch - m_min_pitch < 35) {
		determineBestClef(with_alto_clef);
		return NULL;
	}

	if (force_piano) {
		piano = true;
		split[0][0] = 0; split[0][1] = m_piano_splits[1];
		split[1][0] = m_piano_splits[1]; split[1][1] = 129;
		m_staves = 2;
	}
	else if ((m_min_pitch >= m_piano_splits[0]  && m_max_pitch < m_piano_splits[2])) {
		if (m_piano_slots[0] && m_piano_slots[1]) {
			split[0][0] = m_piano_splits[0]; split[0][1] = m_piano_splits[1];
			split[1][0] = m_piano_splits[1]; split[1][1] = m_piano_splits[2];
			piano = true;
			m_staves = 2;
		}
		else {
			determineBestClef(with_alto_clef);
			return NULL;
		}
	}
	else {
		m_staves = 0;
		for (i = 0; i < 4; i++) {
			if (m_slots[i]) {
				split[m_staves][0] = m_splits[i];
				split[m_staves][1] = m_splits[i+1];
				m_staves++;
			}
		}
		if (m_staves == 1) {
			determineBestClef(with_alto_clef);
			return NULL;
		}
	}

	for (i = 0; i < m_staves - 1; i++) {
		instrument[i] = new NedTableInstrument(m_midi_nr);
		instrument[i]->m_midi_channel = m_midi_channel;
		instrument[i]->m_midi_volume = m_midi_volume;
		instrument[i]->m_keysig = m_keysig;
		if (m_instrument_name[0] != '\0') {
			strcpy(instrument[i]->m_instrument_name, m_instrument_name);
		}
		if (!piano) {
			if (split[i+1][0] == m_splits[0]) {
				instrument[i]->m_clef = BASS_CLEF;
				instrument[i]->m_octave_shift = -1;
			}
			else if (split[i+1][0] == m_splits[1]) {
				instrument[i]->m_clef = BASS_CLEF;
			}
			else if (split[i+1][0] == m_splits[2]) {
				instrument[i]->m_clef = TREBLE_CLEF;
			}
			else {
				instrument[i]->m_clef = TREBLE_CLEF;
				instrument[i]->m_octave_shift = 1;
			}
		 }
		new_instruments = g_list_insert(new_instruments, instrument[i], 0);
	}
	if (piano) {
		instrument[0]->m_clef = TREBLE_CLEF;
		instrument[0]->m_octave_shift = 0;
		m_clef = BASS_CLEF;
		m_octave_shift = 0;
	}
	else {
		if (split[0][0] == m_splits[0]) {
			m_clef = BASS_CLEF;
			m_octave_shift = -1;
		}
		else if (split[0][0] == m_splits[1]) {
			m_clef = BASS_CLEF;
		}
		else if (split[0][0] == m_splits[2]) {
			m_clef = TREBLE_CLEF;
		}
		else {
			m_clef = TREBLE_CLEF;
			m_octave_shift = 1;
		}
	}

	//for (lptr = g_list_first(m_note_list); lptr; lptr = g_list_next(lptr)) {
	if (g_list_first(m_note_list) != NULL)  {
		track = (GList *) g_list_first(m_note_list)->data;	
		lptr2 = g_list_first(track);
		while (lptr2) {
			do_continue = false;
			table_note = (NedTableNote *) lptr2->data;
			for (i = 1; i < m_staves; i++) {
				if (table_note->m_midi_pitch >= split[i][0] && table_note->m_midi_pitch < split[i][1]) {
					track = g_list_delete_link(track, lptr2);
					instrument[i-1]->addNote(table_note);
					lptr2 = g_list_first(track);
					do_continue = true;
					break;
				}
			}
			if (do_continue) continue;
			lptr2 = g_list_next(lptr2);
		}
		g_list_first(m_note_list)->data = (void *) track;
	}

	//}
	if (!piano) {
		determineBestClef(with_alto_clef);
		for (i = 0; i < m_staves - 1; i++) {
			instrument[i]->determineBestClef(with_alto_clef);
		}
	}
	return new_instruments;
}

void NedTableInstrument::setBestClef(bool with_alto_clef) {
	determineBestClef(with_alto_clef);
}
			

void NedInstrumentTable::prepareNotes() {
	GList *lptr, *lptr2, *lptr3;
	GList *new_instruments;
	NedTableInstrument *instr;
	unsigned long long base = NOTE_2;
	GList *ori_list = NULL;
	int channel, i;

	for (i = 0, lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr), i++) {
		instr = (NedTableInstrument *) lptr->data;
		instr->testUnclosed(i);
	}
	channel = 0;
	for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
		instr = (NedTableInstrument *) lptr->data;
		if (instr->getMidiNr() == -1) {
			instr->m_midi_channel = 9;
		}
		else {
			instr->m_midi_channel = channel++;
			if (channel == 9) channel++;
		}
	}
	NedResource::DbgMsg(DBG_TESTING, "Stelle B\n");
	if (!m_force_one_staff) {
		for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
			ori_list = g_list_append(ori_list, lptr->data);
		}
		for (lptr = g_list_first(ori_list); lptr; lptr = g_list_next(lptr)) {
			instr = (NedTableInstrument *) lptr->data;
			new_instruments = instr->splitRange(m_force_piano, m_with_alto_clef);
			for (lptr2 = g_list_first(new_instruments); lptr2; lptr2 = g_list_next(lptr2)) {
				if ((lptr3 = g_list_find(m_instrument_list, lptr->data)) == NULL) {
					NedResource::Abort("NedInstrumentTable::prepareNotes");
				}
				m_instrument_list = g_list_insert_before(m_instrument_list, lptr3, lptr2->data);
			}
			g_list_free(new_instruments);
		}
		g_list_free(ori_list);
		NedResource::DbgMsg(DBG_TESTING, "Stelle C\n");
	}
	else {
		for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
			instr = (NedTableInstrument *) lptr->data;
			instr->setBestClef(m_with_alto_clef);
		}
	}
	for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
		((NedTableInstrument *) lptr->data)->snapNotesToMeasures(false, this);
	}
	NedResource::DbgMsg(DBG_TESTING, "Stelle D\n");
	for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
		((NedTableInstrument *) lptr->data)->testForMeasureCrossingNotes(this);
	}
	for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
		((NedTableInstrument *) lptr->data)->determineLines();
	}
	if (m_with_triplets) {
		for (base = WHOLE_NOTE; base >= NOTE_4; base /= 2) {
		//for (base = NOTE_4; base <= WHOLE_NOTE; base *= 2) {
			for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
				((NedTableInstrument *) lptr->data)->tripletRecognition(base);
			}
			for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
				((NedTableInstrument *) lptr->data)->combinePossibleTripletChords();
			}
			for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
				((NedTableInstrument *) lptr->data)->buildTriplets(base, false);
			}
			for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
				((NedTableInstrument *) lptr->data)->handleTuplets(base);
			}
			for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
				((NedTableInstrument *) lptr->data)->freeUnhandledTuplets();
			}
		}
		if (m_with_single_triplets) {
			for (base = NOTE_4; base >= NOTE_4; base /= 2) {
			//for (base = NOTE_4; base <= WHOLE_NOTE; base *= 2) {
				for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
					((NedTableInstrument *) lptr->data)->tripletRecognition(base);
				}
				for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
					((NedTableInstrument *) lptr->data)->combinePossibleTripletChords();
				}
				for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
					((NedTableInstrument *) lptr->data)->buildTriplets(base, true);
				}
				for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
					((NedTableInstrument *) lptr->data)->handleTuplets(base);
				}
				for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
					((NedTableInstrument *) lptr->data)->freeUnhandledTuplets();
				}
			}
		}
	}
	for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
		((NedTableInstrument *) lptr->data)->snapNotes();
	}
	for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
		((NedTableInstrument *) lptr->data)->resort();
	}
	for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
		((NedTableInstrument *) lptr->data)->detectClashes();
	}
	for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
		((NedTableInstrument *) lptr->data)->buildChords();
	}
	for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
		((NedTableInstrument *) lptr->data)->distributeOverlapping(m_voice_num);
	}
	for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
		((NedTableInstrument *) lptr->data)->fillUpUnwantedRests();
	}
	for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
		((NedTableInstrument *) lptr->data)->determineLines();
	}
	/*
	for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
		((NedTableInstrument *) lptr->data)->testForMeasureCrossingNotes(this);
	}
	*/
/*
	for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
		((NedTableInstrument *) lptr->data)->testForRests();
	}
	for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
		((NedTableInstrument *) lptr->data)->insertRests();
	}
	for (lptr = g_list_first(m_instrument_list); lptr; lptr = g_list_next(lptr)) {
		((NedTableInstrument *) lptr->data)->snapNotesToMeasures(true, m_numerator * WHOLE_NOTE / m_denominator);
	}
	*/
}

int NedInstrumentTable::getPartCount() {
	return g_list_length(m_instrument_list);
}

int NedInstrumentTable::getOctaveShift(int staff_nr) {
	GList *lptr;

	lptr = g_list_nth(m_instrument_list, staff_nr);
	if (lptr == NULL) {
		NedResource::Abort("NedInstrumentTable::getOctaveShift");
	}
	return ((NedTableInstrument *) lptr->data)->getOctaveShift();
}

int NedInstrumentTable::getMidiPgm(int staff_nr) {
	GList *lptr;

	lptr = g_list_nth(m_instrument_list, staff_nr);
	if (lptr == NULL) {
		NedResource::Abort("NedInstrumentTable::getMidiPgm");
	}
	return ((NedTableInstrument *) lptr->data)->getMidiPgm();
}

int NedInstrumentTable::getChannel(int staff_nr) {
	GList *lptr;

	lptr = g_list_nth(m_instrument_list, staff_nr);
	if (lptr == NULL) {
		NedResource::Abort("NedInstrumentTable::getChannel");
	}
	return ((NedTableInstrument *) lptr->data)->m_midi_channel;
}

int NedInstrumentTable::getVolume(int staff_nr) {
	GList *lptr;

	lptr = g_list_nth(m_instrument_list, staff_nr);
	if (lptr == NULL) {
		NedResource::Abort("NedInstrumentTable::getVolume");
	}
	return ((NedTableInstrument *) lptr->data)->m_midi_volume;
}

int NedInstrumentTable::getClef(int staff_nr) {
	GList *lptr;

	lptr = g_list_nth(m_instrument_list, staff_nr);
	if (lptr == NULL) {
		NedResource::Abort("NedInstrumentTable::getClef");
	}
	return ((NedTableInstrument *) lptr->data)->getClef();
}

int NedInstrumentTable::getKeySig(int staff_nr) {
	GList *lptr;

	lptr = g_list_nth(m_instrument_list, staff_nr);
	if (lptr == NULL) {
		NedResource::Abort("NedInstrumentTable::getKeySig");
	}
	return ((NedTableInstrument *) lptr->data)->m_keysig;
}

char *NedInstrumentTable::getInstrumentName(int staff_nr) {
	GList *lptr;

	lptr = g_list_nth(m_instrument_list, staff_nr);
	if (lptr == NULL) {
		NedResource::Abort("NedInstrumentTable::getInstrumentName");
	}
	return ((NedTableInstrument *) lptr->data)->getInstrumentName();
}

int NedInstrumentTable::getNumerator() {
	GList *lptr;
	NedTableTimesig *timesig;

	lptr = g_list_first(timesig_list);
	if (lptr == NULL) {NedResource::DbgMsg(DBG_TESTING, "lptr == NULL\n"); return 4;}

	timesig = (NedTableTimesig *) lptr->data;
	if (timesig->m_midi_time != 0) {NedResource::DbgMsg(DBG_TESTING, "ltimesig->m_midi_time != 0\n"); return 4;}
	NedResource::DbgMsg(DBG_TESTING, " getNumerator = %d time = %llu\n", timesig->m_numerator, timesig->m_midi_time);
	return timesig->m_numerator;
}


int NedInstrumentTable::getDenominator() {
	GList *lptr;
	NedTableTimesig *timesig;

	lptr = g_list_first(timesig_list);
	if (lptr == NULL) return 4;

	timesig = (NedTableTimesig *) lptr->data;
	if (timesig->m_midi_time != 0) return 4;
	return timesig->m_denominator;
}

void NedInstrumentTable::getMeasureDurationAt(unsigned long long midi_time, unsigned long long *measure_start, unsigned long long *measure_end) {
	GList *lptr;
	unsigned int duration;
	NedTableTimesig *timesig;
	unsigned long long dist, fac;

	for (lptr = g_list_last(timesig_list); lptr; lptr = g_list_previous(lptr)) {
		timesig = (NedTableTimesig *) lptr->data;
		if (timesig->m_midi_time <= midi_time) {
			duration = timesig->m_numerator * WHOLE_NOTE / timesig->m_denominator;
			dist = midi_time - timesig->m_midi_time;
			fac = dist / duration;
			*measure_start = timesig->m_midi_time + fac * duration;
			*measure_end = *measure_start + duration;
			return;
		}
	}

	fac = midi_time / WHOLE_NOTE;
	*measure_start = fac * WHOLE_NOTE;
	*measure_end = *measure_start + WHOLE_NOTE;
}


NedTableTemposig *NedInstrumentTable::getCurrentTempo(int *tempo_so_far, unsigned long long midi_time) {
	GList *lptr;
	NedTableTemposig *temposig;
	int dist;

	for (lptr = g_list_last(temposig_list); lptr; lptr = g_list_previous(lptr)) {
		temposig = (NedTableTemposig *) lptr->data;
		//if (temposig->m_used) continue;
		if (temposig->m_midi_time > midi_time) continue;
		dist = *tempo_so_far - temposig->m_tempo;
		if (dist < 0) dist = -dist;
		if (100.0 * (double) dist / (double) (*tempo_so_far + 1) < m_tempo_change_density) {
			return NULL;
		}
		//temposig->m_used = true;
		*tempo_so_far = temposig->m_tempo;
		return temposig;
	}
	return NULL;
}

int NedInstrumentTable::getStartTempo() {
	GList *lptr;

	if ((lptr = g_list_first(temposig_list)) == NULL) {
		return 120;
	}

	return ((NedTableTemposig *) lptr->data)->m_tempo;
}
	

void NedInstrumentTable::computeTimeSigMeasureNumbers(NedMainWindow *main_window) {
	unsigned int num = 0;
	unsigned long long last_timesig_time = 0;
	int dummy;
	GList *lptr;
	unsigned int duration = WHOLE_NOTE;
	NedTableTimesig *timesig;

	for (lptr = g_list_first(timesig_list); lptr; lptr = g_list_next(lptr)) {
		timesig = (NedTableTimesig *) lptr->data;
		num = timesig->m_measure_number = num + (timesig->m_midi_time - last_timesig_time) / duration;
		if (timesig->m_midi_time > 0) {
			main_window->setSpecialTimesig(num + 1, timesig->m_numerator, timesig->m_denominator, &dummy, &dummy, false);
		}
		duration = timesig->m_numerator * WHOLE_NOTE / timesig->m_denominator;
		last_timesig_time = timesig->m_midi_time;
	}
}

GList *NedInstrumentTable::convertToNtEdNotes(int staff_nr, int voice_nr) {
	GList *lptr;

/*
	lptr = g_list_first(m_instrument_list);
	return ((NedTableInstrument *) lptr->data)->convertToNtEdNotes(voice_nr, m_numerator * WHOLE_NOTE / m_denominator);
	*/
	lptr = g_list_nth(m_instrument_list, staff_nr);
	if (lptr == NULL) {
		NedResource::Abort("NedInstrumentTable::convertToNtEdNotes");
	}
	return ((NedTableInstrument *) lptr->data)->convertToNtEdNotes(voice_nr, this, staff_nr == 0 && voice_nr == 0);
}
