/*
 ** Copyright(C) 2004 Eric Leblond <regit@inl.fr>
 **              2005 : deal with seeded/unseeded cases. Patch from Julian Reich <jreich@epplehaus.de>
 **	       	2005 : "partial" rewrite by Julian Reich <jreich@epplehaus.de>
 **
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, version 2 of the License.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software
 ** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <auth_srv.h>
#include <tls.h>
#include <gcrypt.h>
#include <sasl/saslutil.h>

/**
 * verify user password against user authentication module.
 */
int verify_user_password(const char *given_password, const char *stored_password){
    int algorithm = GCRY_MD_NONE;
    int seeded = 0;
    int seeded_comparable_length = 0;
    char *comparable_given;
    char *comparable_stored;
    
#ifdef DEBUG_ENABLE
    if (DEBUG_OR_NOT(DEBUG_LEVEL_VERBOSE_DEBUG, DEBUG_AREA_MAIN)) {
        g_message("User given password is: %s", given_password);
	g_message("Stored password is: %s", stored_password);
    }
#endif
    
    if (!strncmp("{SHA}", stored_password, 5)) {
        // SHA1
	
	algorithm = GCRY_MD_SHA1;
	
#ifdef DEBUG_ENABLE
	if (DEBUG_OR_NOT(DEBUG_LEVEL_VERBOSE_DEBUG, DEBUG_AREA_MAIN)) {
	    g_message("Stored password is SHA1 encrypted");
	}
#endif
    } else if (!strncmp("{SHA1}", stored_password, 6)) {
        // SHA1
	
	algorithm = GCRY_MD_SHA1;
	
#ifdef DEBUG_ENABLE
	if (DEBUG_OR_NOT(DEBUG_LEVEL_VERBOSE_DEBUG, DEBUG_AREA_MAIN)) {
	    g_message("Stored password is SHA1 encrypted");
	}
#endif
    } else if (!strncmp("{SSHA}", stored_password, 6)) {
        // SSHA (seeded SHA1)
	
	algorithm = GCRY_MD_SHA1;
	seeded = 1;
	seeded_comparable_length = 20;
        
#ifdef DEBUG_ENABLE
	if (DEBUG_OR_NOT(DEBUG_LEVEL_VERBOSE_DEBUG, DEBUG_AREA_MAIN)) {
	    g_message("Stored password is SHA1 encrypted (seeded)");
	}
#endif
    } else if (!strncmp("{MD5}", stored_password, 5)) {
        // MD5
	
	algorithm = GCRY_MD_MD5;
	
#ifdef DEBUG_ENABLE
	if (DEBUG_OR_NOT(DEBUG_LEVEL_VERBOSE_DEBUG, DEBUG_AREA_MAIN)) {
	    g_message("Stored password is MD5 encrypted");
	}
#endif
    } else if (!strncmp("{SMD5}", stored_password, 6)) {
	// SMD5 (seeded MD5)
	
	algorithm = GCRY_MD_MD5;
	seeded = 1;
	seeded_comparable_length = 16;
	
#ifdef DEBUG_ENABLE
	if (DEBUG_OR_NOT(DEBUG_LEVEL_VERBOSE_DEBUG, DEBUG_AREA_MAIN)) {
	    g_message("Stored password is MD5 encrypted (seeded)");
	}
#endif
    } else {
	// plaintext
	
#ifdef DEBUG_ENABLE
	if (DEBUG_OR_NOT(DEBUG_LEVEL_VERBOSE_DEBUG, DEBUG_AREA_MAIN)) {
	    g_message("Stored password is plaintext");
	}
#endif
    }
    
#ifdef DEBUG_ENABLE
    if (DEBUG_OR_NOT(DEBUG_LEVEL_VERBOSE_DEBUG, DEBUG_AREA_MAIN)) {
	if (seeded) {
	    g_message("Comparable length is %d characters because stored password hash is seeded", seeded_comparable_length);
	}
    }
#endif
    
    if (algorithm == GCRY_MD_NONE) {
	    comparable_given  = g_strdup(given_password);
	    comparable_stored = g_strdup(stored_password);
    } else {
	    gcry_md_hd_t gcrypt_handle;
	    char *temp_given_hash;
	    char temp_given_decoded[50];
	    int temp_given_decoded_length = 0;
	    char **temp_stored;
	    
	    gcry_md_open(&gcrypt_handle, algorithm, 0);
	    gcry_md_write(gcrypt_handle, given_password, strlen(given_password));
            
	    temp_stored = g_strsplit(stored_password, "}", 2);
	    
	    if (seeded) {
                char *temp;
		int length;
		
		temp = malloc(seeded_comparable_length + 4);
		
		sasl_decode64(temp_stored[1], strlen(temp_stored[1]), temp, (seeded_comparable_length + 4), &length);
		
		gcry_md_write(gcrypt_handle, (temp + seeded_comparable_length), 4);
		
		free(temp);
	    }
	    
	    temp_given_hash = gcry_md_read(gcrypt_handle, algorithm);
            
#ifdef DEBUG_ENABLE
	    if (DEBUG_OR_NOT(DEBUG_LEVEL_VERBOSE_DEBUG, DEBUG_AREA_MAIN)) {
	        g_message("Hash of user given password is: %s", temp_given_hash);
            }
#endif
	    
	    sasl_encode64(
                temp_given_hash,
		strlen(temp_given_hash),
		temp_given_decoded,
		50,
		&temp_given_decoded_length
            );
	    
#ifdef DEBUG_ENABLE
	    if (DEBUG_OR_NOT(DEBUG_LEVEL_VERBOSE_DEBUG, DEBUG_AREA_MAIN)) {
	        g_message("Decoded hash of user given password is: %s", temp_given_decoded);
	    }
#endif
	    
	    if (seeded) {
		comparable_given  = g_strndup(temp_given_decoded, seeded_comparable_length);
		comparable_stored = g_strndup(temp_stored[1], seeded_comparable_length);
	    } else {
	        comparable_given  = g_strdup(temp_given_decoded);
	        comparable_stored = g_strdup(temp_stored[1]);
	    }
	    
	    g_strfreev(temp_stored);
    }
    
#ifdef DEBUG_ENABLE
    if (DEBUG_OR_NOT(DEBUG_LEVEL_VERBOSE_DEBUG, DEBUG_AREA_MAIN)) {
	g_message("Comparable part of user given password is: %s", comparable_given);
	g_message("Comparable part of stored password is: %s\n", comparable_stored);
    }
#endif
    
    if (!strcmp(comparable_given, comparable_stored)) {
	free(comparable_given);
	free(comparable_stored);
	
        return SASL_OK;
    } else {
	free(comparable_given);
	free(comparable_stored);
	
        return SASL_BADAUTH;
    }
}
