program shanks;
(* file exemples/pascal/shanks.p: square root modulo an odd prime
 *-----------------------------------------------------------------------+
 |  Copyright 2005, Michel Quercia (michel.quercia@prepas.org)           |
 |                                                                       |
 |  This file is part of Numerix. Numerix is free software; you can      |
 |  redistribute it and/or modify it under the terms of the GNU Lesser   |
 |  General Public License as published by the Free Software Foundation; |
 |  either version 2.1 of the License, or (at your option) any later     |
 |  version.                                                             |
 |                                                                       |
 |  The Numerix Library is distributed in the hope that it will be       |
 |  useful, but WITHOUT ANY WARRANTY; without even the implied warranty  |
 |  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU  |
 |  Lesser General Public License for more details.                      |
 |                                                                       |
 |  You should have received a copy of the GNU Lesser General Public     |
 |  License along with the GNU MP Library; see the file COPYING. If not, |
 |  write to the Free Software Foundation, Inc., 59 Temple Place -       |
 |  Suite 330, Boston, MA 02111-1307, USA.                               |
 +-----------------------------------------------------------------------+
 |                                                                       |
 |            Racine carre modulo p, algorithme de Shanks               |
 |                                                                       |
 +-----------------------------------------------------------------------*)

uses _name_;

                     (* +----------------------------+
                        |  Test de pseudo-primalit  |
                        +----------------------------+ *)
(* p = naturel impair
   retourne 0 si p est compos
            1 si p est premier
            2 si p n'a pas de facteurs triviaux et est pseudo-premier fort
              modulo nbases bases alatoires
*)
const nbases  = 10;
const cr_size = 1000;
   
var crible:array[0..cr_size-1] of boolean; (* crible[i] = (2i+3) est premier *)
const cr_ok : boolean =  false;            (* crible dj calcul ?          *)
   
function pseudo_prime(p:xint):longint;
var x,q: xint;
    i,j,k,n:longint;
begin
   
   (* calcule crible par la mthode d'Etatosthne *)
   if not(cr_ok) then begin
      for i:=0 to cr_size-1 do crible[i] := true;
      i := 0; j := 3;
      while j < cr_size do begin
         crible[j] := false;
         j := j + 2*i + 3;
         if j >= cr_size then begin
            repeat i := i+1 until (i >= cr_size) or crible[i];
            j := 2*i*(i+3) + 3;
         end
      end;
      cr_ok := true;
   end;

   (* si p est petit, cherche dans le crible *)
   if inf_1(p,2*cr_size+3) then begin
      if crible[(int_of(p)-3) div 2] then pseudo_prime := 1
      else pseudo_prime := 0;
      exit;
   end;

   (* sinon, cherche si p a un diviseur dans le crible *)
   for i := 0 to cr_size-1 do
      if mod_1(p,2*i+3) = 0 then begin
         pseudo_prime := 0;
         exit;
      end;

   (* sinon, et si p < (2*cr_size+3)^2 alors p est premier *)
   if inf_1(p,(2*cr_size+3)*(2*cr_size+3)) then begin
      pseudo_prime := 1;
      exit;
   end;

   (* dcompose p-1 = 2^k*q, q impair *)
   k := 1; while not(nth_bit(p,k)) do k := k+1;
   q := f_shr(p, k);
   x := xnew();
   n := nbits(p);

   (* Test de Rabin-Miller *)
   for i := 0 to nbases-1 do begin

      (* tire une base au hasard en vitant 0,1,-1 *)
      repeat nrandom(x,n); gmod(x,x,p,1) until nbits(x) >= 2;

      (* x <- |x^q mod p|, si x = 1, c'est bon *)
      gpowmod(x,x,q,p,1); abs(x,x);
      if neq_1(x,1) then begin

         (* lve au carr jusqu' trouver 0,1,-1 ou atteindre l'exposant (p-1)/2 *)
         j := 1;
         while (nbits(x) > 1) and (j < k) do begin
            sqr(x,x); gmod(x,x,p,1);
            j := j+1;
         end;

        (* p est compos si x <> -1 *)
        if neq_1(x,-1) then begin
           xfree(x);
           xfree(q);
           pseudo_prime := 0;
           exit;
        end
      end
         
   end;

   (* p est probablement premier *)
   xfree(x);
   xfree(q);
   pseudo_prime := 2;

end;


                       (* +------------------------+
                          |  Symbole de Kronecker  |
                          +------------------------+ *)

(* a = naturel quelconque, b = naturel impair *)
function kronecker(a,b:xint):longint;
var x,y,z : xint;
    i,k   : longint;

begin
   
   x := f_copy(a);
   y := f_copy(b);
   
   (* Algorithme d'Euclide + rciprocit quadratique *)
   k := 1;
   while neq_1(x,0) do begin

      (* i <- valuation 2-adique de x, x <- x/2^i *)
      i := 0; while not(nth_bit(x,i)) do i := i+1;
      shiftr(x, x, i);

      (* (2^i/y) = -1 ssi i est impair et y = 3 ou 5 mod 8 *)
      if ((i and 1) = 1) and (((nth_word(y,0) + 2) and 7) > 4) then k := -k;

      (* (x/y) = (y/x) si x ou y = 1 mod 4, -(y/x) sinon *)
      if nth_bit(x,1) and nth_bit(y,1) then k := -k;

      (* y <- x, x <- y mod x *)
      z := x; x := y; y := z; gmod(x, x, y, 1);

      (* (-1/y) = 1 ssi y = 1 mod 4 *)
      if sgn(x) < 0 then begin
         abs(x,x);
         if nth_bit(y,1) then k := -k;
      end
      
   end;

   (* si pgcd(x,y) != 1 alors (x/y) = 0 *)
   if neq_1(y,1) then k := 0;

   xfree(x);
   xfree(y);
   kronecker := k;

end;



                      (* +---------------------------+
                         |  Racine carre modulaire  |
                         +---------------------------+ *)

(* p = premier impair, a = b^2 mod p, retourne b (compris entre 0 et p/2) *)
procedure sqrtmod(var b:xint; a,p:xint);
var q,x,y,z : xint;
    k,l     : longint;
begin
   
   q := xnew();
   x := xnew();
   y := xnew();
   z := xnew();

   (* dcompose p = 1 + 2^k*(2*q+1) *)
   k := 1; while not(nth_bit(p,k)) do k := k+1;
   shiftr(q, p, k+1);

   (* b <- a^(q+1) mod p, x <- a^(2*q+1) mod p (donc b^2 = a*x mod p) *)
   powmod(x, a, q, p);
   mul(b, x, a); modulo(b, b, p);
   mul(x, x, b); modulo(x, x, p);

   if sup_1(x,1) then begin

      (* y <- lment d'ordre 2^k *)
      repeat nrandom(y, nbits(p)) until kronecker(y,p) = -1;
      shiftl(q, q, 1); add_1(q, q, 1);
      powmod(y, y, q, p);

      repeat
       
         (* ordre de x = 2^l *)
         sqr(z, x); modulo(z, z, p);
         l := 1; while (l < k) and neq_1(z,1) do begin
            l := l+1;
            sqr(z, z); modulo(z, z, p);
         end;
         if l >= k then begin
            writeln('internal error: l >= k');
            exit;
         end;

         (* b <- b*y^(2^(k-l-1))
            x <- x*y^(2^(k-l)), 
            y <- y^(2^(k-l))
            donc b^2 = a*x mod p, ordre(y) = 2^l et ordre(x) < 2^l *)
         while l < k-1 do begin
            k := k-1;
            sqr(y, y); modulo(y, y, p);
         end;
         mul(b, b, y); modulo(b, b, p);
         sqr(y, y);    modulo(y, y, p); k := k-1;
         mul(x, x, y); modulo(x, x, p);

      until eq_1(x,1);
      
   end;

   (* rduit b  l'intervalle [0,p/2] *)
   gmod(b, b, p, 1); abs(b, b);

   (* termin *)
   xfree(x);
   xfree(y);
   xfree(z);

end;

                        (* +-----------------------+
                           |  Fonction principale  |
                           +-----------------------+ *)

var help,test,p_ok,a_ok : boolean;
    n,i   : longint;
    p,a,b : xint;
    s,cmd : ansistring;
    c     : word;
   
const test_p = '100000000000000000000000000000000000133';
      test_a = '123456';
      test_b = '36705609512061164177523976477230978260';
begin
   help := false; test := false; p_ok := false; a_ok := false;
   n := 200;
  
    p := xnew();
    a := xnew();
    b := xnew();
    cmd := paramstr(0);

   (* dcode les arguments *)
   i := 1;
   while not(help) and not(test) and (i <= paramcount) do begin
      if paramstr(i) = '-p' then begin
         i := i+1;
         if i <= paramcount then begin
            s := paramstr(i);
            copy_string(p,pchar(s)); p_ok := true;
         end
         else help := true;
      end
      else if paramstr(i) = '-a' then begin
         i := i+1;
         if i <= paramcount then begin
            s := paramstr(i);
            copy_string(a,pchar(s)); a_ok := true;
         end
         else help := true;
      end
      else if paramstr(i) = '-bits' then begin
         i := i+1;
         if i <= paramcount then val(paramstr(i),n,c)
         else help := true;
      end
      else if paramstr(i) = '-test' then begin
         copy_string(p,test_p);
         copy_string(a,test_a);
         test := true; p_ok := true; a_ok := true;
      end
      else help := true;
      i := i+1;
   end;
   
   if help then writeln('usage: ',cmd,' [-p <odd prime>] [-a <quadres mod p>] [-test] [-bits n]')
   else begin

      random_init(0);

      (* si p est dfini, vrifie qu'il est premier impair.
         sinon tire un nombre premier impair au hasard *)
      if p_ok then begin
         if not(nth_bit(p,0)) or (pseudo_prime(p) = 0) then begin
            writeln('p is not an odd prime');
            exit;
         end
      end
      else begin
         nrandom1(p, n); if not(nth_bit(p,0)) then add_1(p,p,1);
         repeat add_1(p, p, 2) until pseudo_prime(p) > 0;
         s := string_of(p); writeln('p = ',s);
      end;

      (* si a est dfini, vrifie que c'est un carr non nul modulo p.
         sinon tire un rsidu quadratique au hasard.*)
      if a_ok then begin
         if kronecker(a,p) <> 1 then begin
            writeln('kronecker(a,p) != 1');
            exit;
         end
      end
      else begin
         nrandom(a,n);
         repeat add_1(a, a, 1) until kronecker(a,p) = 1;
         s := string_of(a); writeln('a = ',s);
      end;

      (* calcule la racine carre *)
      sqrtmod(b, a, p);
      s := string_of(b);
      if test then begin
         if s <> test_b then writeln('error in the ',cmd,' test')
         else                writeln(cmd,#9'test ok');
      end
      else                   writeln('b = ',s);

   end;

   xfree(p);
   xfree(a);
   xfree(b);

end.
