// file kernel/n/c/add.c: addition/subtraction of natural integers
/*-----------------------------------------------------------------------+
 |  Copyright 2005, Michel Quercia (michel.quercia@prepas.org)           |
 |                                                                       |
 |  This file is part of Numerix. Numerix is free software; you can      |
 |  redistribute it and/or modify it under the terms of the GNU Lesser   |
 |  General Public License as published by the Free Software Foundation; |
 |  either version 2.1 of the License, or (at your option) any later     |
 |  version.                                                             |
 |                                                                       |
 |  The Numerix Library is distributed in the hope that it will be       |
 |  useful, but WITHOUT ANY WARRANTY; without even the implied warranty  |
 |  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU  |
 |  Lesser General Public License for more details.                      |
 |                                                                       |
 |  You should have received a copy of the GNU Lesser General Public     |
 |  License along with the GNU MP Library; see the file COPYING. If not, |
 |  write to the Free Software Foundation, Inc., 59 Temple Place -       |
 |  Suite 330, Boston, MA 02111-1307, USA.                               |
 +-----------------------------------------------------------------------+
 |                                                                       |
 |                         Addition/soustraction                         |
 |                                                                       |
 +-----------------------------------------------------------------------*/

/* ---------------------------------------- Addition
  entre :
  a = naturel de longueur la
  b = naturel de longueur lb <= la
  c = naturel de longueur la

  sortie :
  c <- a + b
  retourne la retenue
*/
#ifndef assembly_sn_add
chiffre xn(add)(chiffre *a, long la, chiffre *b, long lb, chiffre *c) {
  ndouble r;
  long    i;

  for (r=0, i=0; i<lb; i++)
       {r += (ndouble)a[i] + (ndouble)b[i]; c[i] = r; r >>= HW;}
  for ( ; i < la; i++) {r += (ndouble)a[i]; c[i] = r; r >>= HW;}
  return(r);
}
#endif /* assembly_sn_add */

/* ---------------------------------------- Incrmentation
  entre :
  a = naturel de longueur la
  b = naturel de longueur lb avec lb <= la

  sortie :
  a <- a + b
  retourne la retenue

  remarque :
  xn(inc) peut tre utilis pour diviser par 1 - BASE^p si p >= 1
  la version assembleur x86 ne peut tre utilise dans ce but que
  pour p >= 2
*/
#ifndef assembly_sn_inc
chiffre xn(inc)(chiffre *a, long la, chiffre *b, long lb) {
  ndouble r;
  long    i;

  for (r=0, i=0; i < lb; i++)
                {r += (ndouble)a[i] + (ndouble)b[i]; a[i] = r; r >>= HW;}
  for ( ; (i < la) && (r); i++) {r += (ndouble)a[i]; a[i] = r; r >>= HW;}
  return(r);
}
#endif /* assembly_sn_inc */

/* ---------------------------------------- Incrmentation de 1
  entre :
  a = naturel de longueur la

  sortie :
  a <- a + 1
  retourne la retenue
*/
#ifndef assembly_sn_inc1
chiffre xn(inc1)(chiffre *a, long la) {
  ndouble r;
  long    i;

  for (r=1, i=0; (i < la) && (r); i++)
    {r += (ndouble)a[i]; a[i] = r; r >>= HW;}
  return(r);
}
#endif /* assembly_sn_inc1 */

/* ---------------------------------------- Soustraction
  entre :
  a = naturel de longueur la
  b = naturel de longueur lb <= la
  c = naturel de longueur la

  sortie :
  c <- a - b
  retourne la retenue
*/
#ifndef assembly_sn_sub
chiffre xn(sub)(chiffre *a, long la, chiffre *b, long lb, chiffre *c) {
  zdouble r;
  long    i;

  for (r=0, i=0; i < lb; i++)
       {r += (ndouble)a[i] - (ndouble)b[i]; c[i] = r; r >>= HW;}
  for ( ; i < la; i++) {r += (ndouble)a[i]; c[i] = r; r >>= HW;}
  return(-r);
}
#endif /* assembly_sn_sub */

/* ---------------------------------------- Dcrmentation
  entre :
  a = naturel de longueur la
  b = naturel de longueur lb avec lb <= la

  sortie :
  a <- a - b
  retourne la retenue
*/
#ifndef assembly_sn_dec
chiffre xn(dec)(chiffre *a, long la, chiffre *b, long lb) {
  zdouble r;
  long    i;

  for (r=0, i=0; i < lb; i++)
                {r += (ndouble)a[i] - (ndouble)b[i]; a[i] = r; r >>= HW;}
  for ( ; (i < la) && (r); i++) {r += (ndouble)a[i]; a[i] = r; r >>= HW;}
  return(-r);
}
#endif /* assembly_sn_dec */

/* ---------------------------------------- Dcrmentation de 1
  entre :
  a = naturel de longueur la

  sortie :
  a <- a - 1
  retourne la retenue
*/
#ifndef assembly_sn_dec1
chiffre xn(dec1)(chiffre *a, long la) {
  zdouble r;
  long    i;

  for (r=-1, i=0; (i < la) && (r); i++)
    {r += (ndouble)a[i]; a[i] = r; r >>= HW;}
  return(-r);
}
#endif /* assembly_sn_dec1 */
