// file kernel/n/c/div_n2.c: O(n^2) division of natural integers
/*-----------------------------------------------------------------------+
 |  Copyright 2005, Michel Quercia (michel.quercia@prepas.org)           |
 |                                                                       |
 |  This file is part of Numerix. Numerix is free software; you can      |
 |  redistribute it and/or modify it under the terms of the GNU Lesser   |
 |  General Public License as published by the Free Software Foundation; |
 |  either version 2.1 of the License, or (at your option) any later     |
 |  version.                                                             |
 |                                                                       |
 |  The Numerix Library is distributed in the hope that it will be       |
 |  useful, but WITHOUT ANY WARRANTY; without even the implied warranty  |
 |  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU  |
 |  Lesser General Public License for more details.                      |
 |                                                                       |
 |  You should have received a copy of the GNU Lesser General Public     |
 |  License along with the GNU MP Library; see the file COPYING. If not, |
 |  write to the Free Software Foundation, Inc., 59 Temple Place -       |
 |  Suite 330, Boston, MA 02111-1307, USA.                               |
 +-----------------------------------------------------------------------+
 |                                                                       |
 |                         Division quadratique                          |
 |                                                                       |
 +-----------------------------------------------------------------------*/


                        /* +-------------------------+
                           |  Division par un long   |
                           +-------------------------+ */

/*
  entre :
  a = naturel de longueur la >= 0
  b = long > 0
  c = naturel de longueur la, peut tre confondu avec a

  sortie :
  c <- floor(a/b)
  retourne a mod b
*/
#ifndef assembly_sn_div_1
unsigned long xn(div_1)(chiffre *a, long la, unsigned long b, chiffre *c) {
  ndouble r;

#if chiffres_per_long == 2
  if (b > BASE) { /* division  deux chiffres */
    ndouble b0,b1,q,s,t;
    long l;

    /* vacue les petits dividendes */
    if (la == 0) return(0);
    if (la == 1) {r = (ndouble)a[0]; c[0] = 0; return(r);}

    /* dcale b pour avoir b1 >= BASE/2 */
    for (l=HW; (b & SIGN_m) == 0; l--, b<<=1);
    b0 = b & (BASE-1);
    b1 = b >> HW;

    /* divise a*2^(HW-l) par b*2^(HW-l) */
    a += la-3; c += la-2; la -= 2;
    r = ((ndouble)a[2] << HW) + (ndouble)a[1];
    c[1] = 0;
    s = r >> l;
    for (; la>=0; la--, a--, c--) {
      if (la) r = (r<<HW) + (ndouble)a[0]; else r <<= HW;
      q = s/b1;
      s = ((s - q*b1) << HW) + ((r >> l) & (BASE-1));
      t = q*b0;
      while (s < t) {q--; t -= s; s = b;}
      c[0] = q;
      s -= t;
    }

    /* retourne le reste dcal de HW-l bits */
    return(s>>(HW-l));

  }
#endif /* chiffres_per_long == 2 */

  for (r = 0, a+=la-1, c+=la-1; la ; la--, a--, c--) {
    r = (r << HW) + (ndouble)a[0];
    c[0] = r/b;
    r %= b;
  }

  return(r);
}
#endif /* assembly_sn_div_1 */

/*
  entre :
  a = naturel de longueur la >= 0
  b = long > 0

  sortie :
  retourne a mod b
*/
#ifndef assembly_sn_mod_1
unsigned long xn(mod_1)(chiffre *a, long la, unsigned long b) {
  ndouble r;

#if chiffres_per_long == 2
  if (b > BASE) { /* division  deux chiffres */
    ndouble b0,b1,q,s,t;
    long l;

    /* vacuel les petits dividendes */
    if (la == 0) return(0);
    if (la == 1) return((ndouble)a[0]);

    /* dcale b pour avoir b >= BASE/2 */
    for (l=HW; (b & SIGN_m) == 0; l--, b<<=1);
    b0 = b & (BASE-1);
    b1 = b >> HW;

    /* divise a*2^(HW-l) par b*2^(HW-l) */
    a += la-3; la -= 2;
    r = ((ndouble)a[2] << HW) + (ndouble)a[1];
    s = r >> l;
    for (; la>=0; la--, a--) {
      if (la) r = (r<<HW) + (ndouble)a[0]; else r <<= HW;
      q = s/b1;
      s = ((s - q*b1) << HW) + ((r >> l) & (BASE-1));
      t = q*b0;
      while (s < t) {q--; t -= s; s = b;}
      s -= t;
    }

    /* retourne le reste dcal de HW-l bits */
    return(s>>(HW-l));

  }
#endif /* chiffres_per_long == 2 */

  for (r = 0, a+=la-1; la ; la--, a--) {
    r = (r << HW) + (ndouble)a[0];
    r %= b;
  }

  return(r);

}
#endif /* assembly_sn_mod_1 */





                   /* +------------------------+
                      |  Division quadratique  |
                      +------------------------+ */

/*
  entre :
  a = naturel de longueur lc+lb
  b = naturel de longueur lb
  c = naturel de longueur lc

  contraintes :
  lb >= 2, lc > 0, le bit de poids fort de b est non nul,
  a < BASE^lc*b
  a,b,c non confondus

  sortie :
  a <- a mod b
  c <- floor(a/b)
*/

#ifndef assembly_sn_div_n2
#ifdef debug_div_n2
void xn(div_n2_buggy)
#else
void xn(div_n2)
#endif
(chiffre *a, long lc, chiffre *b, long lb, chiffre *c) {
  chiffre q, b1=b[lb-1];
  ndouble r;
  zdouble z;
  long i;

  for(a += lc-1, c += lc-1; lc; lc--,a--,c--) {

    /* quotient approch, peut tre trop grand d'au plus 2 units */
    if (a[lb] >= b1) {q = BASE-1;}
    else {
      r = (ndouble)a[lb-1] + ((ndouble)a[lb] << HW);
      q = r/b1;
    }

    /* a <- a - q*b */
    for(r=0, z=0, i=0; i<lb; i++) {
      r += q*(ndouble)b[i];
      z += (ndouble)a[i] - (r & (BASE-1));
      a[i] = z;
      r >>=HW; z >>= HW;
    }
    z += (ndouble)a[lb] - r;
    a[lb] = z;

    /* corrige le quotient et le reste si < 0 */
    while(a[lb]) {q--; xn(inc)(a,lb+1,b,lb);}

    /* quotient dfinitif */
    c[0] = q;
  }
}
#endif /* assembly_sn_div_n2 */

                         /* +------------+
                            |  Contrle  |
                            +------------+ */

#ifdef debug_div_n2
void xn(div_n2_buggy)(chiffre *a, long la, chiffre *b, long lb, chiffre *c);
void xn(div_n2)(chiffre *a, long lc, chiffre *b, long lb, chiffre *c) {
  long la = lc+lb;
  chiffre *x, *y;

  x = xn(alloc_tmp)(2*la); y = x+la;

  /* validit des longueurs ? */
  if ((lb < 2) || (la < lb))
      xn(internal_error)("error, div_n2 is called with lb < 2 or la < lb",0);

  /* le bit de poids fort de b est non nul ? */
  if ((ndouble)b[lb-1] < BASE/2)
    xn(internal_error)("error, div_n2 is called with msb(b) = 0",0);

  /* a < BASE^(la-lb)*b ? */
  if (xn(cmp)(a+lc,lb,b,lb) >= 0)
       xn(internal_error)("error, div_n2 is called with a >= BASE^lc*b",2,a,la,b,lb);

  /* effectue la division douteuse */
  xn(move)(a,la,x);
  xn(div_n2_buggy)(a,lc,b,lb,c);

  /* a_entre = a_sortie + b*c et a_sortie < b ?*/
  if (lc < lb) xn(toommul)(b,lb,c,lc,y); else xn(toommul)(c,lc,b,lb,y);
  if (     (xn(inc)(y,la,a,lb) != 0)
        || (xn(cmp)(x,la,y,la) != 0)
        || (xn(cmp)(a,la,b,lb) >= 0))
      xn(internal_error)("error in div_n2", 4,x,la,b,lb,a,la,c,lc);

  xn(free_tmp)(x);

}
#endif /* debug_div_n2 */

