// file kernel/n/c/random.c: random numbers
/*-----------------------------------------------------------------------+
 |  Copyright 2005, Michel Quercia (michel.quercia@prepas.org)           |
 |                                                                       |
 |  This file is part of Numerix. Numerix is free software; you can      |
 |  redistribute it and/or modify it under the terms of the GNU Lesser   |
 |  General Public License as published by the Free Software Foundation; |
 |  either version 2.1 of the License, or (at your option) any later     |
 |  version.                                                             |
 |                                                                       |
 |  The Numerix Library is distributed in the hope that it will be       |
 |  useful, but WITHOUT ANY WARRANTY; without even the implied warranty  |
 |  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU  |
 |  Lesser General Public License for more details.                      |
 |                                                                       |
 |  You should have received a copy of the GNU Lesser General Public     |
 |  License along with the GNU MP Library; see the file COPYING. If not, |
 |  write to the Free Software Foundation, Inc., 59 Temple Place -       |
 |  Suite 330, Boston, MA 02111-1307, USA.                               |
 +-----------------------------------------------------------------------+
 |                                                                       |
 |                            Nombres alatoires                         |
 |                                                                       |
 +-----------------------------------------------------------------------*/

/* ---------------------------------------- Naturel alatoire
  entre :
  a = naturel de longueur la

  sortie :
  a <- chiffres alatoires
*/

/*
  Sous Linux-i386, RAND_MAX = 2^31 - 1.
  Sur la machine beaune de l'Inria, RAND_MAX n'est pas dfini ...
  et [man random] indique que random retourne 31 bits.

  Je suppose que de manire gnrale, RAND_MAX = 2^x - 1 avec 4x >= HW
  Si x >= HW on peut utiliser directement le rsultat de random(),
  sinon on calcule HW bits alatoires avec deux ou quatre appels  random()

  Pour garantir une identit de rsultats entre les diffrents modes,
  on tire un nombre pair de chiffres lorsque chiffres_per_long = 2.
*/
#ifndef RAND_MAX
#define RAND_MAX 0x7FFFFFFF
#endif

#if RAND_MAX >= BASE-1
void xn(random)(chiffre *a, long la) {
  long i;
  for (i=0; i < la; a[i++] = random());
#if chiffres_per_long == 2
  if (la&1) random();
#endif
}

#elif RAND_MAX >= (BASE >> (HW/2)) - 1
void xn(random)(chiffre *a, long la) {
  long i;
  chiffre mask = (1<<(HW/2)) - 1;
  for (i=0; i < la; a[i++] = (random() & mask) + (random() << (HW/2)));
#if chiffres_per_long == 2
  if (la&1) {random(); random();}
#endif
}

#elif RAND_MAX >= (BASE >> (3*HW/4)) - 1
void xn(random)(chiffre *a, long la) {
  long i;
  chiffre mask = (1<<(HW/4)) - 1, c;
  for (i=0; i < la; i++) {
      c = random() & mask;
      c = (c << (HW/4)) + (random() & mask);
      c = (c << (HW/4)) + (random() & mask);
      c = (c << (HW/4)) + (random() & mask);
      a[i] = c;
  }
#if chiffres_per_long == 2
  if (la&1) {random(); random(); random(); random();}
#endif
}

#else
#error "unexpected small value for RAND_MAX"
#endif

