(* file kernel/ocaml/ml/big.ml: Objective-Caml bignum interface
 +-----------------------------------------------------------------------+
 |  Copyright 2005, Michel Quercia (michel.quercia@prepas.org)           |
 |                                                                       |
 |  This file is part of Numerix. Numerix is free software; you can      |
 |  redistribute it and/or modify it under the terms of the GNU Lesser   |
 |  General Public License as published by the Free Software Foundation; |
 |  either version 2.1 of the License, or (at your option) any later     |
 |  version.                                                             |
 |                                                                       |
 |  The Numerix Library is distributed in the hope that it will be       |
 |  useful, but WITHOUT ANY WARRANTY; without even the implied warranty  |
 |  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU  |
 |  Lesser General Public License for more details.                      |
 |                                                                       |
 |  You should have received a copy of the GNU Lesser General Public     |
 |  License along with the GNU MP Library; see the file COPYING. If not, |
 |  write to the Free Software Foundation, Inc., 59 Temple Place -       |
 |  Suite 330, Boston, MA 02111-1307, USA.                               |
 +-----------------------------------------------------------------------+
 |                                                                       |
 |                       Dfinition du module Big                        |
 |                                                                       |
 +-----------------------------------------------------------------------*)

                           (* +----------------+
                              |   module Big   |
                              +----------------+ *)

module Big = struct
  open Big_int

  type t     = big_int
  type tref  = t ref
  let name() = String.copy "Big"
  let zero   = zero_big_int
  let one    = unit_big_int
  let two    = big_int_of_int 2
  exception Error of string

  (* gestion des rfrences *)
  let make_ref   a = ref a
  let copy_in  r a = r := a
  let copy_out r   = !r
  let look     r   = !r

  (* addition/soustraction/multiplication *)
  let abs        a   =      abs_big_int      a
  let neg        a   =      minus_big_int    a
  let add        a b =      add_big_int      a b
  let add_1      a b =      add_int_big_int  b a
  let sub        a b =      sub_big_int      a b
  let sub_1      a b =      sub_big_int      a (big_int_of_int b)
  let mul        a b =      mult_big_int     a b
  let mul_1      a b =      mult_int_big_int b a
  let sqr        a   =      square_big_int   a
  let abs_in   r a   = r := abs_big_int      a
  let neg_in   r a   = r := minus_big_int    a
  let add_in   r a b = r := add_big_int      a b
  let add_1_in r a b = r := add_int_big_int  b a
  let sub_in   r a b = r := sub_big_int      a b
  let sub_1_in r a b = r := sub_big_int      a (big_int_of_int b)
  let mul_in   r a b = r := mult_big_int     a b
  let mul_1_in r a b = r := mult_int_big_int b a
  let sqr_in   r a   = r := square_big_int   a

  (* division euclidienne *)
  let quomod     a b = if sign_big_int(b) > 0
                       then quomod_big_int a b
                       else let a1 = minus_big_int a
                            and b1 = minus_big_int b in
                            let q,r = quomod_big_int a1 b1
                            in (q,minus_big_int r)

  let quo        a b = let q,_ = quomod a b in q
  let modulo     a b = let _,r = quomod a b in r

  let gquomod m  a b = match m with

    | Floor        -> quomod a b

    | Ceil         -> let q,r = quomod a b in if sign_big_int(r) = 0
                      then q,r
                      else (succ_big_int q),(sub_big_int r b)

    | Nearest_up   -> let a1 = add_big_int a (add_big_int a b)
                      and b1 = add_big_int b b in
                      let q,r = quomod a1 b1 in
                      (q, div_big_int (sub_big_int r b) two)

    | Nearest_down -> let a1 = add_big_int a (add_big_int a b)
                      and b1 = add_big_int b b in
                      let q,r = quomod a1 b1 in if sign_big_int(r) = 0
                      then (pred_big_int q, div_big_int (minus_big_int b) two)
                      else (q, div_big_int (sub_big_int r b) two)

  let gquo       m     a b = let q,_ = gquomod m a b in q
  let gmod       m     a b = let _,r = gquomod m a b in r
  let quomod_in    r s a b = let x,y = quomod    a b in r := x; s := y
  let quo_in       r   a b = let x,_ = quomod    a b in r := x
  let mod_in         s a b = let _,y = quomod    a b in s := y
  let gquomod_in m r s a b = let x,y = gquomod m a b in r := x; s := y
  let gquo_in    m r   a b = let x,_ = gquomod m a b in r := x
  let gmod_in    m   s a b = let _,y = gquomod m a b in s := y

  let quomod_1         a b = let q,r = quomod    a (big_int_of_int b) in q,     (int_of_big_int r)
  let quo_1            a b = let q,_ = quomod    a (big_int_of_int b) in q
  let mod_1            a b = let _,r = quomod    a (big_int_of_int b) in        (int_of_big_int r)
  let gquomod_1      m a b = let q,r = gquomod m a (big_int_of_int b) in q,     (int_of_big_int r)
  let gquo_1         m a b = let q,_ = gquomod m a (big_int_of_int b) in q
  let gmod_1         m a b = let _,r = gquomod m a (big_int_of_int b) in         int_of_big_int r
  let quomod_1_in    r a b = let x,y = quomod    a (big_int_of_int b) in r := x; int_of_big_int y
  let quo_1_in       r a b = let x,y = quomod    a (big_int_of_int b) in r := x
  let gquomod_1_in m r a b = let x,y = gquomod m a (big_int_of_int b) in r := x; int_of_big_int y
  let gquo_1_in    m r a b = let x,y = gquomod m a (big_int_of_int b) in r := x

  (* reprsentation binaire *)
  let words_in_digit = Nat.length_of_digit/16
  let nat_of b = (Obj.obj(Obj.field (Obj.repr b) 1) : Nat.nat)

  let nth_word x n =
    if n < 0 then 0
    else let n1 = n/words_in_digit
         and n2 = 16*(n mod words_in_digit) in
         if n1 >= num_digits_big_int x then 0
         else begin
           let tmp = Nat.create_nat 2 in
           Nat.blit_nat tmp 1 (nat_of x) n1 1;
           Nat.shift_right_nat tmp 1 1 tmp 0 n2;
           (Nat.nth_digit_nat tmp 1) land 0xffff
         end

  let masque = [| 0x0001; 0x0002; 0x0004; 0x0008;
                  0x0010; 0x0020; 0x0040; 0x0080;
                  0x0100; 0x0200; 0x0400; 0x0800;
                  0x1000; 0x2000; 0x4000; 0x8000 |]
  let nth_bit n k =
    (k >= 0) && ((nth_word n (k/16)) land masque.(k land 15) <> 0)

 let nbits x =
    let a = nat_of x             in
    let l = num_digits_big_int x in
    l * Nat.length_of_digit - (Nat.num_leading_zero_bits_in_digit a (l-1))

  let lowbits x = (Nat.nth_digit_nat (nat_of x) 0) land 0x7fffffff

  let highbits x =
    let a = nat_of x
    and l = num_digits_big_int x in
    let m = Nat.length_of_digit - (Nat.num_leading_zero_bits_in_digit a (l-1)) in
    let tmp = Nat.create_nat 2 in
    if l < 2 then begin
      Nat.blit_nat tmp 1 a (l-1) 1;
      Nat.set_digit_nat tmp 0 0;
    end
    else Nat.blit_nat tmp 0 a (l-2) 2;
    if m < 31 then Nat.shift_left_nat  tmp 0 2 tmp 0 (31-m)
              else Nat.shift_right_nat tmp 1 1 tmp 0 (m-31);
    Nat.nth_digit_nat tmp 1

  (* dcalages *)
  let shl a b =
    if (sign_big_int(a) = 0) or (b <= -nbits(a)) then zero_big_int
    else if b = 0 then a
    else if b > 0 then mult_big_int a (power_int_positive_int 2 b)
    else if sign_big_int(a) > 0 then div_big_int a (power_int_positive_int 2 (-b))
    else minus_big_int(div_big_int (minus_big_int a) (power_int_positive_int 2 (-b)))

  let shr a b =
    if (sign_big_int(a) = 0) or (b >= nbits(a)) then zero_big_int
    else if b = 0 then a
    else if b < 0 then mult_big_int a (power_int_positive_int 2 (-b))
    else if sign_big_int(a) > 0 then div_big_int a (power_int_positive_int 2 b)
    else minus_big_int(div_big_int (minus_big_int a) (power_int_positive_int 2 b))

  let split a b =
    if b < 0 then raise (Error "negative index")
    else if (b = 0) or (sign_big_int(a) = 0) then a,zero_big_int
    else if b >= nbits(a)                    then zero_big_int,a
    else let q,r = quomod_big_int (abs_big_int a) (power_int_positive_int 2 b) in
         if sign_big_int(a) > 0 then q,r else minus_big_int(q),minus_big_int(r)

  let join a b c =
    if c < 0 then raise (Error "negative index") else add a (shl b c)

  let shl_in   r   a b = r := shl a b
  let shr_in   r   a b = r := shr a b
  let split_in r s a b = let x,y = split a b in r := x; s := y
  let join_in  r a b c = r := join a b c

  (* exponentiation *)

  (*let pow        a b =      power_big_int_positive_int a b *)
  (* bogu : 416577169024505710327332685500 ^ 45 -> ...442939392 *)

  let pow a b =
    if b < 0 then raise (Error "negative exponent");
    let rec loop n =
      if n = 0 then unit_big_int
      else let x = square_big_int(loop (n/2)) in
           if n land 1 = 0 then x else mult_big_int a x
    in loop b

  let pow_in   r a b = r := pow a b
  let pow_1      a b =      power_int_positive_int     a b
  let pow_1_in r a b = r := power_int_positive_int     a b

  let gpowmod m a b c =
    if sign_big_int(b) < 0 then raise (Error "negative exponent");
    let r = ref(unit_big_int) in
    for i=nbits(b)-1 downto 0 do
      r := mod_big_int (square_big_int !r) c;
      if nth_bit b i then r := mod_big_int (mult_big_int !r a) c;
    done;
    gmod m !r c

  let powmod         a b c =      gpowmod Floor a b c
  let powmod_in    r a b c = r := gpowmod Floor a b c
  let gpowmod_in m r a b c = r := gpowmod m     a b c
        
  (* racine carre *)
  let sqrt    a = sqrt_big_int a
  let gsqrt m a = match m with

    | Floor -> sqrt_big_int a

    | Ceil  -> let b = sqrt_big_int a in
               let c = sub_big_int a (square_big_int b) in
               if sign_big_int(c) = 0 then b else succ_big_int(b)

    | _     -> let a1 = mult_int_big_int 4 a in
               let b1 = sqrt_big_int a1 in
               let b,r = quomod_big_int b1 two in
               if sign_big_int(r) = 0 then b else succ_big_int b

  let sqrt_in    r a = r := sqrt_big_int a
  let gsqrt_in m r a = r := gsqrt m a

  (* racine p-me *)
  let root a p =

    let p1 = p-1 in

    (* itration de Newton : cas a > 0 *)
    let rec loop_positif a x =
      let y = pow x p1  in
      let z = sub_big_int a (mult_big_int x y) in
      if sign_big_int(z) >= 0 then x
      else loop_positif a (add_big_int x (div_big_int z (mult_int_big_int p y)))

    (* itration de Newton : cas a < 0 *)
    and loop_negatif a x =
      let y = pow x p1  in
      let z = sub_big_int a (mult_big_int x y) in
      match sign_big_int(z) with
        | 0  -> x
        | -1 -> pred_big_int x
        | _  -> let y = mult_int_big_int p y
                in loop_negatif a (add_big_int x (quo (add_big_int z (pred_big_int y)) y))
    in

    (* racine rcursive *)
    let rec root_rec a =

      (* valeur initiale : puissance de 2 si a est petit, *)
      (* sinon racine p-me de la partie haute            *)
      let init =
        let n = nbits(a)  in
        let q = n/(2*p)   in
        if q < 2 then shl (big_int_of_int (sign_big_int a)) ((n+p1)/p)
                 else shl (add_int_big_int (sign_big_int a) (root_rec (shr a (p*q)))) q
      in
      (if sign_big_int(a) >= 0 then loop_positif else loop_negatif) a init
    in

    (* contrle de signe *)
    if p <= 0                                   then raise (Error "negative exponent");
    if ((p mod 2) = 0) && (sign_big_int(a) < 0) then raise (Error "negative base");
    if (sign_big_int(a) = 0) then zero_big_int else root_rec a

  let groot m a p = match m with
    | Floor -> root a p
    | Ceil  -> let r = root a p in
               let b = sub_big_int a (pow r p) in
                 if sign_big_int(b) = 0 then r else succ_big_int r
    | _     -> let r = root (shl a p) p in div_big_int (succ_big_int r) two

  let root_in    r a p = r := root    a p
  let groot_in m r a p = r := groot m a p

  (* factorielle *)
  let fact a = 
    if a < 0 then raise (Error "negative base")
    else begin
      let r = ref(unit_big_int) in
      for i=2 to a do r := mul !r (big_int_of_int i) done;
      !r
    end
  let fact_in r a = r := fact a

  (* pgcd *)
  let gcd a b = gcd_big_int a b

  let rec gcd_ex a b = match sign_big_int(a) with

    | 0 -> if sign_big_int(b) >= 0
              then b,zero_big_int,big_int_of_int(-1)
              else minus_big_int(b),zero_big_int,unit_big_int

    | 1 -> let q,r = quomod b a      in
           let d,u1,v1 = gcd_ex r a  in
           let u = minus_big_int(u1) in d,(sub_big_int (mult_big_int q u) v1),u

    |_  -> let d,u,v = gcd_ex (minus_big_int a) b in d,(minus_big_int u),v

  let cfrac a b =
    if (sign_big_int(a) = 0) & (sign_big_int(b) = 0)
    then zero_big_int,unit_big_int,zero_big_int,unit_big_int,zero_big_int
    else let d,u,v = gcd_ex a b in d,u,v,(div_big_int a d),(div_big_int b d)

  let gcd_in    d         a b = d := gcd_big_int a b
  let gcd_ex_in d u v     a b = let     x,y,z = gcd_ex a b in d := x; u := y; v := z
  let cfrac_in  d u v p q a b = let r,s,x,y,z = cfrac  a b in d := r; u := s; v := x; p := y; q := z

  (* comparaisons *)
  let sgn     a   = sign_big_int a
  let cmp     a b = compare_big_int a b
  let eq      a b = eq_big_int      a b
  let neq     a b = not(eq_big_int  a b)
  let inf     a b = lt_big_int      a b
  let infeq   a b = le_big_int      a b
  let sup     a b = gt_big_int      a b
  let supeq   a b = ge_big_int      a b
  let cmp_1   a b = compare_big_int a (big_int_of_int b)
  let eq_1    a b = eq_big_int      a (big_int_of_int b)
  let neq_1   a b = not(eq_big_int  a (big_int_of_int b))
  let inf_1   a b = lt_big_int      a (big_int_of_int b)
  let infeq_1 a b = le_big_int      a (big_int_of_int b)
  let sup_1   a b = gt_big_int      a (big_int_of_int b)
  let supeq_1 a b = ge_big_int      a (big_int_of_int b)

  (* conversions *)
  let of_int      a =      big_int_of_int a
  let of_int_in r a = r := big_int_of_int a

  let of_string ch =

    let l = String.length(ch)
    and i = ref(0) in

    (* dcode le signe *)
    let s =      if (l > 0) & ch.[0] = '-' then (incr i; -1)
            else if (l > 0) & ch.[0] = '+' then (incr i;  1) else 1 in

    (* dcode la base *)
    let b = if (l > !i+1) & (ch.[!i] = '0') then match ch.[!i+1] with
            | 'x' | 'X' -> i := !i + 2; 16
            | 'o' | 'O' -> i := !i + 2;  8
            | 'b' | 'B' -> i := !i + 2;  2
            | _         -> incr i;      10
            else 10 in

    (* s'il n'y a plus de caractres disponibles, chane invalide *)
    if !i >= l then raise (Error "invalid string");

    (* dcode le reste de la chane (algorithme en n^2) *)
    let r = ref(zero_big_int) in
    while !i < l do
      let c = match ch.[!i] with
      | '0'..'9' as x -> Char.code(x)-48
      | 'a'..'f' as x -> Char.code(x)-87
      | 'A'..'F' as x -> Char.code(x)-55
      | _             -> raise (Error "invalid string")
      in
      if c > b then raise (Error "invalid string");
      r := add_int_big_int c (mult_int_big_int b !r);
      incr i
    done;

    (* rsultat *)
    if s = 1 then !r else minus_big_int !r

  let of_string_in r ch = r := of_string ch

  let int_of a    = int_of_big_int a
  let string_of a = string_of_big_int a

  let bstring_of a =
    if sign_big_int(a) = 0 then "0"
    else begin
      let n = nbits(a)
      and s = if sign_big_int(a) < 0 then 1 else 0 in
      let r = String.create (n+2+s) in
        if (s=1) then r.[0] <- '-';
        r.[s]   <- '0';
        r.[s+1] <- 'b';
        for i=0 to n-1 do r.[n+1+s-i] <- if nth_bit a i then '1' else '0' done;
        r
    end

  let hstring_of a =
    if sign_big_int(a) = 0 then "0"
    else begin
      let n = (nbits(a)+3)/4
      and s = if sign_big_int(a) < 0 then 1 else 0 in
      let r = String.create (n+2+s) in
        if (s=1) then r.[0] <- '-';
        r.[s]   <- '0';
        r.[s+1] <- 'x';
        for i=0 to n-1 do
          let c = ((nth_word a (i/4)) lsr (4*(i land 3))) land 15 in
          r.[n+1+s-i] <- if c < 10 then Char.chr(48+c) else Char.chr(55+c)
        done;
        r
    end

  let ostring_of a =
    if sign_big_int(a) = 0 then "0"
    else begin
      let n = (nbits(a)+2)/3
      and s = if sign_big_int(a) < 0 then 1 else 0 in
      let r = String.create (n+2+s) in
        if (s=1) then r.[0] <- '-';
        r.[s]   <- '0';
        r.[s+1] <- 'o';
        for i=0 to n-1 do
          let c = (if nth_bit a (3*i)   then 1 else 0)
                + (if nth_bit a (3*i+1) then 1 else 0)*2
                + (if nth_bit a (3*i+2) then 1 else 0)*4 in
          r.[n+1+s-i] <- Char.chr(48+c)
        done;
        r
    end

  (* nombres alatoires *)
  let nrandom n =
    if n < 0 then raise (Error "negative size")
    else if n = 0 then zero_big_int
    else begin
      let p = if n land 15 = 0 then 16 else n land 15 in
      let mask = (1 lsl p) - 1 in
      let r = ref(big_int_of_int(Random.bits() land mask)) in
      for i=1 to (n-p)/16 do
        r := add_int_big_int (Random.bits() land 0xffff) (mult_int_big_int 0x10000 !r)
      done;
      !r
    end

  let nrandom1 n =
    if n < 0 then raise (Error "negative size")
    else if n = 0 then zero_big_int
    else begin
      let p = if n land 15 = 0 then 16 else n land 15 in
      let mask = 1 lsl (p-1) in
      let r = ref(big_int_of_int((Random.bits() land (mask-1)) lor mask)) in
      for i=1 to (n-p)/16 do
        r := add_int_big_int (Random.bits() land 0xffff) (mult_int_big_int 0x10000 !r)
      done;
      !r
    end

  let zrandom n = match Random.bits() land 1 with
    | 0 -> nrandom n
    | _ -> minus_big_int(nrandom n)

  let zrandom1 n = match Random.bits() land 1 with
    | 0 -> nrandom1 n
    | _ -> minus_big_int(nrandom1 n)

  let nrandom_in  r n = r := nrandom  n
  let zrandom_in  r n = r := zrandom  n
  let nrandom1_in r n = r := nrandom1 n
  let zrandom1_in r n = r := zrandom1 n

  let random_init x = if x = 0 then Random.self_init() else Random.init(x)

  (* affichage tronqu  1000 caractres *)
  let toplevel_print(a) =
    let s = string_of a      in
    let l = String.length(s) in
    if l < 1000 then Format.print_string s
    else begin
      Format.print_string (String.sub s 0 100);
      Format.print_string " ... (";
      Format.print_int    (l-200);
      Format.print_string " digits) ... ";
      Format.print_string (String.sub s (l-100) 100)
    end

  let toplevel_print_tref(a) =
    Format.print_string "tref(";
    toplevel_print(look a);
    Format.print_char ')'

end
