// file kernel/n/x86/add.S: addition/subtraction of natural integers
/*-----------------------------------------------------------------------+
 |  Copyright 2005-2006, Michel Quercia (michel.quercia@prepas.org)      |
 |                                                                       |
 |  This file is part of Numerix. Numerix is free software; you can      |
 |  redistribute it and/or modify it under the terms of the GNU Lesser   |
 |  General Public License as published by the Free Software Foundation; |
 |  either version 2.1 of the License, or (at your option) any later     |
 |  version.                                                             |
 |                                                                       |
 |  The Numerix Library is distributed in the hope that it will be       |
 |  useful, but WITHOUT ANY WARRANTY; without even the implied warranty  |
 |  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU  |
 |  Lesser General Public License for more details.                      |
 |                                                                       |
 |  You should have received a copy of the GNU Lesser General Public     |
 |  License along with the GNU MP Library; see the file COPYING. If not, |
 |  write to the Free Software Foundation, Inc., 59 Temple Place -       |
 |  Suite 330, Boston, MA 02111-1307, USA.                               |
 +-----------------------------------------------------------------------+
 |                                                                       |
 |                         Addition/soustraction                         |
 |                                                                       |
 +-----------------------------------------------------------------------*/


#ifdef use_sse2
#------------------------------ Code SSE2 ------------------------------

# Remarque :
# Les fonctions sn_fadd,sn_fadd_1,sn_fsub,sn_fsub_1,sn_finc,sn_finc_1
# sn_fdec et sn_fdec_1 retournent la retenue  la fois dans CF et dans eax.
# Le fait que la retenue soit dans eax ne figure pas dans les spcifications
# par compatibilit avec le code non-sse2.
               
                               # +------------+
                               # |  Addition  |
                               # +------------+

# sn_fadd
# -------
# entre :
#   a = naturel de longueur la     esi = &a, edx = la
#   b = naturel de longueur lb     ebx = &b, ecx = lb
#   c = naturel de longueur la     edi = &c, peut tre confondu avec a ou b
# contraintes : la >= lb > 0
#
# sortie :
#   c <- a+b
#
# registres modifis :
#   esi = &a[la],  ebx = &b[lb],   edi = &c[la]
#   ecx = 0
#   eax,edx,ebp = ind.
#   CF  = retenue sortante
#
# sn_fadd_1
# ---------
# entre :
#   a = naturel de longueur l      esi = &a, ecx = l
#   b = naturel de longueur l      ebx = &b
#   c = naturel de longueur l      edi = &c, peut tre confondu avec a ou b
# contraintes : l > 0
#
# sortie :
#   c <- a+b
#
# registres modifis :
#   esi = &a[l],  ebx = &b[l],   edi = &c[l]
#   ecx = 0
#   eax,edx,ebp = ind.
#   CF  = retenue sortante
        
#undef L                
#define L(x) .Lsn_fadd_##x

        ALIGN_32
.Lsn_fadd:

        subl   %ecx,    %edx
        leal   (%esi,%ecx,4), %esi      # esi <- &a[lb]
        leal   (%ebx,%ecx,4), %ebx      # ebx <- &b[lb]
        leal   (%edi,%ecx,4), %edi      # edi <- &c[lb]
        negl   %ecx

        # boucle d addition
        movd   (%esi,%ecx,4), %mm2
        movd   (%ebx,%ecx,4), %mm1
        incl   %ecx
        paddq  %mm1,    %mm2
        movd   %mm2, -4(%edi,%ecx,4)
        pshufw $0xfe,   %mm2, %mm2
        jz     2f
        ALIGN_4
1:
        movd   (%esi,%ecx,4), %mm0
        movd   (%ebx,%ecx,4), %mm1
        incl   %ecx
        paddq  %mm0,    %mm2
        paddq  %mm1,    %mm2
        movd   %mm2, -4(%edi,%ecx,4)
        pshufw $0xfe,   %mm2, %mm2
        jne    1b
2:

        # propage la retenue
        leal   (%esi,%edx,4), %esi      # esi <- &a[la]
        leal   (%edi,%edx,4), %edi      # edi <- &c[la]
        negl   %edx
        jz  2f
        ALIGN_4
1:
        movd   (%esi,%edx,4), %mm0
        paddq  %mm0,    %mm2
        movd   %mm2,    (%edi,%edx,4)
        incl   %edx
        pshufw $0xfe,   %mm2, %mm2
        jne    1b
        
        # rcupre la dernire retenue
2:
        movd   %mm2,    %eax
        bt     $0,      %eax
        emms
        ret
        
        ALIGN_32
.Lsn_fadd_1:

        leal   (%esi,%ecx,4), %esi      # esi <- &a[l]
        leal   (%ebx,%ecx,4), %ebx      # ebx <- &b[l]
        leal   (%edi,%ecx,4), %edi      # edi <- &c[l]
        negl   %ecx

        # boucle d addition
        movd   (%esi,%ecx,4), %mm2
        movd   (%ebx,%ecx,4), %mm1
        incl   %ecx
        paddq  %mm1,    %mm2
        movd   %mm2, -4(%edi,%ecx,4)
        pshufw $0xfe,   %mm2, %mm2
        jz     2f
        ALIGN_4
1:
        movd   (%esi,%ecx,4), %mm0
        movd   (%ebx,%ecx,4), %mm1
        incl   %ecx
        paddq  %mm0,    %mm2
        paddq  %mm1,    %mm2
        movd   %mm2, -4(%edi,%ecx,4)
        pshufw $0xfe,   %mm2, %mm2
        jne    1b
2:

        # rcupre la dernire retenue
        movd   %mm2,    %eax
        bt     $0,      %eax
        emms
        ret

                
#  chiffre xn(add)(chiffre *a, long la, chiffre *b, long lb, chiffre *c)
# 
#  entre :
#  a = naturel de longueur la
#  b = naturel de longueur lb <= la
#  c = naturel de longueur la
# 
#  sortie :
#  c <- a + b
#  retourne la retenue

#ifdef assembly_sn_add
#undef L
#define L(x) .Lsn_add_##x

ENTER(sn_add)

        movl   arg1,    %esi            # esi <- &a  
        movl   arg2,    %edx            # edx <- la 
        movl   arg3,    %ebx            # ebx <- &b  
        movl   arg4,    %ecx            # ecx <- lb 
        movl   arg5,    %edi            # edi <- &c
        jecxz  L(b_nul)                 # si b = 0, recopie a
        call   .Lsn_fadd                # sinon, effectue l addition
        RETURN_WITH_SP

L(b_nul):
        movl   %edx,    %ecx            # c <- a
        jecxz  L(a_nul)
        cld;   REP(movsl)
L(a_nul):
        xorl   %eax,    %eax            # eax <- 0 (retenue)
        RETURN_WITH_SP

#endif /* assembly_sn_add */

                             # +----------------+
                             # |  Soustraction  |
                             # +----------------+

        
# sn_fasub
# -------
# entre :
#   a = naturel de longueur la     esi = &a, edx = la
#   b = naturel de longueur lb     ebx = &b, ecx = lb
#   c = naturel de longueur la     edi = &c, peut tre confondu avec a ou b
# contraintes : la >= lb > 0
#
# sortie :
#   c <- |a-b|
#
# registres modifis :
#   ecx = 0
#   eax,edx,esi,ebx,edi,ebp = ind.
#   CF  = 0 si a >= b, 1 si a < b

#undef L                
#define L(x) .Lsn_fsub_##x

        ALIGN_32
.Lsn_fasub:
        
        leal   (%esi,%ecx,4), %esi      # esi <- &a[lb]
        leal   (%ebx,%ecx,4), %ebx      # ebx <- &b[lb]
        leal   (%edi,%ecx,4), %edi      # edi <- &c[lb]
        movl   %edx,    %eax            # eax <- la

        # on parie que a >= b si la > lb ou si a[la-1] >= b[la-1]
        subl   %ecx,    %edx
        jnz    1f
        movl -4(%esi),  %ebp
        cmpl -4(%ebx),  %ebp
        jae    1f
        xchgl  %esi,    %ebx            # sinon, change a et b
1:
        rcll   $1,      %ebp            # ebp[0] <- signe prvu pour a-b

        # soustrait les chiffres communs        
        negl   %ecx
        movd   (%esi,%ecx,4), %mm2
        movd   (%ebx,%ecx,4), %mm1
        incl   %ecx
        psubq  %mm1,    %mm2
        movd   %mm2, -4(%edi,%ecx,4)
        pshufw $0xfe,   %mm2, %mm2
        jz     2f
        ALIGN_4
1:
        movd   (%esi,%ecx,4), %mm0
        movd   (%ebx,%ecx,4), %mm1
        incl   %ecx
        paddq  %mm0,    %mm2
        psubq  %mm1,    %mm2
        movd   %mm2, -4(%edi,%ecx,4)
        pshufw $0xfe,   %mm2, %mm2
        jne    1b
2:
        
        # propage la retenue
        testl  %edx,    %edx
        jz     2f
        leal   (%esi,%edx,4), %esi      # esi <- &a[la]
        leal   (%edi,%edx,4), %edi      # edi <- &c[la]
        negl   %edx
        ALIGN_4
1:
        movd   (%esi,%edx,4), %mm0
        paddq  %mm0,    %mm2
        movd   %mm2,    (%edi,%edx,4)
        incl   %edx
        pshufw $0xfe,   %mm2, %mm2
        jne    1b
        
        # si < 0, change de signe
2:
        movd   %mm2,    %edx
        testl  %edx,    %edx
        jz     2f
        incl   %ebp
        negl   %eax                     # eax <- -la
        pxor   %mm2,    %mm2            # mm2 <- 0 (retenue)
        ALIGN_4
1:
        movd  (%edi,%eax,4), %mm0
        psubq  %mm0,    %mm2
        movd   %mm2,   (%edi,%eax,4)
        incl   %eax
        pshufw $0xfe,   %mm2, %mm2
        jne    1b
        
2:
        bt     $0,      %ebp            # CF <- signe(a-b)
        emms
        ret
        
        
# sn_fsub
# -------
# entre :
#   a = naturel de longueur la     esi = &a, edx = la
#   b = naturel de longueur lb     ebx = &b, ecx = lb
#   c = naturel de longueur la     edi = &c, peut tre confondu avec a ou b
# contraintes : la >= lb > 0
#
# sortie :
#   c <- a-b
#
# registres modifis :
#   esi = &a[la],  ebx = &b[lb],   edi = &c[la]
#   ecx = 0
#   eax,edx,ebp = ind.
#   CF  = retenue sortante
#
# sn_fsub_1
# ---------
# entre :
#   a = naturel de longueur l      esi = &a, ecx = l
#   b = naturel de longueur l      ebx = &b
#   c = naturel de longueur l      edi = &c, peut tre confondu avec a ou b
# contraintes : l > 0
#
# sortie :
#   c <- a-b
#
# registres modifis :
#   esi = &a[l],  ebx = &b[l],   edi = &c[l]
#   ecx = 0
#   eax,edx,ebp = ind.
#   CF  = retenue sortante

        ALIGN_32
.Lsn_fsub:

        subl   %ecx,    %edx
        leal   (%esi,%ecx,4), %esi      # esi <- &a[lb]
        leal   (%ebx,%ecx,4), %ebx      # ebx <- &b[lb]
        leal   (%edi,%ecx,4), %edi      # edi <- &c[lb]
        negl   %ecx
        pxor   %mm2,    %mm2            # mm2 <- 0 (retenue)

        # boucle de soustraction
        movd   (%esi,%ecx,4), %mm2
        movd   (%ebx,%ecx,4), %mm1
        incl   %ecx
        psubq  %mm1,    %mm2
        movd   %mm2, -4(%edi,%ecx,4)
        pshufw $0xfe,   %mm2, %mm2
        jz     2f
        ALIGN_4
1:
        movd   (%esi,%ecx,4), %mm0
        movd   (%ebx,%ecx,4), %mm1
        incl   %ecx
        paddq  %mm0,    %mm2
        psubq  %mm1,    %mm2
        movd   %mm2, -4(%edi,%ecx,4)
        pshufw $0xfe,   %mm2, %mm2
        jne    1b
2:
        
        # propage la retenue
        leal   (%esi,%edx,4), %esi      # esi <- &a[la]
        leal   (%edi,%edx,4), %edi      # edi <- &c[la]
        negl   %edx
        jz     2f
        ALIGN_4
1:
        movd   (%esi,%edx,4), %mm0
        paddq  %mm0,    %mm2
        movd   %mm2,    (%edi,%edx,4)
        incl   %edx
        pshufw $0xfe,   %mm2, %mm2
        jne    1b
        
        # rcupre la dernire retenue
2:
        movd   %mm2,    %eax
        negl   %eax
        emms
        ret
        
        ALIGN_32
.Lsn_fsub_1:

        leal   (%esi,%ecx,4), %esi      # esi <- &a[l]
        leal   (%ebx,%ecx,4), %ebx      # ebx <- &b[l]
        leal   (%edi,%ecx,4), %edi      # edi <- &c[l]
        negl   %ecx

        # boucle de soustraction
        movd   (%esi,%ecx,4), %mm2
        movd   (%ebx,%ecx,4), %mm1
        incl   %ecx
        psubq  %mm1,    %mm2
        movd   %mm2, -4(%edi,%ecx,4)
        pshufw $0xfe,   %mm2, %mm2
        jz     2f
        ALIGN_4
1:
        movd   (%esi,%ecx,4), %mm0
        movd   (%ebx,%ecx,4), %mm1
        incl   %ecx
        paddq  %mm0,    %mm2
        psubq  %mm1,    %mm2
        movd   %mm2, -4(%edi,%ecx,4)
        pshufw $0xfe,   %mm2, %mm2
        jne    1b
2:
        
        # rcupre la dernire retenue
        movd   %mm2,    %eax
        negl   %eax
        emms
        ret
        

#  chiffre xn(sub)(chiffre *a, long la, chiffre *b, long lb, chiffre *c)
# 
#  entre :
#  a = naturel de longueur la
#  b = naturel de longueur lb <= la
#  c = naturel de longueur la
# 
#  sortie :
#  c <- a - b
#  retourne la retenue

#ifdef assembly_sn_sub
#undef L
#define L(x) .Lsn_sub_##x

ENTER(sn_sub)

        movl   arg1,    %esi            # esi <- &a  
        movl   arg2,    %edx            # edx <- la 
        movl   arg3,    %ebx            # ebx <- &b  
        movl   arg4,    %ecx            # ecx <- lb 
        movl   arg5,    %edi            # edi <- &c
        jecxz  L(b_nul)                 # si b = 0, recopie a
        call   .Lsn_fsub                # sinon, effectue la soustraction
        RETURN_WITH_SP

L(b_nul):
        movl   %edx,    %ecx            # c <- a
        jecxz  L(a_nul)
        cld;   REP(movsl)
L(a_nul):
        xorl   %eax,    %eax            # eax <- 0 (retenue)
        RETURN_WITH_SP

#endif /* assembly_sn_sub */

                            # +------------------+
                            # |  Incrmentation  |
                            # +------------------+
# sn_finc
# -------
# entre :
#   a = naturel de longueur la     esi = &a, edx = la
#   b = naturel de longueur lb     ebx = &b, ecx = lb
# contraintes : la >= lb > 0
#
# sortie :
#   a <- a+b
#
# registres modifis :
#   esi = &a[la],  ebx = &b[lb]
#   ecx = 0
#   eax,edx,ebp = ind.
#   CF  = retenue sortante
#
# sn_finc_1
# ---------
# entre :
#   a = naturel de longueur l      esi = &a, ecx = l
#   b = naturel de longueur l      ebx = &b
# contraintes : l > 0
#
# sortie :
#   a <- a+b
#
# registres modifis :
#   esi = &a[l],  ebx = &b[l]
#   ecx = 0
#   eax,edx,ebp = ind.
#   CF = retenue sortante
#
# remarque :
#  sn_finc_1 peut tre utilis pour diviser par 1 - BASE^p si p >= 2

#undef L                
#define L(x) .Lsn_finc_##x

        ALIGN_32
.Lsn_finc:

        subl   %ecx,    %edx
        leal   (%esi,%ecx,4), %esi      # esi <- &a[lb]
        leal   (%ebx,%ecx,4), %ebx      # ebx <- &b[lb]
        negl   %ecx

        # boucle d addition
        movd   (%esi,%ecx,4), %mm2
        movd   (%ebx,%ecx,4), %mm1
        incl   %ecx
        paddq  %mm1,    %mm2
        movd   %mm2, -4(%esi,%ecx,4)
        pshufw $0xfe,   %mm2, %mm2
        jz     2f
        ALIGN_4
1:
        movd   (%esi,%ecx,4), %mm0
        movd   (%ebx,%ecx,4), %mm1
        incl   %ecx
        paddq  %mm0,    %mm2
        paddq  %mm1,    %mm2
        movd   %mm2, -4(%esi,%ecx,4)
        pshufw $0xfe,   %mm2, %mm2
        jne    1b
2:
        
        # propage la retenue
        leal   (%esi,%edx,4), %esi      # esi <- &a[la]
        leal   (%edi,%edx,4), %edi      # edi <- &c[la]
        movd   %mm2,    %eax            # eax <- retenue finale
        bt     $0,      %eax            # CF <- retenue finale
        jnc    3f
        negl   %edx
        jz     2f
        xorl   %eax,    %eax            # prdit que la retenue sera absorbe
1:
        incl  (%esi,%edx,4)
        jnz    3f
        incl   %edx
        jnz    1b
        incl   %eax                     # loup ... eax <- 1
2:
        stc                             # et CF <- 1
3:
        emms
        ret
        
        ALIGN_32
.Lsn_finc_1:

        leal   (%esi,%ecx,4), %esi      # esi <- &a[l]
        leal   (%ebx,%ecx,4), %ebx      # ebx <- &b[l]
        negl   %ecx

        # boucle d addition
        movd   (%esi,%ecx,4), %mm2
        movd   (%ebx,%ecx,4), %mm1
        incl   %ecx
        paddq  %mm1,    %mm2
        movd   %mm2, -4(%esi,%ecx,4)
        pshufw $0xfe,   %mm2, %mm2
        jz     2f
        ALIGN_4
1:
        movd   (%esi,%ecx,4), %mm0
        movd   (%ebx,%ecx,4), %mm1
        incl   %ecx
        paddq  %mm0,    %mm2
        paddq  %mm1,    %mm2
        movd   %mm2, -4(%esi,%ecx,4)
        pshufw $0xfe,   %mm2, %mm2
        jne    1b
2:

        # rcupre la dernire retenue
        movd   %mm2,    %eax
        bt     $0,      %eax
        emms
        ret
                
#  chiffre xn(inc)(chiffre *a, long la, chiffre *b, long lb)
#  entre :
#  a = naturel de longueur la
#  b = naturel de longueur lb avec lb <= la
#
#  sortie :
#  a <- a + b
#  retourne la retenue

#ifdef assembly_sn_inc
#undef L
#define L(x) .Lsn_inc_##x

ENTER(sn_inc)

        movl   arg1,    %esi            # esi <- &a  
        movl   arg2,    %edx            # edx <- la  
        movl   arg3,    %ebx            # ebx <- &b 
        movl   arg4,    %ecx            # ecx <- lb
        jecxz  L(b_nul)                 # si b = 0, c est fini
        call   .Lsn_finc                # sinon, effectue l incrmentation
        RETURN_WITH_SP
        
L(b_nul):
        xorl   %eax,    %eax            # eax <- 0 (retenue)
        RETURN_WITH_SP

#endif /* assembly_sn_inc */
        

                            # +------------------+
                            # |  Dcrmentation  |
                            # +------------------+

# sn_fdec
# -------
# entre :
#   a = naturel de longueur la     esi = &a, edx = la
#   b = naturel de longueur lb     ebx = &b, ecx = lb
# contraintes : la >= lb > 0
#
# sortie :
#   a <- a-b
#
# registres modifis :
#   esi = &a[la],  ebx = &b[lb]
#   ecx = 0
#   eax,edx,ebp = ind.
#   CF  = retenue sortante
#
# sn_fdec_1
# ---------
# entre :
#   a = naturel de longueur l      esi = &a, ecx = l
#   b = naturel de longueur l      ebx = &b
# contraintes : l > 0
#
# sortie :
#   a <- a-b
#
# registres modifis :
#   esi = &a[l],  ebx = &b[l]
#   ecx = 0
#   eax,edx,ebp = ind.
#   CF  = retenue sortante

#undef L                
#define L(x) .Lsn_fdec_##x

        ALIGN_32
.Lsn_fdec:      

        subl   %ecx,    %edx
        leal   (%esi,%ecx,4), %esi      # esi <- &a[lb]
        leal   (%ebx,%ecx,4), %ebx      # ebx <- &b[lb]
        negl   %ecx

        # boucle de soustraction
        movd   (%esi,%ecx,4), %mm2
        movd   (%ebx,%ecx,4), %mm1
        incl   %ecx
        psubq  %mm1,    %mm2
        movd   %mm2, -4(%esi,%ecx,4)
        pshufw $0xfe,   %mm2, %mm2
        jz     2f
        ALIGN_4
1:
        movd   (%esi,%ecx,4), %mm0
        movd   (%ebx,%ecx,4), %mm1
        incl   %ecx
        paddq  %mm0,    %mm2
        psubq  %mm1,    %mm2
        movd   %mm2, -4(%esi,%ecx,4)
        pshufw $0xfe,   %mm2, %mm2
        jne    1b
2:

        # propage la retenue
        leal   (%esi,%edx,4), %esi      # esi <- &a[la]
        leal   (%edi,%edx,4), %edi      # edi <- &c[la]
        movd   %mm2,    %eax            # eax <- retenue finale
        negl   %eax                     # CF <- retenue finale
        jnc    3f
        negl   %edx
        jz     2f
        xorl   %eax,    %eax            # prdit que la retenue sera absorbe
1:
        subl   $1,     (%esi,%edx,4)
        jnb    3f
        incl   %edx
        jnz    1b
        incl   %eax                     # loup ... eax <- 1
2:
        stc                             # et CF <- 1
3:
        emms
        ret
        
        ALIGN_32
.Lsn_fdec_1:

        leal   (%esi,%ecx,4), %esi      # esi <- &a[l]
        leal   (%ebx,%ecx,4), %ebx      # ebx <- &b[l]
        negl   %ecx

        # boucle de soustraction
        movd  (%esi,%ecx,4), %mm2
        movd  (%ebx,%ecx,4), %mm1
        incl   %ecx
        psubq  %mm1,    %mm2
        movd   %mm2, -4(%esi,%ecx,4)
        pshufw $0xfe,   %mm2, %mm2
        jz     2f
        ALIGN_4
1:
        movd  (%esi,%ecx,4), %mm0
        movd  (%ebx,%ecx,4), %mm1
        incl   %ecx
        paddq  %mm0,    %mm2
        psubq  %mm1,    %mm2
        movd   %mm2, -4(%esi,%ecx,4)
        pshufw $0xfe,   %mm2, %mm2
        jne    1b
2:

        # rcupre la dernire retenue
        movd   %mm2,    %eax
        bt     $0,      %eax
        emms
        ret
        

#  chiffre xn(dec)(chiffre *a, long la, chiffre *b, long lb)
#  entre :
#  a = naturel de longueur la
#  b = naturel de longueur lb avec lb <= la
#
#  sortie :
#  a <- a - b
#  retourne la retenue

#ifdef assembly_sn_dec
#undef L
#define L(x) .Lsn_dec_##x

ENTER(sn_dec)
        
        movl   arg1,    %esi            # esi <- a  
        movl   arg2,    %edx            # edx <- la  
        movl   arg3,    %ebx            # ebx <- b 
        movl   arg4,    %ecx            # ecx <- lb
        jecxz  L(b_nul)                 # si b = 0, c est fini
        call   .Lsn_fdec                # sinon, effectue la dcrmentation
        RETURN_WITH_SP

L(b_nul):
        xorl   %eax,    %eax            # eax <- 0 (retenue)
        RETURN_WITH_SP
        
#endif /* assembly_sn_dec */
        


                         # +--------------------+
                         # |  somme/diffrence  |
                         # +--------------------+

# entre :
#   a = naturel de longueur l       esi = &a,   ecx = l
#   b = naturel de longueur l       ebx = &b
#   c = nautrel de longueur l       edi = &c
#   d = naturel de longueur l       edx = &d
# contraintes : l > 0, c,d non confondus
#
# sortie :
#   c <- a+b
#   d <- a-b
#
# registres modifis :
#   ecx <- 0
#   esi <- &a[l], ebx <- &b[l], edi <- &c[l], edx <- &d[l]
#   CF  <- retenue(a+b)
#   eax <- retenue(a-b)
#
# remarque :
#   cette fonction n est pas disponible en code x86 gnrique

#undef L
#define L(x) .Lsn_fadd_sub_##x

        ALIGN_32
.Lsn_fadd_sub:
        
        leal  (%esi,%ecx,4), %esi       # esi <- &a[l]
        leal  (%ebx,%ecx,4), %ebx       # ebx <- &b[l]
        leal  (%edi,%ecx,4), %edi       # edi <- &c[l]
        leal  (%edx,%ecx,4), %edx       # edx <- &d[l]
        negl   %ecx
        pxor   %mm0,    %mm0            # init retenues
        pxor   %mm1,    %mm1

        ALIGN_4
1:
        movd  (%esi,%ecx,4), %mm2
        movd  (%ebx,%ecx,4), %mm3
        paddq  %mm2,    %mm0
        paddq  %mm2,    %mm1
        paddq  %mm3,    %mm0
        psubq  %mm3,    %mm1
        movd   %mm0,   (%edi,%ecx,4)
        movd   %mm1,   (%edx,%ecx,4)
        incl   %ecx
        pshufw $0xfe, %mm0, %mm0
        pshufw $0xfe, %mm1, %mm1
        jne    1b

        movd   %mm0,    %eax
        shrl   $1,      %eax            # CF  <- retenue(a+b)
        movd   %mm1,    %eax            # eax <- retenue(a-b)

        emms
        ret


                         # +-------------------------+
                         # |  Demi-somme/diffrence  |
                         # +-------------------------+

# entre :
#   a = naturel de longueur l       esi = &a,   ecx = l
#   b = naturel de longueur l       ebx = &b
#   c = nautrel de longueur l       edi = &c
#   d = naturel de longueur l       edx = &d
# contraintes : l > 0, c,d non confondus, a+b pair
#
# sortie :
#   c <- (a+b)/2
#   d <- (a-b)/2
#
# registres modifis :
#   ecx <- 0
#   esi <- &a[l], ebx <- &b[l], edi <- &c[l], edx <- &d[l]
#   eax <- retenue(a-b)
#
# remarque :
#   cette fonction n est pas disponible en code x86 gnrique

#undef L
#define L(x) .Lsn_fhalf_add_sub_##x

        ALIGN_32
.Lsn_fhalf_add_sub:

        leal  (%esi,%ecx,4), %esi       # esi <- &a[l]
        leal  (%ebx,%ecx,4), %ebx       # ebx <- &b[l]
        leal  (%edi,%ecx,4), %edi       # edi <- &c[l]
        leal  (%edx,%ecx,4), %edx       # edx <- &d[l]
        negl   %ecx

        movd  (%esi,%ecx,4), %mm4       # mm4 <- a[0]
        movd  (%ebx,%ecx,4), %mm2       # mm2 <- b[0]
        incl   %ecx
        paddq  %mm4,    %mm2            # mm2 <- a[0] + b[0] = r:x[0]
        movl   $-1,     %eax
        movd   %eax,    %mm7            # mm7 <- 00000000:ffffffff (masque)
        jz     2f
        
        ALIGN_4
1:
        movq   %mm2,    %mm3            # mm3 <- r:x[i-1]
        pshufw $0xfe,   %mm2, %mm2      # retenue pour a+b
        movd  (%esi,%ecx,4), %mm0       # mm0 <- a[i]
        movd  (%ebx,%ecx,4), %mm1       # mm1 <- b[i]
        incl   %ecx
        paddq  %mm0,    %mm2
        paddq  %mm1,    %mm2            # mm2 <- a[i] + b[i] + retenue = r:x[i]
        punpckldq %mm2, %mm3
        psrlq  $1,      %mm3            # mm3 <- (x[i]:x[i-1])/2
        movd   %mm3, -8(%edi,%ecx,4)    # sauve c[i-1]
        pand   %mm7,    %mm3            # mm3 <- c[i-1]
        psubq  %mm3,    %mm4            # mm4 <- a[i-1] - ret - c[i-1]
        movd   %mm4, -8(%edx,%ecx,4)    # sauve d[i-1]
        pshufw $0xfe,   %mm4, %mm4      # nouvelle retenue pour a-b
        paddq  %mm0,    %mm4            # mm4 <- a[i] - ret
        jne    1b
2:
        psrlq  $1,      %mm2            # mm2 <- (r:x[l-1])/2
        movd   %mm2, -4(%edi)           # sauve c[l-1]
        psubq  %mm2,    %mm4            # mm4 <- a[l-1] - ret - c[l-1]
        movd   %mm4, -4(%edx)           # sauve d[l-1]
        pshufw $0xfe,   %mm4, %mm4
        movd   %mm4,    %eax            # eax <- retenue(a-b)

        emms
        ret


#else
#------------------------------ Code x86 ------------------------------
               
                               # +------------+
                               # |  Addition  |
                               # +------------+

# sn_fadd
# -------
# entre :
#   a = naturel de longueur la     esi = &a, edx = la
#   b = naturel de longueur lb     ebx = &b, ecx = lb
#   c = naturel de longueur la     edi = &c, peut tre confondu avec a ou b
# contraintes : la >= lb > 0
#
# sortie :
#   c <- a+b
#
# registres modifis :
#   esi = &a[la],  ebx = &b[lb],   edi = &c[la]
#   ecx = 0
#   eax,edx,ebp = ind.
#   CF  = retenue sortante
#
# sn_fadd_1
# ---------
# entre :
#   a = naturel de longueur l      esi = &a, ecx = l
#   b = naturel de longueur l      ebx = &b
#   c = naturel de longueur l      edi = &c, peut tre confondu avec a ou b
# contraintes : l > 0
#
# sortie :
#   c <- a+b
#
# registres modifis :
#   esi = &a[l],  ebx = &b[l],   edi = &c[l]
#   ecx = 0
#   eax,edx,ebp = ind.
#   CF  = retenue sortante
        
#undef L                
#define L(x) .Lsn_fadd_##x

        ALIGN_32
.Lsn_fadd:
        
        subl   %ecx,    %edx            # edx <- la - lb
        jz     .Lsn_fadd_1
        pushl  %edx                     # sauve la - lb
        call   .Lsn_fadd_1              # additionne les parties communes
        popl   %edx
        movl   %edx,    %ecx            # ecx <- la - lb
        not    %edx
        incl   %edx                     # edx <- lb - la
        cld;   REP(movsl)               # copie la fin de a
1:
        adcl   %ecx,    (%edi,%edx,4)   # propage la retenue
        jnc    2f
        incl   %edx
        jne    1b
2:
        ret

        ALIGN_32
.Lsn_fadd_1:
        
        movl   (%esi),  %eax            # eax <- a[0]
        leal   (%esi,%ecx,4), %esi      # esi <- &a[l]
        leal   (%ebx,%ecx,4), %ebx      # ebx <- &b[l]
        leal   (%edi,%ecx,4), %edi      # edi <- &c[l]

        # calcule l adresse de saut dans la boucle
        decl   %ecx
        negl   %ecx
        jz     L(last)
        call   L(here)
L(here):
        popl   %ebp                     # ebp <- L(here)
        movl   %ecx,    %edx
        andl   $15,     %edx            # edx <- (1-l) mod 16
        xorl   %edx,    %ecx            # ecx <- 16*((1-l)/16)
        leal   (%edx,%edx,2), %edx      # ebp <- L(begin) + 12*reste
        leal   L(begin)-L(here)(%ebp,%edx,4), %ebp
        movl   %eax,    %edx
        jmp    *%ebp
        
        # corps de boucle  drouler. taille du code = 24 octets
        # entrer avec eax = edx = 1er chiffre de a, CF = 0
#undef BODY
#define BODY(x,y,z) \
          adcl  x(%ebx,%ecx,4), %eax; \
       /* movl  y(%esi,%ecx,4), %edx */ .byte 0x8B, 0x54, 0x8E, y; \
          movl  %eax, x(%edi,%ecx,4); \
       /* adcl  y(%ebx,%ecx,4), %edx */ .byte 0x13, 0x54, 0x8B, y; \
          movl  z(%esi,%ecx,4), %eax; \
       /* movl  %edx, y(%edi,%ecx,4) */ .byte 0x89, 0x54, 0x8f, y

        # boucle d addition droule pour 16 chiffres
        ALIGN_4
L(begin):
        BODY(-4,0,4);	BODY(4,8,12);	BODY(12,16,20); BODY(20,24,28)
        BODY(28,32,36); BODY(36,40,44); BODY(44,48,52); BODY(52,56,60)

        leal   15(%ecx), %ecx
        incl   %ecx
        jne    L(begin)
L(last):
        adcl   -4(%ebx), %eax           # termine la dernire addition
        movl   %eax,    -4(%edi)
        ret
        
#  chiffre xn(add)(chiffre *a, long la, chiffre *b, long lb, chiffre *c)
# 
#  entre :
#  a = naturel de longueur la
#  b = naturel de longueur lb <= la
#  c = naturel de longueur la
# 
#  sortie :
#  c <- a + b
#  retourne la retenue

#ifdef assembly_sn_add
#undef L
#define L(x) .Lsn_add_##x

ENTER(sn_add)

        movl   arg1,    %esi            # esi <- &a  
        movl   arg2,    %edx            # edx <- la 
        movl   arg3,    %ebx            # ebx <- &b  
        movl   arg4,    %ecx            # ecx <- lb 
        movl   arg5,    %edi            # edi <- &c
        jecxz  L(b_nul)                 # si b = 0, recopie a
        call   .Lsn_fadd                # sinon, effectue l addition
        movl   $0,      %eax            # eax <- retenue
        adcl   $0,      %eax
        RETURN_WITH_SP

L(b_nul):
        movl   %edx,    %ecx            # c <- a
        jecxz  L(a_nul)
        cld;   REP(movsl)
L(a_nul):
        xorl   %eax,    %eax            # eax <- 0 (retenue)
        RETURN_WITH_SP

#endif /* assembly_sn_add */

                             # +----------------+
                             # |  Soustraction  |
                             # +----------------+

        
# sn_fasub
# -------
# entre :
#   a = naturel de longueur la     esi = &a, edx = la
#   b = naturel de longueur lb     ebx = &b, ecx = lb
#   c = naturel de longueur la     edi = &c, peut tre confondu avec a ou b
# contraintes : la >= lb > 0
#
# sortie :
#   c <- |a-b|
#
# registres modifis :
#   ecx = 0
#   eax,edx,esi,ebx,edi,ebp = ind.
#   CF  = 0 si a >= b, 1 si a < b

#undef L                
#define L(x) .Lsn_fsub_##x

        ALIGN_32
.Lsn_fasub:
        
        cmpl  %ecx,     %edx
        je    L(compare)
        
        # si la > lb, recopie la fin de a dans c en testant s il y a
        # un chiffre non nul
        xorl   %ebp,    %ebp
        ALIGN_4
1:
        movl  -4(%esi,%edx,4), %eax
        orl   %eax,     %ebp
        movl  %eax,     -4(%edi,%edx,4)
        decl  %edx
        cmpl  %edx,     %ecx
        jne   1b
        testl %ebp,     %ebp            # pfort(a) = 0 ?
        jz    L(compare)

        # ici la > lb et pfort(a) > 0, donc a > b
        call  .Lsn_fsub_1               # soustrait les chiffres communs
1:
        sbbl  %ecx,     (%edi)          # propage la retenue
        leal  4(%edi),  %edi
        jb    1b
        ret

        # si la = lb, soustrait les chiffres de poids fort jusqu 
        # trouver une diffrence
        ALIGN_4
L(compare):
        movl  -4(%esi,%ecx,4), %eax
        subl  -4(%ebx,%ecx,4), %eax
        ja    .Lsn_fsub_1
        jb    1f
        movl  %eax, -4(%edi,%ecx,4)
        loop  L(compare)
        ret
        ALIGN_4

        # ici la = lb et a[al-1] < b[la-1], donc a < b
1:
        xchgl  %esi,    %ebx
        call   .Lsn_fsub_1              # c <- b - a
        stc                             # retour avec CF = 1
        ret

# sn_fsub
# -------
# entre :
#   a = naturel de longueur la     esi = &a, edx = la
#   b = naturel de longueur lb     ebx = &b, ecx = lb
#   c = naturel de longueur la     edi = &c, peut tre confondu avec a ou b
# contraintes : la >= lb > 0
#
# sortie :
#   c <- a-b
#
# registres modifis :
#   esi = &a[la],  ebx = &b[lb],   edi = &c[la]
#   ecx = 0
#   eax,edx,ebp = ind.
#   CF  = retenue sortante
#
# sn_fsub_1
# ---------
# entre :
#   a = naturel de longueur l      esi = &a, ecx = l
#   b = naturel de longueur l      ebx = &b
#   c = naturel de longueur l      edi = &c, peut tre confondu avec a ou b
# contraintes : l > 0
#
# sortie :
#   c <- a-b
#
# registres modifis :
#   esi = &a[l],  ebx = &b[l],   edi = &c[l]
#   ecx = 0
#   eax,edx,ebp = ind.
#   CF  = retenue sortante

        ALIGN_32
.Lsn_fsub:

        subl   %ecx,    %edx            # edx <- la - lb
        jz     .Lsn_fsub_1
        pushl  %edx                     # sauve la - lb
        call   .Lsn_fsub_1              # soustrait les parties communes
        popl   %edx
        movl   %edx,    %ecx            # ecx <- la - lb
        not    %edx
        incl   %edx                     # edx <- lb - la
        cld;   REP(movsl)               # copie la fin de a
1:
        sbbl   %ecx,    (%edi,%edx,4)   # propage la retenue
        jnb    2f
        incl   %edx
        jne    1b
2:
        ret

        ALIGN_32
.Lsn_fsub_1:
        
        movl   (%esi),  %eax            # eax <- a[0]
        leal   (%esi,%ecx,4), %esi      # esi <- &a[l]
        leal   (%ebx,%ecx,4), %ebx      # ebx <- &b[l]
        leal   (%edi,%ecx,4), %edi      # edi <- &c[l]

        # calcule l adresse de saut dans la boucle
        decl   %ecx
        negl   %ecx
        jz     L(last)
        call   L(here)
L(here):
        popl   %ebp                     # ebp <- L(here)
        movl   %ecx,    %edx
        andl   $15,     %edx            # edx <- (1-l) mod 16
        xorl   %edx,    %ecx            # ecx <- 16*((1-l)/16)
        leal   (%edx,%edx,2), %edx      # ebp <- L(begin) + 12*reste
        leal   L(begin)-L(here)(%ebp,%edx,4), %ebp
        movl   %eax,    %edx
        jmp    *%ebp

        # corps de boucle  drouler. taille du code = 24 octets
        # entrer avec eax = edx = 1er chiffre de a, CF = 0
#undef BODY
#define BODY(x,y,z) \
          sbbl  x(%ebx,%ecx,4), %eax; \
      /*  movl  y(%esi,%ecx,4), %edx */ .byte 0x8B, 0x54, 0x8E, y; \
          movl  %eax, x(%edi,%ecx,4); \
      /*  sbbl  y(%ebx,%ecx,4), %edx */ .byte 0x1B, 0x54, 0x8B, y; \
          movl  z(%esi,%ecx,4), %eax; \
       /* movl  %edx, y(%edi,%ecx,4) */ .byte 0x89, 0x54, 0x8f, y

        # boucle de soustraction droule pour 16 chiffres
        ALIGN_4
L(begin):
        BODY(-4,0,4);   BODY(4,8,12);   BODY(12,16,20); BODY(20,24,28)
        BODY(28,32,36); BODY(36,40,44); BODY(44,48,52); BODY(52,56,60)

        leal   15(%ecx), %ecx
        incl   %ecx
        jne    L(begin)
L(last):
        sbbl   -4(%ebx), %eax           # termine la dernire soustraction
        movl   %eax,    -4(%edi)
        ret

#  chiffre xn(sub)(chiffre *a, long la, chiffre *b, long lb, chiffre *c)
# 
#  entre :
#  a = naturel de longueur la
#  b = naturel de longueur lb <= la
#  c = naturel de longueur la
# 
#  sortie :
#  c <- a - b
#  retourne la retenue

#ifdef assembly_sn_sub
#undef L
#define L(x) .Lsn_sub_##x

ENTER(sn_sub)

        movl   arg1,    %esi            # esi <- &a  
        movl   arg2,    %edx            # edx <- la 
        movl   arg3,    %ebx            # ebx <- &b  
        movl   arg4,    %ecx            # ecx <- lb 
        movl   arg5,    %edi            # edi <- &c
        jecxz  L(b_nul)                 # si b = 0, recopie a
        call   .Lsn_fsub                # sinon, effectue la soustraction
        movl   $0,      %eax            # eax <- retenue
        adcl   $0,      %eax
        RETURN_WITH_SP

L(b_nul):
        movl   %edx,    %ecx            # c <- a
        jecxz  L(a_nul)
        cld;   REP(movsl)
L(a_nul):
        xorl   %eax,    %eax            # eax <- 0 (retenue)
        RETURN_WITH_SP

#endif /* assembly_sn_sub */

                            # +------------------+
                            # |  Incrmentation  |
                            # +------------------+
# sn_finc
# -------
# entre :
#   a = naturel de longueur la     esi = &a, edx = la
#   b = naturel de longueur lb     ebx = &b, ecx = lb
# contraintes : la >= lb > 0
#
# sortie :
#   a <- a+b
#
# registres modifis :
#   esi = &a[la],  ebx = &b[lb]
#   ecx = 0
#   eax,edx,ebp = ind.
#   CF  = retenue sortante
#
# sn_finc_1
# ---------
# entre :
#   a = naturel de longueur l      esi = &a, ecx = l
#   b = naturel de longueur l      ebx = &b
# contraintes : l > 0
#
# sortie :
#   a <- a+b
#
# registres modifis :
#   esi = &a[l],  ebx = &b[l]
#   ecx = 0
#   eax,edx,ebp = ind.
#   CF = retenue sortante
#
# remarque :
#  sn_finc_1 peut tre utilis pour diviser par 1 - BASE^p si p >= 2

#undef L                
#define L(x) .Lsn_finc_##x

        ALIGN_32
.Lsn_finc:
        
        subl   %ecx,    %edx            # edx <- la - lb
        jz     .Lsn_finc_1
        pushl  %edx                     # sauve la - lb
        call   .Lsn_finc_1              # additionne les parties communes
        popl   %edx                     # edx <- la - lb
        leal   (%esi,%edx,4), %esi      # esi <- &a[la]
        not    %edx
        incl   %edx
1:
        adcl   %ecx,    (%esi,%edx,4)   # propage la retenue
        jnc    2f
        incl   %edx
        jne    1b
2:
        ret
        
        ALIGN_32
.Lsn_finc_1:
        
        movl   (%ebx),  %eax            # eax <- b[0]
        leal   (%esi,%ecx,4), %esi      # esi <- &a[l]
        leal   (%ebx,%ecx,4), %ebx      # ebx <- &b[l]

        # calcule l adresse de saut dans la boucle
        decl   %ecx
        negl   %ecx
        jz     L(last)
        call   L(here)
L(here):
        popl   %ebp                     # ebp <- L(here)
        movl   %ecx,    %edx
        andl   $15,     %edx            # edx <- (1-l) mod 16
        xorl   %edx,    %ecx            # ecx <- 16*((1-l)/16)
        leal   (%edx,%edx,2), %edx      # ebp <- L(begin) + 12*reste
        leal   L(begin)-L(here)(%ebp,%edx,4), %ebp
        movl   %eax,    %edx
        jmp    *%ebp

        # corps de boucle  drouler. taille du code = 24 octets
        # entrer avec eax = edx = 1er chiffre de b, CF = 0
#undef BODY
#define BODY(x,y,z) \
          adcl  x(%esi,%ecx,4), %eax; \
       /* movl  y(%ebx,%ecx,4), %edx */ .byte 0x8B, 0x54, 0x8B, y ; \
          movl  %eax, x(%esi,%ecx,4); \
       /* adcl  y(%esi,%ecx,4), %edx */ .byte 0x13, 0x54, 0x8E, y; \
          movl  z(%ebx,%ecx,4), %eax; \
       /* movl  %edx, y(%esi,%ecx,4) */ .byte 0x89, 0x54, 0x8e, y

        # boucle d addition droule pour 16 chiffres
        ALIGN_4
L(begin):
        BODY(-4,0,4);   BODY(4,8,12);   BODY(12,16,20); BODY(20,24,28)
        BODY(28,32,36); BODY(36,40,44); BODY(44,48,52); BODY(52,56,60)

        leal   15(%ecx), %ecx
        incl   %ecx
        jne    L(begin)
L(last):
        adcl   %eax,    -4(%esi)        # termine la dernire addition
        ret

#  chiffre xn(inc)(chiffre *a, long la, chiffre *b, long lb)
#  entre :
#  a = naturel de longueur la
#  b = naturel de longueur lb avec lb <= la
#
#  sortie :
#  a <- a + b
#  retourne la retenue

#ifdef assembly_sn_inc
#undef L
#define L(x) .Lsn_inc_##x

ENTER(sn_inc)
        movl   arg1,    %esi            # esi <- &a  
        movl   arg2,    %edx            # edx <- la  
        movl   arg3,    %ebx            # ebx <- &b 
        movl   arg4,    %ecx            # ecx <- lb
        jecxz  L(b_nul)                 # si b = 0, c est fini
        call   .Lsn_finc                # sinon, effectue l incrmentation
        movl   $0,      %eax            # eax <- retenue
        adcl   $0,      %eax
        RETURN_WITH_SP
        
L(b_nul):
        xorl   %eax,    %eax            # eax <- 0 (retenue)
        RETURN_WITH_SP

#endif /* assembly_sn_inc */
        

                            # +------------------+
                            # |  Dcrmentation  |
                            # +------------------+

# sn_fdec
# -------
# entre :
#   a = naturel de longueur la     esi = &a, edx = la
#   b = naturel de longueur lb     ebx = &b, ecx = lb
# contraintes : la >= lb > 0
#
# sortie :
#   a <- a-b
#
# registres modifis :
#   esi = &a[la],  ebx = &b[lb]
#   ecx = 0
#   eax,edx,ebp = ind.
#   CF  = retenue sortante
#
# sn_fdec_1
# ---------
# entre :
#   a = naturel de longueur l      esi = &a, ecx = l
#   b = naturel de longueur l      ebx = &b
# contraintes : l > 0
#
# sortie :
#   a <- a-b
#
# registres modifis :
#   esi = &a[l],  ebx = &b[l]
#   ecx = 0
#   eax,edx,ebp = ind.
#   CF  = retenue sortante

#undef L                
#define L(x) .Lsn_fdec_##x

        ALIGN_32
.Lsn_fdec:
        
        subl   %ecx,    %edx            # edx <- la - lb
        jz     .Lsn_fdec_1
        pushl  %edx                     # sauve la - lb
        call   .Lsn_fdec_1              # soustrait les parties communes
        popl   %edx                     # edx <- la - lb
        leal   (%esi,%edx,4), %esi      # esi <- &a[la]
        not    %edx
        incl   %edx
1:
        sbbl   %ecx,    (%esi,%edx,4)   # propage la retenue
        jnb    2f
        incl   %edx
        jne    1b
2:
        ret

        ALIGN_32
.Lsn_fdec_1:
        
        movl   (%esi),  %eax            # eax <- a[0]
        leal   (%esi,%ecx,4), %esi      # esi <- &a[l]
        leal   (%ebx,%ecx,4), %ebx      # ebx <- &b[l]

        # calcule l adresse de saut dans la boucle
        decl   %ecx
        negl   %ecx
        jz     L(last)
        call   L(here)
L(here):
        popl   %ebp                     # ebp <- L(here)
        movl   %ecx,    %edx
        andl   $15,     %edx            # edx <- (1-l) mod 16
        xorl   %edx,    %ecx            # ecx <- 16*((1-l)/16)
        leal   (%edx,%edx,2), %edx      # ebp <- L(begin) + 12*reste
        leal   L(begin)-L(here)(%ebp,%edx,4), %ebp
        movl   %eax,    %edx
        jmp    *%ebp

        # corps de boucle  drouler. taille du code = 24 octets
        # entrer avec eax = edx = 1er chiffre de a, CF = 0
#undef BODY
#define BODY(x,y,z) \
          sbbl  x(%ebx,%ecx,4), %eax; \
       /* movl  y(%esi,%ecx,4), %edx */ .byte 0x8B, 0x54, 0x8E, y; \
          movl  %eax, x(%esi,%ecx,4); \
       /* sbbl  y(%ebx,%ecx,4), %edx */ .byte 0x1B, 0x54, 0x8B, y; \
          movl  z(%esi,%ecx,4), %eax; \
       /* movl  %edx, y(%esi,%ecx,4) */ .byte 0x89, 0x54, 0x8e, y
	
        # boucle de soustraction droule pour 16 chiffres
        ALIGN_4        
L(begin):
        BODY(-4,0,4);   BODY(4,8,12);   BODY(12,16,20); BODY(20,24,28)
        BODY(28,32,36); BODY(36,40,44); BODY(44,48,52); BODY(52,56,60)

        leal   15(%ecx), %ecx
        incl   %ecx
        jne    L(begin)
L(last):
        sbbl   -4(%ebx), %eax           # termine la dernire soustraction
        movl   %eax,    -4(%esi)
        ret

#  chiffre xn(dec)(chiffre *a, long la, chiffre *b, long lb)
#  entre :
#  a = naturel de longueur la
#  b = naturel de longueur lb avec lb <= la
#
#  sortie :
#  a <- a - b
#  retourne la retenue

#ifdef assembly_sn_dec
#undef L
#define L(x) .Lsn_dec_##x

ENTER(sn_dec)
        
        movl   arg1,    %esi            # esi <- a  
        movl   arg2,    %edx            # edx <- la  
        movl   arg3,    %ebx            # ebx <- b 
        movl   arg4,    %ecx            # ecx <- lb
        jecxz  L(b_nul)                 # si b = 0, c est fini
        call   .Lsn_fdec                # sinon, effectue la dcrmentation
        movl   $0,      %eax            # eax <- retenue
        adcl   $0,      %eax
        RETURN_WITH_SP

L(b_nul):
        xorl   %eax,    %eax            # eax <- 0 (retenue)
        RETURN_WITH_SP
        
#endif /* assembly_sn_dec */

#endif /* use_sse2 */
