/* upsc - simple "client" to test communications 

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "common.h"

#include <netdb.h>
#include <netinet/in.h>
#include <sys/socket.h>

#include "upsclient.h"

static void help(char *prog)
{
	printf("Network UPS Tools upsc %s\n\n", UPS_VERSION);

	printf("usage: %s <ups> [<variable>]\n", prog);

	printf("\nDemo program to display UPS variables.\n\n");

	printf("  <ups>      - upsd server, <upsname>@<hostname>[:<port>] form\n");
	printf("  <variable> - optional, display this variable only.\n");
	printf("               Default: list all variables for <host>\n");

	exit(0);
}

static void clean_exit(UPSCONN *ups, char *upsname, char *hostname, int code)
{
	if (upsname)
		free(upsname);

	if (hostname)
		free(hostname);

	upscli_disconnect(ups);

	exit(code);
}

static int old_printvar(UPSCONN *ups, const char *upsname, const char *var)
{
	char	value[SMALLBUF];

	if (upscli_getvar(ups, upsname, var, value, sizeof(value)) < 0) {
		fprintf(stderr, "Error: %s\n", upscli_strerror(ups));
		return 1;
	}

	printf("%s\n", value);

	return 0;
}

static int new_printvar(UPSCONN *ups, const char *upsname, const char *var)
{
	int	ret, numq, numa;
	const	char	*query[4];
	char	**answer;

	/* old-style variable name? */
	if (!strchr(var, '.'))
		return old_printvar(ups, upsname, var);

	if (!upsname) {
		fprintf(stderr, "Error: ups name must be defined\n");
		return 1;
	}

	query[0] = "VAR";
	query[1] = upsname;
	query[2] = var;

	numq = 3;

	ret = upscli_get(ups, numq, query, &numa, &answer);

	if (ret < 0) {

		/* new var and old upsd?  try to explain the situation */
		if (upscli_upserror(ups) == UPSCLI_ERR_UNKCOMMAND) {
			fprintf(stderr, "Error: variable unknown (old upsd detected)\n");
			return 1;
		}

		fprintf(stderr, "Error: %s\n", upscli_strerror(ups));
		return 1;
	}

	if (numa < numq) {
		fprintf(stderr, "Error: insufficient data "
			"(got %d args, need at least %d)\n", numa, numq);
		return 1;
	}

	printf("%s\n", answer[3]);
	return 0;
}

static int old_list(UPSCONN *ups, const char *upsname)
{
	char	vars[LARGEBUF], *v, *ptr;

	if (upscli_getlist(ups, upsname, UPSCLI_LIST_VARS, vars, 
		sizeof(vars)) < 0) {
		fprintf(stderr, "Unable to get variable list - %s\n", 
		        upscli_strerror(ups));

		return 1;
	}

	if (upsname)
		printf("UPS: %s\n", upsname);
	else
		printf("UPS: (default)\n");

	if (strlen(vars) == 0) {
		fprintf(stderr, "No data available - check your "
			"configuration (ups.conf)\n");

		return 1;
	}

	v = vars;
	while (v != NULL) {
		ptr = strchr(v, ' ');
		if (ptr)
			*ptr++ = '\0';

		printf("%s: ", v);
		old_printvar(ups, upsname, v);

		v = ptr;
	}		

	return 0;
}

static int new_list(UPSCONN *ups, const char *upsname)
{
	int	ret, numq, numa;
	const	char	*query[4];
	char	**answer;

	if (!upsname)
		return old_list(ups, upsname);

	query[0] = "VAR";
	query[1] = upsname;
	numq = 2;

	ret = upscli_list_start(ups, numq, query);

	if (ret < 0) {

		/* old upsd --> fall back on old LISTVARS technique */
		if (upscli_upserror(ups) == UPSCLI_ERR_UNKCOMMAND)
			return old_list(ups, upsname);

		fprintf(stderr, "Error: %s\n", upscli_strerror(ups));
		return 1;
	}

	ret = upscli_list_next(ups, numq, query, &numa, &answer);

	while (ret == 1) {

		/* VAR <upsname> <varname> <val> */
		if (numa < 4) {
			fprintf(stderr, "Error: insufficient data "
				"(got %d args, need at least 4)\n", numa);
			return 1;
		}

		printf("%s: %s\n", answer[2], answer[3]);

		ret = upscli_list_next(ups, numq, query, &numa, &answer);
	}

	return 0;
}

int main(int argc, char **argv)
{
	int	port, ret;
	char	*upsname, *hostname;
	UPSCONN	ups;

	if (argc < 2)
		help(argv[0]);

	/* special cases since we're not using getopt */
	if (!strcmp(argv[1], "-V")) {
		printf("Network UPS Tools upsc %s\n", UPS_VERSION);
		exit(0);
	}

	if (!strcmp(argv[1], "-h"))
		help(argv[0]);

	upscli_splitname(argv[1], &upsname, &hostname, &port);

	if (upscli_connect(&ups, hostname, port, UPSCLI_CONN_TRYSSL) < 0) {
		fprintf(stderr, "Error: %s\n", upscli_strerror(&ups));

		clean_exit(&ups, upsname, hostname, 1);
	}

	if (argc >= 3) {
		ret = new_printvar(&ups, upsname, argv[2]);
		clean_exit(&ups, upsname, hostname, ret);
	}

	ret = new_list(&ups, upsname);
	clean_exit(&ups, upsname, hostname, ret);

	/* NOTREACHED */
	return 0;
}
