/* apcsmart.c - model specific routines for APC smart protocol units

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "main.h"
#include "apcsmart.h"

#define DRV_VERSION "0.72"

	extern	int	sddelay;
	int	cap_measureupsii = 0, upslevel = 0;
	int	shutdowncmdset = 0; /* The shutdown command set to use */

#define ALT_CABLE_1 "940-0095B"

/* try to find AMBTEMP, CONTACTS and AMBHUMID */
static void test_measureupsii(void)
{
	int	ret;
	char	temp[256];

	upssendchar(REQ_AMBTEMP);
	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if ((ret < 1) || (!strcmp(temp, "NA")))	/* not supported */
		return;

	/* we got something, so it must be supported */

	cap_measureupsii = 1;		/* log this fact */

	dstate_setinfo("ambient.temperature", "%s", temp);

	/* Now probe the contacts */

	upssendchar(REQ_CONTACTS);
	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if ((ret > 0) && (strcmp(temp, "NA") != 0))
		dstate_setinfo("ups.contacts", "%s", temp);

	/* now try for ambient humidity support */

	upssendchar(REQ_AMBHUMID);
	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if ((ret > 0) && (strcmp(temp, "NA") != 0))
		dstate_setinfo("ambient.humidity", "%s", temp);
}

static void addcap(char cmdchar, const char *value)
{
	switch (cmdchar) {
		case 'u': dstate_addenum("input.transfer.high", "%s", value);
			  dstate_setflags("input.transfer.high", ST_FLAG_RW);
			  break;

		case 'l': dstate_addenum("input.transfer.low", "%s", value);
			  dstate_setflags("input.transfer.low", ST_FLAG_RW);
			  break;

		case 'e': dstate_addenum("battery.charge.restart", "%s", value);
			  dstate_setflags("battery.charge.restart", ST_FLAG_RW);
			  break;

		case 'o': break;	/* output voltages, unused now	*/

		case 's': dstate_addenum("input.sensitivity", "%s", value);
			  dstate_setflags("input.sensitivity", ST_FLAG_RW);
			  break;

		case 'q': break;	/* TODO: low batt warning	*/

		case 'p': dstate_addenum("ups.delay.shutdown", "%s", value);
			  dstate_setflags("ups.delay.shutdown", ST_FLAG_RW);
			  break;

		case 'k': dstate_addenum("battery.alarm.threshold", "%s", value);
			  dstate_setflags("battery.alarm.threshold", ST_FLAG_RW);
			  break;

		case 'r': dstate_addenum("ups.delay.start", "%s", value);
			  dstate_setflags("ups.delay.start", ST_FLAG_RW);
			  break;

		case 'E': break;	/* TODO: selftest intervals	*/
		case 0x06: break;	/* TODO: find out what ^F is	*/
		case 0x0c: break;	/* TODO: front panel language	*/
		case 'w': break;	/* TODO: runtime conservation	*/
		default:
			printf("Unrecognized cmdchar %c, value %s\n",
			        cmdchar, value);
	}		
}	

static void poll_data(const char *var, char reqchar)
{
	int	ret;
	char	tmp[SMALLBUF];

	upssendchar(reqchar);
	ret = upsrecv(tmp, sizeof(tmp), ENDCHAR, IGNCHARS);

	if ((ret < 0) || (!strcmp(tmp, "NA")))
		dstate_delinfo(var);
	else
		dstate_setinfo(var, "%s", tmp);
}

static void do_capabilities(void)
{
	char	upsloc, temp[512], *ptr, cmd, loc, *entptr, etmp[16], *endtemp;
	int	nument, entlen, i, matrix, ret;

	/* just in case ... */
	usleep(50000);

	/* get location byte for later comparisons */
	upssendchar('V');

	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if ((ret < 1) || (!strcmp(temp, "NA"))) {
		upslogx(LOG_ERR, "Get location byte failed");
		return;
	}

	upsloc = temp[strlen(temp) - 1];

	/* add base entries for these variables */

	/* create them as normal variables for now - enum and rw flags    *
	 * get added later if the model actually supports such operations */

	poll_data("input.transfer.low", REQ_LOWXFER);
	poll_data("input.transfer.high", REQ_HIGHXFER);
	poll_data("ups.delay.start", REQ_WAKEDELAY);
	poll_data("input.sensitivity", REQ_LINESENS);
	poll_data("ups.delay.shutdown", REQ_GRACEDELAY);
	poll_data("battery.charge.restart", REQ_RTHRESH);
	poll_data("battery.alarm.threshold", REQ_ALRMDEL);

	/* battery change date */
	poll_data("battery.date", REQ_BATTDATE);

	/* writable string, max 8 bytes */
	dstate_setflags("battery.date", ST_FLAG_RW | ST_FLAG_STRING);
	dstate_setaux("battery.date", 8);

	/* get capability string */
	upssendchar(REQ_CAPABILITIES);	/* ^Z */

	snprintf(temp, sizeof(temp), "NA");
	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if ((ret == -1) || (!strcmp(temp, "NA"))) {
		/* Early Smart-UPS, not as smart as later ones */
		return;
	}

	/* recv always puts a \0 at the end, so this is safe */
	/* however it assumes a zero byte cannot be embedded */
	endtemp = &temp[0] + strlen(temp);

	if (temp[0] != '#') {
		printf("Unrecognized capability start char %c\n", temp[0]);
		printf("Please report this error [%s]\n", temp);
		return;
	}

	if (temp[1] == '#') {		/* Matrix-UPS */
		matrix = 1;
		ptr = &temp[0];
	}
	else {
		ptr = &temp[1];
		matrix = 0;
	}

	/* command char, location, # of entries, entry length */

	while (ptr[0] != '\0') {
		if (matrix)
			ptr += 2;	/* jump over repeating ## */

		/* check for idiocy */
		if (ptr >= endtemp) {
			printf("Capability string has overflowed\n");
			printf("Please report this error\n");
			fatalx("ERROR: capability overflow!");
		}

		cmd = ptr[0];
		loc = ptr[1];
		nument = ptr[2] - 48;
		entlen = ptr[3] - 48;
		entptr = &ptr[4];

		for (i = 0; i < nument; i++) {
			snprintf(etmp, entlen + 1, "%s", entptr);
			if ((loc == upsloc) || (loc == '4'))
				addcap (cmd, etmp);
			entptr += entlen;
		}
		ptr = entptr;
	}
}

static int do_status(void)
{
	char	temp[256];
	int	ret, tval;

	/* parse the status */
	upssendchar(REQ_STATUS);
	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if ((ret < 1) || (!strcmp(temp, "NA"))) {
		dstate_datastale();
		return 0;
	}

	/* APC SU600 models return this when the FPTEST is still active */
	if (!strncmp(temp, "BC", 2))
		return 1;		/* bogus, we don't want to parse it */

	tval = strtol(temp, 0, 16);

	status_init();

	if (tval & 1)
		status_set("CAL");		/* calibration */
	if (tval & 2)
		status_set("TRIM");		/* SmartTrim */
	if (tval & 4)
		status_set("BOOST");		/* SmartBoost */
	if (tval & 8)
		status_set("OL");		/* on line */
	if (tval & 16)
		status_set("OB");		/* on battery */
	if (tval & 32)
		status_set("OVER");		/* overload */
	if (tval & 64)
		status_set("LB");		/* low battery */
	if (tval & 128)
		status_set("RB");		/* replace batt */

	if (!strcmp(temp, "00"))
		status_set("OFF");

	status_commit();
	dstate_dataok();

	return 1;
}

static void getbaseinfo(const char *port)
{
	char	temp[256];
	int	silentfail = 0, oldmatrix = 0;
	int	ret = 0;

	upslevel = 1;

	/* really old models ignore REQ_MODEL, so find them first */
	upssendchar(REQ_MODEL);
	snprintf(temp, sizeof(temp), "NA");

	/* disable timeout complaints temporarily */
	flag_timeoutfailure = -1;
	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);
	flag_timeoutfailure = 0;

	if ((ret == -1) || (!strcmp(temp, "NA"))) {
		/* force the model name */
		dstate_setinfo("ups.model", "Smart-UPS");
		silentfail = 1;
	}
	else
		dstate_setinfo("ups.model", "%s", temp);

	/* see if this might be an old Matrix-UPS instead */
	upssendchar('/');
	flag_timeoutfailure = -1;

	snprintf(temp, sizeof(temp), "NA");
	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);
	flag_timeoutfailure = 0;

	/* it returned a useful number, so it must be a Matrix-UPS */
	if ((ret != -1) && (strcmp(temp, "NA")) != 0) {
		oldmatrix = 1;	/* doesn't do 'b' */
		dstate_setinfo("ups.model", "Matrix-UPS");
	}

	poll_data("ups.serial", REQ_SERIAL);

	do_status();

	/* upslevel 1: Smart-UPS v/s, older Back-UPS Pros */
	upssendchar(REQ_UTILITY);

	snprintf(temp, sizeof(temp), "NA");
	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);
	if ((!strcmp(temp, "NA")) || (ret == -1)) {
		upslevel = 1;
		return;
	}

	/* upslevel 2: at least a modern Smart-UPS / Back-UPS Pro */
	upslevel = 2;

	/* add other things to monitor and prime them with values */

	poll_data("input.voltage", REQ_UTILITY);
	poll_data("input.frequency", REQ_ACFREQ);
	poll_data("ups.load", REQ_LOADPCT);
	poll_data("battery.charge", REQ_BATTPCT);
	poll_data("battery.voltage", REQ_BATTVOLT);
	poll_data("output.voltage", REQ_OUTVOLT);

	/* upslevel 3: at least a Smart-UPS (beyond the Back-UPS Pro) */

	/* test for UPSTEMP support */
	snprintf(temp, sizeof(temp), "NA");

	upssendchar(REQ_UPSTEMP);
	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if ((ret > 0) && (strcmp(temp, "NA") != 0)) {
		upslevel = 3;
		dstate_setinfo("ups.temperature", "%s", temp);
	}

	poll_data("ups.id", REQ_UPSIDENT);

	/* another 8 char rw string */
	dstate_setflags("ups.id", ST_FLAG_RW | ST_FLAG_STRING);
	dstate_setaux("ups.id", 8);

	/* now add the instant commands */
	dstate_addcmd("test.panel.start");
	dstate_addcmd("test.battery.start");
	dstate_addcmd("load.off");
	dstate_addcmd("load.on");
	dstate_addcmd("calibrate.start");
	dstate_addcmd("calibrate.stop");

	do_capabilities();

	/* that's all for 'silent failure' units */
	if (silentfail)
		return;

	/* see if we have a Measure-UPS II connected */
	test_measureupsii();
}

/* get the UPS talking to us in smart mode */
static int smartmode(void)
{
	int	tries, ret;
	char	temp[256];

	for (tries = 0; tries < 5; tries++) {
		upssendchar(GO_SMART);
		ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

		if ((ret > 0) && (!strcmp(temp, "SM")))
			return 1;	/* success */

		sleep(1);	/* wait before trying again */
	}

	return 0;	/* failure */
}

/* normal idle loop - keep up with the current state of the UPS */
void upsdrv_updateinfo(void)
{
	/* try to wake up a dead ups once in awhile */
	if (flag_timeoutfailure == 1) {

		if (!smartmode) {
			dstate_datastale();
			return;
		}
	}

	if (!do_status())
		return;

	/* upslevel 1 only has a status readback */
	if (upslevel < 2)
		return;

	poll_data("input.voltage", REQ_UTILITY);
	poll_data("input.frequency", REQ_ACFREQ);
	poll_data("ups.load", REQ_LOADPCT);
	poll_data("battery.charge", REQ_BATTPCT);
	poll_data("battery.voltage", REQ_BATTVOLT);
	poll_data("output.voltage", REQ_OUTVOLT);

	/* need at least 3 for ups temperature and Measure-UPS II */
	if (upslevel < 3)
		return;

	poll_data("ups.temperature", REQ_UPSTEMP);

	if (cap_measureupsii) {
		poll_data("ambient.humidity", REQ_AMBHUMID);
		poll_data("ambient.temperature", REQ_AMBTEMP);
		poll_data("ups.contacts", REQ_CONTACTS);
	}
}

/* TODO: roll this into upscommon ala bestups */
static void sendstring(char * string, int len, int delay)
{
	int	i;

	for (i=0; (i<len); i++) {
	        upssendchar(string[i]);
		usleep(delay);
	}
}

/* 940-0095B support: set DTR, lower RTS */
static void init_serial_0095B(void)
{
	int	dtr_bit = TIOCM_DTR;
	int	rts_bit = TIOCM_RTS;

	ioctl(upsfd, TIOCMBIS, &dtr_bit);
	ioctl(upsfd, TIOCMBIC, &rts_bit);
}

/* set a value in the hardware using the <cmdchar> '-' (repeat) approach */
static void hw_set(char cmdchar, const char *newval)
{
	char	temp[256], orig[256];
	int	tries, ret;

	upssendchar(cmdchar);
	ret = upsrecv(orig, sizeof(orig), ENDCHAR, IGNCHARS);

	if ((ret < 1) || (!strcmp(orig, "NA")))
		return;

	/* don't change it if already set - save wear and tear on the eeprom */
	if (!strcmp(orig, newval))
		return;

	tries = 0;
	while (tries < 6) {
		upssendchar(NEXT_VAL);
		ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

		if ((ret < 1) || (!strcmp(temp, "NO")))
			return;

		upssendchar(cmdchar);
		ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

		if ((ret < 1) || (!strcmp(temp, "NO")))
			return;

		if (!strcmp(temp, newval)) 		/* got it */
			return;

		if (!strcmp(temp, orig)) {		/* wrapped around */
			upslogx(LOG_ERR, "setvar: variable %c wrapped!\n",
			        cmdchar);
			return;
		}

		tries++;
	}

	upslogx(LOG_ERR, "setvar: gave up after 6 tries\n");
}

/* set string values in the UPS EEPROM (battery date, local identity) */
static void setstring(const char *data, char reqchar)
{
	char	temp[256];
	int	i, ret;

	upssendchar(reqchar);
	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if ((ret < 1) || (!strcmp(temp, "NO")))
		return;

	upssendchar(NEXT_VAL);
	usleep(50000);

	for (i = 0; i < strlen(data); i++) {
		upssendchar(data[i]);
		usleep(50000);
	}

	/* pad to 8 chars with CRs */
	for (i = strlen(data); i < 8; i++) {
		upssendchar(13);
		usleep(50000);
	}

	usleep(50000);
	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if ((ret < 1)  || (!strcmp(temp, "NO")))
		return;

	usleep(50000);
}

int setvar(const char *varname, const char *val)
{
	if (!strcasecmp(varname, "ups.delay.start")) {
		hw_set(REQ_WAKEDELAY, val);
		poll_data("ups.delay.start", REQ_WAKEDELAY);
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "input.transfer.low")) {
		hw_set(REQ_LOWXFER, val);
		poll_data("input.transfer.low", REQ_LOWXFER);
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "input.transfer.high")) {
		hw_set(REQ_HIGHXFER, val);
		poll_data("input.transfer.high", REQ_HIGHXFER);
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "input.sensitivity")) {
		hw_set(REQ_LINESENS, val);
		poll_data("input.sensitivity", REQ_LINESENS);
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "ups.delay.shutdown")) {
		hw_set(REQ_GRACEDELAY, val);
		poll_data("ups.delay.shutdown", REQ_GRACEDELAY);
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "battery.charge.restart")) {
		hw_set(REQ_RTHRESH, val);
		poll_data("battery.charge.restart", REQ_RTHRESH);
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "battery.alarm.threshold")) {
		hw_set(REQ_ALRMDEL, val);
		poll_data("battery.alarm.threshold", REQ_ALRMDEL);
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "ups.id")) {
		setstring(val, REQ_UPSIDENT);
		poll_data("ups.id", REQ_UPSIDENT);
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "battery.date")) {
		setstring(val, REQ_BATTDATE);
		poll_data("battery.date", REQ_BATTDATE);
		return STAT_SET_HANDLED;
	}

	upslogx(LOG_NOTICE, "setvar: unknown var [%s]", varname);
	return STAT_SET_UNKNOWN;
}

/* check for calibration status and either start or stop */
static void do_cal(int start)
{
	char	temp[256];
	int	tval, ret;

	upssendchar(REQ_STATUS);
	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if (!ret)		/* didn't get status, bail out */
		return;

	tval = strtol(temp, 0, 16);

	if (tval & 1) {		/* calibration currently happening */
		if (start == 0)	{		/* stop requested */
			upslogx(LOG_INFO, "Stopping runtime calibration");
			upssendchar(REQ_CAL);

			/* eat the reply */
			ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);
			return;
		}

		/* requested start while calibration still running */
		upslogx(LOG_INFO, "Runtime calibration already in progress");
		return;
	}

	/* calibration not happening */

	if (start == 0) {		/* stop requested */
		upslogx(LOG_INFO, "Runtime calibration not occurring");
		return;
	}

	upslogx(LOG_INFO, "Starting runtime calibration");
	upssendchar(REQ_CAL);

	/* eat the reply */
	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);
}

/* handle the off command with some paranoia */
static void do_off(void)
{
	static	time_t lastcmd = 0;
	time_t	now, elapsed;

#ifdef CONFIRM_DANGEROUS_COMMANDS
	time(&now);
	elapsed = now - lastcmd;

	/* reset the timer every call - this means if you call it too      *
	 * early, then you have to wait MINCMDTIME again before sending #2 */
	lastcmd = now;

	if ((elapsed < MINCMDTIME) || (elapsed > MAXCMDTIME)) {

		/* FUTURE: tell the user (via upsd) to try it again */
		/* msgreply (UPSMSG_REPAGAIN); */
		return;
	}
#endif

	upslogx(LOG_INFO, "Sending powerdown command to UPS\n");
	upssendchar(REQ_OFF);
	usleep(1500000);
	upssendchar(REQ_OFF);
}

int instcmd(const char *cmdname, const char *extra)
{
	char	temp[256];

	if (!strcasecmp(cmdname, "test.panel.start")) {
		upssendchar(REQ_FPTEST);
		upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);
		return STAT_INSTCMD_HANDLED;
	}

	if (!strcasecmp(cmdname, "test.battery.start")) {
		upssendchar(REQ_BTEST);
		return STAT_INSTCMD_HANDLED;
	}

	if (!strcasecmp(cmdname, "load.off")) {
		do_off();
		return STAT_INSTCMD_HANDLED;
	}

	if (!strcasecmp(cmdname, "load.on")) {
		upssendchar(REQ_ON);
		usleep(500000);
		upssendchar(REQ_ON);
		return STAT_INSTCMD_HANDLED;
	}

	if (!strcasecmp(cmdname, "calibrate.stop")) {
		do_cal(0);
		return STAT_INSTCMD_HANDLED;
	}

	if (!strcasecmp(cmdname, "calibrate.start")) {
		do_cal(1);
		return STAT_INSTCMD_HANDLED;
	}

	upslogx(LOG_NOTICE, "instcmd: unknown command [%s]", cmdname);
	return STAT_INSTCMD_UNKNOWN;
}

void upsdrv_banner(void)
{
	printf("Network UPS Tools - APC Smart protocol driver %s (%s)\n", 
		DRV_VERSION, UPS_VERSION);
	printf("\n");
	printf("*** This driver will be replaced by newapc in the future\n");
}

/* prep the serial port */
void upsdrv_initups(void)
{
	char	*cable;

	open_serial(device_path, B2400);

	cable = getval("cable");

	if (cable)
		if (!strcasecmp(cable, ALT_CABLE_1))
			init_serial_0095B();
}

/* set up the variables we intend to monitor */
void upsdrv_initinfo(void)
{
	if (smartmode() == 0) {
		printf("Unable to detect an APC Smart protocol UPS on port %s\n", 
			device_path);
		printf("Check the cabling, port name or model name and try again\n");
		exit(1);
	}

	getbaseinfo(device_name);

	/* hardcode the manufacturer - it's pretty obvious here */
        dstate_setinfo("ups.mfr", "APC");

	dstate_setinfo("driver.version.internal", "%s", DRV_VERSION);

	printf("Detected %s [%s] on %s (level %i)\n", 
		dstate_getinfo("ups.model"), dstate_getinfo("ups.serial"), 
		device_path, upslevel);

	if (cap_measureupsii)
		printf("Measure-UPS II detected!\n");

	/* install handlers */
	upsh.new_instcmd = instcmd;
	upsh.new_setvar = setvar;
}

/* power down the attached load immediately */
void upsdrv_shutdown(void)
{
	char	temp[32];
	int	tval, ret;

	if (smartmode() == 0)
		upslogx(LOG_WARNING, "Warning: unable to contact the UPS, but trying the shutdown anyway.");
	
	/* check the line status */

	upssendchar(REQ_STATUS);
	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if ((ret < 1) || (!strcmp(temp, "NA"))) {
		printf("Warning: status unavailable - assuming on battery\n");
		tval = 64 | 16;		/* LB + OB */
	} else {
		tval = strtol(temp, 0, 16);
	}

	if (testvar("sdtype"))
		shutdowncmdset = atoi(getval("sdtype"));

	switch (shutdowncmdset) {
		case 3:		/* shutdown with grace period */
			printf("Sending delayed power off command to UPS\n");
			upssendchar(REQ_SHUTDOWN);
			usleep(1500000);
			upssendchar(REQ_SHUTDOWN);
			break;

		case 2:		/* instant shutdown */
			printf("Sending power off command to UPS\n");
			upssendchar(REQ_OFF);
			usleep(1500000);
			upssendchar(REQ_OFF);
			break;

		case 1:

	/* Send a combined set of shutdown commands which can work better */
	/* if the UPS gets power during shutdown process */
	/* Specifically it sends both the soft shutdown 'S' */
	/* and the powerdown after grace period - '@000' commands */
			printf("UPS - currently %s - sending shutdown/powerdown\n",
			       (tval & 8) ? "on-line" : "on battery");
			sendstring("S@000", 5, 50000);
			break;

		default:

		/* @000 - shutdown after 'p' grace period             */
		/*      - returns after 000 minutes (i.e. right away) */

		/* S    - shutdown after 'p' grace period, only on battery */
		/*        returns after 'e' charge % plus 'r' seconds      */

			if (tval & 8) {                 /* on line */
				printf("On line, sending shutdown+return command...\n");
				sendstring("@000", 4, 50000);
			}
			else {
				printf("On battery, sending normal shutdown command...\n");
				upssendchar('S');
			}
	}
}

/* throw in our two cents when someone calls for -h from main */
void upsdrv_help(void)
{
	printf("\nShutdown types:\n");
	printf("  0: soft shutdown or powerdown, depending on battery status\n");
	printf("  1: soft shutdown followed by powerdown\n");
	printf("  2: instant power off\n");
	printf("  3: power off with grace period\n");
	printf("Modes 0-1 will make the UPS come back when power returns\n");
	printf("Modes 2-3 will make the UPS stay turned off when power returns\n");
}

/* list flags and values that you want to receive via -x */
void upsdrv_makevartable(void)
{
	addvar(VAR_VALUE, "cable", "Specify alternate cable (940-0095B)");
	addvar(VAR_VALUE, "sdtype", "Specify shutdown type (1-3)");
}

void upsdrv_cleanup(void)
{
}

