/* belkin.c - model specific routines for Belkin Smart-UPS units.

   Copyright (C) 2000 Marcus Mller <marcus@ebootis.de>
     
   based on:

   apcsmart.c - model specific routines for APC smart protocol units

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "main.h"
#include "belkin.h"

#define DRV_VERSION "0.11"

int init_communication(void)
{
	int	i;
	int	res;
	char	temp[SMALLBUF];

	res = -1;
	for (i = 1; i <= 10 && res == -1; i++) {
		send_belkin_command(STATUS,MANUFACTURER,"");
    		res = get_belkin_reply(temp);
	}

	if (res == -1 || strcmp(temp,"BELKIN")) 
		return res;

	return 0;
}

void do_status(void)
{
	char	temp[SMALLBUF], st[SMALLBUF];
	int	res;

	send_belkin_command(STATUS,STAT_STATUS,"");
	res = get_belkin_reply(temp);
	if (res == -1) {
		dstate_datastale();
		return;
	}

	status_init();

	get_belkin_field(temp, st, sizeof(st), 6);
	if (*st == '1') {
		status_set("OFF");

	} else {	/* (OFF) and (OB | OL) are mutually exclusive */

		get_belkin_field(temp, st, sizeof(st), 2);
		if (*st == '1') {
			status_set("OB");

			send_belkin_command(STATUS,STAT_BATTERY,"");
			res = get_belkin_reply(temp);
			if (res == -1)
				return;

			get_belkin_field(temp, st, sizeof(st), 10);
			res = atoi(st);
			get_belkin_field(temp, st, sizeof(st), 2);

			if (*st == '1' || res < LOW_BAT) 
				status_set("LB");		/* low battery */
		}
		else
			status_set("OL");	/* on line */
	}

	status_commit();
	dstate_dataok();
}

int init_ups_data(void)
{
	int	res;
	double	low, high;
	char	temp[SMALLBUF], st[SMALLBUF];

	send_belkin_command(STATUS, MODEL, "");
	res = get_belkin_reply(temp);
	if (res == -1)
		return res;

	dstate_setinfo("ups.model", "%s", temp);

	send_belkin_command(STATUS, VERSION, "");
	res = get_belkin_reply(temp);
	if (res == -1)
		return (res);

	dstate_setinfo("ups.firmware", "%s", temp);

	send_belkin_command(STATUS, RATING, "");
	res = get_belkin_reply(temp);

	get_belkin_field(temp, st, sizeof(st), 8);
	low = atof(st) / 0.88;

	get_belkin_field(temp, st, sizeof(st), 9);
	high = atof(st) * 0.88;

	tcflush(upsfd,TCIOFLUSH);

	dstate_setinfo("input.transfer.low", "%03.1f", low);
	dstate_setinfo("input.transfer.high", "%03.1f", high);

	dstate_addcmd("load.off");
	dstate_addcmd("load.on");
	upsdrv_updateinfo();
	return 0;
}

/* normal idle loop - keep up with the current state of the UPS */
void upsdrv_updateinfo(void)
{
	int	res;
	double	val;
	char	temp[SMALLBUF], st[SMALLBUF];

	do_status();

	send_belkin_command(STATUS, STAT_INPUT, "");
	res = get_belkin_reply(temp);
	if (res == -1)
		return;

	get_belkin_field(temp, st, sizeof(st), 3);
	val = atof(st) / 10;
	dstate_setinfo("input.voltage", "%05.1f", val);

	send_belkin_command(STATUS,STAT_BATTERY, "");
	res = get_belkin_reply(temp);
	if (res == -1)
		return;

	get_belkin_field(temp, st, sizeof(st), 10);
	val = atof(st);
	dstate_setinfo("battery.charge", "%03.0f", val);

	get_belkin_field(temp, st, sizeof(st), 7);
	val = atof(st) / 10;
	dstate_setinfo("battery.voltage", "%4.1f", val);

	get_belkin_field(temp, st, sizeof(st), 9);
	val = atof(st);
	dstate_setinfo("ups.temperature", "%03.0f", val);

	send_belkin_command(STATUS, STAT_OUTPUT, "");
	res = get_belkin_reply(temp);
	if (res == -1)
		return;

	get_belkin_field(temp, st, sizeof(st), 2);
	val = atof(st) / 10;
	dstate_setinfo("input.frequency", "%.1f", val);

	get_belkin_field(temp, st, sizeof(st), 4);
	val = atof(st) / 10;
	dstate_setinfo("output.voltage", "%05.1f", val);

	get_belkin_field(temp, st, sizeof(st), 7);
	val = atof(st);
	dstate_setinfo("ups.load", "%03.0f", val);
}


void get_belkin_field(char *temp, char *data, size_t datalen, int n)
{
	char	st[SMALLBUF];
	int	i = 0, f = 1;

	strlcpy(st, temp, sizeof(st));

	while (f < n) {
		while (st[i] && st[i] != ';') 
			i++;

		st[i++] = 0;
		f++;
	}

	f = i;

	while (st[i] && st[i] != ';') 
		i++;

	st[i] = 0;

	snprintf(data, datalen, "%s", st+f);
}

int get_belkin_reply(char *buf)
{
	int	res, cnt;

	res = recvbinary(buf, 7);

	if (res)
		return res;

	buf[7] = 0;
	cnt = atoi(buf + 4);
	res = recvbinary(buf, cnt);
	buf[cnt] = 0;

	return res;
}

void send_belkin_command(char command, const char *subcommand, const char *data)
{
	upssend("~00%c%03d%s%s", command, strlen(data) + 3, subcommand, data);
}

/* power down the attached load immediately */
void upsdrv_shutdown(void)
{
	int	res;
	char	temp[SMALLBUF], st[SMALLBUF];

	res = init_communication();
	if (res == -1) {
		printf("Detection failed.  Trying a shutdown command anyway.\n");
		send_belkin_command(CONTROL, POWER_OFF, "1;1");
		exit(-1);
	}

	send_belkin_command(STATUS, STAT_STATUS, "");
	res = get_belkin_reply(temp);
	get_belkin_field(temp, st, sizeof(st), 2);

        if (*st == '1') {
		printf("On battery - sending shutdown comand...\n");
		send_belkin_command(CONTROL,POWER_OFF,"1;1");
		return;
	}

	printf("Power restored - sending shutdown+return command...\n");

	/* turn off the outlet in 5 seconds */
	send_belkin_command(CONTROL, POWER_OFF, "1;5");

	/* ... and turn it back on 10 seconds after that */ 
	send_belkin_command(CONTROL, POWER_ON, "1;15");
}

/* handle the "load.off" with some paranoia */
void do_off(void)
{
	static	time_t lastcmd = 0;
	time_t	now, elapsed;
#ifdef CONFIRM_DANGEROUS_COMMANDS
	time(&now);
	elapsed = now - lastcmd;

	/* reset the timer every call - this means if you call it too      *
	 * early, then you have to wait MINCMDTIME again before sending #2 */
	lastcmd = now;

	if ((elapsed < MINCMDTIME) || (elapsed > MAXCMDTIME)) {

		/* FUTURE: tell the user (via upsd) to try it again */
		/* msgreply(UPSMSG_REPAGAIN); */
		return;
	}
#endif

	upslogx(LOG_INFO, "Sending powerdown command to UPS\n");
        send_belkin_command(CONTROL,POWER_OFF,"1;1");
	usleep(1500000);
        send_belkin_command(CONTROL,POWER_OFF,"1;1");
}

int instcmd(const char *cmdname, const char *extra)
{
	if (!strcasecmp(cmdname, "load.off")) {
		do_off();
		return STAT_INSTCMD_HANDLED;
	}

	if (!strcasecmp(cmdname, "load.on")) {
	        send_belkin_command(CONTROL,POWER_ON,"1;1");
		return STAT_INSTCMD_HANDLED;
	}

	upslogx(LOG_NOTICE, "instcmd: unknown command [%s]", cmdname);
	return STAT_INSTCMD_UNKNOWN;
}

/* install pointers to functions for msg handlers called from msgparse */
void setuphandlers(void)
{
	upsh.new_instcmd = instcmd;
}

int send_string(char *data)
{
	tcflush(upsfd, TCIFLUSH);
	return(write(upsfd, data, strlen(data)));
}

int recvbinary(char *buf, int buflen)
{
	unsigned char	in;
	int	ret, counter = 0, retval = 0;
	struct	sigaction sa;
	sigset_t	bel_sigmask;
 
	sa.sa_handler = timeout;
	sigemptyset(&bel_sigmask);
	sa.sa_mask = bel_sigmask;
	sa.sa_flags = 0;
	sigaction(SIGALRM, &sa, NULL);

	alarm(3);

	while (counter < buflen) {
		ret = read(upsfd, &in, 1);

		if (ret > 0) {
			buf[counter] = in;
			counter++;
			nolongertimeout();	   
		}
		else {
			upslogx(LOG_DEBUG, "error reading from serial device!");
			retval = -1;
			break;
		}
	}
	
	alarm(0);
	signal(SIGALRM, SIG_IGN);

	return retval;
}

void set_serialDTR0RTS1(void)
{
	int dtr_bit = TIOCM_DTR;
	int rts_bit = TIOCM_RTS;

	/* set DTR to low and RTS to high */
	ioctl(upsfd, TIOCMBIC, &dtr_bit);
	ioctl(upsfd, TIOCMBIS, &rts_bit);
}

void upsdrv_banner(void)
{
	printf("Network UPS Tools - Belkin Smart protocol driver %s (%s)\n",
		DRV_VERSION, UPS_VERSION);
}

void upsdrv_help(void)
{
}

void upsdrv_makevartable(void)
{
}

/* prep the serial port */
void upsdrv_initups(void)
{
	open_serial(device_path, B2400);
	
	set_serialDTR0RTS1();

	sleep(1);
  	tcflush(upsfd,TCIOFLUSH);
}

void upsdrv_initinfo(void)
{
	int	res;

	res = init_communication();
	if (res == -1) {
		printf("Unable to detect an Belkin Smart protocol UPS on port %s\n", 
			device_path);
		printf("Check the cabling, port name or model name and try again\n");
		exit(1);
	}

	/* manufacturer ID - hardcoded in this particular module */
	dstate_setinfo("ups.mfr", "BELKIN", 0, 0);

	dstate_setinfo("driver.version.internal", "%s", DRV_VERSION);

	/* see what's out there */
	init_ups_data();

	printf("Detected %s on %s\n", dstate_getinfo("ups.model"),
		device_path);

	setuphandlers();
}

void upsdrv_cleanup(void)
{
}
