/* ferrups.c - model specific routines for PowerWare (formerly Best Power)
 *                     FerrUPS 500VA to 18KVA models with 8.x firmware
 *
 * Copyright (C) 2003  Robert Woodcock <rcw@debian.org>
 *                     Arnaud Quette <aquette@debian.org>
 * Copyright (C) 2002  Jonathan Corbin <jonboy@rocketmonkey.org>
 * Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "main.h"

#define DRIVER_VERSION "0.02"

#define ENDCHAR	'>'
#define IGNCHARS	"f\n\r"

/* Retrievable/settable UPS operational parameters */
#define HWGETBUFSIZ 60
#define TIME "0"
#define DATE "10"
#define LOAD_PERCENTAGE "16"
#define POWER "17"
#define SERIALNUMBER "40"
#define MODELINDEX "41"
#define BATTERY_RUNTIME_LOW "68"

/* Powerware factory default */
char *password = "377";

/* Alarms A-Z and 0-5, in order */
char *alarms[] = {
	/* 1st register, A-H */
	"Low Battery",      "Near Low Battery",
	"High Battery",     "Low Runtime",
	"Low AC Output",    "High AC Output",
	"Output Overload",  "High Ambient Temp",
	/* 2nd register, I-P */
	"Hi Heatsink Temp", "User Test Alarm",
	"Hi Transfmr Temp", "Check Charger",
	"Check Battery",    "Check Inverter",
	"Memory Check",     "Emergency PwrOff",
	/* 3rd register, Q-X */
	"Hi PFM Res Temp",  "Probe Missing",
	"High AC Input",    "Call Service",
	"Unknown Alarm U",  "Unknown Alarm V",
	"Fan Alarm",        "Unknown Alarm X",
	/* 4th register, Y-5 */
	"Unknown Alarm Y",  "Unknown Alarm Z",
	"Unknown Alarm 0",  "Unknown Alarm 1",
	"Unknown Alarm 2",  "Unknown Alarm 3",
	"Unknown Alarm 4",  "Unknown Alarm 5"
};

/* Disclaimer: This password yields a higher access to the ups than is
 * normally available.  This driver only uses this password to change the date
 * and time (which, according to the spec, don't need this password to change,
 * but at least with firmware version 8.01, the spec is wrong and you need the
 * password).  Use this password at your own risk.  You could very easily break
 * the ups with this.
 *
 * This password can be changed, so if you have changed it on the ups, you will
 * need to change it in the driver as well.
 *
 */

/* This function clears the buffer on the ups.  It takes several seconds to
 * execute, so it should only be called when necessary
 */
void clearUpsBuffer() {
	char data[1];

	/* Turn off timeout failure logging */
	flag_timeoutfailure = -1;
	
	/* clear the happy buffer */
	while (upsrecvchars (data, 1) != -1);
	
	/* Turn timeout failure logging back on */
	flag_timeoutfailure = 0;
}

void hw_set (char *paramNum, const char *newval) {

	/* send a carriage return to clear anything extraneous from the ups keyboard
	 * buffer
	 */
	upssend ("\r");

	/* send the user password to the ups */
	upssend("pw ");
	upssend(password);
	upssend("\r");

	/* change the parameter */
	upssend("pr ");
	upssend(paramNum);
	upssend(" ");
	upssend(newval);
	upssend("\r");

	/* logout */
	upssend("pw\r");
	
	clearUpsBuffer();
}

/* data should be long enough to hold whatever data will be returned, a
 * maximum of datalen chars.
 *
 * This function currently does not handle parameters where the desired token
 * is not the last token in the first line. It has limited support for
 * parameters where the desired data is on another line, via special casing.
 *
 * The ups buffer should be clear before this function is called.  This can be
 * accomplished by calling clearUpsBuffer() before calling this function.  This
 * function will leave the buffer cleared, so clearUpsBuffer does not have to be
 * called between calls of this function.
 */

char *hw_get (char *paramNum, char *data, size_t datalen) {
	int ret, offset, tokensize;
	char *temp, *end, *begin;

	if ((temp = calloc(datalen, 1)) == NULL) return data;
	
	/* ask the ups for the parameter's value */
	upssend("p ");
	upssend(paramNum);
	upssend("\r");

	/* get the value from the ups */
	ret = upsrecv(temp, datalen, ENDCHAR, "");
	if (ret < 0) {
		/* Nothing checks the return value from hw_get currently
		 * so just dying won't help. Return an empty string. */
		data[0] = 0;
		free(temp);
		return data;
	}
	
	/* This newline is the one we just sent */
	if ((end = strchr(temp, '\n')) == NULL) {
		free(temp);
		return data;
	}
	offset = end - temp + 1;

	/* Find end of first real newline */
	if ((end = strchr(&temp[offset], '\n')) == NULL) {
		free(temp);
		return data;
	}

	/* Check the second line instead if needed */
	if ((paramNum == MODELINDEX) || (paramNum == SERIALNUMBER)) {
		offset = end - temp + 1;
		if ((end = strchr(&temp[offset], '\n')) == NULL) {
			free(temp);
			return data;
		}
	}
	
	/* And erase everything after it, including trailing \r\n */
	offset = end - temp;
	memset(&temp[offset], 0, datalen - offset);

	/* Find first space back from that, or \n for serial number */
	if (paramNum != SERIALNUMBER) {
		if ((begin = strrchr(temp, ' ')) == NULL) {
			free(temp);
			return data;
		}
	} else {
		if ((begin = strrchr(temp, '\n')) == NULL) {
			free(temp);
			return data;
		}
	}	

	begin++;

	tokensize = datalen;
	if (strlen(begin) < datalen) {
		tokensize = strlen(begin);
	}

	/* Copy token */
	strlcpy(data, begin, tokensize);
	free(temp);
	return data;
}

/* atoi() without the freebie octal conversion */
int bcd2i (const char *bcdstring, const int bcdlen)
{
	int i, digit, total = 0, factor = 1;
	for (i = 1; i < bcdlen; i++)
		factor *= 10;
	for (i = 0; i < bcdlen; i++) {
		digit = bcdstring[i] - '0';
		if (digit > 9) {
			digit = 0;
		}
		total += digit * factor;
		factor /= 10;
	}
	return total;
}

int setvar (const char *varname, const char *val)
{
	char newval[HWGETBUFSIZ]; /* of guaranteed length */

	upslog(LOG_INFO, "test10");
	/* TODO: look this up from a table? */
	if (!strcasecmp(varname, "ups.time")) {
		upslog(LOG_INFO, "test11");
		hw_set(TIME, val);
		dstate_setinfo("ups.time", "%s", hw_get(TIME, newval, HWGETBUFSIZ));
		return STAT_SET_HANDLED;
	}

	if (!strcasecmp(varname, "ups.date")) {
		upslog(LOG_INFO, "test12");
		hw_set(DATE, val);
		dstate_setinfo("ups.date", "%s", hw_get(DATE, newval, HWGETBUFSIZ));
		return STAT_SET_HANDLED;
	}
	
	upslogx(LOG_ERR, "setvar: unknown type 0x%04x\n", varname);
	upslog(LOG_INFO, "test13");
	return STAT_SET_UNKNOWN;
}

void upsdrv_initinfo(void)
{
	char data[HWGETBUFSIZ];
	int tempint;
	
	dstate_setinfo("ups.mfr", "Powerware");
	dstate_setinfo("ups.model", "FERRUPS %s", hw_get(MODELINDEX, data, HWGETBUFSIZ));
	dstate_setinfo("ups.serial", "%s", hw_get(SERIALNUMBER, data, HWGETBUFSIZ));

	dstate_setinfo("ups.load", "%s", hw_get(LOAD_PERCENTAGE, data, HWGETBUFSIZ));

#if 0	/* undefined */
	/* output.voltamps and output.watts is ad-hoc */
	dstate_setinfo("output.voltamps", "");
	dstate_setinfo("output.watts", "%s", hw_get(POWER, data, HWGETBUFSIZ));
#endif

	hw_get(BATTERY_RUNTIME_LOW, data, HWGETBUFSIZ);
	tempint = bcd2i(data, strlen(data) - 1) * 60; /* "5m" -> 5 -> 300 */
	dstate_setinfo("battery.runtime.low", "%d", tempint);
	dstate_setinfo("ups.time", "");
	dstate_setflags("ups.time", ST_FLAG_RW | ST_FLAG_STRING);
	dstate_setaux("ups.time", 8);
	dstate_setinfo("ups.date", "");
	dstate_setflags("ups.date", ST_FLAG_RW | ST_FLAG_STRING);
	dstate_setaux("ups.date", 8);

	/* Install handler for changing variables. */
	upsh.new_setvar = setvar;
}

void upsdrv_updateinfo(void)
{
	/* int flags; */
	char data[HWGETBUFSIZ];
	char temp[86];
	unsigned char bytes[40]; /* ints of parsed hex temp[] data */
	char byte;
	int sum = 0;
	/* total size of alarms[], plus delimiting, plus null termination */
	char alarmreport[517];
	int alarmregisters = 0;
	char time[9];
	int i, offset, tempint;
	
	clearUpsBuffer();

	upssend("f\r");

	if (upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS) == -1) {
		dstate_datastale();
		return;
	}

	/* The value in chars 78 and 79 will cause a no-carry
	 * twos-compliment addition of all of the bytes to equal 0.
	 * This is contradictory to PowerWare's documentation in TIP 503
	 * which states that this value is "Equal to the 2's compliment
	 * hex sum, without carry, of the preceding 39 two-digit
	 * hexadecimal numbers." */

	/* Convert from hexadecimal to a raw array */
	for (i = 0; i < 40; i++) {
		byte = temp[i * 2] - '0';
		if (byte > 9) byte -= 7; /* A through F */
		bytes[i] = byte << 4;
		byte = temp[i * 2 + 1] - '0';
		if (byte > 9) byte -= 7;
		bytes[i] += byte;
	}

	/* Add everything including the checksum value */
	for (i = 0; i < 40; i++) {
		sum += bytes[i];
	}

	/* Make sure the last 8 bits are 0 */
	if ((sum % 256) != 0) {
		dstate_datastale();
		return;
	}

	/* Parse the format line */

	/* Time */
	offset = 0;
	for (i = 0; i < 8; i++) {
		if (i == 2 || i == 5) {
			time[i] = ':';
			offset++;
		} else {
			time[i] = temp[4 + i - offset];
		}
	}

	time[8] = '\0';
	dstate_setinfo("ups.time", "%s", time);
	
	status_init();

	/* Check if ups is running on the battery */
	if (temp[17] == '1')
		status_set("OB");
	else
		status_set("OL");

	/* Check the alarm registers - chars 20-21, 22-23, 66-67, 68-69 */
	alarmregisters = bytes[10];
	alarmregisters |= bytes[11] << 8;
	alarmregisters |= bytes[33] << 16;
	alarmregisters |= bytes[34] << 24;

	memset(&alarmreport, 0, sizeof(alarmreport));
	
	for (i = 0; i < sizeof(alarmregisters) * 8; i++) {
		/* check if bit is set */
		if (alarmregisters & (1 << i)) {
			/* yes - add to alarm string */
			strcat(alarmreport, alarms[i]);
			strcat(alarmreport, ", ");
		}
	}
	/* eat trailing ", " */
	if (strlen(alarmreport) > 2) {
		alarmreport[strlen(alarmreport) - 1] = 0;
		alarmreport[strlen(alarmreport) - 1] = 0;
	}

#if 0	/* undefined */
	/* ups.alarms is ad-hoc */
	dstate_setinfo("ups.alarms", alarmreport);
#endif

	/* Some special cases */
	
	/* H = High Ambient Temp */
	if (alarmregisters & (1 << ('H' - 'A'))) {
		dstate_setinfo("ambient.temperature.alarm", "1");
	} else {
		dstate_setinfo("ambient.temperature.alarm", "0");
	}
	/* G = Output Overload */
	if (alarmregisters & (1 << ('G' - 'A'))) {
		status_set("OVER");
	}
	/* M = Check Battery */
	if (alarmregisters & (1 << ('M' - 'A'))) {
		status_set("RB");
	}
	/* S = High AC Input */
	if (alarmregisters & (1 << ('S' - 'A'))) {
		status_set("TRIM");
	}
	/* A = Low Battery, D = Low Runtime */
	if ((alarmregisters & (1 << ('A' - 'A'))) ||
	    (alarmregisters & (1 << ('D' - 'A')))) {
		status_set("LB");
	}

	status_commit();

	dstate_setinfo("input.voltage", "%d", bcd2i(&temp[24], 4));
	dstate_setinfo("output.voltage", "%d", bcd2i(&temp[28], 4));
	dstate_setinfo("output.current", "%.1f", (float)bcd2i(&temp[36], 4) / 10);

#if 0	/* undefined */
	dstate_setinfo("output.voltamps", "%d", bcd2i(&temp[40], 6));
#endif

	dstate_setinfo("battery.current", "%d", bcd2i(&temp[46], 4));
	dstate_setinfo("battery.voltage", "%.1f", (float)bcd2i(&temp[50], 4) / 10);
	dstate_setinfo("input.frequency", "%.2f", (float)bcd2i(&temp[54], 4) / 100);
	dstate_setinfo("battery.runtime", "%d", bcd2i(&temp[58], 4) * 60);
	dstate_setinfo("ambient.temperature", "%d", bcd2i(&temp[62], 4));
	dstate_setinfo("ups.firmware", "v%.2f", (float)bcd2i(&temp[74], 4) / 100);

	clearUpsBuffer();

	/* Time to get some more happy info from the ups */

	/* such as load percentage */
	dstate_setinfo("ups.load", "%s", hw_get(LOAD_PERCENTAGE, data, HWGETBUFSIZ));

#if 0	/* undefined */
	/* power (watts) */
	dstate_setinfo("output.watts", "%s", hw_get(POWER, data, HWGETBUFSIZ));
#endif

	/* the all-important date */
	dstate_setinfo("ups.date", "%s", hw_get(DATE, data, HWGETBUFSIZ));

	/* And how much longer we have left when we hit "low battery" */
	hw_get(BATTERY_RUNTIME_LOW, data, HWGETBUFSIZ);
	tempint = bcd2i(data, strlen(data)-1) * 60; /* "5m" -> 5 -> 300 */
	dstate_setinfo("battery.runtime.low", "%d", tempint);

	/* we can be pretty sure of this now */
	dstate_dataok();
}

void upsdrv_shutdown(void)
{
	/* The "off" command needs a minimum of 5 seconds for a time value.
	 * Autorestart after shutdown so that things come back up when power
	 * returns.
	 */
	upssend("o 5 a\r");
}

void upsdrv_help(void)
{
}

/* list flags and values that you want to receive via -x */
void upsdrv_makevartable(void)
{
	/* char msg[64]; */
	/* TODO: allow password change as command line option => vartable */
	/* sprintf(msg, "Set user password (default=%s).", password);
	   addvar (VAR_VALUE, "password", msg); */
}

void upsdrv_banner(void)
{
	printf("Network UPS Tools - Ferrups 8.01-8.07 UPS driver %s (%s)\n",
	       DRIVER_VERSION, UPS_VERSION);
}


static void sync_serial(void)
{
	int ret;
	char buffer[10];

	upssend("\r");
	ret = upsrecv(buffer, sizeof(buffer), ENDCHAR, IGNCHARS);
	if (ret < 0) {
		return;
	}

	while (buffer[0] != '=') {
		upssend("\r");
		ret = upsrecv(buffer, sizeof(buffer), ENDCHAR, IGNCHARS);
		if (ret < 0) {
			return;
		}
	}
}

/* Begin code stolen from bestups.c */
static void setup_serial(void)
{  
	struct   termios  tio;
			     
	if (tcgetattr(upsfd, &tio) == -1)
		fatal("tcgetattr");
				     
	tio.c_iflag = IXON | IXOFF;
	tio.c_oflag = 0;
	tio.c_cflag = (CS8 | CREAD | HUPCL | CLOCAL);
	tio.c_lflag = 0;
	tio.c_cc[VMIN] = 1;
	tio.c_cc[VTIME] = 0;

#ifdef HAVE_CFSETISPEED
	cfsetispeed(&tio, B1200); /* baud change here */
	cfsetospeed(&tio, B1200);
#else
#error This system lacks cfsetispeed() and has no other means to set the speed
#endif

	if (tcsetattr(upsfd, TCSANOW, &tio) == -1)
		fatal("tcsetattr");
/* end code stolen from bestups.c */

	sync_serial();
}

void upsdrv_initups(void)
{
	/* TODO: not broken anymore, but not yet stable */
	experimental_driver = 1;

	open_serial(device_path, B1200);
	setup_serial();

	/* TODO: get user password */
	/* if (getval ("password"))
	   password = atoi (getval ("password")); */
}

void upsdrv_cleanup(void)
{
}
