/*
   newapc.c - model specific routines for APC smart protocol units

	$Id: newapc.c,v 1.2 2002/03/04 13:31:32 cvs Exp $

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>
             (C) 2000  Nigel Metheringham <Nigel.Metheringham@Intechnology.co.uk>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#define APC_DRIVER_VERSION	"1.99.1a"

#include "main.h"
#include "newapc.h"

#define ALT_CABLE_1 "940-0095B"

static struct apc_vartab_t *vartab_lookup_char(char cmdchar)
{
	int	i;

	for (i = 0; apc_vartab[i].name != NULL; i++)
		if (apc_vartab[i].cmd == cmdchar)
			return &apc_vartab[i];

	return NULL;
}

static struct apc_vartab_t *vartab_lookup_name(const char *var)
{
	int	i;

	for (i = 0; apc_vartab[i].name != NULL; i++)
		if (!strcasecmp(apc_vartab[i].name, var))
			return &apc_vartab[i];

	return NULL;
}

/* FUTURE: change to use function pointers */

/* convert APC formatting to NUT formatting */
static char *convert_data(struct apc_vartab_t *cmd_entry, char *upsval)
{
	static	char tmp[128];
	int	tval;

	switch(cmd_entry->flags & APC_FORMATMASK) {
		case APC_F_PERCENT:
		case APC_F_VOLT:
		case APC_F_AMP:
		case APC_F_CELSIUS:
		case APC_F_HEX:
		case APC_F_DEC:
		case APC_F_SECONDS:
		case APC_F_LEAVE:

			/* no conversion for any of these */
			return upsval;

		case APC_F_HOURS:
			/* convert to seconds */

			tval = 60 * 60 * strtol(upsval, NULL, 10);

			snprintf(tmp, sizeof(tmp), "%d", tval);
			return tmp;

		case APC_F_MINUTES:
			/* Convert to seconds - NUT standard time measurement */
			tval = 60 * strtol(upsval, NULL, 10);
			/* Ignore errors - Theres not much we can do */
			snprintf(tmp, sizeof(tmp), "%d", tval);
			return tmp;

		default:
			upslogx(LOG_NOTICE, "Unable to handle conversion of %s",
				cmd_entry->name);
			return upsval;
	}

	/* NOTREACHED */
	return upsval;
}

static void poll_data(struct apc_vartab_t *vt)
{
	int	ret;
	char	tmp[SMALLBUF];

	if ((vt->flags & APC_PRESENT) == 0)
		return;

	upsdebugx(4, "poll_data: %s", vt->name);

	upssendchar(vt->cmd);

	ret = upsrecv(tmp, sizeof(tmp), ENDCHAR, IGNCHARS);

	if (ret < 1) {			/* comm failure */
		dstate_datastale();
		return;
	}

	/* no longer supported by the hardware somehow */
	if (!strcmp(tmp, "NA")) {
		dstate_delinfo(vt->name);
		return;
	}

	dstate_setinfo(vt->name, "%s", convert_data(vt, tmp));
	dstate_dataok();
}

/* check for support or just update a named variable */
static int query_ups(const char *var, int first)
{
	int	ret;
	char	temp[256], *ptr;
	struct	apc_vartab_t *vt;

	vt = vartab_lookup_name(var);

	if (!vt) {
		upsdebugx(1, "query_ups: unknown variable %s", var);
		return 0;
	}

	/* already known to not be supported? */
	if (vt->flags & APC_IGNORE)
		return 0;

	upsflushin(0, nut_debug_level, IGNCHARS);

	upssendchar(vt->cmd);

	if (first)
		flag_timeoutfailure = -1;

	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if ((ret < 1) || (!strcmp(temp, "NA"))) {	/* not supported */
		vt->flags |= APC_IGNORE;
		return 0;
	}

	ptr = convert_data(vt, temp);
	dstate_setinfo(vt->name, "%s", ptr);

	return 1;	/* success */
}

static void do_capabilities(void)
{
	const	char	*ptr, *entptr;
	char	upsloc, temp[512], cmd, loc, etmp[16], *endtemp;
	int	nument, entlen, i, matrix, ret;
	struct	apc_vartab_t *vt;

	upsdebugx(1, "APC - About to get capabilities string");
	/* If we can do caps, then we need the Firmware revision which has
	   the locale descriptor as the last character (ugh)
	*/
	ptr = dstate_getinfo("ups.firmware");
	if (ptr)
		upsloc = ptr[strlen(ptr) - 1];
	else
		upsloc = 0;

	/* get capability string */
	upssendchar(APC_CAPABILITY);		/* ^Z */

	ret = upsrecv(temp, sizeof(temp), ENDCHAR, MINIGNCHARS);

	if ((ret < 1) || (!strcmp(temp, "NA"))) {

		/* Early Smart-UPS, not as smart as later ones */
		/* This should never happen since we only call
		   this if the REQ_CAPABILITIES command is supported
		*/
		upslogx(LOG_ERR, "ERROR: APC cannot do capabilites but said it could!");
		return;
	}

	/* recv always puts a \0 at the end, so this is safe */
	/* however it assumes a zero byte cannot be embedded */
	endtemp = &temp[0] + strlen(temp);

	if (temp[0] != '#') {
		printf("Unrecognized capability start char %c\n", temp[0]);
		printf("Please report this error [%s]\n", temp);
		upslogx(LOG_ERR, "ERROR: unknown capability start char %c!", 
			temp[0]);

		return;
	}

	if (temp[1] == '#') {		/* Matrix-UPS */
		matrix = 1;
		ptr = &temp[0];
	}
	else {
		ptr = &temp[1];
		matrix = 0;
	}

	/* command char, location, # of entries, entry length */

	while (ptr[0] != '\0') {
		if (matrix)
			ptr += 2;	/* jump over repeating ## */

		/* check for idiocy */
		if (ptr >= endtemp) {
			printf("Capability string has overflowed\n");
			printf("Please report this error\n");
			fatalx("ERROR: capability overflow!");
		}

		cmd = ptr[0];
		loc = ptr[1];
		nument = ptr[2] - 48;
		entlen = ptr[3] - 48;
		entptr = &ptr[4];

		vt = vartab_lookup_char(cmd);

		/* mark this as writable */
		if (vt && ((loc == upsloc) || (loc == '4'))) {
			upsdebugx(1, "Supported capability: %02x (%c) - %s", 
				cmd, loc, vt->name);

			dstate_setflags(vt->name, ST_FLAG_RW);

			/* make sure setvar knows what this is */
			vt->flags |= APC_RW | APC_ENUM;
		}

		for (i = 0; i < nument; i++) {
			snprintf(etmp, entlen + 1, "%s", entptr);

			if (vt && ((loc == upsloc) || (loc == '4')))
				dstate_addenum(vt->name, convert_data(vt, etmp));

			entptr += entlen;
		}

		ptr = entptr;
	}
}

static void update_status(void)
{
	int	ret, tval;
	char	buf[SMALLBUF];

	upsflushin(0, nut_debug_level, IGNCHARS);

	upssendchar(APC_STATUS);
	ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);

	if ((ret < 1) || (!strcmp(buf, "NA"))) {
		dstate_datastale();
		return;
	}

	tval = strtol(buf, 0, 16);
	tval &= 0xff;

	status_init();
	if (tval & 1)
		status_set("CAL");		/* calibration */
	if (tval & 2)
		status_set("TRIM");		/* SmartTrim */
	if (tval & 4)
		status_set("BOOST");		/* SmartBoost */
	if (tval & 8)
		status_set("OL");		/* on line */
	if (tval & 16)
		status_set("OB");		/* on battery */
	if (tval & 32)
		status_set("OVER");		/* overload */
	if (tval & 64)
		status_set("LB");		/* low battery */
	if (tval & 128)
		status_set("RB");		/* replace batt */

	if (!strcmp(buf, "00"))
		status_set("OFF");

	status_commit();
	dstate_dataok();
}

static void oldapcsetup(void)
{
	int	ret = 0;

	/* really old models ignore REQ_MODEL, so find them first */
	ret = query_ups("ups.model", 1);

	if (ret != 1) {
		/* force the model name */
		dstate_setinfo("ups.model", "Smart-UPS");
	}

	/* see if this might be an old Matrix-UPS instead */
	if (query_ups("output.current", 1))
		dstate_setinfo("ups.model", "Matrix-UPS");

	query_ups("ups.serial", 1);
	query_ups("input.voltage", 1); /* This one may fail... no problem */

	update_status();

	/* If we have come down this path then we dont do capabilities and
	   other shiny features
	*/
}

static void protocol_verify(unsigned char cmd)
{
	int	i, found;

	/* we might not care about this one */
	if (strchr(CMD_IGN_CHARS, cmd))
		return;

	/* see if it's a variable */
	for (i = 0; apc_vartab[i].name != NULL; i++) {

		/* 1:1 here, so the first match is the only match */

		if (apc_vartab[i].cmd == cmd) {
			upsdebugx(3, "UPS supports variable [%s]",
				apc_vartab[i].name);

			/* load initial data */
			apc_vartab[i].flags |= APC_PRESENT;
			poll_data(&apc_vartab[i]);

			/* handle special data for our two strings */
			if (apc_vartab[i].flags & APC_STRING) {
				dstate_setflags(apc_vartab[i].name, 
					ST_FLAG_RW | ST_FLAG_STRING);
				dstate_setaux(apc_vartab[i].name, APC_STRLEN);

				apc_vartab[i].flags |= APC_RW;
			}

			return;
		}
	}

	/* check the command list */

	/* some cmdchars map onto multiple commands (start and stop) */

	found = 0;

	for (i = 0; apc_cmdtab[i].name != NULL; i++) {
		if (apc_cmdtab[i].cmd == cmd) {
			upsdebugx(2, "UPS supports command [%s]",
				apc_cmdtab[i].name);

			dstate_addcmd(apc_cmdtab[i].name);

			apc_cmdtab[i].flags |= APC_PRESENT;
			found = 1;
		}
	}

	if (found)
		return;

	if (isprint(cmd))
		upsdebugx(1, "protocol_verify: 0x%02x [%c] unrecognized", 
			cmd, cmd);
	else
		upsdebugx(1, "protocol_verify: 0x%02x unrecognized", cmd);
}

static void getbaseinfo(void)
{
	int	i, ret = 0;
	char 	*alrts, *cmds, temp[512];

	upsdebugx(1, "APC - Attempting to find command set");
	/* Initially we ask the UPS what commands it takes
	   If this fails we are going to need an alternate
	   strategy - we can deal with that if it happens
	*/

	upssendchar(APC_CMDSET);

	/* disable timeout complaints temporarily */
	flag_timeoutfailure = -1;
	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);
	flag_timeoutfailure = 0;

	if ((ret < 1) || (!strcmp(temp, "NA"))) {

		/* We have an old dumb UPS - go to specific code for old stuff */
		oldapcsetup();
		return;
	}

	upsdebugx(1, "APC - Parsing out command set");
	/* We have the version.alert.cmdchars string
	   NB the alert chars are normally in IGNCHARS
	   so will have been pretty much edited out.
	   You will need to change the upsrecv above if
	   you want to check those out too....
	*/
 	alrts = strchr(temp, '.');
	if (alrts == NULL) {
		printf("Unable to split APC version string\n");
		printf("Bailing out\n");
		exit(1);
	}
	*alrts++ = 0;

	cmds = strchr(alrts, '.');
	if (cmds == NULL) {
		printf("Unable to find APC command string\n");
		printf("Bailing out\n");
		exit(1);
	}
	*cmds++ = 0;

	for (i = 0; i < strlen(cmds); i++)
		protocol_verify(cmds[i]);

	/* if capabilities are supported, add them here */
	if (strchr(cmds, APC_CAPABILITY))
		do_capabilities();

	upsdebugx(1, "APC - UPS capabilities determined");
}

/* TODO: roll this into upscommon ala bestups */
static void sendstring(char * string, int len, int delay)
{
	int	i;

	upsflushin(0, nut_debug_level, IGNCHARS);
	for (i=0; (i<len); i++) {
	        upssendchar(string[i]);
		usleep(delay);
	}
}

/* check for calibration status and either start or stop */
static int do_cal(int start)
{
	char	temp[256];
	int	tval, ret;

	upssendchar(APC_STATUS);
	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

	/* if we can't check the current calibration status, bail out */
	if ((ret < 1) || (!strcmp(temp, "NA")))
		return STAT_INSTCMD_HANDLED;		/* FUTURE: failure */

	tval = strtol(temp, 0, 16);

	if (tval & 1) {		/* calibration currently happening */
		if (start == 1) {
			/* requested start while calibration still running */
			upslogx(LOG_INFO, "Runtime calibration already in progress");
			return STAT_INSTCMD_HANDLED;	/* FUTURE: failure */
		}

		/* stop requested */

		upslogx(LOG_INFO, "Stopping runtime calibration");
		upssendchar(APC_CMD_CALTOGGLE);

		ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

		if ((ret < 1) || (!strcmp(temp, "NA")) || (!strcmp(temp, "NO"))) {
			upslogx(LOG_WARNING, "Stop calibration failed", temp);
			return STAT_INSTCMD_HANDLED;	/* FUTURE: failure */
		}

		return STAT_INSTCMD_HANDLED;	/* FUTURE: success */
	}

	/* calibration not happening */

	if (start == 0) {		/* stop requested */
		upslogx(LOG_INFO, "Runtime calibration not occurring");
		return STAT_INSTCMD_HANDLED;		/* FUTURE: failure */
	}

	upslogx(LOG_INFO, "Starting runtime calibration");
	upssendchar(APC_CMD_CALTOGGLE);

	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if ((ret < 1) || (!strcmp(temp, "NA")) || (!strcmp(temp, "NO"))) {
		upslogx(LOG_WARNING, "Start calibration failed", temp);
		return STAT_INSTCMD_HANDLED;	/* FUTURE: failure */
	}

	return STAT_INSTCMD_HANDLED;			/* FUTURE: success */
}

/* get the UPS talking to us in smart mode */
static int smartmode(void)
{
	int	ret, tries;
	char	temp[256];

	for (tries = 0; tries < 5; tries++) {

		upssendchar(APC_GOSMART);
		ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

		if (ret > 0)
			if (!strcmp(temp, "SM"))
				return 1;	/* success */

		sleep(1);	/* wait before trying again */

		/* it failed, so try to bail out of menus on newer units */

		upssendchar(27);	/* ESC */

		/* eat the response (might be NA, might be something else) */
		upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);
	}

	return 0;	/* failure */
}

/* power down the attached load immediately */
void upsdrv_shutdown(void)
{
	char	temp[32];
	int	ret, tval, sdtype = 0;

	if (!smartmode())
		printf("Detection failed.  Trying a shutdown command anyway.\n");

	/* check the line status */

	upssendchar(APC_STATUS);
	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if (ret < 1) {
		printf("Status read failed!  Assuming you're on battery...\n");
		tval = 64 | 16;		/* LB + OB */

	} else {
		tval = strtol(temp, 0, 16);
	}

	if (testvar("sdtype"))
		sdtype = atoi(getval("sdtype"));

	switch (sdtype) {
	case 3:		/* shutdown with grace period */
		printf("Sending delayed power off command to UPS\n");

		upssendchar(APC_CMD_SHUTDOWN);
		usleep(1500000);
		upssendchar(APC_CMD_SHUTDOWN);

		break;

	case 2:		/* instant shutdown */
		printf("Sending power off command to UPS\n");

		upssendchar(APC_CMD_OFF);
		usleep(1500000);
		upssendchar(APC_CMD_OFF);

		break;

	case 1:

		/* Send a combined set of shutdown commands which can work better */
		/* if the UPS gets power during shutdown process */
		/* Specifically it sends both the soft shutdown 'S' */
		/* and the powerdown after grace period - '@000' commands */
		printf("UPS - currently %s - sending shutdown/powerdown\n",
			(tval & 8) ? "on-line" : "on battery");
		sendstring("S@000", 5, 50000);
		break;

	default:

		/* @000 - shutdown after 'p' grace period             */
		/*      - returns after 000 minutes (i.e. right away) */

		/* S    - shutdown after 'p' grace period, only on battery */
		/*        returns after 'e' charge % plus 'r' seconds      */

		if (tval & 8) {                 /* on line */
			printf("On line, sending shutdown+return command...\n");
			sendstring("@000", 4, 50000);
		}
		else {
			printf("On battery, sending normal shutdown command...\n");
			upssendchar(APC_CMD_SOFTDOWN);
		}
	}
}

/* 940-0095B support: set DTR, lower RTS */
static void init_serial_0095B()
{
	int	dtr_bit = TIOCM_DTR;
	int	rts_bit = TIOCM_RTS;

	ioctl(upsfd, TIOCMBIS, &dtr_bit);
	ioctl(upsfd, TIOCMBIC, &rts_bit);
}

/* normal idle loop - keep up with the current state of the UPS */
/* If updateall is non-zero query all known UPS variables */
static void updateinfo(int updateall)
{
	int	i;

	/* try to wake up a dead ups once in awhile */
	if (flag_timeoutfailure == 1) {
		if (!smartmode()) {
			upslogx(LOG_ERR, "Communication with UPS lost");
			return;
		}
	}

	for (i = 0; apc_vartab[i].name != NULL; i++) {

		/* "all" mode - do everything regardless */
		if (updateall) {
			poll_data(&apc_vartab[i]);
			continue;
		}

		/* normal mode: check the reduced set */
		if (apc_vartab[i].flags & APC_POLL)
			poll_data(&apc_vartab[i]);
	}
}

static int setvar_enum(struct apc_vartab_t *vt, const char *val)
{
	int	i, ret;
	char	orig[256], temp[256], *ptr;

	upssendchar(vt->cmd);

	ret = upsrecv(orig, sizeof(orig), ENDCHAR, IGNCHARS);

	if ((ret < 1) || (!strcmp(orig, "NA")))
		return STAT_SET_HANDLED;	/* FUTURE: failed */

	ptr = convert_data(vt, orig);

	/* suppress redundant changes - easier on the eeprom */
	if (!strcmp(ptr, val)) {
		upslogx(LOG_INFO, "Ignoring enum SET %s='%s' (unchanged value)",
			vt->name, val);

		return STAT_SET_HANDLED;	/* FUTURE: no change */
	}

	for (i = 0; i < 6; i++) {
		upssendchar(APC_NEXTVAL);

		/* this should return either OK (if rotated) or NO (if not) */
		ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

		if ((ret < 1) || (!strcmp(temp, "NA")))
			return STAT_SET_HANDLED;	/* FUTURE: failed */

		/* sanity checks */
		if (!strcmp(temp, "NO"))
			return STAT_SET_HANDLED;	/* FUTURE: failed */
		if (strcmp(temp, "OK") != 0)
			return STAT_SET_HANDLED;	/* FUTURE: failed */

		/* see what it rotated onto */
		upssendchar(vt->cmd);
		ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

		if ((ret < 1) || (!strcmp(temp, "NA")))
			return STAT_SET_HANDLED;	/* FUTURE: failed */

		ptr = convert_data(vt, temp);

		upsdebugx(1, "Rotate value: got [%s], want [%s]",
			ptr, val);

		if (!strcmp(ptr, val)) {	/* got it */
			upslogx(LOG_INFO, "SET %s='%s'", vt->name, val);

			/* refresh data from the hardware */
			query_ups(vt->name, 0);

			return STAT_SET_HANDLED;	/* FUTURE: success */
		}

		/* check for wraparound */
		if (!strcmp(ptr, orig)) {
			upslogx(LOG_ERR, "setvar: variable %s wrapped",
				vt->name);

			return STAT_SET_HANDLED;	/* FUTURE: failed */
		}			
	}

	upslogx(LOG_ERR, "setvar: gave up after 6 tries for %s",
		vt->name);

	/* refresh data from the hardware */
	query_ups(vt->name, 0);

	return STAT_SET_HANDLED;
}

static int setvar_string(struct apc_vartab_t *vt, const char *val)
{
	int	i, ret;
	char	temp[256];

	upsflushin(0, nut_debug_level, IGNCHARS);

	upssendchar(vt->cmd);

	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if ((ret < 1) || (!strcmp(temp, "NA")))
		return STAT_SET_HANDLED;	/* FUTURE: failed */

	/* suppress redundant changes - easier on the eeprom */
	if (!strcmp(temp, val)) {
		upslogx(LOG_INFO, "Ignoring string SET %s='%s' (unchanged value)",
			vt->name, val);

		return STAT_SET_HANDLED;	/* FUTURE: no change */
	}

	upssendchar(APC_NEXTVAL);
	usleep(50000);

	for (i = 0; i < strlen(val); i++) {
		upssendchar(val[i]);
		usleep(50000);
	}

	/* pad to 8 chars with CRs */
	for (i = strlen(val); i < APC_STRLEN; i++) {
		upssendchar(13);
		usleep(50000);
	}

	ret = upsrecv(temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if (ret < 1) {
		upslogx(LOG_ERR, "setvar_string: short final read");
		return STAT_SET_HANDLED;	/* FUTURE: failed */
	}

	if (!strcmp(temp, "NO")) {
		upslogx(LOG_ERR, "setvar_string: got NO at final read");
		return STAT_SET_HANDLED;	/* FUTURE: failed */
	}

	/* refresh data from the hardware */
	query_ups(vt->name, 0);

	upslogx(LOG_INFO, "SET %s='%s'", vt->name, val);

	return STAT_SET_HANDLED;	/* FUTURE: failed */
}

int setvar(const char *varname, const char *val)
{
	struct	apc_vartab_t	*vt;

	vt = vartab_lookup_name(varname);

	if (!vt)
		return STAT_SET_UNKNOWN;

	if ((vt->flags & APC_RW) == 0) {
		upslogx(LOG_WARNING, "setvar: [%s] is not writable", varname);
		return STAT_SET_UNKNOWN;
	}

	if (vt->flags & APC_ENUM)
		return setvar_enum(vt, val);

	if (vt->flags & APC_STRING)
		return setvar_string(vt, val);

	upslogx(LOG_WARNING, "setvar: Unknown type for [%s]", varname);
	return STAT_SET_UNKNOWN;
}

/* actually send the instcmd's char to the ups */
static int do_cmd(struct apc_cmdtab_t *ct)
{
	int	ret;
	char	buf[SMALLBUF];

	upssendchar(ct->cmd);

	/* some commands have to be sent twice with a 1.5s gap */
	if (ct->flags & APC_REPEAT) {
		usleep(1500000);
		upssendchar(ct->cmd);
	}

	ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);

	if (ret < 1)
		return STAT_INSTCMD_HANDLED;		/* FUTURE: failed */

	if (strcmp(buf, "OK") != 0) {
		upslogx(LOG_WARNING, "Got [%s] after command [%s]",
			buf, ct->name);

		return STAT_INSTCMD_HANDLED;		/* FUTURE: failed */
	}

	upslogx(LOG_INFO, "Command: %s", ct->name);
	return STAT_INSTCMD_HANDLED;			/* FUTURE: success */
}

/* some commands must be repeated in a window to execute */
static int instcmd_chktime(struct apc_cmdtab_t *ct)
{
	double	elapsed;
	time_t	now;
	static	time_t	last = 0;

	time(&now);

	elapsed = difftime(now, last);
	last = now;

	/* you have to hit this in a small window or it fails */
	if ((elapsed < MINCMDTIME) || (elapsed > MAXCMDTIME)) {
		upsdebugx(1, "instcmd_chktime: outside window for %s (%2.0f)",
				ct->name, elapsed);
		return STAT_INSTCMD_HANDLED;		/* FUTURE: again */
	}

	return do_cmd(ct);
}

int instcmd(const char *cmdname, const char *extra)
{
	int	i;
	struct	apc_cmdtab_t	*ct;

	ct = NULL;

	for (i = 0; apc_cmdtab[i].name != NULL; i++)
		if (!strcasecmp(apc_cmdtab[i].name, cmdname))
			ct = &apc_cmdtab[i];

	if (!ct) {
		upslogx(LOG_WARNING, "instcmd: unknown command [%s]", cmdname);
		return STAT_INSTCMD_UNKNOWN;
	}

	if ((ct->flags & APC_PRESENT) == 0) {
		upslogx(LOG_WARNING, "instcmd: command [%s] is not supported",
			cmdname);
		return STAT_INSTCMD_UNKNOWN;
	}

	if (!strcasecmp(cmdname, "calibrate.start"))
		return do_cal(1);

	if (!strcasecmp(cmdname, "calibrate.stop"))
		return do_cal(0);

	if (ct->flags & APC_NASTY)
		return instcmd_chktime(ct);

	/* nothing special here */
	return do_cmd(ct);
}	

/* install pointers to functions for msg handlers called from msgparse */
static void setuphandlers(void)
{
	upsh.new_setvar = setvar;
	upsh.new_instcmd = instcmd;
}

/* functions that interface with main.c */

void upsdrv_banner(void)
{
	printf("Network UPS Tools (version %s) - APC Smart protocol driver\n",
		UPS_VERSION);
	printf("\tDriver version %s, command table %s\n",
		APC_DRIVER_VERSION,
		APC_TABLE_VERSION);
}

void upsdrv_makevartable(void)
{
	addvar(VAR_VALUE, "cable", "Specify alternate cable (940-0095B)");
	addvar(VAR_VALUE, "sdtype", "Specify shutdown type (1-3)");
}

void upsdrv_initups(void)
{
	char	*cable;

	open_serial(device_path, B2400);

	cable = getval("cable");

	if (cable)
		if (!strcasecmp(cable, ALT_CABLE_1))
			init_serial_0095B();
}

void upsdrv_help(void)
{
	printf("\nShutdown types:\n");
	printf("  0: soft shutdown or powerdown, depending on battery status\n");
	printf("  1: soft shutdown followed by powerdown\n");
	printf("  2: instant power off\n");
	printf("  3: power off with grace period\n");
	printf("Modes 0-1 will make the UPS come back when power returns\n");
	printf("Modes 2-3 will make the UPS stay turned off when power returns\n");
}

void upsdrv_initinfo(void)
{
	if (!smartmode()) {
		printf("Unable to detect an APC Smart protocol UPS on port %s\n", 
			device_path);
		printf("Check the cabling, port name or model name and try again\n");
		exit(1);
	}

	/* manufacturer ID - hardcoded in this particular module */
	dstate_setinfo("ups.mfr", "APC");

	dstate_setinfo("driver.version.internal", "%s", APC_DRIVER_VERSION);

	getbaseinfo();

	printf("Detected %s [%s] on %s\n", dstate_getinfo("ups.model"),
		dstate_getinfo("ups.serial"), device_path);

	setuphandlers();
}

void upsdrv_updateinfo(void)
{
	static	time_t	last_full = 0;
	time_t	now;

	update_status();

	time(&now);

	/* refresh all variables hourly */
	/* does not catch measure-ups II insertion/removal */
	if (difftime(now, last_full) > 3600) {
		updateinfo(1);
		last_full = now;
		return;
	}

	updateinfo(0);
}

/*
 * Local variables:
 *  c-indent-level: 8
 *  c-basic-offset: 8
 * End:
 */

void upsdrv_cleanup(void)
{
}
