/* tripplite.c - model specific routines for Tripp Lite SmartUPS models
   (tested with "SMART 700" [on back -- "SmartPro UPS" on front])

   tripplite.c was derived from Russell Kroll's bestups.c by Rik Faith.

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>
   Copyright (C) 2001  Rickard E. (Rik) Faith <faith@alephnull.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

/* REFERENCE 1

   A few magic numbers were derived from the GPL'd file
   opensrc_server/upscmd.cpp, available from Tripp Lite at
   http://www.tripplite.com/linux/.  The copyright notice that applies
   to this file is as follows:

   The PowerAlert server, the PowerAlert client, and all supporting 
   documents and files, are Copyright (C) 1998, 1999 by Tripp Lite,
   unless expressly stated therein.

   This program is free software; you can redistribute it and/or 
   modify it under the terms of the GNU General Public License as 
   published by the Free Software Foundation; either version 2 of 
   the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of 
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
   General Public License for more details.

   You should have received a copy of the GNU General Public License 
   along with this program; if not, write to the Free Software 
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* REFERENCE 2

   Other hints on commands were found on these web pages:
   http://www.kursknet.ru/~boa/ups/stinfo_command.html
   http://www.kursknet.ru/~boa/ups/rtinfo_command.html

   These pages confirm the information in the Tripp Lite source code
   referenced above and add more details.
   
   The first page tells how to derive the VA rating from w_value and
   l_value.  It's a confusing explanation because shifts are used to
   mask out bits.  Here is an example starting with the formula on the
   web page and proceeding to a formula that uses C syntax.

         I have a SMART 700 (700VA, 450W)
         w_value = 0x84 [available from upsc as REG1]
         l_value=- 0x60 [available from upsc as REG2]

         Unit Type: bit 6 of w_value is 0 so I have a Smart (vs. a Unison)

         VA Rating: ((([V:W Result]<<2)*8)+([V:L Result]>>3))*5
                    = (((w_value<<2)*8)+([l_value]>>3)) * 5
                    = ((w_value & 0x3f)*32 + (l_value >> 3)) * 5
                    = (4 * 32 + 12) * 5
                    = 700
*/

#define DRV_VERSION "0.02"

#include "main.h"
#include <ctype.h>

#define ENDCHAR  '\n'	/* replies end with CR LF -- use LF to end */
#define IGNCHARS "\r"   /* ignore CR */
#define UPSDELAY  1
#define MAXTRIES  3

static  char    *w_value, *l_value, *v_value, *x_value;
static	int	poll_failures = 0;

static int hex2d(char *start, int len)
{
    char buf[256];

    strncpy(buf, start, len);
    buf[len] = '\0';
    return strtol(buf, NULL, 16);
}

int instcmd(const char *cmdname, const char *extra)
{
	char    buf[256];

	if (!strcasecmp(cmdname, "test.battery.start")) {
		upssend("%s", ":A\r"); /* Start self test */
		sleep(UPSDELAY);
		upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);

		return STAT_INSTCMD_HANDLED;
	}

	upslogx(LOG_NOTICE, "instcmd: unknown command [%s]", cmdname);
	return STAT_INSTCMD_UNKNOWN;
}

void upsdrv_initinfo(void)
{
    char *model;
    int  va;
    long w, l;

    dstate_addcmd("test.battery.start");	/* Turns off automatically */
    
    dstate_setinfo("ups.mfr", "%s", "Tripp Lite");

    w = hex2d(w_value, 2);
    l = hex2d(l_value, 2);
    
    if (w & 0x40) model = "Unison %d";
    else          model = "Smart %d";

    if (w & 0x80) {       /* New VA rating formula */
        va = ((w & 0x3f) * 32 + (l >> 3)) * 5;
    } else {                    /* Old VA rating formula */
        va = l / 2;
    }
    
    dstate_setinfo("ups.model", model, va);
    dstate_setinfo("ups.firmware", "%c%c", 
		'A'+v_value[0]-'0', 'A'+v_value[1]-'0');

/* we need more details on what these really are */
#if 0
    setinfo(INFO_FIRMREV1,   "%s", v_value);
    setinfo(INFO_FIRMREV2,   "%s", x_value);
    setinfo(INFO_REG1,       "%s", w_value);
    setinfo(INFO_REG2,       "%s", l_value);
#endif
    
    upsh.new_instcmd = instcmd;     /* Set here instead of upsdrv_initups */
    
    printf("Detected %s %s on %s\n",
           dstate_getinfo("ups.mfr"), dstate_getinfo("ups.model"), device_path);

    dstate_setinfo("driver.version.internal", "%s", DRV_VERSION);
}

static void setup_serial(void)
{	
    struct	termios	tio;
    
    if (tcgetattr(upsfd, &tio) == -1) fatal("tcgetattr");
    
    tio.c_iflag     = IXON | IXOFF;
    tio.c_oflag     = 0;
    tio.c_cflag     = (CS8 | CREAD | HUPCL | CLOCAL);
    tio.c_lflag     = 0;
    tio.c_cc[VMIN]  = 1;
    tio.c_cc[VTIME] = 0; 
    
#ifdef HAVE_CFSETISPEED
    cfsetispeed(&tio, B2400);
    cfsetospeed(&tio, B2400);
#else
#error This system lacks cfsetispeed() and has no other means to set the speed
#endif

    if (tcsetattr(upsfd, TCSANOW, &tio) == -1) fatal("tcsetattr");
}

void ups_sync(void)
{
    char	buf[256];
    int	        tries = 0;
    int         ret;

    for (;;) {
        tries++;
        if (tries > MAXTRIES) fatalx("\nFailed - giving up...");
        
        upssend("%s", ":W\r");
        sleep(UPSDELAY);
        ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
        ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
        
        if ((ret > 0) && isdigit((unsigned char) buf[0])) break;
    }
}

void upsdrv_shutdown(void)      /* FIXME: Not yet tested */
{
    char	buf[256];
    int	        ret;
    
    upssend("%s", ":N40\r"); /* Delay 64 seconds */
    sleep(UPSDELAY);
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
    
    upssend("%s", ":G\r");   /* Inverter off */
    sleep(UPSDELAY);
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);

#if 1
    /* FIXME: This may prevent UPS from powering up when utility comes
     * back. */
    upssend("%s", ":K0\r");  /* Relay off */
    sleep(UPSDELAY);
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
#endif
}

void ups_ident(void)
{
    char	buf[256];
    int	        ret;
    
    upssend("%s", ":W\r");
    sleep(UPSDELAY);
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
    if (w_value) free(w_value);
    w_value = xstrdup(buf);

    upssend("%s", ":L\r");
    sleep(UPSDELAY);
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
    if (l_value) free(l_value);
    l_value = xstrdup(buf);
    
    upssend("%s", ":V\r");
    sleep(UPSDELAY);
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
    if (v_value) free(v_value);
    v_value = xstrdup(buf);

    upssend("%s", ":X\r");
    sleep(UPSDELAY);
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
    if (x_value) free(x_value);
    x_value = xstrdup(buf);
}

void pollfail(char *why)
{
    poll_failures++;

    /* ignore the first few since these UPSes tend to drop characters */
    if (poll_failures == 3) upslogx(LOG_ERR, why);
    
    return;
}

void upsdrv_updateinfo(void)
{
    char    buf[256], temp[32];
    int	ret;
    int     bp;
    float   bv;
    
    upssend("%s", ":D\r");
    sleep(UPSDELAY); 
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
    
    if (strlen(buf) < 21) {
        pollfail("Poll failed: short read from UPS");
	dstate_datastale();
        return;
    }
    
    if (strlen(buf) > 21) {
        pollfail("Poll failed: oversized read from UPS");
	dstate_datastale();
        return;
    }
    
    /* only say this if it got high enough to log a failure note */
    if (poll_failures >= 3)
        upslogx(LOG_NOTICE, "UPS poll succeeded");
    
    poll_failures = 0;

/* this looks more like a candidate for upsdebugx, not a variable */
#if 0
    setinfo(INFO_REG3,    "%s",     buf);
    /* 0123456789abcdef01234
       ABCCDEFFGGGGHHIIIIJJJ
       A    0=LB   1=OK
       B    0=OVER 1=OK
       CC   INFO_UTILITY
       D    0=normal 1=TRIM 2=BOOST 3="EXTRA BOOST"
       E    0=OFF 1=OB 2=OL 3=OB (1 and 3 are the same?)
       FF   f(INFO_UPSTEMP)
       GG   ? INFO_BATTPCT (00 when OB, values don't match table we use)
       HH   ? (always 00)
       II   INFO_LOADPCT
       JJJJ ? (e.g., 5B82 5B82 5982 037B 0082)
       KKK  INFO_ACFREQ * 10
    */
#endif
    
    dstate_setinfo("input.voltage", "%03d",   hex2d(buf + 2, 2));
    dstate_setinfo("ups.temperature", "%03d",  (int)(hex2d(buf + 6, 2)*0.3636 - 21.0));
    dstate_setinfo("ups.load", "%03d",   hex2d(buf + 12, 2));
    dstate_setinfo("input.frequency",  "%02.2f", hex2d(buf + 18, 3) / 10.0);
    
    status_init();
    
    /* Battery Voltage Condition */
    switch (buf[0]) {
    case '0': status_set("LB");                   break; /* Low Battery */
    case '1':                                     break; /* Normal */
    default:  sprintf(temp, "BAT-%c?", buf[0]);  
              status_set(temp);
                                                  break; /* Unknown */
    }
    
    /* Load State */
    switch (buf[1]) {
    case '0': status_set("OVER");                 break; /* Overload */
    case '1':                                     break; /* Normal */
    default:  sprintf(temp, "LOAD-%c?", buf[1]); 
              status_set(temp);
                                                  break; /* Unknown */
    }
    
    /* Tap State */
    switch (buf[4]) {
    case '0':                                     break; /* Normal */
    case '1': status_set("TRIM");                 break; /* Reducing */ 
    case '2': status_set("BOOST");                break; /* Boost */
    case '3': status_set("BOOST");                break; /* Extra Boost */
    default:  sprintf(temp, "TAP-%c?", buf[4]);  
              status_set(temp);
                                                  break; /* Unknown */
    }
    
    /* Mode */
    switch (buf[5]) {
    case '0': status_set("OFF");                  break; /* Off */
    case '1': status_set("OB");                   break; /* On Battery */
    case '2': status_set("OL");                   break; /* On Line */
    case '3': status_set("OB");                   break; /* On Battery */
    default: sprintf(temp, "MODE-%c?", buf[5]);
              status_set(temp);
                                                  break; /* Unknown */
    }

    status_commit();
    
    upssend("%s", ":B\r");
    sleep(UPSDELAY); 
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
    bv = hex2d(buf, 2) / 10;
    if (bv <= 11.0)      bp =  10;
    else if (bv <= 11.2) bp =  20;
    else if (bv <= 11.4) bp =  30;
    else if (bv <= 11.6) bp =  40;
    else if (bv <= 11.8) bp =  50;
    else if (bv <= 12.0) bp =  60;
    else if (bv <= 12.2) bp =  70;
    else if (bv <= 12.5) bp =  80;
    else if (bv <= 13.2) bp =  90;
    else                 bp = 100;
    
    dstate_setinfo("battery.voltage", "%.1f", bv);
    dstate_setinfo("battery.charge",  "%03d",   bp);
    
    upssend("%s", ":M\r");
    sleep(UPSDELAY); 
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
    dstate_setinfo("input.voltage.maximum", "%d", hex2d(buf, 2));
    
    upssend("%s", ":I\r");
    sleep(UPSDELAY); 
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
    dstate_setinfo("input.voltage.minimum", "%d", hex2d(buf, 2));
    
    upssend("%s", ":C\r");
    sleep(UPSDELAY);
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
    ret = upsrecv(buf, sizeof(buf), ENDCHAR, IGNCHARS);
    switch (atoi(buf)) {
    case 0:  dstate_setinfo("ups.test.result", "%s", "OK");                    break;
    case 1:  dstate_setinfo("ups.test.result", "%s", "Battery Bad (Replace)"); break;
    case 2:  dstate_setinfo("ups.test.result", "%s", "In Progress");           break;
    case 3:  dstate_setinfo("ups.test.result", "%s", "Bad Inverter");          break;
    default: dstate_setinfo("ups.test.result", "Unknown (%s)", buf);           break;
    }

    dstate_dataok();
}

void upsdrv_help(void)
{
}

void upsdrv_makevartable(void)
{
}

void upsdrv_banner(void)
{
    printf("Network UPS Tools - Tripp-Lite SmartUPS driver %s (%s)\n",
           DRV_VERSION, UPS_VERSION);
}

void upsdrv_initups(void)
{
    open_serial(device_path, B2400);
    setup_serial();
    
    /* don't let upscommon warn about it since it happens way too often */
    /* this driver does its own checking with better handling */
    flag_timeoutfailure = -1;
    
    ups_sync();
    ups_ident();
}

void upsdrv_cleanup(void)
{
}
