/*
 * powercom.c - model specific routines for following units:
 *  -Trust 425/625
 *  -Powercom
 *  -Advice Partner/King PR750
 *    See http://www.advice.co.il/product/inter/ups.html for its specifications.
 *    This model is based on PowerCom (www.powercom.com) models.
 *  -Socomec Sicon Egys 420
 *
 * Read docs/powercom.txt for other models and manufactures
 *
 * $Id: - will be filled in on next CVS add/update $
 *
 * Status:
 *  20011022/Revision 0.1 - Peter Bieringer <pb@bieringer.de>
 *   - porting done, looks like working
 *  20011208/Revision 0.2 - Peter Bieringer <pb@bieringer.de>
 *   - add some debug code
 *   - add new option "subtype"
 *   - add support for a 16 byte sending UPS (KP625AP)
 *   - shutdown function checked, but 'Trust' wakes up after a few seconds...
 *  20020629/Revision 0.3 - Simon Rozman <simon@rozman.net>
 *   - add different serial port init. for each subtype
 *   - add support for Socomec Sycon Egys 420
 *   - add support for different calculation parameters for each subtype
 *   - add support for output voltage and output frequency monitoring
 *  20020701/Revision 0.4 - Simon Rozman <simon@rozman.net>,
 *                                          proposed by Shaul Karl
 *   - subtypes specific info moved to struct subtype
 *  20030506/Revision 0.5 - Shaul Karl <shaulka@bezeqint.net>
 *   - converted to dstate
 *   - modify the validation code
 *   - replace subtype with type
 *   	
 * Copyrights:
 * (C) 2002 Simon Rozman <simon@rozman.net>
 *  Added support for Egys
 
 * (C) 2001 Peter Bieringer <pb@bieringer.de>
 *  Porting old style "powercom" to new style "newpowercom"
 *  
 * (C) 2000  Shaul Karl <shaulk@israsrv.net.il>
 *  Creating old style "powercom"
 *   Heavily based on
 *    ups-trust425+625.c - model specific routines for Trust UPS 425/625
 *    Copyright (C) 1999  Peter Bieringer <pb@bieringer.de>
 *                              
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *                            
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *    
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 */ 

#include "main.h"
#include "powercom.h"

#define POWERCOM_DRIVER_VERSION      "$Revision: 0.5 $"
#define NUM_OF_SUBTYPES              (sizeof (types) / sizeof (*types))

/* variables used by module */
static unsigned char powercom_raw_data[MAX_NUM_OF_BYTES_FROM_UPS]; /* raw data reveived from UPS */
static unsigned int  powercom_linevoltage = 230U; /* line voltage, can be defined via command line option */
static char *powercom_manufacturer = "PowerCom";
static char *powercom_modelname    = "Unknown";
static char *powercom_serialnumber = "Unknown";
static char *powercom_type_name    = "Trust";
static unsigned int type_index = 0;

/* forward declaration of functions used to setup flow control */
static void dtr0rts1 ();
static void no_flow_control ();

/* struct defining types */
static struct type types[] = {
        {
                "Trust",
                11,
                dtr0rts1,
                { { 5, 0 }, { 7, 0 }, { 8, 0 } },
                {  0.00020997, 0.00020928 },
                {  6.1343, -0.3808,  4.3110,  0.1811 },
                {  5.0000,  0.3268,  -825.00,  4.5639, -835.82 },
                {  1.9216, -0.0977,  0.9545,  0.0000 },
        },
        {
                "KP625AP",
                16,
                dtr0rts1,
                { { 5, 0x80 }, { 7, 0 }, { 8, 0 } },
                {  0.00020997, 0.00020928 },
                {  6.1343, -0.3808,  4.3110,  0.1811 },
                {  5.0000,  0.3268,  -825.00,  4.5639, -835.82 },
                {  1.9216, -0.0977,  0.9545,  0.0000 },
        },
        {
                "KIN2200AP",
                16,
                dtr0rts1,
                { { 7, 0 }, { 8, 0 }, { 8, 0 } },
                {  0.00020997, 0.0 },
                {  6.1343, -0.3808,  1.075,  0.1811 },
                {  5.0000,  0.3268,  -825.00,  0.46511, 0 },
                {  1.9216, -0.0977,  0.82857,  0.0000 },
        },
        {
                "Egys",
                16,
                no_flow_control,
                { { 5, 0x80 }, { 7, 0 }, { 8, 0 } },
                {  0.00020997, 0.00020928 },
                {  6.1343, -0.3808,  1.3333,  0.6667 },
                {  5.0000,  0.3268,  -825.00,  2.2105, -355.37 },
                {  1.9216, -0.0977,  0.9545,  0.0000 },
        },
};


/* used external variables */
extern int sddelay;          /* shutdown delay, set by "-d $delay" in main.c */
extern int do_forceshutdown; /* shutdown delay, set by "-k" in main.c */


/*
 * local used functions
 */

static void powercom_shutdown(void)
{
	printf ("Initiating UPS shutdown!\n");
        
	upssendchar (SHUTDOWN);
	upssendchar (SEC_FOR_POWERKILL);
			
	exit (0);
}


/* registered instant commands */
static int instcmd (const char *cmdname, const char *extra)
{
	if (!strcasecmp(cmdname, "test.battery.start")) { 
	    upssendchar (BATTERY_TEST);
	    return STAT_INSTCMD_HANDLED;
    }

    upslogx(LOG_NOTICE, "instcmd: unknown command [%s]", cmdname);
    return STAT_INSTCMD_UNKNOWN;
}


/* wait for buflen bytes from upsfd and store in buf */
static int read_raw_data (int upsfd, unsigned char *buf, int buflen)
{
	int     counter = 0;
	struct  sigaction sa;
	sigset_t pow_sigmask;

	sa.sa_handler = timeout;
        sigemptyset (&pow_sigmask);
        sa.sa_mask = pow_sigmask;
        sa.sa_flags = 0;
        sigaction (SIGALRM, &sa, NULL);

        alarm (3);

        for (counter = 0; counter < buflen; counter++) 

		if (!(read (upsfd, buf + counter, 1))) {

			alarm (0);
		        signal (SIGALRM, SIG_IGN);
			return counter;
			
		}

	nolongertimeout();
		
	return counter;
}
			

/* set DTR and RTS lines on a serial port to supply a passive
 * serial interface: DTR to 0 (-V), RTS to 1 (+V)
 */
static void dtr0rts1 ()
{
	int dtr_bit = TIOCM_DTR;
	int rts_bit = TIOCM_RTS;

	upsdebugx(2, "DTR => 0, RTS => 1");

	/* set DTR to low and RTS to high */
	ioctl(upsfd, TIOCMBIC, &dtr_bit);
	ioctl(upsfd, TIOCMBIS, &rts_bit);
}


/* clear any flow control */
static void no_flow_control ()
{
	struct termios tio;
	
	tcgetattr (upsfd, &tio);
	
	tio.c_iflag &= ~ (IXON | IXOFF);
	tio.c_cc[VSTART] = _POSIX_VDISABLE;
	tio.c_cc[VSTOP] = _POSIX_VDISABLE;
				
	upsdebugx(2, "Flow control disable");

	/* disable any flow control */
	tcsetattr(upsfd, TCSANOW, &tio);
}


/* sane check for returned buffer */
static int validate_raw_data (void)
{
	int i = 0, 
	    num_of_tests = 
			sizeof types[0].validation / 
		                    sizeof types[0].validation[0];
	
	for (i = 0;
		 i < num_of_tests  && 
		   powercom_raw_data[
	           types[type_index].validation[i].index_of_byte
	                        ] ==
          		   types[type_index].validation[i].required_value;
		 i++)  ;
	return (i < num_of_tests) ? 1 : 0;
}


/*
 * global used functions
 */

/* update information */
void upsdrv_updateinfo(void)
{
	char	val[32];
	float	tmp;
	int	i, c;
	
	/* send trigger char to UPS */
	if (upssendchar (SEND_DATA) != 1) {
		upslogx(LOG_NOTICE, "writing error");
		dstate_datastale();
		return;
	} else {
		upsdebugx(5, "Num of bytes requested for reading from UPS: %d", types[type_index].num_of_bytes_from_ups);

		c = read_raw_data(upsfd, powercom_raw_data, types[type_index].num_of_bytes_from_ups);
	
		if (c != types[type_index].num_of_bytes_from_ups) {
			upslogx(LOG_NOTICE, "data receiving error (%d instead of %d bytes)", c, types[type_index].num_of_bytes_from_ups);
			dstate_datastale();
			return;
		} else
			upsdebugx(5, "Num of bytes received from UPS: %d", c);

	};

	/* optional dump of raw data */
	if (nut_debug_level > 4) {
		printf("Raw data from UPS:\n");
		for (i = 0; i < types[type_index].num_of_bytes_from_ups; i++) {
	 		printf("%2d 0x%02x (%c)\n", i, powercom_raw_data[i], powercom_raw_data[i]>=0x20 ? powercom_raw_data[i] : ' ');
		};
	};

	/* validate raw data for correctness */
	if (validate_raw_data() != 0) {
		upslogx(LOG_NOTICE, "data receiving error (validation check)");
		dstate_datastale();
		return;
	};
	
	/* input.frequency */
	upsdebugx(3, "input.frequency   (raw data): [raw: %u]",
	                            powercom_raw_data[INPUT_FREQUENCY]);
	dstate_setinfo("input.frequency", "%02.2f",
	    powercom_raw_data[INPUT_FREQUENCY] ? 
			1.0 / (types[type_index].freq[0] *
	                powercom_raw_data[INPUT_FREQUENCY] +
	                        types[type_index].freq[1]) : 0);
	upsdebugx(2, "input.frequency: %s", dstate_getinfo("input.frequency"));

	/* output.frequency */
	upsdebugx(3, "output.frequency   (raw data): [raw: %u]",
	                            powercom_raw_data[OUTPUT_FREQUENCY]);
	dstate_setinfo("output.frequency", "%02.2f",
	    powercom_raw_data[OUTPUT_FREQUENCY] ? 
			1.0 / (types[type_index].freq[0] *
	                powercom_raw_data[OUTPUT_FREQUENCY] +
	                        types[type_index].freq[1]) : 0);
	upsdebugx(2, "output.frequency: %s", dstate_getinfo("output.frequency"));

	/* ups.load */	
	upsdebugx(3, "ups.load  (raw data): [raw: %u]",
	                            powercom_raw_data[UPS_LOAD]);
	dstate_setinfo("ups.load", "%03.1f", 
	    tmp = powercom_raw_data[STATUS_A] & MAINS_FAILURE ?
	 		types[type_index].loadpct[0] *
	            powercom_raw_data[UPS_LOAD] +
	                types[type_index].loadpct[1] :
			types[type_index].loadpct[2] *
	            powercom_raw_data[UPS_LOAD] +
	                types[type_index].loadpct[3]);
	upsdebugx(2, "ups.load: %s", dstate_getinfo("ups.load"));

	/* battery.charge */
	upsdebugx(3, "battery.charge (raw data): [raw: %u]",
	                            powercom_raw_data[BATTERY_CHARGE]);
	dstate_setinfo("battery.charge", "%03.1f",
	    powercom_raw_data[STATUS_A] & MAINS_FAILURE ?
			types[type_index].battpct[0] *
	            powercom_raw_data[BATTERY_CHARGE] +
			        types[type_index].battpct[1] * tmp +
			                types[type_index].battpct[2] :
			types[type_index].battpct[3] *
	            powercom_raw_data[BATTERY_CHARGE] +
			                types[type_index].battpct[4]);
	upsdebugx(2, "battery.charge: %s", dstate_getinfo("battery_charge"));

	/* input.voltage */	
	upsdebugx(3, "input.voltage (raw data): [raw: %u]",
	                            powercom_raw_data[INPUT_VOLTAGE]);
	dstate_setinfo("input.voltage", "%03.1f",
	    tmp = powercom_linevoltage >= 220 ?
			types[type_index].voltage[0] *
	            powercom_raw_data[INPUT_VOLTAGE] +
			            types[type_index].voltage[1] :
			types[type_index].voltage[2] *
	            powercom_raw_data[INPUT_VOLTAGE] +
			            types[type_index].voltage[3]);
	upsdebugx(2, "input.voltage: %s", dstate_getinfo("input.voltage"));
	
	/* output.voltage */	
	upsdebugx(3, "input.voltage (raw data): [raw: %u]",
	                            powercom_raw_data[OUTPUT_VOLTAGE]);
	dstate_setinfo("output.voltage", "%03.1f",
	    powercom_linevoltage >= 220 ?
			types[type_index].voltage[0] *
	            powercom_raw_data[OUTPUT_VOLTAGE] +
			            types[type_index].voltage[1] :
			types[type_index].voltage[2] *
	            powercom_raw_data[OUTPUT_VOLTAGE] +
			            types[type_index].voltage[3]);
	upsdebugx(2, "output.voltage: %s", dstate_getinfo("output.voltage"));

	status_init();
	
	*val = 0;
	if (!(powercom_raw_data[STATUS_A] & MAINS_FAILURE)) {
		!(powercom_raw_data[STATUS_A] & OFF) ? 
			status_set("OL") : status_set("OFF");
	} else {
		status_set("OB");
	}

	if (powercom_raw_data[STATUS_A] & LOW_BAT)  status_set("LB");

	if (powercom_raw_data[STATUS_A] & AVR_ON) {
		tmp < powercom_linevoltage ? 
			status_set("BOOST") : status_set("TRIM");
	}

	if (powercom_raw_data[STATUS_A] & OVERLOAD)  status_set("OVER");

	if (powercom_raw_data[STATUS_B] & BAD_BAT)  status_set("RB");

	if (powercom_raw_data[STATUS_B] & TEST)  status_set("TEST");

	status_commit();

	upsdebugx(2, "STATUS: %s", dstate_getinfo("ups.status"));
	dstate_dataok();
}


/* shutdown UPS */
void upsdrv_shutdown(void)
{
	if (do_forceshutdown == 1) {
		/* power down the attached load immediately */
		printf("Forced UPS shutdown triggered, do it...\n");
		powercom_shutdown();
	} else {
		/* power down the attached load after the given delay */
		printf("UPS shutdown with '%d' seconds delay triggered, wait now...\n", sddelay);
		sleep(sddelay);
		powercom_shutdown();
	};
}


/* initialize UPS */
void upsdrv_initups(void)
{
	int tmp, i;

	/* check manufacturer name from arguments */
	if (getval("manufacturer") != NULL) {
		powercom_manufacturer = getval("manufacturer");
	};
	
	/* check model name from arguments */
	if (getval("modelname") != NULL) {
		powercom_modelname = getval("modelname");
	};
	
	/* check serial number from arguments */
	if (getval("serialnumber") != NULL) {
		powercom_serialnumber = getval("serialnumber");
	};
	
	/* get and check type */
	if (getval("type") != NULL) {
		powercom_type_name = getval("type");
		for (i = 0; i < NUM_OF_SUBTYPES; i++) {
			if (strcmp(types[i].name, powercom_type_name) == 0)
				break;
		};
		if (i < NUM_OF_SUBTYPES) {
			/* we found a type */
			type_index = i;	
		} else {
			/* no such type exists */
			printf("Given UPS type '%s' isn't valid!\n", powercom_type_name);
			exit (1);
		};
	};
	
	/* check line voltage from arguments */
	if (getval("linevoltage") != NULL) {
		tmp = atoi(getval("linevoltage"));
		if (! ( (tmp >= 220 && tmp <= 240) || (tmp >= 110 && tmp <= 120) ) ) {
			printf("Given line voltage '%d' is out of range (110-120 or 220-240 V)\n", tmp);
			exit (1);
		};
		powercom_linevoltage  = (unsigned int) tmp;
	};

	upsdebugx(1, "Values of arguments:");
	upsdebugx(1, " manufacturer : '%s'", powercom_manufacturer);
	upsdebugx(1, " model name   : '%s'", powercom_modelname);
	upsdebugx(1, " serial number: '%s'", powercom_serialnumber);
	upsdebugx(1, " line voltage : '%u'", powercom_linevoltage);
	upsdebugx(1, " type         : '%s'", powercom_type_name);

	/* open serial port */
	open_serial (device_path, B1200);
	
	/* setup flow control */
	types[type_index].setup_flow_control();
}


/* display help */
void upsdrv_help(void)
{
	printf("\n");
	return;
}


/* display banner */
void upsdrv_banner(void)
{
	printf ("Network UPS Tools (version %s) - PowerCom and similars protocol driver\n",
		UPS_VERSION);
	printf ("\tDriver %s\n",
		POWERCOM_DRIVER_VERSION);
}


/* initialize information */
void upsdrv_initinfo(void)
{
	/* write constant data for this model */
	dstate_setinfo ("ups.mfr", "%s", powercom_manufacturer);
	dstate_setinfo ("ups.model", "%s", powercom_modelname);
	dstate_setinfo ("ups.serial", "%s", powercom_serialnumber);
	dstate_setinfo ("ups.model.type", "%s", powercom_type_name);
	dstate_setinfo ("input.voltage.nominal", "%u", powercom_linevoltage);

	/* now add the instant commands */
    dstate_addcmd ("test.battery.start");
	upsh.new_instcmd = instcmd;
}


/* define possible arguments */
void upsdrv_makevartable(void)
{
	addvar(VAR_VALUE, "manufacturer",  "Specify manufacturer name (default: 'PowerCom')");
	addvar(VAR_VALUE, "linevoltage",   "Specify line voltage (110-120 or 220-240 V), because it cannot detect automagically (default: 230 V)");
	addvar(VAR_VALUE, "modelname",     "Specify model name, because it cannot detect automagically (default: Unknown)");
	addvar(VAR_VALUE, "serialnumber",  "Specify serial number, because it cannot detect automagically (default: Unknown)");
	addvar(VAR_VALUE, "type",       "Type of UPS like 'Trust', 'KP625AP', 'KIN2200AP' or 'Egys' (default: 'Trust')");
}



void upsdrv_cleanup(void)
{
}
