/* NVClock 0.8 - Linux overclocker for NVIDIA cards
 * 
 * Copyright(C) 2001-2005 Roderick Colenbrander
 *
 * site: http://nvclock.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "config.h"
#include "nvclock.h"
#ifdef HAVE_NVCONTROL
    #include "nvcontrol.h"
#endif
#include "backend.h"

/* At the moment NV3x/NV4x cards are supported through
/  both a pure low-level and a hybrid backend called 'nvcontrol'
/  which can use both the NV-CONTROL extension for X and low-level.
/  The advantage of NV-CONTROL is that it can handle the
/  2d/3d clocks which appear on various boards, and further
/  it might be more stable. Further the low-level backend
/  should work fine too.
/
/  The task of this function is to switch between the low-level
/  and NV-CONTROL backend by adjusting the function pointers.
*/
static void set_backend()
{
#ifdef HAVE_NVCONTROL
    int irq;
    int have_coolbits = 0;
    nv_card.have_coolbits = 0;

    /* We need an X display to check if NV-CONTROL support exists */    
    if(nv_card.dpy)
    {
	/* Check if we have NV-CONTROL support */
	if(init_nvcontrol(nv_card.dpy))
	{
	    int tmp;
	    have_coolbits = NVGetAttribute(nv_card.dpy, 0, 0, NV_GPU_OVERCLOCKING_STATE, &tmp);
	    
	    /* Also retrieve the irq which is used to sync nvclock and NV-CONTROL */
	    NVGetAttribute(nv_card.dpy, 0, 0, NV_IRQ, &irq);
	}
    }
    
    /* The NVCONTROL backend is only supported when the NV-CONTROL
    /  extension is present on Desktop GPUs. Further only enable
    /  this backend when both nvclock and NVCONTROL are 'talking'
    /  about the same card which is likely the case when the IRQs
    /  match. Note the nvcontrol backend supports both low-level
    /  and NV-CONTROL. You can switch between both using set_state().
    /
    /  FIXME: somehow add support for multiple Coolbits cards
    */
    if(have_coolbits && (nv_card.gpu == DESKTOP) && (nv_card.irq == irq))
    {
	nv_card.have_coolbits = 1;

	if(nvclock.card[nv_card.number].state == 0)
	    set_state(STATE_3D); /* Default to 3D */
	    
        nv_card.get_gpu_speed = nvcontrol_get_gpu_speed;
        nv_card.get_memory_speed = nvcontrol_get_memory_speed;
        nv_card.set_gpu_speed = nvcontrol_set_gpu_speed;
        nv_card.set_memory_speed = nvcontrol_set_memory_speed;
        nv_card.reset_gpu_speed = nvcontrol_reset_gpu_speed;
        nv_card.reset_memory_speed = nvcontrol_reset_memory_speed;
    }
    else
#endif
    {
	nv_card.have_coolbits = 0;

	/* GeforceFX 5600/5700 */
	if(nv_card.arch == NV31)
	{
    	    nv_card.get_gpu_speed = nv31_get_gpu_speed;
    	    nv_card.get_memory_speed = nv31_get_memory_speed;
    	    nv_card.set_memory_speed = nv31_set_memory_speed;
    	    nv_card.set_gpu_speed = nv31_set_gpu_speed;
    	    nv_card.reset_gpu_speed = nv31_reset_gpu_speed;
    	    nv_card.reset_memory_speed = nv31_reset_memory_speed;
	}
	/* GeforceFX 5800/5900 */
	else if(nv_card.arch & (NV30|NV35))
	{
    	    nv_card.get_gpu_speed = nv30_get_gpu_speed;
    	    nv_card.get_memory_speed = nv30_get_memory_speed;
    	    nv_card.set_gpu_speed = nv30_set_gpu_speed;
    	    nv_card.set_memory_speed = nv30_set_memory_speed;
    	    nv_card.reset_gpu_speed = nv30_reset_gpu_speed;
    	    nv_card.reset_memory_speed = nv30_reset_memory_speed;
	}
	/* Geforce 6200/6600/6800 */
	else if(nv_card.arch & NV4X)
	{
    	    nv_card.get_gpu_speed = nv40_get_gpu_speed;
    	    nv_card.get_memory_speed = nv40_get_memory_speed;
    	    nv_card.set_memory_speed = nv40_set_memory_speed;
    	    nv_card.set_gpu_speed = nv40_set_gpu_speed;
    	    nv_card.reset_gpu_speed = nv40_reset_gpu_speed;
    	    nv_card.reset_memory_speed = nv40_reset_memory_speed;
	}
    }
}

/* This function is actually a basic version of set_card.
/  It mainly copies the entries of the card list and maps 
/  the video registers. We need this function as we need access
/  to the videocard from the config file creation code. Further it
/  is being used from set_card.
*/
int set_card_info(int number)
{
    nv_card.card_name = nvclock.card[number].card_name;
    nv_card.number = number;
    nv_card.device_id = nvclock.card[number].device_id;
    nv_card.arch = nvclock.card[number].arch;
    nv_card.devbusfn = nvclock.card[number].devbusfn;
    nv_card.irq = nvclock.card[number].irq;
    nv_card.reg_address = nvclock.card[number].reg_address;
    nv_card.gpu = nvclock.card[number].gpu;
    nv_card.bios = nvclock.card[number].bios;
    nv_card.supported = 0; /* Reset the capabilities of the card */
    nv_card.state = nvclock.card[number].state;

    if(!map_mem(nvclock.card[number].dev_name))
    {
	/* The card isn't set, so set it to no card */
	nv_card.number = -1;
	return 0; /* map_mem already took care of the error */
    }
    
    return 1;
}

/* Set the card object to the requested card */
int set_card(int number)
{
    /* When the internal card number is the same as the number passed, don't do anything. */
    if((nv_card.number == number) && (nv_card.device_id != 0)) 
	return 1;

    if(!set_card_info(number))
	return 0;

    /* For hardware monitoring support we need access to the i2c bus. Each card
    /  has 2 or 3 of these busses. Currently we will create these busses here in set_card
    /  and store them in the card list. It should also be possible to create just 3 busses and store
    /  those in the nv_card struct and have each card use them and limit the number of used busses by
    /  a variable but the issue is that this is not very nice and perhaps even incorrect. For example
    /  different cards might need different i2c bus timings and further we mess up the xfree86 i2c structures
    /  if we mix dev/bus structures from different cards.
    /
    /  Once the busses have been created they will be probed for sensor chips. If a sensor is found
    /  a i2c device pointer will be stored. Using this pointer we can access the chip using standard
    /  xf86 i2c functions.
    */
    if(nvclock.card[number].busses[0] == NULL)
    {
	nvclock.card[number].num_busses = 2;
        nvclock.card[number].busses[0] = NV_I2CCreateBusPtr("BUS0", 0x3e); /* available on riva128 and higher */
        nvclock.card[number].busses[1] = NV_I2CCreateBusPtr("BUS1", 0x36); /* available on rivatnt hardware and higher */

	/* There's an extra bus available on geforce4mx, geforcefx and geforce6 cards (not on geforce4ti!).
	/  The check below looks for geforce4mx/geforcefx/geforce6 architecture.
	*/
	if(nv_card.arch & (NV17 | NV3X | NV4X))
	{
	    nvclock.card[number].num_busses = 3;
    	    nvclock.card[number].busses[2] = NV_I2CCreateBusPtr("BUS2", 0x50); 
	}

        nvclock.card[number].sensor = I2cProbeDevices(nvclock.card[number].busses, nvclock.card[number].num_busses);
    }

    /* When a sensor is available, enable the correct function pointers */
    if(nvclock.card[number].sensor)
    {
	nv_card.sensor = nvclock.card[number].sensor;
	nv_card.sensor_name = nv_card.sensor->chip_name;
	
	switch(nv_card.sensor->chip_id)
	{
	    case LM99:
	    case MAX6559:
		nv_card.supported |= BOARD_TEMP_MONITORING_SUPPORTED | GPU_TEMP_MONITORING_SUPPORTED;
		nv_card.get_board_temp = lm99_get_board_temp;
        	nv_card.get_gpu_temp = lm99_get_gpu_temp;
		break;
	    case F75375:
		nv_card.supported |= BOARD_TEMP_MONITORING_SUPPORTED | GPU_TEMP_MONITORING_SUPPORTED | I2C_FANSPEED_MONITORING_SUPPORTED;
		nv_card.get_board_temp = f75375_get_board_temp;
        	nv_card.get_gpu_temp = f75375_get_gpu_temp;
		nv_card.get_i2c_fanspeed_rpm = f75375_get_fanspeed_rpm;
		nv_card.get_i2c_fanspeed_pwm = f75375_get_fanspeed_pwm;
		nv_card.set_i2c_fanspeed_pwm = f75375_set_fanspeed_pwm;
		break;
	    case W83L785R:
		nv_card.supported |= BOARD_TEMP_MONITORING_SUPPORTED | GPU_TEMP_MONITORING_SUPPORTED | I2C_FANSPEED_MONITORING_SUPPORTED;
		nv_card.get_board_temp = w83l785r_get_board_temp;
        	nv_card.get_gpu_temp = w83l785r_get_gpu_temp;
		nv_card.get_i2c_fanspeed_rpm = w83l785r_get_fanspeed_rpm;
		nv_card.get_i2c_fanspeed_pwm = w83l785r_get_fanspeed_pwm;
		nv_card.set_i2c_fanspeed_pwm = w83l785r_set_fanspeed_pwm;
		break;
	}
    }
    else
    {
	nv_card.sensor = NULL;    
    }
	
    /* Set the default pll's */
    nv_card.mpll = nvclock.card[number].mpll;
    nv_card.mpll2 = nvclock.card[number].mpll2;
    nv_card.nvpll = nvclock.card[number].nvpll;
    nv_card.nvpll2 = nvclock.card[number].nvpll2;

    /* gpu arch/revision */
    nv_card.get_gpu_architecture = get_gpu_architecture;
    nv_card.get_gpu_revision = get_gpu_revision;

    /* Find out what memory is being used */
    nv_card.mem_type = (nv_card.PFB[0x200/4] & 0x01) ? DDR : SDR;
    nv_card.get_bus_type = get_bus_type;

    if(strcmp(nv_card.get_bus_type(), "AGP") == 0)
    {
	nv_card.get_agp_rate = get_agp_rate;
	nv_card.get_agp_status = get_agp_status;
	nv_card.get_fw_status = get_fw_status;
	nv_card.get_sba_status = get_sba_status;
	nv_card.get_supported_agp_rates = get_supported_agp_rates;
    }
    else
    {
	nv_card.get_agp_rate = get_agp_rate;
	nv_card.get_agp_status = dummy_str;
	nv_card.get_fw_status = dummy_str;
	nv_card.get_sba_status = dummy_str;
	nv_card.get_supported_agp_rates = dummy_str;
    }
    
    nv_card.get_memory_size = get_memory_size;
    nv_card.get_memory_type = get_memory_type;
    nv_card.get_memory_width = get_memory_width;

    /* GeforceFX 5600/5700 */
    if(nv_card.arch == NV31)
    {
	nv_card.supported |= (GPU_OVERCLOCKING_SUPPORTED | MEM_OVERCLOCKING_SUPPORTED);
	set_backend();
    }
    /* GeforceFX 5800/5900 */
    else if(nv_card.arch & (NV30|NV35))
    {
	nv_card.supported |= (GPU_OVERCLOCKING_SUPPORTED | MEM_OVERCLOCKING_SUPPORTED);
	set_backend();

	/* HW monitoring; bit 31 is an indication if fanspeed monitoring is available
	/  Note this bit isn't very reliable as it is set on cards with advanced sensors too.
	/
	/  Only support this on NV35/NV38 hardware for now as it works differently on NV30 boards
	*/
	if((nv_card.PMC[0x10f0/4] & 0x80000000) && (nv_card.arch == NV35) && !(nv_card.supported & I2C_FANSPEED_MONITORING_SUPPORTED))
	{
	    nv_card.supported |= GPU_FANSPEED_MONITORING_SUPPORTED;
	    nv_card.get_fanspeed = nv30_get_fanspeed;
	    nv_card.set_fanspeed = nv30_set_fanspeed;
	}
    }
    /* Geforce 6200/6600/6800 */
    else if(nv_card.arch & NV4X)
    {
	nv_card.supported |= (GPU_OVERCLOCKING_SUPPORTED | MEM_OVERCLOCKING_SUPPORTED);
	set_backend();

	nv_card.get_default_mask = nv40_get_default_mask;
	nv_card.get_hw_masked_units = nv40_get_hw_masked_units;
	nv_card.get_sw_masked_units = nv40_get_sw_masked_units;
	nv_card.get_pixel_pipelines = nv40_get_pixel_pipelines;
	nv_card.get_vertex_pipelines = nv40_get_vertex_pipelines;

	/* For now enable modding on NV40 cards and NV43 revisions prior to A4; other cards are locked */
	if((nv_card.arch & NV40) || ((nv_card.arch & NV43) && (get_gpu_revision() < 0xA4)))
	{
	    nv_card.supported |= PIPELINE_MODDING_SUPPORTED;
	    nv_card.set_pixel_pipelines = nv40_set_pixel_pipelines;
	    nv_card.set_vertex_pipelines = nv40_set_vertex_pipelines;
	}

	/* For now support Smartdimmer on 6200Go laptops */
	if(((nv_card.device_id & 0xff0) == 0x160) && nv_card.gpu == MOBILE)
	{
	    nv_card.supported |= SMARTDIMMER_SUPPORTED;
	    nv_card.mobile_get_smartdimmer = nv44_mobile_get_smartdimmer;
	    nv_card.mobile_set_smartdimmer = nv44_mobile_set_smartdimmer;
	}

	/* Temperature monitoring; all cards after the NV40 feature an internal temperature sensor.
	/  Only it is disabled on most 6200/6600(GT) cards but we can re-enable it ;)
	*/
	if((nv_card.arch & (NV43 | NV44 | NV47)) && !(nv_card.supported & GPU_TEMP_MONITORING_SUPPORTED))
	{
	    nv_card.supported |= GPU_TEMP_MONITORING_SUPPORTED;
	    nv_card.sensor_name = (char*)strdup("GPU Internal Sensor");
	    nv_card.get_gpu_temp = (int(*)(I2CDevPtr))nv43_get_gpu_temp;
	}

	/* Fanspeed monitoring; bit 31 is an indication if fanspeed monitoring is available
	/  Note this bit isn't very reliable as it is set on cards with advanced sensors too.
	/  Should the NV47 use the NV40 codepath? Should the NV44 use the NV43 codepath?
	*/
	if(((nv_card.arch & (NV40 | NV41 | NV47)) && (nv_card.PMC[0x10f0/4] & 0x80000000)) && !(nv_card.supported & I2C_FANSPEED_MONITORING_SUPPORTED))
	{
	    nv_card.supported |= GPU_FANSPEED_MONITORING_SUPPORTED;
	    nv_card.get_fanspeed = nv40_get_fanspeed;
	    nv_card.set_fanspeed = nv40_set_fanspeed;
	}
	else if(((nv_card.arch & (NV43 | NV44)) && (nv_card.PMC[0x15f4/4] & 0x80000000)) && !(nv_card.supported & I2C_FANSPEED_MONITORING_SUPPORTED))
	{
	    nv_card.supported |= GPU_FANSPEED_MONITORING_SUPPORTED;
	    nv_card.get_fanspeed = nv43_get_fanspeed;
	    nv_card.set_fanspeed = nv43_set_fanspeed;
	}
    }
    else if(nv_card.gpu == NFORCE)
    {
	/* Only support gpu overclocking as the memory is (shared) system memory */
	nv_card.supported |= GPU_OVERCLOCKING_SUPPORTED;
    	nv_card.get_gpu_speed = get_gpu_speed;
	nv_card.set_gpu_speed = set_gpu_speed;
	nv_card.get_memory_speed = nforce_get_memory_speed;
	nv_card.set_memory_speed = dummy;
        nv_card.reset_gpu_speed = reset_gpu_speed;
        nv_card.reset_memory_speed = dummy;
    }
    /* All other cards including the GeforceFX 5200 and GeforceFX 5500 */
    else
    {
	nv_card.supported |= (GPU_OVERCLOCKING_SUPPORTED | MEM_OVERCLOCKING_SUPPORTED);
	nv_card.get_gpu_speed = get_gpu_speed;
    	nv_card.set_gpu_speed = set_gpu_speed;
	nv_card.get_memory_speed = get_memory_speed;
	nv_card.set_memory_speed = set_memory_speed;
        nv_card.reset_gpu_speed = reset_gpu_speed;
        nv_card.reset_memory_speed = reset_memory_speed;
    }

    /* Mobile GPU check; we don't want to overclock those unless the user wants it */
    if(nv_card.gpu == MOBILE)
    {
	/* Don't support overclocking as laptops aren't made for it; the user should be able to override this for atleast underclocking */
	nv_card.supported = ~(~nv_card.supported | GPU_OVERCLOCKING_SUPPORTED | MEM_OVERCLOCKING_SUPPORTED);
    }

    /* Set the speed range */
    set_speed_range();

    return 1;
}

void set_state(int state)
{
    nvclock.card[nv_card.number].state = state;
    nv_card.state = state;
}

void unset_card()
{
    nv_card.number=-1;
    unmap_mem();
}
