/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

//
// Kernel module for NW802 based webcams
//
//  for more informations : nw802.sourceforge.net
//
 
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/usb.h>

#include "usbvideo.h"


// ============================================================================
// Module Description
// ============================================================================

MODULE_AUTHOR("Munaut Sylvain <tnt@246tNt.com>");
MODULE_DESCRIPTION("Driver for the NW802 webcam chip");
MODULE_LICENSE("GPL");


// ============================================================================
// Typedefs, struct, defines, ...
// ============================================================================

// Constants

#define ENABLE_DEBUG

#define	MAX_NW802CAMS	1
#define MAX_VEIO_LEN	64
#define JPGL_HEADERLEN	8

#define DIVIO_VENDORID	0x06A5
#define NW802_PRODUCTID	0xD001


// Macros

#define INFO(fmt,args...)	printk( KERN_INFO "nw802.c: " fmt "\n", ## args )
#define ERR(fmt,args...)	printk( KERN_ERR "nw802.c: " fmt "\n" , ## args )
#ifdef ENABLE_DEBUG
#define DEBUG(level, fmt, args...)						\
	do {									\
		if ( debug >= level )						\
			printk( KERN_DEBUG "nw802.c: " fmt "\n", ## args );	\
	} while(0)
#else
#define DEBUG(level, fmt, args...)
#endif

#define LIMIT_0_255(x)	(((x) < 0) ? 0 : (((x) > 255) ? 255 : (x)))

#define NW802_T(uvd) ((nw802_t *)((uvd)->user_data))


// Structs

typedef struct		// This structure lives in uvd_t->user field.
{
	unsigned char veio_buf[MAX_VEIO_LEN];	// Buffer for vendor usb I/O
} nw802_t;


		// Allow us to read bit by bit in a RingQueue_t. We start by the
typedef struct	// MSB and consider data as Little Endian 16bits word
{
	int cur_bit;
	unsigned int cur_data;
	RingQueue_t *rq;
} rqBitReader_t;


// Enums
enum
{
	SIZE_160x120 = 0,
	SIZE_176x144,
	SIZE_320x240,
	SIZE_352x288,
	SIZE_640x480,
	SIZE_END
};

// ============================================================================
// Global vars
// ============================================================================

// Module parameters
#ifdef ENABLE_DEBUG
static int debug = 0;		// The debugging level of our module
static int debug_uv = 0;	// The debugging level of USB video
#endif

static int size = SIZE_320x240;


// Internal vars
static usbvideo_t *nw802_cams = NULL;

static int canvasX = 320;
static int canvasY = 240;


// ============================================================================
// Module parameters, options
// ============================================================================

#ifdef ENABLE_DEBUG
MODULE_PARM(debug, "i");
MODULE_PARM_DESC(debug, "Debug level: 0-5 (default=0)");
MODULE_PARM(debug_uv, "i");
MODULE_PARM_DESC(debug_uv, "Debug level of USB Video: 0-2 (default=0)");
#endif

MODULE_PARM(size, "i");
MODULE_PARM_DESC(size, "Video size: 0=160x120 1=176x144 2=320x240 4=352x288 5=640x580 (default=2)" );


// ===========================================================================
// RingQueue bit reader
// ===========================================================================

static void rqBR_init( rqBitReader_t *br, RingQueue_t *rq )
{
	br->cur_bit = 15;
	br->cur_data =	RING_QUEUE_PEEK( rq, 2 )	| 
			RING_QUEUE_PEEK( rq, 3 ) << 8	|
			RING_QUEUE_PEEK( rq, 0 ) << 16	|
			RING_QUEUE_PEEK( rq, 1 ) << 24	;
	RING_QUEUE_DEQUEUE_BYTES( rq, 2 );
	br->rq = rq;
}

static int rqBR_readBit( rqBitReader_t *br )
{
	int b;
	b = ( br->cur_data >> ( 16 + br->cur_bit ) ) & 0x01;
	if ( --br->cur_bit < 0 )
	{
		// No more bit, need to read more data
		br->cur_data <<= 16;
		br->cur_data |= RING_QUEUE_PEEK( br->rq, 2 )	 |
				RING_QUEUE_PEEK( br->rq, 3 ) << 8;
		RING_QUEUE_DEQUEUE_BYTES( br->rq, 2 );	// FIXME May be check
		br->cur_bit = 15;			// for data availability ?
	}
	return b;
}

static int rqBR_readBits( rqBitReader_t *br, int n )
{
	int rv = 0;
	rv = ( br->cur_data >> ( 17 + br->cur_bit - n ) ) & ( 0xFFFF >> ( 16 - n ) );
	if ( ( br->cur_bit -= n	) < 0 )
	{
		// No more bits, need to read more
		br->cur_data <<= 16;
		br->cur_data |= RING_QUEUE_PEEK( br->rq, 2 )	 |
				RING_QUEUE_PEEK( br->rq, 3 ) << 8;
		RING_QUEUE_DEQUEUE_BYTES( br->rq, 2 );	// FIXME May be check
		br->cur_bit += 16;			// for data availability ?
	}
	
	return rv;
}

// ============================================================================
// Jpeg-Lite stuff
// ============================================================================

// Some helpers

static int jpgl_readAC( rqBitReader_t *br, int *run, int *amp )
{
	// Some table used for decoding
	#define VLC_NUM	28
	static const int vlc_len[] = 
		{ 2, 2, 3, 3, 4, 5, 5, 6, 6, 6, 6, 7, 7, 7, 7, 7,
		  8 ,8 ,8 ,9, 9, 9, 10, 10, 10, 10, 10, 10 };
	static const int vlc_run[] =
		{ 0, 0, 0, 1, 0, 2, 3, 1, 0, 4, 0, 5, 1, 0, -1, -2,
		  2, 6, 0, 3, 1, 0, 1, 0, 7, 2, 0, 8 };
	static const int vlc_amp[] =
		{ 0, 1, 2, 1, 3, 1, 1, 2, 4, 1 ,5 ,1 ,3 ,6, -1, -2,
		  2, 1, 7, 2, 4, 8, 5, 9, 1 ,3, 10, 1 };
	static const int vlc_cod[] =
		{ 0x000, 0x002, 0x003, 0x006, 0x00E, 0x008, 0x00B, 0x012,
		  0x014, 0x03D, 0x03E, 0x078, 0x079, 0x07E, 0x026, 0x027,
		  0x054, 0x057, 0x0FF, 0x0AA, 0x0AC, 0x1FC, 0x156, 0x157,
		  0x15A, 0x15B, 0x3FA, 0x3FB };

	// Now read bit by bit and try to find a match
	int pos = 0;
	int cur_cod = 0;
	int cur_len = 0;
	
	while ( cur_len < vlc_len[VLC_NUM-1] )
	{
		// Read a bit
		cur_cod = ( cur_cod << 1 ) | ( rqBR_readBit(br) );
		cur_len++;

		// Is there a match ?
		while ( (pos < VLC_NUM) & ( cur_len == vlc_len[pos] ) )
		{
			if ( vlc_cod[pos] == cur_cod )
			{
				// A match has been found
				if ( vlc_amp[pos] == 0 )	// EOB
					return 0;
				else if ( vlc_amp[pos] < 0 )	// Special cases
				{
					cur_cod = rqBR_readBits(br, 9);
					if ( vlc_amp[pos] == -1 )
					{	// 0100110srrraaaaa
						*run = ( cur_cod >> 5 ) & 0x07;
						*amp = ( cur_cod & 0x100) ?
							-(cur_cod&0x1F) : (cur_cod&0x1F);
					}
					else	// 0100111srrrraaaa
					{
						*run = ( cur_cod >> 4 ) & 0x0F;
						*amp = ( cur_cod & 0x100) ?
							-(cur_cod&0x0F) : (cur_cod&0x0F);
					}
				}
				else				// Normal
				{
					*run = vlc_run[pos];
					*amp = rqBR_readBit(br) ?
						-vlc_amp[pos] : vlc_amp[pos];
				}

				return 1;
			}
			pos++;
		}
	}
	
	// Nothing found ...
	return 0;
}

static void jpgl_lineIDCT( int *base, int dir )
{
	int yw0, yw1, yw2, yw3;
	
	dir <<= 1;
	yw0 = base[0];
	yw1 = base[1 << dir];
	yw2 = base[2 << dir];
	yw3 = base[3 << dir];

	base[0]		= yw0 + ( (  74804 * yw1 + 60454 * yw2 +  37402 * yw3 ) >> 16 );
	base[1 << dir]	= yw0 + ( (  38386 * yw1 + 74898 * yw2 - 111878 * yw3 ) >> 16 );
	base[2 << dir]	= yw0 + ( ( -38386 * yw1 - 74898 * yw2 + 111878 * yw3 ) >> 16 );
	base[3 << dir]  = yw0 + ( ( -74804 * yw1 + 60454 * yw2 -  37402 * yw3 ) >> 16 );
}


static int jpgl_decodeBlock( rqBitReader_t *br, int *block, int *old_dc )
{
	int coeff[16];		// Coefficient matrix
	int t;			// Type of DC ( 0=direct 1=differential )
	int dc;			// DC value
	int q;			// Quantifier factor

	int run, amp, cc;	// Used in coeff reading
	
	// Read block header
	q = rqBR_readBits( br, 2 );
	t = rqBR_readBit( br );

	if ( t )
	{
		// Differential mode
		dc = rqBR_readBits( br, 5 );
		if ( dc & 0x10 )
			dc |= ~0x1F;

		dc += *old_dc;
	}
	else
	{
		// Direct mode
		dc = rqBR_readBits( br, 8 );
		if ( dc & 0x80 )
			dc |= ~0xFF;
	}
	
	*old_dc = dc;
	
	// Build coeffitient matrix
	memset( coeff, 0x00, 16*sizeof(int) );
	coeff[0] = dc;
	cc = 0;
	while ( jpgl_readAC( br, &run, &amp ) )
	{
		cc += run + 1;
		if ( cc > 15 )
		{
			DEBUG( 3, "Warning, too many AC coeff !" );
			break;
		}
		coeff[cc] = amp;
	}
	
	// Dequantification & DeZigzag
	block[0] = coeff[0];
	block[1] = coeff[1] << q;
	block[2] = coeff[5] << q;
	block[4] = coeff[2] << q;
	block[5] = coeff[4] << q;
	block[8] = coeff[3] << q;
	q++;	// Switch to zone 1
	block[3] = coeff[6] << q;
	block[6] = coeff[7] << q;
	block[7] = coeff[12] << q;
	block[9] = coeff[8] << q;
	block[10] = coeff[11] << q;
	block[11] = coeff[13] << q;
	block[12] = coeff[9] << q;
	block[13] = coeff[10] << q;
	block[14] = coeff[14] << q;
	block[15] = coeff[15] << q;
	
	// iDCT
	jpgl_lineIDCT( block +  0, 0 );
	jpgl_lineIDCT( block +  4, 0 );
	jpgl_lineIDCT( block +  8, 0 );
	jpgl_lineIDCT( block + 12, 0 );
	
	jpgl_lineIDCT( block + 0, 1 );
	jpgl_lineIDCT( block + 1, 1 );
	jpgl_lineIDCT( block + 2, 1 );
	jpgl_lineIDCT( block + 3, 1 );

	return !( t || dc );
}

// Main functions

static int jpgl_processFrame( RingQueue_t *rq, unsigned char *framebuffer )
{
	rqBitReader_t br;
	int x, y;
	
	int yb[16*4];		// Blocks
	int ib[16];
	int cb[16];

	int dc_y;		// DC Values
	int dc_i;
	int dc_c;
	
	int xs, ys;		// Used for copying data into frame buffer
	int rc, gc, bc, yc, ic, cc;// Used for YIC -> RGB conversion
	
	// Prepare a bit reader
	rqBR_init( &br, rq );
	
	// Now read all block. We got YYYYIC. Actually it's not really YIC, nor
	// YUV ... It's weird ... but let's call it YIC because it's the closer
	// format
	for ( y=0 ; y<canvasY ; y+=4 )
	{
		dc_y = dc_i = dc_c = 0;
		
		for ( x=0 ; x<canvasX ; x+=16 )
		{
			// Read blocks. If the first block of the line has
			// DC = 0 with direct mode => ABORT ! ( stream error )
			if ( jpgl_decodeBlock( &br, yb, &dc_y ) && ( x == 0x00 ) )
				return 1;
			jpgl_decodeBlock( &br, yb + 16, &dc_y );
			jpgl_decodeBlock( &br, yb + 32, &dc_y );
			jpgl_decodeBlock( &br, yb + 48, &dc_y );
			jpgl_decodeBlock( &br, ib, &dc_i );
			jpgl_decodeBlock( &br, cb, &dc_c );

			// Now, need to copy theses in the frame buffer
			// FIXME Need to rewrite this
			for ( ys=0 ; ys<4 ; ys++ )
				for ( xs=0 ; xs<16 ; xs++ )
				{
					yc = LIMIT_0_255( yb[ ((xs >> 2) * 16) + (xs & 0x03) + (ys<<2) ] + 128 );
					ic = LIMIT_0_255( ib[ (ys << 2) + (xs >> 2 ) ] + 128 );
					cc = LIMIT_0_255( cb[ (ys << 2) + (xs >> 2 ) ] + 128 );
					
					rc = yc + ( ( 62652 * ( cc - 127 ) + 40698 * ( ic - 127 ) ) >> 16 );
					gc = yc + ( ( -42402 * ( cc - 127 ) + 17826 * ( ic - 127 ) ) >> 16 );
					bc = yc + ( ( -72417 * ( cc - 127 ) + 111542 * ( ic - 127 ) ) >> 16 );
					
					framebuffer[3*((y+ys)*canvasX+x+xs)+0]	= LIMIT_0_255(bc);
					framebuffer[3*((y+ys)*canvasX+x+xs)+1]	= LIMIT_0_255(gc);
					framebuffer[3*((y+ys)*canvasX+x+xs)+2]	= LIMIT_0_255(rc);
				}
		}
	}
	
	return 0;
}

static int jpgl_findHeader( RingQueue_t *rq, int ofs )
{
	int header_pos = 0;
	int rq_ofs = ofs;
	int rq_len;
	
	static const unsigned char header_mask[JPGL_HEADERLEN] = 
		{ 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF };
	unsigned char header_data[JPGL_HEADERLEN] =
		{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF };
	
	// Adjust variable header bytes
	header_data[2] = (unsigned char)( canvasY >> 2 );
	header_data[3] = (unsigned char)( canvasX >> 2 );
	
	// Look for a complete header in the RingQueue, starting at ofs
	rq_len = RingQueue_GetLength( rq );

	while ( (rq_ofs < rq_len) && (header_pos < JPGL_HEADERLEN) )
	{
		if ( ( RING_QUEUE_PEEK(rq, rq_ofs) & header_mask[header_pos] ) ==
		     header_data[header_pos] )
			header_pos++;
		else
			header_pos = 0;

		rq_ofs++;
	}

	return ( header_pos == JPGL_HEADERLEN ) ? rq_ofs - JPGL_HEADERLEN : -1; 
}


// ============================================================================
// Module options related funcs
// ============================================================================

static void nw802_validate_params()
{
	#ifdef ENABLE_DEBUG
	RESTRICT_TO_RANGE( debug, 0, 5 );
	RESTRICT_TO_RANGE( debug_uv, 0, 2 );
	#endif

	RESTRICT_TO_RANGE( size, 0, SIZE_END - 1 );
}

static int nw802_size_to_videosize( int size )
{
	switch (size)
	{
		case SIZE_160x120:
			return VIDEOSIZE( 160, 120 );
		case SIZE_176x144:
			return VIDEOSIZE( 176, 144 );
		case SIZE_320x240:
			return VIDEOSIZE( 320, 240 );
		case SIZE_352x288:
			return VIDEOSIZE( 352, 288 );
		case SIZE_640x480:
			return VIDEOSIZE( 640, 480 );
		default:
			ERR( "Invalid video size !" );
			return VIDEOSIZE( 320, 240 );
	}
}


// ============================================================================
// USB Video driver stuff
// ============================================================================

// Other functions

static int nw802_vendor_send( uvd_t *uvd, int idx, int val, void *buf, int len )
{
	int rv;
	
	RESTRICT_TO_RANGE( len, 0, MAX_VEIO_LEN );
	memcpy( (void*)NW802_T(uvd)->veio_buf, buf, len );
	
	rv = usb_control_msg( uvd->dev,
			      usb_sndctrlpipe( uvd->dev, 0 ),
			      0,
			      USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_ENDPOINT,
			      val,
			      idx,
			      (void*)NW802_T(uvd)->veio_buf,
			      len,
			      0.1 * HZ );

	DEBUG( 2, "Vendor send report : index=%i value=%i rv=%i", idx, val, rv );
	
	return rv;
}

static int nw802_init_camera( uvd_t *uvd )
{
	// TODO Need to rewrite this & understand what it means ...
	int i;
	
	static const
	int index[] = { 0x0406, 0x0000, 0x0040, 0x0080, 0x0200, 0x0300, 0x0400, 0x0600,
			0x0800, 0x1000, 0x1040, 0x1080, 0x0200, 0x101a, 0x1000, 0x0000,
			0x1085, 0x101b, 0x1011, 0x101d, 0x100e, 0x1041, 0x1003, 0x100f,
			0x1003, 0x1041, 0x100b, 0x100d, 0x100c, 0x0406, 0x0404
		      };
	
	static const
	int value[] = {	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
			0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
			0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
			0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000
		      };
	
	static const
	int len[] = {	0x0001, 0x0040, 0x0040, 0x001f, 0x0012, 0x0003, 0x0007, 0x0002,
			0x0015, 0x0040, 0x0040, 0x001b, 0x0011, 0x0001, 0x0001, 0x0001,
			0x0008, 0x0002, 0x0008, 0x0008, 0x0001, 0x0011, 0x0001, 0x0002,
			0x0001, 0x0011, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001
		    };
	
	static const
	unsigned char buffer[][MAX_VEIO_LEN] =	{
		{ 0x04 },
		{ 0x10, 0x00, 0x00, 0xf9, 0x02, 0x10, 0x00, 0x4d,
		  0x0f, 0x1f, 0x00, 0x0d, 0x02, 0x01, 0x00, 0x19,
		  0x00, 0x01, 0x00, 0x19, 0x00, 0x01, 0x00, 0x19, 
		  0x00, 0x01, 0x00, 0x19, 0x00, 0xce, 0x00, 0xf4,
		  0x05, 0x3e, 0x00, 0x86, 0x00, 0x3e, 0x00, 0x86,
		  0x00, 0x3e, 0x00, 0x86, 0x00, 0x01, 0x00, 0x01,
		  0x00, 0x56, 0x00, 0x9e, 0x00, 0x56, 0x00, 0x9e,
		  0x00, 0x56, 0x00, 0x9e, 0x00, 0x01, 0x00, 0x01 },
		{ 0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
		  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
		  0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
		  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
		  0x00, 0xca, 0x03, 0x46, 0x04, 0xca, 0x03, 0x46,
		  0x04, 0x10, 0x00, 0x36, 0x00, 0xd2, 0x00, 0xee,
		  0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0xf0,
		  0x00, 0x3e, 0x00, 0xaa, 0x00, 0x88, 0x00, 0x2e },
		{ 0xb4, 0x6f, 0x3f, 0x0f, 0x88, 0x20, 0x68, 0x00,
		  0x00, 0x00, 0x00, 0x00, 0xa8, 0x08, 0x00, 0x11, 
		  0x00, 0x0c, 0x02, 0x01, 0x00, 0x16, 0x00, 0x94,
		  0x00, 0x10, 0x06, 0x08, 0x00, 0x18, 0x00 },
		{ 0x78, 0xa0, 0x9e, 0x78, 0xa0, 0x00, 0x00, 0x00,
		  0x00, 0xf0, 0x18, 0x0b, 0x06, 0x62, 0x82, 0xa0,
		  0x40, 0x20 },
		{ 0x03, 0x00, 0x00 },
		{ 0x01, 0x10, 0x00, 0x00, 0x00, 0x21, 0x00 },
		{ 0x09, 0x99 },
		{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x00, 0x00, 0x00, 0x00, 0x00 } ,
		{ 0xa1, 0x02, 0x80, 0x00, 0x12, 0x00, 0x00, 0x00,
		  0x00, 0x00, 0x00, 0x10, 0x10, 0x10, 0x08, 0x0a,
		  0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x00, 0x49, 0x13, 0x00, 0x00, 0xc0, 0x00, 0x14,
		  0x02, 0x00, 0x01, 0x00, 0x00, 0x20, 0x00, 0x00,
		  0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x10, 0x08,
		  0x03, 0x00, 0x00, 0x00, 0x00, 0x20, 0x10, 0x06,
		  0xf7, 0xee, 0x1c, 0x1c, 0xe9, 0xfc, 0x10, 0x80 },
		{ 0x80, 0x00, 0x05, 0x35, 0x5e, 0x78, 0x8b, 0x99,
		  0xa4, 0xae, 0xb5, 0xbc, 0xc1, 0xc6, 0xc9, 0xcc,
		  0xcf, 0xd0, 0x00, 0x11, 0x22, 0x32, 0x43, 0x54,
		  0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3, 0xd2,
		  0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32, 0x43,
		  0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3,
		  0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32,
		  0x43, 0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3 },
		{ 0xc3, 0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x00, 0x00,
		  0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x05, 0x82,
		  0x02, 0xe4, 0x01, 0x40, 0x01, 0xf0, 0x00, 0x40,
		  0x01, 0xf0, 0x00 },
		{ 0x3c, 0x50, 0x9e, 0x3c, 0x50, 0x00, 0x00, 0x00,
		  0x00, 0x78, 0x3f, 0x10, 0x02, 0xf2, 0x8f, 0x78,
		  0x40 },
		{ 0x00 },
		{ 0xad },
		{ 0x08 },
		{ 0x00, 0x00, 0x3f, 0x01, 0x00, 0x00, 0xef, 0x00 },
		{ 0x00, 0x00 },
		{ 0x51, 0x00, 0xf0, 0x00, 0x3d, 0x00, 0xb4, 0x00 },
		{ 0x00, 0xa0, 0x00, 0xa0, 0x00, 0xa0, 0x00, 0xa0 },
		{ 0x08 },
		{ 0x00, 0x0e, 0x35, 0x4f, 0x62, 0x71, 0x7f, 0x8b,
		  0x96, 0xa0, 0xa9, 0xb2, 0xbb, 0xc3, 0xca, 0xd2,
		  0xd8 },
		{ 0x00 },
		{ 0x14, 0x14 },
		{ 0x0c },
		{ 0x00, 0x0e, 0x35, 0x4f, 0x62, 0x71, 0x7f, 0x8b,
		  0x96, 0xa0, 0xa9, 0xb2, 0xbb, 0xc3, 0xca, 0xd2,
		  0xd8 },
		{ 0x10 },
		{ 0x10 },
		{ 0x2a },
		{ 0x03 },
		{ 0x00 }
					};

	// Select alternate setting to the active one
	usb_set_interface( uvd->dev, 0x00, uvd->ifaceAltActive );

	// Send all the packets 
	for ( i=0 ; i < 31 ; i++ )
		if ( nw802_vendor_send( uvd, index[i], value[i],
					(void*)buffer[i], len[i] ) < 0 )
			return -1;

	return 0;
}

static void nw802_configure_video( uvd_t *uvd )
{
	if ( !uvd )
		return;

	// Picture settings ( FIXME: Take them as args ??? )
	memset( &uvd->vpic, 0x00, sizeof(uvd->vpic) );
	memset( &uvd->vpic_old, 0x00, sizeof(uvd->vpic_old) );
	
	uvd->vpic.colour = 128 << 8;
	uvd->vpic.hue = 128 << 8;
	uvd->vpic.brightness = 128 << 8;
	uvd->vpic.contrast = 128 << 8;
	uvd->vpic.whiteness = 128 << 8;
	uvd->vpic.depth = 24;
	uvd->vpic.palette = VIDEO_PALETTE_RGB24;

	// Video capabilities & channel setting
	memset( &uvd->vcap, 0, sizeof(uvd->vcap) );
	strcpy( uvd->vcap.name, "NW802 base camera" );
	        
	uvd->vcap.type = VID_TYPE_CAPTURE;
	uvd->vcap.channels = 1;
	uvd->vcap.audios = 0;
	uvd->vcap.maxwidth = VIDEOSIZE_X(uvd->canvas);
	uvd->vcap.maxheight = VIDEOSIZE_Y(uvd->canvas);
	uvd->vcap.minwidth = 4;		// FIXME Don't really know
	uvd->vcap.minheight = 4;	// what these values means ...
								

	memset( &uvd->vchan, 0, sizeof(uvd->vchan) );
	strcpy( uvd->vchan.name, "Camera view" );

	uvd->vchan.flags = 0;
	uvd->vchan.tuners = 0;
	uvd->vchan.channel = 0;
	uvd->vchan.type = VIDEO_TYPE_CAMERA;
}


// Call backs

static void nw802_processIsocData(uvd_t *uvd, usbvideo_frame_t *frame)
{
	int rv;
	
	DEBUG( 5, "nw802_processIsocData" );

	// Try to find the first header
	rv = jpgl_findHeader( &uvd->dp, 0 );
	if ( rv >= 0 )
	{
		// Remove junk data preceding the header
		if ( rv )
			RING_QUEUE_DEQUEUE_BYTES(&uvd->dp, rv);

		// Try to find another header
		rv = jpgl_findHeader( &uvd->dp, 1 );
		if ( rv > 0 )
		{
			// Ok, we found two headers, so between them, there is
			// a frame waiting for decoding
			RING_QUEUE_DEQUEUE_BYTES(&uvd->dp, JPGL_HEADERLEN);
			DEBUG( 4, "Frame ready for decoding" );
			
			if ( !jpgl_processFrame(&uvd->dp, frame->data) )
			{
				// Frame processing was sucessful
				frame->frameState = FrameState_Done;
				uvd->curframe = -1;
				uvd->stats.frame_num++;
			}
			else
				DEBUG(3, "Invalid frame detected !");
		}
	}
}

static int nw802_setupOnOpen(uvd_t *uvd)
{
	DEBUG( 1, "nw802_setupOnOpen(...)" );
	
	// TODO : Nothing more todo here ???
	nw802_init_camera( uvd );
	
	return 0;
}

static void *nw802_probe( struct usb_device *dev,
			  unsigned int ifnum,
			  const struct usb_device_id *devid )
{
	uvd_t *uvd = NULL;
	int nas, i;
	int actSetting = -1;
	int inactSetting = -1;
	int maxPS = 0;
	unsigned char video_ep = 0;
	
	DEBUG( 1, "nw802_probe(...)" );

	// We don't want multiple configuration camera
	if ( dev->descriptor.bNumConfigurations != 1 )
		return NULL;
	
	// Check Vendor & Product ID
	if ( ( dev->descriptor.idVendor != DIVIO_VENDORID ) ||
	     ( dev->descriptor.idProduct != NW802_PRODUCTID ) )
	     return NULL;
		
	// Ok it's a supported cam ( at least seems to )
	INFO( "Divio NW802 based webcam found !" );
	
	// Let's find endpoint, altsettings, ... and validate all this !
	nas = dev->actconfig->interface[ifnum].num_altsetting;
	DEBUG( 2, "Number of alternate settings : %d", nas );
	
	for ( i=0 ; i<nas ; i++ )
	{
		const struct usb_interface_descriptor *interface;
		const struct usb_endpoint_descriptor *endpoint;
		
		interface = &dev->actconfig->interface[ifnum].altsetting[i];
		if ( interface->bNumEndpoints != 3 )
		{
			ERR( "Interface %u Alt %i has %i endpoints!", 
			     ifnum, i, interface->bNumEndpoints );
			return NULL;
		}
		
		endpoint = &interface->endpoint[1];
		
		if ( video_ep == 0 )
			video_ep = endpoint->bEndpointAddress;
		else if ( video_ep != endpoint->bEndpointAddress )
		{
			ERR( "Alternate settings have different endpoint addresses!");
			return NULL;	
		}		
		
		if ( ( endpoint->bmAttributes & USB_ENDPOINT_XFERTYPE_MASK ) !=
		     USB_ENDPOINT_XFER_ISOC )
		{
			ERR( "Interface %u Alt %i has non-ISO endpoint 0!", ifnum, i );
			return NULL;    	
		}
		
		if ( ( endpoint->bEndpointAddress & USB_ENDPOINT_DIR_MASK ) == USB_DIR_OUT )
		{
			ERR( "Interface %u Alt %i has ISO OUT endpoint 0!", ifnum, i );
			return NULL;    	
		}

		if ( endpoint->wMaxPacketSize == 0 )
		{
			if ( inactSetting < 0 )
				inactSetting = i;
			else
			{
				ERR( "More thant one inactive alt. setting!" );
				return NULL;	
			}
		}
		else
		{
			if ( actSetting < 0 )
			{
				actSetting = i;
				maxPS = endpoint->wMaxPacketSize;
				DEBUG( 2, "Active setting=%i maxPS=%i", i, maxPS );
			}
			else if ( maxPS < endpoint->wMaxPacketSize )
			{
				// This one is better
				actSetting = i;	
				maxPS = endpoint->wMaxPacketSize;
				DEBUG( 2, "Better active setting=%i maxPS=%i", i, maxPS );
			}
		}		
	}
	
	if ( ( maxPS == 0 ) || ( actSetting < 0 ) || ( inactSetting < 0 ) )
	{
		ERR( "No suitable endpoits! Failed to recognize camera!" );
		return NULL;
	}
	
	// Check the module options
	nw802_validate_params();
	
	// All is Ok, let's register a video device
	MOD_INC_USE_COUNT;	// Code below may sleep, use this as a lock
	
	uvd = usbvideo_AllocateDevice(nw802_cams);
	if ( uvd )
	{
		// Let's setup our freshly allocated usb video driver
		#ifdef ENABLE_DEBUG
		uvd->flags = ( debug >= 3 ) ? FLAGS_DISPLAY_HINTS | FLAGS_OVERLAY_STATS : 0;
		uvd->debug = debug_uv;
		#else
		uvd->flags = 0;
		uvd->debug = 0;
		#endif
		
		uvd->dev = dev;
		uvd->iface = ifnum;
		uvd->ifaceAltInactive = inactSetting;
		uvd->ifaceAltActive = actSetting;
		uvd->video_endp = video_ep;
		uvd->iso_packet_len = maxPS;
		uvd->paletteBits = 1L << VIDEO_PALETTE_RGB24;
		uvd->defaultPalette = VIDEO_PALETTE_RGB24;
		uvd->canvas = nw802_size_to_videosize( size );
		uvd->videosize = uvd->canvas;
		
		// Init the nw802 specific part of uvd & global var
		canvasX = VIDEOSIZE_X( uvd->canvas );
		canvasY = VIDEOSIZE_Y( uvd->canvas );
		
		// Configure video & register video device
		nw802_configure_video( uvd );

		if ( usbvideo_RegisterVideoDevice(uvd) )
		{
			ERR( "Failed to register video device!" );
			uvd = NULL;
		}
	}
	else
		ERR( "Failed to allocate usbvideo device!" );
		
	MOD_DEC_USE_COUNT;	// Release the 'lock'
	
	return uvd;
}




// ============================================================================
// Kernel Module init & exit
// ============================================================================

#if defined(usb_devide_id_ver)
static __devinitdata struct usb_device_id nw802_usb_ids[] =
	{
		{ USB_DEVICE( DIVIO_VENDORID, NW802_PRODUCTID ) },
		{}	// End entry
	};

MODULE_DEVICE_TABLE( usb, nw802_usb_ids );
#endif

static int __init nw802_init()
{
	int rv;
	
	// Setup callbacks
	usbvideo_cb_t cbTbl;
	memset( &cbTbl, 0, sizeof(cbTbl) );

	cbTbl.probe = nw802_probe;
	cbTbl.setupOnOpen = nw802_setupOnOpen;
	cbTbl.processData = nw802_processIsocData;
	
	// Register usbvideo driver
	rv = usbvideo_register( &nw802_cams,
				MAX_NW802CAMS,
				sizeof(nw802_t),
				"nw802",
				&cbTbl,
				THIS_MODULE );	// FIXME 2.5.x kernels also need
						// the nw802_usb_ids as a 7th param
	
	// Status
	if ( !rv )
		INFO( "Module loaded" );
	else
		ERR( "Error loading module. Errcode = %i", rv );
		
	return rv;
}

static void __exit nw802_exit()
{
	usbvideo_Deregister( &nw802_cams );
	INFO( "Module unloaded" );
}

module_init( nw802_init );
module_exit( nw802_exit );


//
// gcc -O2 -D__KERNEL__ -DMODULE -Wall -DMODVERSIONS -nostdinc -I /usr/src/linux/include -I /usr/lib/gcc-lib/i386-slackware-linux/3.0.4/include/ -include /usr/src/linux/include/linux/modversions.h -c -o nw802.o nw802.c
//
