/*
 *  Nextview EPG GUI: Database statistics and main window status line
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License Version 2 as
 *  published by the Free Software Foundation. You find a copy of this
 *  license in the file COPYRIGHT in the root directory of this release.
 *
 *  THIS PROGRAM IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL,
 *  BUT WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF
 *  MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *
 *  Description:
 *
 *    Implements pop-up windows with statistics about the current state
 *    of the databases and the acquisition process.  They can be opened
 *    separately for the browser and acquisition database and are updated
 *    dynamically while acquisition is running.
 *
 *    The statistics window is generated by Tcl/Tk procedures. It consists
 *    of two vertically separated parts: the upper part contains db stats,
 *    such as number of PI in db, percentage of expired PI etc. The lower
 *    part describes the acq state, e.g. number of received AI blocks etc.
 *    Both parts consist of a canvas widget (graphics output) at the left
 *    and a message widget (multi-line, fixed-width string output) at the
 *    right.
 *
 *    This module also generates the status line at the bottom of the
 *    main window, which is a single line of text which is assembled from
 *    carefully selected db stats and acq state information.
 *
 *  Author: Tom Zoerner
 *
 *  $Id: statswin.c,v 1.65 2003/10/05 19:32:09 tom Exp tom $
 */

#define DEBUG_SWITCH DEBUG_SWITCH_EPGUI
#define DPRINTF_OFF

#include <string.h>
#include <time.h>

#include <tcl.h>
#include <tk.h>

#include "epgctl/mytypes.h"
#include "epgctl/debug.h"

#include "epgvbi/ttxdecode.h"
#include "epgdb/epgblock.h"
#include "epgdb/epgdbfil.h"
#include "epgdb/epgdbif.h"
#include "epgctl/epgacqctl.h"
#include "epgctl/epgscan.h"
#include "epgctl/epgctxmerge.h"
#include "epgctl/epgacqclnt.h"
#include "epgui/menucmd.h"
#include "epgui/epgmain.h"
#include "epgui/uictrl.h"
#include "epgui/statswin.h"


// state of DbStatsWin window
static struct
{
   bool   open;
   bool   isForAcq;
   int    lastHistPos;
   time_t lastHistReset;
   Tcl_TimerToken updateHandler;
} dbStatsWinState[2];

const char * const dbswn[2] =
{
   ".dbstats_ui",
   ".dbstats_acq"
};


static void StatsWin_UpdateDbStatsWinTimeout( ClientData clientData );

// ----------------------------------------------------------------------------
// Update the history diagram in the db statistics window
//
static void StatsWin_UpdateHist( int target )
{
   const EPGDB_STATS * sv;
   uint idx;

   sv = EpgAcqCtl_GetAcqStats();
   if (sv != NULL)
   {
      if (dbStatsWinState[target].lastHistReset != sv->acqStartTime)
      {  // acquisition was reset -> clear history
         dbStatsWinState[target].lastHistReset = sv->acqStartTime;
         dbStatsWinState[target].lastHistPos   = STATS_HIST_WIDTH;

         if (dbStatsWinState[target].lastHistReset != 0)
         {
            sprintf(comm, "DbStatsWin_ClearHistory %s\n", dbswn[target]);
            eval_check(interp, comm);
         }
      }

      if ( (dbStatsWinState[target].lastHistPos != sv->histIdx) &&
           (sv->histIdx > 1) &&
           (sv->histIdx != STATS_HIST_WIDTH - 1) )
      {
         idx = (dbStatsWinState[target].lastHistPos + 1) % STATS_HIST_WIDTH;

         for (idx=0; idx < sv->histIdx; idx++)
         {
            sprintf(comm, "DbStatsWin_AddHistory %s %d %d %d %d %d %d\n",
                          dbswn[target], idx+1,
                          sv->hist[idx].expir,
                          sv->hist[idx].s1cur, sv->hist[idx].s1old,
                          sv->hist[idx].s2cur, sv->hist[idx].s2old);
            eval_check(interp, comm);
         }
         dbStatsWinState[target].lastHistPos = sv->histIdx;
      }
   }
   else
   {  // acq not running -> clear history
      sprintf(comm, "DbStatsWin_ClearHistory %s\n", dbswn[target]);
      eval_check(interp, comm);
   }
}

// ----------------------------------------------------------------------------
// Update the database statistics window
// - consists of two parts:
//   + the upper one describes the current state of the database, with a pie chart
//     and statistics of PI block counts against the numbers from the AI block
//   + the lower one describes the current acquisition state and a histroy diagram
//     of current and past PI block count percentages
//
static void StatsWin_UpdateDbStatsWin( ClientData clientData )
{
   const EPGDB_BLOCK_COUNT * count;
   EPGDB_BLOCK_COUNT myCount[2];
   EPGDB_CONTEXT * dbc;
   EPGACQ_DESCR acqState;
   EPGDB_STATS mySv;
   const EPGDB_ACQ_VPS_PDC * pAcqVpsPdc;
   const EPGDB_STATS *sv;
   const AI_BLOCK *pAi;
   uchar netname[30+1], datestr[25+1];
   uchar version, versionSwo;
   ulong duration;
   uint  cni;
   uint32_t  total, allVersionsCount, curVersionCount, obsolete;
   time_t acqMinTime[2];
   time_t lastAiUpdate;
   time_t now = time(NULL);
   uint target;

   target = PVOID2UINT(clientData);
   if (target == DB_TARGET_ACQ)
      dbc = pAcqDbContext;
   else if (target == DB_TARGET_UI)
      dbc = pUiDbContext;
   else
      dbc = NULL;

   if (dbStatsWinState[target].open)
   {
      dprintf1("StatsWin-UpdateDbStatsWin: for %s\n", ((target == DB_TARGET_UI) ? "ui" : "acq"));

      if ( (target == DB_TARGET_ACQ) || (pAcqDbContext == pUiDbContext) )
      {
         count = EpgAcqCtl_GetDbStats();
      }
      else
         count = NULL;
      if (count == NULL)
      {  // acq not running for this database -> display db stats only
         memset(myCount, 0, sizeof(myCount));
         if (dbc != NULL)
         {
            memset(acqMinTime, 0, sizeof(acqMinTime));
            EpgDbGetStat(dbc, myCount, acqMinTime, 0);
         }
         count = myCount;
      }

      if (EpgDbContextIsMerged(dbc) == FALSE)
      {
         // default values if database is empty
         strcpy(netname, "none yet");
         strcpy(datestr, "none yet");
         version = versionSwo = 0;
         cni = 0;

         if (dbc != NULL)
         {  // get provider's network name and db version from AI block
            EpgDbLockDatabase(dbc, TRUE);
            pAi = EpgDbGetAi(dbc);
            if (pAi != NULL)
            {
               strncpy(netname, AI_GET_NETWOP_NAME(pAi, pAi->thisNetwop), sizeof(netname) - 1);
               netname[sizeof(netname) - 1] = 0;
               lastAiUpdate = EpgDbGetAiUpdateTime(dbc);
               strftime(datestr, 25, "%H:%M:%S %a %d.%m.", localtime(&lastAiUpdate));
               version = pAi->version;
               versionSwo = pAi->version_swo;
               cni = AI_GET_CNI(pAi);
            }
            EpgDbLockDatabase(dbc, FALSE);
         }

         total            = count[0].ai + count[1].ai;
         obsolete         = count[0].expired + count[0].defective +
                            count[1].expired + count[1].defective;
         allVersionsCount = count[0].allVersions + count[1].allVersions + obsolete;
         curVersionCount  = count[0].curVersion + count[1].curVersion + obsolete;

         sprintf(comm, "%s.browser.stat configure -text \""
                       "EPG provider:     %s (CNI %04X)\n"
                       "last AI update:   %s\n"
                       "Database version: %d/%d\n"
                       "Blocks in AI:     %d (%d + %d swo)\n"
                       "Block count db:   %d (%d + %d swo)\n"
                       "current version:  %d (%d + %d swo)\n"
                       "filled:           %d%% / %d%% current version\n"
                       "expired stream:   %d%%: %d (%d + %d)\n"
                       "expired total:    %d%%: %d\n"
                       "defective blocks: %d%%: %d (%d + %d)"
                       "\"\n",
                       dbswn[target],
                       netname, cni,
                       datestr,
                       version, versionSwo,
                       total, count[0].ai, count[1].ai,
                       allVersionsCount,
                          count[0].allVersions + count[0].expired + count[0].defective,
                          count[1].allVersions + count[1].expired + count[1].defective,
                       curVersionCount,
                          count[0].curVersion + count[0].expired + count[0].defective,
                          count[1].curVersion + count[1].expired + count[1].defective,
                       ((total > 0) ? (int)((double)allVersionsCount * 100.0 / total) : 100),
                       ((total > 0) ? (int)((double)curVersionCount * 100.0 / total) : 100),
                       ((allVersionsCount > 0) ? ((int)((double)(count[0].expired + count[1].expired) * 100.0 / allVersionsCount)) : 0),
                          count[0].expired + count[1].expired, count[0].expired, count[1].expired,
                       ((allVersionsCount + count[0].extra > 0) ? ((int)((double)count[0].extra * 100.0 / (allVersionsCount + count[0].extra))) : 0),
                          count[0].extra,
                       ((allVersionsCount > 0) ? ((int)((double)(count[0].defective + count[1].defective) * 100.0 / allVersionsCount)) : 0),
                          count[0].defective + count[1].defective, count[0].defective, count[1].defective
                );
         eval_check(interp, comm);

         if (total > 0)
         {
            sprintf(comm, "DbStatsWin_PaintPie %s %d %d %d %d %d %d %d\n",
                          dbswn[target],
                          (int)((double)(count[0].defective + count[0].expired) / total * 359.9),
                          (int)((double)(count[0].curVersion + count[0].defective + count[0].expired) / total * 359.9),
                          (int)((double)(count[0].allVersions + count[0].defective + count[0].expired) / total * 359.9),
                          (int)((double)count[0].ai / total * 359.9),
                          (int)((double)(count[0].ai + count[1].defective + count[1].expired) / total * 359.9),
                          (int)((double)(count[0].ai + count[1].curVersion + count[1].defective + count[1].expired) / total * 359.9),
                          (int)((double)(count[0].ai + count[1].allVersions + count[1].defective + count[1].expired) / total * 359.9));
         }
         else
         {
            sprintf(comm, "DbStatsWin_ClearPie %s\n", dbswn[target]);
         }
         eval_check(interp, comm);
      }
      else
      {  // Merged database

         allVersionsCount = count[0].allVersions + count[0].expired + count[0].defective +
                            count[1].allVersions + count[1].expired + count[1].defective;

         sprintf(comm, "%s.browser.stat configure -text \""
                       "EPG Provider:     Merged database\n"
                       "Block count db:   %d\n"
                       "\nFor more info please refer to the\n"
                       "original databases (more statistics\n"
                       "will be added here in the future)\n"
                       "\"\n",
                       dbswn[target],
                       allVersionsCount
                );
         eval_check(interp, comm);

         sprintf(comm, "DbStatsWin_ClearPie %s\n", dbswn[target]);
         eval_check(interp, comm);
      }


      //  Acquisition status

      if ((target == DB_TARGET_ACQ) || (pAcqDbContext == pUiDbContext))
      {
         sv = EpgAcqCtl_GetAcqStats();
         if (sv == NULL)
         {
            memset(&mySv, 0, sizeof(mySv));
            sv = &mySv;
         }
         if (dbStatsWinState[target].isForAcq == FALSE)
         {
            sprintf(comm, "pack %s.acq -side top -anchor nw -fill both -after %s.browser\n",
                          dbswn[target], dbswn[target]);
            eval_check(interp, comm);
            dbStatsWinState[target].isForAcq = TRUE;
         }

         // display graphical database fill percentage histogramm
         StatsWin_UpdateHist(target);

         EpgAcqCtl_DescribeAcqState(&acqState);

         if ((sv->acqStartTime > 0) && (sv->acqStartTime <= now))
         {
            duration = now - sv->acqStartTime;
            if (duration == 0)
               duration = 1;
         }
         else
            duration = 0;

         sprintf(comm, "%s.acq.stat configure -text {", dbswn[target]);

         #ifdef USE_DAEMON
         if (acqState.isNetAcq)
         {
            EPGDBSRV_DESCR netState;
            EpgAcqClient_DescribeNetState(&netState);
            switch(netState.state)
            {
               case NETDESCR_ERROR:
                  sprintf(comm + strlen(comm), "Network state:    error\nError cause:      %s\n", ((netState.cause != NULL) ? netState.cause : "unknown"));
                  break;
               case NETDESCR_CONNECT:
                  sprintf(comm + strlen(comm), "Network state:    server contacted\n");
                  break;
               case NETDESCR_STARTUP:
                  sprintf(comm + strlen(comm), "Network state:    connect in progress\n");
                  break;
               case NETDESCR_LOADING:
                  sprintf(comm + strlen(comm), "Network state:    loading database\n");
                  break;
               case NETDESCR_RUNNING:
                  sprintf(comm + strlen(comm), "Network state:    connected (rx %ld bytes)\n", netState.rxTotal);
                  break;
               case NETDESCR_DISABLED:
               default:
                  sprintf(comm + strlen(comm), "Network state:    unconnected\n");
                  break;
            }
         }
         #endif

         sprintf(comm + strlen(comm), "Acq Runtime:      %02d:%02d\n",
                       (uint)(duration / 60), (uint)(duration % 60));

         pAcqVpsPdc = EpgAcqCtl_GetVpsPdc(VPSPDC_REQ_STATSWIN);
         if ((pAcqVpsPdc != NULL) && (pAcqVpsPdc->cni != 0))
         {
            if ( VPS_PIL_IS_VALID(pAcqVpsPdc->pil) )
            {
               sprintf(comm + strlen(comm), "Channel VPS/PDC:  CNI %04X, PIL %02d.%02d. %02d:%02d\n",
                                            pAcqVpsPdc->cni,
                                            (pAcqVpsPdc->pil >> 15) & 0x1F, (pAcqVpsPdc->pil >> 11) & 0x0F,
                                            (pAcqVpsPdc->pil >>  6) & 0x1F, (pAcqVpsPdc->pil) & 0x3F);
            }
            else
            {
               sprintf(comm + strlen(comm), "Channel VPS/PDC:  CNI %04X\n", pAcqVpsPdc->cni);
            }
         }
         else
            strcat(comm, "Channel VPS/PDC:  ---\n");

         sprintf(comm + strlen(comm),
                       "TTX data rate:    %d baud\n"
                       "EPG data rate:    %d baud (%1.1f%% of TTX)\n"
                       "EPG page rate:    %1.2f pages/sec\n"
                       "AI recv. count:   %d\n"
                       "AI min/avg/max:   %d/%2.2f/%d sec\n"
                       "PI rx repetition: %d/%.2f/%.2f now/s1/s2\n",
                       ((duration > 0) ? (int)((sv->ttx.ttxPkgCount*45*8)/duration) : 0),
                       ((duration > 0) ? (int)((sv->ttx.epgPkgCount*45*8)/duration) : 0),
                       ((sv->ttx.ttxPkgCount > 0) ? ((double)sv->ttx.epgPkgCount*100.0/sv->ttx.ttxPkgCount) : 0.0),
                       ((duration > 0) ? ((double)sv->ttx.epgPagCount / duration) : 0),
                       sv->ai.aiCount,
                       (int)sv->ai.minAiDistance,
                          ((sv->ai.aiCount > 1) ? ((double)sv->ai.sumAiDistance / (sv->ai.aiCount - 1)) : 0),
                          (int)sv->ai.maxAiDistance,
                       sv->nowNextMaxAcqRepCount, sv->count[0].avgAcqRepCount, sv->count[1].avgAcqRepCount
                );

         switch (acqState.mode)
         {
            case ACQMODE_PASSIVE:
               strcat(comm, "Acq mode:         passive\n");
               break;
            case ACQMODE_EXTERNAL:
               strcat(comm, "Acq mode:         external\n");
               break;
            case ACQMODE_FORCED_PASSIVE:
               if (acqState.state == ACQDESCR_DISABLED)
               {
                  strcat(comm, "Acq mode:         disabled\n");
               }
               else
               {
                  strcat(comm, "Acq mode:         forced passive\n");
                  #ifdef WIN32
                  if (MenuCmd_CheckTvCardConfig() == FALSE)
                  {
                     strcat(comm,    "Passive reason:   TV card not configured");
                  }
                  else
                  #endif
                  switch (acqState.passiveReason)
                  {
                     case ACQPASSIVE_NO_TUNER:
                        strcat(comm, "Passive reason:   input source is not a tuner\n");
                        break;
                     case ACQPASSIVE_NO_FREQ:
                        strcat(comm, "Passive reason:   frequency unknown\n");
                        break;
                     case ACQPASSIVE_NO_DB:
                        strcat(comm, "Passive reason:   database missing\n");
                        break;
                     case ACQPASSIVE_ACCESS_DEVICE:
                        strcat(comm, "Passive reason:   video device busy\n");
                        break;
                     default:
                        break;
                  }
               }
               break;

            case ACQMODE_FOLLOW_UI:
            case ACQMODE_FOLLOW_MERGED:
               if (acqState.cniCount <= 1)
                  strcat(comm, "Acq mode:         follow browser database\n");
               else
                  sprintf(comm + strlen(comm), "Acq mode:         follow browser (%d databases)\n", acqState.cniCount);
               break;
            case ACQMODE_CYCLIC_2:
               if (acqState.cniCount <= 1)
                  strcat(comm, "Acq mode:         manual\n");
               else
               {
                  if (acqState.cyclePhase == ACQMODE_PHASE_STREAM2)
                     strcat(comm, "Acq mode:         manual, phase 'All'\n");
                  else
                     strcat(comm, "Acq mode:         manual, phase 'Complete'\n");
               }
               break;
            default:
               switch (acqState.cyclePhase)
               {
                  case ACQMODE_PHASE_NOWNEXT:
                     strcat(comm, "Acq mode:         cyclic, phase 'Now'\n");
                     break;
                  case ACQMODE_PHASE_STREAM1:
                     strcat(comm, "Acq mode:         cyclic, phase 'Near'\n");
                     break;
                  case ACQMODE_PHASE_STREAM2:
                     strcat(comm, "Acq mode:         cyclic, phase 'All'\n");
                     break;
                  case ACQMODE_PHASE_MONITOR:
                     strcat(comm, "Acq mode:         cyclic, phase 'Complete'\n");
                     break;
                  default:
                     break;
               }
               break;
         }

         if (ACQMODE_IS_CYCLIC(acqState.mode))
         {
            sprintf(comm + strlen(comm), "Network variance: %1.2f / %1.2f\n",
                                         sv->count[0].variance, sv->count[1].variance);
         }

         strcat(comm, "}\n");
         eval_check(interp, comm);

         // remove the old update timer
         if (dbStatsWinState[target].updateHandler != NULL)
            Tcl_DeleteTimerHandler(dbStatsWinState[target].updateHandler);
         // set up a timer to re-display the stats in case there's no EPG reception
         if ((acqState.state != ACQDESCR_DISABLED) && (acqState.isNetAcq == FALSE))
            dbStatsWinState[target].updateHandler =
               Tcl_CreateTimerHandler(((sv->ai.aiCount < 2) ? 2*1000 : 15*1000), StatsWin_UpdateDbStatsWinTimeout, UINT2PVOID(target));
      }
      else
      {  // acq not running (at least for the same db)
         if (dbStatsWinState[target].isForAcq)
         {  // acq stats are still being displayed -> remove them
            sprintf(comm, "%s.acq.stat configure -text {}\n"
                          "pack forget %s.acq\n",
                          dbswn[target], dbswn[target]);
            eval_check(interp, comm);

            sprintf(comm, "DbStatsWin_ClearHistory %s\n", dbswn[target]);
            eval_check(interp, comm);

            dbStatsWinState[target].isForAcq = FALSE;
         }
      }
   }
}

// ----------------------------------------------------------------------------
// Schedule update of stats window after timeout
// - used only for stats windows with acquisition
// - timeout occurs when no AI is received
//
static void StatsWin_UpdateDbStatsWinTimeout( ClientData clientData )
{
   uint target = PVOID2UINT(clientData);

   if (target < 2)
   {
      dbStatsWinState[target].updateHandler = NULL;
      AddMainIdleEvent(StatsWin_UpdateDbStatsWin, clientData, FALSE);
   }
   else
      fatal1("StatsWin-UpdateDbStatsWinTimeout: illegal target %d", target);
}

// ----------------------------------------------------------------------------
// Update status line of browser window
// - possible bits of information about browser db:
//   + allversion percentage of PI
//   + time & date of last AI reception
//   + percentage of expired PI
// - possible bits of information about acq:
//   + state: Off, Startup, Running, Stalled
//   + CNI in acq db
//   + mode: forced-passive & reason, passive, cycle state
//
static void StatsWin_UpdateDbStatusLine( ClientData clientData )
{
   const EPGDB_BLOCK_COUNT * count;
   EPGDB_BLOCK_COUNT myCount[2];
   EPGACQ_DESCR acqState;
   const AI_BLOCK *pAi;
   char * pProvName, provName[20];
   ulong aiTotal, nearCount, allCount, expiredCount, expiredBase;
   time_t dbAge, acqMinTime[2];
   #ifdef USE_DAEMON
   EPGDBSRV_DESCR netState;
   #endif

   dprintf0("StatsWin-UpdateDbStatusLine: called\n");

   strcpy(comm, "set dbstatus_line {");

   if (IsDemoMode())
   {  // Demo database -> do not display statistics
      strcat(comm, "Demo database: start times are not real. ");
   }
   else
   if ((EpgDbContextIsMerged(pUiDbContext) == FALSE) && (EpgDbContextGetCni(pUiDbContext) != 0))
   {
      if (pAcqDbContext == pUiDbContext)
      {  // acq runs for the same db -> reuse acq statistics
         count = EpgAcqCtl_GetDbStats();
      }
      else
         count = NULL;
      if (count == NULL)
      {  // not the same db or acq not ready yet -> compute statistics now
         memset(acqMinTime, 0, sizeof(acqMinTime));
         EpgDbGetStat(pUiDbContext, myCount, acqMinTime, 0);
         count = myCount;
      }

      // get name of UI provider network
      EpgDbLockDatabase(pUiDbContext, TRUE);
      pAi = EpgDbGetAi(pUiDbContext);
      if (pAi != NULL)
         sprintf(comm + strlen(comm), "%s database", AI_GET_NETWOP_NAME(pAi, pAi->thisNetwop));
      else
         strcat(comm, "Browser database");
      EpgDbLockDatabase(pUiDbContext, FALSE);

      aiTotal      = count[0].ai + count[1].ai;
      allCount     = count[0].allVersions + count[0].expired + count[0].defective +
                     count[1].allVersions + count[1].expired + count[1].defective;
      expiredCount = count[0].expired + count[1].expired;
      expiredBase  = count[0].allVersions + count[1].allVersions + expiredCount;

      // print fill percentage across all db versions
      if ( (allCount == 0) || (aiTotal == 0) ||
           (expiredCount + count[0].defective + count[1].defective >= allCount) )
      {
         strcat(comm, " is empty");
      }
      else if (allCount < aiTotal)
      {
         sprintf(comm + strlen(comm), " %d%% complete", ACQ_COUNT_TO_PERCENT(allCount, aiTotal));
      }
      else
         strcat(comm, " 100% complete");

      // warn about age of database if more than one hour old
      dbAge = (time(NULL) - EpgDbGetAiUpdateTime(pUiDbContext)) / 60;
      if (dbAge >= 24*60)
      {
         sprintf(comm + strlen(comm), ", %1.1f days old", (double)dbAge / (24*60));
      }
      else if (dbAge >= 60)
      {
         sprintf(comm + strlen(comm), ", %ld:%02ld hours old", dbAge / 60, dbAge % 60);
      }

      // print how much of PI are expired
      if ( (expiredBase > 0) && (expiredCount > 0) &&
           ((dbAge >= 60) || (expiredCount * 10 >= expiredBase)) )
      {
         sprintf(comm + strlen(comm), ", %d%% expired", ACQ_COUNT_TO_PERCENT(expiredCount, expiredBase));
      }
      else if ((allCount > 0) && (allCount < aiTotal) && (count[0].ai > 0) && (aiTotal > 0))
      {  // append near percentage
         if ( ((count[0].allVersions + count[0].defective) / count[0].ai) > (allCount / aiTotal) )
         {
            sprintf(comm + strlen(comm), ", near data %d%%", ACQ_COUNT_TO_PERCENT(count[0].allVersions + count[0].defective, count[0].ai));
         }
      }
      strcat(comm, ". ");
   }

   // fetch current state and acq db statistics from acq control
   EpgAcqCtl_DescribeAcqState(&acqState);
   aiTotal = nearCount = allCount = 0;
   pProvName = NULL;
   count = NULL;
   if (pAcqDbContext != NULL)
   {
      // get name of acq provider network
      EpgDbLockDatabase(pAcqDbContext, TRUE);
      pAi = EpgDbGetAi(pAcqDbContext);
      if (pAi != NULL)
      {
         strncpy(provName, AI_GET_NETWOP_NAME(pAi, pAi->thisNetwop), sizeof(provName) - 1);
         provName[sizeof(provName) - 1] = 0;
         pProvName = provName;
      }
      EpgDbLockDatabase(pAcqDbContext, FALSE);

      if (acqState.state == ACQDESCR_RUNNING)
      {
         count = EpgAcqCtl_GetDbStats();
         if (count != NULL)
         {
            // evaluate some totals for later use in percentage evaluations
            aiTotal   = count[0].ai + count[1].ai;
            nearCount = count[0].curVersion + count[0].expired + count[0].defective;
            allCount  = nearCount +
                        count[1].curVersion + count[1].expired + count[1].defective;
         }
      }
   }


   // describe acq state (disabled, network connect, wait AI, running)

   #ifdef USE_DAEMON
   if (acqState.isNetAcq)
   {
      if ( EpgAcqClient_DescribeNetState(&netState) )
      {
         // network connection to acquisition server is not set up -> describe network state instead
         switch (netState.state)
         {
            case NETDESCR_CONNECT:
            case NETDESCR_STARTUP:
               strcat(comm, "Acquisition: connecting to server..");
               break;
            case NETDESCR_ERROR:
            case NETDESCR_DISABLED:
               strcat(comm, "Acquisition stalled: network error");
               break;
            case NETDESCR_LOADING:
               strcat(comm, "Acquisition: loading database from server..");
               break;
            case NETDESCR_RUNNING:
               if (acqState.state == ACQDESCR_NET_CONNECT)
               {  // note: in state RUNNING display "loading" until the first stats report is available
                  strcat(comm, "Acquisition: loading database from server..");
                  acqState.state = ACQDESCR_NET_CONNECT;
               }
               break;
            default:
               fatal1("StatsWin-UpdateDbStatusLine: unknown net state %d", netState.state);
               break;
         }

         if (netState.state != NETDESCR_RUNNING)
         {  // set acq state to suppress additional output (e.g. acq phase)
            acqState.state = ACQDESCR_NET_CONNECT;
         }
      }
   }
   #endif

   switch (acqState.state)
   {
      case ACQDESCR_DISABLED:
         strcat(comm, "Acquisition is disabled");
         break;
      case ACQDESCR_NET_CONNECT:
      case ACQDESCR_SCAN:
         break;
      case ACQDESCR_STARTING:
         if ( (acqState.dbCni == acqState.cycleCni) && (acqState.dbCni != 0) &&
              (acqState.mode != ACQMODE_FORCED_PASSIVE) &&
              (acqState.mode != ACQMODE_PASSIVE) &&
              (acqState.mode != ACQMODE_EXTERNAL) &&
              (pProvName != NULL) )
         {
            sprintf(comm + strlen(comm), "Acquisition is starting up for %s", pProvName);
         }
         else
            strcat(comm, "Acquisition is waiting for reception");
         break;
      case ACQDESCR_NO_RECEPTION:
         #ifdef WIN32
         if (MenuCmd_CheckTvCardConfig() == FALSE)
         {
            strcat(comm, "Acquisition: TV card not configured");
         }
         else
         #endif
         if ( (acqState.dbCni == acqState.cycleCni) && (acqState.dbCni != 0) &&
              (acqState.mode != ACQMODE_FORCED_PASSIVE) &&
              (acqState.mode != ACQMODE_PASSIVE) &&
              (acqState.mode != ACQMODE_EXTERNAL) &&
              (pProvName != NULL) )
         {
            sprintf(comm + strlen(comm), "Acquisition: no reception on %s", pProvName);
         }
         else
            strcat(comm, "Acquisition: no reception");
         break;
      case ACQDESCR_STALLED:
         strcat(comm, "Acquisition stalled");
         if ( (acqState.mode != ACQMODE_PASSIVE) &&
              (acqState.mode != ACQMODE_EXTERNAL) &&
              (acqState.mode != ACQMODE_FORCED_PASSIVE) )
         {
            if (pProvName != NULL)
               sprintf(comm + strlen(comm), " on %s", pProvName);
         }
         break;
      case ACQDESCR_RUNNING:
         if (pProvName != NULL)
         {
            sprintf(comm + strlen(comm), "Acquisition working on %s", pProvName);
         }
         else
         {  // internal inconsistancy
            fatal0("StatsWin-UpdateDbStatusLine: no AI block while in state RUNNING");
            sprintf(comm + strlen(comm), "Acquisition in progress..");
            // patch state to prevent further output
            acqState.state = ACQDESCR_NET_CONNECT;
         }
         break;

      default:
         fatal1("StatsWin-UpdateDbStatusLine: unknown acq state %d", acqState.state);
   }

   // describe acq mode, warn the user if passive
   if ( (acqState.state != ACQDESCR_NET_CONNECT) &&
        (acqState.state != ACQDESCR_DISABLED) &&
        (acqState.state != ACQDESCR_SCAN))
   {
      if (acqState.mode == ACQMODE_PASSIVE)
      {
         if ((acqState.state == ACQDESCR_RUNNING) && (count != NULL))
         {
            sprintf(comm + strlen(comm), ", %d%% complete", ACQ_COUNT_TO_PERCENT(allCount, aiTotal));
         }
         strcat(comm, " (passive mode)");
      }
      else if (acqState.mode == ACQMODE_EXTERNAL)
      {
         if ((acqState.state == ACQDESCR_RUNNING) && (count != NULL))
         {
            sprintf(comm + strlen(comm), ", %d%% complete", ACQ_COUNT_TO_PERCENT(allCount, aiTotal));
         }
         strcat(comm, " (external)");
      }
      else if (acqState.mode == ACQMODE_FORCED_PASSIVE)
      {
         if ((acqState.state == ACQDESCR_RUNNING) && (count != NULL))
         {
            sprintf(comm + strlen(comm), ", %d%% complete", ACQ_COUNT_TO_PERCENT(allCount, aiTotal));
         }
         strcat(comm, " (forced passive)");
      }
      else if ((acqState.dbCni != acqState.cycleCni) && (acqState.cycleCni != 0))
      {
         debug2("StatsWin-UpdateDbStatusLine: prov change pending: db %04X  cycle %04X", acqState.dbCni, acqState.cycleCni);
         strcat(comm, " (provider change pending)");
      }
      else if ( (acqState.mode == ACQMODE_FOLLOW_UI) ||
                (acqState.mode == ACQMODE_FOLLOW_MERGED) ||
                (acqState.mode == ACQMODE_CYCLIC_2) )
      {
         if (count != NULL)
         {
            sprintf(comm + strlen(comm), ", %d%% complete", ACQ_COUNT_TO_PERCENT(allCount, aiTotal));
         }
      }
      else
      {
         switch (acqState.cyclePhase)
         {
            case ACQMODE_PHASE_NOWNEXT:
               strcat(comm, ", phase 'Now'");
               break;
            case ACQMODE_PHASE_STREAM1:
               if (count != NULL)
                  sprintf(comm + strlen(comm), " phase 'Near', %d%% complete", ACQ_COUNT_TO_PERCENT(nearCount, count[0].ai));
               else
                  strcat(comm, " phase 'Near'");
               break;
            case ACQMODE_PHASE_STREAM2:
               if (count != NULL)
                  sprintf(comm + strlen(comm), " phase 'All', %d%% complete", ACQ_COUNT_TO_PERCENT(allCount, aiTotal));
               else
                  strcat(comm, ", phase 'All'");
               break;
            case ACQMODE_PHASE_MONITOR:
               if (count != NULL)
                  sprintf(comm + strlen(comm), " phase 'Complete', %d%% complete", ACQ_COUNT_TO_PERCENT(allCount, aiTotal));
               else
                  strcat(comm, ", phase 'Complete'");
               break;
            default:
               break;
         }
      }
   }

   if (acqState.state == ACQDESCR_STARTING)
      strcat(comm, "...");
   else if (acqState.state != ACQDESCR_SCAN)
      strcat(comm, ".");

   strcat(comm, "}\n");
   eval_global(interp, comm);
}

// ----------------------------------------------------------------------------
// Open or close window with db and acq statistics
// - called by the "View (acq) statistics" menu entries (toggle mode, argc=2)
//   or as a callback when a statistics window is destroyed
//
static int StatsWin_ToggleDbStats( ClientData ttp, Tcl_Interp *interp, int argc, CONST84 char *argv[] )
{
   const char * const pUsage = "Usage: C_StatsWin_ToggleDbStats ui|acq [0|1]";
   EPGDB_CONTEXT * dbc;
   int target, newState;
   int result = TCL_OK;

   dbc = NULL;  // avoid compiler warnings
   target = -1;

   if ((argc == 2) || (argc == 3))
   {
      // determine target from first parameter: ui or acq db context
      if ((strcmp(argv[1], "ui") == 0) || (strcmp(argv[1], dbswn[DB_TARGET_UI]) == 0))
      {
         dbc = pUiDbContext;
         target = DB_TARGET_UI;
      }
      else if ((strcmp(argv[1], "acq") == 0) || (strcmp(argv[1], dbswn[DB_TARGET_ACQ]) == 0))
      {
         dbc = pAcqDbContext;
         target = DB_TARGET_ACQ;
      }
   }

   if (target != -1)
   {
      result = TCL_OK;
      // determine new state from optional second parameter: 0, 1 or toggle
      if (argc == 2)
      {  // no state parameter -> toggle state
         newState = ! dbStatsWinState[target].open;
      }
      else
      {
         result = Tcl_GetBoolean(interp, argv[2], &newState);
      }

      if ( (result == TCL_OK) && (newState != dbStatsWinState[target].open) )
      {
         if (dbStatsWinState[target].open == FALSE)
         {  // window shall be opened
            sprintf(comm, "DbStatsWin_Create %s\n", dbswn[target]);
            eval_check(interp, comm);

            dbStatsWinState[target].open = TRUE;
            dbStatsWinState[target].isForAcq = FALSE;
            dbStatsWinState[target].lastHistPos = 0;
            dbStatsWinState[target].lastHistReset = 0;
            dbStatsWinState[target].updateHandler = NULL;

            if ((target == DB_TARGET_ACQ) || (pAcqDbContext == pUiDbContext))
            {
               StatsWin_UpdateHist(target);
            }
            // enable extended statistics reports
            EpgAcqCtl_EnableAcqStats(TRUE);

            // display initial summary
            StatsWin_UpdateDbStatsWin(UINT2PVOID(target));
         }
         else
         {  // destroy the window

            // note: set to FALSE before window destruction to avoid recursion
            dbStatsWinState[target].open = FALSE;

            sprintf(comm, "destroy %s", dbswn[target]);
            eval_check(interp, comm);

            // disable extended statistics reports when all windows are closed
            if ( (dbStatsWinState[DB_TARGET_ACQ].open == FALSE) &&
                 (dbStatsWinState[DB_TARGET_UI].open == FALSE) )
            {
               EpgAcqCtl_EnableAcqStats(FALSE);
            }
         }
         // set the state of the checkbutton of the according menu entry
         sprintf(comm, "set menuStatusStatsOpen(%s) %d\n",
                       ((target == DB_TARGET_UI) ? "ui" : "acq"),
                       dbStatsWinState[target].open);
         eval_check(interp, comm);
      }
   }
   else
   {  // parameter error -> display usage
      Tcl_SetResult(interp, (char *)pUsage, TCL_STATIC);
      result = TCL_ERROR;
   }

   return result;
}

// ----------------------------------------------------------------------------
// Schedule an update of all statistics output
// - argument target says which statistics popup needs to be updated.
//   Since the main window status line contains info about both acq and ui db
//   its update is unconditional.
// - the function is triggered by acq control regularily upon every AI reception
//   and after provider changes or after acquisition parameters are changed
//   or after a provider change by the user for the browser database
// - the function is also triggered by GUI provider changes
//   plus regularily every minute (to update expiration statistics)
// - execution is delayed, mainly for calls from acq control; then we must wait
//   until the EPG block queue is processed and the latest acq db stats are
//   available (the function is called from a callback before the AI block is
//   actually in the db)
//
void StatsWin_StatsUpdate( int target )
{
   // update the db statistics window of the given db, if open
   if (dbStatsWinState[target].open)
   {
      AddMainIdleEvent(StatsWin_UpdateDbStatsWin, UINT2PVOID(target), FALSE);
   }

   // if the ui db is the same as the acq db, forward updates from acq to ui stats popup
   if ( (target == DB_TARGET_ACQ) &&
        dbStatsWinState[DB_TARGET_UI].open &&
        ((pUiDbContext == pAcqDbContext) || dbStatsWinState[DB_TARGET_UI].isForAcq) )
   {
      AddMainIdleEvent(StatsWin_UpdateDbStatsWin, (ClientData)DB_TARGET_UI, FALSE);
   }

   // update the main window status line
   AddMainIdleEvent(StatsWin_UpdateDbStatusLine, NULL, TRUE);
}

// ----------------------------------------------------------------------------
// Initialize module state variables
// - this should be called only once during start-up
//
void StatsWin_Create( void )
{
   Tcl_CmdInfo cmdInfo;

   if (Tcl_GetCommandInfo(interp, "C_StatsWin_ToggleDbStats", &cmdInfo) == 0)
   {
      Tcl_CreateCommand(interp, "C_StatsWin_ToggleDbStats", StatsWin_ToggleDbStats, (ClientData) NULL, NULL);
   }
   else
      fatal0("StatsWin-Create: commands are already created");

   memset(dbStatsWinState, 0, sizeof(dbStatsWinState));
}

